/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2019 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include "dqm_dev.h"
#include "dqm_dbg.h"

#define PROC_DIR		"driver/dqm"
#define STATUS_PROC_FILE	"status"
#define CMD_PROC_FILE		"cmd"
#define MIB_PROC_FILE		"mib"
#define MIBALL_PROC_FILE	"miball"

static int dqm_mib_proc_open(struct inode *inode, struct file *file);
static int dqm_miball_proc_open(struct inode *inode, struct file *file);
static void *dqm_mib_proc_start(struct seq_file *seq, loff_t *pos);
static void dqm_mib_proc_stop(struct seq_file *seq, void *v);
static void *dqm_mib_proc_next(struct seq_file *seq, void *v,
				  loff_t *pos);
static int dqm_mib_proc_show(struct seq_file *seq, void *v);
static int dqm_miball_proc_show(struct seq_file *seq, void *v);
static int dqm_status_proc_open(struct inode *inode, struct file *file);
static void *dqm_status_proc_start(struct seq_file *seq, loff_t *pos);
static void dqm_status_proc_stop(struct seq_file *seq, void *v);
static void *dqm_status_proc_next(struct seq_file *seq, void *v,
				  loff_t *pos);
static int dqm_status_proc_show(struct seq_file *seq, void *v);
static int dqm_proc_cmd_reset_stats(void *data, int argc, char *argv[]);

static const struct seq_operations mib_seq_ops = {
	.start	= dqm_mib_proc_start,
	.stop	= dqm_mib_proc_stop,
	.next	= dqm_mib_proc_next,
	.show	= dqm_mib_proc_show,
};

static const struct proc_ops mib_fops = {
	.proc_open		= dqm_mib_proc_open,
	.proc_read		= seq_read,
	.proc_lseek		= seq_lseek,
	.proc_release	= seq_release,
};

static const struct seq_operations miball_seq_ops = {
	.start	= dqm_mib_proc_start,
	.stop	= dqm_mib_proc_stop,
	.next	= dqm_mib_proc_next,
	.show	= dqm_miball_proc_show,
};

static const struct proc_ops miball_fops = {
	.proc_open		= dqm_miball_proc_open,
	.proc_read		= seq_read,
	.proc_lseek		= seq_lseek,
	.proc_release	= seq_release,
};

static const struct seq_operations status_seq_ops = {
	.start	= dqm_status_proc_start,
	.stop	= dqm_status_proc_stop,
	.next	= dqm_status_proc_next,
	.show	= dqm_status_proc_show,
};

static const struct proc_ops status_fops = {
	.proc_open		= dqm_status_proc_open,
	.proc_read		= seq_read,
	.proc_lseek		= seq_lseek,
	.proc_release	= seq_release,
};

static int dqm_show_total(struct seq_file *seq, struct dqmdev *qdev)
{
	int bank;
	u32 offUsed, offPushed, offPopped;

	if (!qdev->cfg || (qdev->intc[0].type == DQM_INTC_LEGACY))
		return 0;
	offUsed = 25;
	offPushed = 26;
	offPopped = 27;

	pr_seq(seq, "\nDEVICE: %s Cumulative count for all queues\n",
	       qdev->name);
	for (bank = 0; bank < qdev->bank_count; bank++) {
		pr_seq(seq, "Bank %d: Used %4u, Pushed %10u, Popped %10u\n",
		       bank,
		       dqm_reg_read(qdev->cfg[bank] + offUsed),
		       dqm_reg_read(qdev->cfg[bank] + offPushed),
		       dqm_reg_read(qdev->cfg[bank] + offPopped));
	}
	pr_seq(seq, "\n");

	return 0;
}

static int dqm_show(struct seq_file *seq, struct dqmdev *qdev, int all)
{
	int i;
	struct dqm *q;
	u32 offF, offE, offP;

	if (qdev->intc[0].type == DQM_INTC_LEGACY) {
		offF = 0;
		offE = 32;
		offP = 64;
	} else {
		offF = 0;
		offE = 1;
		offP = 2;
	}

	pr_seq(seq, "\nDEVICE: %s\n", qdev->name);
	for (i = 0; i < qdev->q_count; i++) {

		q = &qdev->dqm[i];
		if (!q->mib)
			continue;
		if (all || DQM_GET_NUM_TOK(q))
		pr_seq(seq, "%*s [%2d] N %4u, A %4u, U %4u, nF %10u, nE %10u, nP %10u\n",
			   (int) sizeof(q->name), q->name, i,
			   DQM_GET_NUM_TOK(q), DQM_GET_Q_SPACE(q),
			   DQM_GET_NUM_TOK(q) - DQM_GET_Q_SPACE(q),
			   dqm_reg_read(q->mib + offF),
			   dqm_reg_read(q->mib + offE),
			   dqm_reg_read(q->mib + offP));

	}
	pr_seq(seq, "\n\n");
	return 0;
}

static void dqm_proc_cmd_show_help(char *str)
{
	pr_alert("%s show: Show MIB counters\n", str);
}

static int dqm_proc_cmd_show(void *data, int argc, char *argv[])
{
	struct dqmdev *qdev;

	rcu_read_lock();
	if (data) {
		qdev = (struct dqmdev *) data;
		dqm_show(NULL, qdev, 0);
	} else {
		list_for_each_entry_rcu(qdev, &dqmdevs, list)
			dqm_show(NULL, qdev, 0);
	}
	rcu_read_unlock();

	return 0;
}

static void dqm_proc_cmd_reset_stats_help(char *str)
{
	pr_alert("%s reset: Reset stats\n", str);
}

static int dqm_proc_cmd_mon(void *data, int argc, char *argv[])
{
	int qnum, hi_thresh, lo_thresh, log;
	struct dqm_mon_cb cb = {};

	if (argc == 6) {
		if (kstrtou32(argv[2], 0, &qnum)) {
			goto help;
		} else if (kstrtou32(argv[3], 0, &hi_thresh)) {
			goto help;
		} else if (kstrtou32(argv[4], 0, &lo_thresh)) {
			goto help;
		} else if (kstrtou32(argv[5], 0, &log)) {
			goto help;
		}
		dqm_mon(argv[1], qnum, hi_thresh, lo_thresh, log, &cb);
		goto done;
	}
help:
	pr_info("%s <qname> <qnum> <hi_thresh> <lo_thresh>\n", argv[0]);
done:
       return 0;
}

static void dqm_proc_cmd_mon_help(char *str)
{
	pr_alert("%s mon: Configure queue monitoring\n", str);
}

static int dqm_proc_cmd_mon_start(void *data, int argc, char *argv[])
{
	int start;

	if (argc == 2) {
		if (kstrtou32(argv[1], 0, &start) == 0) {
			dqm_mon_start(start);
			goto done;
		}
	}
	pr_info("%s <1|0>\n", argv[0]);
done:
       return 0;
}

static void dqm_proc_cmd_mon_start_help(char *str)
{
	pr_alert("%s mon_start: Start queue monitoring\n", str);
}

static struct proc_cmd_ops command_entries[] = {
	PROC_CMD_DATA_INIT("reset", dqm_proc_cmd_reset_stats),
	PROC_CMD_DATA_INIT("show",  dqm_proc_cmd_show),
	PROC_CMD_DATA_INIT("mon", dqm_proc_cmd_mon),
	PROC_CMD_DATA_INIT("mon_start",  dqm_proc_cmd_mon_start),
};

struct proc_cmd_table dqm_command_table = {
	.module_name = "DQM",
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

static struct proc_dir_entry *proc_dir;
static struct proc_dir_entry *status_proc_file;
static struct proc_dir_entry *cmd_proc_file;
static struct proc_dir_entry *mib_proc_file;
static struct proc_dir_entry *miball_proc_file;

static int panic_callback(struct notifier_block *self,
			  unsigned long event, void *ctx)
{
	struct dqmdev *qdev;
	time64_t now;
	struct tm tm_val;

	now = ktime_get_real_seconds();
	time64_to_tm(now, 0, &tm_val);

	pr_emerg("---[ DQM dump: Start %d/%d/%ld %02d:%02d:%02d ]---\n",
		 tm_val.tm_mon + 1, tm_val.tm_mday, 1900 + tm_val.tm_year,
		 tm_val.tm_hour, tm_val.tm_min, tm_val.tm_sec);
	rcu_read_lock();
	list_for_each_entry_rcu(qdev, &dqmdevs, list)
		dqm_show(NULL, qdev, 0);
	rcu_read_unlock();
	pr_emerg("---[ DQM dump: End ]---\n");
	return NOTIFY_OK;
}


static struct notifier_block nb_panic = {
	.notifier_call  = panic_callback,
};

int __init dqm_proc_init(void)
{
	int status = 0;

	pr_debug("-->\n");
	proc_dir = proc_mkdir(PROC_DIR, NULL);
	if (!proc_dir) {
		pr_err("Failed to create PROC directory %s.\n",
		       PROC_DIR);
		status = -EIO;
		goto done;
	}
	status_proc_file = proc_create(STATUS_PROC_FILE, S_IRUGO,
					 proc_dir, &status_fops);
	if (!status_proc_file) {
		pr_err("Failed to create %s\n", STATUS_PROC_FILE);
		status = -EIO;
		dqm_proc_exit();
		goto done;
	}
	cmd_proc_file = proc_create_cmd(CMD_PROC_FILE, proc_dir,
					&dqm_command_table);
	if (!cmd_proc_file) {
		pr_err("Failed to create %s\n", CMD_PROC_FILE);
		status = -EIO;
		dqm_proc_exit();
		goto done;
	}
	mib_proc_file = proc_create_data(MIB_PROC_FILE, S_IRUGO,
					 proc_dir, &mib_fops, NULL);
	if (!mib_proc_file) {
		pr_err("Failed to create %s\n", MIB_PROC_FILE);
		status = -EIO;
		dqm_proc_exit();
		goto done;
	}
	miball_proc_file = proc_create_data(MIBALL_PROC_FILE, S_IRUGO,
					    proc_dir, &miball_fops, NULL);
	if (!miball_proc_file) {
		pr_err("Failed to create %s\n", MIBALL_PROC_FILE);
		status = -EIO;
		dqm_proc_exit();
		goto done;
	}
	atomic_notifier_chain_register(&panic_notifier_list,
				       &nb_panic);

done:
	pr_debug("<--\n");
	return status;
}

void dqm_proc_exit(void)
{
	pr_debug("-->\n");
	if (cmd_proc_file) {
		remove_proc_entry(CMD_PROC_FILE, proc_dir);
		cmd_proc_file = NULL;
	}
	if (status_proc_file) {
		remove_proc_entry(STATUS_PROC_FILE, proc_dir);
		status_proc_file = NULL;
	}
	if (mib_proc_file) {
		remove_proc_entry(MIB_PROC_FILE, proc_dir);
		mib_proc_file = NULL;
	}
	if (proc_dir) {
		remove_proc_entry(PROC_DIR, NULL);
		proc_dir = NULL;
	}
	atomic_notifier_chain_unregister(&panic_notifier_list,
					 &nb_panic);
	pr_debug("<--\n");
}

int dqmdev_proc_init(struct dqmdev *qdev)
{
	int status = 0;

	pr_debug("-->\n");
	qdev->proc_dir = proc_mkdir(qdev->name, proc_dir);
	if (!qdev->proc_dir) {
		pr_err("Failed to create PROC directory %s.\n",
		       PROC_DIR);
		status = -EIO;
		goto done;
	}
	qdev->status_proc_file = proc_create_data(STATUS_PROC_FILE, S_IRUGO,
						  qdev->proc_dir,
						  &status_fops, qdev);
	if (!qdev->status_proc_file) {
		pr_err("Failed to create %s/%s\n", qdev->name,
		       STATUS_PROC_FILE);
		status = -EIO;
		dqmdev_proc_exit(qdev);
		goto done;
	}
	qdev->proc_cmd = dqm_command_table;
	qdev->proc_cmd.data = qdev;
	qdev->cmd_proc_file = proc_create_cmd(CMD_PROC_FILE, qdev->proc_dir,
					      &qdev->proc_cmd);
	if (!qdev->cmd_proc_file) {
		pr_err("Failed to create %s/%s\n", qdev->name,
		       CMD_PROC_FILE);
		status = -EIO;
		dqmdev_proc_exit(qdev);
		goto done;
	}
	qdev->mib_proc_file = proc_create_data(MIB_PROC_FILE, S_IRUGO,
					       qdev->proc_dir,
					       &mib_fops, qdev);
	if (!qdev->mib_proc_file) {
		pr_err("Failed to create %s/%s\n", qdev->name,
		       MIB_PROC_FILE);
		status = -EIO;
		dqmdev_proc_exit(qdev);
		goto done;
	}
	qdev->miball_proc_file = proc_create_data(MIBALL_PROC_FILE, S_IRUGO,
						  qdev->proc_dir, &miball_fops,
						  qdev);
	if (!qdev->miball_proc_file) {
		pr_err("Failed to create %s/%s\n", qdev->name,
		       MIBALL_PROC_FILE);
		status = -EIO;
		dqmdev_proc_exit(qdev);
		goto done;
	}

done:
	pr_debug("<--\n");
	return status;
}

void dqmdev_proc_exit(struct dqmdev *qdev)
{
	pr_debug("-->\n");
	if (qdev->cmd_proc_file) {
		remove_proc_entry(CMD_PROC_FILE, qdev->proc_dir);
		qdev->cmd_proc_file = NULL;
	}
	if (qdev->status_proc_file) {
		remove_proc_entry(STATUS_PROC_FILE, qdev->proc_dir);
		qdev->status_proc_file = NULL;
	}
	if (qdev->mib_proc_file) {
		remove_proc_entry(MIB_PROC_FILE, qdev->proc_dir);
		qdev->mib_proc_file = NULL;
	}
	if (qdev->proc_dir) {
		remove_proc_entry(qdev->name, proc_dir);
		qdev->proc_dir = NULL;
	}
	pr_debug("<--\n");
}

static int dqm_status_proc_open(struct inode *inode, struct file *file)
{
	int status;

	pr_debug("-->\n");
	status = seq_open(file, &status_seq_ops);
	if (!status) {
		struct seq_file *sf = file->private_data;

		sf->private = PDE_DATA(inode);
	}
	pr_debug("<--\n");
	return status;
}

static void *dqm_status_proc_start(struct seq_file *seq, loff_t *pos)
{
	struct dqmdev *qdev = NULL;
	struct dqmdev *tmp;
	loff_t off = 0;

	pr_debug("-->\n");

	if (seq->private) {
		if (off++ == *pos)
			return seq->private;
		return NULL;
	}

	rcu_read_lock();
	list_for_each_entry_rcu(tmp, &dqmdevs, list) {
		if (off++ == *pos) {
			qdev = tmp;
			break;
		}
	}
	rcu_read_unlock();

	pr_debug("<--\n");
	return qdev;
}

static void dqm_status_proc_stop(struct seq_file *seq, void *v)
{
	pr_debug("-->\n");
	pr_debug("<--\n");
}

static void *dqm_status_proc_next(struct seq_file *seq, void *v,
				  loff_t *pos)
{
	struct dqmdev *qdev = NULL;
	struct dqmdev *cur = v;
	struct list_head *next;

	pr_debug("-->\n");

	if (seq->private)
		return NULL;

	(*pos)++;
	rcu_read_lock();
	next = cur->list.next;
	if (next != &dqmdevs)
		qdev = container_of(next, struct dqmdev, list);
	rcu_read_unlock();
	pr_debug("<--\n");
	return qdev;
}

#define show_reg_dqm(qdev, name, format, member) ({ \
	int i, first = 0; \
	struct dqm *q; \
	for (i = 0; i < qdev->q_count; i++) { \
		q = &qdev->dqm[i]; \
		if ((q->flags & DQM_F_TX) || (q->flags & DQM_F_RX)) { \
			if (!first) { \
				seq_printf(seq, "\n\t%s", name); \
				first = 1; \
			} \
			seq_printf(seq, format, member); \
		} \
	} })

#define show_reg_dqm_avg_depth(qdev, name, format, member) ({ \
	int i, first = 0; \
	struct dqm *q; \
	for (i = 0; i < qdev->q_count; i++) { \
		q = &qdev->dqm[i]; \
		if ((q->flags & DQM_F_TX) || (q->flags & DQM_F_RX)) { \
			if (!first) { \
				seq_printf(seq, "\n\t%s", name); \
				first = 1; \
			} \
			seq_printf(seq, format, \
				q->stats.member ## _attempts ? \
				q->stats.member ## _q_depth_sum / \
				q->stats.member ## _attempts : 0); \
		} \
	} })

static int dqm_status_proc_show(struct seq_file *seq, void *v)
{
	struct dqmdev *qdev = v;
	struct dqm_intc_reg *intcreg;
	int i, bank;
	struct dqm *q;

	pr_debug("-->\n");

	seq_printf(seq, "DEVICE: %s\n", qdev->name);
	if (!qdev->intc)
		goto print_dqm;
	if (qdev->intc[0].type != DQM_INTC_LEGACY)
		goto print_dqm;
	for (i = 0; i < qdev->intc_count; i++) {
		intcreg = qdev->intc[i].reg;
		seq_printf(seq, "\tIRQ Controller %d: Type %d\n", i,
			   qdev->intc[i].type);
		if (qdev->intc[0].reg[0].l1_irq_mask)
			seq_printf(seq, "\t\tL1 IRQ Mask  : 0x%08x\n",
			   dqm_reg_read(intcreg[0].l1_irq_mask));
		if (qdev->intc[0].reg[0].l1_irq_status)
			seq_printf(seq, "\t\tL1 IRQ Status: 0x%08x\n",
			   dqm_reg_read(intcreg[0].l1_irq_status));
		seq_printf(seq, "\tDQM  | Low Watermark | Low Watermark | "
			   "Not Empty  | Not Empty  | Not Empty ");
		seq_printf(seq, " | Hi Watermark | Hi Watermark");
		seq_printf(seq, " |   Timer    |   Timer   ");
		seq_printf(seq, "\n");
		seq_printf(seq, "\tBank |   IRQ Mask    |   IRQ Status  | "
			   "IRQ Mask   | IRQ Status |   Status  ");
		seq_printf(seq, " |   IRQ Mask   |  IRQ Status ");
		seq_printf(seq, " | IRQ Mask   | IRQ Status");
		seq_printf(seq, "\n");
		for (bank = 0; bank < qdev->bank_count; bank++) {
			seq_printf(seq, "\t%4d  | ", bank);
			seq_printf(seq, "0x%08x    | ",
				   dqm_reg_read(intcreg[bank].lwm_irq_mask));
			seq_printf(seq, "0x%08x    | ",
				   dqm_reg_read(intcreg[bank].lwm_irq_status));
			seq_printf(seq, "0x%08x | ",
				   dqm_reg_read(intcreg[bank].ne_irq_mask));
			seq_printf(seq, "0x%08x | ",
				   dqm_reg_read(intcreg[bank].ne_irq_status));
			seq_printf(seq, "0x%08x",
				   dqm_reg_read(intcreg[bank].ne_status));
			if (intcreg[bank].hwm_irq_mask) {
				seq_printf(seq, " | 0x%08x  ",
				   dqm_reg_read(intcreg[bank].hwm_irq_mask));
				seq_printf(seq, " | 0x%08x  ",
				   dqm_reg_read(intcreg[bank].hwm_irq_status));
			}
			if (intcreg[bank].tmr_irq_mask) {
				seq_printf(seq, " | 0x%08x",
				   dqm_reg_read(intcreg[bank].tmr_irq_mask));
				seq_printf(seq, " | 0x%08x",
				   dqm_reg_read(intcreg[bank].tmr_irq_status));
			}
			seq_printf(seq, "\n");
		}
	}
	seq_printf(seq, "\n");
print_dqm:
	seq_printf(seq, "\tRegistered DQM's:\n");

	seq_printf(seq, "     |   Msgs   |   Msgs   | Attempts | Attempts |   Peak   |   Peak   |    Avg   |    Avg   |\n");
	seq_printf(seq, " DQM |    TX    |    RX    |    TX    |    RX    | Depth TX | Depth RX | Depth TX | Depth RX |\n");
	for (i = 0; i < qdev->q_count; i++) {
		unsigned int avgrx = 0;

		q = &qdev->dqm[i];
		if ((q->flags & DQM_F_TX) || (q->flags & DQM_F_RX)) {

			if (q->stats.rx_attempts)
				avgrx = q->stats.rx_q_depth_sum / q->stats.rx_attempts;
			if (q->type != DQM_INTC_LEGACY)
				avgrx /= q->msg_size;

			seq_printf(seq, " %3d |%10d|%10d|%10d|%10d|%10d|%10d|%10d|%10d|\n",
				   q->num, q->stats.tx_cnt, q->stats.rx_cnt,
				   q->stats.tx_attempts, q->stats.rx_attempts,
				   q->stats.tx_q_depth_peak,
				   (q->type != DQM_INTC_LEGACY) ? q->stats.rx_q_depth_peak /  q->msg_size : q->stats.rx_q_depth_peak,
				   q->stats.tx_attempts ? q->stats.tx_q_depth_sum / q->stats.tx_attempts : 0,
				   avgrx);
		}
	}
	seq_printf(seq, "---------------------------------------------------------------------------------------------------\n");
	seq_printf(seq, "     |  Token   |    QSM   |    Num   |   Avail  |   Low    |    Hi    |  Timeout |   Start  |\n");
	seq_printf(seq, " DQM |   Size   | Allocated|   Token  |   Token  | Water Mrk| Water Mrk|   (us)   |   Addr   |\n");
	for (i = 0; i < qdev->q_count; i++) {
		q = &qdev->dqm[i];
		if ((q->flags & DQM_F_TX) || (q->flags & DQM_F_RX)) {
			seq_printf(seq, " %3d | %8d | %8d | %8d | %8d | %8d | %8d | %8d | 0x%06x |\n",
				   q->num, DQM_GET_TOK_SIZE(q) + 1,
				   DQM_GET_Q_SIZE(q),
				   q->depth, DQM_GET_Q_SPACE(q),
				   q->lwm, q->hwm, q->timeout,
				   DQM_GET_Q_ADDR(q));
		}
	}
	seq_printf(seq, "---------------------------------------------------------------------------------------------------\n");
	seq_printf(seq, " DQM |   NE     |   LWM    |    HWM   |  Timeout |\n");
	for (i = 0; i < qdev->q_count; i++) {
		q = &qdev->dqm[i];
		if ((q->flags & DQM_F_TX) || (q->flags & DQM_F_RX)) {
			seq_printf(seq, " %3d | %8d | %8d | %8d | %8d |\n",
				   q->num, q->stats.cntne, q->stats.cntlwm,
				   q->stats.cnthwm, q->stats.cnttmr);
		}
	}
	seq_printf(seq, "---------------------------------------------------------------------------------------------------\n");
	seq_printf(seq, "-------------------Monitoring-Running-%d-----------------------------------------------------------\n",
		   is_dqm_mon_running());
	seq_printf(seq, " DQM | %%hi %%lo | L | S |    Count   | Name\n");
	for (i = 0; i < qdev->q_count; i++) {
		q = &qdev->dqm[i];
		if ((q->flags & DQM_F_TX) || (q->flags & DQM_F_RX)) {
			seq_printf(seq, " %3d | %3d %3d | %d | %d | %10d | %-*s\n",
				   q->num, q->mon.hi_thresh,
				   q->mon.lo_thresh, q->mon.log,
				   q->mon.state, q->mon.ncnt,
				   (int)sizeof(q->name), q->name);
		}
	}
	seq_printf(seq, "---------------------------------------------------------------------------------------------------\n\n");
	pr_debug("<--\n");
	return 0;
}

static int dqm_proc_cmd_reset_stats(void *data, int argc, char *argv[])
{
	int i;
	struct dqmdev *qdev;
	struct dqm *q;

	rcu_read_lock();
	if (data) {
		qdev = (struct dqmdev *) data;
		for (i = 0; i < qdev->q_count; i++) {
			q = &qdev->dqm[i];
			memset(&q->stats, 0, sizeof(q->stats));
		}
	} else {
		list_for_each_entry_rcu(qdev, &dqmdevs, list) {
			for (i = 0; i < qdev->q_count; i++) {
				q = &qdev->dqm[i];
				memset(&q->stats, 0, sizeof(q->stats));
			}
		}
	}
	rcu_read_unlock();

	return 0;
}

static int dqm_mib_proc_open(struct inode *inode, struct file *file)
{
	int status;

	pr_debug("-->\n");
	status = seq_open(file, &mib_seq_ops);
	if (!status) {
		struct seq_file *sf = file->private_data;

		sf->private = PDE_DATA(inode);
	}
	pr_debug("<--\n");
	return status;
}

static int dqm_miball_proc_open(struct inode *inode, struct file *file)
{
	int status;

	pr_debug("-->\n");
	status = seq_open(file, &miball_seq_ops);
	if (!status) {
		struct seq_file *sf = file->private_data;

		sf->private = PDE_DATA(inode);
	}
	pr_debug("<--\n");
	return status;
}

static void *dqm_mib_proc_start(struct seq_file *seq, loff_t *pos)
{
	struct dqmdev *qdev = NULL;
	struct dqmdev *tmp;
	loff_t off = 0;

	pr_debug("-->\n");

	if (seq->private) {
		if (off++ == *pos)
			return seq->private;
		return NULL;
	}

	rcu_read_lock();
	list_for_each_entry_rcu(tmp, &dqmdevs, list) {
		if (off++ == *pos) {
			qdev = tmp;
			break;
		}
	}
	rcu_read_unlock();

	pr_debug("<--\n");
	return qdev;
}

static void dqm_mib_proc_stop(struct seq_file *seq, void *v)
{
	pr_debug("-->\n");
	pr_debug("<--\n");
}

static void *dqm_mib_proc_next(struct seq_file *seq, void *v,
			       loff_t *pos)
{
	struct dqmdev *qdev = NULL;
	struct dqmdev *cur = v;
	struct list_head *next;

	pr_debug("-->\n");
	if (seq->private)
		return NULL;
	(*pos)++;
	rcu_read_lock();
	next = cur->list.next;
	if (next != &dqmdevs)
		qdev = container_of(next, struct dqmdev, list);
	rcu_read_unlock();
	pr_debug("<--\n");
	return qdev;
}

static int dqm_mib_proc_show(struct seq_file *seq, void *v)
{
	struct dqmdev *qdev = v;

	pr_debug("-->\n");

	dqm_show_total(seq, qdev);
	dqm_show(seq, qdev, 0);

	pr_debug("<--\n");
	return 0;
}

static int dqm_miball_proc_show(struct seq_file *seq, void *v)
{
	struct dqmdev *qdev = v;

	pr_debug("-->\n");

	dqm_show_total(seq, qdev);
	//dqm_show(seq, qdev, 1);

	pr_debug("<--\n");
	return 0;
}
