 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/

#ifndef _ETHSW_PRIV_H_
#define _ETHSW_PRIV_H_

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/phy.h>
#include <linux/spi/spi.h>
#include <linux/workqueue.h>
#include <linux/regulator/consumer.h>

#define ETHSW_53134_SWITCH 0xA750  /* 3384mwvg */
#define ETHSW_53134_SWITCH_ALT1 0x5035  /* 3384mwvg */
#define ETHSW_53134_SWITCH_ALT2 0x5075  /* 3384mwvg */
#define ETHSW_53125_SWITCH 0x53125 /* 3384wvg, 33843mwvg */
#define ETHSW_53124_SWITCH ETHSW_53125_SWITCH
#define ETHSW_53115_SWITCH 0x53115

#define BCHP_JTAG_OTP_UB_GENERAL_STATUS_5 0x3890128
#define BCHP_JTAG_OTP_UB_GENERAL_STATUS_5_FAB_ID_MASK 0x0000000f
#define BCHP_JTAG_OTP_UB_GENERAL_STATUS_5_FAB_ID_SHIFT 15
#define FAB_ID_TSMC_15 0
#define FAB_ID_TSMC_16 2
#define FAB_ID_SMIC    3
/*
 * Build Options
 */

#define ETHSW_FLOW_CONTROL_ENABLED


enum {
	INTERNAL_SW = 0,
	EXTERNAL_SPI_SW
};
/* support for proc file system as debug interface  */
#define PROC_IF_SUPPORT         1

/* support char devices as test interface */
#define CDEV_IF_SUPPORT         0

/* support virtual network device as test interface */
#define VNET_IF_SUPPORT         0

#define LOG_EMERG(...)   pr_emerg(LOG_MODULE ": " __VA_ARGS__)
#define LOG_ALERT(...)   pr_alert(LOG_MODULE ": " __VA_ARGS__)
#define LOG_CRIT(...)    pr_crit(LOG_MODULE ": " __VA_ARGS__)
#define LOG_ERR(...)     pr_err(LOG_MODULE ": " __VA_ARGS__)
#define LOG_WARNING(...) pr_warn(LOG_MODULE ": " __VA_ARGS__)
#define LOG_NOTICE(...)  pr_notice(LOG_MODULE ": " __VA_ARGS__)
#define LOG_INFO(...)    pr_info(LOG_MODULE ": " __VA_ARGS__)

#define LOG_DEBUG(...)   pr_debug(LOG_MODULE ": " __VA_ARGS__)
#define FUNC_ENTER()     LOG_DEBUG(" ---->\n")
#define FUNC_LEAVE()     LOG_DEBUG(" <----\n")

#define VALIDATE_SWDEV(ptr) do {\
	if (!ptr) {\
		LOG_ERR("%s Ethsw switch device not initialized\n",__FUNCTION__);\
		return(-EINVAL);\
	}\
} while (0)

#define VALIDATE_ETH_PWR_STATUS(ptr) do {\
	if (!ptr->ethsw_pwr_status) {\
		return(-EIO);\
	}\
} while (0)

#define VALIDATE_SWCORE(ptr) do {\
	if (!ptr->swcore) {\
		LOG_ERR("%s Ethsw switch core not initialized\n",__FUNCTION__);\
		return(-EINVAL);\
	}\
} while (0)

/*
 * Constant Defines
 */

/* use pseudo PHY to access switch core registers */
#define PSEUDO_PHY_ACCESS       1
#define PSEUDO_PHY_ID           30

#define PORT_INT_LINK_UP        0x01
#define PORT_INT_LINK_DN        0x02
#define PORT_INT_ENERGY_ON      0x04
#define PORT_INT_ENERGY_OFF     0x08
#define PORT_INT_GPHY           0x10

/*
 * Macros
 */

/* short-hand notation for error checking */
#define CHECK_EXIT(x) if ((ret = (x))) goto EXIT
#define CHECK_ERROR(x) if ((ret = (x))) goto ERROR

/* Macros for core register access */
#define CREG_RD(PAGE, CREG, DATA)					\
	ethsw_creg_read(						\
		swdev,							\
		PAGE,							\
		CREG,							\
		(u8 *)DATA,						\
		CREG##_SIZE)

#define CREG_WR(PAGE, CREG, DATA)					\
	ethsw_creg_write(						\
		swdev,							\
		PAGE,							\
		CREG,							\
		(u8 *)DATA,						\
		CREG##_SIZE)

#define CREG_IDX_RD(PAGE, CREG, IDX, DATA)				\
	ethsw_creg_read(						\
		swdev,							\
		PAGE,							\
		CREG + CREG##_STEP * IDX,				\
		(u8 *)DATA,						\
		CREG##_SIZE)

#define CREG_IDX_WR(PAGE, CREG, IDX, DATA)				\
	ethsw_creg_write(						\
		swdev,							\
		PAGE,							\
		CREG + CREG##_STEP * IDX,				\
		(u8 *)DATA,						\
		CREG##_SIZE)

#define CREG_FLD_MSK(CREG, FLD)						\
	(CREG##_##FLD##_MASK)

#define CREG_FLD_SHF(CREG, FLD, FVAL)					\
	((FVAL) << CREG##_##FLD##_SHIFT)

#define CREG_FLD_CLR(CREG, FLD, RVAL)					\
	((RVAL) & ~CREG##_##FLD##_MASK)

#define CREG_FLD_GET(CREG, FLD, RVAL)					\
	(((RVAL) & CREG##_##FLD##_MASK) >>				\
	 CREG##_##FLD##_SHIFT)

#define CREG_FLD_SET(CREG, FLD, RVAL, FVAL)				\
	(((RVAL) & ~CREG##_##FLD##_MASK) |				\
	 ((FVAL) << CREG##_##FLD##_SHIFT))

#define CREG_FLD_IDX_MSK(CREG, FLD, IDX)				\
	(CREG##_##FLD##_MASK <<						\
	 CREG##_##FLD##_BSTEP * IDX)

#define CREG_FLD_IDX_SHF(CREG, FLD, IDX, FVAL)				\
	((FVAL) << (CREG##_##FLD##_SHIFT +				\
	CREG##_##FLD##_BSTEP * IDX))

#define CREG_FLD_IDX_CLR(CREG, FLD, IDX, RVAL)				\
	((RVAL) & ~(CREG##_##FLD##_MASK <<				\
	CREG##_##FLD##_BSTEP * IDX))

#define CREG_FLD_IDX_GET(CREG, FLD, IDX, RVAL)				\
	(((RVAL) & (CREG##_##FLD##_MASK <<				\
	CREG##_##FLD##_BSTEP * IDX)) >>					\
	(CREG##_##FLD##_SHIFT +						\
	CREG##_##FLD##_BSTEP * IDX))

#define CREG_FLD_IDX_SET(CREG, FLD, IDX, RVAL, FVAL)			\
	(((RVAL) & ~(CREG##_##FLD##_MASK <<				\
	CREG##_##FLD##_BSTEP * IDX)) |					\
	((FVAL) << (CREG##_##FLD##_SHIFT +				\
	CREG##_##FLD##_BSTEP * IDX)))


/*
 * Inline Functions
 */

#ifdef __BIG_ENDIAN

/* 00:01:02:03:04:05 -> (MSB) 00010203 04050000 (LSB) */
inline static void pack_mac_to_u48(u64 *val, u8 *mac)
{
	*((u32 *)val    ) = *(u32 *)(mac    );
	*((u16 *)val + 2) = *(u16 *)(mac + 4);
	*((u16 *)val + 3) = 0;
}

/* (MSB) 00010203 04050000 (LSB) -> 00:01:02:03:04:05 */
inline static void unpack_u48_to_mac(u8 *mac, u64 val)
{
	*(u32 *)(mac    ) = *((u32 *)&val    );
	*(u16 *)(mac + 4) = *((u32 *)&val + 2);
}

/* 00:01:02:03:04:05 -> (MSB) 00000001 02030405 (LSB) */
inline static void pack_mac_to_u64(u64 *val, u8 *mac)
{
	*((u16 *)val    ) = 0;
	*((u16 *)val + 1) = *(u16 *)(mac    );
	*((u32 *)val + 1) = *(u32 *)(mac + 2);
}

/* (MSB) 00000001 02030405 (LSB) -> 00:01:02:03:04:05 */
inline static void unpack_u64_to_mac(u8 *mac, u64 val)
{
	*(u16 *)(mac    ) = *((u16 *)&val + 1);
	*(u32 *)(mac + 2) = *((u32 *)&val + 1);
}

/* 00:01:02:03:04:05 VHVL -> (MSB) VHVL0001 02030405 (LSB) */
inline static void pack_macvid_to_u64(u64 *val, u8 *mac, u16 vid)
{
	pack_mac_to_u64(val, mac);
	*((u16 *)val) = vid;
}

#else /* __BIG_ENDIAN */

/* 00:01:02:03:04:05 -> (LSB) 05040302 01000000 (MSB) */
inline static void pack_mac_to_u48(u64 *val, u8 *mac)
{
	*((u32 *)val    ) = be32_to_cpu(*(u32 *)(mac + 2));
	*((u16 *)val + 2) = be16_to_cpu(*(u16 *)(mac    ));
	*((u16 *)val + 3) = 0;
}

/* (LSB) 05040302 01000000 (MSB) -> 00:01:02:03:04:05 */
inline static void unpack_u48_to_mac(u8 *mac, u64 val)
{
	*(u16 *)(mac    ) = cpu_to_be16(*((u16 *)&val + 2));
	*(u32 *)(mac + 2) = cpu_to_be32(*((u32 *)&val    ));
}

/* 00:01:02:03:04:05 -> (LSB) 05040302 01000000 (MSB) */
#define pack_mac_to_u64 pack_mac_to_u48

/* (LSB) 05040302 01000000 (MSB) -> 00:01:02:03:04:05 */
#define unpack_u64_to_mac unpack_u48_to_mac

/* 00:01:02:03:04:05 VLVH -> (LSB) 05040302 0100VLVH (MSB) */
inline static void pack_macvid_to_u64(u64 *val, u8 *mac, u16 vid)
{
	pack_mac_to_u64(val, mac);
	*((u16 *)val + 3) = vid;
}

#endif /* __BIG_ENDIAN */


/*
 * Data Structures
 */

enum {
	INV_PORT = 0,
	PHY_PORT,
	MII_PORT,
	IMP_PORT
};

enum {
	LED_ON = 0,
	LED_OFF,
	MAX_LED_STATE
};

enum {
	ETHSW_PHY_STOP = 0,
	ETHSW_PHY_START,
	MAX_ETHSW_PHY_STATE
};

/* ethsw port control block */
struct ethsw_port {
	int  type;
	const char *name;

	ethsw_port_cb_t cb;
	void *cb_priv;

	/* admin state */
	bool enable;

	/* admin config */
	struct ethsw_port_config config;
	struct ethsw_power_config power_config;

	/* operational status */
	struct ethsw_port_status status;
	struct ethsw_power_status power_status;

	/* PHY port */
	struct {
		int id;
		struct device_node *node;
		struct phy_device *dev;
		phy_interface_t mode;
		atomic_t state;
	} phy;

	/* type specific parameters */
	union {
		/* MII port */
		struct {
			int rgmii;
			int speed;
		} mii;

		/* IMP port */
		struct {
			int speed;
		} imp;
	};

	struct regulator *xcvr_power;
	struct gpio_desc *xcvr_reset;
	struct pinctrl *xcvr_pins;
	struct pinctrl_state *xcvr_pins_state[2];
	unsigned int xcvr_reset_duration;
	unsigned int afe_pwrdn_xcvr_power;
	unsigned int afe_pwrdn_xcvr_core;
	unsigned int afe_pwrdn_xcvr_reset_duration;
	unsigned int afe_pwrdn_xcvr_pins;
	unsigned int afe_pwrdn_xcvr_reset;

	int (*ethsw_phy_set_afe_power_state)(struct ethsw_device *swdev, int port, bool afe_pwrdn);
	bool (*ethsw_phy_get_afe_power_state)(struct ethsw_device *swdev, int port);

	int (*ethsw_phy_set_power_state)(struct ethsw_device *swdev, int port, bool pwrdn);
	bool (*ethsw_phy_get_power_state)(struct ethsw_device *swdev, int port);

	unsigned int qphy_ctrl_reg;
};

/* ethsw device control block */
struct ethsw_device {
	struct spi_device *sdev;
	u32 spi_freq;
	struct delayed_work arl_aging_work;
	struct delayed_work link_status_work;
	struct regulator *vreg;

	/* clk */
	struct clk *clk;

	/* platform device struct */
	struct platform_device *pdev;
	u32 high_speed_imp;

	/* device parameters */
	const char *name;
	u32 chip_id;

	int pm_count;
	const int *pm_params;
	int pi_count;
	const int *pi_params;

	struct notifier_block nb;

	u32 led_map;
	u32 unimac_inf_start;
	u32 unimac_inf_end;

	/* IRQ numbers */
	u32 irq[ETHSW_IRQ_MAX];

	/* L2 IRQ status */
	u32 irq_status[ETHSW_IRQ_MAX];

	/* memory region */
	struct resource *mem_region;

	/* register base */
	volatile u32 *reg_base;
	volatile u32 *periph_base;
	volatile u32 *jtag_otp_base;
	void __iomem *sun_top_ctrl;

	int phy_min_addr;

	/* port control blocks */
	struct ethsw_port port[ETHSW_PORT_MAX];
	u32 phy_port_mask;
	u32 mii_port_mask;
	u32 imp_port_mask;
   u8  imp_port_count;

	/* core control block */
	struct ethsw_core *swcore;

	/* interrupt control block */
	struct ethsw_int *swint;

#if PROC_IF_SUPPORT
	/* ethsw proc control block */
	struct ethsw_proc *swproc;
#endif

#if CDEV_IF_SUPPORT
	/* ethsw cdev control block */
	struct ethsw_cdev *swcdev;
#endif

#if VNET_IF_SUPPORT
	/* ethsw VNET control block */
	struct ethsw_vnet *swvnet;
#endif

	struct ethsw_arl_cache *arlcache;

	u32 eee_support;
	bool ethsw_pwr_status;
	u32 vlan_default_mode;
	/* only used for 3384 w/external switch */
	bool vlan_ipv6_mc_nf_mode_enabled;
	u32 subid_port_offset;
	int (*ethsw_creg_read)(struct ethsw_device *swdev,
						   int page_num, int reg_addr, u8 *data, int size);

	int (*ethsw_creg_write)(struct ethsw_device *swdev,
							int page_num, int reg_addr, u8 *data, int size);

	int (*ethsw_phy_board_init)(struct ethsw_device *swdev);
	int (*ethsw_led_board_init)(struct ethsw_device *swdev);
	int (*ethsw_int_board_init)(struct ethsw_device *swdev);


	int (*ethsw_switch_set_port_stp_state_specific)(struct ethsw_device *swdev, int port_id, unsigned int state);
	int (*ethsw_int_board_isr)(struct ethsw_device *swdev, int irq);

	int (*ethsw_port_idx_to_phy_id)(struct ethsw_device *swdev, int port_idx);
	int (*ethsw_port_idx_to_port_id)(struct ethsw_device *swdev, int port_idx);
	int (*ethsw_port_id_to_port_idx)(struct ethsw_device *swdev, int port_id);

	void (*ethsw_led_board_exit)(struct ethsw_device *swdev);
	int (*ethsw_led_powercntrl)(struct ethsw_device *swdev, int port_id, bool state);
	void (*ethsw_phy_board_exit)(struct ethsw_device *swdev);

	int (*ethsw_int_board_proc)(struct ethsw_device *swdev);

	int (*ethsw_swdev_board_init)(struct ethsw_device *swdev);
	void (*ethsw_swdev_board_exit)(struct ethsw_device *swdev);

	void (*ethsw_int_board_exit)(struct ethsw_device *swdev);
	int (*ethsw_port_status_get)(struct ethsw_device *swdev, int port_id, struct ethsw_port_status *port_status);

	int (*ethsw_low_pwr_mode)(struct ethsw_device *swdev);
	int (*ethsw_normal_pwr_mode)(struct ethsw_device *swdev);
};

/* ethsw driver control block */
struct ethsw_driver {
	/* spi driver struct */
	struct spi_driver *sdrv;
	/* platform driver struct */
	struct platform_driver *pdrv;

	/* ethsw device control blocks */
	int swdev_count;
	struct ethsw_device *swdev[ETHSW_MAX_SWITCHS];
};



/*
 * Global Variables
 */

/* For simplicity for now, it is assumed that there is only ONE ethsw
 * device in the system. This could change in the future.
 */
extern struct ethsw_driver *ethsw_drv[];
extern struct ethsw_device *ethsw_dev[];
extern struct ethsw_device * subid_to_swdev_table[];


inline static struct ethsw_device * subid_to_swdev(int port_id)
{
	//struct ethsw_device *swdev = subid_to_swdev_table[port_id];
	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		return NULL;
	}
	if (subid_to_swdev_table[port_id])
		return subid_to_swdev_table[port_id];
	else
		return ethsw_dev[INTERNAL_SW];
}


/*
 * Function Declarations
 */

int  ethsw_int_init(struct ethsw_device *swdev);
void ethsw_int_exit(struct ethsw_device *swdev);

int  ethsw_phy_init(struct ethsw_device *swdev);
void ethsw_phy_exit(struct ethsw_device *swdev);

int  ethsw_core_init(struct ethsw_device *swdev);
void ethsw_core_exit(struct ethsw_device *swdev);

int  ethsw_proc_init(struct ethsw_device *swdev);
void ethsw_proc_exit(struct ethsw_device *swdev);

int  ethsw_cdev_init(struct ethsw_device *swdev);
void ethsw_cdev_exit(struct ethsw_device *swdev);

int  ethsw_vnet_init(struct ethsw_device *swdev);
void ethsw_vnet_exit(struct ethsw_device *swdev);

int ethsw_phy_port_enable(
	struct ethsw_device *swdev,
	int port_id);

int ethsw_phy_port_disable(
	struct ethsw_device *swdev,
	int port_id);

int ethsw_phy_port_restart(
	struct ethsw_device *swdev,
	int port_id);

int ethsw_phy_port_config_set(
	struct ethsw_device *swdev,
	int port_id,
	struct ethsw_port_config *port_config);

int ethsw_phy_port_power_config_set(
	struct ethsw_device *swdev,
	int port_id,
	struct ethsw_power_config *power_config);

bool ethsw_phy_port_energy_det(
	struct ethsw_device *swdev,
	int port_id);

bool ethsw_phy_port_dll_down(
	struct ethsw_device *swdev,
	int port_id);

bool ethsw_phy_port_eee_capable(
	struct ethsw_device *swdev,
	int port_id);

bool ethsw_switch_port_link_up(struct ethsw_device *swdev,int port_id);
bool ethsw_switch_port_full_duplex(struct ethsw_device *swdev,int port_id);
int  ethsw_switch_port_speed(int port_id);

int ethsw_switch_port_eee_config_set(
	int port_id,
	struct ethsw_power_config *power_config);

#if defined(CONFIG_BCM_MBOX)
int ethsw_phy_mbox_cb_sf2(struct notifier_block *this,
		      unsigned long event, void *ptr);
#endif
int ethsw_phy_mbox_cb_531xx(struct notifier_block *this,
		      unsigned long event, void *ptr);


int ethsw_phy_set_afe_power_state(
	struct ethsw_device *swdev,
	int port_id,
	bool afe_pwrdn);

bool ethsw_phy_get_afe_power_state(struct ethsw_device *swdev,
								   int port_id);

bool ethsw_switch_port_eee_assert(int port_id);
bool ethsw_switch_port_eee_indicate(int port_id);

int ethsw_alt_ethwan(int imp_port, int lan_port, bool enable);
int ethsw_alt_ethwan_core(struct ethsw_device *swdev, int imp_port, int lan_port, bool enable);
int ethsw_alt_ethwan_sf2(struct ethsw_device *swdev,int imp_port, int lan_port, bool enable);
int ethsw_alt_ethwan_53xxx(struct ethsw_device *swdev,int imp_port, int lan_port, bool enable);

int ethsw_port_isolate(int imp_port, int lan_port, bool enable);
int ethsw_port_isolate_core(struct ethsw_device *swdev, int imp_port, int lan_port, bool enable);

bool hash_collision_fix_enabled(void);
int hash_collision_fix_enable(struct ethsw_device *swdev, bool enable);

bool reserved_multicast_enabled(void);
int reserved_multicast_enable(struct ethsw_device *swdev, bool enable);

/*
 * Platform Dependent Descalrations
 */

int  ethsw_swdev_board_init(struct ethsw_device *swdev);
void ethsw_swdev_board_exit(struct ethsw_device *swdev);

int  ethsw_int_board_init(struct ethsw_device *swdev);
void ethsw_int_board_exit(struct ethsw_device *swdev);

int  ethsw_phy_board_init(struct ethsw_device *swdev);
void ethsw_phy_board_exit(struct ethsw_device *swdev);

int  ethsw_led_board_init(struct ethsw_device *swdev);
void ethsw_led_board_exit(struct ethsw_device *swdev);
int ethsw_led_powercntrl(struct ethsw_device *swdev, int port_id, bool state);

int   ethsw_swdev_board_init_531xx(struct ethsw_device *swdev);
void  ethsw_swdev_board_exit_531xx(struct ethsw_device *swdev);

int   ethsw_phy_board_init_531xx(struct ethsw_device *swdev);

int   ethsw_swdev_board_init_sf2(struct ethsw_device *swdev);
void  ethsw_swdev_board_exit_sf2(struct ethsw_device *swdev);

int   ethsw_int_board_init_sf2(struct ethsw_device *swdev);
void  ethsw_int_board_exit_sf2(struct ethsw_device *swdev);

int   ethsw_phy_board_init_sf2(struct ethsw_device *swdev);
void  ethsw_phy_board_exit_sf2(struct ethsw_device *swdev);

int   ethsw_led_board_init_sf2(struct ethsw_device *swdev);
void  ethsw_led_board_exit_sf2(struct ethsw_device *swdev);

int ethsw_core_normal_power_mode_sf2(struct ethsw_device *swdev);
int ethsw_core_low_power_mode_sf2(struct ethsw_device *swdev);
int ethsw_led_powercntrl_sf2(struct ethsw_device *swdev, int port_id, bool state);
int ethsw_led_powercntrl_531xx(struct ethsw_device *swdev, int port_id, bool state);

#ifdef ETHSW_FLOW_CONTROL_ENABLED
#define ETHSW_ACB_QUEUES_PER_PORT 8
int ethsw_acb_enable(struct ethsw_device *swdev);
int ethsw_get_tc2cos_mapping(struct ethsw_device *swdev, u32 port,
			     u32 tc, u32 *cos);
int ethsw_set_tc2cos_mapping(struct ethsw_device *swdev, u32 port,
			     u32 tc, u32 cos);
int ethsw_set_acb_threshold(struct ethsw_device *swdev, u32 port, u32 qid,
			    u32 threshold);
int ethsw_get_acb_threshold(struct ethsw_device *swdev, u32 port, u32 qid,
			    u32 *threshold);
#endif

int ethsw_creg_read(
	struct ethsw_device *swdev,
	int page_num, int reg_addr, u8 *data, int size);

int ethsw_creg_write(
	struct ethsw_device *swdev,
	int page_num, int reg_addr, u8 *data, int size);


int ethsw_creg_read_sf2(
	struct ethsw_device *swdev,
	int page_num, int reg_addr, u8 *data, int size);

int ethsw_creg_write_sf2(
	struct ethsw_device *swdev,
	int page_num, int reg_addr, u8 *data, int size);


int ethsw_creg_read_spi(
	struct ethsw_device *swdev,
	int page_num, int reg_addr, u8 *data, int size);

int ethsw_creg_write_spi(
	struct ethsw_device *swdev,
	int page_num, int reg_addr, u8 *data, int size);

int ethsw_int_board_isr(struct ethsw_device *swdev, int irq);
int ethsw_int_board_proc(struct ethsw_device *swdev);

int ethsw_int_port_enable(
	struct ethsw_device *swdev,
	int port_id, u32 int_map);

int ethsw_int_port_disable(
	struct ethsw_device *swdev,
	int port_id, u32 int_map);

int ethsw_switch_set_port_stp_state_specific(
	struct ethsw_device *swdev,
	int port_id, u32 state);


int ethsw_int_board_isr_sf2(struct ethsw_device *swdev, int irq);
int ethsw_int_board_proc_sf2(struct ethsw_device *swdev);

int ethsw_int_port_enable_sf2(
	struct ethsw_device *swdev,
	int port_id, u32 int_map);

int ethsw_int_port_disable_sf2(
	struct ethsw_device *swdev,
	int port_id, u32 int_map);

int ethsw_switch_set_port_stp_state_specific_sf2(
	struct ethsw_device *swdev,
	int port_id, u32 state);


int ethsw_arl_cache_table_find(u8 *mac, u16 vid, u16 *port);
int __ethsw_arl_cache_table_find(u8 *mac, u16 vid, u16 *port);
int ethsw_arl_cache_table_add(u8 *mac, u16 vid, u16 port);
int ethsw_arl_cache_table_delete(u8 *mac, u16 vid);
int ethsw_arl_cache_table_init(struct ethsw_device *swdev);
void ethsw_arl_cache_table_fini(struct ethsw_device *swdev);
int __ethsw_arl_cache_table_delete(u8 *mac, u16 vid, u16 port);
void ethsw_arl_cache_get_counter(struct ethsw_device *swdev, unsigned int *arl_cache_miss, unsigned int *arl_cache_faked);

int ethsw_arl_switch_table_find(struct ethsw_device *swdev, u8 *mac, u16 vid, u16 *port);
int ethsw_arl_switch_table_add(struct ethsw_device *swdev, u8 *mac, u16 vid, u16 port);
int ethsw_arl_switch_table_delete(struct ethsw_device *swdev, u8 *mac, u16 vid);
void ethsw_arl_cache_table_set_aging_poll(struct ethsw_device *swdev, int seconds);
void ethsw_arl_cache_table_set_aging_period(struct ethsw_device *swdev, int seconds);
void ethsw_set_link_poll(struct ethsw_device *swdev, int seconds);

u8 ethsw_check_link_status(struct ethsw_device *swdev);

void PinMuxSelRange(const int *params, int count);
void PinMuxSel(unsigned mux, unsigned value);
int PinMuxVal(unsigned mux );

int ethsw_spi_write(u8 page, u8 reg_addr, void *reg_val, u8 reg_size);
int ethsw_spi_read(u8 page, u8 reg_addr, void *reg_val, u8 reg_size);
void ethsw_init_invalid_page_map(void);

static inline u32 jtag_otp_reg_read(struct ethsw_device *swdev, u32 offset)
{
	u8 *addr = ((u8 *)swdev->jtag_otp_base) + offset - BCHP_JTAG_OTP_UB_REG_START;
	return __raw_readl((u32 *)addr);
}

static inline u32 ethsw_reg_read(struct ethsw_device *swdev, u32 offset)
{
	u8 *addr = ((u8 *)swdev->reg_base) + offset - BCHP_SWITCH_CORE_REG_START;
	return __raw_readl((u32 *)addr);
}

static inline void ethsw_reg_write(struct ethsw_device *swdev, u32 offset, u32 val)
{
	u8 *addr = ((u8 *)swdev->reg_base) + offset - BCHP_SWITCH_CORE_REG_START;
	__raw_writel(val, (u32 *)addr);
}

static inline void ethsw_reg_set_field(struct ethsw_device *swdev,
				       u32 reg_offset,
				       u32 mask,
				       u32 shift,
				       u32 value)
{
	u32 field = value << shift;
	u32 data = ethsw_reg_read(swdev, reg_offset);

	data &= ~mask;
	data |= field;
	ethsw_reg_write(swdev, reg_offset, data);
}


static inline u32 ethsw_reg_get_field(struct ethsw_device *swdev,
				      u32 reg_offset,
				      u32 mask,
				      u32 shift)
{
	u32 data = ethsw_reg_read(swdev, reg_offset);
	data &= mask;
	return data >> shift;
}

typedef struct {
	u8  mac[6];
	u16 vid;
	u16 port;
	u8  stat;
	u8  valid;
} ethsw_arl_table_entry;

#define ETHSW_ARL_TBL_LAST_ENTRY 1
#define ETHSW_ARL_TBL_ENTRIES_PER_READ 2

int ethsw_arl_table_get_next(struct ethsw_device *swdev,
			     ethsw_arl_table_entry *tbl_entry);

int ethsw_phy_get_property(struct device *dev, struct ethsw_port *port,
		int port_id);
int ethsw_phy_get_pwr_rst_pin(struct device *dev, struct ethsw_port *port);
void ethsw_phy_port_init_ids(struct ethsw_device *swdev);
int ethsw_port_idx_to_phy_id(struct ethsw_device *swdev, int port_idx);
int ethsw_port_idx_to_port_id(struct ethsw_device *swdev, int port_idx);
int ethsw_port_id_to_port_idx(struct ethsw_device *swdev, int port_id);
bool ethsw_isswitch_53134(int chip_id);

int ethsw_port_idx_to_phy_id_531xx(struct ethsw_device *swdev, int port_idx);
int ethsw_port_idx_to_port_id_531xx(struct ethsw_device *swdev, int port_idx);
int ethsw_port_id_to_port_idx_531xx(struct ethsw_device *swdev, int port_id);


int ethsw_port_idx_to_phy_id_sf2(struct ethsw_device *swdev, int port_idx);
int ethsw_port_idx_to_port_id_sf2(struct ethsw_device *swdev, int port_idx);
int ethsw_port_id_to_port_idx_sf2(struct ethsw_device *swdev, int port_id);

int ethsw_switch_set_port_stp_state(int port_id, unsigned int state);
int ethsw_switch_set_port_stp_state_sf2(int port_id, unsigned int state);

void ethsw_phy_link_cb_sf2(struct net_device *dev);
void ethsw_phy_link_cb_531xx(struct net_device *dev);


int ethsw_port_config_set(
    int port_id,
	struct ethsw_port_config *port_config);

int ethsw_port_power_config_set(
    int port_id,
	struct ethsw_power_config *power_config);

int ethsw_port_disable(
   int port_id);

int ethsw_pwr_down_531xx(struct ethsw_device *swdev);
int ethsw_pwr_up_531xx(struct ethsw_device *swdev);

/* Ethernet switch MTU function */
int ethsw_set_mtu(u16 mtu);
int ethsw_set_phy_state(struct ethsw_port *port, int state);
#endif /* _ETHSW_PRIV_H_ */
