 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/

#ifndef BRCM_MBOX_PRIV_H
#define BRCM_MBOX_PRIV_H

#define DEBUG_FS_DIR "brcm_mbox" /* Where to put debugfs files */
#define BRCM_MBOX_MAX_INST 2 /* Maximum open "instances" */

#ifdef CONFIG_BCM_MBOX_DEBUG
#define MSG_TRACE(level, format, ...) pr_debug(level format, ##__VA_ARGS__);
#define ERR_TRACE(level, format, ...) pr_err(level format, ##__VA_ARGS__);
#define FUNC_TRACE(x) \
	pr_debug(MSG_LEVEL_FUNC_TRACE "%s %s\n", x ? "->" : "<-", __func__);
#define FUNC_TRACEF(x, format, ...) \
	pr_debug(MSG_LEVEL_FUNC_TRACE "%s %s " format, x ? "->" : "<-", \
		__func__, __VA_ARGS__);
#define MSG_TRACEN(...) MSG_TRACE(MSG_LEVEL_NOISE, __VA_ARGS__)
#define MSG_TRACEI(...) MSG_TRACE(MSG_LEVEL_INFO, __VA_ARGS__)
#define MSG_TRACEE(...) ERR_TRACE(MSG_LEVEL_ERROR, __VA_ARGS__)

#define MSG_LEVEL_FUNC_TRACE "[MB:f]"
#define MSG_LEVEL_NOISE "[MB:n]"
#define MSG_LEVEL_INFO "[MB:i]"
#define MSG_LEVEL_ERROR "[MB:e]"
#else   /* #ifdef CONFIG_BCM_MBOX_DEBUG */
#define ERR_TRACE(format, ...) pr_err(format, ##__VA_ARGS__);
#define FUNC_TRACE(x)
#define FUNC_TRACEF(x, format, ...)
#define MSG_TRACEN(...)
#define MSG_TRACEI(...)
#define MSG_TRACEE(...) ERR_TRACE(__VA_ARGS__)
#endif /* #ifdef CONFIG_BCM_MBOX_DEBUG */

/**
* struct brcm_mbox_inst
* @mask Which MBOX's are "new" for this instance
* @wait Wait queue for this instance
*/
struct brcm_mbox_inst {
	u32 mask;
	wait_queue_head_t wait;
};

/**
* struct brcm_mbox_dev_info - Device info
* @pdev: platform device
* @states: MBOX register values
* @inst: Per "consumer" info (wait and mask)
* @inst_mask: Which "consumer" instance is in use
* @state_lock: lock for MBOX value storage
* @down: Waited on being set to true before allowing power down
* @irq: IRQ from device tree
* @mbox_dev: device
* @write_mask: mask of mbox #'s that can be written to (from DT)
* @read_mask: mask of mbox #'s that can be read (from DT)
* @mbox_class: class
* @mbox_cdev: character device
* @reg_status: Register to use for status (depends on STB vs. RG)
* @reg_mask: Register to use for mask (depends on STB vs. RG)
* @mbox: MBOX registers
* @comm: COMM registers
*/
struct brcm_mbox_dev_info {
	struct platform_device *pdev;
	struct brcm_mbox_info states;
	struct brcm_mbox_inst inst[BRCM_MBOX_MAX_INST];
	u8 inst_mask;
	spinlock_t state_lock; /* lock for MBOX value storage */
	bool down;
	int irq;
	dev_t mbox_dev;
	u16 write_mask;
	u32 read_mask;
	bool override_bmu_state;
	struct class *mbox_class;
	struct cdev mbox_cdev;
	void __iomem *reg_status;
	void __iomem *reg_mask;
	void __iomem *mbox;
	void __iomem *comm;
};

#define MBOX_LOCK_OWNER_STB 0x42
#define MBOX_LOCK_OWNER_RG 0x41 /* TBD... for 3390? Doesn't matter? */

#ifdef CONFIG_BCM7145A0
#define MAX_LOCKED_MBOX 3 /* MBOX 0-3 are locked and hard coded in HW */
#else
/* TBD */
#define MAX_LOCKED_MBOX 3 /* MBOX 0-3 are locked and hard coded in HW */
#endif

#ifdef CONFIG_BCM_MBOX_PERF

#define MAX_IRQ_METRIC 200

struct mbox_met_irq {
	unsigned long jiff;
	u32 mbox[6];
};

struct mbox_metric {
	unsigned long start_jiff;
	struct mbox_met_irq irq[MAX_IRQ_METRIC];
};

#define METRIC_START do {metrics.start_jiff = jiffies; } while (0)
#define METRIC_IRQ do {\
	if (irqcnt >= MAX_IRQ_METRIC) \
		continue; \
	metrics.irq[irqcnt].jiff = jiffies; \
	memcpy(metrics.irq[irqcnt].mbox, info->states.mbox, \
	       sizeof(metrics.irq[irqcnt].mbox)); \
	irqcnt++; \
	} while (0)
#else
#define METRIC_START
#define METRIC_IRQ
#endif

#define MBOX(i) (info->mbox + (i<<2))
#define BOX_CPUC_DATA0 MBOX(0)
#define BMUPOWER_STATE_MASK 0x3
#define BMUPOWER_STATE_SHIFT 2
#define BMUDISCHARGE_ENABLE_MASK 1
#define BMUDISCHARGE_ENABLE_SHIFT 0

#define BOX_CPUC_DATA1 MBOX(1)
#define POWER_STATE_MASK 0x3
#define POWER_STATE_SHIFT 4
#define RUN_STATE_MASK 0xF
#define RUN_STATE_SHIFT 0
#define CM_RGPING_REQ_MASK 1
#define CM_RGPING_REQ_SHIFT 8
#define CM_RGPING_ACK_MASK 1
#define CM_RGPING_ACK_SHIFT 9
#define CM_RGSHUTDOWN_ACK_MASK 1
#define CM_RGSHUTDOWN_ACK_SHIFT 10
#define CM_STBPING_REQ_MASK 1
#define CM_STBPING_REQ_SHIFT 16
#define CM_STBPING_ACK_MASK 1
#define CM_STBPING_ACK_SHIFT 17
#define CM_STBSHUTDOWN_ACK_MASK 1
#define CM_STBSHUTDOWN_ACK_SHIFT 18
#define CM_STBRESTART_CM_MASK 1
#define CM_STBRESTART_CM_SHIFT 19
#define CM_SVMPING_REQ_MASK 1
#define CM_SVMPING_REQ_SHIFT 24
#define CM_SVMPING_ACK_MASK 1
#define CM_SVMPING_ACK_SHIFT 25
#define CM_SVMSHUTDOWN_ACK_MASK 1
#define CM_SVMSHUTDOWN_ACK_SHIFT 26
#define CM_SVMRESTART_CM_MASK 1
#define CM_SVMRESTART_CM_SHIFT 27

#define BOX_CPUC_DATA2 MBOX(2)
#define RG_CMPING_REQ_MASK 1
#define RG_CMPING_REQ_SHIFT 8
#define RG_CMPING_ACK_MASK 1
#define RG_CMPING_ACK_SHIFT 9
#define RG_CMSHUTDOWN_ACK_MASK 1
#define RG_CMSHUTDOWN_ACK_SHIFT 10
#define RG_STBPING_REQ_MASK 1
#define RG_STBPING_REQ_SHIFT 16
#define RG_STBPING_ACK_MASK 1
#define RG_STBPING_ACK_SHIFT 17
#define RG_STBSHUTDOWN_ACK_MASK 1
#define RG_STBSHUTDOWN_ACK_SHIFT 18
#define RG_STBRESTART_RG_MASK 1
#define RG_STBRESTART_RG_SHIFT 19
#define RG_SVMPING_REQ_MASK 1
#define RG_SVMPING_REQ_SHIFT 24
#define RG_SVMPING_ACK_MASK 1
#define RG_SVMPING_ACK_SHIFT 25
#define RG_SVMSHUTDOWN_ACK_MASK 1
#define RG_SVMSHUTDOWN_ACK_SHIFT 26
#define RG_SVMRESTART_RG_MASK 1
#define RG_SVMRESTART_RG_SHIFT 27

#define BOX_CPUC_DATA3 MBOX(3)
#define STB_CMPING_REQ_MASK 1
#define STB_CMPING_REQ_SHIFT 8
#define STB_CMPING_ACK_MASK 1
#define STB_CMPING_ACK_SHIFT 9
#define STB_CMSHUTDOWN_ACK_MASK 1
#define STB_CMSHUTDOWN_ACK_SHIFT 10
#define STB_RGPING_REQ_MASK 1
#define STB_RGPING_REQ_SHIFT 16
#define STB_RGPING_ACK_MASK 1
#define STB_RGPING_ACK_SHIFT 17
#define STB_RGSHUTDOWN_ACK_MASK 1
#define STB_RGSHUTDOWN_ACK_SHIFT 18
#define STB_SVMPING_REQ_MASK 1
#define STB_SVMPING_REQ_SHIFT 24
#define STB_SVMPING_ACK_MASK 1
#define STB_SVMPING_ACK_SHIFT 25
#define STB_SVMSHUTDOWN_ACK_MASK 1
#define STB_SVMSHUTDOWN_ACK_SHIFT 26
#define STB_SVMRESTART_STB_MASK 1
#define STB_SVMRESTART_STB_SHIFT 27

#define BOX_CPUC_DATA4 MBOX(4)
#define BOX_CPUC_DATA5 MBOX(5)
#define BOX_CPUC_DATA6 MBOX(6)

#define MBOX_CFG(i) (0x40 + MBOX(i))
#define MBOX_CPUC_CFG6 MBOX(22)
#define MBOX_CPUC_CFG7 MBOX(23)
#define MBOX_LOCK_MASK 1
#define MBOX_LOCK_SHIFT 9
#define MBOX_CLR_ON_READ_MASK 1
#define MBOX_CLR_ON_READ_SHIFT 8
#ifdef CONFIG_BCM7145A0 /* TBD.. split to separate h files? */
#define MBOX_LOCK_SRC_MASK 0xFF
#else
#define MBOX_LOCK_SRC_MASK 0x7F
#endif
#define MBOX_LOCK_SRC_SHIFT 0

#define CPU_COMM_REGS_CPUC_HV_MODE (info->comm+0x28)
#define HV_MODE 1

/* Used to determine which IRQ/MASK pair to use */
#ifdef CONFIG_BCM7145
/* 7145 has L1 0 with L2 1, and L1 1 with L2 0 (relative) */
#define CPUC_IRQ_ARM0 161 /* device tree + 32 */
#define CPUC_IRQ_ARM1 160 /* device tree + 32 */
#else
#define CPUC_IRQ_ARM0 160 /* device tree + 32 */
#define CPUC_IRQ_ARM1 161 /* device tree + 32 */
#endif
#define CPUC_IRQ_ARM2 187 /* device tree + 32 */
#define CPUC_IRQ_ARM3 188 /* device tree + 32 */

#define CPU_COMM_REGS_CPUC_L2_IRQ_A15R_MASK (info->comm+0x50)
#define CPU_COMM_REGS_CPUC_L2_IRQ_ARM0_MASK CPU_COMM_REGS_CPUC_L2_IRQ_A15R_MASK
#define CPU_COMM_REGS_CPUC_L2_IRQ_A15S_MASK (info->comm+0x58)
#define CPU_COMM_REGS_CPUC_L2_IRQ_ARM1_MASK CPU_COMM_REGS_CPUC_L2_IRQ_A15S_MASK
#define CPU_COMM_REGS_CPUC_L2_IRQ_ARM2_MASK (info->comm+0x60)
#define CPU_COMM_REGS_CPUC_L2_IRQ_ARM3_MASK (info->comm+0x68)
#define MBOX0_IRQ_MASK_MASK 1
#define MBOX0_IRQ_MASK_SHIFT 0
#define MBOX1_IRQ_MASK_MASK 1
#define MBOX1_IRQ_MASK_SHIFT 1
#define MBOX2_IRQ_MASK_MASK 1
#define MBOX2_IRQ_MASK_SHIFT 2
#define MBOX3_IRQ_MASK_MASK 1
#define MBOX3_IRQ_MASK_SHIFT 3
#define MBOX4_IRQ_MASK_MASK 1
#define MBOX4_IRQ_MASK_SHIFT 4
#define MBOX5_IRQ_MASK_MASK 1
#define MBOX5_IRQ_MASK_SHIFT 5

/* In case register layout changes */
#define SET_CPU_COMM_MBOX_READ_MASK(x) (0x0000FFFF & x) /*bitmask 0-16 to reg*/
#define GET_CPU_COMM_MBOX_READ_MASK(x) (u16)(0x0000FFFF & x) /*reg to bitmask*/
#define MBOX_BIT(x) (1 << x) /* mbox # to register mask bit */

#define CPU_COMM_REGS_CPUC_L2_IRQ_A15R_STATUS (info->comm+0x54)
#define CPU_COMM_REGS_CPUC_L2_IRQ_ARM0_STATUS CPU_COMM_REGS_CPUC_L2_IRQ_A15R_STATUS
#define CPU_COMM_REGS_CPUC_L2_IRQ_A15S_STATUS (info->comm+0x5C)
#define CPU_COMM_REGS_CPUC_L2_IRQ_ARM1_STATUS CPU_COMM_REGS_CPUC_L2_IRQ_A15S_STATUS
#define CPU_COMM_REGS_CPUC_L2_IRQ_ARM2_STATUS (info->comm+0x64)
#define CPU_COMM_REGS_CPUC_L2_IRQ_ARM3_STATUS (info->comm+0x6C)
#define MBOX0_IRQ_STATUS_MASK 1
#define MBOX0_IRQ_STATUS_SHIFT 0
#define MBOX1_IRQ_STATUS_MASK 1
#define MBOX1_IRQ_STATUS_SHIFT 1
#define MBOX2_IRQ_STATUS_MASK 1
#define MBOX2_IRQ_STATUS_SHIFT 2
#define MBOX3_IRQ_STATUS_MASK 1
#define MBOX3_IRQ_STATUS_SHIFT 3
#define MBOX4_IRQ_STATUS_MASK 1
#define MBOX4_IRQ_STATUS_SHIFT 4
#define MBOX5_IRQ_STATUS_MASK 1
#define MBOX5_IRQ_STATUS_SHIFT 5
#define BMU_INTR_BATT_LOW_STATUS_MASK 1
#define BMU_INTR_BATT_LOW_STATUS_SHIFT 28
#define BMU_INTR_AP_RESTORED_STATUS_MASK 1
#define BMU_INTR_AP_RESTORED_STATUS_SHIFT 29
#define CM_INTR_HE_WKUP_STATUS_MASK 1
#define CM_INTR_HE_WKUP_STATUS_SHIFT 31
#define CM_RUN_STATE_SHUTDOWN_REQUESTED 11
#define CM_RUN_STATE_SHUTDOWN_COMPLETE 12
#define CM_RUN_STATE_EXCEPTION_SHUTDOWN_REQ 13

#define R32(var, reg) var = readl_relaxed(reg)
#define W32(reg, val) writel_relaxed(val, reg)
#define RF32(reg, fld) ((readl_relaxed(reg)>>fld##_SHIFT)&fld##_MASK)
#define SRF32(reg, fld) ((reg>>fld##_SHIFT)&fld##_MASK)
#define WF32(reg, fld, val) do { \
	u32 tmp; \
	R32(tmp, reg);  \
	tmp = ((tmp&~(fld##_MASK<<fld##_SHIFT)) | \
	((val&fld##_MASK)<<fld##_SHIFT)); \
	W32(reg, tmp); \
	} while (0)
#define SWF32(reg, fld, val) do { reg = \
	((reg&~(fld##_MASK<<fld##_SHIFT)) | \
	((val&fld##_MASK)<<fld##_SHIFT)); \
	} while (0)

/**
* enum mbox_io_idx - Index to device tree register IO memory
*/
enum mbox_io_idx {
	IO_MEM_MBOX,
	IO_MEM_COMM
};

#endif
