/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2017 Broadcom.  All rights reserved.
* The term “Broadcom” refers to Broadcom Inc. and/or its subsidiaries.
*
* Unless you and Broadcom execute a separate written software license
* agreement governing use of this software, this software is licensed to
* you under the terms of the GNU General Public License version 2 (the
* "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
* the following added to such license:
*
* As a special exception, the copyright holders of this software give you
* permission to link this software with independent modules, and to copy
* and distribute the resulting executable under terms of your choice,
* provided that you also meet, for each linked independent module, the
* terms and conditions of the license of that module. An independent
* module is a module which is not derived from this software. The special
* exception does not apply to any modifications of the software.
*
* Notwithstanding the above, under no circumstances may you combine this
* software in any way with any other Broadcom software provided under a
* license other than the GPL, without Broadcom's express prior written
* consent.
*
****************************************************************************
* Author: Peter Sulc <peter.sulc@broadcom.com>
* Filename: bus_capture.c
* Description: bus capture flattened device tree parsing and early init
****************************************************************************/

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/io.h>
#include <linux/printk.h>
#include <linux/delay.h>
#include <asm/exception.h>
#include <linux/of.h>
#include <linux/of_fdt.h>

#include "gisb_arb.h"
#include "ubus_capture.h"

static int __init fdt_scan_gisb_arb(unsigned long node)
{
	const __be32 *reg;
	int reglen;
	int addr_cells, size_cells;
	u64 base, size;

	reg = of_get_flat_dt_prop(node, "reg", &reglen);
	if (reg == NULL) {
		pr_err("gisb-arb: no reg property\n");
		return -ENOENT;
	}
	addr_cells = size_cells = 1;

	base = dt_mem_next_cell(addr_cells, &reg);
	size = dt_mem_next_cell(size_cells, &reg);

	return gisb_arb_map(base, size);
}

static int __init fdt_scan_ubus_capture(unsigned long node)
{
	const __be32 *reg, *seg;
	int reglen, seglen;
	int addr_cells;
	int size_cells;
	int rc, bases, i;
	const char *p;

	reg = of_get_flat_dt_prop(node, "reg", &reglen);
	if (reg == NULL) {
		pr_err("ubus-capture: no reg property\n");
		return -ENOENT;
	}
	seg = of_get_flat_dt_prop(node, "segments", &seglen);
	if (seg == NULL) {
		pr_err("ubus-capture: no segments property\n");
		return -ENOENT;
	}
	addr_cells = size_cells = 2;
	bases = reglen / (sizeof(__be32) * (addr_cells + size_cells));
	rc = ubus_capt_alloc_bases(bases);
	if (rc)
		return rc;

	for (i = 0; i < bases; i++) {
		u64 base = dt_mem_next_cell(addr_cells, &reg);
		u64 size = dt_mem_next_cell(size_cells, &reg);
		u32 segs = dt_mem_next_cell(1, &seg);
		ubus_capt_map_base(i, base, size, segs);
		pr_info("ubus %d segment capture space: %llx size %llx\n",
			segs, base, size);
	}
	reg = of_get_flat_dt_prop(node, "address-range", &reglen);
	if (reg) {
		ubus_capt_addr_start = dt_mem_next_cell(1, &reg);
		ubus_capt_addr_end = dt_mem_next_cell(1, &reg);
	}
	else {
		pr_info("ubus-capture: no address-range property\n");
	}
	reg = of_get_flat_dt_prop(node, "address-exclude", &reglen);
	if (reg) {
		ubus_capt_addr_exclude = dt_mem_next_cell(1, &reg);
	}
	else {
		pr_info("ubus-capture: no address-exclude property\n");
	}
	reg = of_get_flat_dt_prop(node, "pid-range", &reglen);
	if (reg) {
		ubus_capt_pid_start = dt_mem_next_cell(1, &reg);
		ubus_capt_pid_end = dt_mem_next_cell(1, &reg);
	}
	else {
		pr_info("ubus-capture: no pid-range property\n");
	}
	reg = of_get_flat_dt_prop(node, "pid-exclude", &reglen);
	if (reg) {
		ubus_capt_pid_exclude = dt_mem_next_cell(1, &reg);
	}
	else {
		pr_info("ubus-capture: no address-exclude property\n");
	}
	p = of_get_flat_dt_prop(node, "enabled-at-boot", &reglen);
	if (!p) {
		pr_err("ubus-capture: no enabled-at-boot property\n");
		return -ENOENT;
	}
	if ((reglen == 3) && (strncmp(p, "yes", 3) == 0))
		ubus_capt_on = 1;
	return 0;
}

static int __init fdt_scan_callback(unsigned long node, const char *uname,
				    int depth, void *data)
{
	if (of_flat_dt_is_compatible(node, "brcm,ubus-capture"))
		return fdt_scan_ubus_capture(node);
	if (of_flat_dt_is_compatible(node, "brcm,gisb-arb"))
		return fdt_scan_gisb_arb(node);
	return 0;
}

static int __init bus_capt_parse_flat_dt(void)
{
	int rc = of_scan_flat_dt(fdt_scan_callback, NULL);
	if (rc)
		pr_err("Error in of_scan_flat_dt: %d\n", rc);
	return rc;
}

static int __init bus_capt_early_init(void)
{
	int rc = bus_capt_parse_flat_dt();
	if (rc)
		return rc;
	rc = ubus_capt_early_init();
	return rc;
}

early_initcall(bus_capt_early_init);
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("UBUS capture support");
