 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/workqueue.h>
#include <linux/timer.h>
#include <linux/netfilter.h>
#include <linux/netfilter_bridge.h>
#include <net/netfilter/nf_conntrack_core.h>
#include <net/netfilter/nf_conntrack_l4proto.h>
#include <net/netfilter/nf_conntrack_l4proto.h>
#include <net/netfilter/nf_conntrack_ecache.h>
#include <net/netfilter/nf_conntrack_acct.h>
#include <net/netfilter/nf_conntrack_offload.h>
#include <net/netfilter/nf_conntrack_helper.h>
#include <linux/rculist_nulls.h>

#include "flowmgr.h"

struct delayed_work flowmgr_work;

int flowmgr_counter_start(void)
{
	unsigned long sec = msecs_to_jiffies(flowmgr.counter_update_period * 1000);
	if (sec)
	    schedule_delayed_work(&flowmgr_work, sec);
	return 0;
}

int flowmgr_counter_stop(void)
{
	cancel_delayed_work(&flowmgr_work);
	flush_delayed_work(&flowmgr_work);
	return 0;
}

static void flowmgr_work_handler(struct work_struct *work)
{
	struct nf_conntrack_tuple_hash *hash;
	struct hlist_nulls_head *ct_hash;
	struct hlist_nulls_node *n;
	unsigned int bucket, hsize;
	struct nf_conn *ct;
	struct nf_conn_offload *ct_offload;
	struct offload_info *ct_offload_info;

	nf_conntrack_get_ht(&ct_hash, &hsize);
	rcu_read_lock();
	for (bucket = 0; bucket < hsize; bucket++) {
		n = rcu_dereference(
		   hlist_nulls_first_rcu(&ct_hash[bucket]));

		while (!is_a_nulls(n)) {

			hash = (struct nf_conntrack_tuple_hash *)n;
			ct = nf_ct_tuplehash_to_ctrack(hash);
			if (!ct)
				goto next_ct;

			ct_offload = nf_conn_offload_find(ct);
			if (!ct_offload)
				goto next_ct;

			ct_offload_info = &ct_offload->info[hash->tuple.dst.dir];
			if (nf_ct_is_dying(ct))
				goto next_ct;

			if (ct_offload_info->ctinfo == -1)
				goto next_ct;

			pr_debug("Update Counters for flow %d\n",
				 ct_offload_info->flow_id);

			flowmgr_update_flow_stats(ct, hash->tuple.dst.dir, 1);

next_ct:
			n = rcu_dereference(hlist_nulls_next_rcu(n));
		}
	}
	rcu_read_unlock();
	flowmgr_counter_start();
}

int flowmgr_counter_init(void)
{
	INIT_DELAYED_WORK(&flowmgr_work, flowmgr_work_handler);
	return 0;
}

void flowmgr_counter_exit(void)
{
	flush_work(&flowmgr_work.work);
}
