 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/gpio/consumer.h>

struct t3_priv {
	struct device *dev;
	struct platform_device *pdev;
	struct gpio_desc *gpio;
	int irq;
};

static irqreturn_t t3_isr(int irq, void *d)
{
	struct t3_priv *priv = d;

	dev_info(priv->dev, "%s: IRQ -->\n", __func__);
	dev_info(priv->dev, "%s: IRQ <--\n", __func__);
	return IRQ_HANDLED;
}

static int t3_probe(struct platform_device *pdev)
{
	int status = 0;
	struct device *dev = &pdev->dev;
	struct t3_priv *priv;
	struct gpio_desc *gpio;
	u32 irqflags;

	dev_dbg(dev, "%s: -->\n", __func__);

	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv) {
		dev_err(dev, "unable to allocate private data\n");
		status = -ENOMEM;
		goto done;
	}

	platform_set_drvdata(pdev, priv);
	priv->dev = dev;

	gpio = devm_gpiod_get(dev,"wps");
	if (!gpio) {
		dev_err(dev, "Couldn't get GPIO");
		status = -ENOENT;
		goto done;
	}
	status = gpiod_direction_input(gpio);
	if (status) {
		dev_err(dev, "Couldn't set GPIO direction to input.");
		goto done;
	}

	priv->irq = gpiod_to_irq(priv->gpio);
	if (priv->irq < 0) {
		dev_err(dev, "Couldn't get IRQ\n");
		status = priv->irq;
		goto done;
	}
	irqflags = IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING;

	dev_dbg(dev, "Requesting IRQ %d with flags 0x%08x for GPIO %d.\n",
		 priv->irq, irqflags, desc_to_gpio(gpio));
	status = devm_request_irq(dev, priv->irq, t3_isr, irqflags,
				  "WPS IRQ", priv);
	if (status) {
		dev_err(dev, "Failed registering IRQ with status %d!\n",
			status);
		goto done;
	}

done:
	dev_dbg(dev, "%s: <--\n", __func__);
	return status;
}

static int t3_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct t3_priv *priv = platform_get_drvdata(pdev);

	dev_dbg(dev, "%s: -->\n", __func__);
	free_irq(priv->irq, priv);
	devm_gpiod_put(dev, priv->gpio);
	dev_dbg(dev, "%s: <--\n", __func__);
	return 0;
}

static const struct of_device_id t3_of_match[] = {
	{.compatible = "brcm,t3"},
	{}
};
MODULE_DEVICE_TABLE(of, t3_of_match);
static struct platform_driver t3_driver = {
	.probe	= t3_probe,
	.remove	= t3_remove,
	.driver = {
		.name		= "t3",
		.owner		= THIS_MODULE,
		.of_match_table	= t3_of_match
	}
};

__init int t3_init(void)
{
	platform_driver_register(&t3_driver);
	return 0;
}

void t3_exit(void)
{
	platform_driver_unregister(&t3_driver);
	return;
}

module_init(t3_init);
module_exit(t3_exit);
MODULE_AUTHOR("Tim Ross");
MODULE_LICENSE("GPL v2");
