/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2017 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * vFlash block IO proc driver
 *
 * Author: Tim Ross <tim.ross@broadcom.com>
 * Author: Venky Selvaraj <venky.selvaraj@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/dma-mapping.h>
#include <proc_cmd.h>
#include "vfbio_priv.h"
#include "vfbio_proc.h"

#define VFBIO_PROC_DIR "driver/vfbio"
#define VFBIO_DH_PROC_FILE	"device_health"
#define VFBIO_EMMC_SINFO_PROC_FILE "emmc_standard_info"
#define VFBIO_EMMC_PINFO_PROC_FILE "emmc_proprietary_info"
#define VFBIO_CMD_PROC_FILE	"cmd"
#define EST_MAX 12
#define PRE_EOL_MAX 5
#define TUNNEL_NAME "rg-smc"

char *est[EST_MAX] = {
	"Not Defined",
	"0% - 10% device lifetime used",
	"10% - 20% device lifetime used",
	"20% - 30% device lifetime used",
	"30% - 40% device lifetime used",
	"40% - 50% device lifetime used",
	"50% - 60% device lifetime used",
	"60% - 70% device lifetime used",
	"70% - 80% device lifetime used",
	"80% - 90% device lifetime used",
	"90% - 100% device lifetime used",
	"Exceeded its maximum estimated device lifetime"
};

char *pre_eol[PRE_EOL_MAX] = {
	"Not Defined",
	"Normal",
	"Warning",
	"Urgent",
	"Read Only"
};

struct dh_res {
	bool end;
	u8 est_a;
	u8 est_b;
	u8 pre_eol;
};

struct emmc_info {
	bool end;
	u16 size;
	u16 len;
};

static struct proc_dir_entry *vfbio_proc_dir;
static struct proc_dir_entry *vfbio_dh_proc_file;
static struct proc_dir_entry *vfbio_emmc_sinfo_proc_file;
static struct proc_dir_entry *vfbio_emmc_pinfo_proc_file;
static bool vbio_dh_end;
static bool vbio_dh_begin;
static u32 device_cnt;
static const char lum_swap_delim[] = ",";
static void *emmc_info_buf;
static bool vbio_emmc_info_end;
static bool vbio_emmc_info_begin;
static u32 emmc_info_data_offset;
static DEFINE_MUTEX(emmc_info_mutex);

static struct vfbio_device *vfbio_get_dev_info(char *lun_name)
{
	struct list_head *pos;
	struct vfbio_device *vfdev = NULL;
	bool dev_found = false;

	if (!lun_name) {
		dev_err(vfbio_char_device, "LUN Name NULL\n");
		return NULL;
	}

	list_for_each(pos, &vfdevs_list) {
		vfdev = list_entry(pos, struct vfbio_device, list);
		if (!strcmp(lun_name, vfdev->name)) {
			dev_dbg(vfbio_char_device, "LUN Name %s found\n", lun_name);
			dev_found = true;
			break;
		}
	}

	if (!dev_found) {
		dev_err(vfbio_char_device, "LUN Name %s not found\n", lun_name);
		return NULL;
	}

	return vfdev;
}

static int vfbio_lunswap(char *lun_name, struct vfbio_device *vfdev,
			bool first, bool last)
{
	int status = 0;
	rpc_msg msg;
	bool state;

	if (!vfdev) {
		vfdev = vfbio_get_dev_info(lun_name);
		if (!vfdev) {
			dev_err(vfbio_char_device, "Getting LUN ID failed\n");
			return -EINVAL;
		}
	}
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		SYS_FUNC_GET_LUN_NEXT_BOOT_STATE, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);
	status = rpc_send_request(vfbio_rpc_tunnel, &msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device,
			RED("rpc_send_request failure (%d) on get LUN boot state\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}
	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		dev_err(vfbio_char_device,
			RED("invalid response %d on get LUN boot state\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}
	state = sys_msg_get_lun_state(&msg);

	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		SYS_FUNC_SET_LUN_NEXT_BOOT_STATE, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);
	sys_msg_set_lunswap_flag(&msg, first, last);
	sys_msg_set_lunswap_state(&msg, !state);
	status = rpc_send_request(vfbio_rpc_tunnel, &msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device,
			RED("rpc_send_request failure (%d) on set LUN boot state\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}
	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		dev_err(vfbio_char_device,
			RED("invalid response %d on set LUN boot state\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}

	return status;
}

static int vfbio_lunstat(char *lun_name, struct vfbio_device *vfdev,
			bool *state)
{
	int status = 0;
	rpc_msg msg;

	if (!vfdev) {
		vfdev = vfbio_get_dev_info(lun_name);
		if (!vfdev) {
			dev_dbg(vfbio_char_device, "Getting LUN ID failed\n");
			return -EINVAL;
		}
	}
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		SYS_FUNC_GET_LUN_NEXT_BOOT_STATE, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);
	status = rpc_send_request(vfbio_rpc_tunnel, &msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device,
			RED("rpc_send_request failure (%d) on get LUN\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}
	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		dev_err(vfbio_char_device,
			RED("invalid response %d on get LUN\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}
	*state = sys_msg_get_lun_state(&msg);

	return 0;
};


static int vfbio_get_attr(char *lun_name, struct vfbio_device *vfdev, u32 *attr)
{
	int status = 0;
	rpc_msg msg;

	if (!vfdev) {
		vfdev = vfbio_get_dev_info(lun_name);
		if (!vfdev) {
			dev_dbg(vfbio_char_device, "Getting LUN ID failed\n");
			return -EINVAL;
		}
	}

	rpc_msg_init(&msg, RPC_SERVICE_VFBIO,
		VFBIO_FUNC_GET_ATTR, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);

	status = rpc_send_request(vfbio_rpc_tunnel, &msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device,
			RED("rpc_send_request failure (%d) attempting to "
			    "get LUN attributes\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}

	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		dev_err(vfbio_char_device,
			RED("invalid response %d on lock LUN\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}
	*attr = vfbio_msg_get_attr(&msg);

	return 0;
};

static int vfbio_set_attr(char *lun_name, struct vfbio_device *vfdev, u32 attr)
{
	int status = 0;
	rpc_msg msg;

	if (!vfdev) {
		vfdev = vfbio_get_dev_info(lun_name);
		if (!vfdev) {
			dev_dbg(vfbio_char_device, "Getting LUN ID failed\n");
			return -EINVAL;
		}
	}

	rpc_msg_init(&msg, RPC_SERVICE_VFBIO,
		VFBIO_FUNC_SET_ATTR, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);
	vfbio_msg_set_attr(&msg, attr);

	status = rpc_send_request(vfbio_rpc_tunnel, &msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device,
			RED("rpc_send_request failure (%d) attempting to "
			    "set LUN attributes\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}

	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		dev_err(vfbio_char_device,
			RED("invalid response %d on lock LUN\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}

	return 0;
};

static inline int vfbio_get_emmc_info(struct emmc_info *info,
			bool begin, enum vfbio_func_idx rpc_func)
{
	int status = 0;
	rpc_msg msg;
	dma_addr_t dma_addr;

	BUG_ON(!info);

	dma_addr = dma_map_single(vfbio_char_device, emmc_info_buf, info->size,
				  DMA_FROM_DEVICE);
	if (dma_mapping_error(vfbio_char_device, dma_addr)) {
		dev_err(vfbio_char_device, RED("failure mapping emmc_info_buf\n"));
		status = -ENOMEM;
		goto done;
	}
	dev_dbg(vfbio_char_device, MAG("dma addr: 0x%llx\n"), dma_addr);
	rpc_msg_init(&msg, RPC_SERVICE_VFBIO, rpc_func, 0,
		     0, 0, 0);
	vfbio_msg_set_begin(&msg, begin);
	vfbio_msg_set_emmc_info_addr(&msg, (u64)dma_addr);
	vfbio_msg_set_emmc_info_size(&msg, (u16)info->size);
	status = rpc_send_request(vfbio_rpc_tunnel, &msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device, RED("rpc_send_request failure (%d)\n"),
			status);
		rpc_dump_msg(&msg);
		goto done;
	}
	dma_unmap_single(vfbio_char_device, dma_addr, info->size,
			 DMA_FROM_DEVICE);
	status = vfbio_msg_get_retcode(&msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device, RED("vfbio msg retcode %d\n"), (s8)status);
		rpc_dump_msg(&msg);
		status = -EIO;
	}
	info->len = vfbio_msg_get_emmc_info_len(&msg);
	info->end = vfbio_msg_get_end(&msg);

done:
	return status;
}

static inline int vfbio_get_device_health(
			struct dh_res *res, bool begin)
{
	int status = 0;
	rpc_msg msg;

	BUG_ON(!res);

	rpc_msg_init(&msg, RPC_SERVICE_VFBIO, VFBIO_FUNC_DEVICE_HEALTH, 0,
		     0, 0, 0);
	vfbio_msg_set_begin(&msg, begin);
	status = rpc_send_request(vfbio_rpc_tunnel, &msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device,
			RED("rpc_send_request failure (%d)\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}
	status = vfbio_msg_get_retcode(&msg);
	if (unlikely(status)) {
		dev_err(vfbio_char_device,
			RED("vfbio msg retcode %d\n"), (s8)status);
		rpc_dump_msg(&msg);
		status = -EIO;
	}
	res->end = vfbio_msg_get_end(&msg);
	res->est_a = vfbio_msg_get_dh_est_a(&msg);
	res->est_b = vfbio_msg_get_dh_est_b(&msg);
	res->pre_eol = vfbio_msg_get_dh_pre_eol(&msg);

	return status;
}

static int vfbio_emmc_pinfo_proc_show(struct seq_file *seq, void *v)
{
	struct emmc_info info;
	int status = 0;

	dev_dbg(vfbio_char_device, "-->\n");

	if (!v)
		return -EINVAL;

	memset(&info, 0, sizeof(struct emmc_info));
	info.size = PAGE_SIZE;
	status = vfbio_get_emmc_info(&info, vbio_emmc_info_begin,
				     VFBIO_FUNC_EMMC_PROP_INFO);
	if (unlikely(status))
		return -EIO;
	vbio_emmc_info_begin = false;
	vbio_emmc_info_end = info.end;
	seq_write(seq, (const void *)emmc_info_buf, (size_t)info.len);
	emmc_info_data_offset += info.len;

	dev_dbg(vfbio_char_device, "<--\n");
	return 0;
}

static void *vfbio_emmc_info_proc_start(struct seq_file *seq, loff_t *pos)
{
	dev_dbg(vfbio_char_device, "-->\n");

	mutex_lock(&emmc_info_mutex);
	if (*pos == 0) {
		emmc_info_data_offset = 0;
		vbio_emmc_info_begin = true;
		vbio_emmc_info_end = false;
	} else
		vbio_emmc_info_begin = false;

	if (vbio_emmc_info_end)
		return NULL;

	dev_dbg(vfbio_char_device, "<--\n");
	return vfbio_char_device;
}

static void vfbio_emmc_info_proc_stop(struct seq_file *seq, void *v)
{
	dev_dbg(vfbio_char_device, "-->\n");
	mutex_unlock(&emmc_info_mutex);
	dev_dbg(vfbio_char_device, "<--\n");
}

static void *vfbio_emmc_info_proc_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	dev_dbg(vfbio_char_device, "-->\n");

	(*pos)++;
	if (vbio_emmc_info_end)
		return NULL;

	dev_dbg(vfbio_char_device, "<--\n");
	return vfbio_char_device;
}

static int vfbio_emmc_sinfo_proc_show(struct seq_file *seq, void *v)
{
	struct emmc_info info;
	int status = 0;

	dev_dbg(vfbio_char_device, "-->\n");

	if (!v)
		return -EINVAL;

	memset(&info, 0, sizeof(struct emmc_info));
	info.size = PAGE_SIZE;
	status = vfbio_get_emmc_info(&info, vbio_emmc_info_begin,
					VFBIO_FUNC_EMMC_STANDARD_INFO);
	if (unlikely(status))
		return -EIO;
	vbio_emmc_info_begin = false;
	vbio_emmc_info_end = info.end;
	seq_write(seq, (const void *)emmc_info_buf, (size_t)info.len);
	emmc_info_data_offset += info.len;

	dev_dbg(vfbio_char_device, "<--\n");
	return 0;
}

static const struct seq_operations vfbio_emmc_pinfo_seq_ops = {
	.start = vfbio_emmc_info_proc_start,
	.stop = vfbio_emmc_info_proc_stop,
	.next = vfbio_emmc_info_proc_next,
	.show = vfbio_emmc_pinfo_proc_show,
};

static const struct seq_operations vfbio_emmc_sinfo_seq_ops = {
	.start = vfbio_emmc_info_proc_start,
	.stop = vfbio_emmc_info_proc_stop,
	.next = vfbio_emmc_info_proc_next,
	.show = vfbio_emmc_sinfo_proc_show,
};

static void *vfbio_dh_proc_start(struct seq_file *seq, loff_t *pos)
{
	dev_dbg(vfbio_char_device, "-->\n");

	if (*pos == 0) {
		device_cnt = 0;
		vbio_dh_begin = true;
		vbio_dh_end = false;
		seq_puts(seq, "---------- Flash devices health report ----------\n");
	} else
		vbio_dh_begin = false;

	if (vbio_dh_end)
		return NULL;

	dev_dbg(vfbio_char_device, "<--\n");
	return vfbio_char_device;
}

static void vfbio_dh_proc_stop(struct seq_file *seq, void *v)
{
	dev_dbg(vfbio_char_device, "-->\n");
	dev_dbg(vfbio_char_device, "<--\n");
}

static void *vfbio_dh_proc_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	dev_dbg(vfbio_char_device, "-->\n");

	(*pos)++;
	if (vbio_dh_end)
		return NULL;

	dev_dbg(vfbio_char_device, "<--\n");
	return vfbio_char_device;
}

static int vfbio_dh_proc_show(struct seq_file *seq, void *v)
{
	struct dh_res res;
	int status = 0;

	dev_dbg(vfbio_char_device, "-->\n");

	if (!v)
		return -EINVAL;

	memset(&res, 0, sizeof(struct dh_res));
	device_cnt++;
	status = vfbio_get_device_health(&res, vbio_dh_begin);
	if (unlikely(status))
		return -EIO;
	seq_printf(seq, "Device Number: %u\n", device_cnt);
	seq_printf(seq, "Device Lifetime estimate(EST-A) : Value as per JESD84-B51 standard - %d (%s)\n",
		res.est_a, (res.est_a < EST_MAX) ?
		est[res.est_a] : "Reserved");
	seq_printf(seq, "Device Lifetime estimate(EST-B) : Value as per JESD84-B51 standard - %d (%s)\n",
		res.est_b, (res.est_b < EST_MAX) ?
		est[res.est_b] : "Reserved");
	seq_printf(seq, "Pre EOL Lifetime Estimation(Pre EOL) : Value as per JESD84-B51 standard - %d (%s)\n",
		res.pre_eol, (res.pre_eol < PRE_EOL_MAX) ?
		pre_eol[res.pre_eol] : "Reserved");
	seq_puts(seq, "\n");
	vbio_dh_begin = false;
	vbio_dh_end = res.end;

	dev_dbg(vfbio_char_device, "<--\n");
	return 0;
}

static const struct seq_operations vfbio_dh_seq_ops = {
	.start = vfbio_dh_proc_start,
	.stop = vfbio_dh_proc_stop,
	.next = vfbio_dh_proc_next,
	.show = vfbio_dh_proc_show,
};

static int vfbio_emmc_pinfo_proc_seq_open(struct inode *inode, struct file *file)
{
	int status;

	dev_dbg(vfbio_char_device, "-->\n");

	status = seq_open(file, &vfbio_emmc_pinfo_seq_ops);
	if (!status) {
		struct seq_file *sf = file->private_data;

		sf->private = PDE_DATA(inode);
	}

	dev_dbg(vfbio_char_device, "<--\n");
	return status;
}

static const struct proc_ops vfbio_emmc_pinfo_fops = {
	.proc_open = vfbio_emmc_pinfo_proc_seq_open,
	.proc_read = seq_read,
	.proc_lseek = seq_lseek,
	.proc_release = seq_release,
};

static int vfbio_emmc_sinfo_proc_seq_open(struct inode *inode, struct file *file)
{
	int status;

	dev_dbg(vfbio_char_device, "-->\n");

	status = seq_open(file, &vfbio_emmc_sinfo_seq_ops);
	if (!status) {
		struct seq_file *sf = file->private_data;

		sf->private = PDE_DATA(inode);
	}

	dev_dbg(vfbio_char_device, "<--\n");
	return status;
}

static const struct proc_ops vfbio_emmc_sinfo_fops = {
	.proc_open = vfbio_emmc_sinfo_proc_seq_open,
	.proc_read = seq_read,
	.proc_lseek = seq_lseek,
	.proc_release = seq_release,
};

static int vfbio_dh_proc_open(struct inode *inode, struct file *file)
{
	int status;

	dev_dbg(vfbio_char_device, "-->\n");

	status = seq_open(file, &vfbio_dh_seq_ops);
	if (!status) {
		struct seq_file *sf = file->private_data;

		sf->private = PDE_DATA(inode);
	}

	dev_dbg(vfbio_char_device, "<--\n");
	return status;
}

static const struct proc_ops vfbio_dh_fops = {
	.proc_open = vfbio_dh_proc_open,
	.proc_read = seq_read,
	.proc_lseek = seq_lseek,
	.proc_release = seq_release,
};

static void vfbio_proc_cmd_lunswap_help(char *str)
{
	pr_info("%s lunswap: Swap specified LUN (or) all LUNs\n", str);
	pr_info("%s  lunswap {LUN name(s) seperated by ',' | all}\n", str);
}

static void vfbio_proc_cmd_lunstat_help(char *str)
{
	pr_info("%s lunstat: Provide status of specified LUN (or) all LUNs\n",
			 str);
	pr_info("%s  lunstat  {LUN name(s) seperated by ',' | all}\n", str);
}

static void vfbio_proc_cmd_lunattr_help(char *str)
{
	pr_info("%s lunattr: get or set LUN attributes\n", str);
	pr_info("%s  lunattr {LUN name} [rw|ro]\n", str);
}

static void vfbio_proc_cmd_help_help(char *str)
{
	pr_info("%s help: Help on individual command\n", str);
}

static int vfbio_proc_cmd_help(int argc, char *argv[])
{
	int status = 0;

	if (argc != 2) {
		dev_err(vfbio_char_device, RED("Invalid # of arguments.\n"));
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(argv[1], "lunswap", sizeof("lunswap")))
		vfbio_proc_cmd_lunswap_help("");
	else if (!strncmp(argv[1], "lunstat", sizeof("lunstat")))
		vfbio_proc_cmd_lunstat_help("");
	else if (!strncmp(argv[1], "lunattr", sizeof("lunattr")))
		vfbio_proc_cmd_lunattr_help("");
	else {
		dev_err(vfbio_char_device, RED("Unrecognized command: %s\n"),
			argv[1]);
		status = -EINVAL;
	}

done:
	return status;
}

static int vfbio_proc_cmd_lunswap(int argc, char *argv[])
{
	int status = 0;
	struct list_head *pos;
	struct vfbio_device *vfdev = NULL;
	int cnt = 0, first_vflash = -1, last_vflash = 0;
	char *lun_name = NULL, *tmp = NULL, *tmp_bak = NULL;

	if (argc != 2) {
		dev_err(vfbio_char_device, "Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(argv[1], "all", strlen("all"))) {
		list_for_each(pos, &vfdevs_list) {
			vfdev = list_entry(pos, struct vfbio_device, list);
			if (vfdev->name[strlen(vfdev->name)-1] == 'b') {
				if (first_vflash < 0)
					first_vflash = cnt;
				last_vflash = cnt;
			}
			cnt++;
		}
		cnt = 0;

		list_for_each(pos, &vfdevs_list) {
			vfdev = list_entry(pos, struct vfbio_device, list);
			if (vfdev->name[strlen(vfdev->name)-1] == 'b') {
				status = vfbio_lunswap(NULL, vfdev,
					cnt == first_vflash,
					cnt == last_vflash);
				if (status < 0) {
					dev_err(vfbio_char_device, "Lun Swap Failed for %s.\n",
						vfdev->name);
					goto done;
				}
			}
			cnt++;
		}
	} else {
		tmp = kmalloc(strlen(argv[1])+1, GFP_KERNEL);
		if (!tmp) {
			dev_err(vfbio_char_device, "Unable to allocate memory to copy LUN names.\n");
			status = -ENOMEM;
			goto done;
		}
		tmp_bak = tmp;
		memset(tmp_bak, 0, strlen(argv[1])+1);
		memcpy(tmp_bak, argv[1], strlen(argv[1]));
		while ((lun_name = strsep(&tmp_bak, lum_swap_delim))) {
			if (first_vflash < 0)
				first_vflash = cnt;
			last_vflash = cnt;
			cnt++;
		}
		cnt = 0;
		tmp_bak = tmp;
		memset(tmp_bak, 0, strlen(argv[1])+1);
		memcpy(tmp_bak, argv[1], strlen(argv[1]));
		while ((lun_name = strsep(&tmp_bak, lum_swap_delim))) {
			status = vfbio_lunswap(lun_name, NULL,
				cnt == first_vflash,
				cnt == last_vflash);
			if (status < 0) {
				dev_err(vfbio_char_device, "Lun Swap Failed for %s.\n",
					lun_name);
				goto done;
			}
			cnt++;
		}

	}

done:
	if (tmp)
		kfree(tmp);
	return status;
}

static int vfbio_proc_cmd_lunstat(int argc, char *argv[])
{
	int status = 0;
	struct list_head *pos;
	struct vfbio_device *vfdev = NULL;
	char *lun_name;
	bool state;

	if (argc != 2) {
		dev_err(vfbio_char_device, "Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(argv[1], "all", strlen("all"))) {
		list_for_each(pos, &vfdevs_list) {
			vfdev = list_entry(pos, struct vfbio_device, list);
			if (vfdev->name[strlen(vfdev->name)-1] == 'o') {
				status = vfbio_lunstat(NULL, vfdev, &state);
				if (status < 0) {
					dev_err(vfbio_char_device, "Getting Lun state Failed for %s.\n",
						   vfdev->name);
					goto done;
				}
				pr_info("%s - LUN %d State : %s\n",
					vfdev->name, vfdev->lun,
					(state ? "backup  " : "operational"));
			}
		}
	} else {
		while ((lun_name = strsep(&argv[1], lum_swap_delim))) {
			status = vfbio_lunstat(lun_name, NULL, &state);
			if (status < 0) {
				dev_err(vfbio_char_device, "Get Lun stat Failed for %s.\n",
					lun_name);
				goto done;
			}
			pr_info("%s - State : %s\n",
				lun_name, (state ? "backup  " : "operational"));
		}
	}

done:
	return status;
}

static int vfbio_proc_cmd_lunattr(int argc, char *argv[])
{
	int status = 0;
	struct vfbio_device *vfdev = NULL;
	char *lun_name;
	u32 attr = 0;
	int i;

	if (argc < 2) {
		dev_err(vfbio_char_device, "Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	lun_name = argv[1];

	if (argc == 2) {
		status = vfbio_get_attr(lun_name, vfdev, &attr);
		if (status) {
			dev_err(vfbio_char_device,
				"getting attributes for LUN %s failed.\n",
				lun_name);
			status = -EIO;
		}
		goto done;
	}

	for (i = 2; i < argc; i++) {
		if(!strcmp(argv[i],"rw"))
			attr |= VFBIO_ATTR_RW;
		else if(!strcmp(argv[i],"ro"))
			attr &= ~VFBIO_ATTR_RW;
		else {
			dev_err(vfbio_char_device, "Invalid attribute, %s, specified.\n", argv[i]);
			status = -EINVAL;
			goto done;
		}
	}

	status = vfbio_set_attr(lun_name, vfdev, attr);
	if (status)
		dev_err(vfbio_char_device,
			"setting attributes for LUN %s failed.\n",
			lun_name);

done:
	if (!status)
		pr_info("LUN %s attributes: %s\n", lun_name, attr & VFBIO_ATTR_RW ? "rw " : "ro ");

	return status;
}

static struct proc_cmd_ops command_entries[] = {
	PROC_CMD_INIT("help", vfbio_proc_cmd_help),
	PROC_CMD_INIT("lunswap", vfbio_proc_cmd_lunswap),
	PROC_CMD_INIT("lunstat", vfbio_proc_cmd_lunstat),
	PROC_CMD_INIT("lunattr", vfbio_proc_cmd_lunattr),
};

struct proc_cmd_table vfbio_command_table = {
	.module_name = MODULE_NAME,
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

static struct proc_dir_entry *vfbio_cmd_proc_file;

int __init vfbio_proc_init(void)
{
	int status = 0;

	dev_dbg(vfbio_char_device, "-->\n");
	vfbio_proc_dir = proc_mkdir(VFBIO_PROC_DIR, NULL);
	if (!vfbio_proc_dir) {
		dev_err(vfbio_char_device, "Failed to create PROC directory %s.\n",
		       VFBIO_PROC_DIR);
		status = -EIO;
		goto done;
	}
	vfbio_dh_proc_file = proc_create(VFBIO_DH_PROC_FILE, 0444,
					 vfbio_proc_dir, &vfbio_dh_fops);
	if (!vfbio_dh_proc_file) {
		dev_err(vfbio_char_device, "Failed to create %s\n", VFBIO_DH_PROC_FILE);
		status = -EIO;
		vfbio_proc_exit();
		goto done;
	}
	vfbio_emmc_sinfo_proc_file = proc_create(VFBIO_EMMC_SINFO_PROC_FILE, 0444,
					 vfbio_proc_dir, &vfbio_emmc_sinfo_fops);
	if (!vfbio_emmc_sinfo_proc_file) {
		dev_err(vfbio_char_device, "Failed to create %s\n", VFBIO_EMMC_SINFO_PROC_FILE);
		status = -EIO;
		vfbio_proc_exit();
		goto done;
	}
	vfbio_emmc_pinfo_proc_file = proc_create(VFBIO_EMMC_PINFO_PROC_FILE, 0444,
					 vfbio_proc_dir, &vfbio_emmc_pinfo_fops);
	if (!vfbio_emmc_pinfo_proc_file) {
		dev_err(vfbio_char_device, "Failed to create %s\n", VFBIO_EMMC_PINFO_PROC_FILE);
		status = -EIO;
		vfbio_proc_exit();
		goto done;
	}
	vfbio_cmd_proc_file = proc_create_cmd(VFBIO_CMD_PROC_FILE,
		vfbio_proc_dir, &vfbio_command_table);
	if (!vfbio_cmd_proc_file) {
		dev_err(vfbio_char_device, "Failed to create %s\n", VFBIO_CMD_PROC_FILE);
		status = -EIO;
		vfbio_proc_exit();
		goto done;
	}

	/*
	 * Since we are requesting PAGE_SIZE bytes we are guaranteed by kmalloc
	 * that it will allocate a buffer that is aligned to this power-of-2
	 * size and also cache line aligned on the end. So it is safe to use
	 * for DMA with the SMC.
	 */
	emmc_info_buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
	if (!emmc_info_buf) {
		dev_err(vfbio_char_device, "Unable to allocate page to get EMMC standard info.\n");
		status = -ENOMEM;
		vfbio_proc_exit();
		goto done;
	}
	dev_dbg(vfbio_char_device, MAG("emmc info_buf: 0x%px\n"), emmc_info_buf);

done:
	dev_dbg(vfbio_char_device, "<--\n");
	return status;
}

void vfbio_proc_exit(void)
{
	dev_dbg(vfbio_char_device, "-->\n");
	if (vfbio_cmd_proc_file) {
		remove_proc_entry(VFBIO_CMD_PROC_FILE, vfbio_proc_dir);
		vfbio_cmd_proc_file = NULL;
	}
	if (vfbio_dh_proc_file) {
		remove_proc_entry(VFBIO_DH_PROC_FILE, vfbio_proc_dir);
		vfbio_dh_proc_file = NULL;
	}
	if (vfbio_emmc_sinfo_proc_file) {
		remove_proc_entry(VFBIO_EMMC_SINFO_PROC_FILE, vfbio_proc_dir);
		vfbio_emmc_sinfo_proc_file = NULL;
	}
	if (vfbio_emmc_pinfo_proc_file) {
		remove_proc_entry(VFBIO_EMMC_PINFO_PROC_FILE, vfbio_proc_dir);
		vfbio_emmc_pinfo_proc_file = NULL;
	}
	if (vfbio_proc_dir) {
		remove_proc_entry(VFBIO_PROC_DIR, NULL);
		vfbio_proc_dir = NULL;
	}
	if (emmc_info_buf) {
		kfree(emmc_info_buf);
		emmc_info_buf = NULL;
	}

	dev_dbg(vfbio_char_device, "<--\n");
}
