/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2017 Broadcom.  All rights reserved.
* The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
*
* Unless you and Broadcom execute a separate written software license
* agreement governing use of this software, this software is licensed to
* you under the terms of the GNU General Public License version 2 (the
* "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
* the following added to such license:
*
* As a special exception, the copyright holders of this software give you
* permission to link this software with independent modules, and to copy
* and distribute the resulting executable under terms of your choice,
* provided that you also meet, for each linked independent module, the
* terms and conditions of the license of that module. An independent
* module is a module which is not derived from this software. The special
* exception does not apply to any modifications of the software.
*
* Notwithstanding the above, under no circumstances may you combine this
* software in any way with any other Broadcom software provided under a
* license other than the GPL, without Broadcom's express prior written
* consent.
*
****************************************************************************
*
* Filename: sgmiiplus2_core.c
*
****************************************************************************
* Description: 28nm SGMIIPLUS2 Multi-rate Serdes/PHY core driver
* Author: Ravi Patel <ravi.patel@broadcom.com>
****************************************************************************/

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/of_net.h>
#include <linux/of_mdio.h>
#include <linux/of_platform.h>
#include "sgmiiplus2.h"
#include "miimdiomux3390.h"

#define MODULE_NAME	"sgmiiplus2"
#define MODULE_VER	"1.0"

static u16 seq_ref50m_vco6p25[] = {
	0x8000, 0x0c2f,
	0x8300, 0x0100,
	0x8308, 0xc000,
	0x8050, 0x5740,
	0x8051, 0x01d0,
	0x8052, 0x19f0,
	0x8053, 0xaab0,
	0x8054, 0x8821,
	0x8055, 0x0044,
	0x8056, 0x8000,
	0x8057, 0x0872,
	0x8058, 0x0000,
	0x8106, 0x0020,
	0x8054, 0x8021,
	0x8054, 0x8821,
};

static int sgmiiplus2_access(struct sgmiiplus2 *serdes, bool connect)
{
	u32 data;

	if (miimdiomux_select(serdes->mux_pdev, serdes->mux_select,
				serdes->mux_values[connect])) {
		dev_err(serdes->dev, "Error setting %s to %s\n", serdes->mux_select,
				serdes->mux_values[connect]);
		return -EIO;
	}

	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data = connect ? data | SGMII_MDIO_EN_MASK : data & ~SGMII_MDIO_EN_MASK;
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);

	return 0;
}

static int sgmiiplus2_extoffset(struct sgmiiplus2 *serdes,
		struct mii_bus *bus, int addr, u16 regnum)
{
	int data = 0, retval = 0;

	if ((regnum > 0x001F) && (regnum < 0x8000)) {
		retval = -ENXIO;
		goto _skip_extoffset;
	}

	/* Prepare MDIO C22 address extension command as required */
	regnum &= 0xFFF0;
	if (regnum & 0x8000) {
		if (serdes->extoffset != regnum) {
			data = regnum;
		}
	} else if (serdes->extoffset &&
			(serdes->extoffset != 0x8000)) {
		data = 0x8000;
	}

	/* Start MDIO C22 address extension transaction */
	if (data) {
		retval = __mdiobus_write(bus, addr, 0x1F, data);
		if (retval == 0)
			serdes->extoffset = regnum;
	}

_skip_extoffset:
	return retval;
}

static int sgmiiplus2_read(struct phy_device *phydev, int devad, u16 regnum)
{
	struct sgmiiplus2 *serdes = (struct sgmiiplus2 *)(phydev->priv);
	struct mii_bus *bus = phydev->mdio.bus;
	int addr = phydev->mdio.addr, retval;

	sgmiiplus2_access(serdes, 1);

	retval = sgmiiplus2_extoffset(serdes, bus, addr, regnum);
	if (retval)
		goto _skip_read;

	if (regnum < 0x20)
		retval = __mdiobus_read(bus, addr, regnum);
	else
		retval = __mdiobus_read(bus, addr, (0x10 + (regnum & 0xF)));

_skip_read:
	sgmiiplus2_access(serdes, 0);

	return retval;
}

static int sgmiiplus2_write(struct phy_device *phydev, int devad, u16 regnum, u16 val)
{
	struct sgmiiplus2 *serdes = (struct sgmiiplus2 *)(phydev->priv);
	struct mii_bus *bus = phydev->mdio.bus;
	int addr = phydev->mdio.addr, retval;

	sgmiiplus2_access(serdes, 1);

	retval = sgmiiplus2_extoffset(serdes, bus, addr, regnum);
	if (retval)
		goto _skip_write;

	if (regnum < 0x20)
		retval = __mdiobus_write(bus, addr, regnum, val);
	else
		retval = __mdiobus_write(bus, addr, (0x10 + (regnum & 0xF)), val);

_skip_write:
	sgmiiplus2_access(serdes, 0);

	return retval;
}

static int sgmiiplus2_suspend(struct brcmphy_core *phycore)
{
	struct sgmiiplus2 *serdes =
		container_of(phycore, struct sgmiiplus2, phycore);
	struct phy_device *core = phycore->core;
	int data;

	mutex_lock(&core->lock);
	if (!serdes->powerup)
		goto _skip_suspend;

	data = phy_read_mmd(core, 0, MII_BMCR);
	phy_write_mmd(core, 0, MII_BMCR, data | BMCR_PDOWN);
	phy_write_mmd(core, 0, 0x8000, 0x0c2f);

	/* Power Down enable except for reference clock */
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data |= (PWRDWN_MASK | IDDQ_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	serdes->powerup = 0;

_skip_suspend:
	mutex_unlock(&core->lock);

	return 0;
}

static int sgmiiplus2_resume(struct brcmphy_core *phycore)
{
	struct sgmiiplus2 *serdes =
		container_of(phycore, struct sgmiiplus2, phycore);
	struct phy_device *core = phycore->core;
	int data;

	mutex_lock(&core->lock);
	if (serdes->powerup)
		goto _skip_resume;

	/* Power Down disable */
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data &= ~(PWRDWN_MASK | IDDQ_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);

	phy_write_mmd(core, 0, 0x8000, 0x2c2f);
	data = phy_read_mmd(core, 0, MII_BMCR);
	phy_write_mmd(core, 0, MII_BMCR, data & ~BMCR_PDOWN);
	serdes->powerup = 1;

_skip_resume:
	mutex_unlock(&core->lock);

	return 0;
}

static int sgmiiplus2_media_speed(struct brcmphy_core *phycore,
		int media, int speed, int duplex)
{
	struct sgmiiplus2 *serdes =
		container_of(phycore, struct sgmiiplus2, phycore);
	struct phy_device *core = phycore->core;
	u32 data, timeout;
	int ret;
/*
 * (dis_seq) 0x8000 = 0x0c2f (disable sequencer)
 * (mode)    0x8300 = 0x0105 (fibre),
 *                    0x0101 (2.5G sgmii),
 *                    0x0100 (1G/100M sgmii)
 * (speed1)  0x8308 = 0xc010 (2.5G),
 *                    0xc000 (1G/100M/10M)
 * (os_mode) 0x834a = 0x0001 (2.5G),
 *                    0x0003 (1G/100M/10M)
 * (speed0)  0x0000 = 0x0140 (2.5G/1G),
 *                    0x2100 (100M),
 *                    0x0100 (10M)
 * (ena_seq) 0x8000 = 0x2c2f (enable sequencer)
 */
	u16 mode = 0x0100, speed1 = 0xc000, os_mode = 0x0003, speed0 = 0x0140;

	if (media == PORT_FIBRE)
		mode = 0x0105;

	switch (speed) {
		case SPEED_10:
			speed0 = 0x0100;
			break;
		case SPEED_100:
			speed0 = 0x2100;
			break;
		case SPEED_1000:
			break;
		case SPEED_2500:
			mode |= 0x0001;
			speed1 = 0xc010;
			os_mode = 0x0001;
			break;
		default:
			speed = SPEED_UNKNOWN;
	}

	ret = phy_write_mmd(core, 0, 0x8000, 0x0c2f);
	if (ret) {
		dev_err(serdes->dev, "Error %d disabling sequencer\n", ret);
		goto _ret;
	}

	if (speed == SPEED_UNKNOWN)
		goto _ret;

	ret = phy_write_mmd(core, 0, 0x8300, mode);
	if (ret) {
		dev_err(serdes->dev, "Error %d configuring mode\n", ret);
		goto _ret;
	}

	ret = phy_write_mmd(core, 0, 0x8308, speed1);
	if (ret) {
		dev_err(serdes->dev, "Error %d configuring speed1\n", ret);
		goto _ret;
	}

	ret = phy_write_mmd(core, 0, 0x834a, os_mode);
	if (ret) {
		dev_err(serdes->dev, "Error %d configuring os_mode\n", ret);
		goto _ret;
	}

	ret = phy_write_mmd(core, 0, 0x0000, speed0);
	if (ret) {
		dev_err(serdes->dev, "Error %d configuring speed0\n", ret);
		goto _ret;
	}

	ret = phy_write_mmd(core, 0, 0x8000, 0x2c2f);
	if (ret) {
		dev_err(serdes->dev, "Error %d enabling sequencer\n", ret);
		goto _ret;
	}

	/* Wait for PLL Lock */
	timeout = 100;
	do {
		data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_STAT);
		udelay(50);
	} while (!(data & PLL_LOCK_MASK) && --timeout);

	if (timeout == 0) {
		dev_err(serdes->dev, "Unable to lock Serdes PLL\n");
		ret = -ETIMEDOUT;
		goto _ret;
	}

	switch (data & SPEED_STATUS_MASK) {
		case SPEED_2P5G_STATUS_MASK:
			  dev_info(serdes->dev, "Configured Link Speed @ 2.5 Gbps\n");
			  break;
		case SPEED_1G_STATUS_MASK:
			  dev_info(serdes->dev, "Configured Link Speed @ 1 Gbps\n");
			  break;
		case SPEED_100M_STATUS_MASK:
			  dev_info(serdes->dev, "Configured Link Speed @ 100 Mbps\n");
			  break;
		case SPEED_10M_STATUS_MASK:
			  dev_info(serdes->dev, "Configured Link Speed @ 10 Mbps\n");
			  break;
	}

_ret:
	return ret;
}

static inline int sgmiiplus2_seq_write(struct sgmiiplus2 *serdes,
		u16 *seq, u16 size)
{
	struct phy_device *core = serdes->phycore.core;
	int num, ret = 0;

	for (num = 0; num < size; num += 2) {
		ret = phy_write_mmd(core, 0, seq[num], seq[num + 1]);
		if (ret) {
			dev_err(serdes->dev, "Error programming block "
					"reg 0x%04x data 0x%04x\n", seq[num], seq[num + 1]);
			break;
		}
	}

	return ret;
}

static struct phy_driver sgmiiplus2_phy_driver = {
	.read_mmd  = sgmiiplus2_read,
	.write_mmd = sgmiiplus2_write,
};

static int sgmiiplus2_phy_init(struct sgmiiplus2 *serdes)
{
	int ret = 0;

	sgmiiplus2_access(serdes, 1);
	serdes->phycore.core = mdiobus_scan(serdes->phycore.bus, serdes->phyaddr);
	sgmiiplus2_access(serdes, 0);
	if (IS_ERR(serdes->phycore.core) || serdes->phycore.core == NULL) {
		serdes->phycore.core = ERR_PTR(-EIO);
		ret = -EIO;
		goto _init_ret;
	}

	serdes->phycore.core->priv = serdes;
	serdes->phycore.core->drv = &sgmiiplus2_phy_driver;

	/* Configure PLL for 50 Mhz ref and 6.25 GHz VCO Frequency */
	if ((ret = sgmiiplus2_seq_write(serdes, seq_ref50m_vco6p25,
				ARRAY_SIZE(seq_ref50m_vco6p25))) != 0) {
		dev_err(serdes->dev, "Error programming seq_ref50m_vco6p25\n");
		goto _init_ret;
	}

	/* Wait at least 1ms */
	udelay(1000);

	/* Configure 2.5Gbps link speed */
	if ((ret = sgmiiplus2_media_speed(&serdes->phycore, PORT_MII,
				serdes->phycore.speed,
				serdes->phycore.duplex)) != 0) {
		dev_err(serdes->dev, "Error programming SGMII 2.5Gbps speed\n");
		goto _init_ret;
	}

	dev_info(serdes->dev, "SGMIIPLUS2 Core initialized\n");

_init_ret:
	return ret;
}

static void sgmiiplus2_phy_exit(struct sgmiiplus2 *serdes)
{
	sgmiiplus2_media_speed(&serdes->phycore,
			PORT_MII, SPEED_UNKNOWN, DUPLEX_HALF);
	serdes->phycore.core->priv = NULL;

	dev_info(serdes->dev, "SGMIIPLUS2 Core deinitialized\n");
}

static void sgmiiplus2_core_init(struct sgmiiplus2 *serdes)
{
	u32 data;

	/* Power on the core */
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data &= ~(PWRDWN_MASK | IDDQ_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	serdes->powerup = 1;

	/* Wait at least 1ms */
	udelay(1000);

	/* Release resets */
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data |= (SERDES_RESET_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data |= (SERDES_RESET_MASK | RESET_MDIOREGS_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data |= (SERDES_RESET_MASK | RESET_MDIOREGS_MASK | RESET_PLL_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);

	/* Configure MDIO clause and port address (prtad) */
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data |= (MDIO_ST_MASK | (serdes->phyaddr << SERDES_PRTAD_SHIFT));
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);

	/* Configure mux to select Port 7 for SGMII */
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data |= (PORT_7_SOURCE_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
}

static void sgmiiplus2_core_exit(struct sgmiiplus2 *serdes)
{
	u32 data;

	/* Put in reset */
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data &= ~(SERDES_RESET_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);

	/* Power off the core */
	data = __raw_readl(serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	data |= (PWRDWN_MASK | IDDQ_MASK);
	__raw_writel(data, serdes->addr + SGMII0_UNI3_SINGLE_SERDES_CNTRL);
	serdes->powerup = 0;
}

static int sgmiiplus2_get_resources(struct sgmiiplus2 *serdes)
{
	struct resource *res;
	struct device_node *node;
	struct brcmphy_core *phycore = &serdes->phycore;
	int i, ret = 0;

	res = platform_get_resource(serdes->pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(serdes->dev, "Get resource reg failed\n");
		ret = -ENXIO;
		goto _ret_func;
	}

	serdes->addr = devm_ioremap(serdes->dev, res->start,
			resource_size(res));
	if (!serdes->addr) {
		dev_err(serdes->dev, "Unable to obtain register region\n");
		ret = -ENOMEM;
		goto _ret_func;
	}

	ret = of_property_read_u32(serdes->node, "phyaddr", &serdes->phyaddr);
	if (ret < 0) {
		dev_err(serdes->dev, "Unable to obtain phyaddr\n");
		goto _ret_func;
	}

	if (serdes->phyaddr >= PHY_MAX_ADDR) {
		dev_err(serdes->dev, "Invalid phyaddr %d\n", serdes->phyaddr);
		serdes->phyaddr = 0;
		ret = -EINVAL;
		goto _ret_func;
	}

	node = of_parse_phandle(serdes->node, "mux-handle", 0);
	if (!node) {
		dev_err(serdes->dev, "Unable to obtain MDIO mux info\n");
		ret = -ENXIO;
		goto _ret_func;
	}

	of_node_put(node);
	serdes->mux_pdev = of_find_device_by_node(node);
	if (of_property_read_string(serdes->node, "mux-select",
				&serdes->mux_select)) {
		dev_err(serdes->dev, "Unable to obtain MDIO mux select info\n");
		ret = -EINVAL;
		goto _ret_mux_node;
	}

	for (i = 0; i < 2; i++) {
		if (of_property_read_string_index(serdes->node, "mux-values", i,
					&serdes->mux_values[i])) {
			dev_err(serdes->dev, "Unable to obtain MDIO mux values[%d] info\n",
					i);
			ret = -EINVAL;
			goto _ret_mux_node;
		}
	}

	ret = brcmphy_get_busxcvr(serdes->dev, phycore);
	if (ret || !phycore->bus || !phycore->xcvr) {
		dev_err(serdes->dev, "Unable to obtain MDIO bus or PHY XCVR info\n");
		ret = -ENXIO;
		goto _ret_mux_node;
	}

	serdes->phycore.media_speed = sgmiiplus2_media_speed;
	serdes->phycore.suspend = sgmiiplus2_suspend;
	serdes->phycore.resume = sgmiiplus2_resume;
	goto _ret_func;

_ret_mux_node:
	put_device(&serdes->mux_pdev->dev);
	serdes->mux_pdev = NULL;

_ret_func:
	return ret;
}

static void sgmiiplus2_put_resources(struct sgmiiplus2 *serdes)
{
	serdes->phycore.media_speed = NULL;
	serdes->phycore.suspend = NULL;
	serdes->phycore.resume = NULL;
	brcmphy_put_busxcvr(&serdes->phycore);

	if (serdes->mux_pdev)
		put_device(&serdes->mux_pdev->dev);
}

static int sgmiiplus2_probe(struct platform_device *pdev)
{
	struct sgmiiplus2 *serdes;
	unsigned long *supported;
	int ret;

	/* alloc sgmiiplus2 device control block */
	serdes = devm_kzalloc(&pdev->dev, sizeof(struct sgmiiplus2), GFP_KERNEL);
	if (!serdes)
		return -ENOMEM;

	platform_set_drvdata(pdev, serdes);
	serdes->pdev = pdev;
	serdes->dev = &pdev->dev;
	serdes->node = pdev->dev.of_node;
	serdes->phycore.speed = SPEED_2500;
	serdes->phycore.duplex = DUPLEX_FULL;

	ret = sgmiiplus2_get_resources(serdes);
	if (ret)
		return ret;

	sgmiiplus2_core_init(serdes);
	ret = sgmiiplus2_phy_init(serdes);
	if (ret) {
		sgmiiplus2_core_exit(serdes);
		sgmiiplus2_put_resources(serdes);
		return ret;
	}

	supported = serdes->phycore.core->supported;
	linkmode_set_bit(ETHTOOL_LINK_MODE_Pause_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_Asym_Pause_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_Autoneg_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_TP_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_MII_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_FIBRE_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_10baseT_Half_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_10baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_100baseT_Half_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_100baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_1000baseT_Half_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_1000baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_1000baseX_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, supported);
	linkmode_set_bit(ETHTOOL_LINK_MODE_2500baseX_Full_BIT, supported);

	return 0;
}

static int sgmiiplus2_remove(struct platform_device *pdev)
{
	struct sgmiiplus2 *serdes = platform_get_drvdata(pdev);

	if (serdes) {
		sgmiiplus2_phy_exit(serdes);
		sgmiiplus2_core_exit(serdes);
		sgmiiplus2_put_resources(serdes);
	}

	return 0;
}

static void sgmiiplus2_shutdown(struct platform_device *pdev)
{
	sgmiiplus2_remove(pdev);
}

static struct of_device_id sgmiiplus2_of_match[] = {
    {.compatible = "brcm,sgmiiplus2"},
    {}
};
MODULE_DEVICE_TABLE(of, sgmiiplus2_of_match);

static struct platform_driver sgmiiplus2_driver = {
	.probe  = sgmiiplus2_probe,
	.remove = sgmiiplus2_remove,
	.shutdown = sgmiiplus2_shutdown,
	.driver = {
		.name  = MODULE_NAME,
		.owner = THIS_MODULE,
		.of_match_table = sgmiiplus2_of_match,
	},
};

module_platform_driver(sgmiiplus2_driver);

MODULE_AUTHOR("Ravi Patel <ravi.patel@broadcom.com>");
MODULE_DESCRIPTION("Broadcom 28nm SGMIIPLUS2 Serdes/PHY core driver");
MODULE_VERSION(MODULE_VER);
MODULE_LICENSE("GPL v2");
