 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
/****************************************************************************
*	Filename:       bcmunimac_dt.c
*	Author:         Mark Newcomer/Tim Ross
*	Creation Date:  5/6/2013
*	PURPOSE: Common Functions for the Unimac driver.
******************************************************************************/

#include <linux/netdevice.h>
#include <hal_device.h>
#include "bcmunimac_priv.h"

int bcmunimac_parse_dt_prop_u32(struct device_node *of_node,
				const char *propname, u32 *dst)
{
	int status = 0;

	if (!of_property_read_u32(of_node, propname, dst)) {
		pr_debug("%s = %xh\n", propname, *dst);
	} else {
		pr_err("Missing %s property!\n", propname);
		status = -EFAULT;
	}
	return status;
}

int bcmunimac_parse_dt_prop_string(struct device_node *of_node,
								   const char *propname, char **dst)
{
	int status = 0;

	if (!of_property_read_string(of_node, propname, (const char **)dst)) {
		pr_debug("%s = %s\n", propname, *dst);
	} else {
		pr_err("Missing %s property!\n", propname);
		status = -EFAULT;
	}
	return status;
}

int bcmunimac_parse_dt_prop_u32_array(struct device_node *of_node,
									  const char *propname, int cols,
									  int *rows, u32 *dst)
{
	int status = 0;
	struct property *prop;
	int num_entries;
	int i, j;
	const __be32 *ptr;

	prop = of_find_property(of_node, propname, NULL);
	if (!prop) {
		pr_err("Unable to find %s property!\n", propname);
		status = -EFAULT;
		goto err;
	}

	if ((prop->length % cols*sizeof(u32)) != 0) {
		pr_err("%s property is malformed!\n", propname);
		status = -EFAULT;
		goto err;
	}

	num_entries = prop->length / sizeof(u32) / cols;
	if (num_entries > *rows) {
		pr_err("Array provided for %s not large enough!\n", propname);
		status = -EFAULT;
		goto err;
	}
	*rows = num_entries;

	ptr = prop->value;
	for (i = 0; i < *rows; i++) {
		for (j = 0; j < cols; j++)
			dst[i*cols + j] = be32_to_cpup(ptr++);
	}
err:
	return status;
}

int bcmunimac_parse_dt_node(struct platform_device *pdev)
{
	int status = 0;
	struct device *pdd = &pdev->dev;
	struct mac_device *macdev = pdd->platform_data;
	struct unimac_device *qdev = macdev_priv(macdev);
	struct device_node *of_node = pdd->of_node;
	const char *str;
	u32 tmp=0;

	status = haldev_get_id(pdd, NULL, UNIMAC_DEV, UNIMAC_MAX_PORTS, &str);
	if (status < 0)
		goto done;

	if (!haldev_available(of_node, "mac-handle", "mii-handle", NULL)) {
		status = -ENODEV;
		goto done;
	}

	qdev->instance = status;
	macdev->name = str;
	qdev->unimac_irq = platform_get_irq(pdev, 0);
	if (qdev->unimac_irq <= 0) {
		pr_err("Unable to retrieve irq.\n");
		status = -EFAULT;
		goto done;
	}

	status = bcmunimac_parse_dt_prop_u32(of_node, "unimac_core_offset",
						 &tmp);
	if (status) {
		pr_err("Unable to retrieve unimac_core_offset.\n");
		status = -EFAULT;
		goto done;
	} else
		qdev->unimac_core_offset = tmp;

	status = bcmunimac_parse_dt_prop_u32(of_node, "unimac_iface_offset",
						 &tmp);
	if (status) {
		pr_err("Unable to retrieve unimac_iface_offset.\n");
		status = -EFAULT;
		goto done;
	} else
		qdev->unimac_iface_offset = tmp;

	status = bcmunimac_parse_dt_prop_u32(of_node, "unimac_mib_offset",
						 &tmp);
	if (status) {
		pr_err("Unable to retrieve unimac_mib_offset.\n");
		status = -EFAULT;
		goto done;
	} else
		qdev->unimac_mib_offset = tmp;

	status = of_property_read_u32(of_node, "skip_mbdma_init",
						 &tmp);
	if (status) {
		status = 0;
		qdev->skip_mbdma_init = 0;
	}
	else {
		qdev->skip_mbdma_init = tmp;
		pr_debug("Skip MBDMA set to %d.\n",tmp);
	}

	if(!qdev->skip_mbdma_init) {
		status = of_property_read_u32(of_node, "mbdma_use_scb",
							 &tmp);
		if (status) {
			status = 0;
			qdev->use_scb = 1;
		}
		else
			qdev->use_scb = tmp;

		status = of_property_read_u32(of_node, "mbdma_burst_size",
							 &tmp);
		if (status) {
			status = 0;
			qdev->burst_size =
#ifdef MAX_SCB_BURST
				qdev->use_scb ? MAX_SCB_BURST :
				MAX_UBUS_BURST;
#else
			0;
#endif
		}
		else
			qdev->burst_size = tmp;
	}

	status = of_property_read_u32(of_node, "mbdma_channel0_rx_macid",
						 &tmp);
	if (status) {
		pr_debug("Defaulting MBDMA RX Channel0 Id to MACID0.\n");
		status = 0;
		qdev->mbdma_channel0_rx_macid = 0;
	}
	else
		qdev->mbdma_channel0_rx_macid = tmp;

	status = of_property_read_u32(of_node, "mbdma_channel0_tx_macid",
						 &tmp);
	if (status) {
		pr_debug("Defaulting MBDMA TX Channel0 Id to MACID0.\n");
		status = 0;
		qdev->mbdma_channel0_tx_macid = 0;
	}
	else
		qdev->mbdma_channel0_tx_macid = tmp;

	status = of_property_read_u32(of_node, "mbdma_channel1_rx_macid",
						 &tmp);
	if (status) {
		pr_debug("Defaulting MBDMA RX Channel1 Id to MACID0.\n");
		status = 0;
		qdev->mbdma_channel1_rx_macid = 1;
	}
	else
		qdev->mbdma_channel1_rx_macid = tmp;

	status = of_property_read_u32(of_node, "mbdma_channel1_tx_macid",
						 &tmp);
	if (status) {
		pr_debug("Defaulting MBDMA TX Channel1 Id to MACID1.\n");
		status = 0;
		qdev->mbdma_channel1_tx_macid = 1;
	}
	else
		qdev->mbdma_channel1_tx_macid = tmp;

	status = of_property_read_u32(of_node, "rgmii_id_mode_disable",
						 &tmp);
	if (status) {
		pr_debug("Defaulting rgmii_id_mode_disable to 1.\n");
		status = 0;
		qdev->rgmii_id_mode_disable = 1;
	}
	else
		qdev->rgmii_id_mode_disable = tmp;

	status = of_property_read_u32(of_node, "tx_clk_dly",
						 &tmp);
	if (status) {
		pr_debug("Defaulting tx_clk_dly to 4.\n");
		status = 0;
		qdev->tx_clk_dly = 4;
	}
	else
		qdev->tx_clk_dly = tmp;

	status = of_property_read_u32(of_node, "rx_clk_dly",
						 &tmp);
	if (status) {
		pr_debug("Defaulting rx_clk_dly to 4.\n");
		status = 0;
		qdev->rx_clk_dly = 4;
	}
	else
		qdev->rx_clk_dly = tmp;

done:
	return status;
}
