 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/if_arp.h>
#include <linux/netfilter_bridge.h>
#include <net/netfilter/nf_conntrack_acct.h>
#include "flowmgr.h"
#include "flowmgr_fap.h"
#include "flowmgr_fap_ops.h"
#include "flowmgr_tunnel.h"
#include "bcmnethooks.h"
#include "proc_cmd.h"
#include "mso_feat.h"

static struct delayed_work tu_port_dwork ;
static struct flowmgr_tu_port_state tu_port_track[65536];

static struct mso_feat *feat;

static int mso_feat_init(void)
{
	get_mso_feat_fptr get_mso_feat;

	if (!feat) {
		get_mso_feat = get_mso_feat_symbol();
		if (!get_mso_feat)
			return -1;

		feat = get_mso_feat(mf_dslite);
		if (!feat)
			return -1;
	}
	return 0;
}

void flowmgr_tu_port_update(u16 dport)
{
	tu_port_track[dport].ts = jiffies;
}

void flowmgr_tu_port_clear(u16 dport)
{
	tu_port_track[dport].ts = 0;
}

int flowmgr_tu_port_test(u16 dport)
{
	if (tu_port_track[dport].ts)
		return 1;
	return 0;
}

struct flowmgr_tu_port_state *flowmgr_tu_port_table(void)
{
	return tu_port_track;
}
EXPORT_SYMBOL(flowmgr_tu_port_table);

void flowmgr_tu_port_show(struct seq_file *s)
{
	int i, first = 1;
	if (!flowmgr.enable_tu_port_track)
		return;
	for (i = 0; i < 65536; i++) {
		if (tu_port_track[i].ts) {
			if (first)
				pr_seq(s, "Open TCP/UDP port usage: Clear timeout %u\n",
				       flowmgr.tu_port_track_timeout);
			pr_seq(s, "%-5u: %lu\n", i,
			       tu_port_track[i].ts);
			first = 0;
		}
	}
}

void flowmgr_demote_tu_port_update(struct nf_conn *ct,
				   struct nf_conn_offload *ct_offload)
{
	if (!flowmgr.enable_tu_port_track)
		return;
	if (ct_offload_orig.tu_port)
		flowmgr_tu_port_update(ct_offload_orig.tu_port);
	if (ct_offload_repl.tu_port)
		flowmgr_tu_port_update(ct_offload_repl.tu_port);
}

static char *flow_directions[] = {
	"Lan2Lan",
	"Wan2Lan",
	"Lan2Wan",
	"Expected"
};

char *flowmgr_get_directions(int index)
{
	return flow_directions[index];
}

void flowmgr_fap_dbg_show(struct seq_file *s)
{
	pr_seq(s, "FAP Type: %u (%s)\n", fap()->type, fap()->name);
	if (fap()->dump)
		fap()->dump(s);
	pr_seq(s, "----------------------\n");
}

int flowmgr_fap_flows_active(void)
{
	return atomic_read(&flowmgr.flow_create_cnt) !=
		atomic_read(&flowmgr.flow_delete_cnt);
}

#define dump_bridge(pr_func, label, mark, info, params)\
do {\
	struct net_device *in = NULL;\
	struct net_device *out = NULL;\
	in = __dev_get_by_index(&init_net, info->iif);\
	out = __dev_get_by_index(&init_net, info->oif);\
	pr_func("FLOWMGR: %s mark(%X)\n",\
		 label, mark);\
	pr_func("    in_dev(%s) vlan untag(%d) out_dev(%s) vlan tag(%d)\n",\
		 in ? in->name:"null", info->vlan_untag,\
		 out ? out->name:"null", info->vlan_id);\
	pr_func("    src_mac %pM\n",\
		 params->mac_src);\
	pr_func("    dst_mac %pM\n",\
		 params->mac_dst);\
	pr_func("    rx_interface  %d\n",\
		 params->rx_interface);\
	pr_func("    rx_subif      %d\n",\
		 params->rx_subif);\
} while (0)

#define dump_ipv4(pr_func, label, mark, info, params)\
do {\
	struct net_device *in = NULL;\
	struct net_device *out = NULL;\
	in = __dev_get_by_index(&init_net, info->iif);\
	out = __dev_get_by_index(&init_net, info->oif);\
	pr_func("FLOWMGR: %s mark(%X)\n",\
		 label, mark);\
	pr_func("    in_dev(%s) vlan untag(%d) out_dev(%s) vlan tag(%d)\n",\
		 in ? in->name:"null", info->vlan_untag,\
		 out ? out->name:"null", info->vlan_id);\
	if (params->type != ft_ipv4_expected) {\
		pr_func("    rep_src_mac %pM\n",\
			 params->replacement_mac_src);\
		pr_func("    rep_dst_mac %pM\n",\
			 params->replacement_mac_dst);\
	} \
	pr_func("    wifi flowring %d pri %d\n",\
		 info->wifi_flowring, info->wifi_pri);\
	pr_func("    decap_tid %d encap_tid %d\n",\
		 info->decap_tid, info->encap_tid);\
	pr_func("    ip_prot  %d\n",\
		 params->ip_prot);\
	if ((params->type == ft_ipv4_nat_src) ||\
	    (params->type == ft_ipv4_gre_decap_nat_src) ||\
	    (params->type == ft_ipv4_gre_encap_nat_src)) {\
		pr_func("    src_ip   %pI4 <- %pI4\n",\
			 &params->src_ip,\
			 &params->replacement_ip);\
		pr_func("    dst_ip   %pI4\n",\
			 &params->dst_ip);\
		pr_func("    src_port %05d <- %05d\n",\
			 ntohs(params->src_port),\
			 ntohs(params->replacement_port));\
		pr_func("    dst_port %05d\n",\
			 ntohs(params->dst_port));\
	} else if ((params->type == ft_ipv4_nat_dst) ||\
		   (params->type == ft_ipv4_gre_decap_nat_dst) ||\
		   (params->type == ft_ipv4_gre_encap_nat_dst)) {\
		pr_func("    src_ip   %pI4\n",\
			 &params->src_ip);\
		pr_func("    dst_ip   %pI4 <- %pI4\n",\
			 &params->dst_ip,\
			 &params->replacement_ip);\
		pr_func("    src_port %05d\n",\
			 ntohs(params->src_port));\
		pr_func("    dst_port %05d <- %05d\n",\
			 ntohs(params->dst_port),\
			 ntohs(params->replacement_port));\
	} else if ((params->type == ft_ipv4_mapt_map) ||\
		   (params->type == ft_ipv4_mapt_gre2map)) {\
		pr_func("    mapt domain   %d\n",\
			 params->mapt.domain_index);\
		pr_func("    v4:\n");\
		pr_func("        src_ip   %pI4 <- %pI4\n",\
			 &params->src_ip,\
			 &params->replacement_ip);\
		pr_func("        dst_ip   %pI4\n",\
			 &params->dst_ip);\
		pr_func("        src_port %05d <- %05d\n",\
			 ntohs(params->src_port),\
			 ntohs(params->replacement_port));\
		pr_func("        dst_port %05d\n",\
			 ntohs(params->dst_port));\
		pr_func("    v6:\n");\
		pr_func("        src_ip   %pI6c\n",\
			 params->mapt.ipv6_src_ip);\
		pr_func("        dst_ip   %pI6c\n",\
			 params->mapt.ipv6_dst_ip);\
	} else {\
		pr_func("    src_ip   %pI4\n",\
			 &params->src_ip);\
		pr_func("    dst_ip   %pI4\n",\
			 &params->dst_ip);\
		pr_func("    src_port %05d\n",\
			 ntohs(params->src_port));\
		pr_func("    dst_port %05d\n",\
			 ntohs(params->dst_port));\
	} \
} while (0)

void flowmgr_dump_ipv4(char *label, int mark,
		       struct offload_info *info,
		       struct flow_ipv4_params *params)
{
	if (info->debug)
		dump_ipv4(pr_alert, label, mark, info, params);
}

#define dump_ipv6(pr_func, label, mark, info, params)\
do {\
	struct net_device *in = NULL;\
	struct net_device *out = NULL;\
	in = __dev_get_by_index(&init_net, info->iif);\
	out = __dev_get_by_index(&init_net, info->oif);\
	pr_func("FLOWMGR: %s mark(%X)\n",\
		 label, mark);\
	pr_func("    in_dev(%s) vlan untag(%d) out_dev(%s) vlan tag(%d)\n",\
		 in ? in->name:"null", info->vlan_untag,\
		 out ? out->name:"null", info->vlan_id);\
	if (params->type != ft_ipv6_expected) {\
		pr_func("    rep_src_mac %pM\n",\
			 params->replacement_mac_src);\
		pr_func("    rep_dst_mac %pM\n",\
			 params->replacement_mac_dst);\
	} \
	pr_func("    wifi flowring %d pri %d\n",\
		 info->wifi_flowring, info->wifi_pri);\
	pr_func("    decap_tid %d encap_tid %d\n",\
		 info->decap_tid, info->encap_tid);\
	pr_func("    ip_prot  %d\n",\
		 params->ip_prot);\
	if ((params->type == ft_ipv6_nat_src) ||\
	    (params->type == ft_ipv6_gre_decap_nat_src) ||\
	    (params->type == ft_ipv6_gre_encap_nat_src)) {\
		pr_func("    src_ip   %pI6c <- %pI6c\n",\
			params->src_ip,\
			params->replacement_ip);\
		pr_func("    dst_ip   %pI6c\n",\
			params->dst_ip);\
		pr_func("    src_port %05d <- %05d\n",\
			ntohs(params->src_port),\
			ntohs(params->replacement_port));\
		pr_func("    dst_port %05d\n",\
			ntohs(params->dst_port));\
	} else if ((params->type == ft_ipv6_nat_dst) ||\
		   (params->type == ft_ipv6_gre_decap_nat_dst) ||\
		   (params->type == ft_ipv6_gre_encap_nat_dst)) {\
		pr_func("    src_ip   %pI6c\n",\
			params->src_ip);\
		pr_func("    dst_ip   %pI6c <- %pI6c\n",\
			params->dst_ip,\
			params->replacement_ip);\
		pr_func("    src_port %05d\n",\
			ntohs(params->src_port));\
		pr_func("    dst_port %05d <- %05d\n",\
			ntohs(params->dst_port),\
			ntohs(params->replacement_port));\
	} else if ((params->type == ft_ipv6_mapt_unmap) ||\
		   (params->type == ft_ipv6_mapt_unmap2gre)) {\
		pr_func("    src_ip   %pI6c\n",\
			params->src_ip);\
		pr_func("    dst_ip   %pI6c\n",\
			params->dst_ip);\
		pr_func("    src_port %05d\n",\
			ntohs(params->src_port));\
		pr_func("    dst_port %05d\n",\
			ntohs(params->dst_port));\
		pr_func("    mapt domain   %d\n",\
			params->mapt.domain_index);\
			pr_func("        src_ip   %pI4\n",\
				&params->mapt.src_ip);\
			pr_func("        dst_ip   %pI4\n",\
				&params->mapt.dst_ip);\
			pr_func("        dst_port %05d\n",\
				ntohs(params->mapt.dst_port));\
	} else {\
		pr_func("    src_ip   %pI6c\n",\
			 params->src_ip);\
		pr_func("    dst_ip   %pI6c\n",\
			 params->dst_ip);\
		pr_func("    src_port %05d\n",\
			 ntohs(params->src_port));\
		pr_func("    dst_port %05d\n",\
			 ntohs(params->dst_port));\
	} \
} while (0)

void flowmgr_dump_ipv6(char *label, int mark,
		       struct offload_info *info,
		       struct flow_ipv6_params *params)
{
	if (info->debug)
		dump_ipv6(pr_alert, label, mark, info, params);
}

void flowmgr_dump_flowid(struct offload_info *ct_offload_info, char *label, int flow_id)
{
	if (ct_offload_info->debug)
		pr_alert("%s: %d ct %px\n", label, flow_id, ct_offload_info->ct);
}

void flowmgr_update_stats(const struct nf_conn *ct)
{
	flowmgr_update_flow_stats(ct, IP_CT_DIR_ORIGINAL, 1);
	flowmgr_update_flow_stats(ct, IP_CT_DIR_REPLY, 1);
}

void flowmgr_flow_usage(struct offload_info *ct_offload_info)
{
	struct flowmgr_flow_usage *usage;
	if (!flowmgr.flow_usage)
		return;
	usage = &flowmgr.flow_usage[ct_offload_info->flow_id];
	jiffies_to_timeval(jiffies - ct_offload_info->tstamp, &usage->ltv);
	if (usage->ltv.tv_sec > usage->mtv.tv_sec)
		 usage->mtv.tv_sec =  usage->ltv.tv_sec;
	usage->del++;
}


#ifdef CONFIG_BCM_FLOWMGR_FAP_WQ
struct flowmgr_flow_work {
	struct work_struct work;
	union {
		struct flow_params params;
		struct {
			int flow_id;
			int ctdebug;
		};
		int tunnel_id;
	};
};

static void flowmgr_flow_add_deferred(struct work_struct *w)
{
	int ret, flow_id;
	struct offload_info *ct_offload_info;
	struct flowmgr_flow_work *work;
	work = container_of(w, struct flowmgr_flow_work, work);
	ct_offload_info = work->params.data;
	ret = fap()->flow_add(&work->params, &flow_id);
	if (!ret) {
		ct_offload_info->flow_id = flow_id;
		atomic_inc(&flowmgr.flow_create_cnt);
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Flow (type=%d|%d) deferred added %d ct %px\n",
				 work->params.type,
				 work->params.ipv4.type,
				 flow_id,  ct_offload_info->ct);
	} else {
		ct_offload_info->create_err = 1;
		ct_offload_info->flow_id = -1;
		atomic_inc(&flowmgr.flow_create_err_cnt);
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - fap %d\n", ret);
	}
	nf_conntrack_put(&ct_offload_info->ct->ct_general);
	kfree(work);
}

int flowmgr_flow_add(struct flow_params *params,
		     int *flow_id,
		     struct offload_info *ct_offload_info)
{
	int ret = 0;
	struct flowmgr_flow_work *work;
	work = kmalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;
	params->data = ct_offload_info;
	ct_offload_info->packets_promote = ct_offload_info->packets_slow;
	nf_conntrack_get(&ct_offload_info->ct->ct_general);
	INIT_WORK(&work->work, flowmgr_flow_add_deferred);
	work->params = *params;
	*flow_id = 0xDEFE22ED;
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}

static void flowmgr_flow_add_expected_deferred(struct work_struct *w)
{
	int ret, flow_id;
	struct offload_info *ct_offload_info;
	struct flowmgr_flow_work *work;
	work = container_of(w, struct flowmgr_flow_work, work);
	ct_offload_info = work->params.data;
	ret = fap()->flow_add_expected(&work->params, &flow_id);
	if (!ret) {
		ct_offload_info->flow_id = flow_id;
		atomic_inc(&flowmgr.flow_expected_create_cnt);
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Flow expected (type=%d|%d) deferred added: %d ct %px\n",
				 work->params.type,
				 work->params.ipv4.type,
				 flow_id, ct_offload_info->ct);
	} else {
		ct_offload_info->create_err = 1;
		atomic_inc(&flowmgr.flow_expected_create_err_cnt);
	}
	nf_conntrack_put(&ct_offload_info->ct->ct_general);
	kfree(work);
}

int flowmgr_flow_add_expected(struct flow_params *params,
			      int *flow_id,
			      struct offload_info *ct_offload_info)
{
	int ret = 0;
	struct flowmgr_flow_work *work;
	work = kmalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;
	params->data = ct_offload_info;
	nf_conntrack_get(&ct_offload_info->ct->ct_general);
	INIT_WORK(&work->work, flowmgr_flow_add_expected_deferred);
	work->params = *params;
	*flow_id = 0xDEFE22ED;
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}

int flowmgr_flow_add_multicast(struct flow_params *params,
			       int *flow_id,
			       void *data)
{
	int ret = 0;
	params->data = data;
	ret = fap()->flow_add(params, flow_id);
	if (!ret)
		atomic_inc(&flowmgr.flow_mcast_create_cnt);
	return ret;
}

static void flowmgr_flow_remove_deferred(struct work_struct *w)
{
	int ret, flow_id;
	struct flowmgr_flow_work *work;
	work = container_of(w, struct flowmgr_flow_work, work);
	flow_id = work->flow_id;
	ret = fap()->flow_remove(flow_id);
	if (!ret) {
		atomic_inc(&flowmgr.flow_delete_cnt);
		if (work->ctdebug) {
			pr_alert("FLOWMGR: Flow deferred deleted: %d\n",
				 flow_id);
		}
	} else {
		//atomic_inc(&flowmgr.flow_delete_err_cnt);
	}
	kfree(work);
}

int flowmgr_flow_remove(int ctdebug, int flow_id)
{
	int ret = 0;
	struct flowmgr_flow_work *work;

	if ((flow_id < 0) || (flow_id >= fap()->flow_max())) {
		pr_alert("FLOWMGR: Flow demote: flow_id %d oob\n",
			 flow_id);
		atomic_inc(&flowmgr.flow_delete_err_cnt);
		return -1;
	}

	work = kmalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;
	work->flow_id = flow_id;
	work->ctdebug = ctdebug;
	INIT_WORK(&work->work, flowmgr_flow_remove_deferred);
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}

static void flowmgr_flow_remove_expected_deferred(struct work_struct *w)
{
	int ret, flow_id;
	struct flowmgr_flow_work *work;
	work = container_of(w, struct flowmgr_flow_work, work);
	flow_id = work->flow_id;
	ret = fap()->flow_remove(flow_id);
	if (!ret) {
		atomic_inc(&flowmgr.flow_expected_delete_cnt);
		if (work->ctdebug)
			pr_alert("FLOWMGR: Flow expected deferred deleted: %d\n",
				 flow_id);
	} else {
		atomic_inc(&flowmgr.flow_expected_delete_err_cnt);
	}
	kfree(work);
}

int flowmgr_flow_remove_expected(int ctdebug, int flow_id)
{
	int ret = 0;
	struct flowmgr_flow_work *work;

	if ((flow_id < 0) || (flow_id >= fap()->flow_max())) {
		pr_alert("FLOWMGR: Flow expected demote: flow_id %d oob\n",
			flow_id);
		atomic_inc(&flowmgr.flow_expected_delete_err_cnt);
		return -1;
	}

	work = kmalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;
	work->flow_id = flow_id;
	work->ctdebug = ctdebug;
	INIT_WORK(&work->work, flowmgr_flow_remove_expected_deferred);
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}
#else
int flowmgr_flow_add(struct flow_params *params,
		     int *flow_id,
		     struct offload_info *data)
{
	int ret;
	params->data = data;
	ret = fap()->flow_add(params, flow_id);
	if (!ret)
		atomic_inc(&flowmgr.flow_create_cnt);
	else {
		data->create_err = 1;
		atomic_inc(&flowmgr.flow_create_err_cnt);
		if (data->debug)
			pr_alert("FLOWMGR: Promotion error fap %d\n", ret);
	}
	return ret;
}

int flowmgr_flow_add_expected(struct flow_params *params,
			      int *flow_id,
			      struct offload_info *data)
{
	int ret;
	params->data = data;
	ret = fap()->flow_add_expected(params, flow_id);
	if (!ret) {
		atomic_inc(&flowmgr.flow_expected_create_cnt);
	} else {
		data->create_err = 1;
		atomic_inc(&flowmgr.flow_expected_create_err_cnt);
	}
	return ret;
}

int flowmgr_flow_add_multicast(struct flow_params *params,
			       int *flow_id,
			       struct flowmgr_mdb_entry *data)
{
	int ret = 0;
	params->data = data;
	ret = fap()->flow_add(params, flow_id);
	if (!ret)
		atomic_inc(&flowmgr.flow_mcast_create_cnt);
	return ret;
}

int flowmgr_flow_remove(int ctdebug, int flow_id)
{
	int ret;

	if ((flow_id < 0) || (flow_id >= fap()->flow_max())) {
		pr_alert("FLOWMGR: Flow demote: flow_id %d oob\n",
			 flow_id);
		atomic_inc(&flowmgr.flow_delete_err_cnt);
		return -1;
	}

	ret = fap()->flow_remove(flow_id);
	if (!ret)
		atomic_inc(&flowmgr.flow_delete_cnt);
	return ret;
}

int flowmgr_flow_remove_expected(int ctdebug, int flow_id)
{
	int ret;

	if ((flow_id < 0) || (flow_id >= fap()->flow_max())) {
		pr_alert("FLOWMGR: Flow expected demote: flow_id %d oob\n",
			 flow_id);
		atomic_inc(&flowmgr.flow_expected_delete_err_cnt);
		return -1;
	}

	ret = fap()->flow_remove(flow_id);
	if (!ret)
		atomic_inc(&flowmgr.flow_expected_delete_cnt);
	else
		atomic_inc(&flowmgr.flow_expected_delete_err_cnt);
	return ret;
}
#endif

void flowmgr_expected_demote(struct offload_info *ct_offload_info)
{
	struct timeval tv;

	if (ct_offload_info->flow_id == -1)
		return;

	if (!ct_offload_info->expected)
		return;

	ct_offload_info->expected = 0;
	jiffies_to_timeval(jiffies - ct_offload_info->tstamp, &tv);
	if (ct_offload_info->debug)
		pr_alert("FLOWMGR: Flow expected removed: %d ct %px duration %ld sec.\n",
			 ct_offload_info->flow_id,  ct_offload_info->ct, tv.tv_sec);
	flowmgr_flow_remove_expected(ct_offload_info->debug,
				ct_offload_info->flow_id);
	flowmgr_flow_usage(ct_offload_info);
	ct_offload_info->flow_id = -1;
	ct_offload_info->flow_type = 0;
	ct_offload_info->tstamp = 0;
	ct_offload_info->bytes = 0;
	ct_offload_info->packets[0] = 0;
	ct_offload_info->packets[1] = 0;
	ct_offload_info->create_err = 0;
}

void flowmgr_demote(struct nf_conn *ct, struct nf_conn_offload *ct_offload)
{
	flowmgr_demote_tu_port_update(ct, ct_offload);
	flowmgr_demote_single(ct, ct_offload, IP_CT_DIR_ORIGINAL);
	flowmgr_demote_single(ct, ct_offload, IP_CT_DIR_REPLY);
}

void flowmgr_demote_single(struct nf_conn *ct, struct nf_conn_offload *ct_offload, int direction)
{
	struct nf_conn_offload *ct_offload_master = NULL;
	struct timeval tv;
	struct nf_bridge_info *nf_bridge = NULL;
	int ret = -1;
	struct offload_info *ct_offload_info;

	if (!ct_offload)
		return;

	spin_lock_bh(&ct->lock);
	ct_offload_info = &ct_offload->info[direction];

	/* reset flow trigger flags for manual promotion */
	flowmgr_manual_trigger_clear(ct, ct_offload_info, direction);

	if (ct_offload_info->flow_id == -1)
		goto done;

	if (!ct_offload_info->flow_type)
		goto done;

	if (ct_offload_info->expected) {
		flowmgr_trace_add_entry(3, &ct->tuplehash[direction].tuple, ct_offload_info);
		flowmgr_expected_demote(ct_offload_info);
		goto done;
	}

	jiffies_to_timeval(jiffies - ct_offload_info->tstamp, &tv);
	if (ct_offload_info->debug)
		pr_alert("FLOWMGR: Flow %s removed: %d ct %px duration %ld sec.\n",
			 ((direction == IP_CT_DIR_ORIGINAL) ? "orig" : "repl"),
			 ct_offload_info->flow_id,  ct_offload_info->ct, tv.tv_sec);
	flowmgr_update_flow_stats(ct, direction, 1);
	flowmgr_trace_add_entry(1, &ct->tuplehash[direction].tuple, ct_offload_info);
	ret = flowmgr_flow_remove(ct_offload_info->debug,
				  ct_offload_info->flow_id);
	if (ret < 0)
		atomic_inc(&flowmgr.flow_delete_err_cnt);
	flowmgr_db_dec_rxflow(ct_offload_info->idb);
	flowmgr_db_dec_txflow(ct_offload_info->odb);
	flowmgr_flow_usage(ct_offload_info);
	nf_bridge = ct_offload_info->nf_bridge;
	memset(ct_offload_info, 0, offsetof(struct offload_info, nf_bridge));
	ct_offload_info->ctinfo = -1;
	ct_offload_info->flow_id = -1;
	ct_offload_info->nf_bridge = nf_bridge;
	ct_offload_info->wifi_flowring = 0xFFFF;
	if (ct->master) {
		if (unlikely(!refcount_inc_not_zero(&ct->master->ct_general.use)))
			goto done;
		ct_offload_master = nf_conn_offload_find(ct->master);
		if (ct_offload_master && atomic_read(&ct_offload_master->slavecnt))
			atomic_dec(&(ct_offload_master->slavecnt));
		nf_ct_put(ct->master);
	}
done:
	spin_unlock_bh(&ct->lock);
}

void flowmgr_update_ct_offload(int flow_id, int flow_type, int lag,
			       struct nf_conntrack_tuple *tuple_match,
			       struct offload_info *ct_offload_info,
			       struct nf_conn_offload *ct_offload,
			       struct nf_conn_offload *ct_offload_master)
{
	ct_offload_info->lag = lag;
	ct_offload_info->flow_id = flow_id;
	ct_offload_info->flow_type = flow_type;
	ct_offload->destructor = flowmgr_demote;
	ct_offload->update_stats = flowmgr_update_stats;
	flowmgr_db_inc_rxflow(ct_offload_info->idb);
	flowmgr_db_inc_txflow(ct_offload_info->odb);
	if (ct_offload_master)
		atomic_inc(&(ct_offload_master->slavecnt));
	if (flowmgr.flow_usage)
		flowmgr.flow_usage[flow_id].add++;
	flowmgr_trace_add_entry(0, tuple_match, ct_offload_info);
}

void flowmgr_update_dscp_params(int dir, struct nf_conn *ct,
				struct offload_info *ct_offload_info,
				struct flow_tx_control *tx_control)
{
	if (ct_offload_info->dscp_old != ct_offload_info->dscp_new) {
		tx_control->dscp_mark = 1;
		tx_control->dscp_val = ct_offload_info->dscp_new;
	} else if ((flowmgr.ctmark_dscp_override & dir) && ct->mark) {
		tx_control->dscp_mark = 1;
		tx_control->dscp_val = ct_offload_info->dscp_new;
		ct_offload_info->dscp_ovr = 1;
	}
}

static
void update_ct_expected(int flow_id, int flow_type,
			struct nf_conntrack_tuple *tuple_match,
			struct offload_info *ct_offload_info,
			struct nf_conn_offload *ct_offload)
{
	ct_offload_info->flow_id = flow_id;
	ct_offload_info->flow_type = flow_type;
	ct_offload_info->expected = 1;
	ct_offload->destructor = flowmgr_demote;
	ct_offload_info->tstamp = jiffies;
	if (flowmgr.flow_usage)
		flowmgr.flow_usage[flow_id].add++;
	flowmgr_trace_add_entry(2, tuple_match, ct_offload_info);
}

int flowmgr_expected_promote(int dir, int prot,
			     struct nf_conn *ct,
			     struct nf_conntrack_tuple *tuple_match,
			     int ctdir,
			     struct offload_info *ct_offload_info)
{
	int flow_id = -1, ret = -1;
	struct flow_params params;
	struct nf_conn_offload *ct_offload = NULL;
	struct net_device *dev_in;

	if (!fap()->flow_add_expected)
		return -1;

	ct_offload = nf_conn_offload_find(ct);
	dev_in = __dev_get_by_index(&init_net, ct_offload_info->iif);
	if (!dev_in) {
		pr_err("flowmgr_expected_promote: error dev in %s dir %d prot %d sport %d dport %d %pM %pM ctinfo %d orig %d repl %d\n",
			   dev_in?dev_in->name:"null",
			   dir, prot,
			   ntohs(tuple_match->src.u.tcp.port),
			   ntohs(tuple_match->dst.u.tcp.port),
			   ct_offload_info->eh.h_source,
			   ct_offload_info->eh.h_dest,
			   ct_offload_info->ctinfo,
			   ct_offload_info == &ct_offload_orig,
			   ct_offload_info == &ct_offload_repl);
		return 0;
	}

	if (!tuple_match->src.u.tcp.port || !tuple_match->dst.u.tcp.port) {
		pr_debug("flowmgr_expected_promote: error dev in %s dir %d prot %d sport %d dport %d %pM %pM ctinfo %d orig %d repl %d\n",
			   dev_in?dev_in->name:"null",
			   dir, prot,
			   ntohs(tuple_match->src.u.tcp.port),
			   ntohs(tuple_match->dst.u.tcp.port),
			   ct_offload_info->eh.h_source,
			   ct_offload_info->eh.h_dest,
			   ct_offload_info->ctinfo,
			   ct_offload_info == &ct_offload_orig,
			   ct_offload_info == &ct_offload_repl);
		atomic_inc(&flowmgr.pkt_ignore_0port_cnt);
		return 0;
	}

	if (flowmgr.enable_tu_port_track) {
		ct_offload_info->tu_port = ntohs(tuple_match->dst.u.tcp.port);
		flowmgr_tu_port_update(ct_offload_info->tu_port);
	}

	if ((prot == IPPROTO_ESP) || (prot == IPPROTO_AH)) {
		struct flow_ipsec_params *ipsec_params = &params.ipsec;
		ipsec_params->spi = ct_offload_info->spi;
	}

	if (nf_ct_l3num(ct) == AF_INET6) {
		/* IPv6 Routed */
		struct flow_ipv6_params *ipv6_params = &params.ipv6;
		params.type = ft_ipv6;
		ipv6_params->type = ft_ipv6_expected;
		ipv6_params->ip_prot = prot;
		memcpy(ipv6_params->src_ip,
		       tuple_match->src.u3.ip6, 16);
		memcpy(ipv6_params->dst_ip,
		       tuple_match->dst.u3.ip6, 16);
		ipv6_params->src_port =
			tuple_match->src.u.tcp.port;
		ipv6_params->dst_port =
			tuple_match->dst.u.tcp.port;
		memcpy(ipv6_params->replacement_mac_src,
		       ct_offload_info->eh.h_source, ETH_ALEN);
		memcpy(ipv6_params->replacement_mac_dst,
		       ct_offload_info->eh.h_dest, ETH_ALEN);
		memset(&params.tx, 0, sizeof(struct flow_tx_params));
		memset(&params.rx, 0, sizeof(struct flow_rx_params));
		params.rx.interface = dev_in->ifindex;
		params.debug = ct_offload_info->debug;
		flowmgr_dump_ipv6(flow_directions[dir],
			  ct->mark, ct_offload_info, ipv6_params);
		ret = flowmgr_flow_add_expected(&params, &flow_id, ct_offload_info);
		if (ret != 0)
			goto err_out;
		flowmgr_dump_flowid(ct_offload_info,
				    "Flow (ipv6 expected) added",
				    flow_id);
		update_ct_expected(flow_id,
				  ft_ipv6 << 16 | ipv6_params->type,
				  tuple_match,
				  ct_offload_info,
				  ct_offload);
		return 0;
	} else if (nf_ct_l3num(ct) == AF_INET) {
		/* IPv4 Routed */
		struct flow_ipv4_params *ipv4_params = &params.ipv4;
		params.type = ft_ipv4;
		ipv4_params->type = ft_ipv4_expected;
		ipv4_params->ip_prot = prot;
		ipv4_params->src_ip   = tuple_match->src.u3.ip;
		ipv4_params->dst_ip   = tuple_match->dst.u3.ip;
		ipv4_params->src_port =
			tuple_match->src.u.tcp.port;
		ipv4_params->dst_port =
			tuple_match->dst.u.tcp.port;
		memcpy(ipv4_params->replacement_mac_src,
		       ct_offload_info->eh.h_source, ETH_ALEN);
		memcpy(ipv4_params->replacement_mac_dst,
		       ct_offload_info->eh.h_dest, ETH_ALEN);
		ipv4_params->replacement_ip = 0;
		ipv4_params->replacement_port = 0;
		memset(&params.tx, 0, sizeof(struct flow_tx_params));
		memset(&params.rx, 0, sizeof(struct flow_rx_params));
		params.rx.interface = dev_in->ifindex;
		params.debug = ct_offload_info->debug;
		flowmgr_dump_ipv4(flow_directions[dir],
			  ct->mark, ct_offload_info, ipv4_params);
		ret = flowmgr_flow_add_expected(&params, &flow_id, ct_offload_info);
		if (ret != 0)
			goto err_out;
		flowmgr_dump_flowid(ct_offload_info,
				    "Flow (ipv4 expected) added",
				    flow_id);
		update_ct_expected(flow_id,
				  ft_ipv4 << 16 | ipv4_params->type,
				  tuple_match,
				  ct_offload_info,
				  ct_offload);
		return 0;
	}
err_out:
	if (ct_offload_info->debug)
		pr_alert("FLOWMGR: Promotion error - expected flow fap error %d\n", ret);
	return ret;
}

bool flowmgr_is_direction_allowed(int dir)
{
	return flowmgr.promote_dir_ctl & (1<<dir);
}

int flowmgr_promote_gre2dslite(int dir, int prot, int pri,
			       struct nf_conn *ct,
			       struct nf_conntrack_tuple *tuple_match,
			       struct nf_conntrack_tuple *tuple_replacement,
			       struct net_device *dev_in,
			       struct net_device *dev_out,
			       struct nf_conn_offload *ct_offload,
			       struct nf_conn_offload *ct_offload_master,
			       struct offload_info *ct_offload_info)
{
	int flow_id = -1, ret = -1;
	struct flow_params params;
	struct flow_ipv4_params *ipv4_params = &params.ipv4;
	struct flow_tunnel_params tunnel;
	struct flowmgr_tdb_entry *tdb;

	mso_feat_init();
	tdb = flowmgr_tdb_find(dev_in->ifindex);
	if (!tdb) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - gre2dslite input gre tunnel not found\n");
		ct_offload_info->create_err = 1;
		return -1;
	}
	/* DSLite encap and gre decap */
	if (ct_offload_info->debug)
		pr_debug("FLOWMGR: dsite encap and gre decap\n");
	params.type = ft_ipv4;
	ipv4_params->ip_prot  = prot;
	ipv4_params->src_ip   = tuple_match->src.u3.ip;
	ipv4_params->dst_ip   = tuple_match->dst.u3.ip;
	ipv4_params->src_port = tuple_match->src.u.tcp.port;
	ipv4_params->dst_port = tuple_match->dst.u.tcp.port;
	ipv4_params->replacement_ip   =
		tuple_replacement->src.u3.ip;
	ipv4_params->replacement_port =
		tuple_replacement->src.u.tcp.port;

	memset(&params.rx, 0, sizeof(struct flow_rx_params));
	memset(&params.tx, 0, sizeof(struct flow_tx_params));
	params.rx.interface =
		flowmgr_tunnel_out_ifindex(dev_in,
					   &tunnel,
					   NULL);
	if (params.rx.interface == 0) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - gre2dslite input gre rx device not set\n");
		ct_offload_info->create_err = 1;
		return -1;
	}

	params.wifi.skb_pri = pri & 7;
	params.tx.interface =
		flowmgr_tunnel_out_ifindex(dev_out,
					   &tunnel,
					   &params.wifi);
	if (params.tx.interface == 0) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - gre2dslite output dslite tx device not set\n");
		ct_offload_info->create_err = 1;
		return -1;
	}
	params.tx.control.priority = pri;

	memcpy(&params.encap_tun, &tunnel,
	       sizeof(struct flow_tunnel_params));
	ct_offload_info->decap_tid = tdb->id;
	params.debug = ct_offload_info->debug;

	if (feat && feat->dslite && feat->dslite->ops1) {
		ret = feat->dslite->ops1(&params, ct_offload_info);
		if (ret != 0) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - gre2dslite feature ops error %d\n", ret);
			ct_offload_info->create_err = 1;
			return ret;
		}
	}
	flowmgr_dump_ipv4(flow_directions[dir],
		  ct->mark, ct_offload_info, ipv4_params);
	ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
	if (ret != 0)
		return ret;

	flowmgr_dump_flowid(ct_offload_info,
			    "FLOWMGR: Flow (ft_ipv4_gre2dslite) added",
			    flow_id);
	flowmgr_update_ct_offload(flow_id,
			  ft_ipv4 << 16 | ipv4_params->type,
			  params.tx.control.lag,
			  tuple_match,
			  ct_offload_info,
			  ct_offload,
			  ct_offload_master);
	return 0;
}

int flowmgr_promote_dslite2gre(int dir, int prot, int pri,
			       struct nf_conn *ct,
			       struct nf_conntrack_tuple *tuple_match,
			       struct nf_conntrack_tuple *tuple_replacement,
			       struct net_device *dev_in,
			       struct net_device *dev_out,
			       struct nf_conn_offload *ct_offload,
			       struct nf_conn_offload *ct_offload_master,
			       struct offload_info *ct_offload_info,
			       int out_dev_cat)
{
	int flow_id = -1, ret = -1;
	struct flow_params params;
	struct flow_ipv4_params *ipv4_params = &params.ipv4;
	struct flow_tunnel_params tunnel;

	mso_feat_init();
	/* DSLite decap and gre encap */
	if (ct_offload_info->debug)
		pr_debug("FLOWMGR: dslite decap and gre encap\n");
	params.type = ft_ipv4;
	ipv4_params->ip_prot  = prot;
	ipv4_params->src_ip   = tuple_match->src.u3.ip;
	ipv4_params->dst_ip   = tuple_match->dst.u3.ip;
	ipv4_params->src_port = tuple_match->src.u.tcp.port;
	ipv4_params->dst_port = tuple_match->dst.u.tcp.port;
	ipv4_params->replacement_ip   =
		tuple_replacement->src.u3.ip;
	ipv4_params->replacement_port =
		tuple_replacement->src.u.tcp.port;

	if (ct_offload_info->encap_tid == -1) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - dslite2gre output gre tunnel not set\n");
		ct_offload_info->create_err = 1;
		return -1;
	}

	memset(&params.rx, 0, sizeof(struct flow_rx_params));
	memset(&params.tx, 0, sizeof(struct flow_tx_params));
	params.rx.interface =
		flowmgr_tunnel_out_ifindex(dev_in,
					   &tunnel,
					   NULL);
	if (params.rx.interface == 0) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - dslite2gre input dslite rx device not set\n");
		ct_offload_info->create_err = 1;
		return -1;
	}

	params.wifi.skb_pri = pri & 7;
	params.tx.interface =
	    flowmgr_tunnelid_out_ifindex(dev_out,
	    ct_offload_info->encap_tid,
	    &tunnel, &params.wifi);
	if (params.tx.interface == 0) {
		if (ct_offload_info->debug)
			pr_alert("FLOWMGR: Promotion error - dslite2gre outout gre tx device not set\n");
		ct_offload_info->create_err = 1;
		return -1;
	}
	params.tx.control.priority = pri;

	memcpy(&params.encap_tun, &tunnel,
	    sizeof(struct flow_tunnel_params));

	if (out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP)
		params.encap_tun.gre.type = ft_ipv4;
	else
		params.encap_tun.gre.type = ft_ipv6;

	params.debug = ct_offload_info->debug;

	if (feat && feat->dslite && feat->dslite->ops2) {
		ret = feat->dslite->ops2(&params, ct_offload_info);
		if (ret != 0) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - dslite2gre feature ops error %d\n", ret);
			ct_offload_info->create_err = 1;
			return ret;
		}
	}
	flowmgr_dump_ipv4(flow_directions[dir],
		  ct->mark, ct_offload_info, ipv4_params);
	ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
	if (ret != 0)
		return ret;

	flowmgr_dump_flowid(ct_offload_info,
			    "FLOWMGR: Flow (ft_ipv4_dslite2gre) added",
			    flow_id);
	flowmgr_update_ct_offload(flow_id,
			  ft_ipv4 << 16 | ipv4_params->type,
			  params.tx.control.lag,
			  tuple_match,
			  ct_offload_info,
			  ct_offload,
			  ct_offload_master);
	return 0;
}

int flowmgr_promote_ipv6nat(int dir, int prot, int pri,
			    struct nf_conn *ct,
			    struct nf_conntrack_tuple *tuple_match,
			    struct nf_conntrack_tuple *tuple_replacement,
			    struct net_device *dev_in,
			    struct net_device *dev_out,
			    struct nf_conn_offload *ct_offload,
			    struct nf_conn_offload *ct_offload_master,
			    struct offload_info *ct_offload_info,
			    bool tuple_replacement_use_dst)
{
	int flow_id = -1, ret = -1;
	struct flow_params params;
	struct flow_ipv6_params *ipv6_params = &params.ipv6;

	params.type = ft_ipv6;
	ipv6_params->type = (dir == flow_wanlan) ?
		ft_ipv6_nat_dst : ft_ipv6_nat_src;
	ipv6_params->ip_prot  = prot;
	memcpy(ipv6_params->src_ip,
	       tuple_match->src.u3.ip6, 16);
	memcpy(ipv6_params->dst_ip,
	       tuple_match->dst.u3.ip6, 16);
	ipv6_params->src_port = tuple_match->src.u.tcp.port;
	ipv6_params->dst_port = tuple_match->dst.u.tcp.port;
	if (tuple_replacement_use_dst) {
		memcpy(ipv6_params->replacement_ip,
			tuple_replacement->dst.u3.ip6, 16);
		ipv6_params->replacement_port =
			tuple_replacement->dst.u.tcp.port;
	} else {
		memcpy(ipv6_params->replacement_ip,
			tuple_replacement->src.u3.ip6, 16);
		ipv6_params->replacement_port =
			tuple_replacement->src.u.tcp.port;
	}
	memcpy(ipv6_params->replacement_mac_src,
	       ct_offload_info->eh.h_source, ETH_ALEN);
	memcpy(ipv6_params->replacement_mac_dst,
	       ct_offload_info->eh.h_dest, ETH_ALEN);
	memset(&params.tx, 0, sizeof(struct flow_tx_params));
	params.tx.interface = dev_out->ifindex;
	params.tx.control.priority = pri;
	params.tx.control.vlan_tci = ct_offload_info->vlan_id;
	flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
	memset(&params.rx, 0, sizeof(struct flow_rx_params));
	params.rx.interface = dev_in->ifindex;
	params.rx.control.vlan_untag =  ct_offload_info->vlan_untag;
	params.wifi.flowring = ct_offload_info->wifi_flowring;
	params.wifi.pri = ct_offload_info->wifi_pri;
	params.debug = ct_offload_info->debug;
	flowmgr_dump_ipv6(flow_directions[dir],
		  ct->mark, ct_offload_info, ipv6_params);
	ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
	if (ret != 0)
		return ret;
	flowmgr_dump_flowid(ct_offload_info,
			    "FLOWMGR: Flow (ipv6_nat) added",
			    flow_id);
	flowmgr_update_ct_offload(flow_id,
			  ft_ipv6 << 16 | ipv6_params->type,
			  params.tx.control.lag,
			  tuple_match,
			  ct_offload_info,
			  ct_offload,
			  ct_offload_master);
	return 0;
}

int flowmgr_promote(int dir, int prot, int pri,
		    struct nf_conn *ct,
		    struct nf_bridge_info *nf_bridge,
		    struct nf_conntrack_tuple *tuple_match,
		    struct nf_conntrack_tuple *tuple_replacement,
		    bool tuple_replacement_use_dst,
		    struct offload_info *ct_offload_info)
{
	int flow_id = -1, ret = -1;
	struct flow_params params;
	struct nf_conn_offload *ct_offload = NULL;
	struct nf_conn_offload *ct_offload_master = NULL;
	struct net_device *dev_out;
	struct net_device *dev_in;
	struct net_device *physdev_in;
	bool gre = false, nat = false;
	int in_dev_cat;
	int out_dev_cat;
	int physdev_in_cat = 0;
	struct flow_tunnel_params tunnel;

	if (!flowmgr_is_direction_allowed(dir)) {
		ct_offload_info->flow_type =
			ft_ignore<<16;
		return 0;
	}

	dev_out = __dev_get_by_index(&init_net, ct_offload_info->oif);
	dev_in = __dev_get_by_index(&init_net, ct_offload_info->iif);
	ct_offload = nf_conn_offload_find(ct);
	if (!dev_out || !dev_in) {
		pr_err("flowmgr_promote: error dev in %s %d out %s %d dir %d prot %d sport %d dport %d %pM %pM ctinfo %d orig %d repl %d\n",
			   dev_in?dev_in->name:"null",
			   ct_offload_info->iif,
			   dev_out?dev_out->name:"null",
			   ct_offload_info->oif,
			   dir, prot,
			   ntohs(tuple_match->src.u.tcp.port),
			   ntohs(tuple_match->dst.u.tcp.port),
			   ct_offload_info->eh.h_source,
			   ct_offload_info->eh.h_dest,
			   ct_offload_info->ctinfo,
			   ct_offload_info == &ct_offload_orig,
			   ct_offload_info == &ct_offload_repl);
		/* Relearn in/out interfaces */
		ct_offload_info->ctinfo = -1;
		return 0;
	}

	in_dev_cat = BCM_NETDEVICE_GROUP_CAT(dev_in->group);
	out_dev_cat = BCM_NETDEVICE_GROUP_CAT(dev_out->group);
	if (nf_bridge && nf_bridge->physindev) {
		physdev_in = __dev_get_by_index(&init_net,
						nf_bridge->physindev->ifindex);
		if (!physdev_in) {
			pr_err("flowmgr_promote: error physdev_in null dir %d sport %d dport %d\n",
			       dir,
			       ntohs(tuple_match->src.u.tcp.port),
			       ntohs(tuple_match->dst.u.tcp.port));
			/* Relearn in/out interfaces */
			ct_offload_info->ctinfo = -1;
			return 0;
		}
		physdev_in_cat = BCM_NETDEVICE_GROUP_CAT(physdev_in->group);
		if (physdev_in_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
		    physdev_in_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6)
			gre = true;
	}
	if (in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
	    in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6 ||
	    out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
	    out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6)
		gre = true;
	if (ct->master)
		ct_offload_master = nf_conn_offload_find(ct->master);

	flowmgr_expected_demote(ct_offload_info);

	if ((prot == IPPROTO_ESP) || (prot == IPPROTO_AH)) {
		struct flow_ipsec_params *ipsec_params = &params.ipsec;
		ipsec_params->spi = ct_offload_info->spi;
	}

	if ((test_bit(IPS_SRC_NAT_BIT, &ct->status) ||
	     test_bit(IPS_DST_NAT_BIT, &ct->status)) && !gre) {
		/* NATed flow */
		struct flow_ipv4_params *ipv4_params = &params.ipv4;

		/* Simultaneous Src and Dst NAT is not allowed */
		if (test_bit(IPS_SRC_NAT_BIT, &ct->status) &&
			test_bit(IPS_DST_NAT_BIT, &ct->status)) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - Simultaneous Src and Dst NAT is not allowed\n");
			ct_offload_info->create_err = 1;
			return -1;
		}

		if (dir == flow_lanlan) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - NAT flow is not allowed for lan2lan direction\n");
			ct_offload_info->create_err = 1;
			return -1;
		}

		if (nf_ct_l3num(ct) == AF_INET6) {
			return flowmgr_promote_ipv6nat(dir, prot, pri,
						       ct, tuple_match,
						       tuple_replacement,
						       dev_in, dev_out,
						       ct_offload,
						       ct_offload_master,
						       ct_offload_info,
						       tuple_replacement_use_dst);
		}
		params.type = ft_ipv4;
		ipv4_params->type = (dir == flow_wanlan) ?
			ft_ipv4_nat_dst : ft_ipv4_nat_src;
		ipv4_params->ip_prot  = prot;
		ipv4_params->src_ip   = tuple_match->src.u3.ip;
		ipv4_params->dst_ip   = tuple_match->dst.u3.ip;
		ipv4_params->src_port = tuple_match->src.u.tcp.port;
		ipv4_params->dst_port = tuple_match->dst.u.tcp.port;
		if (tuple_replacement_use_dst) {
			ipv4_params->replacement_ip   =
				tuple_replacement->dst.u3.ip;
			ipv4_params->replacement_port =
				tuple_replacement->dst.u.tcp.port;
		} else {
			ipv4_params->replacement_ip   =
				tuple_replacement->src.u3.ip;
			ipv4_params->replacement_port =
				tuple_replacement->src.u.tcp.port;
		}
		memcpy(ipv4_params->replacement_mac_src,
		       ct_offload_info->eh.h_source, ETH_ALEN);
		memcpy(ipv4_params->replacement_mac_dst,
		       ct_offload_info->eh.h_dest, ETH_ALEN);
		memset(&params.tx, 0, sizeof(struct flow_tx_params));
		params.tx.interface = dev_out->ifindex;
		params.tx.control.priority = pri;
		params.tx.control.vlan_tci = ct_offload_info->vlan_id;
		flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
		memset(&params.rx, 0, sizeof(struct flow_rx_params));
		params.rx.interface = dev_in->ifindex;
		params.rx.control.vlan_untag =  ct_offload_info->vlan_untag;
		params.wifi.flowring = ct_offload_info->wifi_flowring;
		params.wifi.pri = ct_offload_info->wifi_pri;
		params.debug = ct_offload_info->debug;
		flowmgr_dump_ipv4(flow_directions[dir],
			  ct->mark, ct_offload_info, ipv4_params);
		ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
		if (ret != 0)
			return ret;
		flowmgr_dump_flowid(ct_offload_info,
				    "FLOWMGR: Flow (ipv4_nat) added",
				    flow_id);
		flowmgr_update_ct_offload(flow_id,
				  ft_ipv4 << 16 | ipv4_params->type,
				  params.tx.control.lag,
				  tuple_match,
				  ct_offload_info,
				  ct_offload,
				  ct_offload_master);
		return 0;
	} else if (nf_bridge && flowmgr.use_3tuple_br_flow &&
			nf_bridge->physoutdev && !gre) {
		struct flow_mac_bridge_params *mac_params;

		if (!flowmgr_is_feature_enabled(FLOW_F_MACBR)) {
			if (ct_offload_info->debug)
				pr_alert("FLOWMGR: Promotion error - MAC BR Flows not enabled\n");
			ct_offload_info->create_err = 1;
			return -1;
		}

		mac_params = &params.mac_bridge;
		params.type = ft_mac_bridge;
		mac_params->type = ft_mb_bridge;
		mac_params->rx_interface = dev_in->ifindex;
		mac_params->rx_subif = 0;
		memcpy(mac_params->mac_src,
		       ct_offload_info->eh.h_source, ETH_ALEN);
		memcpy(mac_params->mac_dst,
		       ct_offload_info->eh.h_dest, ETH_ALEN);
		memset(&params.tx, 0, sizeof(struct flow_tx_params));
		params.tx.interface = dev_out->ifindex;
		params.tx.control.priority = pri;
		params.tx.control.vlan_tci = ct_offload_info->vlan_id;
		flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
		memset(&params.rx, 0, sizeof(struct flow_rx_params));
		params.rx.interface = dev_in->ifindex;
		params.rx.control.vlan_untag =  ct_offload_info->vlan_untag;
		params.wifi.flowring = ct_offload_info->wifi_flowring;
		params.wifi.pri = ct_offload_info->wifi_pri;
		params.debug = ct_offload_info->debug;
		if (ct_offload_info->debug)
			dump_bridge(pr_alert, flow_directions[dir],
				    ct->mark, ct_offload_info, mac_params);
		ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
		if (ret != 0)
			return ret;
		flowmgr_dump_flowid(ct_offload_info,
				    "FLOWMGR: Flow (bridge_mac) added",
				    flow_id);
		flowmgr_update_ct_offload(flow_id,
				  ft_mac_bridge << 16 | mac_params->type,
				  params.tx.control.lag,
				  tuple_match,
				  ct_offload_info,
				  ct_offload,
				  ct_offload_master);
		return 0;
	} else if (nf_bridge && (nf_ct_l3num(ct) == AF_INET) &&
			nf_bridge->physoutdev && !gre) {
		/* IPv4 Bridge */
		struct flow_ipv4_params *ipv4_params = &params.ipv4;
		params.type = ft_ipv4;
		if (BCM_NETDEVICE_GROUP_TYPE(dev_in->group) ==
		    BCM_NETDEVICE_GROUP_WAN)
			ipv4_params->type = ft_ipv4_bridge_ds;
		else
			ipv4_params->type = ft_ipv4_bridge;

		ipv4_params->ip_prot  = prot;
		ipv4_params->src_ip   = tuple_match->src.u3.ip;
		ipv4_params->dst_ip   = tuple_match->dst.u3.ip;
		ipv4_params->src_port = tuple_match->src.u.tcp.port;
		ipv4_params->dst_port = tuple_match->dst.u.tcp.port;
		/* Replacement parameters are not used/needed */
		ipv4_params->replacement_ip = tuple_match->dst.u3.ip;
		ipv4_params->replacement_port = tuple_match->dst.u.tcp.port;
		memcpy(ipv4_params->replacement_mac_src,
		       ct_offload_info->eh.h_source, ETH_ALEN);
		memcpy(ipv4_params->replacement_mac_dst,
		       ct_offload_info->eh.h_dest, ETH_ALEN);
		memset(&params.tx, 0, sizeof(struct flow_tx_params));
		params.tx.interface = dev_out->ifindex;
		params.tx.control.priority = pri;
		params.tx.control.vlan_tci = ct_offload_info->vlan_id;
		flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
		memset(&params.rx, 0, sizeof(struct flow_rx_params));
		params.rx.interface = dev_in->ifindex;
		params.rx.control.vlan_untag =  ct_offload_info->vlan_untag;
		params.wifi.flowring = ct_offload_info->wifi_flowring;
		params.wifi.pri = ct_offload_info->wifi_pri;
		params.debug = ct_offload_info->debug;
		flowmgr_dump_ipv4(flow_directions[dir],
			  ct->mark, ct_offload_info, ipv4_params);
		ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
		if (ret != 0)
			return ret;
		flowmgr_dump_flowid(ct_offload_info,
				    "FLOWMGR: Flow (ipv4_bridge) added",
				    flow_id);
		flowmgr_update_ct_offload(flow_id,
				  ft_ipv4 << 16 | ipv4_params->type,
				  params.tx.control.lag,
				  tuple_match,
				  ct_offload_info,
				  ct_offload,
				  ct_offload_master);
		return 0;

	} else if (nf_bridge && (nf_ct_l3num(ct) == AF_INET6) &&
			nf_bridge->physoutdev && !gre) {
		/* IPv6 Bridged */
		struct flow_ipv6_params *ipv6_params = &params.ipv6;
		params.type = ft_ipv6;
		if (BCM_NETDEVICE_GROUP_TYPE(dev_in->group) ==
		    BCM_NETDEVICE_GROUP_WAN)
			ipv6_params->type = ft_ipv6_bridge_ds;
		else
			ipv6_params->type = ft_ipv6_bridge;
		ipv6_params->ip_prot = prot;
		memcpy(ipv6_params->src_ip,
		       tuple_match->src.u3.ip6, 16);
		memcpy(ipv6_params->dst_ip,
		       tuple_match->dst.u3.ip6, 16);
		ipv6_params->src_port =
			tuple_match->src.u.tcp.port;
		ipv6_params->dst_port =
			tuple_match->dst.u.tcp.port;
		/* Replacement parameters are not used/needed */
		memcpy(ipv6_params->replacement_mac_src,
		       ct_offload_info->eh.h_source, ETH_ALEN);
		memcpy(ipv6_params->replacement_mac_dst,
		       ct_offload_info->eh.h_dest, ETH_ALEN);
		memset(&params.tx, 0, sizeof(struct flow_tx_params));
		params.tx.interface = dev_out->ifindex;
		params.tx.control.priority = pri;
		params.tx.control.vlan_tci = ct_offload_info->vlan_id;
		flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
		memset(&params.rx, 0, sizeof(struct flow_rx_params));
		params.rx.interface = dev_in->ifindex;
		params.rx.control.vlan_untag =  ct_offload_info->vlan_untag;
		params.wifi.flowring = ct_offload_info->wifi_flowring;
		params.wifi.pri = ct_offload_info->wifi_pri;
		params.debug = ct_offload_info->debug;
		flowmgr_dump_ipv6(flow_directions[dir],
			  ct->mark, ct_offload_info, ipv6_params);
		ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
		if (ret != 0)
			return ret;
		flowmgr_dump_flowid(ct_offload_info,
				    "FLOWMGR: Flow (ipv6_bridge) added",
				    flow_id);
		flowmgr_update_ct_offload(flow_id,
				  ft_ipv6 << 16 | ipv6_params->type,
				  params.tx.control.lag,
				  tuple_match,
				  ct_offload_info,
				  ct_offload,
				  ct_offload_master);
		return 0;
	} else if (nf_ct_l3num(ct) == AF_INET6) {
		/* IPv6 Routed */
		struct flow_ipv6_params *ipv6_params = &params.ipv6;
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
		if ((dev_in->type == ARPHRD_TUNNEL6) ||
		    (dev_out->type == ARPHRD_TUNNEL6)) {
			ct_offload_info->create_err = 1;
			return 0;
		} else if ((dev_in->type == ARPHRD_IPGRE) ||
				(dev_out->type == ARPHRD_IPGRE)) {
			/* GRE Tunnel */
			ct_offload_info->create_err = 1;
			return 0;
		} else if (gre) {
			if (in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
			    out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
			    physdev_in_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
				in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6 ||
				out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6 ||
				physdev_in_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6) {
				/* Simultaneous Src and Dst NAT is not allowed */
				if (test_bit(IPS_SRC_NAT_BIT, &ct->status) &&
					test_bit(IPS_DST_NAT_BIT, &ct->status)) {
					if (ct_offload_info->debug)
						pr_alert("FLOWMGR: Promotion error - Simultaneous Src and Dst NAT is not allowed\n");
					ct_offload_info->create_err = 1;
					return -1;
				}
				if (test_bit(IPS_SRC_NAT_BIT, &ct->status) ||
					test_bit(IPS_DST_NAT_BIT, &ct->status))
					nat = true;

				params.type = ft_ipv6;
				ipv6_params->ip_prot = prot;
				memcpy(ipv6_params->src_ip,
				       tuple_match->src.u3.ip6, 16);
				memcpy(ipv6_params->dst_ip,
				       tuple_match->dst.u3.ip6, 16);
				ipv6_params->src_port =
					tuple_match->src.u.tcp.port;
				ipv6_params->dst_port =
					tuple_match->dst.u.tcp.port;
				memcpy(ipv6_params->replacement_mac_src,
				       ct_offload_info->eh.h_source, ETH_ALEN);
				memcpy(ipv6_params->replacement_mac_dst,
				       ct_offload_info->eh.h_dest, ETH_ALEN);
				memset(&params.rx, 0,
				       sizeof(struct flow_rx_params));
				memset(&params.tx, 0,
				       sizeof(struct flow_tx_params));
				params.tx.control.priority = pri;
				params.tx.control.vlan_tci =
					ct_offload_info->vlan_id;
				flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
				if (tuple_replacement_use_dst) {
					memcpy(ipv6_params->replacement_ip,
						tuple_replacement->dst.u3.ip6, 16);
					ipv6_params->replacement_port =
						tuple_replacement->dst.u.tcp.port;
				} else {
					memcpy(ipv6_params->replacement_ip,
						tuple_replacement->src.u3.ip6, 16);
					ipv6_params->replacement_port =
						tuple_replacement->src.u.tcp.port;
				}
				if ((out_dev_cat ==
				    BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP) ||
					(out_dev_cat ==
					BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6)) {
					/* GRE: Encapsulation */

					if (ct_offload_info->encap_tid == -1) {
						if (ct_offload_info->debug)
							pr_alert("FLOWMGR: Promotion error - output gre tunnel not set\n");
						ct_offload_info->create_err = 1;
						return -1;
					}
					if (flowmgr.disable_gre_accel & 1) {
						ct_offload_info->flow_type =
							ft_ignore<<16;
						atomic_inc(&flowmgr.flow_ignore_cnt);
						return -1;
					}

					ipv6_params->type = ft_ipv6_gre_encap;
					if (nat) {
						ipv6_params->type = (dir == flow_wanlan) ?
							ft_ipv6_gre_encap_nat_dst : ft_ipv6_gre_encap_nat_src;
					}
					params.wifi.skb_pri = pri & 7;
					params.tx.interface =
					    flowmgr_tunnel_out_ifindex(dev_out,
					    &tunnel, &params.wifi);
					if (params.tx.interface == 0) {
						if (ct_offload_info->debug)
							pr_alert("FLOWMGR: Promotion error - output gre tunnel tx device not set\n");
						ct_offload_info->create_err = 1;
						return -1;
					}
					memcpy(&params.encap_tun, &tunnel,
					    sizeof(struct flow_tunnel_params));
					params.encap_tun.gre.tunnel_id = ct_offload_info->encap_tid;
					params.rx.interface = dev_in->ifindex;
					/* Update offload wifi for flushing */
					ct_offload_info->wifi_flowring = params.wifi.flowring;
					ct_offload_info->wifi_pri = params.wifi.pri;
				} else {
					/* This catches both the case of
					bridged entry towards gretap0pri AND
					direct. Either in_dev_cat OR
					physdev_in_cat is
					BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP */
					/* GRE Decapsulation */
					struct flow_tunnel_params tunnel;
					struct flowmgr_tdb_entry *tdb;

					tdb = flowmgr_tdb_find(dev_in->ifindex);
					if (!tdb) {
						if (ct_offload_info->debug)
							pr_alert("FLOWMGR: Promotion error - input gre tunnel not found\n");
						ct_offload_info->create_err = 1;
						return -1;
					}

					if (flowmgr.disable_gre_accel & 2) {
						ct_offload_info->flow_type =
							ft_ignore<<16;
						atomic_inc(&flowmgr.flow_ignore_cnt);
						return -1;
					}

					ipv6_params->type = ft_ipv6_gre_decap;
					if (nat) {
						ipv6_params->type = (dir == flow_wanlan) ?
							ft_ipv6_gre_decap_nat_dst : ft_ipv6_gre_decap_nat_src;
					}
					params.rx.interface =
						flowmgr_tunnel_out_ifindex(dev_in, &tunnel,
									   NULL);
					params.tx.interface = dev_out->ifindex;
					params.decap_tun.gre.tunnel_id = ct_offload_info->decap_tid = tdb->id;
					params.wifi.flowring = ct_offload_info->wifi_flowring;
					params.wifi.pri = ct_offload_info->wifi_pri;
				}
				if (in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
				    out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
				    physdev_in_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP)
					params.encap_tun.gre.type = ft_ipv4;
				else
					params.encap_tun.gre.type = ft_ipv6;
				params.rx.control.vlan_untag =
					ct_offload_info->vlan_untag;
				if (ct_offload_info->debug)
					pr_alert("FLOWMGR: Flow (ft_ipv6_gre) add: %d\n", ipv6_params->type);
				params.debug = ct_offload_info->debug;
				flowmgr_dump_ipv6(flow_directions[dir],
				    ct->mark, ct_offload_info, ipv6_params);
				ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
				if (ret != 0)
					return ret;

				flowmgr_dump_flowid(ct_offload_info,
						    "FLOWMGR Flow (ft_ipv6_gre) added",
						    flow_id);
				flowmgr_update_ct_offload(flow_id,
						  ft_ipv6<<16|ipv6_params->type,
						  params.tx.control.lag,
						  tuple_match,
						  ct_offload_info,
						  ct_offload,
						  ct_offload_master);
				return 0;

			}
		}
#endif
		params.type = ft_ipv6;
		ipv6_params->type = ft_ipv6_routed;
		ipv6_params->ip_prot = prot;
		memcpy(ipv6_params->src_ip,
		       tuple_match->src.u3.ip6, 16);
		memcpy(ipv6_params->dst_ip,
		       tuple_match->dst.u3.ip6, 16);
		ipv6_params->src_port =
			tuple_match->src.u.tcp.port;
		ipv6_params->dst_port =
			tuple_match->dst.u.tcp.port;
		memcpy(ipv6_params->replacement_mac_src,
		       ct_offload_info->eh.h_source, ETH_ALEN);
		memcpy(ipv6_params->replacement_mac_dst,
		       ct_offload_info->eh.h_dest, ETH_ALEN);
		memset(&params.tx, 0, sizeof(struct flow_tx_params));
		params.tx.interface = dev_out->ifindex;
		params.tx.control.priority = pri;
		params.tx.control.vlan_tci = ct_offload_info->vlan_id;
		flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
		memset(&params.rx, 0, sizeof(struct flow_rx_params));
		params.rx.interface = dev_in->ifindex;
		params.rx.control.vlan_untag =  ct_offload_info->vlan_untag;
		params.wifi.flowring = ct_offload_info->wifi_flowring;
		params.wifi.pri = ct_offload_info->wifi_pri;
		params.debug = ct_offload_info->debug;
		flowmgr_dump_ipv6(flow_directions[dir],
			  ct->mark, ct_offload_info, ipv6_params);
		ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
		if (ret != 0)
			return ret;
		flowmgr_dump_flowid(ct_offload_info,
				    "FLOWMGR: Flow (ipv6) added",
				    flow_id);
		flowmgr_update_ct_offload(flow_id,
				  ft_ipv6 << 16 | ipv6_params->type,
				  params.tx.control.lag,
				  tuple_match,
				  ct_offload_info,
				  ct_offload,
				  ct_offload_master);
		return 0;
	} else if (nf_ct_l3num(ct) == AF_INET) {
		/* IPv4 Routed */
		struct flow_ipv4_params *ipv4_params = &params.ipv4;
#ifdef CONFIG_BCM_FLOWMGR_TUNNEL
		if ((dev_out->type == ARPHRD_TUNNEL6) && gre) {
			return flowmgr_promote_gre2dslite(dir, prot, pri,
							  ct,
							  tuple_match,
							  tuple_replacement,
							  dev_in,
							  dev_out,
							  ct_offload,
							  ct_offload_master,
							  ct_offload_info);
		} else if ((dev_in->type == ARPHRD_TUNNEL6) && gre) {
			return flowmgr_promote_dslite2gre(dir, prot, pri,
							  ct,
							  tuple_match,
							  tuple_replacement,
							  dev_in,
							  dev_out,
							  ct_offload,
							  ct_offload_master,
							  ct_offload_info,
							  out_dev_cat);
		} else if ((dev_in->type == ARPHRD_TUNNEL6) ||
		    (dev_out->type == ARPHRD_TUNNEL6)) {
			params.type = ft_ipv4;
			ipv4_params->ip_prot  = prot;
			ipv4_params->src_ip   = tuple_match->src.u3.ip;
			ipv4_params->dst_ip   = tuple_match->dst.u3.ip;
			ipv4_params->src_port = tuple_match->src.u.tcp.port;
			ipv4_params->dst_port = tuple_match->dst.u.tcp.port;
			memcpy(ipv4_params->replacement_mac_src,
			       ct_offload_info->eh.h_source, ETH_ALEN);
			memcpy(ipv4_params->replacement_mac_dst,
			       ct_offload_info->eh.h_dest, ETH_ALEN);
			memset(&params.rx, 0, sizeof(struct flow_rx_params));
			memset(&params.tx, 0, sizeof(struct flow_tx_params));
			params.tx.control.priority = pri;
			params.tx.control.vlan_tci =
				ct_offload_info->vlan_id;
			flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
			ipv4_params->replacement_ip   =
				tuple_replacement->src.u3.ip;
			ipv4_params->replacement_port =
				tuple_replacement->src.u.tcp.port;
			if (dev_in->type == ARPHRD_TUNNEL6) {
				/* IPv6 Tunnel: Decapsulation */
				ipv4_params->type = ft_ipv4_dslite_decap;
				params.rx.interface =
					flowmgr_tunnel_out_ifindex(dev_in,
								   &tunnel,
								   NULL);
				if (tunnel.dslite.tunnel_id == 0) {
					if (ct_offload_info->debug)
						pr_alert("FLOWMGR: Promotion error - input dslite tunnel id not set\n");
					ct_offload_info->flow_type =
						ft_ignore<<16;
					atomic_inc(&flowmgr.flow_ignore_cnt);
					return -1;
				}
				if (params.rx.interface == 0) {
					if (ct_offload_info->debug)
						pr_alert("FLOWMGR: Promotion error - input dslite tunnel rx device not set\n");
					ct_offload_info->create_err = 1;
					return -1;
				}
				memcpy(&params.decap_tun, &tunnel,
				       sizeof(struct flow_tunnel_params));
				params.tx.interface = dev_out->ifindex;
				params.wifi.flowring = ct_offload_info->wifi_flowring;
				params.wifi.pri = ct_offload_info->wifi_pri;
			} else {
				/* IPv6 Tunnel: Encapsulation */
				ipv4_params->type = ft_ipv4_dslite_encap;
				params.wifi.skb_pri = pri & 7;
				params.tx.interface =
					flowmgr_tunnel_out_ifindex(dev_out,
								   &tunnel,
								   &params.wifi);
				if (tunnel.dslite.tunnel_id == 0) {
					if (ct_offload_info->debug)
						pr_alert("FLOWMGR: Promotion error - output dslite tunnel id not set\n");
					ct_offload_info->flow_type =
						ft_ignore<<16;
					atomic_inc(&flowmgr.flow_ignore_cnt);
					return -1;
				}
				if (params.tx.interface == 0) {
					if (ct_offload_info->debug)
						pr_alert("FLOWMGR: Promotion error - output dslite tunnel tx device not set\n");
					ct_offload_info->create_err = 1;
					return -1;
				}
				memcpy(&params.encap_tun, &tunnel,
				       sizeof(struct flow_tunnel_params));
				params.rx.interface = dev_in->ifindex;
				ct_offload_info->wifi_flowring = params.wifi.flowring;
				ct_offload_info->wifi_pri = params.wifi.pri;
			}
			params.rx.control.vlan_untag =
				ct_offload_info->vlan_untag;
			params.debug = ct_offload_info->debug;
			flowmgr_dump_ipv4(flow_directions[dir],
				  ct->mark, ct_offload_info, ipv4_params);
			ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
			if (ret != 0)
				return ret;

			flowmgr_dump_flowid(ct_offload_info,
					    "FLOWMGR: Flow (ft_ipv4_dslite) added",
					    flow_id);
			flowmgr_update_ct_offload(flow_id,
					  ft_ipv4 << 16 | ipv4_params->type,
					  params.tx.control.lag,
					  tuple_match,
					  ct_offload_info,
					  ct_offload,
					  ct_offload_master);
			return 0;
		} else if ((dev_in->type == ARPHRD_IPGRE) ||
				(dev_out->type == ARPHRD_IPGRE)) {
			/* GRE Tunnel */
		} else if (gre) {
			if (in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
			    out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
			    physdev_in_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
				in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6 ||
				out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6 ||
				physdev_in_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6) {
				/* GRE over IPv4 */
				/* Simultaneous Src and Dst NAT is not allowed */
				if (test_bit(IPS_SRC_NAT_BIT, &ct->status) &&
					test_bit(IPS_DST_NAT_BIT, &ct->status)) {
					if (ct_offload_info->debug)
						pr_alert("FLOWMGR: Promotion error - Simultaneous Src and Dst NAT is not allowed\n");
					ct_offload_info->create_err = 1;
					return -1;
				}
				if (test_bit(IPS_SRC_NAT_BIT, &ct->status) ||
					test_bit(IPS_DST_NAT_BIT, &ct->status))
					nat = true;

				ipv4_params = &params.ipv4;
				params.type = ft_ipv4;
				ipv4_params->ip_prot  = prot;
				ipv4_params->src_ip   =
					tuple_match->src.u3.ip;
				ipv4_params->dst_ip   =
					tuple_match->dst.u3.ip;
				ipv4_params->src_port =
					tuple_match->src.u.tcp.port;
				ipv4_params->dst_port =
					tuple_match->dst.u.tcp.port;
				memcpy(ipv4_params->replacement_mac_src,
				       ct_offload_info->eh.h_source, ETH_ALEN);
				memcpy(ipv4_params->replacement_mac_dst,
				       ct_offload_info->eh.h_dest, ETH_ALEN);
				memset(&params.rx, 0,
				       sizeof(struct flow_rx_params));
				memset(&params.tx, 0,
				       sizeof(struct flow_tx_params));
				params.tx.control.priority = pri;
				params.tx.control.vlan_tci =
					ct_offload_info->vlan_id;
				flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
				if (tuple_replacement_use_dst) {
					ipv4_params->replacement_ip   =
						tuple_replacement->dst.u3.ip;
					ipv4_params->replacement_port =
							tuple_replacement->dst.u.tcp.port;
				} else {
					ipv4_params->replacement_ip   =
						tuple_replacement->src.u3.ip;
					ipv4_params->replacement_port =
						tuple_replacement->src.u.tcp.port;
				}
				if ((out_dev_cat ==
				    BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP) ||
					(out_dev_cat ==
					BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP6)) {
					/* GRE: Encapsulation */

					if (ct_offload_info->encap_tid == -1) {
						if (ct_offload_info->debug)
							pr_alert("FLOWMGR: Promotion error - output gre tunnel not set\n");
						ct_offload_info->create_err = 1;
					}
					if (flowmgr.disable_gre_accel & 1) {
						ct_offload_info->flow_type =
							ft_ignore<<16;
						atomic_inc(&flowmgr.flow_ignore_cnt);
						return -1;
					}

					ipv4_params->type = ft_ipv4_gre_encap;
					if (nat) {
						ipv4_params->type = (dir == flow_wanlan) ?
							ft_ipv4_gre_encap_nat_dst : ft_ipv4_gre_encap_nat_src;
					}
					params.wifi.skb_pri = pri & 7;
					params.tx.interface =
					    flowmgr_tunnelid_out_ifindex(dev_out,
					    ct_offload_info->encap_tid,
					    &tunnel, &params.wifi);
					if (params.tx.interface == 0) {
						if (ct_offload_info->debug)
							pr_alert("FLOWMGR: Promotion error - output gre tunnel tx device not set\n");
						ct_offload_info->create_err = 1;
						return -1;
					}
					memcpy(&params.encap_tun, &tunnel,
					    sizeof(struct flow_tunnel_params));
					params.encap_tun.gre.tunnel_id = ct_offload_info->encap_tid;
					params.rx.interface = dev_in->ifindex;
					ct_offload_info->wifi_flowring = params.wifi.flowring;
					ct_offload_info->wifi_pri = params.wifi.pri;
				} else {
					/* This catches both the case of
					bridged entry towards gretap0pri AND
					direct. Either in_dev_cat OR
					physdev_in_cat is
					BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP */
					/* GRE Decapsulation */
					struct flow_tunnel_params tunnel;
					struct flowmgr_tdb_entry *tdb;

					tdb = flowmgr_tdb_find(dev_in->ifindex);
					if (!tdb) {
						if (ct_offload_info->debug)
							pr_alert("FLOWMGR: Promotion error - input gre tunnel not found\n");
						ct_offload_info->create_err = 1;
						return -1;
					}

					if (flowmgr.disable_gre_accel & 2) {
						ct_offload_info->flow_type =
							ft_ignore<<16;
						atomic_inc(&flowmgr.flow_ignore_cnt);
						return -1;
					}

					ipv4_params->type = ft_ipv4_gre_decap;
					if (nat) {
						ipv4_params->type = (dir == flow_wanlan) ?
							ft_ipv4_gre_decap_nat_dst : ft_ipv4_gre_decap_nat_src;
					}
					params.rx.interface =
						flowmgr_tunnel_out_ifindex(dev_in, &tunnel,
									   NULL);
					params.tx.interface = dev_out->ifindex;
					params.decap_tun.gre.tunnel_id = ct_offload_info->decap_tid = tdb->id;
					params.wifi.flowring = ct_offload_info->wifi_flowring;
					params.wifi.pri = ct_offload_info->wifi_pri;
				}
				if (in_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
					out_dev_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP ||
					physdev_in_cat == BCM_NETDEVICE_GROUP_CAT_VIR_GRETAP)
					params.encap_tun.gre.type = ft_ipv4;
				else
					params.encap_tun.gre.type = ft_ipv6;
				params.rx.control.vlan_untag =
					ct_offload_info->vlan_untag;
				if (ct_offload_info->debug)
					pr_alert("FLOWMGR: Flow (ft_ipv4_gre) add: %d\n", ipv4_params->type);
				params.debug = ct_offload_info->debug;
				flowmgr_dump_ipv4(flow_directions[dir],
				    ct->mark, ct_offload_info, ipv4_params);
				ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
				if (ret != 0)
					return ret;

				flowmgr_dump_flowid(ct_offload_info,
						    "FLOWMGR: Flow (ft_ipv4_gre) added",
						    flow_id);
				flowmgr_update_ct_offload(flow_id,
						  ft_ipv4<<16|ipv4_params->type,
						  params.tx.control.lag,
						  tuple_match,
						  ct_offload_info,
						  ct_offload,
						  ct_offload_master);
				return 0;
			}
		}
#endif
		params.type = ft_ipv4;
		if (BCM_NETDEVICE_GROUP_TYPE(dev_out->group) ==
		    BCM_NETDEVICE_GROUP_WAN)
			ipv4_params->type = ft_ipv4_routed_dst;
		else
			ipv4_params->type = ft_ipv4_routed_src;
		ipv4_params->ip_prot = prot;
		ipv4_params->src_ip   = tuple_match->src.u3.ip;
		ipv4_params->dst_ip   = tuple_match->dst.u3.ip;
		ipv4_params->src_port =
			tuple_match->src.u.tcp.port;
		ipv4_params->dst_port =
			tuple_match->dst.u.tcp.port;
		memcpy(ipv4_params->replacement_mac_src,
		       ct_offload_info->eh.h_source, ETH_ALEN);
		memcpy(ipv4_params->replacement_mac_dst,
		       ct_offload_info->eh.h_dest, ETH_ALEN);
		ipv4_params->replacement_ip = 0;
		ipv4_params->replacement_port = 0;
		memset(&params.tx, 0, sizeof(struct flow_tx_params));
		params.tx.interface = dev_out->ifindex;
		params.tx.control.priority = pri;
		params.tx.control.vlan_tci = ct_offload_info->vlan_id;
		flowmgr_update_dscp_params(dir, ct, ct_offload_info, &params.tx.control);
		memset(&params.rx, 0, sizeof(struct flow_rx_params));
		params.rx.interface = dev_in->ifindex;
		params.rx.control.vlan_untag =  ct_offload_info->vlan_untag;
		params.wifi.flowring = ct_offload_info->wifi_flowring;
		params.wifi.pri = ct_offload_info->wifi_pri;
		params.debug = ct_offload_info->debug;
		flowmgr_dump_ipv4(flow_directions[dir],
			  ct->mark, ct_offload_info, ipv4_params);
		ret = flowmgr_flow_add(&params, &flow_id, ct_offload_info);
		if (ret != 0)
			return ret;
		flowmgr_dump_flowid(ct_offload_info,
				    "FLOWMGR: Flow (ipv4 routed) added",
				    flow_id);
		flowmgr_update_ct_offload(flow_id,
				  ft_ipv4 << 16 | ipv4_params->type,
				  params.tx.control.lag,
				  tuple_match,
				  ct_offload_info,
				  ct_offload,
				  ct_offload_master);
		return 0;
	}
	if (ct_offload_info->debug)
		pr_alert("FLOWMGR: No flow type found %d\n", ret);
	ct_offload_info->create_err = 1;
	return ret;
}

int flowmgr_promote_mcast(struct flowmgr_mdb_entry *mdb, int oif)
{
	int flow_id, ret = -1;
	struct flow_params params;
	struct flow_mc_params *mc_params = &params.multicast;
	int i;

	memset(&params, 0, sizeof(struct flow_params));
	params.type = ft_multicast;
	if (atomic_read(&mdb->use) == 1) {
		/* 1 member */
		if (mdb->num_mc2uc)
			mc_params->type = ft_mcast_single_ucast;
		else
			mc_params->type = ft_mcast_single;
	} else {
		/* 1+ member */
		mc_params->type = ft_mcast_multi;
	}
	memcpy(mc_params->group_id, mdb->group, sizeof(mc_params->group_id));
	mc_params->rx_interface = mdb->iif;
	memset(mc_params->tx_interface_mask, 0,
	       sizeof(mc_params->tx_interface_mask));
	memset(&params.tx, 0, sizeof(struct flow_tx_params));
	for (i = 0; i < NUM_MCTX; i++) {
		mc_params->tx_interface_mask[i] = mdb->oif[i];
		if (mdb->oif[i])
			params.tx.interface = i;
	}
	for (i = 0; i < mdb->num_mc2uc; i++) {
		memcpy(mc_params->mc2uc[i].mac_dst, mdb->mc2uc[i].addr, 6);
		mc_params->mc2uc[i].interface = mdb->mc2uc[i].oif;
		params.tx.interface = mdb->mc2uc[i].oif;
	}
	params.tx.interface = oif;
	memset(&params.rx, 0, sizeof(struct flow_rx_params));
	flow_id = mdb->flow_id;
	ret = flowmgr_flow_add_multicast(&params, &flow_id, mdb);
	if (ret == 0) {
		mdb->flow_id = flow_id;
		mdb->flow_type = ft_multicast;
		pr_debug("FLOWMGR: Flow (multicast) added: %d\n", flow_id);
		return 0;
	} else if (ret == 1) {
		pr_debug("FLOWMGR: Flow (multicast) updated: %d\n", flow_id);
		if (flow_id != mdb->flow_id)
			pr_err("FLOWMGR: Flow (multicast) updated: with new flow %d\n",
			       flow_id);
		return 0;
	}

	atomic_inc(&flowmgr.flow_mcast_create_err_cnt);
	return ret;
}

int flowmgr_get_mcast_conters(struct flowmgr_mdb_entry *mdb)
{
	u32 packets, bytes;

	if (!mdb->flow_id)
		return -1;

	fap()->flow_get_mcast_counter(mdb->flow_id,
					&packets, &bytes, false);

	mdb->packets = packets;
	mdb->bytes = bytes;

	return 0;
}

int flowmgr_demote_mcast(struct flowmgr_mdb_entry *mdb,
			 int oif)
{
	int i, ret = -1;
	if (!mdb->flow_id)
		return -1;

	if (!oif) {
		for (i = 0; i < NUM_MCTX; i++) {
			if (mdb->oif[i]) {
				fap()->flow_remove_mcast(mdb->flow_id,
							 mdb->group, i);
				atomic_add(mdb->oif[i],
					   &flowmgr.flow_mcast_delete_cnt);
			}
		}
		return 0;
	}


	if (fap()->type == GFAP) {
		if (oif && (atomic_read(&mdb->use) > 1)) {
			/* 1 member left, update flow */
			flowmgr_promote_mcast(mdb, oif);
			return 0;
		}
	}

	if (flowmgr_is_mc2uc())
		ret = fap()->flow_remove_mcast(mdb->flow_id, mdb->group, oif);
	else if (!mdb->oif[oif])
		ret = fap()->flow_remove_mcast(mdb->flow_id, mdb->group, oif);
	if (!ret) {
		pr_debug("FLOWMGR: Flow (mulicast) removed: %d", mdb->flow_id);
		atomic_inc(&flowmgr.flow_mcast_delete_cnt);
	}
	return 0;
}

int flowmgr_config_dslite(struct flow_dslite_tunnel_params *param, int oindex)
{
	if (fap()->config_dslite)
		return fap()->config_dslite(param, oindex);
	return -1;
}

#ifdef CONFIG_BCM_FLOWMGR_FAP_WQ
static void flowmgr_unconfig_dslite_deferred(struct work_struct *w)
{
	int ret, tunnel_id;
	struct flowmgr_flow_work *work;
	work = container_of(w, struct flowmgr_flow_work, work);
	tunnel_id = work->tunnel_id;
	ret = fap()->unconfig_dslite(tunnel_id);
	if (ret)
		pr_err("FLOWMGR: DSLite Tunnel %d Delete deferred error: %d\n",
		       tunnel_id, ret);
	kfree(work);
}

int flowmgr_unconfig_dslite(int id)
{
	int ret = 0;
	struct flowmgr_flow_work *work;
	if (!fap()->unconfig_dslite)
		return -1;
	work = kmalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;
	work->tunnel_id = id;
	INIT_WORK(&work->work, flowmgr_unconfig_dslite_deferred);
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}
#else
int flowmgr_unconfig_dslite(int id)
{
	if (fap()->unconfig_dslite)
		return fap()->unconfig_dslite(id);
	return -1;
}
#endif
int flowmgr_config_gre(struct flow_gre_tunnel_params *param, int oindex)
{
	if (fap()->config_gre)
		return fap()->config_gre(param, oindex);
	return -1;
}
#ifdef CONFIG_BCM_FLOWMGR_FAP_WQ
static void flowmgr_unconfig_gre_deferred(struct work_struct *w)
{
	int ret, tunnel_id;
	struct flowmgr_flow_work *work;
	work = container_of(w, struct flowmgr_flow_work, work);
	tunnel_id = work->tunnel_id;
	ret = fap()->unconfig_gre(tunnel_id);
	if (ret)
		pr_err("FLOWMGR: GRE Tunnel %d Delete deferred error: %d\n",
		       tunnel_id, ret);
	kfree(work);
}

int flowmgr_unconfig_gre(int id)
{
	int ret = 0;
	struct flowmgr_flow_work *work;
	if (!fap()->unconfig_gre)
		return -1;
	work = kmalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;
	work->tunnel_id = id;
	INIT_WORK(&work->work, flowmgr_unconfig_gre_deferred);
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}
#else
int flowmgr_unconfig_gre(int id)
{
	if (fap()->unconfig_gre)
		return fap()->unconfig_gre(id);
	return -1;
}
#endif

#ifdef CONFIG_BCM_FLOWMGR_FAP_WQ
static void flowmgr_config_mapt_deferred(struct work_struct *w)
{
	int ret = -1;
	struct flow_mapt_params *param;
	struct flowmgr_flow_work *work;
	work = container_of(w, struct flowmgr_flow_work, work);
	param = &(work->params.map.mapt);
	if (fap()->config_mapt) {
		ret = fap()->config_mapt(param, work->params.tx.interface);
		if (ret)
			pr_err("FLOWMGR: MAP-T Tunnel Config deferred error: %d\n", ret);
	}

	kfree(work);
}

int flowmgr_config_mapt(struct flow_mapt_params *param, int oindex)
{
	int ret = 0;
	struct flowmgr_flow_work *work;

	if (!fap()->config_mapt)
		return -1;
	work = kmalloc(sizeof(*work), GFP_ATOMIC);
	if (!work)
		return -ENOMEM;

	INIT_WORK(&work->work, flowmgr_config_mapt_deferred);
	work->params.map.mapt = *param;
	work->params.tx.interface = oindex;
	queue_work_on(WORK_CPU_UNBOUND, kflowmgr_wq, &work->work);
	return ret;
}
#else
int flowmgr_config_mapt(struct flow_mapt_params *param, int oindex)
{
	if (fap()->config_mapt)
		return fap()->config_mapt(param, oindex);
	return -1;
}
#endif // CONFIG_BCM_FLOWMGR_FAP_WQ


#define PROCSYS_NFCT(x)        "/proc/sys/net/netfilter/nf_conntrack_"#x
#define NSTR(x)                #x"\n"

int procsys_wr_str(unsigned char *path, unsigned char *data)
{
	int ret;
	unsigned char olddata[25] = {0};

	ret = procsys_file_read_string(path, olddata, 25);
	if (ret < 0) {
		pr_err("FLOWMGR: Error (%d) reading %s\n", ret, path);
		return ret;
	}

	olddata[strcspn(olddata, "\n")] = 0;
	ret = procsys_file_write_string(path, data);
	if (ret >= 0)  {
		pr_alert("FLOWMGR: %s=%s->%s", path, olddata, data);
		return 0;
	}

	pr_err("FLOWMGR: Error (%d) writing %s\n", ret, path);
	return ret;
}

int procsys_pr_str(struct seq_file *s, const char *path)
{
	int ret;
	unsigned char olddata[25] = {0};

	ret = procsys_file_read_string(path, olddata, 25);
	if (ret < 0) {
		pr_err("FLOWMGR: Error (%d) reading %s\n", ret, path);
		return ret;
	}

	pr_seq(s, "%s: %s", path, olddata);
	return ret;
}

void flowmgr_nfct_dbg_show(struct seq_file *s)
{
	procsys_pr_str(s, PROCSYS_NFCT(acct));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_be_liberal));
	procsys_pr_str(s, PROCSYS_NFCT(udp_timeout));
	procsys_pr_str(s, PROCSYS_NFCT(udp_timeout_stream));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_timeout_established));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_timeout_syn_recv));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_timeout_syn_sent));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_timeout_fin_wait));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_timeout_close_wait));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_timeout_last_ack));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_timeout_time_wait));
	procsys_pr_str(s, PROCSYS_NFCT(tcp_timeout_close));
	procsys_pr_str(s, PROCSYS_NFCT(max));
	procsys_pr_str(s, PROCSYS_NFCT(expect_max));
	pr_seq(s, "----------------------\n");
}

static void tu_port_work_handler(struct work_struct *work)
{
	int i;
	unsigned long timeout;

	if (!flowmgr.enable_tu_port_track)
		return;
	timeout = flowmgr.tu_port_track_timeout * HZ;
	pr_debug("Clear port state\n");
	for (i = 0; i < 65536; i++) {
		if (tu_port_track[i].ts) {
			unsigned long now;
			unsigned long diff;

			now = jiffies;
			diff = now - tu_port_track[i].ts;

			if (diff > timeout)
				tu_port_track[i].ts = 0;
		}
	}

	queue_delayed_work(system_wq, &tu_port_dwork, 60*HZ);
}

int flowmgr_config(char *pri_lan_dev_name, char *wan_dev_name,
		   char *sec_lan_dev_name)
{
	if (fap()->config)
		fap()->config(pri_lan_dev_name,
			      wan_dev_name,
			      sec_lan_dev_name);

	flowmgr_enable_br_conntrack(flowmgr.enable_br_conntrack_call);

	/* Configure Netfilter Conntrack */
	procsys_wr_str(PROCSYS_NFCT(acct), NSTR(1));
	procsys_wr_str(PROCSYS_NFCT(tcp_be_liberal), NSTR(1));

	/* Configure UDP Connection Timeouts */
	procsys_wr_str(PROCSYS_NFCT(udp_timeout), NSTR(60));
	procsys_wr_str(PROCSYS_NFCT(udp_timeout_stream), NSTR(300));

	/* Configure TCP Connection Timeouts */
	procsys_wr_str(PROCSYS_NFCT(tcp_timeout_established), NSTR(300));
	procsys_wr_str(PROCSYS_NFCT(tcp_timeout_syn_recv), NSTR(60));
	procsys_wr_str(PROCSYS_NFCT(tcp_timeout_syn_sent), NSTR(60));
	procsys_wr_str(PROCSYS_NFCT(tcp_timeout_fin_wait), NSTR(10));
	procsys_wr_str(PROCSYS_NFCT(tcp_timeout_close_wait), NSTR(10));
	procsys_wr_str(PROCSYS_NFCT(tcp_timeout_last_ack), NSTR(1));
	procsys_wr_str(PROCSYS_NFCT(tcp_timeout_time_wait), NSTR(1));
	procsys_wr_str(PROCSYS_NFCT(tcp_timeout_close), NSTR(1));

	if (!tu_port_dwork .work.func) {
		INIT_DELAYED_WORK(&tu_port_dwork, tu_port_work_handler);
		queue_delayed_work(system_wq, &tu_port_dwork, 60*HZ);
	}

	return 0;
}
