 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/ethtool.h>
#include <uapi/linux/ethtool.h>
#include <linux/netdevice.h>
#include <linux/brcmxcvr.h>

#include "dqnet_priv.h"
#include "dqnet_ethtool.h"
#include "dqnet_dbg.h"
#include "dqnet_switch.h"

/* IEEE1905 stats array indicies for Ethernet devices. */
static char macdev_net_stats_strings[MACDEV_NET_STATS_MAX][ETH_GSTRING_LEN] =
MACDEV_NET_STATS_STRINGS;

static void dqnet_get_speed_duplex(struct phy_device *phydev, int *speed, int *duplex)
{
	int s, d;

	if (phydev->link) {
		s = phydev->speed;
		d = phydev->duplex;
	} else {
		struct brcmphy_xcvr *phyxcvr = dev_get_drvdata(&phydev->mdio.dev);

		if (phyxcvr)
			s = phyxcvr->speed;
		else
			s = SPEED_1000;

		d = DUPLEX_UNKNOWN;
	}

	if (speed)
		*speed = s;

	if (duplex)
		*duplex = d;
}

static int dqnet_get_link_ksettings(struct net_device *dev,
		struct ethtool_link_ksettings *cmd)
{
	int status = 0, speed, duplex;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct phy_device *phydev = dev->phydev;

	netdev_dbg(dev, "-->\n");

	memset(cmd, 0, sizeof(*cmd));
	if (ndev->link_type == DQNET_LINK_TYPE_PHY) {
		phy_ethtool_ksettings_get(phydev, cmd);
		dqnet_get_speed_duplex(phydev, &speed, &duplex);
		cmd->base.speed = speed;
		cmd->base.duplex = duplex;
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	} else if (ndev->link_type == DQNET_LINK_TYPE_SWITCH) {
		if (phydev)
			phy_ethtool_ksettings_get(phydev, cmd);

		status = SWITCH->get_port_config(dev);
		if (!status) {
			cmd->base.autoneg = ndev->link_autoneg;
			cmd->base.speed = ndev->link_speed;
			cmd->base.duplex = ndev->link_duplex;
		}
#endif
	} else {
		ethtool_link_ksettings_add_link_mode(cmd, advertising, 1000baseT_Full);
		ethtool_link_ksettings_add_link_mode(cmd, supported, 1000baseT_Full);
		cmd->base.port = PORT_OTHER;
		cmd->base.autoneg = AUTONEG_DISABLE;
		cmd->base.speed = SPEED_1000;
		cmd->base.duplex = DUPLEX_FULL;
	}

	netdev_dbg(dev, "<--\n");
	return status;
}

static int dqnet_set_link_ksettings(struct net_device *dev,
		const struct ethtool_link_ksettings *cmd)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct phy_device *phydev = dev->phydev;

	netdev_dbg(dev, "-->\n");

	if (ndev->link_type == DQNET_LINK_TYPE_PHY) {
		status = phy_ethtool_ksettings_set(phydev, cmd);
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	} else if (ndev->link_type == DQNET_LINK_TYPE_SWITCH) {
		if (phydev && (!brcmphy_link_configurable(phydev, cmd)))
			status = -ENXIO;

		if (!status) {
			ndev->link_autoneg = cmd->base.autoneg;
			ndev->link_speed = cmd->base.speed;
			ndev->link_duplex = cmd->base.duplex;
			ndev->link_pause = -1;
			status = SWITCH->set_port_config(dev);
		}

		if (!status && phydev)
			status = phy_ethtool_ksettings_set(phydev, cmd);
#endif
	}

	netdev_dbg(dev, "<--\n");
	return status;
}

static void dqnet_get_drvinfo(struct net_device *dev,
			      struct ethtool_drvinfo *info)
{
	netdev_dbg(dev, "-->\n");
	memset(info, 0, sizeof(struct ethtool_drvinfo));
	strncpy(info->driver, MODULE_NAME, sizeof(info->driver));
	strncpy(info->version, MODULE_VER, sizeof(info->version));
	info->n_stats = MACDEV_NET_STATS_MAX;
	netdev_dbg(dev, "<--\n");
}

u32 dqnet_get_msglevel(struct net_device *dev)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	netdev_dbg(dev, "-->\n");
	return ndev->msg_enable;
	netdev_dbg(dev, "<--\n");
}

void dqnet_set_msglevel(struct net_device *dev, u32 value)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	netdev_dbg(dev, "-->\n");
	ndev->msg_enable = value;
	netdev_dbg(dev, "<--\n");
}

static int dqnet_sset_count(struct net_device *dev, int cmd)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	enum ethtool_stringset ss = cmd;
	int count = -EINVAL;

	netdev_dbg(dev, "-->\n");

	if (ndev->hal.macdev) {
		count = haldev_get_sset_count(&ndev->hal, cmd);
		goto done;
	}

	if (ss == ETH_SS_STATS)
		count = MACDEV_NET_STATS_MAX;
	else
		count = 0;

done:
	netdev_dbg(dev, "<--\n");
	return count;
}

static void dqnet_get_strings(struct net_device *dev, u32 cmd, u8 *strings)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	enum ethtool_stringset ss = cmd;

	netdev_dbg(dev, "-->\n");

	if (ndev->hal.macdev) {
		haldev_get_strings(&ndev->hal, cmd, strings);
		goto done;
	}

	if (ss == ETH_SS_STATS)
		memcpy(strings, macdev_net_stats_strings,
		       sizeof(macdev_net_stats_strings));

done:
	netdev_dbg(dev, "<--\n");
}

static void dqnet_get_ethtool_stats(struct net_device *dev,
				    struct ethtool_stats *stats, u64 *data)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);

	netdev_dbg(dev, "-->\n");
	if (stats->cmd != ETHTOOL_GSTATS) {
		netdev_err(dev, "Invalid command, %d.\n", stats->cmd);
		goto done;
	}

	if (ndev->hal.macdev) {
		haldev_get_ethtool_stats(&ndev->hal, stats, data);
		goto done;
	}

	if (stats->n_stats != MACDEV_NET_STATS_MAX) {
		netdev_err(dev,
			   "Incorrect stats array size, %d. Must be %d.\n",
			   stats->n_stats, MACDEV_NET_STATS_MAX);
		goto done;
	}

	if (ndev->link_type == DQNET_LINK_TYPE_PHY) {
		const struct net_device_ops *ops = dev->netdev_ops;
		struct rtnl_link_stats64 storage;
		int speed;

		dqnet_get_speed_duplex(dev->phydev, &speed, NULL);
		memset(&storage, 0, sizeof(storage));
		if (ops->ndo_get_stats64)
			ops->ndo_get_stats64(dev, &storage);
		else if (ops->ndo_get_stats)
			netdev_stats_to_stats64(&storage, ops->ndo_get_stats(dev));
		else
			netdev_stats_to_stats64(&storage, &dev->stats);

		data[MACDEV_NET_STATS_TX_BYTES] = storage.tx_bytes;
		data[MACDEV_NET_STATS_TX_PACKETS] = storage.tx_packets;
		data[MACDEV_NET_STATS_TX_ERRORS] = storage.tx_errors;
		data[MACDEV_NET_STATS_TX_CAPACITY] = speed;
		data[MACDEV_NET_STATS_RX_BYTES] = storage.rx_bytes;
		data[MACDEV_NET_STATS_RX_PACKETS] = storage.rx_packets;
		data[MACDEV_NET_STATS_RX_ERRORS] = storage.rx_errors;
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	} else if (ndev->link_type == DQNET_LINK_TYPE_SWITCH) {
		struct dqnet_switch_stats switch_stats;

		if (SWITCH->get_mib_counters(dev, &switch_stats)) {
			netdev_err(dev, "Error retrieving switch stats.\n");
			goto done;
		}

		if (SWITCH->get_port_config(dev)) {
			netdev_err(dev, "Error retrieving switch config.\n");
			goto done;
		}

		data[MACDEV_NET_STATS_TX_BYTES] =
			switch_stats.tx_bytes;
		data[MACDEV_NET_STATS_TX_PACKETS] =
			switch_stats.tx_unicast_pkts +
			switch_stats.tx_broadcast_pkts +
			switch_stats.tx_multicast_pkts;
		data[MACDEV_NET_STATS_TX_ERRORS] =
			switch_stats.tx_drop_pkts +
			switch_stats.tx_excessive_collision +
			switch_stats.tx_frame_in_disc;
		data[MACDEV_NET_STATS_TX_CAPACITY] = ndev->link_speed;
		data[MACDEV_NET_STATS_RX_BYTES] =
			switch_stats.rx_bytes;
		data[MACDEV_NET_STATS_RX_PACKETS] =
			switch_stats.rx_unicast_pkts +
			switch_stats.rx_broadcast_pkts +
			switch_stats.rx_multicast_pkts;
		data[MACDEV_NET_STATS_RX_ERRORS] =
			switch_stats.rx_alignment_errors +
			switch_stats.rx_discard +
			switch_stats.rx_drop_pkts +
			switch_stats.rx_fcs_errors +
			switch_stats.rx_fragments +
			switch_stats.rx_in_range_errors +
			switch_stats.rx_jabbers +
			switch_stats.rx_out_of_range_errors +
			switch_stats.rx_oversize_pkts +
			switch_stats.rx_symbol_errors +
			switch_stats.rx_undersize_pkts;
#endif
	} else {
		data[MACDEV_NET_STATS_TX_BYTES] =
			ndev->stats.netdev.tx_bytes;
		data[MACDEV_NET_STATS_TX_PACKETS] =
			ndev->stats.netdev.tx_packets;
		data[MACDEV_NET_STATS_TX_ERRORS] =
			ndev->stats.netdev.tx_errors;
		data[MACDEV_NET_STATS_TX_CAPACITY] = SPEED_1000;
		data[MACDEV_NET_STATS_RX_BYTES] =
			ndev->stats.netdev.rx_bytes;
		data[MACDEV_NET_STATS_RX_PACKETS] =
			ndev->stats.netdev.rx_packets;
		data[MACDEV_NET_STATS_RX_ERRORS] =
			ndev->stats.netdev.rx_errors;
	}

done:
	netdev_dbg(dev, "<--\n");
}

int dqnet_get_eee(struct net_device *dev, struct ethtool_eee *eee)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);

	netdev_dbg(dev, "-->\n");
	if (eee->cmd != ETHTOOL_GEEE) {
		netdev_err(dev, "Invalid command, %d.\n", eee->cmd);
		status = -EINVAL;
		goto done;
	}

	if (ndev->hal.macdev) {
		status = haldev_get_eee(&ndev->hal, eee);
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	} else if (ndev->link_type == DQNET_LINK_TYPE_SWITCH) {
		struct dqnet_port_pwr pwr;

		status = SWITCH->get_port_pwr(dev, &pwr);
		if (status) {
			netdev_err(dev, "Error retrieving switch power "
					"config.\n");
			goto done;
		}
		eee->eee_enabled = pwr.eee_enabled;
		eee->tx_lpi_enabled = pwr.eee_tx_lpi_enabled;
		eee->tx_lpi_timer = pwr.eee_tx_lpi_timer;
		eee->eee_active = pwr.eee_active;
		eee->advertised = pwr.eee_advertised;
		eee->supported = pwr.eee_supported;
		eee->lp_advertised = pwr.eee_lp_advertised;
#endif
	} else {
		eee->advertised = 0;
		eee->supported = 0;
		eee->lp_advertised = 0;
		eee->eee_enabled = false;
		eee->eee_active = false;
		eee->tx_lpi_enabled = false;
		eee->tx_lpi_timer = 0;
	}

done:
	netdev_dbg(dev, "<--\n");
	return status;
}

int dqnet_set_eee(struct net_device *dev, struct ethtool_eee *eee)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);

	netdev_dbg(dev, "-->\n");
	if (eee->cmd != ETHTOOL_SEEE) {
		netdev_err(dev, "Invalid command, %d.\n", eee->cmd);
		status = -EINVAL;
		goto done;
	}

	if (ndev->hal.macdev) {
		status = haldev_set_eee(&ndev->hal, eee);
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	} else if (ndev->link_type == DQNET_LINK_TYPE_SWITCH) {
		struct dqnet_port_pwr pwr;

		status = SWITCH->get_port_pwr(dev, &pwr);
		if (status) {
			netdev_err(dev, "Error getting switch power "
					"config.\n");
			goto done;
		}
		pwr.eee_enabled = eee->eee_enabled;
		status = SWITCH->set_port_pwr(dev, &pwr);
		if (status) {
			netdev_err(dev, "Error setting switch power "
					"config.\n");
			goto done;
		}
#endif
	} else {
		status = -ENOTSUPP;
	}

done:
	netdev_dbg(dev, "<--\n");
	return status;
}

void dqnet_get_channels(struct net_device *dev,
			struct ethtool_channels *channel)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);

	channel->max_rx = ndev->chan->rx_q_count;
	channel->max_tx = ndev->chan->tx_q_count;
	channel->rx_count = ndev->chan->rx_q_count;
	channel->tx_count = ndev->chan->tx_q_count;
	channel->max_combined = channel->max_rx + channel->max_tx;
	channel->combined_count =  channel->rx_count + channel->tx_count;
}

struct ethtool_ops dqnet_ethernet_ethtool_ops = {
	.get_drvinfo		= dqnet_get_drvinfo,
	.get_link		= ethtool_op_get_link,
	.set_msglevel		= dqnet_set_msglevel,
	.get_msglevel		= dqnet_get_msglevel,
	.get_sset_count		= dqnet_sset_count,
	.get_strings		= dqnet_get_strings,
	.get_ethtool_stats	= dqnet_get_ethtool_stats,
	.get_eee		= dqnet_get_eee,
	.set_eee		= dqnet_set_eee,
	.get_link_ksettings	= dqnet_get_link_ksettings,
	.set_link_ksettings	= dqnet_set_link_ksettings,
	.get_channels		= dqnet_get_channels,
};
