 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/of.h>
#include <linux/io.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/init.h>
#include <linux/brcmstb/brcmstb.h>
#include <linux/bcm_media_gw/bpcm.h>
#include "pmb.h"
#include "bpcm_priv.h"
#include "bpcm_proc.h"


#define MODULE_NAME	"brcm-bpcm"
#define MODULE_VER	"1.0"

struct list_head bpcms;
spinlock_t bpcms_lock;

int bpcm_get_device(char *name, struct bpcm_device **bpcm)
{
	int status = -ENODEV;
	struct list_head *pos;
	struct bpcm_device *tmp;
	unsigned long flags;

	pr_debug("-->\n");
	*bpcm = NULL;
	spin_lock_irqsave(&bpcms_lock, flags);
	list_for_each(pos, &bpcms) {
		tmp = list_entry(pos, struct bpcm_device, list);
		if (!strncmp(tmp->name, name, sizeof(tmp->name))) {
			*bpcm = tmp;
			status = 0;
			break;
		}
	}
	spin_unlock_irqrestore(&bpcms_lock, flags);
	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(bpcm_get_device);

static int bpcm_set_zone_pwr(struct bpcm_device *bpcm, u32 zone, bool on)
{
	int status = 0;
	struct device *pmb_dev = bpcm->pmb_dev;
	struct device *bpcm_dev = &bpcm->pdev->dev;
	u8 id = bpcm->id;
	u32 val, mask;

	pr_debug("-->\n");
	val = ZONE_CTRL_BLK_RST_ASSERT_MASK |
		ZONE_CTRL_MEM_PWR_CNTL_EN_MASK |
		ZONE_CTRL_DPG_CNTL_EN_MASK;
	if (on) {
		dev_dbg(bpcm_dev, "%s: powering on zone %d.\n", bpcm->name,
			zone);
		val |= ZONE_CTRL_PWR_UP_REQ_MASK;
		mask = ZONE_CTRL_PWR_ON_STATE_MASK;
	} else {
		dev_dbg(bpcm_dev, "%s: powering off zone %d.\n", bpcm->name,
			zone);
		val |= ZONE_CTRL_PWR_DN_REQ_MASK;
		mask = ZONE_CTRL_PWR_OFF_STATE_MASK;
	}
	pmb_write(pmb_dev, id, ZONE_CTRL(zone), val);
	do {
		pmb_read(pmb_dev, id, ZONE_CTRL(zone), &val);
	} while (!(val & mask));
	dev_dbg(bpcm_dev, "%s: zone %d powered %s.\n", bpcm->name, zone,
		 on ? "on" : "off");

	pr_debug("<--\n");
	return status;
}

int bpcm_pwr_on(struct bpcm_device *bpcm)
{
	int status = 0;
	int i;
	struct device *bpcm_dev = &bpcm->pdev->dev;

	pr_debug("-->\n");
	for (i = 0; i < bpcm->zones; i++) {
		status = bpcm_set_zone_pwr(bpcm, i, true);
		if (status) {
			dev_err(bpcm_dev, "Failure turning on zone %d "
					  "power.\n", i);
			goto done;
		}
	}
done:
	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(bpcm_pwr_on);

int bpcm_pwr_off(struct bpcm_device *bpcm)
{
	int status = 0;
	int i;
	struct device *bpcm_dev = &bpcm->pdev->dev;

	pr_debug("-->\n");
	for (i = 0; i < bpcm->zones; i++) {
		status = bpcm_set_zone_pwr(bpcm, i, false);
		if (status) {
			dev_err(bpcm_dev, "Failure turning off zone %d "
					  "power.\n", i);
			goto done;
		}
	}
done:
	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(bpcm_pwr_off);

int bpcm_read(struct bpcm_device *bpcm, u32 reg, u32 *val)
{
	int status;
	struct device *pmb_dev = bpcm->pmb_dev;
	struct device *bpcm_dev = &bpcm->pdev->dev;
	u8 id = bpcm->id;

	pr_debug("-->\n");
	if (reg > MAX_REG_OFF) {
		status = -EINVAL;
		goto done;
	}
	status = pmb_read(pmb_dev, id, reg, val);
	dev_dbg(bpcm_dev, "%s: reg 0x%08x: 0x%08x.\n",
		 bpcm->name, reg, *val);
	pr_debug("<--\n");
done:
	return status;
}
EXPORT_SYMBOL(bpcm_read);

int bpcm_write(struct bpcm_device *bpcm, u32 reg, u32 val)
{
	int status = 0;
	struct device *pmb_dev = bpcm->pmb_dev;
	struct device *bpcm_dev = &bpcm->pdev->dev;
	u8 id = bpcm->id;

	pr_debug("-->\n");
	if (reg > MAX_REG_OFF) {
		status = -EINVAL;
		goto done;
	}
	status = pmb_write(pmb_dev, id, reg, val);
	dev_dbg(bpcm_dev, "%s: reg 0x%08x: 0x%08x.\n",
		 bpcm->name, reg, val);
	pr_debug("<--\n");
done:
	return status;
}
EXPORT_SYMBOL(bpcm_write);

int bpcm_assert_soft_reset(struct bpcm_device *bpcm)
{
	int status;
	pr_debug("-->\n");
	status = bpcm_write(bpcm, SOFTRESET, 1);
	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(bpcm_assert_soft_reset);

int bpcm_deassert_soft_reset(struct bpcm_device *bpcm)
{
	int status;
	pr_debug("-->\n");
	status = bpcm_write(bpcm, SOFTRESET, 0);
	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(bpcm_deassert_soft_reset);

int bpcm_set_clk_div(struct bpcm_device *bpcm)
{
	int status = 0;
	struct device *pmb_dev = bpcm->pmb_dev;
	struct device *bpcm_dev = &bpcm->pdev->dev;
	u8 id = bpcm->id;
	u32 val;

	pr_debug("-->\n");
	if (bpcm->clk_div[0] < 0 || bpcm->clk_div[1] < 0) {
		status = -EINVAL;
		goto done;
	}

	pmb_read(pmb_dev, id, ZONE_FREQ_CTRL(0), &val);
	val &= ~ZONE_FREQ_HIGH_GEAR_DIV_MASK;
	val &= ~ZONE_FREQ_LOW_GEAR_DIV_MASK;
	val |= (bpcm->clk_div[0] << ZONE_FREQ_HIGH_GEAR_DIV_SHIFT) &
		ZONE_FREQ_HIGH_GEAR_DIV_MASK;
	val |= (bpcm->clk_div[1] << ZONE_FREQ_LOW_GEAR_DIV_SHIFT) &
		ZONE_FREQ_LOW_GEAR_DIV_MASK;
	val |= ZONE_FREQ_USE_DYN_GEAR_SEL_MASK;
	pmb_write(pmb_dev, id, ZONE_FREQ_CTRL(0), val);
	dev_dbg(bpcm_dev, "%s: dynamic clk scaling enabled.\n", bpcm->name);
	dev_dbg(bpcm_dev,
	        "%s: clk dividers set to 1/%lu, 1/%lu (high, low).\n",
	        bpcm->name, BIT(bpcm->clk_div[0]), BIT(bpcm->clk_div[1]));

done:
	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(bpcm_set_clk_div);

static int bpcm_probe(struct platform_device *pdev)
{
	int status = 0;
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct device_node *parent;
	struct bpcm_device *bpcm;
	const char *str;
	unsigned long flags;
	int zone_init_pwr;
	int i;

	pr_debug("-->\n");

	bpcm = devm_kzalloc(dev, sizeof(struct bpcm_device), GFP_KERNEL);
	if (IS_ERR_OR_NULL(bpcm)) {
		dev_err(dev, "Failed to allocate BPCM device.\n");
		status = PTR_ERR(bpcm);
		goto done;
	}
    bpcm->pdev = pdev;
	INIT_LIST_HEAD(&bpcm->list);
	platform_set_drvdata(pdev, bpcm);

	status = of_property_read_u32(np, "reg", &bpcm->id);
	if (status) {
		dev_err(dev, "Failure reading BPCM reg property.\n");
		goto done;
	}

	status = of_property_read_string(np, "dev-name", &str);
	if (status) {
		dev_err(dev, "Failure reading dev-name property\n");
		goto done;
	}
	strncpy(bpcm->name, str, sizeof(bpcm->name));
	bpcm->name[sizeof(bpcm->name) - 1] = '\0';

	status = of_property_read_u32(np, "zones", &bpcm->zones);
	if (status) {
		dev_err(dev, "Failure reading zones property.\n");
		goto done;
	}
	i = of_property_read_u32(np, "zone-init-pwr",
			&zone_init_pwr);
	if (i)
		zone_init_pwr = i;
	i = of_property_read_u32_array(np, "dyn-clk-scale-div",
				       &bpcm->clk_div[0], 2);
	if (!i) {
		bpcm->clk_div[0] = ffs(bpcm->clk_div[0]) - 1;
		bpcm->clk_div[1] = ffs(bpcm->clk_div[1]) - 1;
	} else {
		bpcm->clk_div[0] = i;
		bpcm->clk_div[1] = i;
	}

	parent = of_get_parent(np);
	bpcm->pmb_dev = pmb_register(parent);
	if (IS_ERR(bpcm->pmb_dev)) {
		dev_err(dev, "Failed to register with PMB device.\n");
		status = PTR_ERR(bpcm->pmb_dev);
		goto done;
	}

	spin_lock_irqsave(&bpcms_lock, flags);
	list_add(&bpcm->list, &bpcms);
	spin_unlock_irqrestore(&bpcms_lock, flags);

	if (zone_init_pwr >= 0)
		for (i = 0; i < bpcm->zones; i++)
			bpcm_set_zone_pwr(bpcm, i,
				(zone_init_pwr >> i) & 1);

	if (bpcm->clk_div[0] >= 0 && bpcm->clk_div[1] >= 0)
		bpcm_set_clk_div(bpcm);

done:
	pr_debug("<--\n");
	return status;
}

static int bpcm_remove(struct platform_device *pdev)
{
	int status = 0;
	struct list_head *pos;
	struct bpcm_device *bpcm = platform_get_drvdata(pdev);
	unsigned long flags;

	pr_debug("-->\n");
	pmb_unregister(bpcm->pmb_dev);
	spin_lock_irqsave(&bpcms_lock, flags);
	list_for_each(pos, &bpcms) {
		bpcm = list_entry(pos, struct bpcm_device, list);
		if (bpcm->pdev == pdev) {
			list_del(pos);
			break;
		}
	}
	spin_unlock_irqrestore(&bpcms_lock, flags);
	pr_debug("<--\n");
	return status;
}

static const struct of_device_id bpcm_of_match[] = {
	{.compatible = "brcm,bpcm"},
	{}
};

MODULE_DEVICE_TABLE(of, bpcm_of_match);

static struct platform_driver bpcm_driver = {
	.probe  = bpcm_probe,
	.remove = bpcm_remove,
	.driver = {
		.name = MODULE_NAME,
		.owner = THIS_MODULE,
		.of_match_table = bpcm_of_match
	}
};

static int __init bpcm_init(void)
{
	pr_debug("%s driver v%s\n", MODULE_NAME, MODULE_VER);
	INIT_LIST_HEAD(&bpcms);
	spin_lock_init(&bpcms_lock);
	bpcm_proc_init();
	return platform_driver_register(&bpcm_driver);
}
arch_initcall(bpcm_init);

static void __exit bpcm_exit(void)
{
	bpcm_proc_exit();
	platform_driver_unregister(&bpcm_driver);
}
module_exit(bpcm_exit);

MODULE_AUTHOR("Tim Ross");
MODULE_DESCRIPTION("Block Power Control Module (BPCM) Driver");
MODULE_LICENSE("GPL");
