/*
 * BCM RPC EDAC (error detection and correction)
 *
 * Copyright (C) 2015-2021 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef CONFIG_BCM_KF_CM

#include <linux/ctype.h>
#include <linux/edac.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/regmap.h>
#include <linux/delay.h>
#include <linux/ktime.h>
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <linux/bcm_media_gw/itc_rpc/itc_msg_defs.h>
#include <linux/bcm_media_gw/itc_rpc/itc_channel_defs.h>
#include "edac_core.h"
#include "edac_module.h"

#define MODULE_NAME "bcm-rpc-edac"

#define ANSI_RED	"\e[31;1m"
#define ANSI_YELLOW	"\e[33;1m"
#define ANSI_RESET	"\e[0m"
#define RED(str)	ANSI_RED str ANSI_RESET
#define YLW(str)	ANSI_YELLOW str ANSI_RESET

#define RPC_SYS_FUNC_MAX	4

#define MAX_RPC_SYS_MC_PID_STR_LEN 8
#define RPC_SYS_MC_PID_LIST \
	RPC_SYS_MC_PID_INIT(RPC_SYS_PID_MEMSYS,		"MEMSYS")\
	RPC_SYS_MC_PID_INIT(RPC_SYS_PID_MEMC0,		"MEMC0")\
	RPC_SYS_MC_PID_INIT(RPC_SYS_PID_MEMC1,		"MEMC1")\
	RPC_SYS_MC_PID_INIT(RPC_SYS_PID_MEMRGNS,	"MEMRGNS")\

#define RPC_SYS_MC_PID_INIT(index, param_name) index,
typedef enum {
	RPC_SYS_MC_PID_LIST
	MAX_RPC_SYS_MC_PID_LIST
} rpc_sys_mc_pid;
#undef RPC_SYS_MC_PID_INIT

/* SYS SVC Param Info */
typedef struct {
	const char name[MAX_RPC_SYS_MC_PID_STR_LEN];
} rpc_sys_mc_pid_struct;

struct sys_memregn_info {
	uint32_t rgn_sz;
	uint32_t sys_phys_addr;
	uint32_t trans_addr;
};

struct sys_memcx_info {
	bool ecc;
	uint32_t phy_width;
	uint32_t dev_width;
	uint32_t dev_size;
};

struct sys_ecc_info {
	uint16_t ce_err_cnt;
	uint16_t ue_err_cnt;
	uint64_t err_addr;
	uint8_t syndrome;
	bool clr_ce_cnt;
	bool clr_ue_cnt;
	bool clr_last_err;
};

struct sys_mem_info {
	struct sys_memregn_info memregn;
	struct sys_memcx_info memcx;
	struct sys_ecc_info ecc;
	uint32_t n_memc;
	uint32_t brd_mem_sz;
};

static rpc_function rpc_sys_mc_services_tbl[] = {
	{ NULL,	0 },
	{ NULL,	0 },
	{ NULL,	0 },
	{ NULL,	0 }
};

enum rpc_sys_mc_func {
	RPC_SYS_FUNC_GET_HW_CFG_PARAM_ID = 1,
	RPC_SYS_FUNC_GET_HW_CFG = 3,
	RPC_SYS_FUNC_GET_ECC_ERR_INFO = 25,
	RPC_SYS_FUNC_INJECT_ECC_ERR = 27,
};

/**
 * This structure represents coordinates of SMC SYS MC data
 */
struct rpc_sys_mc_data {
	struct device *dev;
	int tunnel;
	char tunnel_name[32];
	void *data;
};

/**
 * This structure represents SMC SYS Param ID request data format
 */
struct rpc_sys_mc_pid_req_msg {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t rsvd_w1:32;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t name0:8;
			uint32_t name1:8;
			uint32_t name2:8;
			uint32_t name3:8;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t name4:8;
			uint32_t name5:8;
			uint32_t name6:8;
			uint32_t name7:8;
		};
	};
};

/**
 * This structure represents SMC SYS Param ID response data format
 */
struct rpc_sys_mc_pid_res {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t param_id:16;
			uint32_t rsvd1:8;
			uint32_t rc:8;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t rsvd_w2:32;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t rsvd_w3:32;
		};
	};
};

/**
 * This structure represents SMC SYS MC request data format
 */
struct rpc_sys_mc_req_msg {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t param_id:16;
			uint32_t buf_w1:8;
			uint32_t rsvd1:8;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t buf_w2:32;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t rsvd_w3:32;
		};
	};
};

/**
 * This structure represents SMC SYS memory response data format
 */
struct rpc_sys_memsys_res {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t n_memc:3;
			uint32_t rsvd1:5;
			uint32_t brd_mem_sz:16;
			uint32_t rc:8;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t rsvd_w2:32;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t rsvd_w3:32;
		};
	};
};

/**
 * This structure represents SMC SYS MC response data format
 */
struct rpc_sys_memcx_res {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t dev_size:6;
			uint32_t dev_width:4;
			uint32_t phy_width:4;
			uint32_t ecc:1;
			uint32_t rsvd1:9;
			uint32_t rc:8;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t rsvd_w2:32;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t rsvd_w3:32;
		};
	};
};

/**
 * This structure represents SMC SYS MREGN response data format
 */
struct rpc_sys_memregn_res {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t rgn_sz:20;
			uint32_t rsvd1:4;
			uint32_t rc:8;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t sys_phys_addr:28;
			uint32_t rsvd2:4;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t trans_addr:28;
			uint32_t rsvd3:4;
		};
	};
};

/**
 * This structure represents SMC SYS MC ECC Error request data
 * format
 */
struct rpc_sys_mc_ecc_err_req_msg {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t memcidx:3;
			uint32_t clr_ce_cnt:1;
			uint32_t clr_ue_cnt:1;
			uint32_t clr_last_err:1;
			uint32_t rsvd1:26;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t rsvd_w2:32;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t rsvd_w3:32;
		};
	};
};

/**
 * This structure represents SMC SYS ECC Error response data
 * format
 */
struct rpc_sys_mc_ecc_err_res {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t memcidx:3;
			uint32_t rsvd1:5;
			uint32_t ce:1;
			uint32_t ue:1;
			uint32_t syndrome:6;
			uint32_t err_addr_high:8;
			uint32_t rc:8;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t err_addr_low:32;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t ce_err_cnt:16;
			uint32_t ue_err_cnt:16;
		};
	};
};

/**
 * This structure represents SMC SYS MC Inject ECC Error request
 * data format
 */
struct rpc_sys_mc_inject_ecc_err_req_msg {
	uint32_t hdr;
	union {
		uint32_t w1;
		struct {
			uint32_t memcidx:3;
			uint32_t num_bits:1;
			uint32_t rsvd1:28;
		};
	};
	union {
		uint32_t w2;
		struct {
			uint32_t rsvd_w2:32;
		};
	};
	union {
		uint32_t w3;
		struct {
			uint32_t rsvd_w3:32;
		};
	};
};

struct bcm_rpc_edac_priv {
	struct device *dev;
	spinlock_t lock;
	struct dentry *dfs;
	struct list_head mcus;
	struct mutex mc_lock;
	int mc_active_mask;
	int mc_registered_mask;
	struct rpc_sys_mc_data *sys_mc_data;
};

struct bcm_rpc_edac_mc_ctx {
	struct list_head next;
	char *name;
	struct mem_ctl_info	*mci;
	struct bcm_rpc_edac_priv *edac_priv;
	u32 mc_idx;
};

static inline u8 rpc_sys_msg_get_retcode(rpc_msg *msg)
{
	return ((msg->data[0] >> 24) & 0xff);
}

int register_rpc_sys_mc(struct rpc_sys_mc_data *sys_mc_data, struct device *dev,
	const char *tunnel_name)
{
	if (!dev || !sys_mc_data) {
		pr_err("%s: Input data invalid.\n",
			__func__);
		return -EINVAL;
	}

	sys_mc_data->tunnel_name[0] = '\0';
	if (tunnel_name && *tunnel_name) {
		strncpy(sys_mc_data->tunnel_name, tunnel_name,
			sizeof(sys_mc_data->tunnel_name));
		sys_mc_data->tunnel_name[
			sizeof(sys_mc_data->tunnel_name)-1] = '\0';
	}
	sys_mc_data->tunnel = rpc_get_fifo_tunnel_id(
	   (char *) sys_mc_data->tunnel_name);
	if (sys_mc_data->tunnel < 0) {
		dev_err(dev, "%s: Unable to obtain RPC tunnel ID.\n",
			__func__);
		return -EIO;
	}
	sys_mc_data->dev = dev;

	return 0;
}

void release_rpc_sys_mc(struct rpc_sys_mc_data *sys_mc_data)
{
	if (!sys_mc_data) {
		pr_err("%s: System MC data invalid.\n",
			__func__);
		return;
	}
	memset(sys_mc_data, 0, sizeof(struct rpc_sys_mc_data));
}

int register_rpc_sys_mc_from_platform_device(
		struct rpc_sys_mc_data *sys_mc_data,
		struct platform_device *pdev)
{
	struct device_node *phan_node;
	struct device_node *of_node = pdev->dev.of_node;
	const char *dev_name;

	if (!sys_mc_data) {
		dev_err(&pdev->dev, "%s: System MC data invalid.\n",
			__func__);
		return -EINVAL;
	}

	phan_node = of_parse_phandle(of_node, "rpc-channel", 0);
	if (!phan_node) {
		dev_err(&pdev->dev, "Unable to retrieve rpc-channel phandle ");
		return -EINVAL;
	}

	if (of_property_read_string(phan_node, "dev-name", &dev_name)) {
		dev_err(&pdev->dev, "%s: Missing dev-name property!\n",
				of_node_full_name(of_node));
		of_node_put(phan_node);
		return -EINVAL;
	}
	of_node_put(phan_node);

	return register_rpc_sys_mc(sys_mc_data, &pdev->dev, dev_name);
}

static int rpc_sys_mc_get_hw_cfg(struct rpc_sys_mc_data *sys_mc_data)
{
	rpc_msg msg;
	int rc = 0;
	struct rpc_sys_mc_req_msg *req;

	if (!sys_mc_data) {
		pr_err(RED("%s: System MC data invalid.\n"),
			__func__);
		return -EINVAL;
	}
	req = (struct rpc_sys_mc_req_msg *)sys_mc_data->data;
	rpc_msg_init(&msg, RPC_SERVICE_SYS, RPC_SYS_FUNC_GET_HW_CFG,
				 0, req->w1, 0, 0);

	rc = rpc_send_request(sys_mc_data->tunnel, &msg);
	if (unlikely(rc)) {
		pr_err(RED("%s : rpc_send_request failure (%d)for id: %d\n"),
			   MODULE_NAME, rc, req->param_id);
		rpc_dump_msg(&msg);
		return rc;
	}
	if (rpc_sys_msg_get_retcode(&msg) == 0xff) {
		pr_err(RED("%s : invalid response %d\n"),
		   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}
	memcpy((char *)sys_mc_data->data, (char *)&msg, sizeof(msg));

	return rc;
}

static int rpc_sys_mc_get_ecc_err_info(struct rpc_sys_mc_data *sys_mc_data)
{
	rpc_msg msg;
	int rc = 0;
	struct rpc_sys_mc_ecc_err_req_msg *req;

	if (!sys_mc_data) {
		pr_err(RED("%s: System MC data invalid.\n"),
			__func__);
		return -EINVAL;
	}
	req = (struct rpc_sys_mc_ecc_err_req_msg *)sys_mc_data->data;

	rpc_msg_init(&msg, RPC_SERVICE_SYS, RPC_SYS_FUNC_GET_ECC_ERR_INFO,
				 0, req->w1, 0, 0);

	rc = rpc_send_request(sys_mc_data->tunnel, &msg);
	if (unlikely(rc)) {
		pr_err(RED("%s : rpc_send_request failure (%d)for func: %d\n"),
			   MODULE_NAME, rc, RPC_SYS_FUNC_GET_ECC_ERR_INFO);
		rpc_dump_msg(&msg);
		return rc;
	}
	if (rpc_sys_msg_get_retcode(&msg) == 0xff) {
		pr_err(RED("%s : invalid response %d\n"),
		   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}
	memcpy((char *)sys_mc_data->data, (char *)&msg, sizeof(msg));

	return rc;
}

static int rpc_sys_mc_inject_ecc_err(struct rpc_sys_mc_data *sys_mc_data)
{
	rpc_msg msg;
	int rc = 0;
	struct rpc_sys_mc_inject_ecc_err_req_msg *req;

	if (!sys_mc_data) {
		pr_err(RED("%s: System MC data invalid.\n"),
			__func__);
		return -EINVAL;
	}
	req = (struct rpc_sys_mc_inject_ecc_err_req_msg *)sys_mc_data->data;

	rpc_msg_init(&msg, RPC_SERVICE_SYS, RPC_SYS_FUNC_INJECT_ECC_ERR,
				 0, req->w1, 0, 0);

	rc = rpc_send_request(sys_mc_data->tunnel, &msg);
	if (unlikely(rc)) {
		pr_err(RED("%s : rpc_send_request failure (%d)for func: %d\n"),
			   MODULE_NAME, rc, RPC_SYS_FUNC_INJECT_ECC_ERR);
		rpc_dump_msg(&msg);
		return rc;
	}
	if (rpc_sys_msg_get_retcode(&msg) == 0xff) {
		pr_err(RED("%s : invalid response %d\n"),
		   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}

	return rc;
}

#define RPC_SYS_MC_PID_INIT(index, param_name) {param_name},
static rpc_sys_mc_pid_struct mc_pid_list[] = {
	RPC_SYS_MC_PID_LIST
};
static int rpc_sys_mc_get_hw_cfg_id(struct rpc_sys_mc_data *sys_mc_data,
				rpc_sys_mc_pid param_id, uint32_t *smc_param_id)
{
	int rc = 0;
	struct rpc_sys_mc_pid_req_msg req;
	struct rpc_sys_mc_pid_res *res;
	rpc_msg msg;

	if (param_id >= MAX_RPC_SYS_MC_PID_LIST) {
		pr_err(RED("%s: Param ID invalid.\n"), __func__);
		return -EINVAL;
	}
	memset(&req, 0, sizeof(struct rpc_sys_mc_pid_req_msg));
	memcpy(&req.w2, &mc_pid_list[param_id].name[0],
		   sizeof(req.w2));
	memcpy(&req.w3, &mc_pid_list[param_id].name[4],
		   sizeof(req.w3));

	rpc_msg_init(&msg, RPC_SERVICE_SYS, RPC_SYS_FUNC_GET_HW_CFG_PARAM_ID,
				 0, req.w1, req.w2, req.w3);

	rc = rpc_send_request(sys_mc_data->tunnel, &msg);
	if (unlikely(rc)) {
		pr_err(RED("%s : rpc_send_request failure (%d) for func: %d\n"),
			   MODULE_NAME, rc, RPC_SYS_FUNC_GET_HW_CFG_PARAM_ID);
		rpc_dump_msg(&msg);
		return rc;
	}

	res = (struct rpc_sys_mc_pid_res *)&msg;
	*smc_param_id = res->param_id;

	return rc;
}

static int rpc_sys_mc_get_memsys_cfg(struct rpc_sys_mc_data *sys_mc_data,
			struct sys_mem_info *info, rpc_sys_mc_pid param_id)
{
	int rc = 0;
	struct rpc_sys_mc_req_msg req;
	struct rpc_sys_memsys_res *res;
	uint32_t smc_param_id = 0;

	rc = rpc_sys_mc_get_hw_cfg_id(sys_mc_data, param_id, &smc_param_id);
	if (rc)
		return rc;

	sys_mc_data->data = (struct rpc_sys_mc_req_msg *)&req;
	req.param_id = smc_param_id;
	rc = rpc_sys_mc_get_hw_cfg(sys_mc_data);
	if (rc) {
		pr_info("%s : MC System memory get config failure (%d)\n",
			   MODULE_NAME, rc);
		return rc;
	}
	res = (struct rpc_sys_memsys_res *)sys_mc_data->data;
	info->n_memc = res->n_memc;
	info->brd_mem_sz = res->brd_mem_sz;
	pr_debug("%s : n_memc:%u brd_mem_sz:%x\n",
		MODULE_NAME, res->n_memc, res->brd_mem_sz);

	return rc;
}

static int rpc_sys_mc_get_memcx_cfg(struct rpc_sys_mc_data *sys_mc_data,
			struct sys_mem_info *info, rpc_sys_mc_pid param_id)
{
	int rc = 0;
	struct rpc_sys_mc_req_msg req;
	struct rpc_sys_memcx_res *res;
	uint32_t smc_param_id = 0;

	rc = rpc_sys_mc_get_hw_cfg_id(sys_mc_data, param_id, &smc_param_id);
	if (rc)
		return rc;

	sys_mc_data->data = (struct rpc_sys_mc_req_msg *)&req;
	req.param_id = smc_param_id;

	rc = rpc_sys_mc_get_hw_cfg(sys_mc_data);
	if (rc) {
		pr_info("%s : MC System memory get config failure (%d)\n",
			   MODULE_NAME, rc);
		return rc;
	}
	res = (struct rpc_sys_memcx_res *)sys_mc_data->data;
	info->memcx.ecc = res->ecc;
	info->memcx.phy_width = (0x1 << res->phy_width);
	info->memcx.dev_width = (0x1 << res->dev_width);
	info->memcx.dev_size = (0x1 << res->dev_size);
	pr_debug("%s : ecc:%u phy_width:%x dev_width:%x dev_size:%x\n",
		MODULE_NAME, res->ecc,
		res->phy_width, res->dev_width, res->dev_size);

	return rc;
}

static int rpc_sys_mc_get_memregion_cfg(struct rpc_sys_mc_data *sys_mc_data,
			struct sys_mem_info *info, rpc_sys_mc_pid param_id)
{
	int rc = 0;
	struct rpc_sys_mc_req_msg req;
	struct rpc_sys_memregn_res *res;
	uint32_t smc_param_id = 0;

	rc = rpc_sys_mc_get_hw_cfg_id(sys_mc_data, param_id, &smc_param_id);
	if (rc)
		return rc;

	sys_mc_data->data = (struct rpc_sys_mc_req_msg *)&req;
	req.param_id = smc_param_id;

	rc = rpc_sys_mc_get_hw_cfg(sys_mc_data);
	if (rc) {
		pr_info("%s : MC System memory get config failure (%d)\n",
			   MODULE_NAME, rc);
		return rc;
	}
	res = (struct rpc_sys_memregn_res *)sys_mc_data->data;
	info->memregn.rgn_sz = res->rgn_sz;
	info->memregn.sys_phys_addr = res->sys_phys_addr;
	info->memregn.trans_addr = res->trans_addr;
	pr_debug("%s : rgn_sz:%u sys_phys_addr:%x trans_addr:%x\n",
		MODULE_NAME, res->rgn_sz,
		res->sys_phys_addr, res->trans_addr);

	return rc;
}

static int rpc_sys_mc_get_ecc_info(struct rpc_sys_mc_data *sys_mc_data,
			struct sys_ecc_info *info, int memcidx)
{
	int rc = 0;
	struct rpc_sys_mc_ecc_err_req_msg req;
	struct rpc_sys_mc_ecc_err_res *res;

	sys_mc_data->data = (struct rpc_sys_mc_ecc_err_req_msg *)&req;
	req.memcidx = memcidx;
	req.clr_ce_cnt = info->clr_ce_cnt;
	req.clr_ue_cnt = info->clr_ue_cnt;
	req.clr_last_err = info->clr_last_err;

	rc = rpc_sys_mc_get_ecc_err_info(sys_mc_data);
	if (rc) {
		pr_info("%s : MC System memory get config failure (%d)\n",
			   MODULE_NAME, rc);
		return rc;
	}
	res = (struct rpc_sys_mc_ecc_err_res *)sys_mc_data->data;
	info->ce_err_cnt = res->ce_err_cnt;
	info->ue_err_cnt = res->ue_err_cnt;
	info->syndrome = res->syndrome;
	info->err_addr = (((uint64_t)res->err_addr_high) << 32 |
		res->err_addr_low);
	pr_debug("%s : CE:%u UE:%u SYND:%u ADDR:%llx\n",
		MODULE_NAME, res->ce_err_cnt,
		res->ue_err_cnt, res->syndrome,
		info->err_addr);

	return rc;
}

static int rpc_sys_mc_inject_error(struct rpc_sys_mc_data *sys_mc_data,
			int num_bits, int memcidx)
{
	int rc = 0;
	struct rpc_sys_mc_inject_ecc_err_req_msg req;

	sys_mc_data->data = (struct rpc_sys_mc_inject_ecc_err_req_msg *)&req;
	req.memcidx = memcidx;
	req.num_bits = num_bits;

	rc = rpc_sys_mc_inject_ecc_err(sys_mc_data);
	if (rc) {
		pr_info("%s : MC System inject ecc error failure (%d)\n",
			   MODULE_NAME, rc);
		return rc;
	}

	return rc;
}

static ssize_t bcm_rpc_edac_mc_err_inject_write(struct file *file,
					      const char __user *data,
					      size_t count, loff_t *ppos)
{
	int rc = 0;
	struct mem_ctl_info *mci = file->private_data;
	struct bcm_rpc_edac_mc_ctx *ctx = mci->pvt_info;
	struct rpc_sys_mc_data *sys_mc_data = ctx->edac_priv->sys_mc_data;
	uint32_t num_bits;

	if (count == 2) {
		num_bits = 1;
		edac_printk(KERN_ALERT, EDAC_MC,
			    "Inject Single bit error\n");
	} else if (count == 3) {
		num_bits = 2;
		edac_printk(KERN_ALERT, EDAC_MC,
			    "Inject Double bit error\n");
	} else {
		edac_printk(KERN_ALERT, EDAC_MC,
			    "Inject more then double bit error not supported\n");
		return count;
	}

	/* Inject errors by communication with SMC through RPC */
	rc = rpc_sys_mc_inject_error(sys_mc_data, num_bits, ctx->mc_idx);
	if (rc) {
		pr_info("%s : MC Inject ECC error failure (%d)\n",
			   MODULE_NAME, rc);
		return count;
	}

	return count;
}

static const struct file_operations bcm_rpc_edac_mc_debug_inject_fops = {
	.open = simple_open,
	.write = bcm_rpc_edac_mc_err_inject_write,
	.llseek = generic_file_llseek,
};

static void bcm_rpc_edac_mc_create_debugfs_node(struct mem_ctl_info *mci)
{
	if (!IS_ENABLED(CONFIG_EDAC_DEBUG))
		return;

	if (!mci->debugfs)
		return;

	edac_debugfs_create_file("inject_error", 0200, mci->debugfs, mci,
				 &bcm_rpc_edac_mc_debug_inject_fops);
}

static void bcm_rpc_edac_mc_check(struct mem_ctl_info *mci)
{
	struct bcm_rpc_edac_mc_ctx *ctx = mci->pvt_info;
	struct rpc_sys_mc_data *sys_mc_data = ctx->edac_priv->sys_mc_data;
	struct sys_ecc_info info;
	unsigned long pfn, offset;
	int rc = 0;

	info.clr_ce_cnt = true;
	info.clr_ue_cnt = true;
	info.clr_last_err = true;
	rc = rpc_sys_mc_get_ecc_info(sys_mc_data, &info, ctx->mc_idx);

	if (rc) {
		pr_info("%s : MC ECC get failure (%d)\n",
			   MODULE_NAME, rc);
		return;
	}
	pfn = info.err_addr >> PAGE_SHIFT;
	offset = info.err_addr & ~PAGE_MASK;

	/* Detected uncorrectable memory error */
	if (info.ue_err_cnt)
		edac_mc_handle_error(HW_EVENT_ERR_UNCORRECTED, mci,
			info.ue_err_cnt, pfn, offset, 0, 0, 0, -1,
			mci->ctl_name, "");

	/* Detect correctable memory error */
	if (info.ce_err_cnt)
		edac_mc_handle_error(HW_EVENT_ERR_CORRECTED, mci,
			info.ce_err_cnt, pfn, offset, info.syndrome,
			0, 0, -1, mci->ctl_name, "");
}

static int bcm_rpc_edac_mc_add(
			struct bcm_rpc_edac_priv *edac_priv,
			struct device_node *np)
{
	struct mem_ctl_info *mci;
	struct edac_mc_layer layers[2];
	struct bcm_rpc_edac_mc_ctx tmp_ctx;
	struct bcm_rpc_edac_mc_ctx *ctx;
	struct sys_mem_info info;
	uint64_t mem_size;
	struct dimm_info *dimm;
	int rc;

	memset(&tmp_ctx, 0, sizeof(tmp_ctx));
	tmp_ctx.edac_priv = edac_priv;

	if (!devres_open_group(edac_priv->dev, bcm_rpc_edac_mc_add, GFP_KERNEL))
		return -ENOMEM;

	/* Ignore non-active MCU */
	if (of_property_read_u32(np, "memory-controller", &tmp_ctx.mc_idx)) {
		dev_err(edac_priv->dev, "no memory-controller property\n");
		rc = -ENODEV;
		goto err_group;
	}
	rc = rpc_sys_mc_get_memsys_cfg(edac_priv->sys_mc_data, &info,
			RPC_SYS_PID_MEMSYS);
	if (rc) {
		dev_err(edac_priv->dev,
			"System memory information get failure\n");
		rc = -ENODEV;
		goto err_group;
	}
	if (tmp_ctx.mc_idx > info.n_memc) {
		dev_err(edac_priv->dev,
			"memory-controller %d exceeds hardware limit\n",
			tmp_ctx.mc_idx);
		rc = -ENODEV;
		goto err_group;
	}
	rc = rpc_sys_mc_get_memcx_cfg(edac_priv->sys_mc_data, &info,
			(RPC_SYS_PID_MEMC0+tmp_ctx.mc_idx));
	if (rc) {
		dev_err(edac_priv->dev,
			"Memory controller information get failure\n");
		rc = -ENODEV;
		goto err_group;
	}
	if (!info.memcx.ecc) {
		dev_err(edac_priv->dev,
			"memory-controller %d not supported by hardware\n",
			tmp_ctx.mc_idx);
		rc = -ENODEV;
		goto err_group;
	}
	mem_size = (uint64_t)((info.memcx.phy_width/info.memcx.dev_width)*
		(info.memcx.dev_size/8));

	layers[0].type = EDAC_MC_LAYER_CHIP_SELECT;
	layers[0].size = 1;
	layers[0].is_virt_csrow = true;
	layers[1].type = EDAC_MC_LAYER_CHANNEL;
	layers[1].size = 1;
	layers[1].is_virt_csrow = false;
	mci = edac_mc_alloc(tmp_ctx.mc_idx, ARRAY_SIZE(layers), layers,
			    sizeof(*ctx));
	if (!mci) {
		rc = -ENOMEM;
		goto err_group;
	}

	ctx = mci->pvt_info;
	*ctx = tmp_ctx;		/* Copy over resource value */
	ctx->name = kasprintf(GFP_KERNEL, "BCM RPC EDAC MC%d", tmp_ctx.mc_idx);
	ctx->mci = mci;
	mci->pdev = &mci->dev;
	mci->ctl_name = ctx->name;
	mci->dev_name = dev_name(edac_priv->dev);

	mci->mtype_cap = MEM_FLAG_RDDR | MEM_FLAG_RDDR2 | MEM_FLAG_RDDR3 |
			 MEM_FLAG_DDR | MEM_FLAG_DDR2 | MEM_FLAG_DDR3;
	mci->edac_ctl_cap = EDAC_FLAG_NONE | EDAC_FLAG_EC | EDAC_FLAG_SECDED;
	mci->edac_cap = EDAC_FLAG_EC | EDAC_FLAG_SECDED;
	mci->mod_name = MODULE_NAME;
	mci->mod_ver = "0.1";
	mci->ctl_page_to_phys = NULL;
	mci->scrub_cap = SCRUB_FLAG_HW_SRC;
	mci->scrub_mode = SCRUB_HW_SRC;

	if (edac_op_state == EDAC_OPSTATE_POLL)
		mci->edac_check = bcm_rpc_edac_mc_check;

	dimm = *mci->dimms;
	dimm->nr_pages = ((mem_size - 1) >> PAGE_SHIFT) + 1;
	dimm->grain = (dimm->nr_pages << PAGE_SHIFT);
	dimm->dtype = DEV_UNKNOWN;
	dimm->mtype = MEM_RDDR | MEM_RDDR2 | MEM_RDDR3 |
			 MEM_DDR | MEM_DDR2 | MEM_DDR3;
	dimm->edac_mode = EDAC_EC | EDAC_SECDED;

	if (edac_mc_add_mc(mci)) {
		dev_err(edac_priv->dev, "edac_mc_add_mc failed\n");
		rc = -EINVAL;
		goto err_free;
	}
	bcm_rpc_edac_mc_create_debugfs_node(mci);
	list_add(&ctx->next, &edac_priv->mcus);
	devres_remove_group(edac_priv->dev, bcm_rpc_edac_mc_add);

	dev_info(edac_priv->dev, "BCM RPC EDAC MC registered\n");
	return 0;

err_free:
	edac_mc_free(mci);
err_group:
	devres_release_group(edac_priv->dev, bcm_rpc_edac_mc_add);

	return rc;
}

static int bcm_rpc_edac_mc_remove(struct bcm_rpc_edac_mc_ctx *mcu)
{
	edac_mc_del_mc(&mcu->mci->dev);
	edac_mc_free(mcu->mci);
	return 0;
}

static int bcm_rpc_edac_probe(struct platform_device *pdev)
{
	struct bcm_rpc_edac_priv *edac_priv;
	struct rpc_sys_mc_data *sys_mc_data;
	struct device_node *np;
	int rc;

	edac_priv = (struct bcm_rpc_edac_priv *)
		devm_kzalloc(&pdev->dev, sizeof(*edac_priv), GFP_KERNEL);
	if (!edac_priv)
		return -ENOMEM;

	sys_mc_data = (struct rpc_sys_mc_data *)
		kzalloc(sizeof(struct rpc_sys_mc_data), GFP_KERNEL);
	if (!sys_mc_data)
		return -ENOMEM;

	edac_priv->sys_mc_data = sys_mc_data;
	edac_priv->dev = &pdev->dev;
	platform_set_drvdata(pdev, edac_priv);
	INIT_LIST_HEAD(&edac_priv->mcus);
	mutex_init(&edac_priv->mc_lock);
	rc = register_rpc_sys_mc_from_platform_device(sys_mc_data, pdev);
	if (rc) {
		release_rpc_sys_mc(sys_mc_data);
		kfree(sys_mc_data);
		return rc;
	}

	//edac_priv->dfs = edac_debugfs_create_dir(pdev->dev.kobj.name);
	for_each_child_of_node(pdev->dev.of_node, np) {
		if (!of_device_is_available(np))
			continue;
		if (of_device_is_compatible(np, "brcm,rpc-edac-mc"))
			bcm_rpc_edac_mc_add(edac_priv, np);
	}

	return 0;
}

static int bcm_rpc_edac_remove(struct platform_device *pdev)
{
	struct bcm_rpc_edac_priv *edac_priv = dev_get_drvdata(&pdev->dev);
	struct bcm_rpc_edac_mc_ctx *mcu;
	struct bcm_rpc_edac_mc_ctx *temp_mcu;

	release_rpc_sys_mc(edac_priv->sys_mc_data);
	kfree(edac_priv->sys_mc_data);
	list_for_each_entry_safe(mcu, temp_mcu, &edac_priv->mcus, next)
		bcm_rpc_edac_mc_remove(mcu);

	return 0;
}


static const struct of_device_id bcm_rpc_edac_of_match[] = {
	{ .compatible = "brcm,bcm-rpc-edac" },
	{},
};
MODULE_DEVICE_TABLE(of, bcm_rpc_edac_of_match);

static struct platform_driver bcm_rpc_edac_driver = {
	.probe = bcm_rpc_edac_probe,
	.remove = bcm_rpc_edac_remove,
	.driver = {
		.name = MODULE_NAME,
		.of_match_table = bcm_rpc_edac_of_match,
	},
};


static int __init bcm_rpc_edac_init(void)
{
	int rc = 0;

	edac_op_state = EDAC_OPSTATE_POLL;
	rc = rpc_register_functions(RPC_SERVICE_SYS,
			rpc_sys_mc_services_tbl,
			RPC_SYS_FUNC_MAX);
	if (rc) {
		pr_err("%s: Failed to register SYS RPC function(s).\n",
			MODULE_NAME);
		goto out;
	}

	rc = platform_driver_register(&bcm_rpc_edac_driver);
	if (rc) {
		pr_err("%s: Failed to register platform driver.\n",
			   MODULE_NAME);
		goto err_unreg_sys_rpc;
	}
	return 0;

err_unreg_sys_rpc:
	rpc_unregister_functions(RPC_SERVICE_SYS);
out:
	return rc;
}

static void __exit bcm_rpc_edac_exit(void)
{
	platform_driver_unregister(&bcm_rpc_edac_driver);
	rpc_unregister_functions(RPC_SERVICE_SYS);
}

module_init(bcm_rpc_edac_init);
module_exit(bcm_rpc_edac_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Venky Selvaraj");
MODULE_DESCRIPTION("BCM-RPC EDAC driver");

#endif /* CONFIG_BCM_KF_CM */
