#!/bin/sh
# ============================================================================
# RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
# ============================================================================
# This file (and its contents) are the intellectual property of RDK Management, LLC.
# It may not be used, copied, distributed or otherwise  disclosed in whole or in
# part without the express written permission of RDK Management, LLC.
# ============================================================================
# Copyright (c) 2018 RDK Management, LLC. All rights reserved.
# ============================================================================

if [ -e "/opt/.boops_disable" ]; then
  echo "WARNING: Refusing to start boops because '/opt/.boops_disable' exits! Remove this file to enable boops!"
  exit 1
fi

. /etc/device.properties
rdk_version=$(sed -n 's/^imagename://p' /version.txt)

start_boops_mmc() {
    local block_size=512

    case "${MODEL_NUM}" in
    "AX014AN" )
        read -r mmc_device byte_start byte_count <<< $(mount_emmc_loop EMMC0.NON_VOL.NON_CRIT nomount =100K)
        if [ -e "${mmc_device}" ] && [[ "${byte_start}" =~ ^0x[0-9a-fA-F]{8}$ ]] && [[ "${byte_count}" =~ ^0x[0-9a-fA-F]{8}$ ]]; then
            BOOPS_MMC_BLOCK_START=$((byte_start / block_size))
            BOOPS_MMC_BLOCK_COUNT=$((byte_count / block_size))
        else
            echo "ERROR: Unable to locate OOPS MMC partition! device:${device} byte_offset:${byte_start} byte_size:${byte_count}"
        fi
        ;;
    * )
        mmc_device="${BOOPS_MMC_DEVICE}"
    esac

    # At this point we expect BOOPS_MMC_BLOCK_START and BOOPS_MMC_BLOCK_COUNT to be set
    if [ ! -e "${mmc_device}" ] || [ ! "${BOOPS_MMC_BLOCK_START}" ] && [ ! "${BOOPS_MMC_BLOCK_COUNT}" ]; then
       echo "ERROR: Invalid MMC oops location settings! Boops will not be able to store the log"
       exit 1
    fi

    # Start boops-k
    /sbin/modprobe boops storage_type="mmc" mmc_block_first="${BOOPS_MMC_BLOCK_START}" mmc_block_count="${BOOPS_MMC_BLOCK_COUNT}" build_version="${rdk_version:0:63}"
    [ $? -ne 0 ] && exit 1

    # Start boops
    /usr/bin/boops --in_file "${mmc_device}" --byte_offset "$((BOOPS_MMC_BLOCK_START*block_size))" --monitor --kernel --prefix
    exit $?
}


start_boops_ubi() {
    OOPS_PART_VOLUME_SIZE="100KiB"

    ubi_oops_device_num="${UBI_DEV_NUM:-0}"
    ubi_oops_device="ubi${ubi_oops_device_num}"

    ubinfo /dev/${ubi_oops_device} -N oops_dump 2>/dev/null || ubimkvol /dev/${ubi_oops_device} -s $OOPS_PART_VOLUME_SIZE -N oops_dump
    if [ -e  /dev/${ubi_oops_device} ]; then
        ubi_oops_volume_num=`ubinfo /dev/${ubi_oops_device} -N oops_dump | grep Volume | cut -d' ' -f5`
    fi

    if [ "${ubi_oops_volume_num}" == "" ];then
        echo "ERROR: Unable to create boops UBI volume!"
        exit 1
    fi

    ubi_oops_volume="ubi${ubi_oops_device_num}_${ubi_oops_volume_num}"
    if [ ! -d "/sys/class/ubi/${ubi_oops_volume}" ]; then
        echo "ERROR: Unable to find /sys/class/ubi/${ubi_oops_volume}!"
        exit 1
    fi

    # Copy any previous oops to a /tmp file
    byte_count=$(cat /sys/class/ubi/${ubi_oops_volume}/data_bytes)
    block_size=$(cat /sys/class/ubi/${ubi_oops_volume}/device/min_io_size)
    block_count=$((byte_count/block_size))
    dd if="/dev/ubi${ubi_oops_device_num}_${ubi_oops_volume_num}" of=/tmp/boops.dat bs="${block_size}" count="${block_count}"
    if [ $? -ne 0 ]; then
        echo "ERROR: The UBI volume is corrupt. Recreating it"
        ubirmvol /dev/${ubi_oops_device} -N oops_dump
        ubimkvol /dev/${ubi_oops_device} -s $OOPS_PART_VOLUME_SIZE -N oops_dump
    fi

    # Start boops-k
    /sbin/modprobe boops storage_type="ubi" ubi_device="${ubi_oops_device_num}" ubi_volume="${ubi_oops_volume_num}" build_version="${rdk_version:0:63}"
    [ $? -ne 0 ] && exit 1

    # Start boops
    /usr/bin/boops --in_file /tmp/boops.dat --kernel --prefix
    exit $?
}


if [[ "${MMC_OOPS_ENABLE}" == "true" ]]; then
    start_boops_mmc
else
    start_boops_ubi
fi

exit 0
