 /****************************************************************************
 *
 * Copyright (c) 2016 Broadcom Limited
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 *****************************************************************************/
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/irqreturn.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/types.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <asm/cacheflush.h>
#include <net/dst.h>
#include <net/xfrm.h>
#include <bcmcache.h>
#include "fpm.h"
#include "dqskb_priv.h"

int def_pkt_size = 2048;
int def_autofill = 0;
int def_headroom = 0;
int def_napi_weight = 32;

#ifdef CONFIG_SYSCTL
static struct ctl_table sysctl_table[] = {
	{
		.procname	= "def_pkt_size",
		.data		= &def_pkt_size,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "def_autofill",
		.data		= &def_autofill,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "def_headroom",
		.data		= &def_headroom,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{
		.procname	= "def_napi_weight",
		.data		= &def_napi_weight,
		.maxlen		= sizeof(int),
		.mode		= 0644,
		.proc_handler	= proc_dointvec,
	},
	{}
};
#endif

module_param_named(def_pkt_size, def_pkt_size, int, 0644);
MODULE_PARM_DESC(def_pkt_size, "Default Packet size");

module_param_named(def_autofill, def_autofill, int, 0644);
MODULE_PARM_DESC(def_autofill, "Default Autofill");

module_param_named(def_headroom, def_headroom, int, 0644);
MODULE_PARM_DESC(def_headroom, "Default Headroom");

static inline
phys_addr_t skb_to_phy(struct device *dev, struct sk_buff *skb)
{
	phys_addr_t paddr;
	dma_addr_t dma;

	/* Convert data pointer to physical address */
	dma = dma_map_single(dev, skb->data,
			     skb->end - skb->data,
			     DMA_TO_DEVICE);
	if (dma_mapping_error(dev, dma)) {
		pr_err("%s dmap map single error skb %px\n", __func__,
		       skb);
		return -1;
        }
	paddr = dma;
	return paddr;
}

/*
 * Push element to Available Pool
 */
int dqskb_push(struct dqskb_dev *sdev, struct sk_buff *skb)
{
	int status = -1;
	u32 msgdata[sdev->tx_q_tok_size];

	/* Fill in descriptors */
	msgdata[0] = (uint32_t) skb;
	msgdata[1] = skb_to_phy(&sdev->pdev->dev, skb);

	skb->dev = &sdev->napi_dev;

	if (msgdata[1] == -1)
		return status;

	status = dqm_tx(sdev->tx_q_h, 1, sdev->tx_q_tok_size, msgdata);

	return status;
}

static void skb_release_head_state_local(struct sk_buff *skb)
{
	if (skb->destructor) {
		WARN_ON(in_irq());
		skb->destructor(skb);
		skb->destructor = NULL;
	}
}

/*
 * Allocate and Push element to Available Pool
 */
int dqskb_push_new(struct dqskb_dev *sdev, int num)
{
	struct sk_buff *skb;
	int i, status = 0;

	for (i = 0; (i < num) && (!status); i++) {
		skb = alloc_skb(sdev->pkt_size, GFP_ATOMIC);
		if (!skb)
			break;

		status = dqskb_push(sdev, skb);
		if (status)
			dev_kfree_skb_any(skb);
		else
			atomic_inc(&sdev->cnt_new);
	}

	return status;
}

void dqskb_invalidate_buffer(void *start, u32 size)
{
	cache_invalidate_buffer(start, size);
}
EXPORT_SYMBOL(dqskb_invalidate_buffer);

void dqskb_flush_invalidate_buffer(void *start, u32 size)
{
	cache_flush_invalidate_buffer(start, size);
}
EXPORT_SYMBOL(dqskb_flush_invalidate_buffer);

int dqskb_prepare_skb(struct net_device *dev, struct sk_buff *skb)
{
	struct dqskb_dev *sdev =
		container_of(dev, struct dqskb_dev, napi_dev);

	skb->dev = NULL;
	/* Push one SKB in */
	return dqskb_push_new(sdev, 1);
}
EXPORT_SYMBOL(dqskb_prepare_skb);

static int dqskb_rx_q_msg(struct dqskb_dev *sdev, int budget)
{
	int i, status = 0;
	u32 msgdata[budget * sdev->rx_q_tok_size];
	struct sk_buff *skb;
	int work=0;
	int toksz = sdev->rx_q_tok_size;
	int idx;

	status = dqm_rx(sdev->rx_q_h, budget, sdev->rx_q_tok_size, msgdata);
	if (status < 0) {
		pr_err("Error receiving DQM %d\n on %s.\n",
		       sdev->rx_q, sdev->name);
		atomic_inc(&sdev->cnt_err);
	} else
		work += status;

	for (i = 0, idx = 0; i < work; i++, idx += toksz) {
		skb = (struct sk_buff *)msgdata[idx];
		if (skb) {
			dqskb_invalidate_buffer(skb->data, skb->len);
			dev_kfree_skb_any(skb);
			atomic_inc(&sdev->cnt_free);
		}
	}
	return work;
}

int dqskb_unprepare_skb(struct net_device *napi_dev, struct sk_buff *skb)
{
	struct dqskb_dev *sdev =
		container_of(napi_dev, struct dqskb_dev, napi_dev);

	/* Update memory accounting used by Sock layer
	   If we don't do this then high water mark for
	   SKB recycle DQM should be less than 64 */
	skb_release_head_state_local(skb);
	if (!napi_dev)
		return 0;

	/* Pull one SKB out */
	return dqskb_rx_q_msg(sdev, 1);
}
EXPORT_SYMBOL(dqskb_unprepare_skb);

/*
 * rx_isr handles all dqm rx events
 * for a specific dqm rx channel
 */
static irqreturn_t rx_isr(void *q_h, void *context, u32 flags)
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) context;

	napi_schedule(&sdev->napi);
	dqm_disable_rx_cb(sdev->rx_q_h);
	sdev->cnt_rxint++;
	return IRQ_HANDLED;
}

void tx_tasklet_handler(unsigned long data)
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) data;

	dqskb_push_new(sdev, dqm_space(sdev->tx_q_h));
	dqm_enable_tx_cb(sdev->tx_q_h);
}

/*
 * tx_isr handles all dqm tx events
 * for a specific dqm tx channel
 */
static irqreturn_t tx_isr(void *q_h, void *context, u32 flags)
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) context;

	//dqskb_push_new(sdev, dqm_space(sdev->tx_q_h));
	dqm_disable_tx_cb(sdev->tx_q_h);
	tasklet_hi_schedule(&sdev->tx_tasklet);
	sdev->cnt_txint++;
	return IRQ_HANDLED;
}

static int dqskb_poll_napi(struct napi_struct *napi, int budget)
{
	struct dqskb_dev *sdev =
		container_of(napi, struct dqskb_dev, napi);
	int work;

	work = dqskb_rx_q_msg(sdev, budget);
	if (work < budget) {
		napi_complete(napi);
		dqm_enable_rx_cb(sdev->rx_q_h);
	}
	return work;
}

static int cmd_show(void *data, int argc, char *argv[])
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) data;
	pr_info("Packet Size : %d\n", sdev->pkt_size);
	pr_info("Autofill    : %d\n", sdev->autofill);
	pr_info("Headroom    : %d\n", sdev->headroom);
	pr_info("NAPI Weight : %d\n", sdev->napi.weight);
	pr_info("DQM Info:\n");
	pr_info(" Name : %s\n", sdev->name);
	pr_info(" Dev  : %s\n", sdev->dqm_dev);
	pr_info(" TX   : Num %d handle %p\n", sdev->tx_q, sdev->tx_q_h);
	pr_info(" RX   : Num %d handle %p\n", sdev->rx_q, sdev->rx_q_h);
	pr_info("Counters:\n");
	pr_info(" TX Interrupts: %d\n", sdev->cnt_txint);
	pr_info(" RX Interrupts: %d\n", sdev->cnt_rxint);
	pr_info(" New SKBs     : %d\n", atomic_read(&sdev->cnt_new));
	pr_info(" Freed SKBs   : %d\n", atomic_read(&sdev->cnt_free));
	pr_info(" Error        : %d\n", atomic_read(&sdev->cnt_err));
	pr_info(" RX Push      : %d\n", atomic_read(&sdev->cnt_rxpush));
	return 0;
}

static int cmd_fill(void *data, int argc, char *argv[])
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) data;
	if (argc == 2) {
		int i, num, status;
		if (kstrtoint(argv[1], 0, &num) == 0) {
		       for (i=0; (i<num) && (!status); i++) {
			       status = dqskb_push_new(sdev, 1);
		       }
		}
	}
	return 0;
}

static int cmd_autofill(void *data, int argc, char *argv[])
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) data;
	if (argc == 2) {
		int num, status;
		if (kstrtoint(argv[1], 0, &num) == 0) {
			sdev->autofill = num;
			dev_info(&sdev->pdev->dev, "Autofill=%d\n", num);
			if (num) {
				status = dqm_enable_tx_cb(sdev->tx_q_h);
				if (status) {
					pr_err("%s unable to enable DQM interrupts\n", __func__);
				}
			} else {
				status = dqm_disable_tx_cb(sdev->tx_q_h);
				if (status) {
					pr_err("%s unable to enable DQM interrupts\n", __func__);
				}
			}
		}
	}
	return 0;
}

static int cmd_pktsize(void *data, int argc, char *argv[])
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) data;
	if (argc == 2) {
		int num;
		if (kstrtoint(argv[1], 0, &num) == 0) {
			sdev->pkt_size = num;
			dev_info(&sdev->pdev->dev, "Packet Size=%d\n", num);
		}
	}
	return 0;
}

static int cmd_headroom(void *data, int argc, char *argv[])
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) data;
	if (argc == 2) {
		int num;
		if (kstrtoint(argv[1], 0, &num) == 0) {
			sdev->headroom = num;
			dev_info(&sdev->pdev->dev, "Headroom=%d\n", num);
		}
	}
	return 0;
}

static int cmd_napi_weight(void *data, int argc, char *argv[])
{
	struct dqskb_dev *sdev;
	sdev = (struct dqskb_dev *) data;
	if (argc == 2) {
		int num;
		if (kstrtoint(argv[1], 0, &num) == 0) {
			sdev->napi.weight = num;
			dev_info(&sdev->pdev->dev, "Napi Weight=%d\n", num);
		}
	}
	return 0;
}

static struct proc_cmd_ops command_entries[] = {
	{ .name = "show", .do_command_data = cmd_show},
	{ .name = "fill", .do_command_data = cmd_fill},
	{ .name = "autofill", .do_command_data = cmd_autofill},
	{ .name = "pktsize", .do_command_data = cmd_pktsize},
	{ .name = "headroom", .do_command_data = cmd_headroom},
	{ .name = "napi_weight", .do_command_data = cmd_napi_weight},
};

struct proc_cmd_table command_table_tmpl = {
	.module_name = MODULE_NAME,
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

#define PROC_DIR_NAME	"driver/"

/*
 * Device probe
 *
 * This will be called by Linux for each device tree node matching our
 * driver name we passed when we called platform_driver_register().
 * Each node specifies an skb interface to create.
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqskb_probe(struct platform_device *pdev)
{
	int status = 0;
	struct dqskb_dev *sdev;
	struct dqm_cb cb = {};
	char str[32];

	sdev = kmalloc(sizeof(struct dqskb_dev), GFP_KERNEL);
	if (!sdev) {
		status = -ENOMEM;
		goto done;
	}
	memset(sdev, 0, sizeof(struct dqskb_dev));
	sdev->pkt_size = def_pkt_size;
	sdev->autofill = def_autofill;
	pdev->dev.platform_data = sdev;
	sdev->pdev = pdev;
	status = dqskb_parse_dt_node(pdev);
	if (status)
		goto err_free_dt;

	cb.fn = (dqm_isr_callback)tx_isr;
	cb.context = (void *)sdev;
	cb.name = sdev->name;
	sdev->tx_q_h = dqm_register(sdev->dqm_dev,
				    sdev->tx_q,
				    &cb,
				    &sdev->tx_q_tok_size,
				    DQM_F_TX);
	if (!sdev->tx_q_h) {
		pr_err("%s unable to acquire TX DQM %d on device %s\n",
		       __func__, sdev->tx_q, sdev->dqm_dev);
		status = -EFAULT;
		goto err_free_dt;
	}
	cb.fn = (dqm_isr_callback)rx_isr;
	cb.context = (void *)sdev;
	cb.name = sdev->name;
	sdev->rx_q_h = dqm_register(sdev->dqm_dev,
				    sdev->rx_q,
				    &cb,
				    &sdev->rx_q_tok_size,
				    DQM_F_RX);
	if (!sdev->rx_q_h) {
		pr_err("%s unable to acquire RX DQM %d on device %s\n",
		       __func__, sdev->rx_q, sdev->dqm_dev);
		status = -EFAULT;
		goto err_free_dt;
	}

	status = dqm_enable_rx_cb(sdev->rx_q_h);
	if (status) {
		pr_err("%s unable to enable DQM interrupts\n", __func__);
		goto done;
	}

	tasklet_init(&sdev->tx_tasklet, tx_tasklet_handler, (unsigned long) sdev);
	if (sdev->autofill) {
		status = dqm_enable_tx_cb(sdev->tx_q_h);
		if (status) {
			pr_err("%s unable to enable DQM interrupts\n", __func__);
			goto done;
		}
	}

	snprintf(str, sizeof(str), "%s%s", PROC_DIR_NAME, sdev->name);
	sdev->proc_dir = proc_mkdir(str, NULL);
	if (sdev->proc_dir == NULL) {
		pr_warn("dqskb Warning: cannot create /proc/%s\n",
			str);
		return -1;
	}

	sdev->cmd_tbl = kzalloc(sizeof(struct proc_cmd_table), GFP_KERNEL);
	sdev->cmd_tbl->module_name = command_table_tmpl.module_name;
	sdev->cmd_tbl->size = command_table_tmpl.size;
	sdev->cmd_tbl->ops = command_table_tmpl.ops;
	sdev->cmd_tbl->data = (void *) sdev;
	proc_create_cmd("cmd", sdev->proc_dir, sdev->cmd_tbl);

	init_dummy_netdev(&sdev->napi_dev);
	strncpy(sdev->napi_dev.name, sdev->name, IFNAMSIZ-1);
	netif_napi_add(&sdev->napi_dev, &sdev->napi, dqskb_poll_napi,
		       def_napi_weight);
	napi_enable(&sdev->napi);


	goto done;

  err_free_dt:
	kfree(sdev);

  done:
	return status;
}

/*
 * Device remove
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqskb_remove(struct platform_device *pdev)
{
	int status = 0;
	struct dqskb_dev *sdev;
	char str[32];

	sdev = pdev->dev.platform_data;

	status = dqm_release(sdev->rx_q_h, DQM_F_RX);
	if (status) {
		pr_err("%s unable to release RX DQM %d on device %s\n",
		       __func__, sdev->rx_q, sdev->dqm_dev);
	}
	status = dqm_release(sdev->tx_q_h, DQM_F_TX);
	if (status) {
		pr_err("%s unable to release TX DQM %d on device %s\n",
		       __func__, sdev->tx_q, sdev->dqm_dev);
	}

	if (sdev->proc_dir) {
		snprintf(str, sizeof(str), "%s%s", PROC_DIR_NAME, sdev->name);
		remove_proc_entry("cmd", sdev->proc_dir);
		remove_proc_entry(str, NULL);
	}

	napi_disable(&sdev->napi);
	netif_napi_del(&sdev->napi);
	tasklet_kill(&sdev->tx_tasklet);

	kfree(sdev);
	return status;
}


static const struct of_device_id dqskb_of_match[] = {
	{.compatible = "brcm,dqskb"},
	{}
};
MODULE_DEVICE_TABLE(of, dqskb_of_match);
static struct platform_driver dqskb_driver = {
	.probe	= dqskb_probe,
	.remove	= dqskb_remove,
	.driver	= {
		.name		= MODULE_NAME,
		.owner		= THIS_MODULE,
		.of_match_table	= dqskb_of_match
	}
};

static int match_name(struct device *dev, const void *name)
{
	struct dqskb_dev *sdev;
	sdev = dev->platform_data;
	return strcmp(sdev->name, name) ? 0 : 1;
}

struct net_device *dqskb_get_dev_by_name(char *name)
{
	struct device *dev;
	struct dqskb_dev *sdev;

	dev = driver_find_device(&dqskb_driver.driver, NULL, name,
				 match_name);
	if (!dev)
		return ERR_PTR(-EBADF);

	sdev = dev->platform_data;

	return &sdev->napi_dev;
}
EXPORT_SYMBOL(dqskb_get_dev_by_name);

static struct ctl_table_header	*sysctl_header;

static int __init dqskb_init(void)
{
	platform_driver_register(&dqskb_driver);
#ifdef CONFIG_SYSCTL
	sysctl_header = register_net_sysctl(&init_net,
					    "net/dqskb",
					    sysctl_table);
#endif
	pr_debug("%s driver v%s\n", MODULE_NAME, MODULE_VER);
	return 0;
}

static void dqskb_exit(void)
{
#ifdef CONFIG_SYSCTL
	unregister_net_sysctl_table(sysctl_header);
#endif
	platform_driver_unregister(&dqskb_driver);
	pr_info("%s_removed\n", MODULE_NAME);
}
module_init(dqskb_init);
module_exit(dqskb_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("dq_skb");
