/****************************************************************************
 *
 * Broadcom Proprietary and Confclient_idential.
 * (c) 2017 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsclient_idiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provclient_ided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provclient_ided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Broadcom driver to simulate SMC virtual flash block IO (for test purposes
 * only).
 *
 * Author: Tim Ross <tim.ross@broadcom.com>
 *****************************************************************************/
#include <asm/cacheflush.h>
#include <linux/highmem.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/of.h>
#include <linux/slab.h>
#include <linux/scatterlist.h>
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <vfbio_rpc.h>
#include "vfbioss_priv.h"
#include "vfbioss_proc.h"

#define MODULE_NAME	"brcm-vfbio-smc-sim"
#define MODULE_VER	"1.0"

#define VFBIOSS_MINORS		16
#define VFBIOSS_SG_LIST_ENTRIES	16

#define PAGE_BLOCKS_SHIFT(bs)	(PAGE_SHIFT - (bs))
#define PAGE_BLOCKS(bs)		(1 << PAGE_BLOCKS_SHIFT(bs))

struct vfbioss_device {
	struct platform_device	*pdev;
	struct list_head	list;

	int			tunnel;
	int			client_id;
	const char		*client_name;
};

static struct list_head vfdevs;

enum client_id {
	/* bitfields */
	CLIENT_ID_NONE	= 0,
	CLIENT_ID_CM	= 0x1,
	CLIENT_ID_RG	= 0x2
};

struct lun luns[VFBIO_LUN_MAX] = {
	{ 1024,	1*1024,		0,		"smcbl0",	NULL,	0 },
	{ 1024,	1*1024,		0,		"smcbl1",	NULL,	0 },
	{ 1024,	1*1024,		0,		"smcos0",	NULL,	0 },
	{ 1024,	1*1024,		0,		"smcos1",	NULL,	0 },
	{ 1024,	1*1024,		0,		"smcusrapps0",	NULL,	0 },
	{ 1024,	1*1024,		0,		"smcusrapps1",	NULL,	0 },
	{ 1024,	1*1024,		0,		"smcnonvol0",	NULL,	0 },
	{ 1024,	1*1024,		0,		"smcnonvol1",	NULL,	0 },
	{ 2048,	1*1024,		CLIENT_ID_RG,	"bolt",		NULL,	0 },
	{ 2048,	1*1024,		CLIENT_ID_RG,	"bolt1",	NULL,	0 },
	{ 4096,	8,		CLIENT_ID_RG,	"macadr",	NULL,	0 },
	{ 4096,	8,		CLIENT_ID_RG,	"macadr1",	NULL,	0 },
	{ 2048,	32,		CLIENT_ID_RG,	"nvram",	NULL,	0 },
	{ 2048,	32,		CLIENT_ID_RG,	"nvram1", 	NULL,	0 },
	{ 1024,	256,		CLIENT_ID_RG,	"devtree0",	NULL,	0 },
	{ 1024,	256,		CLIENT_ID_RG,	"devtree1",	NULL,	0 },
	{ 4096,	512,		CLIENT_ID_RG,	"kernel0",	NULL,	0 },
	{ 4096,	512,		CLIENT_ID_RG,	"kernel1",	NULL,	0 },
	{ 4096,	512,		CLIENT_ID_RG,	"rg0",		NULL,	0 },
	{ 4096,	512,		CLIENT_ID_RG,	"rg1",		NULL,	0 },
	{ 4096,	512,		CLIENT_ID_RG,	"rgusrapps0",	NULL,	0 },
	{ 4096,	512,		CLIENT_ID_RG,	"rgusrapps1",	NULL,	0 },
	{ 512,	1024,		CLIENT_ID_RG,	"rgnonvol0",	NULL,	0 },
	{ 512,	1024,		CLIENT_ID_RG,	"rgnonvol1",	NULL,	0 },
	{ 2048,	1024,		CLIENT_ID_RG,	"rgusrnonvol0",	NULL,	0 },
	{ 2048,	1024,		CLIENT_ID_RG,	"rgusrnonvol1",	NULL,	0 },
	{ 4096,	512,		CLIENT_ID_RG,	"trustzone0",	NULL,	0 },
	{ 4096,	512,		CLIENT_ID_RG,	"trustzone1",	NULL,	0 },
	{ 512,	2*1024,		CLIENT_ID_CM,	"cmbl0",	NULL,	0 },
	{ 512,	2*1024,		CLIENT_ID_CM,	"cmbl1",	NULL,	0 },
	{ 512,	2*1024,		CLIENT_ID_CM,	"cm0",		NULL,	0 },
	{ 512,	2*1024,		CLIENT_ID_CM,	"cm1",		NULL,	0 },
	{ 512,	2*1024,		CLIENT_ID_CM,	"cmnonvol0",	NULL,	0 },
	{ 512,	2*1024,		CLIENT_ID_CM,	"cmnonvol1",	NULL,	0 },
	{ 512,	2*1024,		CLIENT_ID_CM,	"gfap0",	NULL,	0 },
	{ 512,	2*1024,		CLIENT_ID_CM,	"gfap1",	NULL,	0 },
};

static void __vfbioss_dump_buf(struct device *dev, u8 *buf,
			       unsigned int len)
{
	unsigned int offset = 0;
	unsigned int line_len;
	u8 line[10 + 32 * 3 + 2 + 32 + 1];
	bool repeat;
	bool print_repeat = true;

	while (len) {
		line_len = len < 16 ? len : 16;
		repeat = offset >= 16 ?
			memcmp(buf, buf - 16, line_len) == 0 : false;
		if (!repeat || len == line_len) {
			snprintf(line, 11, "%8x: ", offset);
			hex_dump_to_buffer(buf, line_len, 16, 1,
					   &line[10], sizeof(line)-10, true);
			pr_debug("%s\n", line);
			print_repeat = true;
		} else if (print_repeat) {
			pr_debug("*\n");
			print_repeat = false;
		}
		len -= line_len;
		buf += line_len;
		offset += line_len;
	}
}

static inline struct vfbioss_device *tunnel_to_dev(int tunnel)
{
	struct vfbioss_device *vfdev;

	rcu_read_lock();
	list_for_each_entry_rcu(vfdev, &vfdevs, list) {
		if (vfdev->tunnel == tunnel)
			return vfdev;
	}
	rcu_read_unlock();
	return NULL;
}

static int vfbioss_alloc(int tunnel, rpc_msg *msg)
{
	int status = 0;
	struct vfbioss_device *vfdev;
	enum client_id id;
	unsigned int lun;

	pr_debug("%s: -->\n", __func__);
	if (!rpc_msg_request(msg) || !vfbio_msg_get_exclusive(msg)) {
		pr_err("%s: Malformed message.\n", __func__);
		rpc_dump_msg(msg);
		status = EINVAL;
		goto done;
	}

	vfdev = tunnel_to_dev(tunnel);
	if (!vfdev) {
		pr_err("%s: No device found for tunnel %d.\n", __func__,
		       tunnel);
		status = EINVAL;
		goto done;
	}

	lun = vfbio_msg_get_lun(msg);
	id = vfdev->client_id;
	if (!(id & luns[lun].perm)) {
		pr_err("%s: %s not allowed to use %s LUN.\n", __func__,
		       vfdev->client_name, luns[lun].name);
		status = EPERM;
		goto done;
	}

	if (luns[lun].client) {
		pr_err("%s: LUN %s in use by %s.\n", __func__,
		       luns[lun].name, luns[lun].client->client_name);
		status = EBUSY;
		goto done;
	}

	luns[lun].client = vfdev;
	pr_info("%s: tunnel: %d, client: %s, lun: %d\n", __func__, tunnel,
		vfdev->client_name, lun);

done:
	vfbio_msg_set_retcode(msg, status);
	rpc_send_reply(tunnel, msg);
	pr_debug("%s: <--\n", __func__);
	return status;
}

static int vfbioss_free(int tunnel, rpc_msg *msg)
{
	int status = 0;
	struct vfbioss_device *vfdev;
	unsigned int lun;

	pr_debug("%s: -->\n", __func__);
	if (!rpc_msg_request(msg)) {
		pr_err("%s: Malformed message.\n", __func__);
		rpc_dump_msg(msg);
		status = EINVAL;
		goto done;
	}

	vfdev = tunnel_to_dev(tunnel);
	if (!vfdev) {
		pr_err("%s: No device found for tunnel %d.\n", __func__,
		       tunnel);
		status = EINVAL;
		goto done;
	}

	lun = vfbio_msg_get_lun(msg);
	if (luns[lun].client != vfdev) {
		pr_err("%s: LUN %s not in use by %s.\n", __func__,
		       luns[lun].name, vfdev->client_name);
		status = EINVAL;
		goto done;
	}

	luns[lun].client = NULL;
	pr_info("%s: tunnel: %d, client: %s, lun: %d\n", __func__, tunnel,
		vfdev->client_name, lun);

done:
	vfbio_msg_set_retcode(msg, status);
	rpc_send_reply(tunnel, msg);
	pr_debug("%s: <--\n", __func__);
	return status;
}

static int vfbioss_lun_info(int tunnel, rpc_msg *msg)
{
	int status = 0;
#if 0
	struct vfbioss_device *vfdev;
	int lun;
#endif
	pr_debug("%s: -->\n", __func__);
#if 0
	if (!rpc_msg_request(msg)) {
		pr_err("%s: Malformed message.\n", __func__);
		rpc_dump_msg(msg);
		status = -EINVAL;
		goto done;
	}

	vfdev = tunnel_to_dev(tunnel);
	if (!vfdev) {
		pr_err("%s: No device found for tunnel %d.\n", __func__,
		       tunnel);
		status = -EINVAL;
		goto done;
	}

	lun = vfbio_msg_get_lun(msg);
	vfbio_msg_set_blk_sz(msg, luns[lun].blk_sz);
	vfbio_msg_set_n_blks(msg, luns[lun].n_blks);
	pr_info("%s: lun: %s, blk_sz: %d, n_blks: %d\n", __func__,
		luns[lun].name, luns[lun].blk_sz, luns[lun].n_blks);
done:
#endif
	rpc_send_reply(tunnel, msg);
	pr_debug("%s: <--\n", __func__);
	return status;
}

static int vfbioss_sync_rw(int tunnel, rpc_msg *msg)
{
	int status = 0;
	struct vfbioss_device *vfdev;
	struct device *dev;
	unsigned int lun, blk, n_blks, blk_sz, blk_sz_shift;
	u64 addr;
	struct scatterlist *sgl_seg;
	struct page *page;
	u8 op;
	void *page_addr;
	unsigned int page_offset;
	unsigned int lun_offset;
	unsigned int len;

	pr_debug("%s: -->\n", __func__);
	if (!rpc_msg_request(msg)) {
		pr_err("%s: Malformed message.\n", __func__);
		rpc_dump_msg(msg);
		status = EINVAL;
		goto done;
	}

	op = rpc_msg_function(msg);
	vfdev = tunnel_to_dev(tunnel);
	if (!vfdev) {
		pr_err("%s: No device found for tunnel %d.\n", __func__,
		       tunnel);
		status = EINVAL;
		goto done;
	}
	dev = &vfdev->pdev->dev;

	lun = vfbio_msg_get_lun(msg);
	if (luns[lun].client != vfdev) {
		pr_err("%s: LUN %s not in use by %s.\n", __func__,
		       luns[lun].name, vfdev->client_name);
		status = EPERM;
		goto done;
	}
	blk = vfbio_msg_get_blk(msg);
	n_blks = vfbio_msg_get_n_blks(msg);
	addr = vfbio_msg_get_addr(msg);

	sgl_seg = (struct scatterlist *)(u32)addr;
	blk_sz = luns[lun].blk_sz;
	blk_sz_shift = __ffs(blk_sz);
	len = n_blks * blk_sz;
	page = sg_page(sgl_seg);
	page_addr = kmap_atomic(page);
	page_offset = sgl_seg->offset;
	lun_offset = blk << blk_sz_shift;

	dev_dbg(dev, "sync %s:\n", op == VFBIO_FUNC_READ ? "read" : "write");
	dev_dbg(dev, "%d blocks @ block %d\n", n_blks, blk);
	dev_dbg(dev, "lun addr: %p, lun offset: 0x%08x\n",
		luns[lun].addr, lun_offset);
	dev_dbg(dev, "page: %p, page_offset: 0x%08x\n", page, page_offset);
	if (op == VFBIO_FUNC_READ) {
		memcpy(page_addr + page_offset, luns[lun].addr + lun_offset, len);
		flush_dcache_page(page);
		__vfbioss_dump_buf(dev, page_addr + page_offset, len);
	} else {
		flush_dcache_page(page);
		__vfbioss_dump_buf(dev, page_addr + page_offset, len);
		memcpy(luns[lun].addr + lun_offset, page_addr + page_offset, len);
	}
	kunmap_atomic(page_addr);

	dev_dbg(dev, "sync %s: tunnel: %d, client: %s, lun: %d\n",
		 op == VFBIO_FUNC_READ ? "read" : "wrote", tunnel,
		 vfdev->client_name, lun);

done:
	vfbio_msg_set_retcode(msg, status);
	rpc_send_reply(tunnel, msg);
	pr_debug("%s: <--\n", __func__);
	return status;
}

static int vfbioss_discard(int tunnel, rpc_msg *msg)
{
	int status = 0;
	struct vfbioss_device *vfdev;
	unsigned int lun, blk, n_blks, blk_sz;

	pr_debug("%s: -->\n", __func__);
	if (!rpc_msg_request(msg)) {
		pr_err("%s: Malformed message.\n", __func__);
		rpc_dump_msg(msg);
		status = EINVAL;
		goto done;
	}

	vfdev = tunnel_to_dev(tunnel);
	if (!vfdev) {
		pr_err("%s: No device found for tunnel %d.\n", __func__,
		       tunnel);
		status = EINVAL;
		goto done;
	}
	lun = vfbio_msg_get_lun(msg);
	if (luns[lun].client != vfdev) {
		pr_err("%s: LUN %s not in use by %s.\n", __func__,
		       luns[lun].name, vfdev->client_name);
		status = EPERM;
		goto done;
	}

	blk = vfbio_msg_get_blk(msg);
	n_blks = vfbio_msg_get_n_blks(msg);
	blk_sz = luns[lun].blk_sz;
	memset(luns[lun].addr + blk * blk_sz, 0, n_blks * blk_sz);

	pr_debug("discarded: tunnel: %d, client: %s, lun: %d\n",
		tunnel, vfdev->client_name, lun);
	pr_debug("blk: %d, n_blks: %d\n", blk, n_blks);

done:
	vfbio_msg_set_retcode(msg, status);
	rpc_send_reply(tunnel, msg);
	pr_debug("%s: <--\n", __func__);
	return status;
}

static int vfbioss_flush(int tunnel, rpc_msg *msg)
{
	int status = 0;
	struct vfbioss_device *vfdev;
	unsigned int lun;

	pr_debug("%s: -->\n", __func__);
	if (!rpc_msg_request(msg)) {
		pr_err("%s: Malformed message.\n", __func__);
		rpc_dump_msg(msg);
		status = EINVAL;
		goto done;
	}

	vfdev = tunnel_to_dev(tunnel);
	if (!vfdev) {
		pr_err("%s: No device found for tunnel %d.\n", __func__,
		       tunnel);
		status = EINVAL;
		goto done;
	}
	lun = vfbio_msg_get_lun(msg);
	if (luns[lun].client != vfdev) {
		pr_err("%s: LUN %s not in use by %s.\n", __func__,
		       luns[lun].name, vfdev->client_name);
		status = EPERM;
		goto done;
	}

	pr_debug("flushed: tunnel: %d, client: %s, lun: %d\n",
		tunnel, vfdev->client_name, lun);

done:
	vfbio_msg_set_retcode(msg, status);
	rpc_send_reply(tunnel, msg);
	pr_debug("%s: <--\n", __func__);
	return status;
}

static int vfbioss_sg_rw(int tunnel, rpc_msg *msg)
{
	int status = 0;
	struct vfbioss_device *vfdev;
	struct device *dev;
	unsigned int lun, blk, n_blks, blk_sz, blk_sz_shift;
	u32 tag;
	int blks_done = 0;
	struct page *page;
	u8 op;
	struct vfbio_hw_sgl *hw_sgl = NULL;
	u8 n_seg;
	u8 seg;
	struct scatterlist *sgl_seg;
	void *page_addr;
	unsigned int page_offset;
	unsigned int lun_offset;
	unsigned int len;

	pr_debug("%s: -->\n", __func__);
	op = rpc_msg_function(msg);
	hw_sgl = (struct vfbio_hw_sgl *)(u32)vfbio_msg_get_addr(msg);
	tag = vfbio_sgl_get_tag(hw_sgl);
	lun = vfbio_msg_get_lun(msg);
	vfdev = tunnel_to_dev(tunnel);
	if (!vfdev) {
		pr_err("%s: No device found for tunnel %d.\n", __func__,
		       tunnel);
		status = EINVAL;
		goto reply;
	}
	dev = &vfdev->pdev->dev;
	if (luns[lun].client != vfdev) {
		pr_err("%s: LUN %s not in use by %s.\n", __func__,
		       luns[lun].name, vfdev->client_name);
		status = EPERM;
		goto reply;
	}
	blk_sz = luns[lun].blk_sz;
	blk_sz_shift = __ffs(blk_sz);

	n_seg = vfbio_sgl_get_n_segs(hw_sgl);
	blk = vfbio_sgl_get_blk(hw_sgl);
	for (seg = 0, blks_done = 0; seg < n_seg;
	      seg++, blk += n_blks, blks_done += n_blks) {
		sgl_seg = (struct scatterlist *)(u32)vfbio_sgl_seg_get_addr(
		   hw_sgl, seg);
		page = sg_page(sgl_seg);
		page_addr = kmap_atomic(page);
		page_offset = sgl_seg->offset;
		lun_offset = blk << blk_sz_shift;
		n_blks = vfbio_sgl_seg_get_n_blks(hw_sgl, seg);
		len = n_blks * blk_sz;

		dev_dbg(dev, "sg %s:\n", op == VFBIO_FUNC_SG_READ ? "read" : "write");
		dev_dbg(dev, "%d blocks @ block %d\n", n_blks, blk);
		dev_dbg(dev, "lun addr: %p, lun offset: 0x%08x\n",
			luns[lun].addr, lun_offset);
		dev_dbg(dev, "page: %p, page_offset: 0x%08x\n", page, page_offset);

		if (op == VFBIO_FUNC_SG_READ) {
			memcpy(page_addr + page_offset, luns[lun].addr + lun_offset, len);
			flush_dcache_page(page);
			__vfbioss_dump_buf(dev, page_addr + page_offset, len);
		} else {
			flush_dcache_page(page);
			__vfbioss_dump_buf(dev, page_addr + page_offset, len);
			memcpy(luns[lun].addr + lun_offset, page_addr + page_offset, len);
		}
		kunmap_atomic(page_addr);
	}

	dev_dbg(dev, "sg %s: %d @ %d on tunnel: %d, client: %s, lun: %d\n",
		op == VFBIO_FUNC_SG_READ ? "read" : "wrote", blks_done,
		vfbio_sgl_get_blk(hw_sgl), tunnel, vfdev->client_name, lun);

reply:
	rpc_msg_init(msg, RPC_SERVICE_VFBIO, VFBIO_FUNC_ASYNC_COMPLETE, 0,
		     0, 0, 0);
	vfbio_msg_set_retcode(msg, status);
	vfbio_msg_set_lun(msg, lun);
	vfbio_msg_set_n_blks(msg, blks_done);
	vfbio_msg_set_tag(msg, tag);
	rpc_send_message(tunnel, msg, false);

	pr_debug("%s: <--\n", __func__);
	return status;
}

static int vfbioss_probe(struct platform_device *pdev)
{
	int status = 0;
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct device_node *rnp;
	char *str;
	struct vfbioss_device *vfbioss;

	vfbioss = devm_kzalloc(dev, sizeof(*vfbioss), GFP_KERNEL);
	if (!vfbioss) {
		dev_err(dev, "%s: Unable to allocate vfbioss.\n", __func__);
		status = -ENOMEM;
		goto done;
	}
	vfbioss->pdev = pdev;

	rnp = of_parse_phandle(np, "rpc-channel", 0);
	if (!rnp) {
		dev_err(dev, "%s: Unable to retrieve rpc-channel phandle.\n",
			__func__);
		status = -EINVAL;
		goto done;
	}
	status = of_property_read_string(rnp, "dev-name", (const char **)&str);
	of_node_put(rnp);
	if (status) {
		dev_err(dev, "%s: RPC node dev-name property missing or "
			     "malformed.\n", __func__);
		goto done;
	}
	vfbioss->tunnel = rpc_get_fifo_tunnel_id(str);
	if (vfbioss->tunnel < 0) {
		dev_err(dev, "%s: Unable to obtain RPC tunnel ID.\n",
			__func__);
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(str, "smc-rg", sizeof("smc-rg"))) {
		vfbioss->client_id = CLIENT_ID_RG;
		vfbioss->client_name = "RG";
	} else if (!strncmp(str, "smc-cm", sizeof("smc-cm"))) {
		vfbioss->client_id = CLIENT_ID_CM;
		vfbioss->client_name = "CM";
	} else {
		dev_err(dev, "%s: %s is not an SMC tunnel.\n", __func__, str);
		status = -EINVAL;
		goto done;
	}

	platform_set_drvdata(pdev, vfbioss);
	list_add_rcu(&vfbioss->list, &vfdevs);
	synchronize_rcu();

	pr_info("tunnel\t%s (ID: %d)\n", str, vfbioss->tunnel);

done:
	return status;
}

static int vfbioss_remove(struct platform_device *pdev)
{
	struct vfbioss_device *vfbioss = platform_get_drvdata(pdev);

	list_del_rcu(&vfbioss->list);
	synchronize_rcu();
	return 0;
}

static rpc_function vfbioss_services_tbl[] =
{
	{ vfbioss_alloc,	0 },
	{ vfbioss_free,		0 },
	{ vfbioss_lun_info,	0 },
	{ vfbioss_sync_rw,	0 },
	{ vfbioss_sync_rw,	0 },
	{ vfbioss_discard,	0 },
	{ vfbioss_flush,	0 },
	{ vfbioss_sg_rw,	0 },
	{ vfbioss_sg_rw,	0 },
	{ NULL,			0 },
	{ NULL			0 }
};

static const struct of_device_id vfbioss_of_match[] = {
	{.compatible = "brcm,vfbioss"},
	{}
};
MODULE_DEVICE_TABLE(of, vfbioss_of_match);
static struct platform_driver vfbioss_driver = {
	.probe		= vfbioss_probe,
	.remove		= vfbioss_remove,
	.driver		= {
		.owner	= THIS_MODULE,
		.name	= "brcm,vfbioss",
		.of_match_table	= vfbioss_of_match
	},
};

static int __init vfbioss_init(void)
{
	int status;
	unsigned int lun;

	pr_info("%s driver v%s\n", MODULE_NAME, MODULE_VER);

	INIT_LIST_HEAD_RCU(&vfdevs);

	for (lun = 0; lun < VFBIO_LUN_MAX; lun++) {
		luns[lun].addr = kzalloc(luns[lun].n_blks * luns[lun].blk_sz,
					  GFP_KERNEL);
		if (!luns[lun].addr) {
			pr_err("%s: failed to alloc memory for LUN %d\n",
				__func__, lun);
			status = -ENOMEM;
			goto err_free_luns;
		}
	}

	status = rpc_register_functions(RPC_SERVICE_VFBIO,
					vfbioss_services_tbl,
					VFBIO_FUNC_MAX);
	if (status) {
		pr_err("%s: Failed to register RPC functions.\n", __func__);
		goto err_free_luns;
	}

	status = vfbioss_proc_init();
	if (status)
		goto err_rm_proc;

	status = platform_driver_register(&vfbioss_driver);
	if (status)
		goto err_unreg_rpc;
	goto done;

err_rm_proc:
	vfbioss_proc_exit();

err_unreg_rpc:
	rpc_unregister_functions(RPC_SERVICE_VFBIO);

err_free_luns:
	for (lun = 0; lun < VFBIO_LUN_MAX; lun++)
		if (luns[lun].addr)
			kfree(luns[lun].addr);

done:
	return status;
}
late_initcall(vfbioss_init);

static void vfbioss_exit(void)
{
	unsigned int lun;

	platform_driver_unregister(&vfbioss_driver);
	vfbioss_proc_exit();
	rpc_unregister_functions(RPC_SERVICE_VFBIO);
	for (lun = 0; lun < VFBIO_LUN_MAX; lun++)
		if (luns[lun].addr)
			kfree(luns[lun].addr);
}
module_exit(vfbioss_exit);

MODULE_AUTHOR("Tim Ross <tim.ross@broadcom.com>");
MODULE_LICENSE("GPL v2");
