 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/spi/spi.h>
#include <linux/of.h>
#include <linux/of_net.h>
#include <linux/ioport.h>
#include <linux/delay.h>

#include "ethsw.h"
#include "ethsw_priv.h"

/*
 This file contains initialization for ethsw driver when the driver is used as a
 spi (as opposed to platform) type driver.
 */

static DEFINE_MUTEX(spi_mutex);

/* define log module */
#define LOG_MODULE "ethsw_init_spi"

struct spi_driver ethsw_sdrv __refdata;


static int __init ethsw_sdev_probe(struct spi_device *spi);
static int __exit ethsw_sdev_remove(struct spi_device *sdev);

static int __init ethsw_swdev_param_init(struct ethsw_device *swdev)
{
	int ret = 0;
	struct spi_device *sdev;
	struct device_node *node;

	FUNC_ENTER();

	sdev = swdev->sdev;
	node = sdev->dev.of_node;

	/* set disabled by default */
	swdev->vlan_ipv6_mc_nf_mode_enabled = false;

	/* get ethsw device name */
	if (of_property_read_string(node, "dev-name", &swdev->name)) {
		LOG_ERR("Unable to obtain name string\n");
		ret = -EINVAL;
		goto EXIT;
	}

	/* get MII port params */
	swdev->pm_params = of_get_property(node, "mii-port", &swdev->pm_count);
	if (!swdev->pm_params) {
		LOG_DEBUG("Unable to obtain MII port properties\n");
		swdev->pm_count = 0;
	} else {
		/* convert size to 3-word group */
		swdev->pm_count /= (sizeof(int) * 3);
	}

	/* get IMP port params */
	swdev->pi_params = of_get_property(node, "imp-port", &swdev->pi_count);
	if (!swdev->pi_params) {
		LOG_ERR("Unable to obtain IMP port properties\n");
		ret = -EINVAL;
		goto EXIT;
	}

	/* convert size to 2-word group */
	swdev->pi_count /= (sizeof(int) * 2);


	/* get LED map */
	if (of_property_read_u32(node, "led-map", &swdev->led_map)) {
		/* default to LED on for all ports */
		swdev->led_map = -1;
	}

	if (of_property_read_u32(node, "spi-max-frequency", &swdev->spi_freq)) {
		LOG_CRIT("Unable to obtain spi frequency\n");
		ret = -EINVAL;
		goto EXIT;
	}

	if (of_property_read_u32(node, "eee-support", &swdev->eee_support)) {
		LOG_CRIT("Unable to obtain Energy-Efficient Ethernet settings\n");
		ret = -EINVAL;
		goto EXIT;
	}

	if (!of_find_property(node, "subid-port-offset",NULL)) {
		swdev->subid_port_offset = 0;
	} else {
		if (of_property_read_u32(node, "subid-port-offset", &swdev->subid_port_offset)) {
			LOG_CRIT("Unable to obtain subid-port-offset\n");
			ret = -EINVAL;
			goto EXIT;
		}
	}

	if (!of_find_property(node, "vlan-default-mode", NULL)) {
		/* if we can not find it then we default to vlans on*/
		swdev->vlan_default_mode = 1;
	} else {
		if (of_property_read_u32(node, "vlan-default-mode", &swdev->vlan_default_mode)) {
			LOG_CRIT("Unable to obtain vlan default mode setting\n");
			ret = -EINVAL;
			goto EXIT;
		}
	}

	if (of_property_read_u32(node, "phy-port-mask", &swdev->phy_port_mask)) {
		swdev->phy_port_mask = 0xF;
	}

EXIT:

	FUNC_LEAVE();
	return(ret);
}


static int __init ethsw_swdev_phy_param_init(struct ethsw_device *swdev)
{

	int i, port_id, ret = 0;
	struct spi_device *sdev;
	struct device_node *node;
	struct ethsw_port *port;
	int num_of_ports_defined =0;

	FUNC_ENTER();

	sdev = swdev->sdev;
	node = sdev->dev.of_node;

	for (i = 0; i < ETHSW_PHY_PORT_MAX; i++) {
		if (!(swdev->phy_port_mask & (1 << i)))
			continue;

		port_id = swdev->ethsw_port_idx_to_port_id(swdev, i);
		port = &swdev->port[port_id];
		port->phy.node = of_parse_phandle(node, "phy-handles", num_of_ports_defined);
		if (!port->phy.node) {
			break;
		}

		num_of_ports_defined++;

		of_node_put(port->phy.node);
		ret = ethsw_phy_get_property(&swdev->pdev->dev, port, port_id);
		if (ret)
			goto EXIT;

		port_id = swdev->subid_port_offset + i;
		if (swdev->chip_id == ETHSW_53124_SWITCH)
			subid_to_swdev_table[port_id + ETHSW_53124_PORT_OFFSET] = swdev;

		subid_to_swdev_table[port_id] = swdev;
	}

EXIT:

	FUNC_LEAVE();
	return(ret);
}


static int __init ethsw_sdev_probe(struct spi_device *spi)
{
	int ret = 0;
	int i;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	/* alloc ethsw driver control block */
	ethsw_drv[EXTERNAL_SPI_SW] = kzalloc(sizeof(struct ethsw_driver), GFP_KERNEL);
	if (!ethsw_drv[EXTERNAL_SPI_SW]) {
		LOG_CRIT("Ethsw driver control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}
	ethsw_drv[EXTERNAL_SPI_SW]->sdrv = &ethsw_sdrv;

	/* alloc ethsw device control block */
	swdev = kzalloc(sizeof(struct ethsw_device), GFP_KERNEL);
	if (!swdev) {
		LOG_CRIT("Ethsw device control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	swdev->sdev = spi;
	swdev->ethsw_creg_read = ethsw_creg_read_spi;
	swdev->ethsw_creg_write = ethsw_creg_write_spi;
	swdev->ethsw_phy_board_init = ethsw_phy_board_init_531xx;
	swdev->ethsw_led_board_init = 0;
	swdev->ethsw_int_board_init = 0;
	swdev->ethsw_led_powercntrl = ethsw_led_powercntrl_531xx;
	swdev->ethsw_switch_set_port_stp_state_specific = 0;
	swdev->ethsw_int_board_isr = 0;
	swdev->ethsw_port_idx_to_phy_id = ethsw_port_idx_to_phy_id_531xx;
	swdev->ethsw_port_idx_to_port_id = ethsw_port_idx_to_port_id_531xx;
	swdev->ethsw_port_id_to_port_idx = ethsw_port_id_to_port_idx_531xx;
	swdev->ethsw_led_board_exit = 0;
	swdev->ethsw_phy_board_exit = 0;
	swdev->ethsw_int_board_proc = 0;
	swdev->ethsw_swdev_board_init = ethsw_swdev_board_init_531xx;
	swdev->ethsw_swdev_board_exit = ethsw_swdev_board_exit_531xx;
	swdev->ethsw_int_board_exit = 0;
	swdev->ethsw_port_status_get = ethsw_port_status_get_531xx;
	swdev->ethsw_low_pwr_mode = ethsw_pwr_down_531xx;
	swdev->ethsw_normal_pwr_mode = ethsw_pwr_up_531xx;
   swdev->imp_port_count = 0;

	ethsw_init_invalid_page_map();

	/* update ethsw driver */
	for (i = 0 ; i < ETHSW_EXTERNAL_DEVICE_MAX; i++) {
		if (!ethsw_drv[EXTERNAL_SPI_SW]->swdev[i]) {
			ethsw_drv[EXTERNAL_SPI_SW]->swdev[i] = swdev;
			ethsw_drv[EXTERNAL_SPI_SW]->swdev_count++;
		}
	}

	/* save ethsw device for global reference */
	if (!ethsw_dev[EXTERNAL_SPI_SW]) {
		ethsw_dev[EXTERNAL_SPI_SW] = swdev;
	}

	ret = ethsw_arl_cache_table_init(swdev);
	if (ret)
		goto ERROR;

	/* get device params */
	ret = ethsw_swdev_param_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw device param init failed\n");
		goto ERROR;
	}
	LOG_INFO("Device name: %s\n", swdev->name);

	for (i = 0; i < swdev->pm_count; i++) {
		int port_id, mii_speed;
		bool rgmii_en;
		struct ethsw_port *port;

		port_id   = be32_to_cpu(swdev->pm_params[i * 3    ]);
		rgmii_en  = be32_to_cpu(swdev->pm_params[i * 3 + 1]);
		mii_speed = be32_to_cpu(swdev->pm_params[i * 3 + 2]);

		swdev->mii_port_mask |= (1 << port_id);

		port = &swdev->port[port_id];
		port->type = MII_PORT;
		/* get rgmii mode */
		port->mii.rgmii = rgmii_en;
		port->mii.speed = mii_speed;

		LOG_DEBUG("MII port params: <0x%x %x 0x%x>\n",
				port_id, rgmii_en, mii_speed);
	}

	for (i = 0; i < swdev->pi_count; i++) {
		int port_id, imp_speed;
		struct ethsw_port *port;

		port_id   = be32_to_cpu(swdev->pi_params[i * 2    ]);
		imp_speed = be32_to_cpu(swdev->pi_params[i * 2 + 1]);

		swdev->imp_port_mask |= (1 << port_id);
      swdev->imp_port_count++;

		port = &swdev->port[port_id];
		port->type = IMP_PORT;
		port->imp.speed = imp_speed;

		LOG_DEBUG("IMP port params: <0x%x 0x%x>\n",
				port_id, imp_speed);
	}

	LOG_DEBUG("LED map: 0x%x\n", swdev->led_map);

	/* remap IO */
	swdev->reg_base = ioremap(
			0xd4e00000,
			0x42000);
	if (!swdev->reg_base) {
		LOG_CRIT("Ethsw device IO remap failed\n");
		ret = -EIO;
		goto ERROR;
	}

	/* init ethsw device, platformat dependent */
	ret = swdev->ethsw_swdev_board_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw device board init failed\n");
		goto ERROR;
	}

	/* get PHY port params */
	ethsw_swdev_phy_param_init(swdev);
	ethsw_phy_port_init_ids(swdev);

	/* init switch core */
	ret = ethsw_core_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw core init failed\n");
		goto ERROR;
	}


	/* init switch PHY */
	ret = ethsw_phy_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw PHY init failed\n");
		goto ERROR;
	}

	/* reset config and disable all ports */
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		if (swdev->port[i].type == PHY_PORT) {
			if (!(swdev->phy_port_mask & (1 << i)))
				continue;

			ethsw_port_config_set(i + swdev->subid_port_offset, NULL);
			ethsw_port_power_config_set(i + swdev->subid_port_offset, NULL);
			ethsw_port_disable(i + swdev->subid_port_offset);
		}
	}

#if PROC_IF_SUPPORT
	/* init proc file system */
	ret = ethsw_proc_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw proc init failed\n");
		goto ERROR;
	}
#endif

#if CDEV_IF_SUPPORT
	/* init cdev device */
	ret = ethsw_cdev_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw cdev init failed\n");
		goto ERROR;
	}
#endif

#if VNET_IF_SUPPORT
	/* init vnet device */
	ret = ethsw_vnet_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw vnet init failed\n");
		goto ERROR;
	}
#endif

#if 0
	LOG_INFO("spi_device 0x%x\n", (u32)spi);
	LOG_INFO("max_spped_hz %d\n", (u32)spi->max_speed_hz);
	LOG_INFO("chip_select 0x%x\n", (u32)spi->chip_select);
	LOG_INFO("bits_per_word 0x%x\n", (u32)spi->bits_per_word);
	LOG_INFO("mode 0x%x\n", (u32)spi->mode);

	ethsw_spi_read(0x02, 0x30, &devID, 4);
	LOG_INFO("Device ID 0x%x\n", devID);

	ethsw_spi_read(0x11, 0x04, &phyID, 2);
	LOG_INFO("Phy ID 0x%x\n", phyID);

	ethsw_spi_read(0x0, 0x18, &LEDmodeMap, 2);
	LOG_INFO("LED mode map 0 0x%x\n", LEDmodeMap);
	LEDmodeMap = 0x1aa;
	ethsw_spi_write(0x0, 0x18, &LEDmodeMap, 2);
	ethsw_spi_read(0x0, 0x18, &LEDmodeMap, 2);
	LOG_INFO("LED mode map 0 0x%x\n", LEDmodeMap);

	ethsw_spi_read(0x0, 0x1A, &LEDmodeMap, 2);
	LOG_INFO("LED mode map 1 0x%x\n", LEDmodeMap);
	LEDmodeMap = 0x1aa;
	ethsw_spi_write(0x0, 0x1A, &LEDmodeMap, 2);
	ethsw_spi_read(0x0, 0x1A, &LEDmodeMap, 2);
	LOG_INFO("LED mode map 1 0x%x\n", LEDmodeMap);
#endif

	FUNC_LEAVE();
	return(ret);

ERROR:

	FUNC_LEAVE();
	return ret;
}

static int __exit ethsw_sdev_remove(struct spi_device *spi)
{
	FUNC_ENTER();

	FUNC_LEAVE();
	return 0;
}


static struct of_device_id ethsw_of_match[] = {
	{.compatible = "brcm,ethsw"},
	{}
};

static const struct spi_device_id ethsw_ids[] = {
	{ "ethsw", },
	{},
};

struct spi_driver ethsw_sdrv = {
	.probe    = ethsw_sdev_probe,
	.remove	  = ethsw_sdev_remove,
	.id_table = ethsw_ids,
	.driver   = {
		.name	= "ethsw",
		.owner	= THIS_MODULE,
		.of_match_table = ethsw_of_match
	},
};
#ifdef CONFIG_BCM3384
module_spi_driver(ethsw_sdrv);
#endif
MODULE_LICENSE("GPL");
