 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/netdevice.h>
#include <linux/of_net.h>
#include <linux/pinctrl/consumer.h>
#include <linux/gpio/consumer.h>
#include <linux/brcmxcvr.h>

#define KERNEL_DEBUG 1

#include "ethsw.h"
#include "ethsw_priv.h"
#include "ethsw_db.h"
#include "ethsw_core.h"

/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_phy"

int ethsw_phy_init(struct ethsw_device *swdev)
{
	int ret = 0;

	FUNC_ENTER();

	/* init switch PHY, platform dependent */
	if (swdev->ethsw_phy_board_init) {
		ret = swdev->ethsw_phy_board_init(swdev);
		if (ret) {
			LOG_CRIT("PHY board init failed\n");
			goto ERROR;
		}
	}

	/* init LED, platform dependent */
	if (swdev->ethsw_led_board_init) {
		ret = swdev->ethsw_led_board_init(swdev);
		if (ret) {
			LOG_CRIT("LED board init failed\n");
			goto ERROR;
		}
	}

	FUNC_LEAVE();
	return(ret);

ERROR:
	/* exit PHY properly */
	ethsw_phy_exit(swdev);

	FUNC_LEAVE();
	return(ret);
}

void ethsw_phy_exit(struct ethsw_device *swdev)
{
	FUNC_ENTER();

	/* exit LED, platform dependent */
	if (swdev->ethsw_led_board_exit) {
		swdev->ethsw_led_board_exit(swdev);
	}

	/* exit switch PHY, platform dependent */
	if (swdev->ethsw_phy_board_exit) {
		swdev->ethsw_phy_board_exit(swdev);
	}

	LOG_DEBUG("Ethsw PHY exited\n");

	FUNC_LEAVE();
}

int ethsw_phy_get_property(struct device *dev,
		struct ethsw_port *port, int port_id)
{
	int status;

	port->phy.id = of_mdio_parse_addr(dev, port->phy.node);
	if (port->phy.id < 0) {
		port->phy.node = NULL;
		port->phy.id = 0;
		LOG_CRIT("Unable to obtain phy addr for phy %d\n", port_id);
		return -EINVAL;
	}

	dev_dbg(dev, "%s: port->phy.id %d\n", __func__, port->phy.id);
	status = of_get_phy_mode(port->phy.node, &port->phy.mode);
	if (status) {
		port->phy.node = NULL;
		port->phy.id = 0;
		port->phy.mode = PHY_INTERFACE_MODE_NA;
		LOG_CRIT("Unable to obtain phy mode for phy %d\n", port_id);
		return -EINVAL;
	}

	return 0;
}

static int ethsw_phyxcvr_set_power_state(struct ethsw_device *swdev,
		int port_id,
		bool afe_pwrdn)
{
	struct ethsw_port *port = &swdev->port[port_id];
	struct phy_device *phydev = port->phy.dev;
	struct brcmphy_xcvr *phyxcvr;
	struct brcmphy_core *phycore;
	int ret = 0;

	if (!afe_pwrdn) {
		if (port->xcvr_power && port->afe_pwrdn_xcvr_power) {
			ret = regulator_enable(port->xcvr_power);
			if (ret) {
				pr_err("Couldn't set xcvr power on, err %d\n",
						ret);
				goto EXIT;
			}

			port->afe_pwrdn_xcvr_power = 0;
		}

		if (phydev) {
			phyxcvr = dev_get_drvdata(&phydev->mdio.dev);
			if (phyxcvr) {
				phycore = phyxcvr->phycore;
				if (phycore && phycore->resume &&
						port->afe_pwrdn_xcvr_core &&
						!in_interrupt()) {
					phycore->resume(phycore);
					port->afe_pwrdn_xcvr_core = 0;
				}
			}
		}

		if (port->xcvr_reset_duration && port->afe_pwrdn_xcvr_reset_duration) {
			mdelay(port->xcvr_reset_duration);
			port->afe_pwrdn_xcvr_reset_duration = 0;
		}

		if (port->xcvr_pins && port->afe_pwrdn_xcvr_pins) {
			ret = pinctrl_select_state(port->xcvr_pins,
					port->xcvr_pins_state[1]);
			if (ret) {
				pr_err("Couldn't set xcvr on pins, err %d\n",
						ret);
				goto EXIT;
			}

			port->afe_pwrdn_xcvr_pins = 0;
		}

		if (port->xcvr_reset && port->afe_pwrdn_xcvr_reset) {
			gpiod_set_value(port->xcvr_reset, 1);
			port->afe_pwrdn_xcvr_reset = 0;
		}

	} else {
		if (port->xcvr_reset && !port->afe_pwrdn_xcvr_reset) {
			gpiod_set_value(port->xcvr_reset, 0);
			port->afe_pwrdn_xcvr_reset = 1;
		}

		if (port->xcvr_reset_duration && !port->afe_pwrdn_xcvr_reset_duration) {
			mdelay(port->xcvr_reset_duration);
			port->afe_pwrdn_xcvr_reset_duration = 1;
		}

		if (port->xcvr_pins && !port->afe_pwrdn_xcvr_pins) {
			ret = pinctrl_select_state(port->xcvr_pins,
					port->xcvr_pins_state[0]);
			if (ret) {
				pr_err("Couldn't set xcvr off pins, err %d\n",
						ret);
				goto EXIT;
			}

			port->afe_pwrdn_xcvr_pins = 1;
		}

		if (phydev) {
			phyxcvr = dev_get_drvdata(&phydev->mdio.dev);
			if (phyxcvr) {
				phycore = phyxcvr->phycore;
				if (phycore && phycore->suspend &&
						!port->afe_pwrdn_xcvr_core &&
						!in_interrupt()) {
					phycore->suspend(phycore);
					port->afe_pwrdn_xcvr_core = 1;
				}
			}
		}

		if (port->xcvr_power && !port->afe_pwrdn_xcvr_power) {
			ret = regulator_disable(port->xcvr_power);
			if (ret) {
				pr_err("Couldn't set xcvr power off, err %d\n",
						ret);
				goto EXIT;
			}

			port->afe_pwrdn_xcvr_power = 1;
		}
	}

EXIT:
	return ret;
}

static bool ethsw_phyxcvr_get_power_state(struct ethsw_device *swdev,
		int port_id)
{
	struct ethsw_port *port = &swdev->port[port_id];

	if (port->xcvr_power && !regulator_is_enabled(port->xcvr_power))
		return true;

	return false;
}

int ethsw_phy_get_pwr_rst_pin(struct device *dev, struct ethsw_port *port)
{
	int ret = 0;

	port->xcvr_power = devm_regulator_get(dev, "xcvr-power");
	if (IS_ERR(port->xcvr_power)) {
		ret = PTR_ERR(port->xcvr_power);
		dev_err(dev, "Couldn't get xcvr power, err %d", ret);
		goto EXIT;
	}

	port->xcvr_reset = devm_gpiod_get(dev, "xcvr-reset", GPIOD_OUT_LOW);
	if (IS_ERR(port->xcvr_reset)) {
		devm_regulator_put(port->xcvr_power);
		ret = PTR_ERR(port->xcvr_reset);
		dev_err(dev, "Couldn't get xcvr reset, err %d", ret);
		goto EXIT;
	}

	ret = gpiod_direction_output(port->xcvr_reset, 0);
	if (ret) {
		dev_err(dev, "Couldn't set xcvr reset gpio direction, err %d",
				ret);
		goto EXIT;
	}

	port->xcvr_pins = devm_pinctrl_get(dev);
	if (IS_ERR(port->xcvr_pins)) {
		devm_regulator_put(port->xcvr_power);
		devm_gpiod_put(dev, port->xcvr_reset);
		ret = PTR_ERR(port->xcvr_pins);
		dev_err(dev, "Couldn't get xcvr pins, err %d", ret);
		goto EXIT;
	}

	if (port->xcvr_pins) {
		port->xcvr_pins_state[0] = pinctrl_lookup_state(port->xcvr_pins,
				"xcvr_off");
		if (IS_ERR(port->xcvr_pins_state[0])) {
			devm_regulator_put(port->xcvr_power);
			devm_gpiod_put(dev, port->xcvr_reset);
			devm_pinctrl_put(port->xcvr_pins);
			ret = PTR_ERR(port->xcvr_pins_state[0]);
			dev_err(dev, "Couldn't get xcvr off pins, err %d", ret);
			goto EXIT;
		}

		port->xcvr_pins_state[1] = pinctrl_lookup_state(port->xcvr_pins,
				"xcvr_on");
		if (IS_ERR(port->xcvr_pins_state[1])) {
			devm_regulator_put(port->xcvr_power);
			devm_gpiod_put(dev, port->xcvr_reset);
			devm_pinctrl_put(port->xcvr_pins);
			ret = PTR_ERR(port->xcvr_pins_state[1]);
			dev_err(dev, "Couldn't get xcvr on pins, err %d", ret);
			goto EXIT;
		}
	}

	if (of_property_read_u32(dev->of_node, "xcvr-reset-duration", &port->xcvr_reset_duration))
		port->xcvr_reset_duration = 0;

	if (port->xcvr_power || port->xcvr_reset || port->xcvr_pins) {
		port->ethsw_phy_set_power_state =
			ethsw_phyxcvr_set_power_state;
		port->ethsw_phy_get_power_state =
			ethsw_phyxcvr_get_power_state;
		port->afe_pwrdn_xcvr_power = 1;
		port->afe_pwrdn_xcvr_core = 0;
		port->afe_pwrdn_xcvr_reset_duration = 1;
		port->afe_pwrdn_xcvr_pins = 1;
		port->afe_pwrdn_xcvr_reset = 1;
	}

EXIT:
	return ret;
}

void ethsw_phy_port_init_ids(struct ethsw_device *swdev)
{
	int i;

	FUNC_ENTER();

	for(i = 0; i < ETHSW_PHY_PORT_MAX; i ++) {
		int port_id, phy_id;
		struct ethsw_port *port;

		if (!(swdev->phy_port_mask & (1 << i)))
			continue;

		port_id = swdev->ethsw_port_idx_to_port_id(swdev, i);
		phy_id  = swdev->ethsw_port_idx_to_phy_id(swdev, i);

		port = &swdev->port[port_id];
		port->type = PHY_PORT;
		port->phy.id = phy_id;

		/* default to auto neg */
		port->config.auto_neg = true;

		LOG_DEBUG("PHY port params for %d: <0x%x 0x%x>\n",
				i, port_id, phy_id);
	}

	FUNC_LEAVE();
}

void ethsw_phy_link_cb_sf2(struct net_device *dev)
{
	int i, found = false, ret;
	u32 val;
	struct phy_device *phydev = dev->phydev;

	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];
	struct ethsw_port *port;
	struct ethsw_port_status new_status = {0};


	FUNC_ENTER();


	for(i = 0; i < ETHSW_PORT_MAX; i ++) {
		port = &swdev->port[i];
		if(phydev == port->phy.dev) {
			found = true;
			break;
		}
	}

	if (!found) {
		LOG_ERR("Matching phy device not found for port %s\n", dev->name);
		goto EXIT;
	}

	if(ethsw_reg_read(swdev, BCHP_SWITCH_CORE_LOW_POWER_CTRL) != 0)
		goto EXIT;

	new_status.link_up = phydev->link;
	new_status.full_duplex = phydev->duplex;
	new_status.speed = 0;

	switch (phydev->speed) {
	case SPEED_10:
		new_status.speed = PORT_SPEED_10M;
		break;
	case SPEED_100:
		new_status.speed = PORT_SPEED_100M;
		break;
	case SPEED_1000:
		new_status.speed = PORT_SPEED_1000M;
		break;
	case SPEED_2500:
		new_status.speed = PORT_SPEED_2500M;
		break;
	}

	if (new_status.link_up     == port->status.link_up &&
			new_status.full_duplex == port->status.full_duplex &&
			new_status.speed       == port->status.speed)
		goto EXIT;

	ethsw_port_tx_enable(swdev, i, 0);
	ethsw_port_rx_enable(swdev, i, 0);

	//phy_print_status(phydev);

	/* update port status */
	port->status = new_status;

	LOG_DEBUG("port: %d, phy: %d, link: %d, duplex: %d,"
			"speed: %d\n", i, port->phy.id, port->status.link_up,
			port->status.full_duplex, port->status.speed);

	if (port->type == MII_PORT) {

		CHECK_EXIT(CREG_IDX_RD(PAGE_CTLREG, CREG_STS_OVERRIDE_GMII, i, &val));

		val |= (1 << CREG_STS_OVERRIDE_GMII_SW_OVERRIDE_SHIFT);

		if(port->status.link_up) {
			val &= ~(CREG_STS_OVERRIDE_GMII_SPEED_MASK);
			if (phydev->speed > SPEED_100)
				val |= (2 << CREG_STS_OVERRIDE_GMII_SPEED_SHIFT);
			else
				val |= (1 << CREG_STS_OVERRIDE_GMII_SPEED_SHIFT);
			val |= (1 << CREG_STS_OVERRIDE_GMII_LINK_STS_SHIFT);
		} else
			val &= ~(1 << CREG_STS_OVERRIDE_GMII_LINK_STS_SHIFT);

		CHECK_EXIT(CREG_IDX_WR(PAGE_CTLREG, CREG_STS_OVERRIDE_GMII, i, &val));
	}

	if(port->cb) {
		(*port->cb)(port->cb_priv, &port->status);
	} else
		LOG_CRIT("Failed to find callback\n");

	if(port->status.link_up)
	{
		ethsw_port_tx_enable(swdev, i, 1);
		ethsw_port_rx_enable(swdev, i, 1);
	}

	if (new_status.link_up == false)
		ethsw_arl_fast_age_set(i);

EXIT:
	FUNC_LEAVE();
}

void ethsw_phy_link_cb_531xx(struct net_device *dev)
{
	int i, found = false;
	struct phy_device *phydev = dev->phydev;

	struct ethsw_device *swdev = ethsw_dev[EXTERNAL_SPI_SW];
	struct ethsw_port *port;
	struct ethsw_port_status new_status = {0};


	FUNC_ENTER();

	//phy_print_status(phydev);

	for(i = 0; i < ETHSW_PORT_MAX; i ++) {
		port = &swdev->port[i];
		if(phydev == port->phy.dev) {
			found = true;
			break;
		}
	}
	if(found == false) {
		LOG_CRIT("Failed to find port\n");
		goto EXIT;
	}

	new_status.link_up = phydev->link;
	new_status.full_duplex = phydev->duplex;
	new_status.speed = 0;
	switch (phydev->speed) {
	case SPEED_10:
		new_status.speed = PORT_SPEED_10M;
		break;
	case SPEED_100:
		new_status.speed = PORT_SPEED_100M;
		break;
	case SPEED_1000:
		new_status.speed = PORT_SPEED_1000M;
		break;
	}

	if (new_status.link_up     == port->status.link_up &&
			new_status.full_duplex == port->status.full_duplex &&
			new_status.speed       == port->status.speed)
		goto EXIT;

	/* update port status */
	port->status = new_status;

	LOG_DEBUG("port: %d, phy: %d, link: %d, duplex: %d,"
			"speed: %d\n", i, port->phy.id, port->status.link_up,
			port->status.full_duplex, port->status.speed);

	if(port->cb)
		(*port->cb)(port->cb_priv, &port->status);
	else
		LOG_CRIT("Failed to find callback\n");
	if (new_status.link_up == false) {
		ethsw_db_delete_by_port(i);
	}

EXIT:
	FUNC_LEAVE();
}
