/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2018 Broadcom. All rights reserved.
* The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
*
* This program is the proprietary software of Broadcom and/or
* its licensors, and may only be used, duplicated, modified or distributed
* pursuant to the terms and conditions of a separate, written license
* agreement executed between you and Broadcom (an "Authorized License").
* Except as set forth in an Authorized License, Broadcom grants no license
* (express or implied), right to use, or waiver of any kind with respect to
* the Software, and Broadcom expressly reserves all rights in and to the
* Software and all intellectual property rights therein. IF YOU HAVE NO
* AUTHORIZED LICENSE, THEN YOU HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY,
* AND SHOULD IMMEDIATELY NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE
* SOFTWARE.
*
* Except as expressly set forth in the Authorized License,
*
* 1. This program, including its structure, sequence and organization,
* constitutes the valuable trade secrets of Broadcom, and you shall use all
* reasonable efforts to protect the confidentiality thereof, and to use this
* information only in connection with your use of Broadcom integrated circuit
* products.
*
* 2. TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED
* "AS IS" AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS
* OR WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH
* RESPECT TO THE SOFTWARE. BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL
* IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR
* A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET
* ENJOYMENT, QUIET POSSESSION OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME
* THE ENTIRE RISK ARISING OUT OF USE OR PERFORMANCE OF THE SOFTWARE.
*
* 3. TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM
* OR ITS LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL,
* INDIRECT, OR EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY
* RELATING TO YOUR USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM
* HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN
* EXCESS OF THE AMOUNT ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1,
* WHICHEVER IS GREATER. THESE LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY
* FAILURE OF ESSENTIAL PURPOSE OF ANY LIMITED REMEDY.
*
****************************************************************************
*
* Filename:    ramlogrpc.c
*
****************************************************************************
*
* Description: ramlog RPC definitions
*
****************************************************************************/

#include "ramlogrpc.h"

#include <linux/kernel.h>
#include <asm/byteorder.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/of.h>
#include <linux/ktime.h>
#include <linux/dma-mapping.h>
#include <asm/div64.h>
#include "bcmcache.h"
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <linux/bcm_media_gw/itc_rpc/itc_msg_defs.h>
#include <linux/bcm_media_gw/itc_rpc/itc_channel_defs.h>

enum
{
	RPC_FUNCTION_GET_ENCRYPTED_RAMLOG = 9,
	RPC_FUNCTION_ADD_TO_RAMLOG,
	RPC_FUNCTION_CLEAR_RAMLOG,
	RPC_FUNCTION_SET_RAMLOG_LEVEL,
	RPC_FUNCTION_SET_RAMLOG_UART_FLAG,
	RPC_FUNCTION_GET_RAMLOG_COUNT,
	RPC_FUNCTION_GET_RAMLOG_LAST_UPDATE
};

#define	RAMLOG_DQM_TIMEOUT			10000	/* milliseconds */

int register_ramlog_rpc(struct ramlog_rpc_data *ramlog, struct device *dev,
	const char *tunnel_name)
{
	if (!ramlog)
		return -EINVAL;

	ramlog->tunnel_name[0] = '\0';
	if (tunnel_name && *tunnel_name) {
		strncpy(ramlog->tunnel_name, tunnel_name,
			sizeof(ramlog->tunnel_name));
		ramlog->tunnel_name[sizeof(ramlog->tunnel_name)-1] = '\0';
	}
	ramlog->tunnel = rpc_get_fifo_tunnel_id((char *) ramlog->tunnel_name);
	if (ramlog->tunnel < 0) {
		dev_err(dev, "%s: Unable to obtain RPC tunnel ID.\n",
			__func__);
		return -EIO;
	}
	ramlog->dev = dev;
	sema_init(&ramlog->sem, 1);
	return 0;
}

void release_ramlog_rpc(struct ramlog_rpc_data *ramlog)
{
	if (!ramlog)
		return;
	memset(ramlog, 0, sizeof(struct ramlog_rpc_data));
}

int register_ramlog_rpc_from_platform_device(struct ramlog_rpc_data *ramlog,
	struct platform_device *pdev)
{
	struct device_node *phan_node;
	struct device_node *of_node = pdev->dev.of_node;
	const char *dev_name;

	phan_node = of_parse_phandle(of_node, "rpc-channel", 0);
	if (!phan_node) {
		dev_err(&pdev->dev, "Unable to retrieve rpc-channel phandle ");
		return -EINVAL;
	}

	if (of_property_read_string(phan_node, "dev-name", &dev_name)) {
		dev_err(&pdev->dev, "%s: Missing dev-name property!\n",
				of_node_full_name(of_node));
		of_node_put(phan_node);
		return -EINVAL;
	}
	of_node_put(phan_node);

	return register_ramlog_rpc(ramlog, &pdev->dev, dev_name);
}

uint32_t get_encrypted_ramlog_entries(dma_addr_t encrypted_dma_addr,
	uint32_t maxbufsize, struct ramlog_rpc_data *ramlog,
	uint32_t *count_ptr, uint32_t *update_ptr,
	uint32_t start_index)
{
	rpc_msg msg;
	int status;

	if (!ramlog)
		return -EINVAL;
#ifdef	SMC_RAMLOG_VERBOSE
	dev_info(ramlog->dev, "buffer=%#lx size=%u (%#x) index=%#x\n",
		(unsigned long) encrypted_dma_addr,
		(unsigned int) maxbufsize, (unsigned int) maxbufsize,
		(unsigned int) start_index);
#endif	/* SMC_RAMLOG_VERBOSE */
	/* alignment to 8 bytes needed to acvoid lock up of SMC IOP DMA rev. A0 */
	maxbufsize = RAMLOG_ALIGNED(maxbufsize);
	memset(&msg, 0, sizeof(msg));
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		RPC_FUNCTION_GET_ENCRYPTED_RAMLOG, 0,
		(uint32_t) (((unsigned long) encrypted_dma_addr) &
			0xffffffff),
		((maxbufsize & 0xffffff) |
		(uint32_t) ((((uint64_t) encrypted_dma_addr) >> 8) &
			0xff000000)),
		(start_index & 0xffffff));

	barrier();
	if (maxbufsize)
		dma_sync_single_for_device(ramlog->dev, encrypted_dma_addr,
			maxbufsize, DMA_BIDIRECTIONAL);
	status = rpc_send_request_timeout(ramlog->tunnel, &msg,
		RAMLOG_DQM_TIMEOUT / 1000);
	if (status < 0)
		return status;
	start_index = (msg.data[0] & 0xffffff);
	if ((start_index & 0x800000) != 0)
		start_index |= 0xff000000;
	if (count_ptr)
		*count_ptr = (msg.data[1] & 0xffffff);
	if (update_ptr)
		*update_ptr = msg.data[2];
	if (maxbufsize)
		dma_sync_single_for_cpu(ramlog->dev, encrypted_dma_addr,
			maxbufsize, DMA_BIDIRECTIONAL);
	return start_index;
}

int clear_ramlog(struct ramlog_rpc_data *ramlog)
{
	rpc_msg msg;

	memset(&msg, 0, sizeof(msg));
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		RPC_FUNCTION_CLEAR_RAMLOG, 0, 0, 0, 0);
	return rpc_send_request_timeout(ramlog->tunnel, &msg,
		RAMLOG_DQM_TIMEOUT / 1000);
}

int add_to_ramlog(struct ramlog_rpc_data *ramlog, unsigned int severity,
	unsigned int length, dma_addr_t message_dma_addr)
{
	rpc_msg msg;
	int status;

	if (!ramlog)
		return -EINVAL;
	memset(&msg, 0, sizeof(msg));
	/* alignment to 8 bytes needed to acvoid lock up of SMC IOP DMA rev. A0 */
	length = RAMLOG_ALIGNED(length);

	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		RPC_FUNCTION_ADD_TO_RAMLOG, 0,
		(uint32_t) (((unsigned long) message_dma_addr) &
			0xffffffff),
		((length & 0xffffff) |
			(uint32_t) ((((uint64_t) message_dma_addr) >> 8) &
				0xff000000)),
		(uint32_t) (severity & 0xffff));

	barrier();
	if (length)
		dma_sync_single_for_device(ramlog->dev, message_dma_addr,
			length, DMA_BIDIRECTIONAL);
	status = rpc_send_request_timeout(ramlog->tunnel, &msg,
		RAMLOG_DQM_TIMEOUT / 1000);
	if (status < 0)
		return status;
	status = (int) ((msg.data[0] & 0xff000000) >> 24);
	if (status > 0)
		status = -status;
	if (length)
		dma_sync_single_for_cpu(ramlog->dev, message_dma_addr,
			length, DMA_BIDIRECTIONAL);
	return status;
}

int set_ramlog_level(struct ramlog_rpc_data *ramlog, unsigned int level,
	unsigned int length, dma_addr_t source_dma_addr)
{
	rpc_msg msg;
	int status;

	if (!ramlog)
		return -EINVAL;
	memset(&msg, 0, sizeof(msg));
	/* alignment to 8 bytes needed to acvoid lock up of SMC IOP DMA rev. A0 */
	length = RAMLOG_ALIGNED(length);
	if (length > 0)
		rpc_msg_init(&msg, RPC_SERVICE_SYS,
			RPC_FUNCTION_SET_RAMLOG_LEVEL, 0,
			(uint32_t) (((unsigned long) source_dma_addr) &
				0xffffffff),
			((length & 0xffffff) |
				(uint32_t) ((((uint64_t) source_dma_addr)
					>> 8) & 0xff000000)),
			(uint32_t) (level & 0xffff));
	else
		rpc_msg_init(&msg, RPC_SERVICE_SYS,
			RPC_FUNCTION_SET_RAMLOG_LEVEL, 0, 0, 0,
			(uint32_t) (level & 0xffff));

	barrier();
	if (length)
		dma_sync_single_for_device(ramlog->dev, source_dma_addr,
			length, DMA_BIDIRECTIONAL);
	status = rpc_send_request_timeout(ramlog->tunnel, &msg,
		RAMLOG_DQM_TIMEOUT / 1000);
	if (status < 0)
		return status;
	status = (int) ((msg.data[0] & 0xff000000) >> 24);
	if (status > 0)
		status = -status;
	if (length)
		dma_sync_single_for_cpu(ramlog->dev, source_dma_addr,
			length, DMA_BIDIRECTIONAL);
	return status;
}

int set_ramlog_uart_flag(struct ramlog_rpc_data *ramlog, unsigned int flag)
{
	rpc_msg msg;
	int status;

	memset(&msg, 0, sizeof(msg));
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		RPC_FUNCTION_SET_RAMLOG_UART_FLAG, 0,
		(uint32_t) (flag ? 1 : 0), 0, 0);
	status = rpc_send_request_timeout(ramlog->tunnel, &msg,
		RAMLOG_DQM_TIMEOUT / 1000);
	if (status < 0)
		return status;
	status = (int) ((msg.data[0] & 0xff000000) >> 24);
	if (status > 0)
		status = -status;
	return status;
}

unsigned int get_ramlog_count(struct ramlog_rpc_data *ramlog)
{
	rpc_msg msg;
	int status;

	memset(&msg, 0, sizeof(msg));
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		RPC_FUNCTION_GET_RAMLOG_COUNT, 0, 0, 0, 0);
	status = rpc_send_request_timeout(ramlog->tunnel, &msg,
		RAMLOG_DQM_TIMEOUT / 1000);
	if (status < 0)
		return 0;
	return (unsigned int) (msg.data[0] & 0xffffff);
}

uint32_t get_ramlog_last_update(struct ramlog_rpc_data *ramlog)
{
	rpc_msg msg;
	int status;

	memset(&msg, 0, sizeof(msg));
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		RPC_FUNCTION_GET_RAMLOG_LAST_UPDATE, 0, 0, 0, 0);
	status = rpc_send_request_timeout(ramlog->tunnel, &msg,
		RAMLOG_DQM_TIMEOUT / 1000);
	if (status < 0)
		return 0;
	return msg.data[1];
}
