/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2016 Broadcom.
* All rights reserved.
*
* Unless you and Broadcom execute a separate written software license
* agreement governing use of this software, this software is licensed to
* you under the terms of the GNU General Public License version 2 (the
* "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
* the following added to such license:
*
* As a special exception, the copyright holders of this software give you
* permission to link this software with independent modules, and to copy
* and distribute the resulting executable under terms of your choice,
* provided that you also meet, for each linked independent module, the
* terms and conditions of the license of that module. An independent
* module is a module which is not derived from this software. The special
* exception does not apply to any modifications of the software.
*
* Notwithstanding the above, under no circumstances may you combine this
* software in any way with any other Broadcom software provided under a
* license other than the GPL, without Broadcom's express prior written
* consent.
*
****************************************************************************
*
* Filename:       wbid.c
* Author:         Russell Enderby <russell.enderby@broadcom.com>
* Creation Date:  o3/24/2o16
*
****************************************************************************
* Description: This driver is an i2c device client used for accessing the
*		WiFi Board Identifier (WBID).
*
****************************************************************************/
#include "wbid.h"
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/of.h>

/*_ Everyone knows this version is the best version */
#define DRV_VERSION     "1.0"
#define DT_BOLT_NODE_NAME   "bolt"
#define DT_WBID_ID   "wifi-board-id"
#define WBID_INVALID	0xFFFFFFFF

/* WBID main data structure */
struct wbid_data {
	struct i2c_client *i2c_client;
	u8     i2c_addr;		/* i2c addr */
	u32    wbid;			/* wbid val */
};

/*_ quick! make a copy while nobody is looking! */
static struct wbid_data g_wbid_data;

static int wbid_detect(struct i2c_client *client, struct i2c_board_info *info)
{
	struct i2c_adapter *adapter = client->adapter;

	if (!i2c_check_functionality(adapter, I2C_FUNC_I2C |
				     I2C_FUNC_SMBUS_EMUL))
		return -ENODEV;

	strlcpy(info->type, "wbid", I2C_NAME_SIZE);
	return 0;
}
/* Read WBID parameter from DT, if not present then read it through
   I2C from a register that was programmed */
static u32 wbid_param_get(struct i2c_client *client)
{
	struct device_node *node;
	u32 wbid = WBID_INVALID;
	int ret;

	/* Find bolt node by name */
	node = of_find_node_by_name(NULL, DT_BOLT_NODE_NAME);
	if (!node) {
		pr_debug("WBID read through i2c-Bolt node not found in DT\n");
		goto exit;
	}
	/* Get WBID value */
	if (of_property_read_u32(node, DT_WBID_ID, &wbid))
		goto exit;

	pr_debug("WBID:0x%x read through DT\n", wbid);
	return wbid;
exit:
	pr_debug("WBID read through i2c, since it is not present in DT\n");
	/* Read WBID through I2C */
	ret = i2c_smbus_read_word_data(client, 0);
	if (ret < 0)
		wbid = WBID_INVALID;
	else
		wbid = ret;
	return wbid;
}

/* WBID API read function.  Call this from kernel space to read 16 bits
   of data from the WBID. */
u32 wbid_read(void)
{
	/* Get WBID params */
	return g_wbid_data.wbid;
}
EXPORT_SYMBOL(wbid_read);

static int wbid_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{
	struct wbid_data *data;
	int err;

	pr_info("WBID driver version %s\n", DRV_VERSION);

	data = devm_kzalloc(&client->dev,
			sizeof(g_wbid_data), GFP_KERNEL);
	if (!data) {
		err = -ENOMEM;
		goto exit;
	}

	/* Store off our internal values for later use */
	data->i2c_client = g_wbid_data.i2c_client = client;
	pr_info("WBID i2c client registered with addr:0x%x\n",
		client->addr);

	i2c_set_clientdata(client, data);

	/* Get WBID and store it */
	data->wbid = g_wbid_data.wbid = wbid_param_get(client);
	/*_ Insert our money, turn the crank three times, see if
		the machine spits out on paper our expected answer */
	pr_info("WBID(x.y):%d.%d WBID:0x%x\n", (data->wbid & 0xFFF0)>>4,
			(data->wbid  & 0x000F), data->wbid);

	return 0;
exit:
	return err;
}

static struct of_device_id wbid_of_match[] = {
	{.compatible = "brcm,wbid"},
	{}
};

static const struct i2c_device_id wbid_id[] = {
		{"wbid", 0},
		{}
};

MODULE_DEVICE_TABLE(i2c, wbid_id);

/* This is the driver that will be inserted */
static struct i2c_driver wbid_driver = {
		.class = I2C_CLASS_HWMON,
		.driver =  {
			.name = "wbid",
			.owner = THIS_MODULE,
			.of_match_table = wbid_of_match
		},
		.probe = wbid_probe,
		.id_table = wbid_id,
		.detect = wbid_detect,
};

/*_ Handy dandy boilerplate code remover macro */
module_i2c_driver(wbid_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Russell Enderby <russell.enderby@broadcom.com>");
MODULE_DESCRIPTION("Broadcom WBID Driver");
MODULE_VERSION(DRV_VERSION);
