/****************************************************************************
 *
 * Copyright (c) 2020 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Broadcom Hardware Abstraction Library for layered devices
 *
 * Media Access Control (macdev)
 * Media Independent Interface (miidev)
 *
 * Author: Ravi Patel <ravi.patel@broadcom.com>
 ****************************************************************************/

#ifndef _HAL_DEVICE_H_
#define _HAL_DEVICE_H_

#include <linux/netdevice.h>
#include <linux/of_platform.h>
#include <linux/regulator/consumer.h>
#include <linux/phy.h>

struct hal_device;
#define mac_device hal_device
#define mii_device hal_device

#define MACDEV_NET_STATS_STRINGS				\
{								\
	[MACDEV_NET_STATS_TX_BYTES]    = "Tx Bytes:      ",	\
	[MACDEV_NET_STATS_TX_PACKETS]  = "Tx Packets:    ", 	\
	[MACDEV_NET_STATS_TX_ERRORS]   = "Tx Errors:     ",	\
	[MACDEV_NET_STATS_TX_CAPACITY] = "Tx Capacity:   ",	\
	[MACDEV_NET_STATS_RX_BYTES]    = "Rx Bytes:      ",	\
	[MACDEV_NET_STATS_RX_PACKETS]  = "Rx Packets:    ",	\
	[MACDEV_NET_STATS_RX_ERRORS]   = "Rx Errors:     ",	\
}

enum macdev_net_stats {
	MACDEV_NET_STATS_TX_BYTES,
	MACDEV_NET_STATS_TX_PACKETS,
	MACDEV_NET_STATS_TX_ERRORS,
	MACDEV_NET_STATS_TX_CAPACITY,
	MACDEV_NET_STATS_RX_BYTES,
	MACDEV_NET_STATS_RX_PACKETS,
	MACDEV_NET_STATS_RX_ERRORS,
	MACDEV_NET_STATS_MAX
};

#define MACDEV_MIB_STATS_STRINGS							\
{											\
	[MACDEV_MIB_STATS_RX_64_BYTES_PACKETS]     = "Rx 64 Bytes Packets:     ",	\
	[MACDEV_MIB_STATS_RX_127_BYTES_PACKETS]    = "Rx 127 Bytes Packets:    ",	\
	[MACDEV_MIB_STATS_RX_255_BYTES_PACKETS]    = "Rx 255 Bytes Packets:    ",	\
	[MACDEV_MIB_STATS_RX_511_BYTES_PACKETS]    = "Rx 511 Bytes Packets:    ",	\
	[MACDEV_MIB_STATS_RX_1023_BYTES_PACKETS]   = "Rx 1023 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_RX_1518_BYTES_PACKETS]   = "Rx 1518 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_RX_1522_BYTES_PACKETS]   = "Rx 1522 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_RX_2047_BYTES_PACKETS]   = "Rx 2047 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_RX_4095_BYTES_PACKETS]   = "Rx 4095 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_RX_9216_BYTES_PACKETS]   = "Rx 9216 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_RX_16383_BYTES_PACKETS]  = "Rx 16383 Bytes Packets:  ",	\
	[MACDEV_MIB_STATS_RX_PACKETS]              = "Rx Packets:              ",	\
	[MACDEV_MIB_STATS_RX_UNICAST_PACKETS]      = "Rx Unicast Packets:      ",	\
	[MACDEV_MIB_STATS_RX_MULTICAST_PACKETS]    = "Rx Multicast Packets:    ",	\
	[MACDEV_MIB_STATS_RX_BROADCAST_PACKETS]    = "Rx Broadcast Packets:    ",	\
	[MACDEV_MIB_STATS_RX_FCS_ERRORS]           = "Rx FCS Errors:           ",	\
	[MACDEV_MIB_STATS_RX_CONTROL_PACKETS]      = "Rx Control Packets:      ",	\
	[MACDEV_MIB_STATS_RX_PAUSE_PACKETS]        = "Rx Pause Packets:        ",	\
	[MACDEV_MIB_STATS_RX_OPCODE_ERRORS]        = "Rx Opcode Errors:        ",	\
	[MACDEV_MIB_STATS_RX_ALIGNMENT_ERRORS]     = "Rx Alignment Errors:     ",	\
	[MACDEV_MIB_STATS_RX_LENGTH_ERRORS]        = "Rx Length Errors:        ",	\
	[MACDEV_MIB_STATS_RX_CODE_ERRORS]          = "Rx Code Errors:          ",	\
	[MACDEV_MIB_STATS_RX_CARRIER_ERRORS]       = "Rx Carrier Errors:       ",	\
	[MACDEV_MIB_STATS_RX_OVERSIZED_ERRORS]     = "Rx Oversized Errors:     ",	\
	[MACDEV_MIB_STATS_RX_JABBER_ERRORS]        = "Rx Jabber Errors:        ",	\
	[MACDEV_MIB_STATS_RX_MTU_ERRORS]           = "Rx MTU Errors:           ",	\
	[MACDEV_MIB_STATS_RX_MATCHED_CRC_PACKETS]  = "Rx Matched CRC Packets:  ",	\
	[MACDEV_MIB_STATS_RX_SINGLE_VLAN_PACKETS]  = "Rx Single VLAN Packets:  ",	\
	[MACDEV_MIB_STATS_RX_DOUBLE_VLAN_PACKETS]  = "Rx Double VLAN Packets:  ",	\
	[MACDEV_MIB_STATS_RX_TRUNCATED_ERRORS]     = "Rx Truncated Errors:     ",	\
	[MACDEV_MIB_STATS_RX_GOOD_PACKETS]         = "Rx Good Packets:         ",	\
	[MACDEV_MIB_STATS_RX_BYTES]                = "Rx Bytes:                ",	\
	[MACDEV_MIB_STATS_RX_RUNT_PACKETS]         = "Rx RUNT Packets:         ",	\
	[MACDEV_MIB_STATS_RX_UNDERSIZED_ERRORS]    = "Rx Undersized Errors:    ",	\
	[MACDEV_MIB_STATS_RX_FRAGMENT_ERRORS]      = "Rx Fragment Errors:      ",	\
	[MACDEV_MIB_STATS_RX_RUNT_BYTES]           = "Rx RUNT Bytes:           ",	\
	[MACDEV_MIB_STATS_RX_ERRORS]               = "Rx Errors:               ",	\
	[MACDEV_MIB_STATS_TX_CAPACITY]             = "Tx Capacity:             ",	\
	[MACDEV_MIB_STATS_TX_64_BYTES_PACKETS]     = "Tx 64 Bytes Packets:     ",	\
	[MACDEV_MIB_STATS_TX_127_BYTES_PACKETS]    = "Tx 127 Bytes Packets:    ",	\
	[MACDEV_MIB_STATS_TX_255_BYTES_PACKETS]    = "Tx 255 Bytes Packets:    ",	\
	[MACDEV_MIB_STATS_TX_511_BYTES_PACKETS]    = "Tx 511 Bytes Packets:    ",	\
	[MACDEV_MIB_STATS_TX_1023_BYTES_PACKETS]   = "Tx 1023 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_TX_1518_BYTES_PACKETS]   = "Tx 1518 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_TX_1522_BYTES_PACKETS]   = "Tx 1522 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_TX_2047_BYTES_PACKETS]   = "Tx 2047 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_TX_4095_BYTES_PACKETS]   = "Tx 4095 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_TX_9216_BYTES_PACKETS]   = "Tx 9216 Bytes Packets:   ",	\
	[MACDEV_MIB_STATS_TX_16383_BYTES_PACKETS]  = "Tx 16383 Bytes Packets:  ",	\
	[MACDEV_MIB_STATS_TX_GOOD_PACKETS]         = "Tx Good Packets:         ",	\
	[MACDEV_MIB_STATS_TX_PACKETS]              = "Tx Packets:              ",	\
	[MACDEV_MIB_STATS_TX_UNICAST_PACKETS]      = "Tx Unicast Packets:      ",	\
	[MACDEV_MIB_STATS_TX_MULTICAST_PACKETS]    = "Tx Multicast Packets:    ",	\
	[MACDEV_MIB_STATS_TX_BROADCAST_PACKETS]    = "Tx Broadcast Packets:    ",	\
	[MACDEV_MIB_STATS_TX_PAUSE_PACKETS]        = "Tx Pause Packets:        ",	\
	[MACDEV_MIB_STATS_TX_JABBER_ERRORS]        = "Tx Jabber Errors:        ",	\
	[MACDEV_MIB_STATS_TX_FCS_ERRORS]           = "Tx FCS Errors:           ",	\
	[MACDEV_MIB_STATS_TX_CONTROL_PACKETS]      = "Tx Control Packets:      ",	\
	[MACDEV_MIB_STATS_TX_OVERSIZED_ERRORS]     = "Tx Oversized Errors:     ",	\
	[MACDEV_MIB_STATS_TX_SINGLE_DEFERRALS]     = "Tx Single Deferrals:     ",	\
	[MACDEV_MIB_STATS_TX_MULTIPLE_DEFERRALS]   = "Tx Multiple Deferrals:   ",	\
	[MACDEV_MIB_STATS_TX_SINGLE_COLLISIONS]    = "Tx Single Collisions:    ",	\
	[MACDEV_MIB_STATS_TX_MULTIPLE_COLLISIONS]  = "Tx Multiple Collisions:  ",	\
	[MACDEV_MIB_STATS_TX_LATE_COLLISIONS]      = "Tx Late Collisions:      ",	\
	[MACDEV_MIB_STATS_TX_EXCESSIVE_COLLISIONS] = "Tx Excessive Collisions: ",	\
	[MACDEV_MIB_STATS_TX_FRAGMENT_ERRORS]      = "Tx Fragment Errors:      ",	\
	[MACDEV_MIB_STATS_TX_SYSTEM_ERRORS]        = "Tx System Errors:        ",	\
	[MACDEV_MIB_STATS_TX_SINGLE_VLAN_PACKETS]  = "Tx Single VLAN Packets:  ",	\
	[MACDEV_MIB_STATS_TX_DOUBLE_VLAN_PACKETS]  = "Tx Double VLAN Packets:  ",	\
	[MACDEV_MIB_STATS_TX_RUNT_PACKETS]         = "Tx RUNT Packets:         ",	\
	[MACDEV_MIB_STATS_TX_UNDERRUN_ERRORS]      = "Tx Underrun Packets:     ",	\
	[MACDEV_MIB_STATS_TX_TOTAL_COLLISIONS]     = "Tx Total Collisions:     ",	\
	[MACDEV_MIB_STATS_TX_BYTES]                = "Tx Bytes:                ",	\
	[MACDEV_MIB_STATS_TX_ERRORS]               = "Tx Errors:               ",	\
}

enum macdev_mib_stats {
	MACDEV_MIB_STATS_RX_64_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_127_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_255_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_511_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_1023_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_1518_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_1522_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_2047_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_4095_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_9216_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_16383_BYTES_PACKETS,
	MACDEV_MIB_STATS_RX_PACKETS,
	MACDEV_MIB_STATS_RX_UNICAST_PACKETS,
	MACDEV_MIB_STATS_RX_MULTICAST_PACKETS,
	MACDEV_MIB_STATS_RX_BROADCAST_PACKETS,
	MACDEV_MIB_STATS_RX_FCS_ERRORS,
	MACDEV_MIB_STATS_RX_CONTROL_PACKETS,
	MACDEV_MIB_STATS_RX_PAUSE_PACKETS,
	MACDEV_MIB_STATS_RX_OPCODE_ERRORS,
	MACDEV_MIB_STATS_RX_ALIGNMENT_ERRORS,
	MACDEV_MIB_STATS_RX_LENGTH_ERRORS,
	MACDEV_MIB_STATS_RX_CODE_ERRORS,
	MACDEV_MIB_STATS_RX_CARRIER_ERRORS,
	MACDEV_MIB_STATS_RX_OVERSIZED_ERRORS,
	MACDEV_MIB_STATS_RX_JABBER_ERRORS,
	MACDEV_MIB_STATS_RX_MTU_ERRORS,
	MACDEV_MIB_STATS_RX_MATCHED_CRC_PACKETS,
	MACDEV_MIB_STATS_RX_SINGLE_VLAN_PACKETS,
	MACDEV_MIB_STATS_RX_DOUBLE_VLAN_PACKETS,
	MACDEV_MIB_STATS_RX_TRUNCATED_ERRORS,
	MACDEV_MIB_STATS_RX_GOOD_PACKETS,
	MACDEV_MIB_STATS_RX_BYTES,
	MACDEV_MIB_STATS_RX_RUNT_PACKETS,
	MACDEV_MIB_STATS_RX_UNDERSIZED_ERRORS,
	MACDEV_MIB_STATS_RX_FRAGMENT_ERRORS,
	MACDEV_MIB_STATS_RX_RUNT_BYTES,
	MACDEV_MIB_STATS_RX_ERRORS,
	MACDEV_MIB_STATS_TX_CAPACITY,
	MACDEV_MIB_STATS_TX_64_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_127_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_255_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_511_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_1023_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_1518_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_1522_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_2047_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_4095_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_9216_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_16383_BYTES_PACKETS,
	MACDEV_MIB_STATS_TX_GOOD_PACKETS,
	MACDEV_MIB_STATS_TX_PACKETS,
	MACDEV_MIB_STATS_TX_UNICAST_PACKETS,
	MACDEV_MIB_STATS_TX_MULTICAST_PACKETS,
	MACDEV_MIB_STATS_TX_BROADCAST_PACKETS,
	MACDEV_MIB_STATS_TX_PAUSE_PACKETS,
	MACDEV_MIB_STATS_TX_JABBER_ERRORS,
	MACDEV_MIB_STATS_TX_FCS_ERRORS,
	MACDEV_MIB_STATS_TX_CONTROL_PACKETS,
	MACDEV_MIB_STATS_TX_OVERSIZED_ERRORS,
	MACDEV_MIB_STATS_TX_SINGLE_DEFERRALS,
	MACDEV_MIB_STATS_TX_MULTIPLE_DEFERRALS,
	MACDEV_MIB_STATS_TX_SINGLE_COLLISIONS,
	MACDEV_MIB_STATS_TX_MULTIPLE_COLLISIONS,
	MACDEV_MIB_STATS_TX_LATE_COLLISIONS,
	MACDEV_MIB_STATS_TX_EXCESSIVE_COLLISIONS,
	MACDEV_MIB_STATS_TX_FRAGMENT_ERRORS,
	MACDEV_MIB_STATS_TX_SYSTEM_ERRORS,
	MACDEV_MIB_STATS_TX_SINGLE_VLAN_PACKETS,
	MACDEV_MIB_STATS_TX_DOUBLE_VLAN_PACKETS,
	MACDEV_MIB_STATS_TX_RUNT_PACKETS,
	MACDEV_MIB_STATS_TX_UNDERRUN_ERRORS,
	MACDEV_MIB_STATS_TX_TOTAL_COLLISIONS,
	MACDEV_MIB_STATS_TX_BYTES,
	MACDEV_MIB_STATS_TX_ERRORS,
	MACDEV_MIB_STATS_MAX
};

#define phydev_lss_status_poll(p) (!p || (p->is_c45 && \
	!linkmode_test_bit(ETHTOOL_LINK_MODE_Autoneg_BIT, p->supported)))

struct mac_device_ops {
	int	(*init)(struct mac_device *macdev);
	void	(*uninit)(struct mac_device *macdev);
	int	(*open)(struct mac_device *macdev);
	int	(*stop)(struct mac_device *macdev);
	int	(*set_mac_address)(struct mac_device *macdev, void *addr);
	int	(*change_mtu)(struct mac_device *macdev, int new_mtu);
	struct rtnl_link_stats64 *(*link_stats)(struct mac_device *macdev,
			struct rtnl_link_stats64 *storage);
	void	(*get_strings)(struct mac_device *macdev, u32 stringset,
			u8 *strings);
	void	(*get_ethtool_stats)(struct mac_device *macdev,
			struct ethtool_stats *stats, u64 *data);
	int	(*get_sset_count)(struct mac_device *macdev, int stringset);
	int	(*get_eee)(struct mac_device *macdev, struct ethtool_eee *eee);
	int	(*set_eee)(struct mac_device *macdev, struct ethtool_eee *eee);
	int	(*lss_status_poll_init)(struct mac_device *macdev, bool setup);
	void	(*lss_status_poll_exit)(struct mac_device *macdev, bool flush);
	int	(*read)(struct mac_device *macdev, uint32_t addr,
			uint64_t *data);
	int	(*write)(struct mac_device *macdev, uint32_t addr,
			uint64_t data);
};

struct mii_device_ops {
	void	(*power)(struct mii_device *miidev, int power);
	int	(*init)(struct mii_device *miidev);
	void	(*uninit)(struct mii_device *miidev);
	int	(*open)(struct mii_device *miidev);
	int	(*stop)(struct mii_device *miidev);
	int	(*read)(struct mii_device *miidev, uint32_t addr,
			uint16_t *data);
	int	(*write)(struct mii_device *miidev, uint32_t addr,
			uint16_t data, uint16_t mask);
};

enum hal_link_state {
	HAL_LINK_STOP = 0,
	HAL_LINK_OPEN,
	HAL_LINK_STOP_EXIT,
	HAL_LINK_INIT_OPEN,
	HAL_LINK_STOP_IFF,
	HAL_LINK_OPEN_IFF,
	HAL_LINK_REMOVE,
	HAL_LINK_ADJUST,
};

#define HAL_LINK_REG BIT(0)
#define HAL_LINK_IFF BIT(1)
#define HAL_LINK_AFE BIT(2)
#define HAL_LINK_PHY BIT(3)
#define HAL_LINK_RUN BIT(4)
#define HAL_LINK_SPD BIT(5)

#define HAL_LINK_PHY_AFE		(HAL_LINK_AFE | HAL_LINK_PHY)
#define HAL_LINK_PHY_AFE_REG		(HAL_LINK_REG | HAL_LINK_PHY_AFE)
#define HAL_LINK_PHY_AFE_IFF		(HAL_LINK_IFF | HAL_LINK_PHY_AFE)
#define HAL_LINK_PHY_AFE_IFF_REG	(HAL_LINK_REG | HAL_LINK_PHY_AFE_IFF)

struct hal_atomic {
	struct mutex lock;
	unsigned int value;
	unsigned int flags;
};

struct hal_layers {
	struct net_device	*netdev;
	struct mac_device	*macdev;
	struct mii_device	*miidev;
	struct phy_device	*phydev;
	struct hal_atomic	power;
	struct hal_atomic	speed;
	int8_t lss_enable;
};

struct hal_device {
	union {
		const struct mac_device_ops	*macdev_ops;
		const struct mii_device_ops	*miidev_ops;
	};
	const char		*name;
	struct device		*pdd;
	struct hal_layers	*hal;
	unsigned int		size;
	unsigned int		padded;
};

struct phy_regulator {
	const char **names;
	struct regulator **regulators;
	int nr_regulators;
};

/**
 *	haldev_priv - access mac/mii device private data
 *	@dev: mac/mii hal device
 *
 * Get mac/mii device's private data
 */
static inline void *haldev_priv(const struct hal_device *dev)
{
	return (char *)dev + ALIGN(sizeof(struct hal_device), NETDEV_ALIGN);
}

/**
 *	haldev_base - get mac/mii device from private data
 *	@dev: mac/mii hal device
 *
 * Get mac/mii from it's private data
 */
static inline void *haldev_base(const void *priv)
{
	return (char *)priv - ALIGN(sizeof(struct hal_device), NETDEV_ALIGN);
}

static inline int haldev_alloc(struct device *pdd, int sizeof_priv)
{
	struct hal_device *haldev;
	size_t alloc_size;
	struct hal_device *p;

	alloc_size = sizeof(struct hal_device);
	if (sizeof_priv) {
		/* ensure 32-byte alignment of private area */
		alloc_size = ALIGN(alloc_size, NETDEV_ALIGN);
		alloc_size += sizeof_priv;
	}
	/* ensure 32-byte alignment of whole construct */
	alloc_size += NETDEV_ALIGN - 1;

	p = kzalloc(alloc_size, GFP_KERNEL | __GFP_NOWARN | __GFP_RETRY_MAYFAIL);
	if (!p)
		p = vzalloc(alloc_size);
	if (!p)
		return -ENOMEM;

	haldev = PTR_ALIGN(p, NETDEV_ALIGN);
	haldev->padded = (char *)haldev - (char *)p;
	haldev->size = alloc_size;
	haldev->pdd = pdd;
	pdd->platform_data = haldev;

	return 0;
}

static inline void haldev_free(struct device *pdd)
{
	struct hal_device *haldev;
	char *addr;

	haldev = pdd->platform_data;
	if (!haldev)
		return;

	pdd->platform_data = NULL;
	addr = (char *)haldev - haldev->padded;
	memset(addr, 0, haldev->size);
	kvfree(addr);
}

#define macdev_priv haldev_priv
#define macdev_alloc haldev_alloc
#define macdev_free haldev_free

#define miidev_priv haldev_priv
#define miidev_alloc haldev_alloc
#define miidev_free haldev_free

static inline void haldev_move(const char *compat)
{
	struct platform_device *pdev;
	struct device_node *of_node;

	for_each_compatible_node(of_node, NULL, compat) {
		if (!of_device_is_available(of_node))
			continue;

		pdev = of_find_device_by_node(of_node);
		if (pdev)
			device_move(&pdev->dev, pdev->dev.parent, DPM_ORDER_DEV_LAST);
	}
}

static inline void *haldev_find(const char *compat, char *name, int port,
		bool priv)
{
	struct platform_device *pdev;
	struct device_node *of_node;
	const char *devname;
	void *data = NULL;
	int idx, len = strlen(name);

	for_each_compatible_node(of_node, NULL, compat) {
		if (!of_device_is_available(of_node))
			continue;

		if (of_property_read_string(of_node, "dev-name",
					(const char **)&devname))
			continue;

		if (strncmp(devname, name, len))
			continue;

		if ((port >= 0) && (kstrtouint(&devname[len], 0, &idx) ||
					(idx != port)))
			continue;

		pdev = of_find_device_by_node(of_node);
		if (!pdev)
			continue;

		data = pdev->dev.platform_data;
		if (priv && (port >= 0) && data)
			data = haldev_priv(data);

		break;
	}

	return data;
}

static inline bool haldev_available(struct device_node *dev,
		const char *dev_prop, const char *dep_prop, u32 *phyad)
{
	struct device_node *net, *phy, *dep;
	u32 val;
	bool rc = false;

	for_each_node_with_property(net, dev_prop) {
		if (!of_device_is_available(net))
			continue;

		if (dev != of_parse_phandle(net, dev_prop, 0))
			continue;

		if (dep_prop) {
			dep = of_parse_phandle(net, dep_prop, 0);
			if (!dep || !of_device_is_available(dep))
				break;
		}

		phy = of_parse_phandle(net, "phy-handle", 0);
		if (!phy || !of_device_is_available(phy))
			break;

		if (!of_property_read_u32(phy, "reg", &val) && val >= PHY_MAX_ADDR)
			break;

		if (phyad)
			*phyad = val;

		rc = true;
		break;
	}

	return rc;
}

static inline int haldev_get_id(struct device *pdd, struct device_node *of_node,
		const char *dev, int max_ports, const char **devname)
{
	const char *name;
	int port, rc = 0, len = strlen(dev);

	if (!of_node)
		of_node = pdd->of_node;

	rc = of_property_read_string(of_node, "dev-name", &name);
	if (rc) {
		dev_err(pdd, "%s: Unable to retrieve name.\n", __func__);
		goto _ret_func;
	}

	if (strncmp(name, dev, len)) {
		dev_err(pdd, "%s: Invalid dev-name %s.\n", __func__, name);
		rc = -EINVAL;
		goto _ret_func;
	}

	rc = kstrtouint(&name[len], 0, &port);
	if (rc) {
		dev_err(pdd, "%s: Unable to retrieve port.\n", __func__);
		goto _ret_func;
	}

	if (port >= max_ports) {
		dev_err(pdd, "%s: %s port %d not present.\n", __func__, dev, port);
		rc = -EINVAL;
		goto _ret_func;
	}

	if (devname)
		*devname = name;

	rc = port;

_ret_func:
	return rc;
}

static inline struct hal_device *haldev_get(struct device *pdd,
		const char *name, struct hal_layers *hal)
{
	struct device_node *of_node;
	struct platform_device *pdev;
	struct hal_device *haldev = NULL;

	of_node = of_parse_phandle(pdd->of_node, name, 0);
	if (!of_node)
		goto _ret_func;

	of_node_put(of_node);
	pdev = of_find_device_by_node(of_node);
	if (!pdev)
		goto _ret_func;

	haldev = pdev->dev.platform_data;
	if (!haldev) {
		put_device(&pdev->dev);
		goto _ret_func;
	}

	haldev->hal = hal;

_ret_func:
	return haldev;
}

static inline void hal_init(struct device *pdd, struct hal_layers *hal)
{
	memset(hal, 0, sizeof(struct hal_layers));
	mutex_init(&hal->speed.lock);
	mutex_init(&hal->power.lock);
	hal->macdev = haldev_get(pdd, "mac-handle", hal);
	hal->miidev = haldev_get(pdd, "mii-handle", hal);
	hal->power.flags = HAL_LINK_REG;
}

static inline void hal_exit(struct hal_layers *hal)
{
	if (hal->miidev)
		put_device(hal->miidev->pdd);

	if (hal->macdev)
		put_device(hal->macdev->pdd);

	mutex_destroy(&hal->power.lock);
	mutex_destroy(&hal->speed.lock);
	memset(hal, 0, sizeof(struct hal_layers));
}

static inline int haldev_init(struct hal_layers *hal)
{
	struct phy_device *phydev = hal->phydev;
	struct mii_device *miidev = hal->miidev;
	struct mac_device *macdev = hal->macdev;
	int rc = 0;

	if (phydev_lss_status_poll(phydev))
		hal->lss_enable = 1;

	if (miidev && miidev->miidev_ops->init)
		rc = miidev->miidev_ops->init(miidev);

	if (macdev && macdev->macdev_ops->init)
		rc |= macdev->macdev_ops->init(macdev);

	if (macdev && macdev->macdev_ops->lss_status_poll_init && hal->lss_enable)
		rc |= macdev->macdev_ops->lss_status_poll_init(macdev, true);

	return rc;
}

static inline void haldev_uninit(struct hal_layers *hal)
{
	struct mac_device *macdev = hal->macdev;
	struct mii_device *miidev = hal->miidev;

	if (macdev && macdev->macdev_ops->lss_status_poll_exit && hal->lss_enable)
		macdev->macdev_ops->lss_status_poll_exit(macdev, true);

	if (macdev && macdev->macdev_ops->uninit)
		macdev->macdev_ops->uninit(macdev);

	if (miidev && miidev->miidev_ops->uninit)
		miidev->miidev_ops->uninit(miidev);
}

static inline int haldev_speed(struct hal_layers *hal, int value)
{
	struct mac_device *macdev = hal->macdev;
	struct mii_device *miidev = hal->miidev;
	struct hal_atomic *speed = &hal->speed;
	int rc = 0, update = 0;

	if (speed->value && value && (speed->value != value)) {
		update = value;
		value = 0;
	} else if (speed->value == value) {
		goto _ret_func;
	}

_exe_func:
	if (value) {
		if (macdev && macdev->macdev_ops->open)
			rc = macdev->macdev_ops->open(macdev);

		if (miidev && miidev->miidev_ops->open)
			rc |= miidev->miidev_ops->open(miidev);
	} else {
		if (miidev && miidev->miidev_ops->stop)
			rc = miidev->miidev_ops->stop(miidev);

		if (macdev && macdev->macdev_ops->stop)
			rc |= macdev->macdev_ops->stop(macdev);
	}

	speed->value = value;

	if (update) {
		value = update;
		update = 0;
		goto _exe_func;
	}

_ret_func:
	return rc;
}

static inline int haldev_config_speed(struct hal_layers *hal, unsigned int mask, unsigned int flags)
{
	struct mac_device *macdev;
	struct hal_atomic *speed;
	int phydev_speed, rc = 0;

	speed = &hal->speed;
	mutex_lock(&speed->lock);

	/* adjust/remove link speed */
	if (mask & HAL_LINK_SPD) {
		if (!(speed->flags & HAL_LINK_IFF))
			goto _ret_func;

		if (!(flags & HAL_LINK_SPD)) {
			rc = haldev_speed(hal, 0);
			goto _ret_func;
		}

		phydev_speed = hal->phydev ? hal->phydev->speed : SPEED_1000;
		if (phydev_speed && (phydev_speed != SPEED_UNKNOWN))
			rc = haldev_speed(hal, phydev_speed);

		goto _ret_func;
	}

	/* reg power up */
	if ((mask & HAL_LINK_REG) && (flags & HAL_LINK_REG) && (!(speed->flags & HAL_LINK_REG)))
		speed->flags |= HAL_LINK_REG;

	/* ifconfig up */
	if ((mask & HAL_LINK_IFF) && (flags & HAL_LINK_IFF) && (!(speed->flags & HAL_LINK_IFF)))
		speed->flags |= HAL_LINK_IFF;

	if (!(speed->flags & HAL_LINK_REG) || !(speed->flags & HAL_LINK_IFF))
		goto _skip_afe;

	if (!(mask & HAL_LINK_AFE))
		goto _skip_afe;

	macdev = hal->macdev;

	/* afe power up */
	if ((flags & HAL_LINK_AFE) && !(speed->flags & HAL_LINK_AFE)) {
		if (macdev && macdev->macdev_ops->lss_status_poll_init && hal->lss_enable)
			rc = macdev->macdev_ops->lss_status_poll_init(macdev, false);

		speed->flags |= HAL_LINK_AFE;
	/* afe power dn */
	} else if (!(flags & HAL_LINK_AFE) && (speed->flags & HAL_LINK_AFE)) {

		rc = haldev_speed(hal, 0);
		if (macdev && macdev->macdev_ops->lss_status_poll_exit && hal->lss_enable)
			macdev->macdev_ops->lss_status_poll_exit(macdev, false);

		speed->flags &= ~HAL_LINK_AFE;
	}

_skip_afe:
	/* ifconfig dn */
	if ((mask & HAL_LINK_IFF) && !(flags & HAL_LINK_IFF) && (speed->flags & HAL_LINK_IFF))
		speed->flags &= ~HAL_LINK_IFF;

	/* reg power dn */
	if ((mask & HAL_LINK_REG) && !(flags & HAL_LINK_REG) && (speed->flags & HAL_LINK_REG))
		speed->flags &= ~HAL_LINK_REG;

_ret_func:
	mutex_unlock(&speed->lock);

	return rc;
}

static inline int phy_regulator_power(struct mii_device *miidev, int power)
{
	struct phy_regulator *supply = miidev_priv(miidev);
	struct device *dev = (miidev->hal && miidev->hal->phydev) ? &miidev->hal->phydev->mdio.dev : NULL;
	int supplies = supply ? supply->nr_regulators : 0;
	int i, rs, rc = 0;

	for (i = 0; i < supplies; i++) {
		if (!supply->names[i])
			continue;

		if (power) {
			if (regulator_is_enabled(supply->regulators[i]))
				continue;

			rs = regulator_enable(supply->regulators[i]);
		} else {
			if (!regulator_is_enabled(supply->regulators[i]))
				continue;

			rs = regulator_disable(supply->regulators[i]);
		}

		if (rs)
			pr_err("Unable to %s %s regulator.\n", (power ? "enable" : "disable"), supply->names[i]);
		else if (dev)
			dev_info(dev, "Regulator power %s %s\n", supply->names[i], (power ? "enabled" : "disabled"));

		rc |= rs;
	}

	return rc;
}

static inline int haldev_config_power(struct hal_layers *hal, unsigned int mask, unsigned int flags)
{
	struct mii_device *miidev;
	struct phy_device *phydev;
	struct hal_atomic *power;
	int rc = 0;

	power = &hal->power;
	mutex_lock(&power->lock);

	miidev = hal->miidev;

	/* reg power up */
	if ((mask & HAL_LINK_REG) && (flags & HAL_LINK_REG) && (!(power->flags & HAL_LINK_REG))) {
		rc = phy_regulator_power(miidev, 1);
		power->flags |= HAL_LINK_REG;
	}

	/* ifconfig up */
	if ((mask & HAL_LINK_IFF) && (flags & HAL_LINK_IFF) && (!(power->flags & HAL_LINK_IFF)))
		power->flags |= HAL_LINK_IFF;

	if (!(power->flags & HAL_LINK_REG) || !(power->flags & HAL_LINK_IFF))
		goto _skip_afe_phy;

	/* afe power up */
	if ((mask & HAL_LINK_AFE) && (flags & HAL_LINK_AFE) && !(power->flags & HAL_LINK_AFE)) {
		if (miidev && miidev->miidev_ops->power)
			miidev->miidev_ops->power(miidev, 1);

		power->flags |= HAL_LINK_AFE;
	}

	phydev = hal->phydev;
	if (!phydev || !(mask & HAL_LINK_PHY))
		goto _skip_phy;

	if (flags & HAL_LINK_PHY) {
		/* phy power up and start */
		if ((((mask & HAL_LINK_REG) && (flags & HAL_LINK_REG)) ||
					((mask & HAL_LINK_IFF) && (flags & HAL_LINK_IFF))) &&
				!(power->flags & HAL_LINK_RUN)) {
			phy_start(phydev);
			power->flags |= (HAL_LINK_PHY | HAL_LINK_RUN);
			dev_info(&phydev->mdio.dev, "PHY power enabled and started\n");
		/* phy power up */
		} else if (!(power->flags & HAL_LINK_PHY) && phydev->drv && phydev->drv->resume) {
			mutex_lock(&phydev->lock);
			phydev->drv->resume(phydev);
			mutex_unlock(&phydev->lock);
			power->flags |= HAL_LINK_PHY;
			dev_info(&phydev->mdio.dev, "PHY power enabled\n");
		}
	} else {
		/* phy stop and power dn */
		if ((((mask & HAL_LINK_REG) && !(flags & HAL_LINK_REG)) ||
					((mask & HAL_LINK_IFF) && !(flags & HAL_LINK_IFF))) &&
				(power->flags & HAL_LINK_RUN)) {
			phy_stop(phydev);
			power->flags &= ~(HAL_LINK_RUN | HAL_LINK_PHY);
			dev_info(&phydev->mdio.dev, "PHY stopped and power disabled\n");
		/* phy power dn */
		} else if ((power->flags & HAL_LINK_PHY) && phydev->drv && phydev->drv->suspend) {
			phydev->drv->suspend(phydev);
			power->flags &= ~HAL_LINK_PHY;
			dev_info(&phydev->mdio.dev, "PHY power disabled\n");
		}
	}

_skip_phy:
	/* afe power dn */
	if ((mask & HAL_LINK_AFE) && !(flags & HAL_LINK_AFE) && (power->flags & HAL_LINK_AFE)) {
		if (miidev && miidev->miidev_ops->power)
			miidev->miidev_ops->power(miidev, 0);

		power->flags &= ~HAL_LINK_AFE;
	}

_skip_afe_phy:
	/* ifconfig dn */
	if ((mask & HAL_LINK_IFF) && !(flags & HAL_LINK_IFF) && (power->flags & HAL_LINK_IFF))
		power->flags &= ~HAL_LINK_IFF;

	/* reg power dn */
	if ((mask & HAL_LINK_REG) && !(flags & HAL_LINK_REG) && (power->flags & HAL_LINK_REG)) {
		rc |= phy_regulator_power(miidev, 0);
		power->flags &= ~HAL_LINK_REG;
	}

	mutex_unlock(&power->lock);

	return rc;
}

static inline int haldev_config_link(struct hal_layers *hal, enum hal_link_state state)
{
	int rc = 0;

	switch (state) {
	case HAL_LINK_STOP:
		rc = haldev_config_power(hal, HAL_LINK_PHY_AFE, 0);
		rc |= haldev_config_speed(hal, HAL_LINK_PHY_AFE, 0);
		break;

	case HAL_LINK_OPEN:
		rc = haldev_config_speed(hal, HAL_LINK_PHY_AFE, HAL_LINK_PHY_AFE);
		rc |= haldev_config_power(hal, HAL_LINK_PHY_AFE, HAL_LINK_PHY_AFE);
		break;

	case HAL_LINK_STOP_EXIT:
		rc = haldev_config_power(hal, HAL_LINK_PHY_AFE_REG, 0);
		rc |= haldev_config_speed(hal, HAL_LINK_PHY_AFE_REG, 0);
		break;

	case HAL_LINK_INIT_OPEN:
		rc = haldev_config_speed(hal, HAL_LINK_PHY_AFE_REG, HAL_LINK_PHY_AFE_REG);
		rc |= haldev_config_power(hal, HAL_LINK_PHY_AFE_REG, HAL_LINK_PHY_AFE_REG);
		break;

	case HAL_LINK_STOP_IFF:
		rc = haldev_config_power(hal, HAL_LINK_PHY_AFE_IFF, 0);
		rc |= haldev_config_speed(hal, HAL_LINK_PHY_AFE_IFF, 0);
		break;

	case HAL_LINK_OPEN_IFF:
		rc = haldev_config_speed(hal, HAL_LINK_PHY_AFE_IFF_REG, HAL_LINK_PHY_AFE_IFF_REG);
		rc |= haldev_config_power(hal, HAL_LINK_PHY_AFE_IFF_REG, HAL_LINK_PHY_AFE_IFF_REG);
		break;

	case HAL_LINK_REMOVE:
		rc = haldev_config_speed(hal, HAL_LINK_SPD, 0);
		break;

	case HAL_LINK_ADJUST:
		rc = haldev_config_speed(hal, HAL_LINK_SPD, HAL_LINK_SPD);
		break;
	}

	return rc;
}

static inline int haldev_open(struct hal_layers *hal)
{
	return haldev_config_link(hal, HAL_LINK_OPEN_IFF);
}

static inline int haldev_stop(struct hal_layers *hal)
{
	return haldev_config_link(hal, HAL_LINK_STOP_IFF);
}

static inline void haldev_adjust_link(struct hal_layers *hal)
{
	if (hal->netdev->flags & IFF_UP)
		haldev_config_link(hal, HAL_LINK_ADJUST);
}

static inline void haldev_remove_link(struct hal_layers *hal)
{
	if ((hal->netdev->flags & IFF_UP) && !hal->lss_enable)
		haldev_config_link(hal, HAL_LINK_REMOVE);
}

static inline int haldev_set_mac_address(struct hal_layers *hal, void *addr)
{
	struct mac_device *macdev = hal->macdev;
	int rc = 0;

	if (macdev && macdev->macdev_ops->set_mac_address)
		rc = macdev->macdev_ops->set_mac_address(macdev, addr);

	return rc;
}

static inline int haldev_change_mtu(struct hal_layers *hal, int new_mtu)
{
	struct mac_device *macdev = hal->macdev;
	int rc = 0;

	if (macdev && macdev->macdev_ops->change_mtu)
		rc = macdev->macdev_ops->change_mtu(macdev, new_mtu);

	return rc;
}

static inline struct rtnl_link_stats64 *haldev_link_stats(struct hal_layers *hal,
		struct rtnl_link_stats64 *storage)
{
	struct mac_device *macdev = hal->macdev;

	return macdev->macdev_ops->link_stats(macdev, storage);
}

static inline void haldev_get_strings(struct hal_layers *hal, u32 stringset,
		u8 *strings)
{
	struct mac_device *macdev = hal->macdev;

	if (macdev && macdev->macdev_ops->get_strings)
		macdev->macdev_ops->get_strings(macdev, stringset, strings);
}

static inline void haldev_get_ethtool_stats(struct hal_layers *hal,
		struct ethtool_stats *stats, u64 *data)
{
	struct mac_device *macdev = hal->macdev;

	if (macdev && macdev->macdev_ops->get_ethtool_stats)
		macdev->macdev_ops->get_ethtool_stats(macdev, stats, data);
}

static inline int haldev_get_sset_count(struct hal_layers *hal, int stringset)
{
	struct mac_device *macdev = hal->macdev;
	int rc = 0;

	if (macdev && macdev->macdev_ops->get_sset_count)
		rc = macdev->macdev_ops->get_sset_count(macdev, stringset);

	return rc;
}

static inline int haldev_get_eee(struct hal_layers *hal, struct ethtool_eee *eee)
{
	struct mac_device *macdev = hal->macdev;
	int rc = 0;

	if (macdev && macdev->macdev_ops->get_eee)
		rc = macdev->macdev_ops->get_eee(macdev, eee);

	return rc;
}

static inline int haldev_set_eee(struct hal_layers *hal, struct ethtool_eee *eee)
{
	struct mac_device *macdev = hal->macdev;
	int rc = 0;

	if (macdev && macdev->macdev_ops->set_eee)
		rc = macdev->macdev_ops->set_eee(macdev, eee);

	return rc;
}

static inline int phy_regulator_probe(struct mii_device *miidev)
{
	struct device *dev = miidev->pdd;
	struct phy_regulator *supply = miidev_priv(miidev);
	const char *name;
	int i, supplies;

	supplies = of_property_count_strings(dev->of_node, "supply-names");
	if (supplies <= 0)
		return 0;

	supply->names = devm_kcalloc(dev, supplies, sizeof(char *), GFP_KERNEL);
	if (!supply->names) {
		dev_err(dev, "Failed to allocate memory for names\n");
		return -ENOMEM;
	}

	supply->regulators = devm_kcalloc(dev, supplies, sizeof(struct regulator *), GFP_KERNEL);
	if (!supply->regulators) {
		dev_err(dev, "Failed to allocate memory for regulators\n");
		return -ENOMEM;
	}

	supply->nr_regulators = supplies;
	memset(supply->names, 0, supplies * sizeof(char *));
	memset(supply->regulators, 0, supplies * sizeof(struct regulator *));

	for (i = 0; i < supplies; i++) {
		if (of_property_read_string_index(dev->of_node, "supply-names", i,
						  &name))
			continue;

		supply->names[i] = name;
		supply->regulators[i] = devm_regulator_get(dev, name);
		if (IS_ERR(supply->regulators[i])) {
			dev_err(dev, "Unable to get %s supply, err=%d\n",
				name, (int)PTR_ERR(supply->regulators[i]));
			supply->names[i] = NULL;
			supply->regulators[i] = NULL;
			continue;
		}
	}

	return phy_regulator_power(miidev, 1);
}

static inline void phy_regulator_remove(struct mii_device *miidev)
{
	struct device *dev = miidev->pdd;
	struct phy_regulator *supply = miidev_priv(miidev);
	int supplies = supply ? supply->nr_regulators : 0;

	if (!supplies)
		return;

	phy_regulator_power(miidev, 0);
	memset(supply->regulators, 0, supplies * sizeof(struct regulator *));
	memset(supply->names, 0, supplies * sizeof(char *));
	devm_kfree(dev, supply->regulators);
	devm_kfree(dev, supply->names);
	memset(supply, 0, sizeof(struct phy_regulator));
}

#endif /* _HAL_DEVICE_H_ */
