/*
 * Driver for Broadcom RPC BA Service
 *
 * Copyright (C) 2022 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/types.h>
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <linux/notifier.h>
#include <linux/of.h>
#include <linux/reboot.h>
#include <linux/delay.h>
#include <linux/cdev.h>
#include <proc_cmd.h>
#include <brcm_ba_rpc.h>
#if defined(CONFIG_BCM_PWR_RPC)
#include <brcm_pwr_rpc.h>
#endif

#define MODULE_NAME "brcm-ba-rpc"
#define BA_NAME_MAX_LEN 9

#define BA_RPC_OF_MATCH "brcm,brcm-ba-rpc"
#define ANSI_RED	"\e[31;1m"
#define ANSI_YELLOW	"\e[33;1m"
#define ANSI_RESET	"\e[0m"
#define RED(str)	ANSI_RED str ANSI_RESET
#define YLW(str)	ANSI_YELLOW str ANSI_RESET

#define BA_RPC_CLASS	"RPC boot assist"
#define BA_RPC_DEVNAME	"ba-rpc"
#define BA_RPC_MAX_DEVS	1
#define BA_RPC_RG_SMC_TUNNEL_NAME "rg-smc"
#define RG_BATT_MODE_PROPERTY "rg-batt-mode"
#define RG_BATT_MODE_OFF "off"
#define RG_BATT_MODE_CPU_ON "cpu-on"

/* Define Run state hand shake mutex */
static DEFINE_MUTEX(runstate_hs_mutex);
static bool rs_req_hs_flag = false;

struct fasync_struct	*ba_rpc_sigio_list;
#if defined(CONFIG_BCM_PWR_RPC)
/* work bottom half */
struct work_struct work;
#endif

struct ba_rpc_dev_info {
	struct platform_device *pdev;
	struct class	*ba_rpc_class;
	dev_t			ba_rpc_dev;
	struct cdev		ba_rpc_cdev;
	struct device	*ba_rpc_device;
};

enum ba_rpc_func {
	GET_CPU_ID,
	GET_CPU_NAME,
	GET_RUN_STATE_ID,
	GET_RUN_STATE_NAME,
	GET_RUN_STATE,
	NOTIFY_RUN_STATE,
	REQUEST_RUN_STATE,
	REQUEST_RUN_STATE_RESPONSE,
	SET_RUN_STATE,
	GET_BOOT_CFG,
	SET_BOOT_CFG,
	PREPARE_RUN_STATE,
	PREPARE_RUN_STATE_RESPONSE,
	BA_MAX_FUNC
};

static enum rg_battery_mode rg_batt_mode = RG_CPU_OFF_UTILITY;

#define BA_RPC_RS_INIT(index, rs_name) rs_name,

const char *ba_rpc_rs_name[BA_RPC_RS_MAX_STATE] = {
	BA_RPC_RS_LIST
};

static char ba_rpc_rs_state[BA_NAME_MAX_LEN] = {0};
static char ba_rpc_cpu_name[BA_NAME_MAX_LEN] = {0};

static int ba_rpc_rs_id[BA_RPC_RS_MAX_STATE] = {-1};
static int ba_rpc_rg_cpu_id = -1;
static int ba_rpc_all_cpu_id = -1;
static int ba_rpc_tunnel = -1;
static bool kernel_core_req = false;
static bool kernel_emerg_restart = false;

struct ba_msg {
	uint32_t	hdr;
	union {
		uint32_t	rsvd0;
		struct {
			uint8_t	cpu_id:8;
			union {
				uint8_t	rs_id:8;

				uint8_t ac_en:1;
				uint8_t batt_en:1;
				uint8_t rsvd4:6;
			};
			union {
				uint8_t	rude:1;
				uint8_t	rsvd1:7;

				uint8_t	response:4;
				uint8_t rsvd2:4;
			};
			uint8_t	rc:8;
		};
	};
	union {
		uint32_t	rsvd3[2];
		char		name[8];
	};
};

enum {
	BA_CMD_NULL = 0,
	BA_CMD_NOTIFY_RS,
	BA_CMD_GET_RS,
	BA_CMD_GET_REQ_RS,
	BA_CMD_REQ_RS
};

struct ba_rs_info {
	int command;
	int state;
	char cpu_name[BA_NAME_MAX_LEN];
	char rs_name[BA_NAME_MAX_LEN];
	char req_rs_name[BA_NAME_MAX_LEN];
};

static struct ba_rs_info rs_info;
static int tunnel_id = -1;
static char reqested_rs_name[BA_NAME_MAX_LEN];

static inline void ba_msg_set_batt_en(rpc_msg *msg, bool v)
{
	msg->data[0] = (msg->data[0] & ~0x200) | v<<9;
}

static inline void ba_msg_set_ac_en(rpc_msg *msg, bool v)
{
	msg->data[0] = (msg->data[0] & ~0x100) | v<<8;
}

static void *ba_rs_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (!*pos)
		return SEQ_START_TOKEN;
	return 0;
}

static void *ba_rs_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	(*pos)++;
	return 0;
}

static void ba_rs_seq_stop(struct seq_file *seq, void *v)
{
}

static int ba_rs_seq_show(struct seq_file *seq, void *v)
{
	switch (rs_info.command) {
	case BA_CMD_NOTIFY_RS:
		seq_printf(seq,"Notified state %s for CPU %s\n",
			ba_rpc_rs_name[rs_info.state], rs_info.cpu_name);
		break;
	case BA_CMD_GET_RS:
		seq_printf(seq,"%s\n", rs_info.rs_name);
		break;
	case BA_CMD_GET_REQ_RS:
		seq_printf(seq,"%s\n", rs_info.req_rs_name);
		break;

	default:
		break;
	}

	return 0;
}

static const struct seq_operations ba_rs_seq_ops = {
	.start	= ba_rs_seq_start,
	.next	= ba_rs_seq_next,
	.stop	= ba_rs_seq_stop,
	.show	= ba_rs_seq_show,
};

static int ba_get_cpu_id(int tunnel, const char *cpu_name, uint8_t *cpu_id)
{
	struct ba_msg ba_msg;
	rpc_msg *msg = (rpc_msg *)&ba_msg;
	int rc = 0;

	if (!cpu_name || (strlen(cpu_name) == 0)) {
		pr_err(RED("%s : Invalid CPU name (%s)\n"),
			   MODULE_NAME, cpu_name);
		rc = -EINVAL;
		goto done;
	}

	rpc_msg_init(msg, RPC_SERVICE_BA, GET_CPU_ID, 0, 0, 0, 0);
	strncpy(ba_msg.name, cpu_name, sizeof(ba_msg.name));

	rc = rpc_send_request(tunnel, msg);
	if (rc) {
		pr_err(RED("%s : GET_CPU_ID failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}
	if (ba_msg.rc == 0xff) {
		pr_err(RED("%s : invalid return code(%d) for CPU(%s)\n"),
			   MODULE_NAME, ba_msg.rc, cpu_name);
		rpc_dump_msg(msg);
		rc = -EINVAL;
		goto done;
	}
	*cpu_id = ba_msg.cpu_id;

done:
	return rc;
}

static int ba_get_run_state_id(int tunnel, const char *rs_name, uint8_t *rs_id)
{
	struct ba_msg ba_msg;
	rpc_msg *msg = (rpc_msg *)&ba_msg;
	int rc = 0;

	if (!rs_name || (strlen(rs_name) == 0)) {
		pr_err(RED("%s : Invalid RS name (%s)\n"),
			   MODULE_NAME, rs_name);
		rc = -EINVAL;
		goto done;
	}

	rpc_msg_init(msg, RPC_SERVICE_BA, GET_RUN_STATE_ID, 0, 0, 0, 0);
	strncpy(ba_msg.name, rs_name, sizeof(ba_msg.name));

	rc = rpc_send_request(tunnel, msg);
	if (rc) {
		pr_err(RED("%s : GET_RUN_STATE_ID failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}
	if (ba_msg.rc == 0xff) {
		pr_err(RED("%s : invalid return code(%d) for RS name(%s)\n"),
			   MODULE_NAME, ba_msg.rc, rs_name);
		rpc_dump_msg(msg);
		rc = -EINVAL;
		goto done;
	}
	*rs_id = ba_msg.rs_id;

done:
	return rc;
}

static int ba_get_cpu_run_state(int tunnel, uint8_t cpu_id, uint8_t *rs_id)
{
	struct ba_msg ba_msg;
	rpc_msg *msg = (rpc_msg *)&ba_msg;
	int rc = 0;

	rpc_msg_init(msg, RPC_SERVICE_BA, GET_RUN_STATE, 0, 0, 0, 0);
	ba_msg.cpu_id = cpu_id;

	rc = rpc_send_request(tunnel, msg);
	if (rc) {
		pr_err(RED("%s : GET_RUN_STATE failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}
	if (ba_msg.rc == 0xff) {
		pr_err(RED("%s : invalid return code(%d) for CPU(%d)\n"),
			   MODULE_NAME, ba_msg.rc, cpu_id);
		rpc_dump_msg(msg);
		rc = -EINVAL;
		goto done;
	}
	*rs_id = ba_msg.rs_id;

done:
	return rc;
}

static int ba_get_cpu_run_state_name(int tunnel, char *name, uint8_t rs_id)
{
	struct ba_msg ba_msg;
	rpc_msg *msg = (rpc_msg *)&ba_msg;
	int rc = 0;

	rpc_msg_init(msg, RPC_SERVICE_BA, GET_RUN_STATE_NAME, 0, 0, 0, 0);
	ba_msg.rs_id = (uint8_t)rs_id;

	rc = rpc_send_request(tunnel, msg);
	if (rc) {
		pr_err(RED("%s : GET_RUN_STATE_NAME failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}
	if (ba_msg.rc == 0xff) {
		pr_err(RED("%s : invalid return code(%d) for state id(%d)\n"),
			   MODULE_NAME, ba_msg.rc, rs_id);
		rpc_dump_msg(msg);
		rc = -EINVAL;
		goto done;
	}
	strncpy(name, ba_msg.name, sizeof(ba_msg.name));

done:
	return rc;
}

static int ba_notify_cpu_run_state(int tunnel, uint8_t cpu_id, uint8_t rs_id)
{
	struct ba_msg ba_msg;
	rpc_msg *msg = (rpc_msg *)&ba_msg;
	int rc = 0;

	rpc_msg_init(msg, RPC_SERVICE_BA, NOTIFY_RUN_STATE, 0, 0, 0, 0);
	ba_msg.rs_id = rs_id;
	ba_msg.cpu_id = cpu_id;

	rc = rpc_send_message(tunnel, msg, true);
	if (rc) {
		pr_err(RED("%s : NOTIFY_RUN_STATE failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}

done:
	return rc;
}

static int ba_req_cpu_run_state(int tunnel, uint8_t cpu_id,
			uint8_t rs_id, bool rude)
{
	struct ba_msg ba_msg;
	rpc_msg *msg = (rpc_msg *)&ba_msg;
	int rc = 0;

	rpc_msg_init(msg, RPC_SERVICE_BA, REQUEST_RUN_STATE, 0, 0, 0, 0);
	ba_msg.rude = rude;
	ba_msg.rs_id = rs_id;
	ba_msg.cpu_id = cpu_id;

	rc = rpc_send_message(tunnel, msg, !(rude));
	if (rc) {
		pr_err(RED("%s : REQUEST_RUN_STATE failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}

done:
	return rc;
}

static int ba_set_cpu_run_state_cb(int tunnel, rpc_msg *msg)
{
	struct ba_msg *ba_msg = (struct ba_msg *)msg;
	/* Notify back to SMC that you completed the handshake */
	ba_notify_cpu_run_state(ba_rpc_tunnel, (uint8_t)ba_rpc_rg_cpu_id,
							(uint8_t)ba_rpc_rs_id[RUNSTATE_BAUP]);
	rs_req_hs_flag = true;
	mutex_unlock(&runstate_hs_mutex);

	if (ba_msg->rs_id == (uint8_t)ba_rpc_rs_id[RUNSTATE_SHUTDOWN] ||
		ba_msg->rs_id == (uint8_t)ba_rpc_rs_id[RUNSTATE_BOOT]) {
		/* Handle the state change */
		if ((!strncmp(reqested_rs_name, BA_RPC_RS_BOOT, strlen(BA_RPC_RS_BOOT))) ||
				 (ba_msg->rs_id == (uint8_t)ba_rpc_rs_id[RUNSTATE_BOOT])) {
			if (kernel_emerg_restart)
				brcm_rpc_ba_rg_emergency_restart(false, kernel_core_req);
		}
	}

	return 0;
}

static int ba_prepare_run_state_response(int tunnel, rpc_msg *msg)
{
	int rc = 0;
	struct ba_msg *ba_msg = (struct ba_msg *)msg;
	uint8_t cpu_id, rs_id;

	rs_id = ba_msg->rs_id;
	cpu_id = ba_msg->cpu_id;

	rpc_msg_init(msg, RPC_SERVICE_BA, PREPARE_RUN_STATE_RESPONSE, 0, 0, 0, 0);
	ba_msg->rs_id = rs_id;
	ba_msg->cpu_id = cpu_id;
	ba_msg->response = 0; //Ready

	rc = rpc_send_message(tunnel, msg, false);
	if (rc) {
		pr_err(RED("%s : Req run state response failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}

done:
	return rc;
}

static int ba_prepare_run_state_cb(int tunnel, rpc_msg *msg)
{
	int rc = 0;

	/* Do the necessory steps here */

	/* Respond back to SMC that you are performing the RUN state change */
	rc = ba_prepare_run_state_response(tunnel, msg);

	if (rc)
		mutex_unlock(&runstate_hs_mutex);

	return rc;
}

static int ba_req_run_state_response(int tunnel, rpc_msg *msg)
{
	int rc = 0;
	struct ba_msg *ba_msg = (struct ba_msg *)msg;
	uint8_t cpu_id, rs_id;

	rs_id = ba_msg->rs_id;
	cpu_id = ba_msg->cpu_id;

	rpc_msg_init(msg, RPC_SERVICE_BA, REQUEST_RUN_STATE_RESPONSE, 0, 0, 0, 0);
	ba_msg->rs_id = rs_id;
	ba_msg->cpu_id = cpu_id;
	ba_msg->response = 0;

	rc = rpc_send_message(tunnel, msg, false);
	if (rc) {
		pr_err(RED("%s : Req run state response failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}

done:
	return rc;
}

static int ba_req_cpu_run_state_cb(int tunnel, rpc_msg *msg)
{
	int rc = 0;

	rc = ba_req_run_state_response(tunnel, msg);

	if (rc)
		mutex_unlock(&runstate_hs_mutex);
	return rc;
}

static int ba_notify_cpu_run_state_cb(int tunnel, rpc_msg *msg)
{
	/* TODO: handle GFAP run state notification changes here */

	return 0;
}

static int ba_set_boot_cfg(int tunnel, uint8_t cpu_id)
{
	struct ba_msg ba_msg;
	rpc_msg *msg = (rpc_msg *)&ba_msg;
	int rc = 0;

	rpc_msg_init(msg, RPC_SERVICE_BA, SET_BOOT_CFG, 0, 0, 0, 0);

	ba_msg.cpu_id = cpu_id;
	ba_msg_set_ac_en(msg, true);
	if (rg_batt_mode == RG_CPU_OFF_UTILITY || rg_batt_mode == RG_CPU_OFF_BATTERY)
		ba_msg_set_batt_en(msg, false);
	else
		ba_msg_set_batt_en(msg, true);

	rc = rpc_send_message(tunnel, msg, false);
	if (rc) {
		pr_err(RED("%s : SET_BOOT_CFG failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(msg);
		goto done;
	}

done:
	return rc;
}

static int brcm_rpc_ba_get_fifo_tunnel_id(char *name)
{
	if (tunnel_id < 0) {
		tunnel_id = rpc_get_fifo_tunnel_id(name);
		if (tunnel_id < 0) {
			pr_err("%s : Error: invalid tunnel %s\n",
					MODULE_NAME, name);
		}
	}

	return tunnel_id;
}

static int brcm_rpc_ba_get_cpu_id(int tunnel, const char *name, uint8_t *cpu_id)
{
	int rc = 0;
	int id = -1;

	if (!name)
		return -EINVAL;

	if (!strncmp(name, BA_RPC_ALL_CPU_NAME,
				 strlen(BA_RPC_ALL_CPU_NAME)))
		id = ba_rpc_all_cpu_id;
	else if (!strncmp(name, BA_RPC_RG_CPU_NAME,
				 strlen(BA_RPC_RG_CPU_NAME)))
		id = ba_rpc_rg_cpu_id;

	if (id < 0) {
		rc = ba_get_cpu_id(tunnel, name, cpu_id);
		if (rc) {
			pr_err("%s : BA get CPU id failure for CPU(%s)\n",
				   MODULE_NAME, name);
		}
		else {
			if (!strncmp(name, BA_RPC_ALL_CPU_NAME,
						 strlen(BA_RPC_ALL_CPU_NAME)))
				ba_rpc_all_cpu_id = *cpu_id;
			else if (!strncmp(name, BA_RPC_RG_CPU_NAME,
							  strlen(BA_RPC_RG_CPU_NAME)))
				ba_rpc_rg_cpu_id = *cpu_id;
		}
	}
	else
		*cpu_id = (uint8_t)id;

	return rc;
}

static int brcm_rpc_ba_get_run_state_id(int tunnel, const char *name, uint8_t *rs_id)
{
	int rc = 0, rs_cnt = 0;

	if (!name)
		return -EINVAL;

	for (rs_cnt = 0; rs_cnt < BA_RPC_RS_MAX_STATE; rs_cnt++) {
		if (!strncmp(ba_rpc_rs_name[rs_cnt], name,
					 strlen(ba_rpc_rs_name[rs_cnt]))) {
			if (ba_rpc_rs_id[rs_cnt] >= 0) {
				*rs_id = (uint8_t)ba_rpc_rs_id[rs_cnt];
				return rc;
			}
		}
	}
	rc = ba_get_run_state_id(tunnel, name, rs_id);
	if (rc) {
		pr_err("%s : BA get CPU run state ID failure for CPU(%s)\n",
			MODULE_NAME, BA_RPC_RG_CPU_NAME);
	}

	return rc;
}

int bcm_rpc_ba_notify_cpu_run_state(const char *rs_name)
{
	int tunnel = 0, rc = 0;
	uint8_t cpu_id = 0, rs_id = 0;

	if (!rs_name)
		return -EINVAL;

	tunnel = brcm_rpc_ba_get_fifo_tunnel_id(BA_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		pr_err("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, BA_RPC_RG_SMC_TUNNEL_NAME);
		rc = tunnel;
		goto done;
	}

	rc = brcm_rpc_ba_get_cpu_id(tunnel, BA_RPC_RG_CPU_NAME, &cpu_id);
	if (rc) {
		pr_err("%s : BA get CPU id failure for CPU(%s)\n",
			MODULE_NAME, BA_RPC_RG_CPU_NAME);
		goto done;
	}
	rc = brcm_rpc_ba_get_run_state_id(tunnel, rs_name, &rs_id);
	if (rc) {
		pr_err("%s : BA get CPU run state ID failure for CPU(%s)\n",
			MODULE_NAME, BA_RPC_RG_CPU_NAME);
		goto done;
	}
	rc = ba_notify_cpu_run_state(tunnel, cpu_id, rs_id);
	if (rc) {
		pr_err("%s : BA notify CPU run state(%s) failure for id(%d)\n",
			MODULE_NAME, rs_name, rs_id);
		goto done;
	}
	strncpy(ba_rpc_rs_state, rs_name, (BA_NAME_MAX_LEN-1));
	strncpy(ba_rpc_cpu_name, BA_RPC_RG_CPU_NAME, (BA_NAME_MAX_LEN-1));

done:
	return rc;
}
EXPORT_SYMBOL(bcm_rpc_ba_notify_cpu_run_state);

static int bcm_rpc_ba_get_cpu_run_state(char *rs_name, char *cpu_name)
{
	int tunnel = 0, rc = 0;
	uint8_t cpu_id = 0, rs_id = 0;

	tunnel = brcm_rpc_ba_get_fifo_tunnel_id(BA_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		pr_err("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, BA_RPC_RG_SMC_TUNNEL_NAME);
		rc = tunnel;
		goto done;
	}

	rc = brcm_rpc_ba_get_cpu_id(tunnel, cpu_name, &cpu_id);
	if (rc) {
		pr_err("%s : BA get CPU id failure for CPU(%s)\n",
			MODULE_NAME, cpu_name);
		goto done;
	}
	rc = ba_get_cpu_run_state(tunnel, cpu_id, &rs_id);
	if (rc) {
		pr_err("%s : BA get CPU run state failure for CPU(%s)\n",
			MODULE_NAME, cpu_name);
		goto done;
	}
	rc = ba_get_cpu_run_state_name(tunnel, rs_name, rs_id);
	if (rc) {
		pr_err("%s : BA get CPU run state name failure for id(%d)\n",
			MODULE_NAME, rs_id);
		goto done;
	}

done:
	return rc;
}

static int brcm_pwr_rpc_set_boot_cfg(char *cpu_name)
{
	int tunnel = 0, rc = 0;
	uint8_t cpu_id = 0;

	tunnel = brcm_rpc_ba_get_fifo_tunnel_id(BA_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		pr_err("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, BA_RPC_RG_SMC_TUNNEL_NAME);
		rc = tunnel;
		goto done;
	}

	rc = brcm_rpc_ba_get_cpu_id(tunnel, cpu_name, &cpu_id);
	if (rc) {
		pr_err("%s : BA get CPU id failure for CPU(%s)\n",
			MODULE_NAME, cpu_name);
		goto done;
	}

	rc = ba_set_boot_cfg(tunnel, cpu_id);
	if (rc) {
		pr_err("%s : BA set boot cfg failure \n",MODULE_NAME);
		goto done;
	}

done:
	return rc;
}

static int bcm_rpc_ba_req_cpu_run_state(const char *rs_name,
				const char *cpu_name, bool rude)
{
	int tunnel = 0, rc = 0;
	uint8_t cpu_id = 0, rs_id = 0;

	rs_req_hs_flag = false;
	mutex_lock(&runstate_hs_mutex);
	tunnel = brcm_rpc_ba_get_fifo_tunnel_id(BA_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		pr_err("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, BA_RPC_RG_SMC_TUNNEL_NAME);
		rc = tunnel;
		goto done;
	}
	rc = brcm_rpc_ba_get_cpu_id(tunnel, cpu_name, &cpu_id);
	if (rc) {
		pr_err("%s : BA get CPU id failure for CPU(%s)\n",
			MODULE_NAME, cpu_name);
		goto done;
	}
	rc = brcm_rpc_ba_get_run_state_id(tunnel, rs_name, &rs_id);
	if (rc) {
		pr_err("%s : BA get CPU run state ID failure for run state(%s)\n",
			MODULE_NAME, rs_name);
		goto done;
	}
	rc = ba_req_cpu_run_state(tunnel, cpu_id, rs_id, rude);
	if (rc) {
		pr_err("%s : BA get CPU run state name failure for id(%d)\n",
			MODULE_NAME, rs_id);
		goto done;
	}

	memset(reqested_rs_name, 0, sizeof(reqested_rs_name));
	strncpy(reqested_rs_name, rs_name, sizeof(reqested_rs_name)-1);

done:
	if (rc)
		mutex_unlock(&runstate_hs_mutex);

	return rc;
}

static void cmd_notify_run_state_help(char *str)
{
	pr_info("%s notifyrunstate:  Notify following run state for RG cpu\n"
			 "%s | %s | %s | %s | %s \n",
			 str, BA_RPC_RS_OFF, BA_RPC_RS_RESET,
			 BA_RPC_RS_BOOT, BA_RPC_RS_SHUTDOWN,
			 BA_RPC_RS_READY);
	pr_info("%s   notifyrunstate <state>\n", str);
}

static int cmd_notify_run_state(int argc, char *argv[])
{
	int state_cnt;
	int rc = 0;

	if (argc  < 2) {
		cmd_notify_run_state_help("");
		return 0;
	}
	/* Set command information */
	memset(&rs_info, 0, sizeof(rs_info));
	rs_info.command = BA_CMD_NOTIFY_RS;
	strncpy(&rs_info.cpu_name[0], BA_RPC_RG_CPU_NAME,
			(BA_NAME_MAX_LEN-1));
	for (state_cnt = 0; state_cnt < BA_RPC_RS_MAX_STATE; state_cnt++) {
		if (!strncmp(argv[1], ba_rpc_rs_name[state_cnt],
					 strlen(ba_rpc_rs_name[state_cnt]))) {
			rc = bcm_rpc_ba_notify_cpu_run_state(
			   ba_rpc_rs_name[state_cnt]);
			if (rc)
				goto done;

			break;
		}
	}
	if (state_cnt  >= BA_RPC_RS_MAX_STATE) {
		pr_err("Error %s: invalid state %s\n",
				argv[0], argv[1]);
		cmd_notify_run_state_help("");
		goto done;
	}
	/* Update state information */
	rs_info.state = state_cnt;
	strncpy(&rs_info.rs_name[0], ba_rpc_rs_name[state_cnt],
		(BA_NAME_MAX_LEN-1));
done:
	return 0;
}

static void cmd_get_run_state_help(char *str)
{
	pr_info("%s getrunstate: Get run state for the cpu\n", str);
	pr_info("%s  getrunstate <cpu>\n", str);
}

static int cmd_get_run_state(int argc, char *argv[])
{
	char rs_name[BA_NAME_MAX_LEN] = {0};
	char cpu_name[BA_NAME_MAX_LEN] = {0};
	int rc = 0;

	if (argc  < 2) {
		cmd_get_run_state_help("");
		return 0;
	}
	/* Set command information */
	memset(&rs_info, 0, sizeof(rs_info));
	rs_info.command = BA_CMD_GET_RS;
	strncpy(&rs_info.cpu_name[0], argv[1], (BA_NAME_MAX_LEN-1));

	strncpy(cpu_name, argv[1], (BA_NAME_MAX_LEN-1));
	rc = bcm_rpc_ba_get_cpu_run_state(rs_name, cpu_name);
	if (rc)
		goto done;

	/* Update state information */
	strncpy(&rs_info.rs_name[0], rs_name, (BA_NAME_MAX_LEN-1));
done:
	return 0;
}

static void cmd_get_requested_run_state_help(char *str)
{
	pr_info("%s getreqrunstate: Get the last requested run state for the cpu\n", str);
	pr_info("%s  getreqrunstate \n", str);
}

static int cmd_get_requested_run_state(int argc, char *argv[])
{
	if (argc  < 1) {
		cmd_get_requested_run_state_help("");
		return 0;
	}
	/* Set command information */
	memset(&rs_info, 0, sizeof(rs_info));
	rs_info.command = BA_CMD_GET_REQ_RS;
	strncpy(&rs_info.cpu_name[0], BA_RPC_RG_CPU_NAME, (BA_NAME_MAX_LEN-1));

	/* Update state information */
	strncpy(&rs_info.req_rs_name[0], reqested_rs_name, (BA_NAME_MAX_LEN-1));

	return 0;
}

static void cmd_req_run_state_help(char *str)
{
	pr_info("%s reqrunstate:  Request following run state for a cpu\n"
			 "%s | %s | %s | %s\n",
			 str, BA_RPC_RS_OFF, BA_RPC_RS_RESET,
			 BA_RPC_RS_BOOT, BA_RPC_RS_SHUTDOWN);
	pr_info("%s   reqrunstate <cpu> <state>\n", str);
}

static int cmd_req_run_state(int argc, char *argv[])
{
	int state_cnt;
	char cpu_name[BA_NAME_MAX_LEN] = {0};
	int rc = 0;

	if (argc  < 3) {
		cmd_req_run_state_help("");
		return 0;
	}
	/* Set command information */
	memset(&rs_info, 0, sizeof(rs_info));
	rs_info.command = BA_CMD_REQ_RS;
	strncpy(&rs_info.cpu_name[0], argv[1], (BA_NAME_MAX_LEN-1));

	strncpy(cpu_name, argv[1], (BA_NAME_MAX_LEN-1));
	for (state_cnt = 0; state_cnt < BA_RPC_RS_MAX_STATE; state_cnt++) {
		if (!strncmp(argv[2], ba_rpc_rs_name[state_cnt],
					 strlen(ba_rpc_rs_name[state_cnt]))) {
			rc = bcm_rpc_ba_req_cpu_run_state(
			   ba_rpc_rs_name[state_cnt],
			   (const char*)cpu_name, false);
			if (rc)
				goto done;

			break;
		}
	}
	if (state_cnt  >= BA_RPC_RS_MAX_STATE) {
		pr_err("Error %s: invalid state %s\n",
				argv[0], argv[1]);
		cmd_req_run_state_help("");
		goto done;
	}

	/* Update state information */
	rs_info.state = state_cnt;
	strncpy(&rs_info.rs_name[0], ba_rpc_rs_name[state_cnt],
		(BA_NAME_MAX_LEN-1));
done:
	return 0;
}

int bcm_ba_get_cpu_run_state(char *run_state, char *cpu_name)
{
	int rc = 0;

	if (!run_state || !cpu_name)
		return -EINVAL;

	if(!strncmp(cpu_name, ba_rpc_cpu_name, strlen(ba_rpc_cpu_name)))
		strncpy(run_state, ba_rpc_rs_state, sizeof(ba_rpc_rs_state)-1);
	else {
		rc = bcm_rpc_ba_get_cpu_run_state(run_state, cpu_name);
		if (rc)
			return rc;
	}
	return 0;
}
EXPORT_SYMBOL(bcm_ba_get_cpu_run_state);

int bcm_ba_req_cpu_poweroff(const char *cpu, bool rude)
{
	int rc = 0;

	if (!cpu)
		return -EINVAL;

	rc = bcm_rpc_ba_req_cpu_run_state(BA_RPC_RS_OFF, cpu, rude);

	return rc;
}
EXPORT_SYMBOL(bcm_ba_req_cpu_poweroff);

int bcm_ba_req_cpu_halt(const char *cpu, bool rude)
{
	int rc = 0;

	if (!cpu)
		return -EINVAL;

	rc = bcm_rpc_ba_req_cpu_run_state(BA_RPC_RS_RESET, cpu, rude);

	return rc;
}
EXPORT_SYMBOL(bcm_ba_req_cpu_halt);

int bcm_ba_req_cpu_restart(const char *cpu, bool rude)
{
	int rc = 0;

	if (!cpu)
		return -EINVAL;

	rc = bcm_rpc_ba_req_cpu_run_state(BA_RPC_RS_BOOT, cpu, rude);

	return rc;
}
EXPORT_SYMBOL(bcm_ba_req_cpu_restart);

static struct proc_cmd_ops rpc_ba_cmd_entries[] = {
	PROC_CMD_INIT("notifyrunstate",   cmd_notify_run_state),
	PROC_CMD_INIT("getrunstate",   cmd_get_run_state),
	PROC_CMD_INIT("getreqrunstate",   cmd_get_requested_run_state),
	PROC_CMD_INIT("reqrunstate",   cmd_req_run_state),
};

struct proc_cmd_table rpc_ba_cmd_table = {
	.module_name = MODULE_NAME,
	.size = ARRAY_SIZE(rpc_ba_cmd_entries),
	.data_seq_read = (void *) &ba_rs_seq_ops,
	.ops = rpc_ba_cmd_entries,
};

#if defined(CONFIG_BCM_PWR_RPC)
static void brcm_pwr_rpc_mbox_work_handler(struct work_struct *work)
{
	brcm_rpc_ba_rg_poweroff(BA_RPC_RG_CPU_NAME, true, kernel_core_req);
}
/*
 * NOTE: This function is designed to be called in ISR context. Make sure
 * any future additions are ISR safe.
 */
static int ba_brcm_pwr_rpc_mbox_event(
		struct notifier_block *this,
		unsigned long event, void *ptr)
{
	int status = NOTIFY_OK;
	struct brcm_pwr_rpc_mbox_info *states = ptr;

	switch (event) {
	case PWR_MBOX_CHANGE_EVENT:
	{
		/*
		 * Currently, we only handle notifications for AC to battery
		 * transitions. Transitions to AC from battery will be handled
		 * by the powerman application.
		 */
		if (BATT_STATE(states->mbox[PWR_MBOX_BS]) == BS_BATTERY) {
			bcm_rpc_ba_notify_cpu_run_state(BA_RPC_RS_BATTERY);
			if(rg_batt_mode == RG_CPU_OFF_UTILITY ||
			   rg_batt_mode == RG_CPU_OFF_BATTERY) {
				schedule_work_on(0, &work);
				status = NOTIFY_DONE;
			}
		}
		else if (BATT_STATE(states->mbox[PWR_MBOX_BS]) == BS_AC) {
			bcm_rpc_ba_notify_cpu_run_state(BA_RPC_RS_READY);
			status = NOTIFY_DONE;
		}
		break;
	}
	default:
		status = NOTIFY_DONE;
		goto done;
		break;
	}

done:
	return status;
}

/**
* Returns the configured RG behavior when the system is running
* on battery power. Available to other kernel modules.
*/
enum rg_battery_mode brcm_rpc_ba_rg_batt_mode(void)
{
	return rg_batt_mode;
}
EXPORT_SYMBOL(brcm_rpc_ba_rg_batt_mode);

static void brcm_rpc_ba_kernel_poweroff(void)
{
	kernel_core_req = false;
	kernel_power_off();
	do_exit(0);
}

int brcm_rpc_ba_rg_poweroff(const char *cpu_name, bool request, bool core)
{
	char rs_name[BA_NAME_MAX_LEN] = {0};
	int rc = 0;

	kernel_core_req = core;
#if defined(CONFIG_BCM_BA_TO_POWERMAN_NOTIF)
	if (core) {
		if (request)
			bcm_ba_req_cpu_poweroff(cpu_name, false);
		else
			brcm_rpc_ba_kernel_poweroff();
		return 0;
	}
	rc = bcm_ba_get_cpu_run_state(rs_name, BA_RPC_RG_CPU_NAME);
	if (rc) {
		pr_err("Getting Run state Fails, initiate poweroff sequence...\n");
		if (request)
			bcm_ba_req_cpu_poweroff(cpu_name, false);
		else
			brcm_rpc_ba_kernel_poweroff();
		return 0;
	}
	if (!strncmp(rs_name, BA_RPC_RS_READY, strlen(BA_RPC_RS_READY))) {
		if (rg_batt_mode == RG_CPU_OFF_BATTERY) {
			/*while (!brcm_pwr_rpc_is_registered_domain_turned_off(PWR_DOMAINS_MAX)) {
				pr_debug("Wating for registered Power Domain to switch off...\n");
			}*/
			if (request)
				bcm_ba_req_cpu_poweroff(cpu_name, false);
			else
				brcm_rpc_ba_kernel_poweroff();
		}
		else {
			pr_debug("RG is ready and we boot on utility, so"
					 " powerman will do the poweroff sequence\n");
			if (request)
				kill_fasync(&ba_rpc_sigio_list, SIGIO, POLL_IN);
			else
				brcm_rpc_ba_kernel_poweroff();
		}
	}
	else {
		pr_debug("RG is not ready, initiate poweroff sequence\n");
		/*while (!brcm_pwr_rpc_is_registered_domain_turned_off(PWR_DOMAINS_MAX)) {
			pr_debug("Wating for registered Power Domain to switch off...\n");
		}*/
		if (request)
			bcm_ba_req_cpu_poweroff(cpu_name, false);
		else
			brcm_rpc_ba_kernel_poweroff();
	}
#else
	rc = bcm_ba_get_cpu_run_state(rs_name, BA_RPC_RG_CPU_NAME);
	if (!rc) {
		if (!(!strncmp(rs_name, BA_RPC_RS_READY, strlen(BA_RPC_RS_READY)) ||
			!strncmp(rs_name, BA_RPC_RS_BATTERY, strlen(BA_RPC_RS_BATTERY)))) {
			pr_info("Current run state in %s\n", rs_name);
			/* Return without poweroff if run state not ready (or) battery */
			//return 0;
		}
	}
	if (request)
		bcm_ba_req_cpu_poweroff(cpu_name, false);
	else
		brcm_rpc_ba_kernel_poweroff();
#endif
	return 0;
}
EXPORT_SYMBOL(brcm_rpc_ba_rg_poweroff);

static void brcm_rpc_ba_kernel_restart(char *buffer)
{
	kernel_core_req = false;
	kernel_restart(buffer);
}

int brcm_rpc_ba_rg_restart(char *buffer, bool request, bool core)
{
	char rs_name[BA_NAME_MAX_LEN] = {0};
	int rc = 0;

	kernel_core_req = core;
#if defined(CONFIG_BCM_BA_TO_POWERMAN_NOTIF)
	if (core) {
		if (request)
			bcm_ba_req_cpu_restart(BA_RPC_ALL_CPU_NAME, false);
		else
			brcm_rpc_ba_kernel_restart(buffer);
		return 0;
	}
	rc = bcm_ba_get_cpu_run_state(rs_name, BA_RPC_RG_CPU_NAME);
	if (rc) {
		pr_err("Getting Run state Fails, initiate poweroff sequence...\n");
		if (request)
			bcm_ba_req_cpu_restart(BA_RPC_ALL_CPU_NAME, false);
		else
			brcm_rpc_ba_kernel_restart(buffer);
		return 0;
	}
	if (!strncmp(rs_name, BA_RPC_RS_READY, strlen(BA_RPC_RS_READY))) {
		if (rg_batt_mode == RG_CPU_OFF_BATTERY) {
			/*while (!brcm_pwr_rpc_is_registered_domain_turned_off(PWR_DOMAINS_MAX)) {
				pr_debug("Wating for registered Power Domain to switch off...\n");
			}*/
			if (request)
				bcm_ba_req_cpu_restart(BA_RPC_ALL_CPU_NAME, false);
			else
				brcm_rpc_ba_kernel_restart(buffer);
		}
		else {
			pr_debug("RG is ready and we boot on utility, so"
					 " powerman will do the poweroff sequence\n");
			if (request)
				kill_fasync(&ba_rpc_sigio_list, SIGIO, POLL_IN);
			else
				brcm_rpc_ba_kernel_restart(buffer);
		}
	}
	else {
		pr_debug("RG is not ready, initiate poweroff sequence\n");
		/*while (!brcm_pwr_rpc_is_registered_domain_turned_off(PWR_DOMAINS_MAX)) {
			pr_debug("Wating for registered Power Domain to switch off...\n");
		}*/
		if (request)
			bcm_ba_req_cpu_restart(BA_RPC_ALL_CPU_NAME, false);
		else
			brcm_rpc_ba_kernel_restart(buffer);
	}
#else
	rc = bcm_ba_get_cpu_run_state(rs_name, BA_RPC_RG_CPU_NAME);
	if (!rc) {
		if (!(!strncmp(rs_name, BA_RPC_RS_READY, strlen(BA_RPC_RS_READY)) ||
			!strncmp(rs_name, BA_RPC_RS_BATTERY, strlen(BA_RPC_RS_BATTERY)))) {
			pr_info("Current run state in %s\n", rs_name);
			/* Return without reboot if run state not ready (or) battery */
			//return 0;
		}
	}
	if (request)
		bcm_ba_req_cpu_restart(BA_RPC_ALL_CPU_NAME, false);
	else
		brcm_rpc_ba_kernel_restart(buffer);
#endif
	return 0;
}
EXPORT_SYMBOL(brcm_rpc_ba_rg_restart);

int brcm_rpc_ba_rg_emergency_restart(bool request, bool core)
{
    kernel_core_req = core;
    if (request) {
        kernel_emerg_restart = true;
		bcm_ba_req_cpu_restart(BA_RPC_ALL_CPU_NAME, false);
    } else {
        if (kernel_emerg_restart == true) {
            kernel_emerg_restart = false;
            kernel_core_req = false;
            machine_emergency_restart();
        }
    }

    return 0;
}
EXPORT_SYMBOL(brcm_rpc_ba_rg_emergency_restart);

static void brcm_rpc_ba_kernel_halt(void)
{
	kernel_core_req = false;
	kernel_halt();
	do_exit(0);
	panic("cannot halt");
	kernel_power_off();
	do_exit(0);
}

int brcm_rpc_ba_rg_halt(bool request, bool core)
{
	char rs_name[BA_NAME_MAX_LEN] = {0};
	int rc = 0;

	kernel_core_req = core;
#if defined(CONFIG_BCM_BA_TO_POWERMAN_NOTIF)
	if (core) {
		if (request)
			bcm_ba_req_cpu_halt(BA_RPC_RG_CPU_NAME, false);
		else
			brcm_rpc_ba_kernel_halt();
		return 0;
	}
	rc = bcm_ba_get_cpu_run_state(rs_name, BA_RPC_RG_CPU_NAME);
	if (rc) {
		pr_err("Getting Run state Fails, initiate poweroff sequence...\n");
		if (request)
			bcm_ba_req_cpu_halt(BA_RPC_RG_CPU_NAME, false);
		else
			brcm_rpc_ba_kernel_halt();
		return 0;
	}
	if (!strncmp(rs_name, BA_RPC_RS_READY, strlen(BA_RPC_RS_READY))) {
		if (rg_batt_mode == RG_CPU_OFF_BATTERY) {
			/*while (!brcm_pwr_rpc_is_registered_domain_turned_off(PWR_DOMAINS_MAX)) {
				pr_debug("Wating for registered Power Domain to switch off...\n");
			}*/
			if (request)
				bcm_ba_req_cpu_halt(BA_RPC_RG_CPU_NAME, false);
			else
				brcm_rpc_ba_kernel_halt();
		}
		else {
			pr_debug("RG is ready and we boot on utility, so"
					 " powerman will do the poweroff sequence\n");
			if (request)
				kill_fasync(&ba_rpc_sigio_list, SIGIO, POLL_IN);
			else
				brcm_rpc_ba_kernel_halt();
		}
	}
	else {
		pr_debug("RG is not ready, initiate poweroff sequence\n");
		/*while (!brcm_pwr_rpc_is_registered_domain_turned_off(PWR_DOMAINS_MAX)) {
			pr_debug("Wating for registered Power Domain to switch off...\n");
		}*/
		if (request)
			bcm_ba_req_cpu_halt(BA_RPC_RG_CPU_NAME, false);
		else
			brcm_rpc_ba_kernel_halt();
	}
#else
	rc = bcm_ba_get_cpu_run_state(rs_name, BA_RPC_RG_CPU_NAME);
	if (!rc) {
		if (!(!strncmp(rs_name, BA_RPC_RS_READY, strlen(BA_RPC_RS_READY)) ||
			!strncmp(rs_name, BA_RPC_RS_BATTERY, strlen(BA_RPC_RS_BATTERY)))) {
			pr_info("Current run state in %s\n", rs_name);
			/* Return without halt if run state not ready (or) battery */
			//return 0;
		}
	}
	if (request)
		bcm_ba_req_cpu_halt(BA_RPC_RG_CPU_NAME, false);
	else
		brcm_rpc_ba_kernel_halt();
#endif

	return 0;
}
EXPORT_SYMBOL(brcm_rpc_ba_rg_halt);

static struct notifier_block brcm_pwr_rpc_mbox_notifier = {
	.notifier_call  = ba_brcm_pwr_rpc_mbox_event,
};
#endif

void bcm_rpc_ba_notify_crash_state(void)
{
	time64_t now;
	struct tm tm_val;

	now = ktime_get_real_seconds();
	time64_to_tm(now, 0, &tm_val);

	pr_emerg("---[ BA Reboot due to panic: %d/%d/%ld %02d:%02d:%02d ]---\n",
		 tm_val.tm_mon + 1, tm_val.tm_mday, 1900 + tm_val.tm_year,
		 tm_val.tm_hour, tm_val.tm_min, tm_val.tm_sec);
	/* If crash runstate supported in SMC use it instead of we handle it */
	ba_notify_cpu_run_state(ba_rpc_tunnel, (uint8_t)ba_rpc_rg_cpu_id,
							(uint8_t)ba_rpc_rs_id[RUNSTATE_CRASH]);
}
EXPORT_SYMBOL(bcm_rpc_ba_notify_crash_state);

static int panic_callback(struct notifier_block *self,
			  unsigned long event, void *ctx)
{
    //bcm_rpc_ba_notify_crash_state();

	return NOTIFY_OK;
}

static struct notifier_block nb_panic = {
	.notifier_call  = panic_callback,
	.priority       = INT_MIN,
};

static int brcm_rpc_ba_reboot_notifier(struct notifier_block *self,
			  unsigned long event, void *ctx)
{
	int rc = NOTIFY_OK;

	if (event == SYS_POWER_OFF)
		brcm_rpc_ba_rg_poweroff(BA_RPC_ALL_CPU_NAME, true, true);
	else if (event == SYS_RESTART)
		brcm_rpc_ba_rg_restart(NULL, true, true);
	else if (event == SYS_HALT)
		brcm_rpc_ba_rg_halt(true, true);

	/* Block here until handshake is done */
	mutex_lock(&runstate_hs_mutex);
	if (rs_req_hs_flag) {
		pr_debug("Run state request handshake complete\n");
		rc = NOTIFY_OK;
	} else {
		pr_err("Run state request handshake failure\n");
		rc = NOTIFY_BAD;
	}
	mutex_unlock(&runstate_hs_mutex);

	return rc;
}

static struct notifier_block nb_brcm_rpc_ba_reboot = {
	.notifier_call  = brcm_rpc_ba_reboot_notifier,
	.next           = NULL,
	.priority       = INT_MAX, /* before any real devices */
};

struct proc_dir_entry *rpc_ba_proc_dir;

static rpc_function ba_services_tbl[] = {
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ ba_notify_cpu_run_state_cb,	0 },
	{ ba_req_cpu_run_state_cb,	0 },
	{ NULL,			0 },
	{ ba_set_cpu_run_state_cb,	0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ ba_prepare_run_state_cb,		0 },
	{ NULL,			0 },
};


int ba_rpc_async(int fd, struct file *filp, int onflag)
{
	return fasync_helper(fd, filp, onflag, &ba_rpc_sigio_list);
}

int ba_rpc_file_open(struct inode *inode, struct file *file)
{
	file->private_data = 0;
	return 0;
}

int ba_rpc_file_release(struct inode *inode, struct file *file)
{
	ba_rpc_async(-1, file, 0);
	return 0;
}

static ssize_t ba_rpc_file_read(struct file *file, char __user *buf, size_t size,
			loff_t *ppos)
{
	return size;
}

static ssize_t ba_rpc_file_write(struct file *file, const char __user *buf,
			size_t size, loff_t *ppos)
{
	return size;
}

static const struct file_operations ba_rpc_fops = {
	.owner =		THIS_MODULE,
	.open =			ba_rpc_file_open,
	.release =		ba_rpc_file_release,
	.read =			ba_rpc_file_read,
	.write =		ba_rpc_file_write,
	.fasync = 		ba_rpc_async,
};

static int brcm_ba_rpc_probe(struct platform_device *pdev)
{
	int rc = 0, tunnel = -1;
	struct device *dev = &pdev->dev;
	struct ba_rpc_dev_info *info;
	const char *str = NULL;
	const char *pwr_str = NULL;
	bool delay_batt_mode = false;
	int rs_cnt = 0;
	uint8_t cpu_id = 0, rs_id = 0;;

	pr_debug("-->\n");
	info = (struct ba_rpc_dev_info *)devm_kzalloc(dev,
		sizeof(struct ba_rpc_dev_info), GFP_KERNEL);
	if (IS_ERR_OR_NULL(info)) {
		dev_err(dev, "unable to allocate private data\n");
		rc = -ENOMEM;
		goto error_exit;
	}
	platform_set_drvdata(pdev, (void *)info);
	info->pdev = pdev;

	if (of_property_read_string(dev->of_node, RG_BATT_MODE_PROPERTY, &str))
		dev_info(dev,"%s property not found\n", RG_BATT_MODE_PROPERTY);
	else {
		if (!str) {
			pr_err("Invalid %s property\n", RG_BATT_MODE_PROPERTY);
			rc = -EINVAL;
			goto error_exit;
		} else if (!strcmp(str, RG_BATT_MODE_OFF)) {
			brcm_pwr_rpc_get_boot_state(&pwr_str);
			if (pwr_str) {
				if (!strcmp(pwr_str, DT_BOOT_STATE_BATTERY_STR)) {
					rg_batt_mode = RG_CPU_OFF_BATTERY;
				} else {
					brcm_pwr_rpc_get_delay_batt_mode(&delay_batt_mode);
					if (delay_batt_mode)
						rg_batt_mode = RG_CPU_OFF_UTILITY_DELAYED;
					else
						rg_batt_mode = RG_CPU_OFF_UTILITY;
				}
			}
		} else if (!strcmp(str, RG_BATT_MODE_CPU_ON)) {
			brcm_pwr_rpc_get_boot_state(&pwr_str);
			if (pwr_str) {
				if (!strcmp(pwr_str, DT_BOOT_STATE_BATTERY_STR))
					rg_batt_mode = RG_CPU_ON_BATTERY;
				else
					rg_batt_mode = RG_CPU_ON_UTILITY;
			}
		}
		else {
			pr_err("Invalid RG battery mode specified!\n");
			rc = -EINVAL;
			goto error_exit;
		}
	}

	tunnel = rpc_get_fifo_tunnel_id(BA_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		dev_err(dev, "%s : Error: invalid tunnel %s\n",
			MODULE_NAME, BA_RPC_RG_SMC_TUNNEL_NAME);
		rc = tunnel;
		goto error_exit;
	}
	ba_rpc_tunnel = tunnel;

	rc = ba_get_cpu_id(tunnel, BA_RPC_RG_CPU_NAME, &cpu_id);
	if (rc) {
		dev_err(dev, "%s : BA get CPU id failure for CPU(%s)\n",
			MODULE_NAME, BA_RPC_RG_CPU_NAME);
		goto error_exit;
	}
	ba_rpc_rg_cpu_id = cpu_id;

	rc = ba_get_cpu_id(tunnel, BA_RPC_ALL_CPU_NAME, &cpu_id);
	if (rc) {
		dev_err(dev, "%s : BA get CPU id failure for CPU(%s)\n",
			MODULE_NAME, BA_RPC_ALL_CPU_NAME);
		goto error_exit;
	}
	ba_rpc_all_cpu_id = cpu_id;

	/* Get All Run state ID's from String */
	for (rs_cnt = 0; rs_cnt < BA_RPC_RS_MAX_STATE; rs_cnt++) {
		ba_rpc_rs_id[rs_cnt] = -1;
		rc = ba_get_run_state_id(tunnel, ba_rpc_rs_name[rs_cnt],
					&rs_id);
		if (rc) {
			dev_err(dev, "%s : BA get CPU run state ID %s failure\n",
				   MODULE_NAME, ba_rpc_rs_name[rs_cnt]);
			//goto error_exit;
		}
		ba_rpc_rs_id[rs_cnt] = rs_id;
	}

	/* Initialize run state at bootup */
	rc = bcm_rpc_ba_get_cpu_run_state(ba_rpc_rs_state, BA_RPC_RG_CPU_NAME);
	if (rc)
		goto error_exit;

	/* Initialize battery/utility state at bootup */
	brcm_pwr_rpc_set_boot_cfg(BA_RPC_RG_CPU_NAME);

	info->ba_rpc_class = class_create(THIS_MODULE, BA_RPC_CLASS);
	if (IS_ERR(info->ba_rpc_class)) {
		dev_err(dev, "class_create() failed for ba_rpc_class\n");
		rc = PTR_ERR(info->ba_rpc_class);
		goto error_exit;
	}

	/* Allocate device nodes */
	rc = alloc_chrdev_region(&info->ba_rpc_dev, 0, BA_RPC_MAX_DEVS, BA_RPC_CLASS);
	if (rc < 0) {
		dev_err(dev, "%s: can't alloc chrdev region\n", __func__);
		class_destroy(info->ba_rpc_class);
		goto error_exit;
	}
	pr_info("Broadcom BA RPC MAJOR is %d\n", MAJOR(info->ba_rpc_dev));

	cdev_init(&info->ba_rpc_cdev, &ba_rpc_fops);
	rc = cdev_add(&info->ba_rpc_cdev, info->ba_rpc_dev, 1);
	if (rc < 0) {
		dev_err(dev, "Error adding cdev for %s\n", BA_RPC_DEVNAME);
		class_destroy(info->ba_rpc_class);
		unregister_chrdev_region(info->ba_rpc_dev, 1);
		goto error_exit;
	}

	info->ba_rpc_device = device_create(info->ba_rpc_class,
		NULL, info->ba_rpc_dev, NULL, BA_RPC_DEVNAME);
	if (IS_ERR(info->ba_rpc_device)) {
		dev_err(dev, "%s: can't register class device\n", __func__);
		info->ba_rpc_device = NULL;
		class_destroy(info->ba_rpc_class);
		cdev_del(&info->ba_rpc_cdev);
		unregister_chrdev_region(info->ba_rpc_dev, 1);
		rc = PTR_ERR(info->ba_rpc_device);
		goto error_exit;
	}
	atomic_notifier_chain_register(&panic_notifier_list,
				       &nb_panic);
	register_reboot_notifier(&nb_brcm_rpc_ba_reboot);
	/* Register notifier callback for power change requests */
#if defined(CONFIG_BCM_PWR_RPC)
	INIT_WORK(&work, brcm_pwr_rpc_mbox_work_handler);
	brcm_pwr_rpc_mbox_register_notifier(&brcm_pwr_rpc_mbox_notifier);
#endif
	pr_debug("<--\n");
	return 0;

error_exit:
#if defined(CONFIG_BCM_PWR_RPC)
	brcm_pwr_rpc_mbox_unregister_notifier(&brcm_pwr_rpc_mbox_notifier);
#endif
	unregister_reboot_notifier(&nb_brcm_rpc_ba_reboot);
	atomic_notifier_chain_unregister(&panic_notifier_list,
					 &nb_panic);
	if (info) {
		devm_kfree(dev, (void *)info);
		info = NULL;
	}
	pr_info("Error Exit <--\n");

	return rc;
}

static int brcm_ba_rpc_remove(struct platform_device *pdev)
{
	struct ba_rpc_dev_info *info;

	pr_debug("-->\n");

	info = (struct ba_rpc_dev_info *)platform_get_drvdata(pdev);
	if (!info)
		goto error_exit;

#if defined(CONFIG_BCM_PWR_RPC)
	brcm_pwr_rpc_mbox_unregister_notifier(&brcm_pwr_rpc_mbox_notifier);
	cancel_work_sync(&work);
#endif
	unregister_reboot_notifier(&nb_brcm_rpc_ba_reboot);
	atomic_notifier_chain_unregister(&panic_notifier_list,
					 &nb_panic);
    class_destroy(info->ba_rpc_class);
    cdev_del(&info->ba_rpc_cdev);
    unregister_chrdev_region(info->ba_rpc_dev, 1);
	devm_kfree(&pdev->dev, (void *)info);
	info = NULL;

error_exit:
	pr_debug("<--\n");
	return 0;
}

static const struct of_device_id brcm_ba_rpc_of_match[] = {
	{ .compatible = BA_RPC_OF_MATCH},
	{}
};
MODULE_DEVICE_TABLE(of, brcm_ba_rpc_of_match);
static struct platform_driver brcm_ba_rpc_driver = {
	.probe = brcm_ba_rpc_probe,
	.remove = brcm_ba_rpc_remove,
	.driver	= {
		.name		= MODULE_NAME,
		.owner		= THIS_MODULE,
		.of_match_table	= brcm_ba_rpc_of_match
	}
};

static int __init brcm_ba_rpc_init(void)
{
	int rc = 0;

	rc = rpc_register_functions(RPC_SERVICE_BA,
					ba_services_tbl,
					sizeof(ba_services_tbl)/
					sizeof(rpc_function));
	if (rc) {
		pr_err("%s: Failed to register BA RPC function(s).\n",
			MODULE_NAME);
		return rc;
	}

	rpc_ba_proc_dir = proc_mkdir("driver/ba", NULL);
	if (!rpc_ba_proc_dir) {
		pr_err("%s: Warning: cannot create proc file\n",
		       MODULE_NAME);
		rpc_unregister_functions(RPC_SERVICE_BA);
		return -ENOMEM;
	}
	proc_create_cmd("cmd", rpc_ba_proc_dir, &rpc_ba_cmd_table);
	pr_info("%s: BA RPC Service module\n", __func__);
	platform_driver_register(&brcm_ba_rpc_driver);
	return rc;
}
subsys_initcall(brcm_ba_rpc_init);


static void __exit brcm_ba_rpc_exit(void)
{
	rpc_unregister_functions(RPC_SERVICE_BA);
	platform_driver_unregister(&brcm_ba_rpc_driver);

}
module_exit(brcm_ba_rpc_exit);

MODULE_DESCRIPTION("BRCM BA RPC service Module");
MODULE_LICENSE("GPL v2");
