/****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Peter Sulc <peter.sulc@broadcom.com>
 * Filename: ubus_capture.c
 * Description: Ubus capture driver adapted from Mike Sieweke's
 * UbusCapture3390.js BBS script
 ****************************************************************************/

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/io.h>
#include <linux/printk.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <asm/exception.h>
#include <linux/seq_file.h>

#include "ubus_capture.h"

struct ubus_base {
	phys_addr_t pbase;
	void *base;
	u32 num_engines;
};

struct ubus_proc_seq {
	int ubase_num;
	int engine_num;
	int segment;
	int error_cnt;
};

struct decode_info {
	u32 buf0;
	u32 buf1;
	u32 buf2;
	u32 buf3;
	u32 srcpid;
	u32 dstpid;
	u32 command;
	u32 timestamp;
	u32 header;
	u32 pktid;
	u32 dlen;
	u32 addr;
	u32 lastaddr;
	int error;
};

struct filter_range {
	int fnum;
	u32 min;
	u32 max;
	int exclude;
};

static const char *segment_names[] = {
	"Seg A0 UBUS0",
	"Seg A1 UBUS0",
	"Seg B0 UBUS0",
	"Seg B1 UBUS0",
	"Seg C0 UBUS0",
	"Seg C1 UBUS0",
	"Seg A0 UBUS1",
	"Seg A1 UBUS1",
	"Seg B0 UBUS1",
	"Seg B1 UBUS1",
	"Seg C0 UBUS1",
	"Seg C1 UBUS1"
};

// Blank names are reserved port IDs.
static const char *port_names[] = {
        "MEMC0_P0",
        "MEMC1_P0",
        "UNIWAN",
        "DAVIC",
        "SEGDMA",
        "UTP",
        "CRYPTO",
        "USMAC",
        "DTP",
        "DFAP",
        "D3MAC0",
        "D3OFDMA",
        "D3OFDMB",
        "TCOFDM",
        "APM",
        "DECT",
        "LEAP",
        "LEAP",
        "MIPS",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "U2U",
        "EMC",
        "CPUCOM",
        "FPM",
        "PERIPH",
        "UBUS_CORE",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "MEMC0_P1",
        "MEMC1_P1",
        "B15CPU-C0",
        "B15CPU-C1",
        "",
        "",
        "",
        "SF2",
        "DBR",
        "DBR1",
        "RABR",
        "RBBR",
        "NATC",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "U2U",
        "UGB",
        "UBUS_CORE"
};

static const char *req_rep[] = {
	"Reqst",
	"Reply"
};

static const char *commandname[] = {
	"WRITE",
	"READ",
	"WR_ACK"
};

static u32 buf0[256];
static u32 buf1[256];
static u32 buf2[256];
static u32 buf3[256];
static int firstentry;
static int numentries;

u32 ubus_capt_addr_start;
u32 ubus_capt_addr_end;
u32 ubus_capt_addr_exclude;
u32 ubus_capt_pid_start;
u32 ubus_capt_pid_end;
u32 ubus_capt_pid_exclude;
u32 ubus_capt_on;

static struct ubus_base *ubus_bases;
static unsigned num_ubus_bases;
static unsigned num_segments;

static const char *portname(int p)
{
	const char *name;

	if (p < 0 || p >= (sizeof(port_names) / sizeof(char *))) {
		return "BAD";
	}
	name = port_names[p];
	if (name[0] == 0)
		return "RSVD";
	return name;
}

static void *capt_base(struct ubus_base *ubase, int bus, int rep)
{
	void *eng_base = ubase->base + CAPT_BASE_OFFSET;
	return eng_base +
		(bus * CAPT_ENGINE_OFFSET) +
		(rep * CAPT_REPLY_OFFSET);
}

static u32 capt_reg_read(struct ubus_base *ubase, int bus, int rep, u32 offset)
{
	void *reg = capt_base(ubase, bus, rep) + offset;
	return __raw_readl(reg);
}

static void capt_reg_write(struct ubus_base *ubase, int bus, int rep,
			   u32 offset, u32 val)
{
	void *reg = capt_base(ubase, bus, rep) + offset;
	__raw_writel(val, reg);
}

// Write a register in a capture block, "or"ing with the value.
static void capt_reg_mask_on(struct ubus_base *ubase, int bus, int rep,
			     u32 offset, u32 val)
{
	void *reg = capt_base(ubase, bus, rep) + offset;
	u32 curr_val = __raw_readl(reg);
	__raw_writel(curr_val | val, reg);
}


// Write a register in a capture block, "and"ing with the negated value.
static void capt_reg_mask_off(struct ubus_base *ubase, int bus, int rep,
			      u32 offset, u32 val)
{
	void *reg = capt_base(ubase, bus, rep) + offset;
	u32 curr_val = __raw_readl(reg);
	__raw_writel(curr_val & ~val, reg);
}

// ===========================================================================
// Decode functions
// ===========================================================================


// Read the capture data into arrays.
// This isn't really necessary, but it helped in early debugging.
// This is useful if you want to try decoding with multiple options.
static void read_buf(struct ubus_base *ubase, int bus, int rep)
{
	u32 i;
	u32 lasttimestamp = 0xffffffff;
	numentries = capt_reg_read(ubase, bus, rep, CAPTURE_STATUS);

	if (numentries > 256)
		numentries = 256;
	firstentry = 0;

	for (i = 0; i < numentries; i++) {
		buf3[i] = capt_reg_read(ubase, bus, rep, CAPT_BUF_WORD3);
		buf2[i] = capt_reg_read(ubase, bus, rep, CAPT_BUF_WORD2);
		buf1[i] = capt_reg_read(ubase, bus, rep, CAPT_BUF_WORD1);
		buf0[i] = capt_reg_read(ubase, bus, rep, CAPT_BUF_WORD0);
		if ((buf3[i] <= lasttimestamp) && (i != 0 ))
			firstentry = i;
		lasttimestamp = buf3[i];
	}
}

void decode_buf(struct decode_info *info, int n)
{
	info->buf0 = buf0[n];
	info->buf1 = buf1[n];
	info->buf2 = buf2[n];
	info->buf3 = buf3[n];
	if ((info->buf2 & CAPT_VALID) == 0)
		pr_info("Found invalid buffer at %d\n", n);

	// Buf 0 = header
	info->dstpid = ( info->buf0 >> 0 )  & 0xff;
	info->srcpid = ( info->buf0 >> 8 )  & 0xff;
	info->pktid  = ( info->buf0 >> 16 ) & 0xff;
	info->dlen   = ( info->buf0 >> 24 ) & 0xff;
	// Buf 1 = address
	info->addr   = info->buf1;
	// Buf 2 = extended header
	info->lastaddr = info->buf2 & 7;
	info->command  = ( info->buf2 >> 4 )  & 7;
	info->header = ( info->buf2 >> 29 ) & 1;
	info->error  = ( info->buf2 >> 7 ) & 1;
	// Buf 3 = timestamp
	info->timestamp = info->buf3;
}

static void ubus_printf(struct seq_file *seq, const char *f, ...)
{
	va_list args;

	va_start(args, f);
	if (seq)
		seq_vprintf(seq, f, args);
	else
		vprintk(f, args);
	va_end(args);
}

static int iterate_engines(int (*doitoit)(struct ubus_base *, int, int, void *),
			   void *arg)
{
	int i, j;
	int seg = 0;
	int cnt = 0;
	struct ubus_base *ubase;
	for (j = 0, ubase = ubus_bases; j < num_ubus_bases; j++, ubase++) {
		for (i = 0; i < ubase->num_engines; i++, seg++) {
			int ret = doitoit(ubase, i, seg, arg);
			if (ret < 0)
				return ret;
			cnt += ret;
		}
	}
	return cnt;
}

// Decode and print data from one capture engine.
static int decode_one(struct seq_file *seq, struct ubus_base *ubase,
		      int bus, int rep)
{
	u32 startaddr = 0, endaddr = 0, curaddr = 0, n = 0, i = 0;
	int bytesize = 0;
	const char *arrowstring, *source, *dest, *err_string;
	struct decode_info info;
	int error_cnt = 0;

	// Read data from the capture buffer into an array.
	read_buf(ubase, bus, rep);
	if ( numentries == 0 ) {
		ubus_printf(seq, "Buffer empty...\n" );
		return error_cnt;
	}
	ubus_printf(seq, "Accumulator = %08x %08x\n",
		    capt_reg_read(ubase, bus, rep, ACCUM_HI),
		    capt_reg_read(ubase, bus, rep, ACCUM_LO));

	n = firstentry;

	startaddr = endaddr = curaddr = 0xdeadc0ed;

	// Process entries in buffer
	for ( i = 0; i < numentries; i++ ) {
		decode_buf(&info, n);
		if (info.header) {
			if (info.dlen == 0) {
				endaddr  = startaddr;
				bytesize = 0;
			}
			else {
				endaddr = (curaddr + (info.dlen-1) * 8) +
					info.lastaddr;
				bytesize = endaddr - startaddr + 1;
			}
			if (endaddr < startaddr) {
				ubus_printf(seq,
					    "End addr %08x < Start addr %08x\n",
					    endaddr, startaddr);
			}
			// An even more terse format, suggested by Mike.
			arrowstring = (info.command & 0x01) ? "<-" : "->";
			source = portname(info.srcpid);
			dest   = portname(info.dstpid);
			if (info.error) {
				error_cnt++;
				err_string = "ERROR!";
			}
			else {
				err_string = "";
			}
			if ( bytesize < 0 )
				bytesize = -1;

			ubus_printf(seq,
				    "%s %s %s(%d) %s %s(%d) %08x[%d] "
				    "time %08x %s\n",
				    req_rep[rep],
				    commandname[info.command],
				    source,
				    info.srcpid,
				    arrowstring,
				    dest,
				    info.dstpid,
				    info.addr,
				    bytesize,
				    info.timestamp,
				    err_string);
		}
		else {
			u32 begin, end, nextaddr, i, n;
			char byteenable[26], ch;
			nextaddr = (curaddr + 7);
			// Calculate begin and end byte address on this bus word
			begin = curaddr;
			end   = nextaddr;
			if ((curaddr < startaddr) && (startaddr <= nextaddr))
				begin = startaddr;
			if ((curaddr <= endaddr) && (endaddr < nextaddr))
				end = endaddr;

			// Get begin and end offset within current bus word.
			begin = begin & 7;
			end   = end & 7;
			for (i = 0, n = 0; i < 8; i++) {
				if (i == 4)
					byteenable[n++] = ' ';
				ch = ((i < begin) || (i > end)) ? '-' : 'X';
				byteenable[n++] = ch;
				byteenable[n++] = ch;
			}
			byteenable[n++] = '\n';
			byteenable[n++] = 0;
			ubus_printf(seq, "Byte Enable: ");
			ubus_printf(seq, byteenable);
			ubus_printf(seq, "       Data: %08x %08x\n",
				    info.buf0, info.buf1);
			curaddr += 8;
		}
		n++;
		if (n == 256)
			n = 0;
	}
	return error_cnt;
}

// Decode and print data from one capture engine.
static int capt_decode_engine(struct ubus_base *ubase, int engine, int seg,
			      void *arg)
{
	int error_cnt = 0;
	struct seq_file *seq = arg;
	ubus_printf(seq, "Decode %s request buffer\n", segment_names[seg]);
	error_cnt += decode_one(seq, ubase, engine, 0);
	ubus_printf(seq, "Decode %s reply buffer\n", segment_names[seg]);
	error_cnt += decode_one(seq, ubase, engine, 1);
	return error_cnt;
}

// Decode and print data from all capture engines.
static void capt_decode(void)
{
	int error_cnt = iterate_engines(capt_decode_engine, NULL);
	pr_info("Total Errors: %d\n", error_cnt);
}

// ===========================================================================
// Capture control/configuration functions
// ===========================================================================

// Verify the capture engines' base pointers.
static int capt_check_engine(struct ubus_base *ubase, int engine, int seg,
			     void *arg)
{
	u32 cnt = 0;
	u32 z = capt_reg_read(ubase, engine, 0, CAPTURE_ID) & 0xffffff00;
	if (z == 0x43505400)
		cnt++;
	else
		pr_err("Invalid capture id %08x on bus %d\n", z, seg);
	z = capt_reg_read(ubase, engine, 1, CAPTURE_ID) & 0xffffff00;
	if (z == 0x43505400)
		cnt++;
	else
		pr_err("Invalid reply id %08x on bus %d\n", z, seg);
	return cnt;
}

static int capt_check(void)
{
	int cnt = iterate_engines(capt_check_engine, NULL);

	if (cnt && (cnt == (num_segments * 2)))
		return 0;
	pr_err("Failure in capture engine check: segments %d engines %d\n",
	       num_segments, cnt);
	return -1;
}


static int capt_engine_enabled(struct ubus_base *ubase, int engine, int seg,
			       void *arg)
{
	int enabled = capt_reg_read(ubase, engine, 0, CAPTURE_ENABLE) & CE_EN;
	enabled += capt_reg_read(ubase, engine, 1, CAPTURE_ENABLE) & CE_EN;
	return enabled;
}

// Return true if any engine is enabled.
int ubus_capt_any_enabled(void)
{
	return iterate_engines(capt_engine_enabled, NULL);
}

// Return true if all engines are enabled.
int ubus_capt_all_enabled(void)
{
	int cnt = iterate_engines(capt_engine_enabled, NULL);
	return (cnt == (num_segments * 2));
}


// Reset capture engine state.
static int capt_reset_engine(struct ubus_base *ubase, int engine, int seg,
			     void *arg)
{
	// Do not reset the capture engine,
	// this causes the clocks to get out
	// of sync between the REQ and REP busses.
	u32 rbits = F0_RESET | F1_RESET;

	// Reset request and reply engines and filters
	capt_reg_write(ubase, engine, 0, CAPTURE_ENABLE, rbits);
	while (capt_reg_read(ubase, engine, 0, CAPTURE_ENABLE) & rbits);

	capt_reg_write(ubase, engine, 1, CAPTURE_ENABLE, rbits);
	while (capt_reg_read(ubase, engine, 1, CAPTURE_ENABLE) & rbits);

	capt_reg_mask_on(ubase, engine, 0, CAPTURE_CONTROL, ACC_CLR);
	capt_reg_mask_on(ubase, engine, 1, CAPTURE_CONTROL, ACC_CLR);

	return 0;
}

// Reset capture engine state.
static void capt_reset(void)
{
	iterate_engines(capt_reset_engine, NULL);
}


// Enable and start capture on engine.
static int capt_start_engine(struct ubus_base *ubase, int engine, int seg,
			     void *arg)
{
	u32 val;

	capt_reg_mask_on(ubase, engine, 0, CAPTURE_ENABLE,  CE_EN);
	capt_reg_mask_on(ubase, engine, 1, CAPTURE_ENABLE,  CE_EN);
	val = capt_reg_read(ubase, engine, 0, CAPTURE_CONTROL);
	val = (val & ~STOP_CAPTURE) | START_CAPTURE;
	capt_reg_write(ubase, engine, 0, CAPTURE_CONTROL, val);
	val = capt_reg_read(ubase, engine, 1, CAPTURE_CONTROL);
	val = (val & ~STOP_CAPTURE) | START_CAPTURE;
	capt_reg_write(ubase, engine, 1, CAPTURE_CONTROL, val);
	capt_reg_mask_on(ubase, engine, 0, CAPTURE_ENABLE,  CE_EN | F0_EN);
	capt_reg_mask_on(ubase, engine, 1, CAPTURE_ENABLE,  CE_EN | F0_EN);
	return 0;
}
// Enable and start capture on all engines.
static void capt_start(void)
{
	iterate_engines(capt_start_engine, NULL);
}

// Stop capture on engine.
static int capt_stop_engine(struct ubus_base *ubase, int engine, int seg,
			    void *arg)
{
	capt_reg_write(ubase, engine, 0, CAPTURE_CONTROL, STOP_CAPTURE);
	capt_reg_write(ubase, engine, 1, CAPTURE_CONTROL, STOP_CAPTURE);
	return 0;
}

// Stop capture on all engines.
static void capt_stop(void)
{
	pr_info("Stopping capture engines\n");
	iterate_engines(capt_stop_engine, NULL);
	pr_info("Capture engines stopped\n");
}

// Stop capture on engine full.
static int capt_stop_engine_on_full(struct ubus_base *ubase, int engine,
				    int seg, void *arg)
{
	capt_reg_mask_on(ubase, engine, 0, CAPTURE_CONTROL, STOPC_ON_FULL);
	capt_reg_mask_on(ubase, engine, 1, CAPTURE_CONTROL, STOPC_ON_FULL);
	return 0;
}

// Configure engines to stop when the buffer is full.
void ubus_capt_stop_on_full(void)
{
	iterate_engines(capt_stop_engine_on_full, NULL);
}

// Stop capture on engine error.
static int capt_stop_engine_on_error(struct ubus_base *ubase, int engine,
				     int seg, void *arg)
{
	capt_reg_mask_on(ubase, engine, 0, CAPTURE_CONTROL, STOPC_ON_UBERR);
	capt_reg_mask_on(ubase, engine, 1, CAPTURE_CONTROL, STOPC_ON_UBERR);
	return 0;
}

// Configure engines to stop when the buffer is error.
void ubus_capt_stop_on_error(void)
{
	iterate_engines(capt_stop_engine_on_error, NULL);
}

// Configure engine to stop capture on filter match.
static int capt_stop_on_filter(struct ubus_base *ubase, int engine,
			       int seg, void *arg)
{
	u32 stop_on = (u32)arg;
	capt_reg_mask_on(ubase, engine, 0, CAPTURE_CONTROL, stop_on);
	capt_reg_mask_on(ubase, engine, 1, CAPTURE_CONTROL, stop_on);
	return 0;
}

// Configure engines to stop capture on filter match.
void ubus_capt_stop_on_filter(int fnum)
{
	iterate_engines(capt_stop_on_filter, (void *)fnum);
}


// ===========================================================================
// Filter control/configuration functions
// ===========================================================================


static u32 filter_base(int fnum)
{
	return fnum == 0 ? F0_BASE : F1_BASE;
}

static int capt_engine_enable_filter(struct ubus_base *ubase, int engine,
				     int seg, void *arg)
{
	int fnum = (int)arg;
	u32 f_base = filter_base(fnum);
	u32 f_en   = fnum == 0 ? F0_EN : F1_EN;

	capt_reg_mask_on(ubase, engine, 0, f_base+F_CONTROL, CAPT_EN);
	capt_reg_mask_on(ubase, engine, 0, CAPTURE_ENABLE, f_en);
	capt_reg_mask_on(ubase, engine, 1, f_base+F_CONTROL, CAPT_EN);
	capt_reg_mask_on(ubase, engine, 1, CAPTURE_ENABLE, f_en);

	return 0;
}

// Enable filter and enable capture of matching transactions.
static void enable_filter(int fnum)
{
	iterate_engines(capt_engine_enable_filter, (void *)fnum);
}

// Enable an address window on filter 0 or 1
// If exclude == 1, reverse the sense of the window (exclude this range instead
// of including it).
static int capt_enable_address_window(struct ubus_base *ubase, int engine,
				      int seg, void *arg)
{
	struct filter_range *fr = arg;
	u32 f_base = filter_base(fr->fnum);
	int k;

	for (k = 0; k < 2; k++ ) {
		capt_reg_write(ubase, engine, k, f_base+F_ADDR_MIN, fr->min);
		capt_reg_write(ubase, engine, k, f_base+F_ADDR_MAX, fr->max);
		capt_reg_mask_on(ubase, engine, k, f_base+F_CONTROL,
				 ADDR_EN | ADDR_RANGE_SEL);
		if (fr->exclude)
			capt_reg_mask_on(ubase, engine, k, f_base+F_CONTROL,
					 ADDR_SENSE_SEL);
	}
	return 0;
}

void ubus_capt_enable_address_window(int fnum, u32 min, u32 max, int exclude)
{
	struct filter_range fr = { fnum, min, max, exclude };

	pr_info("Enable address window start %08x end %08x\n", min, max);

	iterate_engines(capt_enable_address_window, &fr);
}


// Enable the source PID filter for the specified range.
// If exclude == 1, reverse the sense of the window (exclude this range instead
// of including it).
static int capt_enable_src_pid_range(struct ubus_base *ubase, int engine,
				     int seg, void *arg)
{
	struct filter_range *fr = arg;
	u32 f_base = filter_base(fr->fnum);
	int k;

	for (k = 0; k < 2; k++ ) {
		capt_reg_mask_off(ubase, engine, k, f_base+F_PID, 0xffff0000);
		capt_reg_mask_on(ubase, engine, k, f_base+F_PID,
				 fr->min<<24 | fr->max<<16);
		capt_reg_mask_on(ubase, engine, k, f_base+F_CONTROL,
				 SPID_EN | SPID_RANGE_SEL);
		if (fr->exclude)
			capt_reg_mask_on(ubase, engine, k, f_base+F_CONTROL,
					 SPID_SENSE_SEL);
	}
	return 0;
}

void ubus_capt_enable_src_pid_range(int fnum, int min, int max, int exclude)
{
	struct filter_range fr = { fnum, min, max, exclude };

	pr_info("Enable source PID filter from %d to %d\n", min, max);

	iterate_engines(capt_enable_src_pid_range, &fr);
}


// Enable the destination PID filter for the specified range.
// If exclude == 1, reverse the sense of the window (exclude this range instead
// of including it).
static int capt_enable_dest_pid_range(struct ubus_base *ubase, int engine,
				      int seg, void *arg)
{
	struct filter_range *fr = arg;
	u32 f_base = filter_base(fr->fnum);
	int k;

	for (k = 0; k < 2; k++ ) {
		capt_reg_mask_off(ubase, engine, k, f_base+F_PID, 0x0000ffff);
		capt_reg_mask_on(ubase, engine, k, f_base+F_PID,
				 fr->min<<8 | fr->max);
		capt_reg_mask_on(ubase, engine, k, f_base+F_CONTROL,
				 DPID_EN | DPID_RANGE_SEL);
		if (fr->exclude)
			capt_reg_mask_on(ubase, engine, k, f_base+F_CONTROL,
					 DPID_SENSE_SEL);
	}
	return 0;
}

void ubus_capt_enable_dest_pid_range(int fnum, int min, int max, int exclude)
{
	struct filter_range fr = { fnum, min, max, exclude };

	pr_info("Enable dest PID filter from %d to %d\n", min, max);

	iterate_engines(capt_enable_dest_pid_range, &fr);
}


// The filter engine ANDs with the mask and compares with the value.
// match = (( headerbits & mask ) == value )
static int enable_engine_header_filter(struct ubus_base *ubase, int engine,
				       int seg, void *arg)
{
	struct filter_range *fr = arg;
	u32 f_base = filter_base(fr->fnum);
	int k;

	for (k = 0; k < 2; k++ ) {
		capt_reg_write(ubase, engine, k, f_base+F_HDR_MASK, fr->min);
		capt_reg_write(ubase, engine, k, f_base+F_HDR_FILTER, fr->max);
		capt_reg_mask_on(ubase, engine, k, f_base+F_CONTROL, HDR_EN);
	}
	return 0;
}
static void enable_header_filter(int fnum, u32 mask, u32 value)
{
	struct filter_range fr = { fnum, mask, value, 0 };
	iterate_engines(enable_engine_header_filter, &fr);
}


// Enable counting (accumulator).
static int enable_engine_filter_accum(struct ubus_base *ubase, int engine,
				      int seg, void *arg)
{
	u32 f_base = filter_base((int)arg);
	capt_reg_mask_on(ubase, engine, 0, f_base+F_CONTROL, ACC_EN);
	capt_reg_mask_on(ubase, engine, 1, f_base+F_CONTROL, ACC_EN);
	return 0;
}

static void enable_filter_accum(int fnum)
{
	iterate_engines(enable_engine_filter_accum, (void *)fnum);
}


// Enable "and" of filter settings -
// i.e. capture only when all parameters match.
// Default is "or" - i.e. capture when any parameters match.
static int enable_engine_filter_and(struct ubus_base *ubase, int engine,
				    int seg, void *arg)
{
	u32 f_base = filter_base((int)arg);
	capt_reg_mask_on(ubase, engine, 0, f_base+F_CONTROL, AND_OR_SEL);
	capt_reg_mask_on(ubase, engine, 1, f_base+F_CONTROL, AND_OR_SEL);
	return 0;
}

static void enable_filter_and(int fnum)
{
	iterate_engines(enable_engine_filter_and, (void *)fnum);
}


// Capture only headers - no data.
static int cap_engine_header_only(struct ubus_base *ubase, int engine,
				  int seg, void *arg)
{
	u32 f_base = filter_base((int)arg);
	capt_reg_mask_on(ubase, engine, 0, f_base+F_CONTROL, HDR_ONLY);
	capt_reg_mask_on(ubase, engine, 1, f_base+F_CONTROL, HDR_ONLY);
	return 0;
}

void ubus_capt_header_only(int fnum)
{
	iterate_engines(cap_engine_header_only, (void *)fnum);
}

static void configure_and_start(void)
{
	ubus_capt_enable_address_window(0,
					ubus_capt_addr_start,
					ubus_capt_addr_end,
					ubus_capt_addr_exclude);
	ubus_capt_enable_dest_pid_range(0,
					ubus_capt_pid_start,
					ubus_capt_pid_end,
					ubus_capt_pid_exclude);
	ubus_capt_enable_src_pid_range(0,
				       ubus_capt_pid_start,
				       ubus_capt_pid_end,
				       ubus_capt_pid_exclude);
	// Capture only when all filters in filter 0 match.
	enable_filter_and(0);
	// Enable counters (accumulators)
	enable_filter_accum(0);
	// Finally enable filter 0.
	enable_filter(0);
	// ------------------------------------------------
	// The capture engines don't stop on UBUS error reply.
	// Configure filter 1 to detect bus errors and stop capture.
	enable_header_filter(1, 1<<23, 1<<23);
	ubus_capt_stop_on_filter(1);
	ubus_capt_stop_on_error();
	enable_filter(1);

	capt_start();
}

struct ubus_base *ubus_capt_get_base(unsigned n)
{
	if (n >= num_ubus_bases)
		return NULL;
	return ubus_bases + n;
}

int ubus_capt_alloc_bases(unsigned num_bases)
{
	ubus_bases = kmalloc(sizeof(struct ubus_base) * num_bases, GFP_KERNEL);
	if (ZERO_OR_NULL_PTR(ubus_bases)) {
		pr_err("Memory allocation failure in %s\n", __func__);
		return -ENOMEM;
	}
	num_ubus_bases = num_bases;
	return 0;
}

static void parse_range(char *str, u32 *from, u32 *to)
{
	unsigned long lval;
	char *s;

	s = strchr(str, '-');
	if (s) {
		*s++ = 0;
		if (!kstrtoul(str, 0, &lval))
			*from = lval;
		if (s)
			if (!kstrtoul(s, 0, &lval))
				*to = lval;
	}
}

static int __init ubus_capt_param(char *str)
{
	char *s;
	while (str) {
		s = strchr(str, ',');
		if (s)
			*s++ = 0;
		if (!strncmp(str, "window=", 7)) {
			str += 7;
			parse_range(str,
				    &ubus_capt_addr_start,
				    &ubus_capt_addr_end);
		}
		else if (!strncmp(str, "pid=", 4)) {
			str += 4;
			parse_range(str,
				    &ubus_capt_pid_start,
				    &ubus_capt_pid_end);
		}
		str = s;
	}
	ubus_capt_on = 1;
	return 0;
}
early_param("ubus_capture", ubus_capt_param);

void ubus_capt_start(void)
{
	if (capt_check() < 0) {
		pr_err("Capture Engine Check failure\n");
		return;
	}
	capt_reset();
	configure_and_start();
	ubus_capt_on = 1;
}

void ubus_capt_stop(void)
{
	capt_stop();
	ubus_capt_on = 0;
}

void ubus_capt_print(void)
{
	capt_stop();
	capt_decode();
}

void *ubus_capt_show_proc_start(loff_t *pos)
{
	static struct ubus_proc_seq ups;
	int segment = (int)*pos;
	if (segment == ups.segment)
		return &ups;
	ups.ubase_num = ups.engine_num = ups.segment = ups.error_cnt = 0;
	while (ups.segment < segment)
		if (!ubus_capt_show_proc_next(&ups))
			return NULL;
	return &ups;
}

void *ubus_capt_show_proc_next(void *v)
{
	struct ubus_proc_seq *ups = v;
	struct ubus_base *ubase = ubus_bases + ups->ubase_num;
	if (ups->ubase_num >= num_ubus_bases)
		return NULL;
	ups->engine_num++;
	ups->segment++;
	if (ups->engine_num >= ubase->num_engines) {
		ups->ubase_num++;
		if (ups->ubase_num >= num_ubus_bases)
			return NULL;
		ups->engine_num = 0;
	}
	return v;
}

int ubus_capt_show_proc_show(struct seq_file *seq, void *v)
{
	struct ubus_proc_seq *ups = v;
	struct ubus_base *ubase = ubus_bases + ups->ubase_num;
	if (ups->ubase_num >= num_ubus_bases) {
		seq_printf(seq, "Total Errors: %d\n", ups->error_cnt);
		return 0;
	}
	ups->error_cnt += capt_decode_engine(ubase, ups->engine_num,
					     ups->segment, seq);
	return 0;
}

void ubus_capture_exception_print(void)
{
	if (!ubus_capt_on)
		return;
	capt_stop();
	capt_decode();
	capt_reset();
	configure_and_start();
}
EXPORT_SYMBOL(ubus_capture_exception_print);

int __init ubus_capt_map_base(int n, u64 base, u32 size, u32 segs)
{
	struct ubus_base *bp = ubus_capt_get_base(n);
	if (!bp) {
		pr_err("ubus capture invalid base number %d\n", n);
		return -1;
	}
	bp->pbase = base;
	bp->base = ioremap(base, size);
	bp->num_engines = segs;
	num_segments += segs;
	return 0;
}

int __init ubus_capt_early_init(void)
{
	if (ubus_capt_on) {
		if (capt_check() < 0) {
			pr_err("Capture Engine Check failure\n");
			return -1;
		}
		capt_reset();
		configure_and_start();
	}
	return 0;
}
