 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>

#include "ethsw.h"
#include "ethsw_priv.h"

/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_cdev"

#if CDEV_IF_SUPPORT

/* char device for switch */
struct ethsw_cdev_device {
	struct device *device;
	struct cdev cdev;
};

/* char device for port */
struct ethsw_cdev_port {
	const char *ndev_name;
	int port_id;
	struct device *device;
	struct cdev cdev;
};

/* ethsw cdev control block */
struct ethsw_cdev {
	struct ethsw_device *swdev;

	/* char device for switch */
	u32 dev_major;
	struct class *dev_class;
	struct ethsw_cdev_device *dev;

	/* char devices for ports */
	u32 port_major;
	struct class *port_class;
	struct ethsw_cdev_port *port[ETHSW_PORT_MAX];
};

static int  ethsw_cdev_device_init(struct ethsw_cdev *swcdev);
static void ethsw_cdev_device_exit(struct ethsw_cdev *swcdev);
static int  ethsw_cdev_port_init(struct ethsw_cdev *swcdev, int port_id);
static void ethsw_cdev_port_exit(struct ethsw_cdev *swcdev, int port_id);

/*
 * Char Device For Switch
 */

static long dev_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	int ret = 0;

	FUNC_ENTER();

	FUNC_LEAVE();
	return(ret);
}

static struct file_operations dev_fops = {
	.owner = THIS_MODULE,
	.unlocked_ioctl = dev_ioctl
};

static int ethsw_cdev_device_init(struct ethsw_cdev *swcdev)
{
	int ret = 0;
	dev_t devno;
	struct ethsw_cdev_device *dev;

	FUNC_ENTER();

	/* alloc char device for switch */
	dev = kzalloc(sizeof(struct ethsw_cdev_device), GFP_KERNEL);
	if (!dev) {
		LOG_ERR("cdev char device alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	swcdev->dev = dev;

	/* use 0 as device minor number */
	devno = MKDEV(swcdev->dev_major, 0);

	/* init and register cdev */
	cdev_init(&dev->cdev, &dev_fops);
	dev->cdev.owner = THIS_MODULE;

	ret = cdev_add(&dev->cdev, devno, 1);
	if (ret) {
		LOG_ERR("cdev char device register failed\n");
		goto ERROR;
	}

	/* create device */
	dev->device = device_create(
		swcdev->dev_class,
		NULL, /* no parent */
		devno,
		NULL, /* no data */
		"cethsw");

	if (IS_ERR(dev->device)) {
		LOG_ERR("cdev char device create failed\n");
		ret = PTR_ERR(dev->device);
		goto ERROR;
	}

	LOG_INFO("cdev char device initialized\n");

	FUNC_LEAVE();
	return(ret);

 ERROR:
	/* exit char device for switch properly */
	ethsw_cdev_device_exit(swcdev);

	FUNC_LEAVE();
	return(ret);
}

static void ethsw_cdev_device_exit(struct ethsw_cdev *swcdev)
{
	struct ethsw_cdev_device *dev;

	FUNC_ENTER();

	dev = swcdev->dev;
	if (!dev) {
		goto EXIT;
	}

	/* destroy device */
	if (dev->device) {
		dev_t devno;

		devno = MKDEV(swcdev->dev_major, 0);
		device_destroy(swcdev->dev_class, devno);
		dev->device = NULL;
	}

	/* unregister cdev */
	cdev_del(&dev->cdev);

	/* free char device for switch */
	kfree(dev);
	swcdev->dev = NULL;

	LOG_INFO("cdev char device exited\n");

 EXIT:
	FUNC_LEAVE();
}

/*
 * Char Device For Port
 */

static long port_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	int ret = 0;

	FUNC_ENTER();

	FUNC_LEAVE();
	return(ret);
}

static struct file_operations port_fops = {
	.owner = THIS_MODULE,
	.unlocked_ioctl = port_ioctl
};

static void ethsw_cdev_port_cb(
	void *cb_priv,
	struct ethsw_port_status *port_status)
{
	struct ethsw_cdev_port *port;

	FUNC_ENTER();

	/* get cdev port */
	port = (struct ethsw_cdev_port *)cb_priv;

	LOG_INFO("cdev port %d status update: "
			 "link_up = %d, full_duplex = %d, speed = %d\n",
			 port->port_id,
			 port_status->link_up,
			 port_status->full_duplex,
			 port_status->speed);

	FUNC_LEAVE();
}

static int ethsw_cdev_port_init(struct ethsw_cdev *swcdev, int port_id)
{
	int ret = 0;
	dev_t devno;
	char device_name[32];
	struct ethsw_device *swdev;
	struct ethsw_cdev_port *port;

	FUNC_ENTER();

	/* alloc char device for port */
	port = kzalloc(sizeof(struct ethsw_cdev_port), GFP_KERNEL);
	if (!port) {
		LOG_ERR("cdev char device alloc for port failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	swcdev->port[port_id] = port;

	/* init char device control block for port */
	port->port_id = ETHSW_PORT_ID_INVALID;

	/* use port_id as device minor number */
	devno = MKDEV(swcdev->port_major, port_id);

	/* init and register cdev */
	cdev_init(&port->cdev, &port_fops);
	port->cdev.owner = THIS_MODULE;

	ret = cdev_add(&port->cdev, devno, 1);
	if (ret) {
		LOG_ERR("cdev char device register for port failed\n");
		goto ERROR;
	}

	/* get net device name */
	swdev = swcdev->swdev;
	port->ndev_name = swdev->port_name[port_id];

	if (port->ndev_name) {
		/* use net device name with prefix 'c' */
		sprintf(device_name, "c%s", port->ndev_name);
	}
	else {
		/* use cport + port_id */
		sprintf(device_name, "cport%d", port_id);
	}

	/* create device */
	port->device = device_create(
		swcdev->port_class,
		swcdev->dev->device, /* parent is char device for switch */
		devno,
		NULL, /* no data */
		"%s", device_name);

	if (IS_ERR(port->device)) {
		LOG_ERR("cdev char device create for port failed\n");
		ret = PTR_ERR(port->device);
		goto ERROR;
	}

	LOG_INFO("cdev char device for port %d (%s) initialized\n",
			 port_id, device_name);

	/* connect to switch port */
	ret = ethsw_port_connect(
		port->ndev_name,
		&port->port_id,
		ethsw_cdev_port_cb,
		port);

	if (ret) {
		LOG_ERR("cdev port %d connect failed\n", port_id);
		goto ERROR;
	}

	/* enable port */
	ret = ethsw_port_enable(port->port_id);
	if (ret) {
		LOG_ERR("cdev port %d enable failed\n", port_id);
		goto ERROR;
	}

	FUNC_LEAVE();
	return(ret);

 ERROR:
	/* exit char device for switch properly */
	ethsw_cdev_port_exit(swcdev, port_id);

	FUNC_LEAVE();
	return(ret);
}

static void ethsw_cdev_port_exit(struct ethsw_cdev *swcdev, int port_id)
{
	struct ethsw_cdev_port *port;

	FUNC_ENTER();

	port = swcdev->port[port_id];
	if (!port) {
		goto EXIT;
	}

	/* disconnect from switch port */
	if (port->port_id != ETHSW_PORT_ID_INVALID) {
		ethsw_port_disconnect(port->port_id);
		port->port_id = ETHSW_PORT_ID_INVALID;
	}

	/* destroy device */
	if (port->device) {
		dev_t devno;

		devno = MKDEV(swcdev->port_major, port_id);
		device_destroy(swcdev->port_class, devno);
		port->device = NULL;
	}

	/* unregister cdev */
	cdev_del(&port->cdev);

	/* free char device for switch */
	kfree(port);
	swcdev->port[port_id] = NULL;

	LOG_INFO("cdev char device for port %d exited\n", port_id);

 EXIT:
	FUNC_LEAVE();
}

/*
 * CDEV Sub-module
 */

int ethsw_cdev_init(struct ethsw_device *swdev)
{
	int ret = 0;
	int i;
	dev_t devno;
	struct ethsw_cdev *swcdev;

	FUNC_ENTER();

	/* alloc ethsw cdev control block */
	swcdev = kzalloc(sizeof(struct ethsw_cdev), GFP_KERNEL);
	if (!swcdev) {
		LOG_ERR("Ethsw cdev control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	swdev->swcdev = swcdev;
	swcdev->swdev = swdev;

	/* allocate char device number for switch */
	ret = alloc_chrdev_region(&devno, 0, 1, "cethsw");
	if (ret) {
		LOG_ERR("cdev device number alloc failed\n");
		goto ERROR;
	}

	swcdev->dev_major = MAJOR(devno);

	/* create char device class for switch */
	swcdev->dev_class = class_create(THIS_MODULE, "cethsw");
	if (IS_ERR(swcdev->dev_class)) {
		LOG_ERR("cdev device class create failed\n");
		ret = PTR_ERR(swcdev->dev_class);
		goto ERROR;
	}

	/* init char device for switch */
	ret = ethsw_cdev_device_init(swcdev);
	if (ret) {
		LOG_ERR("cdev char device init failed\n");
		goto ERROR;
	}

	/* allocate char device numbers for ports */
	ret = alloc_chrdev_region(&devno, 0, ETHSW_PORT_MAX, "cport");
	if (ret) {
		LOG_ERR("cdev device number alloc for ports failed\n");
		goto ERROR;
	}

	swcdev->port_major = MAJOR(devno);

	/* create char device class for ports */
	swcdev->port_class = class_create(THIS_MODULE, "cport");
	if (IS_ERR(swcdev->port_class)) {
		LOG_ERR("cdev device class create for ports failed\n");
		ret = PTR_ERR(swcdev->port_class);
		goto ERROR;
	}

	/* init char devices for ports */
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		if (swdev->port[i].type == PHY_PORT) {
			ret = ethsw_cdev_port_init(swcdev, i);
			if (ret) {
				LOG_ERR("cdev char device init for ports failed\n");
				goto ERROR;
			}
		}
	}

	LOG_INFO("Ethsw cdev initialized\n");

	FUNC_LEAVE();
	return(ret);

 ERROR:
	/* exit cdev properly */
	ethsw_cdev_exit(swdev);

	FUNC_LEAVE();
	return(ret);
}

void ethsw_cdev_exit(struct ethsw_device *swdev)
{
	int i;
	dev_t devno;
	struct ethsw_cdev *swcdev;

	FUNC_ENTER();

	swcdev = swdev->swcdev;
	if (!swcdev) {
		goto EXIT;
	}

	/* exit char devices for ports */
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		if (swcdev->port[i]) {
			ethsw_cdev_port_exit(swcdev, i);
			swcdev->port[i] = NULL;
		}
	}

	/* destroy char device class for ports */
	if (swcdev->port_class) {
		class_destroy(swcdev->port_class);
	}

	/* unregister char device numbers for ports */
	devno = MKDEV(swcdev->port_major, 0);
	unregister_chrdev_region(devno, ETHSW_PORT_MAX);

	/* exit char device for switch */
	if (swcdev->dev) {
		ethsw_cdev_device_exit(swcdev);
		swcdev->dev = NULL;
	}

	/* destroy char device class for switch */
	if (swcdev->dev_class) {
		class_destroy(swcdev->dev_class);
	}

	/* unregister char device numbers for ports */
	devno = MKDEV(swcdev->dev_major, 0);
	unregister_chrdev_region(devno, 1);

	/* free ethsw cdev control block */
	kfree(swcdev);
	swdev->swcdev = NULL;

	LOG_INFO("Ethsw cdev exited\n");

 EXIT:
	FUNC_LEAVE();
}

#endif
