 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
/****************************************************************************
 *	Filename:       bcmmbdma.c
 *	Author:         Mark Newcomer/Mike Siweke
 *	Creation Date:  5/6/2013
 *	PURPOSE: Common Functions for the Unimac MBDMA driver.
 *****************************************************************************/

#include <uapi/linux/ethtool.h>
#include <uapi/linux/if_link.h>
#include "bcmunimac_priv.h"
#include "bcmmbdma.h"

#ifdef BCMUNIMAC_MBDMA_INTERRUPT_SUPPORT
static irqreturn_t mdbma_isr(void);
/*
 * Initialize the MBDMA interrupts. These are mostly used for
 * debugging error conditions.
 *
 * @author newcomer (6/19/2013)
 */
void mbdma_interrupt_init(void)
{
	/* TODO: interrupts not yet implemented */
}

/*
 * Interrupt handler for all MBDMA specific events. Note, this
 * is not the Unimac handler - only DMA interupts.
 *
 * @author newcomer (6/20/2013)
 *
 * @return u32
 */
static irqreturn_t mdbma_isr(void);
{
	/* TODO: interrupts not yet implemented */
	return(0);
}
#endif /* BCMUNIMAC_MBDMA_INTERRUPT_SUPPORT */

/*
 * Initialize the MBDMA device registers.
 *
 * @author newcomer (6/19/2013)
 *
 * @param punimac_device -  pointer to UNIMAC register base.
 */
void mbdma_init(void *dev)
{
	u32 val;
	struct unimac_device *udev = dev;
	void __iomem *base = udev->reg_base;
	struct fpm_hw_info fpm_hwinfo;
	int ret = 0;

	pr_debug("mbdma base = %p\n", base);

	/* Ask the FPM Client about his register layout, to inform the MBDMA. */
	ret = fpm_get_hw_info(&fpm_hwinfo);
	if (ret)
		pr_err("mbdma_init: fpm_get_hw_info() returned = %d\n",ret);

	pr_debug("mbdma_init: FPM Pool0 mem base = %08x\n",
			(u32)fpm_hwinfo.pool_base[0]);
	pr_debug("mbdma_init: FPM Pool1 mem base = %08x\n",
			(u32)fpm_hwinfo.pool_base[1]);

	pr_debug("test read = %08x\n", __raw_readl(base + MBDMA_STATUS));

	/* Add RX Offset from FPM driver */
	val = __raw_readl(base + MBDMA_RXOFF);
	SET_FLD(val, MBDMA_RXOFF_BUF_OFF, udev->fpm_rxbuffer_offset);
	__raw_writel(val, base + MBDMA_RXOFF);

	/* Setup the base addresses required to talk to the FPM HW */
	if(udev->use_scb) {
		pr_debug("using SCB DDR bus\n");
		__raw_writel(fpm_hwinfo.pool_base[0], base + MBDMA_SCBADDRLO_0);
		__raw_writel(0, base + MBDMA_SCBADDRHI_0);
		__raw_writel(fpm_hwinfo.pool_base[1], base + MBDMA_SCBADDRLO_1);
		__raw_writel(0, base + MBDMA_SCBADDRHI_1);

		/* Set up the token allocation engine */
		val = __raw_readl(base + MBDMA_SCBCNTL);
		SET_FLD(val, MBDMA_SCBCNTL_USE_WR_REPLY, 1);
		SET_FLD(val, MBDMA_SCBCNTL_FORCE_SINGLE_QUEUE, 0);
		SET_FLD(val, MBDMA_SCBCNTL_CLIENT_INIT, 1);
		SET_FLD(val, MBDMA_SCBCNTL_COHERENT, 0);
		SET_FLD(val, MBDMA_SCBCNTL_ENABLE,
			(RX0_SCB_ENABLE | TX0_SCB_ENABLE | TX1_SCB_ENABLE));
		__raw_writel(0, base + MBDMA_SCBCNTL);
	} else {
		__raw_writel(fpm_hwinfo.pool_base[0], base + MBDMA_BUFFBASE0);
		__raw_writel(fpm_hwinfo.pool_base[1], base + MBDMA_BUFFBASE1);
	}

	__raw_writel(fpm_hwinfo.alloc_dealloc[fpm_one_buffer],
		     base + MBDMA_TOKENADDR256);
	__raw_writel(fpm_hwinfo.alloc_dealloc[fpm_two_buffers],
		     base + MBDMA_TOKENADDR512);
	__raw_writel(fpm_hwinfo.alloc_dealloc[fpm_three_buffers],
		     base + MBDMA_TOKENADDR1K);
	__raw_writel(fpm_hwinfo.alloc_dealloc[fpm_four_buffers],
		     base + MBDMA_TOKENADDR2K);
	__raw_writel(fpm_hwinfo.alloc_dealloc[fpm_four_buffers],
		     base + MBDMA_TOKENADDRFREE);

	/* First flush any cached tokens */
	__raw_writel(MBDMA_GLOBALCTL_FLUSH_CACHE_MASK, base + MBDMA_GLOBALCTL);
	udelay(100);

	/* Clear the register before going on to turn off the flush command */
	__raw_writel(0, base + MBDMA_GLOBALCTL);

	/* Set up the token allocation engine */
	val = __raw_readl(base + MBDMA_GLOBALCTL);
	SET_FLD(val, MBDMA_GLOBALCTL_FLUSH_CACHE, 0);
	SET_FLD(val, MBDMA_GLOBALCTL_LAN_TX_MSG_ID_3W, MBDMA_TX_ID_3WORD);
	SET_FLD(val, MBDMA_GLOBALCTL_LAN_TX_MSG_ID_2W, LANTX_MSG);
	__raw_writel(val, base + MBDMA_GLOBALCTL);
	pr_debug("global_ctl = 0x%x\n", __raw_readl(base + MBDMA_GLOBALCTL));

	/* Load up the Token Control register via shadow */
	val = __raw_readl(base + MBDMA_TOKCACHECTL);
	SET_FLD(val, MBDMA_TOKCACHECTL_BUFF_SIZE,
		fpm_hwinfo.chunk_size == 256 ?
		MBDMA_BUFF_SIZE_256 : MBDMA_BUFF_SIZE_512);
	SET_FLD(val, MBDMA_TOKCACHECTL_FREE_ENABLE, 1);
	SET_FLD(val, MBDMA_TOKCACHECTL_FREE_MAX_BURST, MBDMA_FREE_BURST);
	SET_FLD(val, MBDMA_TOKCACHECTL_FREE_THRESH, MBDMA_FREE_THRESH);
	__raw_writel(val, base + MBDMA_TOKCACHECTL);
	pr_debug("token control = 0x%x\n", __raw_readl(base + MBDMA_TOKCACHECTL));

	/* Set alloc burst size */
	val = 0;
	SET_FLD(val, MBDMA_TOKCACHECTL2_BURST256, MBDMA_ALLOC_BURST);
	SET_FLD(val, MBDMA_TOKCACHECTL2_BURST512, MBDMA_ALLOC_BURST);
	SET_FLD(val, MBDMA_TOKCACHECTL2_BURST1K, MBDMA_ALLOC_BURST);
	SET_FLD(val, MBDMA_TOKCACHECTL2_BURST2K, MBDMA_ALLOC_BURST);
	__raw_writel(val, base + MBDMA_TOKCACHECTL2);

	/* Enable all 4 allocation FIFOs. */
	__raw_writel(0xf, base + MBDMA_TOKCACHECTL3);

	/*
	 * Set the RX message burst size.  Set the LAN RX message ID, and leave
	 * the MAC ID as 0.  The other bits in this register should be left as
	 * 0.
	 */
	val = __raw_readl(base + MBDMA_CHANCTRL00);
	SET_FLD(val, MBDMA_CHANCTRL00_MAX_BURST, udev->burst_size >> 3);
	__raw_writel(val, base + MBDMA_CHANCTRL00);

	/* Add RX Offset from FPM driver */
	val = __raw_readl(base + MBDMA_RXOFF);
	SET_FLD(val, MBDMA_RXOFF_BUF_OFF, udev->fpm_rxbuffer_offset);
	__raw_writel(val, base + MBDMA_RXOFF);

	/*
	 * Set the TX Status message ID, and set the message burst size.
	 * Leave the max outstanding read requests at the default of 1.
	 */
	val = __raw_readl(base + MBDMA_CHANCTRL01);
	SET_FLD(val, MBDMA_CHANCTRL01_MAX_BURST, udev->burst_size >> 3);
	SET_FLD(val, MBDMA_CHANCTRL01_MAX_REQS, TX_MAX_REQ);
	__raw_writel(val, base + MBDMA_CHANCTRL01);
	val = __raw_readl(base + MBDMA_CHANCTRL02);
	SET_FLD(val, MBDMA_CHANCTRL02_MAX_BURST, udev->burst_size >> 3);
	SET_FLD(val, MBDMA_CHANCTRL02_MAX_REQS, TX_MAX_REQ);
	__raw_writel(val, base + MBDMA_CHANCTRL02);

	/* Clear all interrupts. */
	__raw_writel(0xffffffff, base + MBDMA_STATUS);

#ifdef BCMUNIMAC_MBDMA_INTERRUPT_SUPPORT
	val = __raw_readl(base + MBDMA_STATUS);
	SET_FLD(val, MBDMA_STATUS_TX_MSGQ_OVERFLOW, 1);
	SET_FLD(val, MBDMA_STATUS_INVALID_TX_MSG_INTR, 1);
	SET_FLD(val, MBDMA_STATUS_UBUS_ERROR, 1);
	SET_FLD(val, MBDMA_STATUS_TOKEN_RD_ERROR, 1);
	SET_FLD(val, MBDMA_STATUS_INVALID_TOKEN, 1);
	SET_FLD(val, MBDMA_STATUS_ALLOC_FIFO_EMPTY_256, 1);
	SET_FLD(val, MBDMA_STATUS_ALLOC_FIFO_EMPTY_512, 1);
	SET_FLD(val, MBDMA_STATUS_ALLOC_FIFO_EMPTY_1K, 1);
	SET_FLD(val, MBDMA_STATUS_ALLOC_FIFO_EMPTY_2K, 1);
	SET_FLD(val, MBDMA_STATUS_FREE_FIFO_FULL, 1);
#if 0
		SET_FLD(val, GblIntrMask, mbdma_block_mask, 1);
		SET_FLD(mbdma_regs, val, IntrMask, mbdma_block_mask,
			GMAC_MBDMA_INT_ENABLE);
#endif
	/*
	 * The new mask register has the same layout as the status register, so
	 * enable the interrupts we just cleared.
	 */
	__raw_writel(val, base + MBDMA_INTRMASK);

	pr_debug("status = 0x%x\n", __raw_readl(base + MBDMA_STATUS));
#endif
}

/*
 * Prints the current status of the MBDMA registers, including
 * errors and interrupts.
 * NOTE:  DiagIntr, FreeFifoEmpty and AllocFifoEmpty are not processed;
 * these are typically asserted, and are not errors.
 *
 * @author newcomer (6/20/2013)
 */
void mbdma_dump_status(void *dev)
{
	u32 status;
	u32 tmp;
	u32 val;
	struct unimac_device *udev = dev;
	void __iomem *base = udev->reg_base;

	status = __raw_readl(base + MBDMA_STATUS);

	pr_debug("mbdma_init: status = 0x%x\n", status);

	tmp = GET_FLD(status, MBDMA_STATUS_TX_MSGQ_OVERFLOW);
	if (tmp == 1) {
		pr_err("Tx MSG Fifo Overflow.\n");
		/*
		* NOTE: This interrupt cannot be cleared manually.
		* It is an OR of the Tx overflow bits from all the Tx channels,
		* so the overflow must be cleared in specific channel.
		* Note - read/modify/write is appropriate for these registers.
		* The lower 8 bits need to be preserved, and the remaining
		*	bits are read-only or reserved.
		*/
		val = __raw_readl(base + MBDMA_CHANCTRL01_2);
		SET_FLD(val, MBDMA_CHANCTRL01_2_TX_MSGQ_OVR, 1);
		__raw_writel(val, base + MBDMA_CHANCTRL01_2);
		val = __raw_readl(base + MBDMA_CHANCTRL02_2);
		SET_FLD(val, MBDMA_CHANCTRL02_2_TX_MSGQ_OVR, 1);
		__raw_writel(val, base + MBDMA_CHANCTRL02_2);
	}

	tmp = GET_FLD(status, MBDMA_STATUS_INVALID_TX_MSG_INTR);
	if (tmp == 1)
		pr_err("Invalid Tx message interrupt.\n");

	tmp = GET_FLD(status, MBDMA_STATUS_UBUS_ERROR);
	if (tmp == 1)
		pr_err("Ubus error interrupt.\n");

	tmp = GET_FLD(status, MBDMA_STATUS_TOKEN_RD_ERROR);
	if (tmp == 1)
		pr_err("Invalid read token interrupt");

	tmp = GET_FLD(status, MBDMA_STATUS_INVALID_TOKEN);
	if (tmp == 1)
		pr_err("Invalid FPM token interrupt.\n");

	tmp = GET_FLD(status, MBDMA_STATUS_ALLOC_FIFO_EMPTY_256);
	if (tmp == 1) {
		pr_err("Alloc256 Fifo Empty.\n");
		pr_err("Token Cache Ctrl = %08x\n",
			   __raw_readl(base + MBDMA_TOKCACHECTL3));
		/* We need to re-enable the MBDMA Allocator. */
		__raw_writel(1<<0, base + MBDMA_TOKCACHECTL3);
	}

	tmp = GET_FLD(status, MBDMA_STATUS_ALLOC_FIFO_EMPTY_512);
	if (tmp == 1) {
		pr_err("Alloc512 Fifo Empty.\n");
		pr_err("Token Cache Ctrl = %08x\n",
			   __raw_readl(base + MBDMA_TOKCACHECTL3));
		__raw_writel(1<<1, base + MBDMA_TOKCACHECTL3);
	}

	tmp = GET_FLD(status, MBDMA_STATUS_ALLOC_FIFO_EMPTY_1K);
	if (tmp == 1) {
		pr_err("Alloc1k Fifo Empty.\n");
		pr_err("Token Cache Ctrl = %08x\n",
			   __raw_readl(base + MBDMA_TOKCACHECTL3));
		__raw_writel(1<<2, base + MBDMA_TOKCACHECTL3);
	}

	tmp = GET_FLD(status, MBDMA_STATUS_ALLOC_FIFO_EMPTY_2K);
	if (tmp == 1) {
		pr_err("Alloc2k Fifo Empty.\n");
		pr_err("Token Cache Ctrl = %08x\n",
			   __raw_readl(base + MBDMA_TOKCACHECTL3));
		__raw_writel(1<<3, base + MBDMA_TOKCACHECTL3);
	}

	tmp = GET_FLD(status, MBDMA_STATUS_FREE_FIFO_FULL);
	if (tmp == 1)
		pr_err("Free Fifo Full.\n");

	/*
	 * Clear the interrupts we processed.  Note that this register has a
	 * mix of mask bits and W1C IRQ status bits.  This write clears the
	 * status bits, and preserves the mask bits.
	 */
	__raw_writel(status, base + MBDMA_STATUS);
}

/*
 * This function is called when the MBDMA is shutting down. It
 * will suspend the DMA and then send any cached tokens back to
 * the free pool manager.
 *
 * @author newcomer (6/19/2013)
 */
void mbdma_flush_tokens(void *dev)
{
	u32 val;
	struct unimac_device *udev = dev;
	void __iomem *base = udev->reg_base;

	val = __raw_readl(base + MBDMA_CHANCTRL00);
	SET_FLD(val, MBDMA_CHANCTRL00_START_STOP_FLUSH, TOKEN_FLUSH);
	__raw_writel(val, base + MBDMA_CHANCTRL00);
	val = __raw_readl(base + MBDMA_CHANCTRL01);
	SET_FLD(val, MBDMA_CHANCTRL01_START_STOP_FLUSH, TOKEN_FLUSH);
	__raw_writel(val, base + MBDMA_CHANCTRL01);
	val = __raw_readl(base + MBDMA_CHANCTRL02);
	SET_FLD(val, MBDMA_CHANCTRL02_START_STOP_FLUSH, TOKEN_FLUSH);
	__raw_writel(val, base + MBDMA_CHANCTRL02);

	/* Give HW a chance to finish its thing to avoid token leak */
	udelay(10);
	val = __raw_readl(base + MBDMA_GLOBALCTL);
	SET_FLD(val, MBDMA_GLOBALCTL_FLUSH_CACHE, 1);
	__raw_writel(val, base + MBDMA_GLOBALCTL);
	udelay(10);
}

#ifdef BCMUNIMAC_USE_POWERCONTROL
/*
 * Turns power to the MBDMA off.
 *
 * @author newcomer (6/19/2013)
 */
void mbdma_power_down(void)
{
	return;
}

/*
 * Turns power to the MBDMA block on.
 *
 * @author newcomer (6/19/2013)
 */
void mbdma_power_up(void)
{
	return;
}
#endif
