/******************************************************************************
 *  Copyright (C) 2019-2020 Broadcom.
 *  The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 *  This program is the proprietary software of Broadcom and/or its licensors,
 *  and may only be used, duplicated, modified or distributed pursuant to
 *  the terms and conditions of a separate, written license agreement executed
 *  between you and Broadcom (an "Authorized License").  Except as set forth in
 *  an Authorized License, Broadcom grants no license (express or implied),
 *  right to use, or waiver of any kind with respect to the Software, and
 *  Broadcom expressly reserves all rights in and to the Software and all
 *  intellectual property rights therein. IF YOU HAVE NO AUTHORIZED LICENSE,
 *  THEN YOU HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD
 *  IMMEDIATELY NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 *  Except as expressly set forth in the Authorized License,
 *
 *  1.     This program, including its structure, sequence and organization,
 *  constitutes the valuable trade secrets of Broadcom, and you shall use all
 *  reasonable efforts to protect the confidentiality thereof, and to use this
 *  information only in connection with your use of Broadcom integrated circuit
 *  products.
 *
 *  2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED
 *  "AS IS" AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS
 *  OR WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH
 *  RESPECT TO THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL
 *  IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR
 *  A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET
 *  ENJOYMENT, QUIET POSSESSION OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME
 *  THE ENTIRE RISK ARISING OUT OF USE OR PERFORMANCE OF THE SOFTWARE.
 *
 *  3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM
 *  OR ITS LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL,
 *  INDIRECT, OR EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY
 *  RELATING TO YOUR USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM
 *  HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN
 *  EXCESS OF THE AMOUNT ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1,
 *  WHICHEVER IS GREATER. THESE LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY
 *  FAILURE OF ESSENTIAL PURPOSE OF ANY LIMITED REMEDY.
 ******************************************************************************/
#ifndef __TRACE_H_
#define __TRACE_H_

#include <stdint.h>

#define TRC_MAGIC_WORD		0x54524342  /* trace buffer "TRCB" */
#define TRC_MINOR_VER		5
#define TRC_MAJOR_VER		0
#define TRC_EVENT_PARAMS_MAX	20

/* event id, type conversion macros */
#define EVID(__typ)		(__typ << 4)
#define EVID_FLAG(__typ)	(1 << (__typ & 0x0f))
#define EVID_TYPE(__evid)	((__evid & 0xf0) >> 4)

#define OEN_ARM_START			0
#define OEN_SIP_START			2
#define OEN_OEM_START			3
#define OEN_STD_START			4	/* Standard Service Calls */

/*******************************************************************************
 * Bit definitions inside the function id as per the SMC calling convention
 ******************************************************************************/
#define FUNCID_OEN_SHIFT		24
#define FUNCID_OEN_MASK			0x3f
#define FUNCID_OEN_WIDTH		6

#define GET_SMC_OEN(id)			(((id) >> FUNCID_OEN_SHIFT) & \
					 FUNCID_OEN_MASK)

#define BRCMSTB_OEN_OEM			OEN_OEM_START
#define BRCMSTB_SVC_MOD_RESERVED        0x00
#define BRCMSTB_SVC_MOD_DVFS            0x01
#define BRCMSTB_SVC_MOD_ASTRA           0x02
#define BRCMSTB_SVC_MOD_LINUX           0x03
#define BRCMSTB_SVC_MOD_MBOX            0x04
#define BRCMSTB_SVC_MOD_CRYPTO          0x05

/* Brcmstb service FN consists of module bit[15:8] and func bit[7:0] */
#define BRCMSTB_SVC_MOD_SHIFT           8
#define BRCMSTB_SVC_MOD_MASK            0xFF

/* Macros to extract brcmstb service MOD/FUNC from FN/FID */
#define BRCMSTB_SVC_MOD(fx)             (((fx) >> BRCMSTB_SVC_MOD_SHIFT)  & \
					 BRCMSTB_SVC_MOD_MASK)

enum event_type {
	SCMI,
	PSCI,
	AVS,
	CPUFREQ,
	STRING_IO,
	SMC,
	SGI_INTR,
	EV_MAX
};

enum event_attribute {
	EV_LOG,
	EV_ENTER,
	EV_EXIT,
};

struct trace_header {
	uint32_t magic_word;
	uint16_t major_ver;
	uint16_t minor_ver;
	uint32_t trcbuf_size;
	uint32_t trcbuf_offset;
	uint32_t trcbuf_head_offset;
	uint32_t trcbuf_tail_offset;
	uint32_t trcbuf_head_seqnum;
	uint8_t	 event_sz;
	uint8_t  event_params_max;
	uint16_t flags;
	uint32_t clk_freq;
	uint32_t reserved[6];
	uint32_t crc32;
} __attribute__ ((aligned(64)));

struct ts48 {
	uint16_t t[3];
};

struct event {
	struct ts48 ts_event;
	uint16_t ev_attrib;	/* event trace point attribute */
	uint16_t ev_desc;	/* event specific data */
	uint8_t event_id;	/* type source encoded in 1 byte */
	uint8_t num_params;
	uint32_t params[TRC_EVENT_PARAMS_MAX];
} __attribute__ ((aligned(16)));

struct trace_info {
	struct trace_header *trc_hdr;
	struct event *start_ev_ptr;
	struct event *end_ev_ptr;
	struct ts48 cur_ev_ts;
	struct event *head;
	struct event *tail;
	uint32_t trace_on;
};

static inline uint64_t uint48_get(const uint16_t *a16)
{
	return a16[0] | (uint32_t) a16[1] << 16 | (uint64_t) a16[2] << 32;
}

static inline void uint48_set(uint16_t *a16, uint64_t value)
{
	a16[0] = (uint16_t) value;
	a16[1] = (uint16_t) (value >> 16);
	a16[2] = (uint16_t) (value >> 32);
}

static inline uint8_t trace_header_size(void)
{
	return sizeof(struct trace_header);
}

static inline uint8_t trace_event_size(struct trace_header const *th)
{
	return th->event_sz;
}

static inline uint8_t trace_event_max_params(struct trace_header *th)
{
	return th->event_params_max;
}

void prn_event_decoded(struct trace_header *th, struct event *evp,
		       uint32_t seqnum);
struct trace_info *get_trace_info(void);

/* Clock name decode functions */
int trace_decode_clk_name_init(const char *filename);
void trace_decode_clk_name_unint(void);
void trace_decode_clk_name(unsigned int clk_id, char* buf, size_t bufsize);
#endif /* _TRACE_H_ */
