/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2017 Broadcom.  All rights reserved.
* The term “Broadcom” refers to Broadcom Limited and/or its subsidiaries.
*
* Unless you and Broadcom execute a separate written software license
* agreement governing use of this software, this software is licensed to
* you under the terms of the GNU General Public License version 2 (the
* "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
* the following added to such license:
*
* As a special exception, the copyright holders of this software give you
* permission to link this software with independent modules, and to copy
* and distribute the resulting executable under terms of your choice,
* provided that you also meet, for each linked independent module, the
* terms and conditions of the license of that module. An independent
* module is a module which is not derived from this software. The special
* exception does not apply to any modifications of the software.
*
* Notwithstanding the above, under no circumstances may you combine this
* software in any way with any other Broadcom software provided under a
* license other than the GPL, without Broadcom's express prior written
* consent.
*
****************************************************************************
* Author: Peter Sulc <peter.sulc@broadcom.com>
* Filename: ubus_capture_proc.c
* Description: Proc controls for ubus capture
****************************************************************************/

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/ctype.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <asm/exception.h>

#include "ubus_capture.h"

#define PROC_DIR		"bus/ubus"
#define STATUS_FILE		"status"
#define SHOW_FILE		"show"
#define STOP_FILE		"stop"
#define START_FILE		"start"

struct ubus_proc_val {
	const char *fname;
	const char *pform;
	u32 *value;
	struct proc_dir_entry *proc_file;
};

static struct ubus_proc_val ubus_proc_val_table[] = {
	{
		.fname = "address_start",
		.pform = "0x%08x\n",
		.value = &ubus_capt_addr_start
	},
	{
		.fname = "address_end",
		.pform = "0x%08x\n",
		.value = &ubus_capt_addr_end
	},
	{
		.fname = "address_exclude",
		.pform = "%d\n",
		.value = &ubus_capt_addr_exclude
	},
	{
		.fname = "pid_start",
		.pform = "%d\n",
		.value = &ubus_capt_pid_start
	},
	{
		.fname = "pid_end",
		.pform = "%d\n",
		.value = &ubus_capt_pid_end
	},
	{
		.fname = "pid_exclude",
		.pform = "%d\n",
		.value = &ubus_capt_pid_exclude
	},
};

static struct proc_dir_entry *proc_dir;
static struct proc_dir_entry *status_file;
static struct proc_dir_entry *show_file;
static struct proc_dir_entry *start_file;
static struct proc_dir_entry *stop_file;

static int   status_proc_open(struct inode *inode, struct file *file);
static int   status_proc_show(struct seq_file *seq, void *v);

static int   show_proc_open(struct inode *inode, struct file *file);
static void *show_proc_start(struct seq_file *seq, loff_t *pos);
static void  show_proc_stop(struct seq_file *seq, void *v);
static void *show_proc_next(struct seq_file *seq, void *v, loff_t *pos);

static ssize_t value_read(struct file *, char __user *, size_t, loff_t *);
static ssize_t value_write(struct file *, const char __user *, size_t, loff_t *);
static ssize_t stop_write(struct file *, const char __user *, size_t, loff_t *);
static ssize_t start_write(struct file *,const char __user *, size_t, loff_t *);


static const struct file_operations status_fops = {
	.owner		= THIS_MODULE,
	.open		= status_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= seq_release,
};

static const struct seq_operations show_seq_ops = {
	.start	= show_proc_start,
	.stop	= show_proc_stop,
	.next	= show_proc_next,
	.show	= ubus_capt_show_proc_show,
};

static const struct file_operations show_fops = {
	.owner		= THIS_MODULE,
	.open		= show_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= seq_release,
};

static const struct file_operations value_fops = {
	.owner		= THIS_MODULE,
	.read		= value_read,
	.write		= value_write,
};

static const struct file_operations stop_fops = {
	.owner		= THIS_MODULE,
	.write		= stop_write,
};

static const struct file_operations start_fops = {
	.owner		= THIS_MODULE,
	.write		= start_write,
};

static int status_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, status_proc_show, NULL);
}

static int status_proc_show(struct seq_file *seq, void *v)
{
	seq_printf(seq, "UBUS Capture is %s\n",
		   ubus_capt_on ? "on" : "off");
	seq_printf(seq, "Address range: 0x%08x - 0x%08x %s\n",
		   ubus_capt_addr_start, ubus_capt_addr_end,
		   ubus_capt_addr_exclude ? "exclusive" : "inclusive");
	seq_printf(seq, "Pid range: %d - %d %s\n",
		   ubus_capt_pid_start, ubus_capt_pid_end,
		   ubus_capt_pid_exclude ? "exclusive" : "inclusive");
	return 0;
}

static int   show_proc_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &show_seq_ops);
}

static void *show_proc_start(struct seq_file *seq, loff_t *pos)
{
	return ubus_capt_show_proc_start(pos);
}

static void show_proc_stop(struct seq_file *seq, void *v)
{
}

static void *show_proc_next(struct seq_file *seq, void *v, loff_t *pos)
{
	void *ret = ubus_capt_show_proc_next(v);
	if (ret)
		*pos += 1;
	return ret;
}

/* we need to do this because kstrtoul requires null termination */
static int ascii_to_ul(const char __user *src, int size, unsigned long *val)
{
	char buffer[32];
	const char __user *s = src;
	const char __user *send = s + size;
	char *d = buffer;
	char *dend = buffer + sizeof(buffer) - 1;

	while (!isxdigit(*s) && s < send)
		s++;
	while ((isxdigit(*s) || (tolower(*s) == 'x')) &&
	       ((s < send) && (d < dend)))
		*d++ = *s++;
	*d = 0;
	if (kstrtoul(buffer, 0, val)) {
		pr_err("parse error\n");
		return -EINVAL;
	}
	return (int)(s - src);
}

static struct ubus_proc_val *get_proc_val(const char *name)
{
	int i;
	int num = sizeof(ubus_proc_val_table) / sizeof(struct ubus_proc_val);

	for (i = 0; i < num; i++) {
		if (strcmp(name, ubus_proc_val_table[i].fname) == 0)
			return &ubus_proc_val_table[i];
	}
	return NULL;
}

static ssize_t value_read(struct file *f, char __user *buf,
			  size_t size, loff_t *ppos)
{
	int num;
	struct ubus_proc_val *pv;

	if (*ppos)
		return 0;

	pv = get_proc_val(f->f_path.dentry->d_name.name);
	if (!pv)
		return 0;

	snprintf(buf, size, pv->pform, *pv->value);
	num = strlen(buf);
	*ppos += num;
	return num;
}

static ssize_t value_write(struct file *f, const char __user *buf,
			  size_t size, loff_t *ppos)
{
	unsigned long value;
	int cnt;
	struct ubus_proc_val *pv = get_proc_val(f->f_path.dentry->d_name.name);
	if (!pv)
		return 0;

	cnt = ascii_to_ul(buf, size, &value);
	if (cnt <= 0)
		return -EINVAL;

	*pv->value = value;

	*ppos += size;
	return size;
}

static ssize_t stop_write(struct file *f, const char __user *buf,
			  size_t size, loff_t *ppos)
{
	ubus_capt_stop();
	return size;
}

static ssize_t start_write(struct file *f, const char __user *buf,
			   size_t size, loff_t *ppos)
{
	ubus_capt_start();
	return size;
}


static int __init ubus_capt_proc_init(void)
{
	int i, num;
	struct ubus_proc_val *pv;

	proc_dir = proc_mkdir(PROC_DIR, NULL);
	if (!proc_dir) {
		pr_err("Failed to create PROC directory %s\n", PROC_DIR);
		return -EIO;
	}
	status_file = proc_create(STATUS_FILE, S_IRUGO, proc_dir, &status_fops);
	if (!status_file) {
		pr_err("Failed to create %s\n", STATUS_FILE);
		return -EIO;
	}
	show_file = proc_create(SHOW_FILE, S_IRUGO, proc_dir, &show_fops);
	if (!show_file) {
		pr_err("Failed to create %s\n", SHOW_FILE);
		return -EIO;
	}
	stop_file = proc_create(STOP_FILE, S_IWUGO, proc_dir, &stop_fops);
	if (!stop_file) {
		pr_err("Failed to create %s\n", STOP_FILE);
		return -EIO;
	}
	start_file = proc_create(START_FILE, S_IWUGO, proc_dir, &start_fops);
	if (!start_file) {
		pr_err("Failed to create %s\n", START_FILE);
		return -EIO;
	}
	num = sizeof(ubus_proc_val_table) / sizeof(struct ubus_proc_val);

	for (i = 0, pv = &ubus_proc_val_table[0]; i < num; i++, pv++) {
		pv->proc_file = proc_create(pv->fname, S_IRUGO|S_IWUGO,
					    proc_dir, &value_fops);
		if (!pv->proc_file) {
			pr_err("Failed to create %s\n", pv->fname);
			return -EIO;
		}
	}
	return 0;
}

late_initcall(ubus_capt_proc_init);
