 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2015-2019 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/

#ifndef __PROC_CMD__
#define __PROC_CMD__
#include <linux/version.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/sched.h>
#include <linux/seq_file.h>
#include <linux/panic_notifier.h>
#include <linux/module.h>
#include <linux/fsnotify.h>
#include <linux/uio.h>
#include <bcmport.h>

#define PROC_CMD_STRING_SIZE	200
#define PROC_CMD_ARGS_MAX	20
#define PROC_CMD_TABLE_SIZE_MAX	40

#define pr_seq(s, ...) \
do { \
	if (s) \
		seq_printf(s,  __VA_ARGS__); \
	else \
		pr_emerg(__VA_ARGS__); \
} while (0)

#define pr_seq_cont(s, ...) \
do { \
	if (s) \
		seq_printf(s,  __VA_ARGS__); \
	else \
		pr_cont(__VA_ARGS__); \
} while (0)

#define PROC_CMD_INIT(str, func) { .name = str, .do_command = func, .do_help = func##_help }
#define PROC_CMD_DATA_INIT(str, func) { .name = str, .do_command_data = func, .do_help = func##_help }

struct proc_cmd_ops {
	const char *name;
	int (*do_command)(int argc, char *argv[]);
	int (*do_command_data)(void *data, int argc, char *argv[]);
	void (*do_help)(char *str);
};

struct proc_cmd_table {
	const char *module_name;
	int size;
	struct proc_cmd_ops *ops;
	void *data;
	void *data_seq_read;
	struct mutex lock;
	int state_size;
};

struct proc_private {
	struct proc_cmd_table table;
	void *state;
};

static int proc_cmd_state_open(struct inode *inode, struct file *f)
{
	int ret = 0;
	struct proc_cmd_table *table = PDE_DATA(inode);
	struct proc_private *priv;

	priv = __seq_open_private(f, table->data_seq_read,
				  sizeof(struct proc_private));
	if (priv) {
		memcpy(&priv->table, table, sizeof(struct proc_cmd_table));
		priv->state = kzalloc(table->state_size, GFP_KERNEL_ACCOUNT);
		if (priv->state == NULL) {
			kfree(priv);
			return -ENOMEM;
		}
	}
	return ret;
};

static int proc_cmd_open(struct inode *inode, struct file *f)
{
	int ret;
	struct proc_cmd_table *table = PDE_DATA(inode);

	if (table->state_size)
		return proc_cmd_state_open(inode, f);

	ret = seq_open(f, table->data_seq_read);
	if (ret == 0)
		((struct seq_file *)f->private_data)->private = table;
	return ret;
};

static ssize_t proc_cmd_write(struct file *file, const char __user *buffer,
			  size_t count, loff_t *pos)
{
	char cmd_line[PROC_CMD_STRING_SIZE];
	char *ptr = cmd_line;
	char *argv[PROC_CMD_ARGS_MAX];
	int  i, argc = 0;
	struct proc_cmd_table *table;
	struct seq_file *seq;
	seq = (struct seq_file *) file->private_data;
	table = (struct proc_cmd_table *) seq->private;
	if ((count <= 1) || (buffer == NULL))
		goto help;
	if (count > PROC_CMD_STRING_SIZE)
		count = PROC_CMD_STRING_SIZE;
	memset(cmd_line, 0, PROC_CMD_STRING_SIZE);
	if (copy_from_user(cmd_line, buffer, count))
		return count;

	if (table && table->module_name)
		pr_debug("%s [pid-%d] [cpu-%d]: proc cmd - %s",
				table->module_name, current->pid,
				task_cpu(current), cmd_line);

	/* Parse command line */
	while ((argc < PROC_CMD_ARGS_MAX) &&
	       ((argv[argc] = strsep(&ptr, " ")) != NULL)) {
		if ((argv[argc][0] != 0x00) &&
		    (argv[argc][0] != 0x0A)) {
			/* Ignore white spaces and newline*/
			argc++;
		}
	}

	/* last argument may contain newline, remove it */
	if (argc) {
		ptr = strstr(argv[argc-1], "\n");
		if (ptr)
			*ptr = 0x00;
	}

	if (!argc)
		goto help;

	if (!table)
		return count;

	for (i = 0; table && i < table->size; i++) {
		if (strcasecmp(table->ops[i].name, argv[0]) == 0) {
			mutex_lock(&table->lock);
			if (table->ops[i].do_command)
				table->ops[i].do_command(argc, argv);
			else if (table->ops[i].do_command_data)
				table->ops[i].do_command_data(table->data, argc, argv);
			mutex_unlock(&table->lock);
			goto done;
		}
	}
help:
	if (table) {
		pr_err("List of Commands:\n");
		for (i = 0; i < table->size; i++) {
			if (table->ops[i].do_help)
				table->ops[i].do_help(" ");
			else
				pr_alert("  %s\n", table->ops[i].name);
			pr_alert("--------------------------------\n");
		}
	}
done:
	return count;

}

static ssize_t proc_cmd_read(struct file *file, char __user *buffer,
			     size_t count, loff_t *pos)
{
	int i;
	struct proc_cmd_table *table;
	struct seq_file *seq;
	seq = (struct seq_file *) file->private_data;
	table = (struct proc_cmd_table *) seq->private;
	if (table) {
		pr_alert("List of Commands:\n");
		for (i = 0; i < table->size; i++) {
			if (table->ops[i].do_help)
				table->ops[i].do_help(" ");
			else
				pr_alert("  %s\n", table->ops[i].name);
			pr_alert("--------------------------------\n");
		}
	}
	return 0;
}

static int proc_cmd_release(struct inode *inode, struct file *f)
{
	struct proc_cmd_table *table = PDE_DATA(inode);
	struct seq_file *seq;
	struct proc_private *priv;

	seq = (struct seq_file *) f->private_data;

	if (table->state_size) {
		priv = (struct proc_private *) seq->private;
		kfree(priv->state);
		return seq_release_private(inode, f);
	}

	return seq_release(inode, f);
};

static const struct proc_ops proc_cmd_fops = {
	.proc_open     = proc_cmd_open,
	.proc_read     = proc_cmd_read,
	.proc_write    = proc_cmd_write,
	.proc_lseek    = seq_lseek,
	.proc_release  = proc_cmd_release,
};

static const struct proc_ops proc_cmd_read_fops = {
	.proc_open     = proc_cmd_open,
	.proc_read     = seq_read,
	.proc_write    = proc_cmd_write,
	.proc_lseek    = seq_lseek,
	.proc_release  = proc_cmd_release,
};

static inline
struct proc_dir_entry *proc_create_cmd(const char *name,
				       struct proc_dir_entry *parent,
				       struct proc_cmd_table *table)
{
	if (!table || !name || !parent)
		return NULL;
	if (table->size > PROC_CMD_TABLE_SIZE_MAX)
		return NULL;
	mutex_init(&table->lock);
	if (table->data_seq_read)
		return proc_create_data(name,
					S_IRUGO | S_IWUGO,
					parent,
					&proc_cmd_read_fops,
					table);
	else
		return proc_create_data(name,
					S_IRUGO | S_IWUGO,
					parent,
					&proc_cmd_fops,
					table);
}

static inline
struct file *kfile_open(const char *fs, const char *path, int flags)
{
	struct vfsmount *mnt;
	struct file_system_type *fs_type;
	struct file *filp = NULL;
	int err = 0;

	fs_type = get_fs_type(fs);
	if (!fs_type) {
		pr_err("Failed to find fs type %s for %s\n", fs, path);
		return 0;
	}
	mnt = kern_mount(fs_type);
	module_put(fs_type->owner);
	if (IS_ERR(mnt)) {
		pr_err("Failed to kern_mount %s for %s\n", fs, path);
		return 0;
	}

	filp = file_open_root_mnt(mnt, path, flags, 0);
	if (IS_ERR(filp)) {
		err = PTR_ERR(filp);
		pr_err("Failed file_open_root_mnt %s for %s: error %d %s\n",
			   fs, path, err, mnt->mnt_root->d_iname);
		return NULL;
	}
	return filp;
}

static inline
void kfile_close(struct file *file)
{
	filp_close(file, NULL);
}

static inline
int kfile_read(struct file *file, unsigned char *data, unsigned int size)
{
	struct kvec iov = {
		.iov_base	= data,
		.iov_len	= size,
	};
	struct kiocb kiocb;
	struct iov_iter iter;
	ssize_t ret;

	if (WARN_ON_ONCE(!(file->f_mode & FMODE_READ)))
		return -EINVAL;
	if (!(file->f_mode & FMODE_CAN_READ))
		return -EINVAL;
	if (unlikely(!file->f_op->read_iter || file->f_op->read))
		return -EINVAL;

	init_sync_kiocb(&kiocb, file);
	kiocb.ki_pos = 0;
	iov_iter_kvec(&iter, READ, &iov, 1, iov.iov_len);
	ret = file->f_op->read_iter(&kiocb, &iter);
	return ret;
}

static inline
int procsys_file_read_string(const char *path, unsigned char *data,
							 unsigned int len)
{
	struct file *f;
	int ret;
	if (strncmp(path, "/proc/", 6) == 0)
		f = kfile_open("proc", path+6, O_RDONLY);
	else
		f = kfile_open("proc", path, O_RDONLY);
	if (!f)
		return -ENOENT;
	ret = kfile_read(f, data, len);
	if (ret > 0) {
		kfile_close(f);
		return 0;
	}

	kfile_close(f);
	return ret;
}

static inline
int kfile_write(struct file *file, unsigned char *data, unsigned int size)
{
	struct kvec iov = {
		.iov_base	= data,
		.iov_len	= size,
	};
	struct kiocb kiocb;
	struct iov_iter iter;
	ssize_t ret;

	if (WARN_ON_ONCE(!(file->f_mode & FMODE_WRITE)))
		return -EINVAL;
	if (!(file->f_mode & FMODE_CAN_WRITE))
		return -EINVAL;
	if (unlikely(!file->f_op->write_iter || file->f_op->write))
		return -EINVAL;

	init_sync_kiocb(&kiocb, file);
	kiocb.ki_pos = 0;
	iov_iter_kvec(&iter, WRITE, &iov, 1, iov.iov_len);
	ret = file->f_op->write_iter(&kiocb, &iter);
	return ret;
}

static inline
int procsys_file_write_string(const char *path, unsigned char *data)
{
	struct file *f;
	int ret, len;

	if (strncmp(path, "/proc/", 6) == 0)
		f = kfile_open("proc", path+6, O_WRONLY);
	else
		f = kfile_open("proc", path, O_WRONLY);
	if (!f)
		return -ENOENT;
	len = strlen(data);
	ret = kfile_write(f, data, len);
	if (ret > 0) {
		kfile_close(f);
		return 0;
	}

	kfile_close(f);
	return ret;
}
#endif
