 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/if_link.h>
#include <linux/netdevice.h>
#include <bcmnethooks.h>
#include <linux/module.h>
#include <linux/bcm_media_gw/msgfifo.h>
#include "dqnet_priv.h"
#include "dqnet_qmsg.h"
#include "dqnet_switch.h"
#include "dqnet_brcmtag.h"
#include "fpm.h"

extern struct dqnet_qmsg_ops dqnet_qmsg_gfap_legacy;
#if defined(CONFIG_BCM_RUNNER)
extern struct dqnet_qmsg_ops dqnet_qmsg_rfap;
#else
extern struct dqnet_qmsg_ops dqnet_qmsg_gfap;
#endif

static struct dqnet_qmsg_ops *gfap_ops;
static struct dqnet_qmsg_ops *chipfap_ops;

void dqnet_getsym_qmsg_ops(void)
{
	gfap_ops = &dqnet_qmsg_gfap_legacy;
#if defined(CONFIG_BCM_RUNNER)
	chipfap_ops = symbol_get(dqnet_qmsg_rfap);
#else
	chipfap_ops = symbol_get(dqnet_qmsg_gfap);
#endif
}

void dqnet_putsym_qmsg_ops(void)
{
	gfap_ops = NULL;
#if defined(CONFIG_BCM_RUNNER)
	symbol_put(dqnet_qmsg_rfap);
#else
	symbol_put(dqnet_qmsg_gfap);
#endif
	chipfap_ops = NULL;
}

int dqnet_max_if_id(void)
{
	if (gfap_ops && chipfap_ops) {
		return ((gfap_ops->max_if_id() > chipfap_ops->max_if_id()) ?
			gfap_ops->max_if_id() : chipfap_ops->max_if_id());
	} else if (gfap_ops) {
		return gfap_ops->max_if_id();
	} else if (chipfap_ops) {
		return chipfap_ops->max_if_id();
	}
	return 0;
}

int dqnet_max_if_subid(void)
{
	if (gfap_ops && chipfap_ops) {
		return ((gfap_ops->max_if_subid() > chipfap_ops->max_if_subid()) ?
			gfap_ops->max_if_subid() : chipfap_ops->max_if_subid());
	} else if (gfap_ops) {
		return gfap_ops->max_if_subid();
	} else if (chipfap_ops) {
		return chipfap_ops->max_if_subid();
	}
	return 0;
}

int dqnet_if_id_subid_to_port_fap(u32 if_id, u32 if_sub_id, u8 *port)
{
	int status = 0;
	int max_ports;

	max_ports = SWITCH->get_max_ports();
	if (!max_ports) {
		pr_err("%s: No switch, hence no ports.\n", __func__);
		status = -EINVAL;
		goto done;
	}
	status = chipfap_ops->id_to_port(if_id, if_sub_id, port);

	if (status)
		goto done;

	if (unlikely(*port >= max_ports)) {
		pr_err("%s: Interface ID greater than # of ports on switch.\n",
		       __func__);
		status = -EINVAL;
		goto done;
	}

done:
	return status;
}

int dqnet_port_to_if_id_subid_fap(u8 port, u32 *if_id, u32 *if_sub_id)
{
	int status = 0;
	int max_ports;

	max_ports = SWITCH->get_max_ports();
	if (!max_ports) {
		pr_err("%s: No switch, hence no ports.\n", __func__);
		status = -EINVAL;
		goto done;
	}
	if (unlikely(port >= max_ports)) {
		pr_err("%s: Port # greater than # of ports on switch.\n",
		       __func__);
		status = -EINVAL;
		goto done;
	}
	status = chipfap_ops->port_to_id(port, if_id, if_sub_id);

done:
	return status;
}

int dqnet_max_if_id_gfap(void)
{
	return ((LANRXO_MAC_ID_MASK >> LANRXO_MAC_ID_SHIFT) + 1);
}

int dqnet_max_if_subid_gfap(void)
{
	return ((LANRXO_SUB_ID_MASK >> LANRXO_SUB_ID_SHIFT) + 1);
}

/*
 * Encode a GFAP TX message
 *
 * Parameters
 *      fb	FPM buffer descriptor
 *      msg	DQM message
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_encode_q_msg_gfap(struct dqnet_netdev *ndev,
			    struct fpm_buff *fb,
			    u32 *msgdata)
{
	int status = 0;
	struct dqnet_channel *chan = ndev->chan;
	struct lanrxoffsetmsg *rx_msg =
		(struct lanrxoffsetmsg *)&msgdata[0];
	struct lantxoffsetmsg *tx_msg =
		(struct lantxoffsetmsg *)&msgdata[0];
	u32 msg_id;

	pr_debug("-->\n");

	memset(msgdata, 0, sizeof(u32) * 2);
	switch (chan->type) {
	case DQNET_CHAN_TYPE_FAP_EXCEPT:
	case DQNET_CHAN_TYPE_POINT_TO_POINT:
	case DQNET_CHAN_TYPE_SPOOFED:
		if (fb->if_id == MAC_ID_EROUTER)
			msg_id = USPacketTxMsg;
		else
			msg_id = LANTxMsg;
		tx_msg->msghdr |= msg_id << LANTXO_MSG_ID_SHIFT;
		tx_msg->msghdr |= fb->if_id << LANTXO_MAC_ID_SHIFT;
		tx_msg->msghdr |= fb->if_sub_id << LANTXO_SUB_ID_SHIFT;
		tx_msg->msghdr |= LANTXO_REL_TKN_MASK;
		tx_msg->msghdr |= LANTXO_EOP_MASK;
		tx_msg->msghdr |= (fb->priority << LANTXO_QOS_SHIFT) &
			LANTXO_QOS_MASK;
		if (chan->qmsg_fmt == DQNET_QMSG_FMT_GFAP_FPM) {
			tx_msg->msghdr |=
				(fb->offset & LANTXO_OFFSET_MASK) <<
				LANTXO_OFFSET_SHIFT;
			tx_msg->token = fb->token;
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
		} else {
			tx_msg->msghdr |=
				((u32)(fb->skb->len) & LANTXSKB_LEN_MASK) <<
				LANTXSKB_LEN_SHIFT;
			tx_msg->context = (u32)fb->skb;
			pr_err("TODO: Need to see if Charles needs phys addr\n");
			pr_err("here for SGFAP.\n");
/* 			memcpy(&tx_msg->addr, fb->skb->cb, */
/* 			       sizeof(tx_msg->addr));	   */
#endif
		}
		break;
	case DQNET_CHAN_TYPE_FAP_HOST:
	case DQNET_CHAN_TYPE_POINT_TO_POINT_SWAP:
		msg_id = LANRxMsg;
		rx_msg->msghdr |= msg_id << LANRXO_MSG_ID_SHIFT;
		rx_msg->msghdr |= fb->if_id << LANRXO_MAC_ID_SHIFT;
		rx_msg->msghdr |= fb->if_sub_id << LANRXO_SUB_ID_SHIFT;
		rx_msg->msghdr |= LANRXO_EOP_MASK;
		rx_msg->msghdr |= (fb->priority << LANRXO_QOS_SHIFT) &
			LANRXO_QOS_MASK;
		if (chan->qmsg_fmt == DQNET_QMSG_FMT_GFAP_FPM) {
			rx_msg->msghdr |=
				(fb->offset & LANRXO_OFFSET_MASK) <<
				LANRXO_OFFSET_SHIFT;
			rx_msg->token = fb->token;
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
		} else {
			rx_msg->msghdr |=
				((u32)(fb->skb->len) & LANRXSKB_LEN_MASK) <<
				LANRXSKB_LEN_SHIFT;
			rx_msg->context = (u32)fb->skb;
			memcpy(&rx_msg->addr, fb->skb->cb,
			       sizeof(rx_msg->addr));
#endif
		}
		break;
	default:
		status = -EIO;
	}

	pr_debug("<--\n");
	return status;
}

/*
 * Decode a GFAP RX message.
 *
 * Parameters
 *      msg	DQM message
 *      fb	FPM buffer descriptor
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_decode_q_msg_gfap(struct dqnet_channel *chan,
			    u32 *msgdata,
			    struct fpm_buff *fb)
{
	int status = 0;
	u32 msg_id;

	pr_debug("-->\n");

	msg_id = (msgdata[0] & ID_MASK) >> ID_SHIFT;
	switch (chan->type) {
	case DQNET_CHAN_TYPE_FAP_EXCEPT:
	case DQNET_CHAN_TYPE_POINT_TO_POINT_SWAP:
	case DQNET_CHAN_TYPE_SPOOFED:
		if ((msg_id != LANRxMsg) &&
		    (msg_id != DSPacketRxMsg))
			status = -EIO;
		break;
	case DQNET_CHAN_TYPE_FAP_HOST:
	case DQNET_CHAN_TYPE_POINT_TO_POINT:
		if (msg_id != LANTxMsg)
			status = -EIO;
		break;
	default:
		status = -EIO;
		break;
	}
	fb->if_id = (msgdata[0] & MAC_ID_MASK) >> MAC_ID_SHIFT;
	fb->if_sub_id = (msgdata[0] & SUB_ID_MASK) >> SUB_ID_SHIFT;
	fb->priority = (msgdata[0] & LANRXO_QOS_MASK) >>
		LANRXO_QOS_SHIFT;
	if (chan->qmsg_fmt == DQNET_QMSG_FMT_GFAP_FPM) {
		fb->type = BUF_TYPE_FPM;
		fb->offset = (msgdata[0] & LANRXO_OFFSET_MASK) >>
			LANRXO_OFFSET_SHIFT;
		fb->token = msgdata[1];
		if (unlikely(!fpm_is_valid_token(fb->token))) {
			pr_err("%s: Invalid token 0x%08x received.\n",
			       __func__, fb->token);
			status = -EIO;
			goto done;
		}
		fpm_track_token_rx(fb->token);
		fb->buf = fpm_token_to_buffer(fb->token);
		if (unlikely(!fb->buf)) {
			pr_err("%s: Unable to translate token 0x%08x to "
			       "buffer.\n", __func__, fb->token);
			status = -EIO;
			goto done;
		}
		fb->data = fb->buf + fb->offset;
		fb->len = fpm_get_token_size(fb->token);
		if (!fb->len) {
			status = -EIO;
			goto done;
		}
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
	} else {
		fb->type = BUF_TYPE_SKB;
		fb->skb = (struct sk_buff *)msgdata[1];
		skb_put(fb->skb, (msgdata[0] & LANRXSKB_LEN_MASK) >>
			LANRXSKB_LEN_SHIFT);
#endif
	}
	fb->brcm_tag_len = 0;
	fb->fap_tag_len = 0;

done:
	pr_debug("<--\n");
	return status;
}

/*
 * Translate an interface ID/SubID to a switch port
 *
 * Parameters
 *      if_id		interface ID
 *      if_sub_id	interface sub-ID
 *      port		switch port #
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_if_id_subid_to_port_gfap(u32 if_id, u32 if_sub_id, u8 *port)
{
	int status = 0;
	int max_ports;

	pr_debug("-->\n");

	max_ports = SWITCH->get_max_ports();
	if (!max_ports) {
		pr_err("%s: No switch, hence no ports.\n", __func__);
		status = -EINVAL;
		goto done;
	}
	if (unlikely(if_sub_id >= dqnet_max_if_subid_gfap())) {
		pr_err("%s: Interface ID greater than max LAN ports allowed.\n",
		       __func__);
		status = -EINVAL;
		goto done;
	}
	*port = if_sub_id;
	if (unlikely(*port >= max_ports)) {
		pr_err("%s: Interface ID greater than # of ports on switch.\n",
		       __func__);
		status = -EINVAL;
		goto done;
	}

done:
	pr_debug("<--\n");
	return status;
}

/*
 * Translate a switch port to an interface ID/SubID.
 *
 * Parameters
 *      port		switch port #
 *      if_id		interface ID
 *      if_sub_id	interface sub-ID
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_port_to_if_id_subid_gfap(u8 port, u32 *if_id, u32 *if_sub_id)
{
	int status = 0;
	int max_ports;

	pr_debug("-->\n");

	max_ports = SWITCH->get_max_ports();
	if (!max_ports) {
		pr_err("%s: No switch, hence no ports.\n", __func__);
		status = -EINVAL;
		goto done;
	}
	if (unlikely(port >= max_ports)) {
		pr_err("%s: Port # greater than # of ports on switch.\n",
		       __func__);
		status = -EINVAL;
		goto done;
	}
	*if_sub_id = port;

done:
	pr_debug("<--\n");
	return status;
}

struct dqnet_qmsg_ops dqnet_qmsg_gfap_legacy = {
	.encode_qmsg    = dqnet_encode_q_msg_gfap,
	.decode_qmsg    = dqnet_decode_q_msg_gfap,
	.id_to_port     = dqnet_if_id_subid_to_port_gfap,
	.port_to_id     = dqnet_port_to_if_id_subid_gfap,
	.max_if_id	= dqnet_max_if_id_gfap,
	.max_if_subid	= dqnet_max_if_subid_gfap,
};

static enum buf_type dqnet_buf_type_fpm(void)
{
	return BUF_TYPE_FPM;
}

static enum buf_type dqnet_buf_type_skb(void)
{
	return BUF_TYPE_SKB;
}

/*
 * Init Q message function pointers
 *
 * Parameters
 *      chan	DQNET channel
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_init_q_msg_funcs(struct dqnet_channel *chan)
{
	int status = 0;

	pr_debug("-->\n");

	switch (chan->qmsg_fmt) {
	case DQNET_QMSG_FMT_GFAP_FPM:
	case DQNET_QMSG_FMT_GFAP_SKB:
		if (!gfap_ops) {
			pr_err("dqnet gfap qmsg module not loaded\n");
			status = -EINVAL;
			break;
		}
		chan->encode_q_msg = gfap_ops->encode_qmsg;
		chan->decode_q_msg = gfap_ops->decode_qmsg;
		chan->if_id_subid_to_port = gfap_ops->id_to_port;
		chan->port_to_if_id_subid = gfap_ops->port_to_id;
		break;

	case DQNET_QMSG_FMT_RFAP_FPM:
	case DQNET_QMSG_FMT_RFAP_SKB:
	case DQNET_QMSG_FMT_GFAP3_FPM:
	case DQNET_QMSG_FMT_GFAP3_SKB:
		if (!chipfap_ops) {
			pr_err("dqnet chip fap module not loaded\n");
			status = -EINVAL;
			break;
		}
		chan->encode_q_msg = chipfap_ops->encode_qmsg;
		chan->decode_q_msg = chipfap_ops->decode_qmsg;
		chan->if_id_subid_to_port = dqnet_if_id_subid_to_port_fap;
		chan->port_to_if_id_subid = dqnet_port_to_if_id_subid_fap;
		break;

	default:
		status = -EINVAL;
		goto done;
	}

	switch (chan->qmsg_fmt) {
	case DQNET_QMSG_FMT_GFAP_FPM:
	case DQNET_QMSG_FMT_GFAP3_FPM:
	case DQNET_QMSG_FMT_RFAP_FPM:
		chan->buf_type = dqnet_buf_type_fpm;
		chan->rx_pkt = dqnet_rx_fpm;
		chan->tx_pkt = dqnet_tx_fpm;
		break;

	case DQNET_QMSG_FMT_GFAP_SKB:
	case DQNET_QMSG_FMT_GFAP3_SKB:
	case DQNET_QMSG_FMT_RFAP_SKB:
		chan->buf_type = dqnet_buf_type_skb;
		chan->rx_pkt = dqnet_rx_skb;
		chan->tx_pkt = dqnet_tx_buf;
		break;

	default:
		status = -EINVAL;
		goto done;
	}

done:
	pr_debug("<--\n");
	return status;
}
