 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/of.h>
#include <linux/io.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/brcmstb/brcmstb.h>


#define MODULE_NAME	"brcm-pmb"
#define MODULE_VER	"1.0"

struct pmb_master {
	struct list_head list;
	struct platform_device *pdev;
	void *reg;
	bool direct;
	int ver;
	spinlock_t bus_lock;;
};

#define SLAVE_DIRECT_BLOCK_SIZE		0x4000

#define CFG	((pmb->ver == 1) ? (pmb->reg + 0) : (pmb->reg + 8))
#define CTRL	((pmb->ver == 1) ? (pmb->reg + 4) : (pmb->reg + 0))
#define WR_DATA	((pmb->ver == 1) ? (pmb->reg + 8) : (pmb->reg + 4))
#define RD_DATA	(pmb->reg + 0xc)

#define CFG_RBUS_ERROR_MASK_SHIFT	(1)
#define CFG_RBUS_ERROR_MASK_MASK	(0x1 << CFG_RBUS_ERROR_MASK_SHIFT)
#define CFG_TIMEOUT_MASK		((pmb->ver == 1) ? 0xff0 : 0xff)
#define CFG_TIMEOUT_SHIFT		((pmb->ver == 1) ? 4 : 0)

#define CTRL_PMB_REG_ADDR_SHIFT		(0)
#define CTRL_PMB_REG_ADDR_MASK		(0xfff << CTRL_PMB_REG_ADDR_SHIFT)
#define CTRL_PMB_REG_ADDR(x) \
	(((x) << CTRL_PMB_REG_ADDR_SHIFT) & CTRL_PMB_REG_ADDR_MASK)
#define CTRL_PMB_SLAVE_ID_SHIFT		(12)
#define CTRL_PMB_SLAVE_ID_MASK		(0xff << CTRL_PMB_SLAVE_ID_SHIFT)
#define CTRL_PMB_SLAVE_ID(x) \
	(((x) << CTRL_PMB_SLAVE_ID_SHIFT) & CTRL_PMB_SLAVE_ID_MASK)
#define CTRL_PMB_CMD_SHIFT		(20)
#define CTRL_PMB_CMD_MASK		(0xf << CTRL_PMB_CMD_SHIFT)
#define CTRL_PMB_CMD(x) \
	(((x) << CTRL_PMB_CMD_SHIFT) & CTRL_PMB_CMD_MASK)
#define CTRL_PMB_CMD_READ		(0)
#define CTRL_PMB_CMD_WRITE		(1)
#define CTRL_PMB_BUSY_SHIFT		(28)
#define CTRL_PMB_BUSY_MASK		(0x1 << CTRL_PMB_BUSY_SHIFT)
#define CTRL_SLAVE_ERROR_SHIFT		(29)
#define CTRL_SLAVE_ERROR_MASK		(0x1 << CTRL_SLAVE_ERROR_SHIFT)
#define CTRL_TIMEOUT_ERROR_SHIFT	(30)
#define CTRL_TIMEOUT_ERROR_MASK		(0x1 << CTRL_TIMEOUT_ERROR_SHIFT)
#define CTRL_ERROR_MASK \
	(CTRL_SLAVE_ERROR_MASK | CTRL_TIMEOUT_ERROR_MASK)
#define CTRL_PMB_START_SHIFT		(31)
#define CTRL_PMB_START_MASK		(0x1 << CTRL_PMB_START_SHIFT)

struct list_head masters;
spinlock_t masters_lock;

int pmb_read(struct device *dev, u8 id, u16 addr, u32 *val)
{
	int status = 0;
	struct platform_device *pdev = to_platform_device(dev);
	struct pmb_master *pmb = platform_get_drvdata(pdev);
	unsigned long flags;

	pr_debug("-->\n");
	if (!val) {
		dev_err(dev, "%s: NULL val ptr!\n", __func__);
		status = -EINVAL;
		goto done;
	}
	spin_lock_irqsave(&pmb->bus_lock, flags);
	if (pmb->direct) {
		u32 *reg = pmb->reg + (id * SLAVE_DIRECT_BLOCK_SIZE) +
			(addr << 2);
		*val = __raw_readl(reg);
		pr_debug("read 0x%08x from ID %d, addr %d (reg offset 0x%08x)\n",
			 *val, id, addr, (u32)reg - (u32)(pmb->reg));
	} else {
		u32 ctrl;
		ctrl = CTRL_PMB_REG_ADDR(addr);
		ctrl |= CTRL_PMB_SLAVE_ID(id);
		ctrl |= CTRL_PMB_CMD(CTRL_PMB_CMD_READ);
		ctrl |= CTRL_PMB_START_MASK;
		__raw_writel(ctrl, CTRL);
		do {
			ctrl = __raw_readl(CTRL);
		} while (ctrl & CTRL_PMB_BUSY_MASK);
		if (ctrl & CTRL_ERROR_MASK) {
			dev_err(dev, "PMB error (slave %d, reg %d).\n",
				id, addr);
			status = -EIO;
		}
		*val = __raw_readl(RD_DATA);
		pr_debug("read 0x%08x from ID %d, addr %d\n", *val, id, addr);
	}
	spin_unlock_irqrestore(&pmb->bus_lock, flags);

done:
	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(pmb_read);

int pmb_write(struct device *dev, u8 id, u16 addr, u32 val)
{
	int status = 0;
	struct platform_device *pdev = to_platform_device(dev);
	struct pmb_master *pmb = platform_get_drvdata(pdev);
	unsigned long flags;

	pr_debug("-->\n");
	spin_lock_irqsave(&pmb->bus_lock, flags);
	if (pmb->direct) {
		u32 *reg = pmb->reg + (id * SLAVE_DIRECT_BLOCK_SIZE) +
			(addr << 2);
		__raw_writel(val, reg);
		pr_debug("wrote 0x%08x to ID %d, addr %d (reg offset 0x%08x)\n",
			 val, id, addr, (u32)reg - (u32)(pmb->reg));
	} else {
		u32 ctrl;
		__raw_writel(val, WR_DATA);
		ctrl = CTRL_PMB_REG_ADDR(addr);
		ctrl |= CTRL_PMB_SLAVE_ID(id);
		ctrl |= CTRL_PMB_CMD(CTRL_PMB_CMD_WRITE);
		ctrl |= CTRL_PMB_START_MASK;
		__raw_writel(ctrl, CTRL);
		do {
			ctrl = __raw_readl(CTRL);
		} while (ctrl & CTRL_PMB_BUSY_MASK);
		if (ctrl & CTRL_ERROR_MASK) {
			dev_err(dev, "PMB error (slave %d, reg %d).\n",
				id, addr);
			status = -EIO;
		}
		pr_debug("wrote 0x%08x to ID %d, addr %d\n",
			 val, id, addr);
	}
	spin_unlock_irqrestore(&pmb->bus_lock, flags);

	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(pmb_write);

struct device *pmb_register(struct device_node *pmb_node)
{
	struct list_head *pos;
	struct pmb_master *pmb;
	struct device *dev = ERR_PTR(-ENODEV);
	struct device_node *np;
	unsigned long flags;

	pr_debug("-->\n");
	spin_lock_irqsave(&masters_lock, flags);
	list_for_each(pos, &masters) {
		pmb = list_entry(pos, struct pmb_master, list);
		np = pmb->pdev->dev.of_node;
		if (np == pmb_node) {
			pr_debug("Found matching node\n");
			dev = &pmb->pdev->dev;
			break;
		}
	}
	spin_unlock_irqrestore(&masters_lock, flags);
	if (IS_ERR(dev)) {
		pr_err("%s: Unable to find matching PMB master device.\n",
		       __func__);
		goto done;
	}
	get_device(dev);

done:
	pr_debug("<--\n");
	return dev;
}
EXPORT_SYMBOL(pmb_register);

int pmb_unregister(struct device *dev)
{
	int status = -ENODEV;
	struct list_head *pos;
	struct pmb_master *pmb;
	unsigned long flags;

	pr_debug("-->\n");
	spin_lock_irqsave(&masters_lock, flags);
	list_for_each(pos, &masters) {
		pmb = list_entry(pos, struct pmb_master, list);
		if (&pmb->pdev->dev == dev) {
			status = 0;
			break;
		}
	}
	spin_unlock_irqrestore(&masters_lock, flags);
	if (status) {
		pr_err("%s: Unable to find matching PMB master device.\n",
		       __func__);
		goto done;
	}
	put_device(dev);

done:
	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(pmb_unregister);

static int pmb_probe(struct platform_device *pdev)
{
	int status = 0;
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct pmb_master *pmb;
	struct resource *res;
	unsigned long flags;

	pr_debug("-->\n");

	pmb = devm_kzalloc(dev, sizeof(struct pmb_master), GFP_KERNEL);
	if (IS_ERR_OR_NULL(pmb)) {
		dev_err(dev, "Failed to allocate PMB device.\n");
		status = PTR_ERR(pmb);
		goto done;
	}
	pmb->pdev = pdev;
	INIT_LIST_HEAD(&pmb->list);
	spin_lock_init(&pmb->bus_lock);
	platform_set_drvdata(pdev, pmb);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	res->flags &= ~IORESOURCE_CACHEABLE;
	pmb->reg = devm_ioremap_resource(dev, res);
	if (IS_ERR_OR_NULL(pmb->reg)) {
		dev_err(dev, "Failed to map PMB registers.");
		status = PTR_ERR(pmb->reg);
		goto done;
	}
	pmb->direct = of_property_read_bool(np, "direct-access");
	/* coverity[example_checked] */
	if (of_property_read_u32(np, "version", &pmb->ver) && !pmb->direct) {
		dev_err(dev,"Missing version property\n");
		status = -EINVAL;
		goto done;
	}

	spin_lock_irqsave(&masters_lock, flags);
	list_add(&pmb->list, &masters);
	spin_unlock_irqrestore(&masters_lock, flags);

	status = of_platform_populate(np, NULL, NULL, NULL);

done:
	pr_debug("<--\n");
	return status;
}

static int pmb_remove(struct platform_device *pdev)
{
	int status = 0;
	struct list_head *pos;
	struct pmb_master *pmb;
	unsigned long flags;

	pr_debug("-->\n");
	spin_lock_irqsave(&masters_lock, flags);
	list_for_each(pos, &masters) {
		pmb = list_entry(pos, struct pmb_master, list);
		if (pmb->pdev == pdev) {
			list_del(pos);
			break;
		}
	}
	spin_unlock_irqrestore(&masters_lock, flags);
	pr_debug("<--\n");
	return status;
}

static const struct of_device_id pmb_of_match[] = {
	{.compatible = "brcm,pmb"},
	{}
};

MODULE_DEVICE_TABLE(of, pmb_of_match);

static struct platform_driver pmb_driver = {
	.probe  = pmb_probe,
	.remove = pmb_remove,
	.driver = {
		.name = MODULE_NAME,
		.owner = THIS_MODULE,
		.of_match_table = pmb_of_match
	}
};

static int __init pmb_init(void)
{
	pr_debug("%s driver v%s\n", MODULE_NAME, MODULE_VER);
	INIT_LIST_HEAD(&masters);
	spin_lock_init(&masters_lock);
	return platform_driver_register(&pmb_driver);
}
arch_initcall(pmb_init);

static void __exit pmb_exit(void)
{
	platform_driver_unregister(&pmb_driver);
}
module_exit(pmb_exit);

MODULE_AUTHOR("Tim Ross");
MODULE_DESCRIPTION("Power Management Bus (PMB) Driver");
MODULE_LICENSE("GPL v2");
