 /****************************************************************************
 *
 * Copyright (c) 2019 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Broadcom BCM5xxx Gigabit Ethernet transceivers.
 *
 * Author: Tim Ross <tim.ross@broadcom.com>
 *	   Ravi Patel <ravi.patel@broadcom.com>
 *
 *	   Inspired by code written by Maciej W. Rozycki.
 *****************************************************************************/

#include "bcm-phy-lib.h"
#include <linux/module.h>
#include <linux/phy.h>
#include <linux/brcmphy.h>

MODULE_DESCRIPTION("Broadcom BCM5xxx PHY driver");
MODULE_AUTHOR("Tim Ross");
MODULE_AUTHOR("Ravi Patel");
MODULE_LICENSE("GPL");

static int bcm5xxx_config_init(struct phy_device *phydev)
{
	struct device_node *np = phydev->mdio.dev.of_node;
	const char *pval;
	int rc;
	bool in_band = false, phy_leds = false;

	if (np) {
		if (!of_property_read_string(np, "managed", &pval) &&
				!strcmp(pval, "in-band-status"))
			in_band = true;

		if (!of_property_read_string(np, "phy-leds", &pval) &&
				!strcmp(pval, "link-activity"))
			phy_leds = true;
	}

	if (phydev->phy_id == PHY_ID_BCM54810C0) {
		/* For BCM54810, we need to disable BroadR-Reach function */
		rc = bcm_phy_read_exp(phydev,
				       BCM54810_EXP_BROADREACH_LRE_MISC_CTL);
		if (rc < 0)
			goto do_return;

		rc &= ~BCM54810_EXP_BROADREACH_LRE_MISC_CTL_EN;
		rc = bcm_phy_write_exp(phydev,
					BCM54810_EXP_BROADREACH_LRE_MISC_CTL,
					rc);
		if (rc < 0)
			goto do_return;
	}

	rc = phy_read(phydev, MII_BCM54XX_ECR);
	if (rc < 0)
		goto do_return;

	/* Mask interrupts globally.  */
	rc |= MII_BCM54XX_ECR_IM;
	rc = phy_write(phydev, MII_BCM54XX_ECR, rc);
	if (rc < 0)
		goto do_return;

	/* Unmask events we are interested in.  */
	rc = ~(MII_BCM54XX_INT_DUPLEX |
		MII_BCM54XX_INT_SPEED |
		MII_BCM54XX_INT_LINK);
	rc = phy_write(phydev, MII_BCM54XX_IMR, rc);
	if (rc < 0)
		goto do_return;

	/* Enable in-band signalling */
	if (!in_band)
		goto do_gtxclk;

	rc = bcm54xx_auxctl_read(phydev, MII_BCM54XX_AUXCTL_SHDWSEL_MISC);
	rc |= MII_BCM54XX_AUXCTL_MISC_WREN;
	rc &= ~MII_BCM54XX_AUXCTL_MISC_OOBS_DIS;
	rc = bcm54xx_auxctl_write(phydev, MII_BCM54XX_AUXCTL_SHDWSEL_MISC,
				   rc);
	if (rc < 0)
		goto do_return;

do_gtxclk:
	/* Disable GTXCLK delay */
	rc = bcm_phy_read_shadow(phydev, BCM54XX_SHD_CAC);
	rc &= ~BCM54XX_SHD_CAC_GTXCLK_DEL_EN;
	rc = bcm_phy_write_shadow(phydev, BCM54XX_SHD_CAC, rc);
	if (rc < 0)
		goto do_return;

	if (!phy_leds)
		goto do_return;

	if (phydev->phy_id == PHY_ID_BCM54210EB0 ||
	    phydev->phy_id == PHY_ID_BCM50212EB1) {
		/* Enable LOM-LED Mode */
		rc = bcm_phy_read_exp(phydev,
				MII_BCM54XX_EXP_TOP_MISC_PIN_CTL);
		if (rc < 0)
			goto do_return;

		rc |= MII_BCM54XX_EXP_TOP_MISC_PIN_CTL_LOM;
		rc = bcm_phy_write_exp(phydev,
				MII_BCM54XX_EXP_TOP_MISC_PIN_CTL, rc);
		if (rc < 0)
			goto do_return;
	} else {
		/* Enable Link LED Mode */
		rc = bcm_phy_read_shadow(phydev, BCM54XX_SHD_SCR1);
		if (rc < 0)
			goto do_return;

		rc |= BCM54XX_SHD_SCR1_LINKLED;
		rc = bcm_phy_write_shadow(phydev, BCM54XX_SHD_SCR1, rc);
		if (rc < 0)
			goto do_return;
	}

	/* Enable Activity/Link LED function */
	rc = bcm_phy_read_shadow(phydev, BCM54XX_SHD_LEDCTL);
	if (rc < 0)
		goto do_return;

	rc |= BCM54XX_SHD_LEDCTL_ACTLINK;
	rc = bcm_phy_write_shadow(phydev, BCM54XX_SHD_LEDCTL, rc);
	if (rc < 0)
		goto do_return;

do_return:
	return rc;
}

#define BCM5XXX_GPHY(_oui, _name)			\
{							\
	.phy_id		= (_oui),			\
	.name		= _name,			\
	.phy_id_mask	= 0xffffffff,			\
	/* .features       = PHY_GBIT_FEATURES |     */       \
	/*                   SUPPORTED_Pause |       */       \
	/*                   SUPPORTED_Asym_Pause,   */       \
	/* .flags          = PHY_HAS_MAGICANEG |     */       \
	/*                   PHY_HAS_INTERRUPT,      */       \
	.config_init	= bcm5xxx_config_init,		\
	.ack_interrupt	= bcm_phy_ack_intr,		\
	.config_intr	= bcm_phy_config_intr,		\
	.suspend	= genphy_suspend,		\
	.resume		= genphy_resume,		\
	.config_aneg	= genphy_config_aneg,		\
	.read_status	= genphy_read_status,		\
	.mdiodrv.driver	= { .owner = THIS_MODULE },	\
}

static struct phy_driver bcm5xxx_drivers[] = {
	BCM5XXX_GPHY(PHY_ID_BCM54810C0,  "Broadcom BCM54810 C0"),
	BCM5XXX_GPHY(PHY_ID_BCM54610C1,  "Broadcom BCM54610 C1"),
	BCM5XXX_GPHY(PHY_ID_BCM54210EB0, "Broadcom BCM54210E B0"),
	BCM5XXX_GPHY(PHY_ID_BCM50212EB1, "Broadcom BCM50212E B1"),
};

module_phy_driver(bcm5xxx_drivers);

static struct mdio_device_id __maybe_unused bcm5xxx_tbl[] = {
	{ PHY_ID_BCM54810C0,  0xffffffff },
	{ PHY_ID_BCM54610C1,  0xffffffff },
	{ PHY_ID_BCM54210EB0, 0xfffffffc },
	{ PHY_ID_BCM50212EB1, 0xffffffff },
	{ }
};

MODULE_DEVICE_TABLE(mdio, bcm5xxx_tbl);
