#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
//#include <sys/ioctl.h>
//#include <asm/byteorder.h>
//#include <libgen.h>
#include <errno.h>
#include <libfdt.h>

#define MAX_FDT_SIZE		(256*1024)

static char comp_fdt[MAX_FDT_SIZE];

static void usage(void)
{
	printf("Usage: fdtcomp <options> <device tree blob file>...\n");
	printf("Composite a base device tree binary blob and one or more\n");
	printf("overlay blobs into a resulting blob. The overlay blobs\n");
	printf("will be composited onto the base blob in the order specified\n");
	printf("by their layer properties without regard to the \n");
	printf("overlay blobs' compatible nodes.\n\n");
	printf("Options: -[o:h]\n");
	printf("\t-o <arg>\tOutput file name (default: \"comp.dtb\")\n");
	printf("\t-h\t\tPrint this help and exit\n");
}

static int find_node(void *fdt, const char *match, int max_depth, int *offset)
{
	int status = 0;
	int depth;

	for (*offset = 0, depth = 0; ; ) {
		const char *name;

		*offset = fdt_next_node(fdt, *offset, &depth);
		if (*offset < 0) {
			status = *offset;
			break;
		}
		name = fdt_get_name(fdt, *offset, NULL);
		if (name && !strcmp(name, match) && depth <= max_depth)
			break;
	}

	return status;
}

static int get_layer(void *fdt, int *layer)
{
	int status = 0;
	const void *prop;
	int len;
	char *prop_name = "layer";
	int plat_match;

	plat_match = fdt_path_offset(fdt, "/platform-match");
	if (plat_match < 0) {
		printf("Failed to find /platform-match node offset with err %d\n", plat_match);
		status = plat_match;
		goto done;
	}
	prop = fdt_getprop(fdt, plat_match, prop_name, &len);
	if (!prop) {
		printf("Failed to get %s property with err %d\n",
		       prop_name, len);
		status = len;
		goto done;
	}
	*layer = fdt32_to_cpu(*(const int *)prop);

done:
	return status;
}

static int del_node(void *fdt, char *full_path)
{
	int status = 0;
	int node;

	if (full_path[0] != '/' ||
	    strlen(full_path) < 2 ||
	    !strstr(full_path, "/")) {
		printf("Invalid node path, %s. \n", full_path);
		status = -FDT_ERR_NOTFOUND;
		goto done;
	}
	node = fdt_path_offset(fdt, full_path);
	if (node < 0) {
		printf("Failed to find node %s offset with err %x.\n",
		       full_path, node);
		status = node;
		goto done;
	}
	status = fdt_del_node(fdt, node);
	if (status) {
		printf("Failure deleting %s node.\n", full_path);
		goto done;
	}

done:
	return status;
}

static int check_layer_match(void *fdt, int layer, bool *match)
{
	int status = 0;
	int fdt_layer = -1;

	*match = false;
	status = get_layer(fdt, &fdt_layer);
	if (status) {
		goto done;
	}
	if (fdt_layer == layer)
		*match = true;
done:
	return status;
}

static int apply_overlay(void *comp_fdt, void *overlay_fdt)
{
	int status = 0;

	status = fdt_overlay_apply(comp_fdt, overlay_fdt);
	if (status) {
		printf("%s failed with error %d.\n", __func__, status);
		goto done;
	}

	/*
	 * fdt_overlay_apply() sets the magic on the overlay to 0xffffffff
	 * because it has modified the phandles. However, we need to use each
	 * overlay header's magic and size value to walk the list of overlays
	 * repeatedly. So we rewrite the overlay's magic back to a valid value.
	 * The modified phandles will not matter because it has already been
	 * applied.
	 */
	fdt_set_magic(overlay_fdt, FDT_MAGIC);

done:
	return status;
}

static int open_fdt(char *name, void **buf, int *size)
{
	int status = 0;
	int fd;
	struct stat fs;
	void *fdt = NULL;

	fd = open(name, O_RDONLY);
	if (fd < 0) {
		status = errno;
		printf("Failed to open file %s.\n", name);
		goto done;
	}
	if (fstat (fd, &fs) < 0) {
		status = errno;
		printf("Failed to fstat file %s.\n", name);
		goto done;
	}
	*size = fs.st_size;
	fdt = mmap(NULL, fs.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (fdt == MAP_FAILED) {
		fdt = NULL;
		status = errno;
		printf("Failed to mmap file %s.\n", name);
		goto done;
	}
	*buf = malloc(fs.st_size);
	if (*buf == NULL) {
		status = errno;
		printf("Failed alloc memory for fdt buffer.\n");
		goto done;
	}
	status = fdt_open_into(fdt, *buf, fs.st_size);
	if (status) {
		printf("Failed open fdt from file %s.\n", name);
		goto done;
	}
	status = fdt_check_header(*buf);
	if (status) {
		printf("Invalid DTB header\n");
		goto done;
	}

done:
	if (status) {
		if (*buf)
			free(*buf);
		*buf = NULL;
		*size = 0;
	}
	if (fdt)
		munmap(fdt, fs.st_size);
	if (fd >= 0)
		close(fd);

	return status;
}

static int close_fdt(void *buf)
{
	free(buf);
	return 0;
}

static int composite_fdts(int argc, char **argv, int optind, void *comp_fdt)
{
	int status = 0;
	int curr_layer, i;
	void *fdt;
	char *file;
	int fdt_size = 0;
	bool match;
	int node;

	for (curr_layer = 1; curr_layer < 0xff && !status; curr_layer++) {
		for (i = optind; i < argc && !status; i++) {
			file = argv[i];
			status = open_fdt(file, &fdt, &fdt_size);
			if (status)
				goto done;
			status = check_layer_match(fdt, curr_layer, &match);
			if (status)
				goto done;
			if (!match)
				goto next;

			printf("Applying overlay %s.\n", file);
			status = apply_overlay(comp_fdt, fdt);
			if (status)
				goto done;
			status = find_node(comp_fdt, "chosen", 1, &node);
			if (status) {
				printf("Couldn't find /chosen node.\n");
				goto next;
			}
			status = fdt_appendprop_string(comp_fdt, node,
					"dtbs", file);
			if (status)
				printf("Couldn't add %s file to "
				      "/chosen/dtbs property.\n", file);

next:
			status = close_fdt(fdt);
			fdt = NULL;
			if (status)
				goto done;
		}
	}

done:
	if (fdt)
		close_fdt(fdt);
	return status;
}

int main(int argc, char **argv)
{
	int status = 0;
	int opt;
	char *outfile = "comp.dtb";
	int outfd = -1;
	void *fdt = NULL;
	int fdt_size = 0;
	int layer = 0;
	bool found;
	int i, n;
	void *p;

	while ((opt = getopt(argc, argv, "o:h")) >= 0) {
		switch (opt) {
		case 'h':
			usage();
			goto done;
		case 'o':
			outfile = optarg;
			break;
		case '?':
			printf("Invalid option \"%c\".\n", optopt);
			usage();
			status = -EINVAL;
			goto done;
		}
	}
	if (optind >= argc) {
		printf("No input file(s) specified.\n");
		usage();
		status = -EINVAL;
		goto done;
	}

	/* Find base DTB */
	found = false;
	for (i = optind; i < argc; i++) {
		status = open_fdt(argv[i], &fdt, &fdt_size);
		if (status)
			goto done;
		status = get_layer(fdt, &layer);
		if (status)
			goto done;
		if (layer == 0) {
			printf("Found base layer file %s.\n", argv[i]);
			found = true;
			break;
		}
		status = close_fdt(fdt);
		fdt = NULL;
		if (status)
			goto done;
	}
	if (!found) {
		printf("Failed to find base layer DTB.\n");
		status = -ENOENT;
		goto done;
	}

	status = fdt_open_into(fdt, comp_fdt, MAX_FDT_SIZE);
	if (status) {
		printf("Failed to copy base fdt to composite fdt "
		       "buffer with error %d\n", status);
		goto done;
	}
	free(fdt);
	fdt = NULL;

	printf("Removing platform-match node from base FDT.\n");
	status = del_node(comp_fdt, "/platform-match");
	if (status) {
		printf("Failed removing platform-match node from base FDT.\n");
		goto done;
	}

	status = find_node(comp_fdt, "chosen", 1, &n);
	if (status) {
		printf("Couldn't find /chosen node.\n");
		goto done;
	}
	status = fdt_appendprop_string(comp_fdt, n, "dtbs", argv[i]);
	if (status) {
		printf("Couldn't add %s string to "
		      "\"dtbs\" property.\n", argv[i]);
		goto done;
	}

	/* Apply the overlays */
	printf("Applying FDT overlays.\n");
	status = composite_fdts(argc, argv, optind, comp_fdt);
	if (status) {
		printf("Failed trying to composite FDT's.\n");
		goto done;
	}

	/* Minimize the size of the resulting FDT */
	printf("Packing composited FDT.\n");
	status = fdt_pack(comp_fdt);
	if (status) {
		printf("Failed trying to pack the composited FDT.\n");
		goto done;
	}

	/* Write out the composited FDT */
	outfd = open(outfile, O_CREAT | O_RDWR,
		     S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
	if (outfd < 0) {
		printf("Failed to open output file %s.\n", outfile);
		status = errno;
		goto done;
	}
	p = comp_fdt;
	n = write(outfd, comp_fdt, fdt_totalsize(p));
	if (n <= 0) {
		printf("Failed to write output file %s.\n", outfile);
		status = errno;
		goto done;
	}
	close(outfd);

done:
	if (fdt)
		close_fdt(fdt);
	if (outfd >= 0)
		close(outfd);

	return status;
}
