 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/jiffies.h>
#include <linux/debugfs.h>
#include <linux/sched.h>
#include <linux/phy.h>
#include <linux/interrupt.h>
#include <linux/kmod.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/platform_device.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/of.h>
#include <linux/of_fdt.h>
#include <linux/of_address.h>
#include <linux/io.h>
#include <linux/dma-contiguous.h>
#include <linux/reboot.h>
#include <linux/libfdt.h>
#include <linux/regulator/consumer.h>

#include <brcm_mbox.h>
#include <brcm_ba.h>
#include "brcm_ba_priv.h"
#ifdef CONFIG_BCM_BA_HARDCODE_MAP
#include "brcm_ba_map.h"
#endif

/**
* DOC: Bootassist Overview
*
* This kernel module is responsible for providing boot assist operations for
* Cable Modem CPU (CM) and Residential Gateway CPU (RG). The primary tasks
* are to program Address Range Checker (ARC) HW, program Address Translation
* Window (ATW) HW, and provide boot images for CM and RG. In addition servicing
* power down and reboot requests from CM and RG, and power management functions.
*
* - Public API's
* No API's are meant to be called outside of this module.
* - TBD's
* Post boot power management
* AVS
*
* High Level Usage Flow:
* Below is a high level view of a "normal" boot on ac process.
* Kernel module (KM) and user space "helper" app (US) start. The kernel module
* will obtain memory start location and size from bmem. The KM will obtain
* memory mapping information from hard coded values, OF, or US. US is
* responsible for overall control to keep the KM as static/stable as possible.
* US will query state information from KM, send any configuration data, retrieve
* any configuration data, and power on CM (unless it is already on). US will
* request (via ioctl) that KM applues the mapping that it has obtained. This
* means programming all the ATW's and ARC's. Note that RG ARC is NOT enabled
* until power on time. US will then poll the KM waiting for either CM/RG/BMU
* state change, or a request from CM for data. Once powered on, CM should
* request data (KM will obtain notification either via polling or IRQ depending
* on compile options) which will reach the user space via it's poll on the KM's
* device node. The US will ioctl to get the CM request, and will have mapped
* physical memory The requested data is read from a file directly into mapped
* memory. A second ioctl will inform the KM data is ready for CM and where it
* is. The KM will pass this information to the CM. This continues as the CM
* requests any and all information it requires (boot loader, run time image,
* persistant nonvol, dynamic nonvol, and/or any others). When "done, the CM will
* indicate a "running" state. US will get this indication as it's polling on the
* KM's device node This will trigger the US to send an ioctl to KM to enter "C2"
* state. As part of this, US will send power state (for debugging purposes).
* This determines what "C2" state will do. In the case of "AC power", the US
* will then fill mapped physical memory with RG boot image. When done, it will
* send an ioctl to KM to power on the RG. The "normal" boot process is complete
* at this point.
*
* ARC/ATW Programming:
* Address Range CHecker registers must be configured before
* CM and RG CPU's can be used to assure memory is used correctly.
* Address Translation Windows must also be programmed.
* Bootassist will get a single block of memory to use via bmem (start and size).
* The overall layout of the partitioning is HARD CODED, but the specifics
* (sizes, enable, etc.) can be specified in a variety of ways. Based on compile
* time setting, this mapping can be HARD CODED. In this configuration it CANNOT
* BE CHANGED with recompiling. If not hard coded, the mapping can be obtained
* via device tree and/or from user space (human readable text file). The device
* tree may specify if mapping can be taken from user space or not (default is
* that it can). If allowed via device tree, and if the text file exists, the
* user space mapping will take precedence.
*
* CM boot process:
* CM will be powered on under all circumstances (unless it is already running).
* Power on is triggered by the user space app. Once on, The process will be
* driven by MBOX input from the CM itself (boot ROM/loader).
* - CM indicates run state (i.e. requesting data or not, which data, etc)
* - CM indicates image request
* - Kernel module gets request from CM and passes request to user space.
* - User space writes directly to mapped physical memory and indicates that
* data is "ready"
*    to the kernel module which will pass the memory location to the CM via
* MBOX.
* - This process is repeated as long as CM is requesting image data and
*   run status supports (i.e. not running, or halted, etc.)
*
* C2 state:
* C2 state is entered when CM has finished it's boot requests and is initiated
* by the user space process. It will (in all cases) set reboot_on_power_loss
* register. Then, depending on battery state passed in by the user process (to
* allow for any debug type operations, instead of using direct battery state as
* read by hardware) set either PM_SET_AUTO_BATTERY_MODE_RULES or
* PM_SET_FORCE_BATTERY_MODE.
*
* RG boot process:
* The RG will only be powered on if the battery state is AC (and it's not
* currently running). RG has no boot ROM and will not use the MBOX communication
* method. When the decision is made to power on RG, and BEFORE it is powered on:
* - User space has mapped the physical memory which will be given to RG. It will
* copy the boot
*    image directly to that location.
* - User space will indicate to kernel module (ioctl) to power on the RG
* - The kernel module will enable RG's ARC (to prevent anyone from accessing
* RG's memory)
*    and then power on the RG.
*
* Power Operation(s) Proxy:
* TBD
*
*/

#define MODULE_NAME	"brcm-ba"
#define MODULE_VER	"1.0"

static struct ba_dev_info *info;
static enum battery_state ba_boot_batt = BS_AC;
static enum battery_state ba_c2_batt = BS_UNKNOWN;
static rwlock_t map_lock;
static enum rg_battery_mode rg_batt_mode = RG_CPU_OFF_UTILITY;

struct pwr_supply {
	struct list_head list;
	char name[32];
	struct regulator *regulator;
};
static struct list_head board_supplies;

static const struct of_device_id of_platform_brcmba_table[] = {
	{ .compatible = "brcm,brcm-ba", },
	{},
};

/* Total size of memory set aside for management by BA.
* This will be obtained from device tree */
static unsigned long ba_mem_size;
module_param(ba_mem_size, ulong, S_IRUGO);
MODULE_PARM_DESC(ba_mem_size, "Managed memory size");

/* Physical start address of memory set aside for management by BA.
* This will be obtained from device tree */
static unsigned long ba_mem_start;
module_param(ba_mem_start, ulong, S_IRUGO);
MODULE_PARM_DESC(ba_mem_start, "Managed memory physical start address");

/* FPM Size... this will be obtained from FPM device tree node, defaults to 0 */
static unsigned long ba_fpm_size;
module_param(ba_fpm_size, ulong, S_IRUGO);
MODULE_PARM_DESC(ba_fpm_size, "FPM reserved size");

/* FPM start. this will be obtained from FPM device tree node, defaults to 0 */
static unsigned long ba_fpm_start;
module_param(ba_fpm_start, ulong, S_IRUGO);
MODULE_PARM_DESC(ba_fpm_start, "FPM reserved start");

/* MEMC0 size, obtained from DT, defaults to 1GB */
unsigned long ba_memc0_size = 1*1024*1024*1024;
module_param(ba_memc0_size, ulong, S_IRUGO);
MODULE_PARM_DESC(ba_memc0_size, "MEMC0 size");

static int ba_notify_reboot(struct notifier_block *this,
			    unsigned long code, void *x);

static struct notifier_block brcm_ba_notifier = {
	.notifier_call  = ba_notify_reboot,
	.next           = NULL,
	.priority       = 0,
};

static inline char *ba_ioctl_to_string(int cmd)
{
	switch (cmd) {
	case BA_IOCTL_GET_VERSION:
		return "Get Version";
	case BA_IOCTL_POWER:
		return "Power Request";
	case BA_IOCTL_C2INIT:
		return "C2 Init";
	case BA_IOCTL_C2FINISH:
		return "C2 Finish";
	case BA_IOCTL_GETMAP:
		return "Get map Info";
	case BA_IOCTL_SETMAP:
		return "Program ATW/ARC";
	case BA_IOCTL_GET_DTB:
		return "Get DTB";
	case BA_IOCTL_GET_OTP:
		return "Get OTP";
    case BA_IOCTL_BOOT_ERROR:
        return "Boot error";
	default:
		break;
	}
	return "Unknown/Invalid";
}

#ifdef CONFIG_BCM_BA_PERF

static struct ba_metric metrics;
static unsigned int ioctlcnt;

/**
* ba_dbgfs_metrics_read() - Provide a dump of collected metrics to user.
* Note: very inefficient for small access, will generate full text each time
* it is called, regardless of the size requested. It's usage/nature doesn't
* really call for improvement
*/
static int ba_dbgfs_metrics_read(struct file *fp, char __user *user_buffer,
			size_t count, loff_t *position)
{
	char *l_buf;
	int ret = 0;
	int i = 0;
	int cnt = 0;

	/* No error checking on this size! */
	/* Assume max 100 chars per line for max collected irq's
	 * and ioctls */
	l_buf = kmalloc((MAX_IOCTL_METRIC+1)*100, GFP_KERNEL);
	if (!l_buf)
		return -ENOMEM;

	cnt += sprintf(&l_buf[cnt], "Start jiffy: 0x%lx\n", metrics.start_jiff);
	cnt += sprintf(&l_buf[cnt], "jiffies, ms from start, description\n");

	while (i < ioctlcnt) {
		cnt += sprintf(&l_buf[cnt], "0x%lx, %u, IOCTL %s\n",
			       metrics.ioctl[i].jiff,
			       jiffies_to_msecs(metrics.ioctl[i].jiff-
						metrics.start_jiff),
			       ba_ioctl_to_string(metrics.ioctl[i].ioctl));
		i++;
	}

	ret = simple_read_from_buffer(user_buffer, count, position, l_buf, cnt);
	kfree(l_buf);

	return ret;
}

static const struct file_operations fops_dbg_metric = {
	.read = ba_dbgfs_metrics_read,
};
#endif

#ifdef CONFIG_BCM_BA_DEBUGFS_MAP
/**
* dump_atw() - Provide a dump to user of ATW usage (via debugfs file)
* For now only show memc0 although it could be expanded for memc1
*/
static int dump_atw(char *page, int cnt, int page_size, enum ba_dev_atw dev)
{
	int i;
	u32 tmp;
	u32 source, dest, size;

	if (dev >= ATW_MAX)
		return -1;

	R32(dest, MC_ATW_SOURCE_START_ADDRESS(info->memc0) + ATW_IDX(dev));
	dest = dest<<16;
	R32(source, MC_ATW_DESTINATION_START_ADDRESS(info->memc0) +
	    ATW_IDX(dev));
	source = source<<16;
	R32(size, MC_ATW_SIZE_MASK(info->memc0) + ATW_IDX(dev));
	size = (64*1024)*(size+1);

	R32(tmp, MC_ATW_ENABLE(info->memc0) + ATW_IDX(dev));

	if (tmp & 0x1) {
		cnt += snprintf(page+cnt, page_size-cnt, "ATW %d: ON\n", dev);
		if (size < 1*1024*1024)
			cnt += snprintf(page+cnt, page_size-cnt, "\t %uKB@0x%08x -> 0x%08x\n",
			size/1024, source, dest);
		else
			cnt += snprintf(page+cnt, page_size-cnt, "\t %uMB@0x%08x -> 0x%08x\n",
			size/1024/1024, source, dest);

		cnt += snprintf(page+cnt, page_size-cnt, "\t Imposed on client(s):\n\t\t");

		R32(tmp, MC_ATW_CLIENTS_MATCH_UBUS_31_0(info->memc0) +
		    ATW_IDX(dev));

		for (i = 0; i <= 223; i++) {
			switch (i) {
			case 0:
				R32(tmp,
				    MC_ATW_CLIENTS_MATCH_UBUS_31_0(info->memc0)
				    + ATW_IDX(dev));
				break;
			case 32:
				i = 64;
				R32(tmp,
				    MC_ATW_CLIENTS_MATCH_UBUS_95_64(info->memc0)
				    + ATW_IDX(dev));
				break;
			case 96:
				i = 128;
				R32(tmp,
				    MC_ATW_CLIENTS_MATCH_UBUS_159_128(
				    info->memc0) + ATW_IDX(dev));
				break;
			case 160:
				i = 192;
				R32(tmp,
				    MC_ATW_CLIENTS_MATCH_UBUS_223_192(
				    info->memc0) + ATW_IDX(dev));
				break;
			}

			if ((tmp&0x1) == 0x1)
				cnt += snprintf(page+cnt, page_size-cnt, "0x%02x ", i);

			tmp = tmp>>1;
		}
		cnt += snprintf(page+cnt, page_size-cnt, "\n");
	} else {
		cnt += snprintf(page+cnt, page_size-cnt, "ATW %d: OFF\n", dev);
	}

	return cnt;
}

#ifdef ARC_SUPPORT /* Not supported for all chips */
/**
* dump_arc_single() - Detailed dump of a single ARC
*/
static inline int dump_arc_single(char *page, int cnt, int page_size,
				  void __iomem *base, int m, int b, int i)
{
	u32 data;
	int x;
	u32 client;
	u64 start, end;

	switch (b) {
	case 0:
		R32(data, SCB_ARC_CNTRL(base, i));
		if (!data) {
			cnt += snprintf(page+cnt, page_size-cnt, "MC%d SCB  ARC[%d]: OFF\n", m, i);
			return cnt;
		}

		R32(start, SCB_ARC_LOW(base, i));
		R32(end, SCB_ARC_HIGH(base, i));
		start = start << 3;
		end = end << 3;
		cnt += snprintf(page+cnt, page_size-cnt,
				"MC%d SCB  ARC[%d]: ON %s 0x%llx to 0x%llx\n",
				m, i, (data & SCB_CNTRL_MODE_MASK) ?
				"Exclusive" : "", start, end);
		cnt += snprintf(page+cnt, page_size-cnt, "\t For READ Clients: ");
		for (x = 0; x < MAX_ARC_CLIENTS; x++) {
			client = 32 * x;
			R32(data, SCB_RR(base, i, x));
			while (data) {
				if (data & 0x1)
					cnt += snprintf(page+cnt, page_size-cnt, "0x%x ",
							client);
				client++;
				data = data >> 1;
			}
		}
		cnt += snprintf(page+cnt, page_size-cnt, "\n\t For WRITE Clients: ");
		for (x = 0; x < MAX_ARC_CLIENTS; x++) {
			client = 32 * x;
			R32(data, SCB_WR(base, i, x));
			while (data) {
				if (data & 0x1)
					cnt += snprintf(page+cnt, page_size-cnt, "0x%x ",
							client);
				client++;
				data = data >> 1;
			}
		}
		cnt += snprintf(page+cnt, page_size-cnt, "\n");
		break;
#ifdef UBUS_ARC_CNTRL
	case 1:
		R32(data, UBUS_ARC_CNTRL(base, i));
		if (!data) {
			cnt += snprintf(page+cnt, page_size-cnt, "MC%d UBUS ARC[%d]: OFF\n", m,
					i);
			return cnt;
		}

		R32(start, UBUS_ARC_LOW(base, i));
		R32(end, UBUS_ARC_HIGH(base, i));
		start = start << 3;
		end = end << 3;

		cnt += snprintf(page+cnt, page_size-cnt,
			       "MC%d UBUS ARC[%d]: ON %s 0x%llx to 0x%llx\n",
			       m, i, (data & UBUS_CNTRL_MODE_MASK) ?
			       "Exclusive" : "", start, end);
		cnt += snprintf(page+cnt, page_size-cnt,
				"\t For READ Clients: ");
		for (x = 0; x < MAX_ARC_CLIENTS; x++) {
			client = 32 * x;
			R32(data, UBUS_RR(base, i, x));
			while (data) {
				if (data & 0x1)
					cnt += snprintf(page+cnt, page_size-cnt,
							"0x%x ", client);
				client++;
				data = data >> 1;
			}
		}
		cnt += snprintf(page+cnt, page_size-cnt, "\n\t For WRITE Clients: ");
		for (x = 0; x < MAX_ARC_CLIENTS; x++) {
			client = 32 * x;
			R32(data, UBUS_WR(base, i, x));
			while (data) {
				if (data & 0x1)
					cnt += snprintf(page+cnt, page_size-cnt,
							"0x%x ", client);
				client++;
				data = data >> 1;
			}
		}
		cnt += snprintf(page+cnt, page_size-cnt, "\n");
		break;
#endif
	default:
		return 0;
	}

	return cnt;
}

/**
* dump_arc() - Provide a dump to user of ARC usage (via debugfs file)
*/
static int dump_arc(char *page, int cnt, int page_size)
{
	int m, b, i;
	void __iomem *base;

	for (m = 0; m < MAX_ARC_MEMC; m++) {
		switch (m) {
		case 0:
			base = info->memc0;
			break;
#if MAX_ARC_MEMC > 1
		case 1:
			base = info->memc1;
			break;
#endif
		/* coverity[dead_error_begin] */
		default:
			base = NULL;
			break;
		}
		if (!base)
			continue;

		for (b = 0; b < MAX_ARC_BUS; b++) {
#ifndef UBUS_ARC_CNTRL
			if (b > 0)
				continue;
#endif
			for (i = 0; i < MAX_ARC; i++)
				cnt = dump_arc_single(page, cnt, page_size,
							base, m, b, i);
		}
	}

	return cnt;
}
#else
/**
* dump_arc() - Provide a dump to user of ARC usage (via debugfs file)
*/
static int dump_arc(char *page, int cnt)
{
	return cnt;
}
#endif

#define MAP_BUFFER_LEN 2000
/**
* ba_dbgfs_map_read() - Provide a dump to user of memory configuratin (ATW/ETC)
* (via debugfs file). Note there is no handling for requesting small data
* (i.e. reading x characters at a time, etc.)
*/
static ssize_t ba_dbgfs_map_read(struct file *fp, char __user *user_buffer,
			size_t count, loff_t *position)
{
	int cnt = 0;
	char *l_buf = NULL;

	/* No error checking on this size! */
	l_buf = kmalloc(MAP_BUFFER_LEN, GFP_KERNEL);
	if (!l_buf)
		return -ENOMEM;


	cnt = dump_atw(l_buf, cnt, MAP_BUFFER_LEN, ATW_CM);
	cnt = dump_atw(l_buf, cnt, MAP_BUFFER_LEN, ATW_CM_DSP);
	cnt = dump_atw(l_buf, cnt, MAP_BUFFER_LEN, ATW_CM_BOOT);
	cnt = dump_atw(l_buf, cnt, MAP_BUFFER_LEN, ATW_CM_EXTRA);
	cnt = dump_atw(l_buf, cnt, MAP_BUFFER_LEN, ATW_CM_BOOT_EXTRA);
	cnt = dump_arc(l_buf, cnt, MAP_BUFFER_LEN);
	cnt = simple_read_from_buffer(user_buffer, count, position, l_buf, cnt);
	kfree(l_buf);
	return cnt;
}

static const struct file_operations fops_dbg_map = {
	.read = ba_dbgfs_map_read,
};
#endif

/**
* ba_map_res() - Map a resource
* @pdev: Platform device pointer
* @mem: mem pointer
* @name: Name to search for
* @exclusive: Map exclusive or shared
*/
static inline int ba_map_res(struct platform_device *pdev, void __iomem **mem,
				  char *name, bool exclusive)
{
	int i = 0;
	struct resource *res;

	if (!name)
		return -1;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		return -1;

	if (strstr(res->name, "ba")) {
		/* LEGACY.... some NOT GOOD stuff here.... TO BE
		REMOVED as soon as device tree version 2.0 is
		deemed no longer to be supported! */
		if (strncmp(name, "mem", strlen("mem")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_MEMBLOCK);
		} else if (strncmp(name, "cpuc", strlen("cpuc")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_COMM);
		} else if (strncmp(name, "mc0", strlen("mc0")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_ATW);
			/* For legacy support... convert ATW address
			* to a MEMC0 address */
			/* This is BAD... */
			/* "End" doesn't need to be adjusted */
			res->start -= MC_ATW_OFFSET;
		} else if (strncmp(name, "aon", strlen("aon")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_AON);
		} else if (strncmp(name, "rgtop", strlen("rgtop")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_RG_TOP);
		} else if (strncmp(name, "cmtop", strlen("cmtop")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_CM_TOP);
		} else if (strncmp(name, "otp", strlen("otp")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_OTP);
		} else if (strncmp(name, "leap", strlen("leap")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_LEAP);
		} else if (strncmp(name, "hif", strlen("hif")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_HIF);
		} else if (strncmp(name, "leds", strlen("leds")) == 0) {
			res = platform_get_resource(pdev, IORESOURCE_MEM,
						    IO_MEM_LEDS);
        } else {
			return -1;
		}
	} else {
		do {
			if (strncmp(res->name, name, strlen(name)) == 0)
				break;
		} while ((res = platform_get_resource(pdev, IORESOURCE_MEM,
						       i++)));
	}

	if (!res)
		return -1;

	if (!strncmp(name, "mem", strlen("mem"))) {
		/* Special case here... not really mapping anything */
		/* This defines a memory block */
		/* Really only used in legacy DT and 7145A0 */
		ba_mem_start = res->start;
		ba_mem_size = resource_size(res);
		return 0;
	}

	if (!mem)
		return -1;

	if (exclusive)
		*mem = devm_ioremap_resource(&pdev->dev, res);
	else
		*mem = devm_ioremap(&pdev->dev, res->start, resource_size(res));

	if (unlikely(!*mem)) {
		MSG_TRACEE("Error getting IO resource %s\n", res->name);
		return -1;
	}

	return 0;
}


/**
* ba_config_dbgfs() - Create/Free debugfs files
* @create: If true, create, else free
*
* For debugging purposes, various debugfs files can be created for things such
* as showing memory configuration and prototype/testing (w/o real hardware
* CM/RG)
*
*/
static int ba_config_dbgfs(bool create)
{
	static struct dentry *parent;
	static struct dentry *map_dentry;
	static struct dentry *metric_dentry;

	FUNC_TRACE(1);

	if (create) {
		parent = debugfs_create_dir(DEBUG_FS_DIR, NULL);
		if ((!parent) || (parent == ERR_PTR(-ENODEV))) {
			MSG_TRACEE("Failed to create debugfs directory\n");
			FUNC_TRACE(0);
			parent = NULL;
			return -1;
		}
#ifdef CONFIG_BCM_BA_DEBUGFS_MAP
		map_dentry = debugfs_create_file("map", S_IRUGO, parent, NULL,
				&fops_dbg_map);
#endif

#ifdef CONFIG_BCM_BA_PERF
		metric_dentry = debugfs_create_file("metrics", S_IRUGO, parent,
				NULL, &fops_dbg_metric);
#endif
	} else {
		if ((metric_dentry) && (metric_dentry != ERR_PTR(-ENODEV))) {
			debugfs_remove(metric_dentry);
			metric_dentry = NULL;
		}

		if ((map_dentry) && (map_dentry != ERR_PTR(-ENODEV))) {
			debugfs_remove(map_dentry);
			map_dentry = NULL;
		}

		if ((parent) && (parent == ERR_PTR(-ENODEV))) {
			debugfs_remove(parent);
			parent = NULL;
		}
	}

	FUNC_TRACE(0);
	return 0;
}

/**
* ba_open() - Device node opening
*
* Just puts private data into the file struct
*/
static int ba_open(struct inode *inode, struct file *file)
{
	FUNC_TRACE(1);

	file->private_data = (void *)info;

	FUNC_TRACE(0);
	return 0;
}

/**
* ba_release() - Device node closing
*/
static int ba_release(struct inode *inode, struct file *file)
{
	FUNC_TRACE(1);

	FUNC_TRACE(0);
	return 0;
}

/**
* ba_read() - TBD for future development
*/
static ssize_t ba_read(struct file *file, char __user *buf, size_t size,
			loff_t *ppos)
{
	FUNC_TRACE(1);

	FUNC_TRACE(0);
	return size;
}

/**
* ba_write() - TBD for future development
*/
static ssize_t ba_write(struct file *file, const char __user *buf,
			size_t size, loff_t *ppos)
{
	FUNC_TRACE(1);
	FUNC_TRACE(0);
	return size;
}

/**
* ba_atw() - Program/enable/disable ATW
* @arg idx: (enum eDeviceAtw)
* @arg on: true to enable, false to disable
* @arg source: Source address
* @arg size: Size of ATW
* @arg dest: Destination of ATW
*
* Setup and enable/disable an ATW. Will NOT allow programming if already
* enabled. For now only memc0 supported.
*/
static int ba_atw(struct ba_atw atw, bool on)
{
	int ret = -1, i;

	FUNC_TRACE(1);

	if (!on) {
		MSG_TRACEN("Disable ATW %d\n", atw.index);
		W32(MC_ATW_ENABLE(info->memc0) + ATW_IDX(atw.index), 0);
		ret = 0;
		goto ATW_EXIT;
	}

	/* At this point, know it's a request to program and enable ATW */
	/* Don't allow anything that's previously been enabled to be changed
	* on the fly! */
	if (RF32(MC_ATW_ENABLE(info->memc0) + ATW_IDX(atw.index), ATW_ENABLE)) {
		MSG_TRACEE("ATW %d ALREADY configured!\n", atw.index);
		goto ATW_EXIT;
	}

	if (!atw.size) {
		/* TBD.. not an error? Just no ATW? */
		MSG_TRACEI("No ATW[%d] size specified\n", atw.index);
		ret = 0;
		goto ATW_EXIT;
	}

	for (i = 0; i < NUM_ATW_CLIENTS; i++)
		if (atw.clients[i])
			break;
	if (i >= NUM_ATW_CLIENTS) {
		/* TBD.. not an error? Just no ATW? */
		MSG_TRACEI("No ATW[%d] clients specified\n", atw.index);
		ret = 0;
		goto ATW_EXIT;
	}

	/* Check alignments... start and destination must be aligned
	* to the size, and the size must be >= 64k and a power of 2 */
	/* TBD: ASSUMES source and dest are 32 bit! */
	if ((((u32)atw.source%atw.size) != 0) ||
	    (((u32)atw.dest%atw.size) != 0)) {
		MSG_TRACEE("ATW %d Alignment failure\n", atw.index);
		MSG_TRACEE("ATW source %08x dest %08x size %08x\n",
		       (u32)atw.source, (u32)atw.dest, atw.size);
		goto ATW_EXIT;
	}
	if ((atw.size < 65536) || (atw.size & (atw.size - 1))) {
		MSG_TRACEE("ATW %d Alignment failure\n", atw.index);
		MSG_TRACEE("ATW size %08x\n", atw.size);
		goto ATW_EXIT;
	}


	/* Setup ATW */
	MSG_TRACEN("Setup ATW %d\n", atw.index);
	W32(MC_ATW_SOURCE_START_ADDRESS(info->memc0) + ATW_IDX(atw.index),
	    atw.dest>>16);
	W32(MC_ATW_DESTINATION_START_ADDRESS(info->memc0) + ATW_IDX(atw.index),
	    (atw.source>>16));
	W32(MC_ATW_SIZE_MASK(info->memc0) + ATW_IDX(atw.index),
	    ((atw.size-1)>>16));

	W32(MC_ATW_CLIENTS_MATCH_UBUS_31_0(info->memc0) + ATW_IDX(atw.index),
	    atw.clients[0]);
	W32(MC_ATW_CLIENTS_MATCH_UBUS_95_64(info->memc0) + ATW_IDX(atw.index),
	    atw.clients[1]);
	W32(MC_ATW_CLIENTS_MATCH_UBUS_159_128(info->memc0) + ATW_IDX(atw.index),
	    atw.clients[2]);
	W32(MC_ATW_CLIENTS_MATCH_UBUS_223_192(info->memc0) + ATW_IDX(atw.index),
	    atw.clients[3]);

	MSG_TRACEN("Enable ATW %d\n", atw.index);
	W32(MC_ATW_ENABLE(info->memc0) + ATW_IDX(atw.index), 1);

	/* All went ok */
	ret = 0;

ATW_EXIT:
	FUNC_TRACE(0);
	return ret;
}

#ifdef ARC_SUPPORT /* Not supported for all chips */
/**
* ba_arc() - Program/enable/disable ARC
* @arg idx: (enum ba_dev_arc)
* @arg on: true to enable, false to disable
* @arg source: Source address
* @arg size: Size of ARC
* @arg stb: In case of stb arc, need additional index
*
* Setup and enable/disable an ARC. Will not allow programming if already
* enabled. No process to disable an ARC until/unless some ioctl is
* added for control from user space.
*/
static int ba_arc(u8 memc, u8 bus, u8 index, bool enable)
{
	u32 cntrl;
	int i;
	void __iomem *base = NULL;
	struct ba_arc *arc;

	if (unlikely(!info))
		return -1;

	switch (memc) {
	case 0:
		base = info->memc0;
		break;
#if MAX_ARC_MEMC > 1
	case 1:
		base = info->memc1;
		break;
#endif
	default:
		break;
	}

	arc = &info->map->arc[memc][bus][index];

	/* See if valid */
	if (unlikely(((index >= MAX_ARC) || (bus >= MAX_ARC_BUS) || (!base))))
		return -1;

	if (arc->start == (typeof(arc->start))~0)
		return 0; /* Nothing to do, settings invalid */

	MSG_TRACEI("%s %s ARC %d: %s\n", enable ? "ENABLE" : "DISABLE",
		   bus ? "UBUS" : "", index,
		   arc->exclusive ? "EXCLUSIVE" : "");

	/* Just for simplicity */
	if ((arc->start & 0xFFFFFFFF00000000) ||
	    (arc->end & 0xFFFFFFFF00000000)) {
		pr_err("64bit arc not supported\n");
		return -1;
	}

#ifdef UBUS_ARC_CNTRL
	if (bus)
		R32(cntrl, UBUS_ARC_CNTRL(base, index));
	else
#endif
		R32(cntrl, SCB_ARC_CNTRL(base, index));

	if (enable && cntrl) {
		MSG_TRACEE("ARC %d already enabled\n", index);
		return -1;
	} else if (!enable && !cntrl) {
		MSG_TRACEE("ARC %d already disabled\n", index);
		return -1;
	}

	ARC_CNTRL(cntrl, enable, arc->exclusive);

#ifdef UBUS_ARC_CNTRL
	if (bus)
		ARC_UBUS(base, index, arc, i, cntrl);
	else
#endif
		ARC_SCB(base, index, arc, i, cntrl);

	if (enable) {
		if (info->arc_enabled[memc] == 0) {
			MSG_TRACEI("Enabling MC%d ARC interrupt(s)\n", memc);
			MC_INT_ENABLE(base);
		}
		info->arc_enabled[memc]++;
	} else {
		info->arc_enabled[memc]--;
		if (info->arc_enabled[memc] == 0) {
			MSG_TRACEI("Disabling MC%d ARC interrupt(s)\n", memc);
			MC_INT_DISABLE(base);
		}
	}

	return 0;
}
#else
static int ba_arc(u8 memc, u8 bus, u8 index, bool enable)
{
	return 0;
}
#endif

/**
* brcm_ba_power() - CM/RG power operations
* @arg dev_id: (enum eDevice) Either CM or RG
* @arg on: true to power on, false to power off
*
* Power on/off RG or CM. In the case of RG, this also enables/disables the
* associated ARC (since STB needs access temporarily to write boot image)
*/
static void brcm_ba_power(enum ba_device dev_id, bool on)
{
	u32 tmp = 0;

	FUNC_TRACE(1);

	switch (dev_id) {
	case DEV_CM:
		MSG_TRACEI("Powering %s CM\n", on ? "on" : "off");
		SWF32(tmp, EXT_SYS_SW_INIT, 1);
		SWF32(tmp, CPU_SW_INIT, 1);
		SWF32(tmp, CM_SYS_CTRL_SW_INIT, 1);
		if (on)
			W32(CM_TOP_CTRL_SW_INIT_0_CLEAR, tmp);
		else
			W32(CM_TOP_CTRL_SW_INIT_0_SET, tmp);
		break;
	default:
		MSG_TRACEE("Invalid power request (%d)\n", dev_id);
		break;
	}

	FUNC_TRACE(0);
	return;
}

/**
* brcm_ba_config_memory() - Configure managed memory
*
* Responsible for determining reserved memory (bmem for now), and applying
* mapping (provided as hard coded data, or passed from device tree or from user
* space) to the reserved memory, programming ATW's and ARC's for the various
* CPU's being managed. Note that the RG has a special case where it's ARC is
* NOT enabled here, but only before it is turned on.
*/
static int brcm_ba_config_memory(void)
{
	struct ba_map *map = info->map;
	void __iomem *leap;
	int ret = 0;
	u32 tmp;
	int i, m, b;

	FUNC_TRACE(1);

	if (!info->flags&MAP_VALID) {
		MSG_TRACEE("Mapping NOT configured!\n");
		FUNC_TRACE(0);
		return -EAGAIN;
	}

	if (info->flags&MAP_PROGRAMMED) {
		MSG_TRACEE("Mapping already set, not reprogramming\n");
		FUNC_TRACE(0);
		return -EBUSY;
	}

	read_lock(&map_lock);

	/* CM is next block */
	if (map->docsis.cm.cm_atw.dest != 0)
		MSG_TRACEE("Non-standard CM ATW. System may not function\n");

	ret |= ba_atw(map->docsis.cm.cm_atw, 1);
	if (map->docsis.cm.cm_atw_extra.index < MAX_ATW) {
		MSG_TRACEI("index %d source %08x size %08x dest %08x clients %08x\n",
			   map->docsis.cm.cm_atw_extra.index,
			   (u32)map->docsis.cm.cm_atw_extra.source,
			   map->docsis.cm.cm_atw_extra.size,
			   (u32)map->docsis.cm.cm_atw_extra.dest,
			   map->docsis.cm.cm_atw_extra.clients[0]);
		ret |= ba_atw(map->docsis.cm.cm_atw_extra, 1);
	}

	/* LEAP */
	/* Note that LEAP ATW is handled by a built in LEAP specific ATW */
	/* LEAP physical must be 1MB aligned */
	if (LEAP_BASE(info) & 0x000FFFFF)
		MSG_TRACEE("LEAP Alignment ERROR! LEAP MAY NOT FUNCTION\n");

	if (unlikely(ba_map_res(info->pdev, &leap, "leap", 0) != 0)) {
		MSG_TRACEE("Error getting LEAP resource...\n");
		MSG_TRACEE("LEAP might be non-functional\n");
	} else {
		/* 20-31 is location in MB. 0 is enable */
		tmp = LEAP_BASE(info) & 0xFFF00000;
		tmp |= 0x1;
		tmp = cpu_to_be32(tmp);
		/* Have to setup byte swap first */
		W32(LEAP_CTRL_BYTE_SWAP_EN(leap), 0x03030303);
		W32(LEAP_CTRL_ADDR_TRANS(leap), tmp);

		iounmap(leap);
	}

	/* CM-DSP */
	ret |= ba_atw(map->docsis.dsp.atw, 1);

	/* CM BOOT */
	if (BOOT_ROM_BYPASS(info)) {
		ret |= ba_atw(map->docsis.cm.boot_atw, 1);
		if (map->docsis.cm.boot_atw_extra.index < MAX_ATW) {
			ret |= ba_atw(map->docsis.cm.boot_atw_extra, 1);
		}
	}

	/* DEBUG */
	if (map->arm.btrace_atw.index < MAX_ATW) {
		ret |= ba_atw(map->arm.btrace_atw, 1);
	}

	/* CM-STB communication, nothing to do */

	/* FPM, nothing to do */

	/* Any ARC's */
	for (m = 0; m < MAX_ARC_MEMC; m++)
		for (b = 0; b < MAX_ARC_BUS; b++)
			for (i = 0; i < MAX_ARC; i++)
				ba_arc(i, b, m, 1);

	info->flags |= MAP_PROGRAMMED;
	read_unlock(&map_lock);

	FUNC_TRACE(0);
	return ret;
}

/**
* ba_ioctl() - ioctls for boot assist user/kernel interaction
* @arg cmd: Ioctl/command
* @arg arg: data from/to user space
*
* Supported ioctls are
* BA_IOCTL_BLKSIZE - Get MBOX block size from user space
* BA_IOCTL_POWER - Request from user space to power on/off a processor
* BA_IOCTL_C2INIT - Initiate C2 stage
* BA_IOCTL_C2FINISH - Complete C2 stage in NON-AC case.
* BA_IOCTL_MAPINFO - If enabled, allow user space to provide mapping infomation
* BA_IOCTL_SETMAP - Apply current mapping information (one time operation)
* BA_IOCTL_MEMLOC - Provide physical memory locations for xfer to user space
* BA_IOC_GET_STATUS_16 - Provide status_16 to user space
*/
static long ba_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct ba_dev_info *pinfo = (struct ba_dev_info *)file->private_data;
	struct power_request power_req;
	struct c2_req c2;
	struct dtb_req dtb_request;
	int ret = 0;
	u32 tmp;

	FUNC_TRACEF(1, "Ioctl %s\n", ba_ioctl_to_string(cmd));

	METRIC_IOCTL(cmd);

	/* Shut down much of this until basic requirements are met */
	if (!info) {
		FUNC_TRACE(0);
		return -EAGAIN;
	}

	switch (cmd) {
	case BA_IOCTL_GETMAP:
		/* Provide mapping info to user space */
		read_lock(&map_lock);
		if (!pinfo->map) {
			read_unlock(&map_lock);
			MSG_TRACEI("No mapping info exists\n");
			ret = -ENXIO;
			break;
		}
		ret = copy_to_user((void *)arg, pinfo->map,
				   sizeof(*pinfo->map));
		read_unlock(&map_lock);

#if !defined(CONFIG_BCM_BA_HARDCODE_MAP) && \
defined(CONFIG_BCM_BA_ALLOW_USER_MAP)
		/* Note: Unless getting mapping information is prevented at
		* compile time, free up memory here and allow user space to
		* track/maintain mapping information */
		if ((ret == 0) && info->allow_user_map) {
			write_lock(&map_lock);
			devm_kfree(&pinfo->pdev->dev, info->map);
			pinfo->map = NULL;
			info->flags &= ~MAP_VALID;
			write_unlock(&map_lock);
		}
#endif
		break;
	case BA_IOCTL_SETMAP:
		/* Apply stored mapping info */
		/* A few scenario's here. If mapping is hard coded, or mapping
		* from user space is not allowed, then pinfo->map MUST already
		* exist and should be persistent. Otherwise, user space will
		* provide mapping info, and it will maintain the info for any
		* re-use needs */
		if (pinfo->flags & MAP_PROGRAMMED)
			break; /* Don't need the map... can't be reprogrammed */
#if !defined(CONFIG_BCM_BA_HARDCODE_MAP) && \
defined(CONFIG_BCM_BA_ALLOW_USER_MAP)
		/* Only copy from user if allowed */
		if (info->allow_user_map) {
			write_lock(&map_lock);
			if (!pinfo->map) {
				pinfo->map = devm_kzalloc(&pinfo->pdev->dev,
							 sizeof(*pinfo->map),
							 GFP_KERNEL);
				if (!pinfo->map) {
					ret = -ENOMEM;
					write_unlock(&map_lock);
					break;
				}
			}

			ret = copy_from_user(info->map, (void *)arg,
					     sizeof(*info->map));
			write_unlock(&map_lock);

			if (ret != 0)
				break;
			info->flags |= MAP_VALID;
		} else {
			/* Not allowing anything from user space.. copy back
			the used map (either hard coded or from DT) so that
			the user knows what is what */
			read_lock(&map_lock);
			ret = copy_to_user((void *)arg, info->map,
					   sizeof(*info->map));
			read_unlock(&map_lock);
			if (ret != 0)
				break;
		}

		MSG_TRACEN("Requested to program ARC/ATW\n");
		ret = brcm_ba_config_memory();

		/* Mapping only set once... don't bother keeping around a copy
		if the programming was succesfull.. TBD */
		if ((ret == 0) && info->allow_user_map) {
			write_lock(&map_lock);
			devm_kfree(&pinfo->pdev->dev, info->map);
			pinfo->map = NULL;
			info->flags &= ~MAP_VALID;
			write_unlock(&map_lock);
		}
#else
		if (unlikely(!(pinfo->map))) {
			ret = -EAGAIN;
			break;
		}
		MSG_TRACEN("Requested to program ARC/ATW\n");
		ret = brcm_ba_config_memory();
#endif
		break;
	case BA_IOCTL_POWER:
		/* Power on/off component from user request */
		if (unlikely(!(pinfo->flags&MAP_PROGRAMMED))) {
			MSG_TRACEE("Cannot power. Mapping not programmed\n");
			ret = -EAGAIN;
			break;
		}
		MSG_TRACEN("Power request from user space\n");
		ret = copy_from_user(&power_req, (void *)arg,
				   sizeof(struct power_request));
		if (ret != 0) {
			MSG_TRACEE("Failed to copy from user\n");
			break;
		}
		brcm_ba_power(power_req.device, power_req.on);
		break;
	case BA_IOCTL_C2INIT:
	{
		u8 bmu_ap_restored;
		void __iomem *comm;

		/* Init C2 stage */
		/* In theory, this would be tied to "real" battery state,
		* however, allow user space to pass in a battery state
		* to allow for debug type operations (i.e. such as NOT
		* shutting down STB when AC is actually present) */
		if (unlikely(!(pinfo->flags&MAP_PROGRAMMED))) {
			MSG_TRACEE("Cannot C2. Mapping not programmed\n");
			ret = -EAGAIN;
			break;
		}
		ret = copy_from_user(&c2, (void *)arg, sizeof(struct c2_req));
		if (ret != 0) {
			MSG_TRACEE("Failed to copy from user\n");
			break;
		}

		ba_c2_batt = c2.battery;
		ba_boot_batt = c2.boot_battery;

		/* No need to reset on AC loss since Docsis is already booted */
		WF32(AON_CTRL_PM_CTRL, REBOOT_ON_POWERLOSS, 0);

		if (unlikely(ba_map_res(pinfo->pdev, &comm, "cpuc", 0) != 0)) {
			MSG_TRACEE("Error getting IO resource\n");
			MSG_TRACEE("C2 will be incorrect\n");
		} else {
			/* Allows system to prepare for transition to BBM if AC
			 * lost */
			W32(CPU_COMM_REGS_CPUC_AP_LOST_MAJOR_DELAY(comm),
			    c2.ap_lost_major_dly&AP_LOST_DELAY_CNT_MASK);
			iounmap(comm);
		}

		bmu_ap_restored = RF32(AON_PM_BBM_L2_CPU_STATUS,
			BMU_INTR_AP_RESTORED);

		if ((ba_boot_batt != BS_AC) && (c2.battery != BS_AC) &&
		    (ba_boot_batt != BS_UNKNOWN) && (c2.battery != BS_UNKNOWN)) {
			if (bmu_ap_restored == 1) {
				tmp = 0;
				SWF32(tmp, BMU_INTR_AP_RESTORED, 1);
				W32(AON_PM_BBM_L2_CPU_CLEAR, tmp);
			}
		}

		if (c2.battery == BS_AC || c2.battery == BS_UNKNOWN) {
			/* TBD... Unmask AVS_top power interrupts */

			/*
			 * Only set the RG power plane to auto shutdown if we do
			 * not want any of the RG blocks powered while on battery.
			 */
			if(rg_batt_mode == RG_CPU_OFF_UTILITY ||
			   rg_batt_mode == RG_CPU_OFF_BATTERY)
				WF32(AON_CTRL_PM_CTRL,
				     PM_SET_AUTO_BATTERY_MODE_RULES, 1);
		} else {
			/* On battery... start "manual" shutdown process */
			/* User space will call "C2FINISH" once CM has indicated
			it's done with resources */
		}
		break;
	}
	case BA_IOCTL_C2FINISH:
		/* At this point, STB should have satisfied any wake
		* on battery interrupt should it have occurred.*/

		if (RF32(AON_PM_BBM_L2_CPU_STATUS, DCS_HEADEND_INTR)) {
			/* CM has woken STB for some proxy service... */
			/* TBD... can this happen while on AC? If so, this code
			* is NOT right! */
			tmp = 0;
			SWF32(tmp, DCS_HEADEND_INTR, 1);
			W32(AON_PM_BBM_L2_CPU_CLEAR, tmp);

			/* TBD.. indicate to CM it's ok to assert future
			ubus_software_intr wakeup interrupts */
		}

		/* TBD... AVS COMMUNICATION to DISENGAGE */
		/* TBD... AVS COMMUNICATION wait for ACK */

		/*
		 * Only set the RG power plane to auto shutdown if we do
		 * not want any of the RG blocks powered while on battery.
		 */
		if(rg_batt_mode == RG_CPU_OFF_UTILITY ||
		   rg_batt_mode == RG_CPU_OFF_BATTERY)
			/* Force BBM */
			WF32(AON_CTRL_PM_CTRL, PM_SET_FORCE_BATTERY_MODE, 1);
		break;
	case BA_IOCTL_GET_DTB: {
		void *dtb;
		unsigned int dtb_total_size;
		/*
		bfr = (uintptr_t)p.addr;
		bfr_len = p.num_bytes;
		*/
		dtb = initial_boot_params;

		ret = copy_from_user(&dtb_request, (void *)arg,
				sizeof(struct dtb_req));
		if (ret != 0) {
			MSG_TRACEE("invalid ioctl request!\n");
			break;
		}

		/*
		* The ePAPR spec says addr+totalsize should point to the
		* absolute end of the DTB.
		*/
		dtb_total_size = fdt_totalsize(dtb);

		/* derived from fdt.c::__unflatten_device_tree() */
		MSG_TRACEN("magic:   %08xh\n", fdt_magic(dtb));
		MSG_TRACEN("size:    %08xh\n", fdt_totalsize(dtb));
		MSG_TRACEN("version: %08xh\n", fdt_version(dtb));

		if ((dtb_request.buffer != NULL) &&
		    (dtb_request.size < dtb_total_size)) {
			MSG_TRACEE("Invalid buffer size\n");
			break;
		}

		dtb_request.size = dtb_total_size;
		ret = copy_to_user((void *)arg, &dtb_request,
				   sizeof(struct dtb_req));
		if (ret != 0) {
			MSG_TRACEE("Failed to return ioctl data!\n");
			break;
		}

		if (dtb_request.buffer == NULL) {
			/* User passes a NULL just to get the size */
			break;
		}

		ret = copy_to_user(dtb_request.buffer, dtb, dtb_total_size);
		if (ret)
			MSG_TRACEE("failed copy of DTB to user-space\n");
		break;
	}
	case BA_IOCTL_GET_OTP:
		ret = copy_to_user((void *)arg, &pinfo->otp_regs, sizeof(pinfo->otp_regs));
		if (ret)
			MSG_TRACEE("failed copy of OTP to user-space\n");
		break;
	case BA_IOCTL_GET_VERSION:
		tmp = BA_KVERSION;
		ret = copy_to_user((void *)arg, &tmp, sizeof(u32));
		if (ret)
			MSG_TRACEE("failed copy version to user-space\n");
		break;
	case BA_IOCTL_GET_RG_BATT_MODE:
		ret = copy_to_user((void *)arg, &rg_batt_mode,
				   sizeof(enum rg_battery_mode));
		if (ret)
			MSG_TRACEE("failed copy RG battery config to user-space\n");
		break;
	case BA_IOCTL_BOOT_ERROR:
        if (pinfo->leds) {

#define LED_60MS_FLASH          0x000000c0
#define LED_120MS_BLINK         0x00000006
#define LED_MODE_0_FAILURE      0x0000aaaa  // All flashing; indicates failure

            uint32_t *ledRegs = (uint32_t *)pinfo->leds;
            *ledRegs = LED_120MS_BLINK | LED_60MS_FLASH;
            *(ledRegs + 1) = LED_MODE_0_FAILURE;  // All flashing; indicates failure
        }
		break;
	default:
		MSG_TRACEE("Unsupported IOCTL 0x%x\n", cmd);
		ret = -EINVAL;
		break;
	}

	FUNC_TRACE(0);

	return ret;
}

#if defined(CONFIG_OF) && !defined(CONFIG_BCM_BA_HARDCODE_MAP)
/**
* brcm_ba_version_check() - Check version of device tree.
* Although this check may fail in a some cases which may still "work",
* generally any failure should be reported to the user to keep the device tree
* and driver in sync.
*
* version - version read from the device tree
*/
static int brcm_ba_version_check(u32 version)
{
	if (BA_KVER_MAJOR(version) < BA_KVER_MAJOR(BA_MIN_KVERSION))
		return 1;

	if (BA_KVER_MAJOR(version) == BA_KVER_MAJOR(BA_MIN_KVERSION))
		if (BA_KVER_MINOR(version) < BA_KVER_MINOR(BA_MIN_KVERSION))
			return 1;

	if (BA_KVER_MAJOR(version) > BA_KVER_MAJOR(BA_MAX_KVERSION))
		return 1;

	if (BA_KVER_MAJOR(version) == BA_KVER_MAJOR(BA_MAX_KVERSION))
		if (BA_KVER_MINOR(version) > BA_KVER_MINOR(BA_MAX_KVERSION))
			return 1;

	return 0;
}


#ifdef ARC_SUPPORT /* Not supported for all chips */
/**
* brcm_ba_of_map_arc() - Parse optional ARC information from DT
* @node: Device node
*/
static inline int brcm_ba_of_map_arc(struct device_node *node)
{
	const __be32 *prop_reg;
	int len;
	u32 tmp;
	int i, j, k;
	u16 num_arcs;
	u32 memc, bus, index;
	struct ba_arc *p_arc;

	prop_reg = (u32 *)of_get_property(node, "arc", &len);
	if (!prop_reg)
		return -1;

	tmp = len / sizeof(u32);
	if ((tmp % ARC_DT_FIELD_COUNT) != 0)
		return -1;

	num_arcs = tmp / ARC_DT_FIELD_COUNT;
	for (i = j = 0; i < num_arcs; i++) {
		memc = be32_to_cpu(prop_reg[j++]);
		if (memc >= MAX_ARC_MEMC)
			continue;
		index = be32_to_cpu(prop_reg[j++]);
		if (index >= MAX_ARC)
			continue;
		bus = be32_to_cpu(prop_reg[j++]);
		if (bus >= MAX_ARC_BUS)
			continue;

		p_arc = &info->map->arc[memc][bus][index];
		p_arc->exclusive = be32_to_cpu(prop_reg[j++]);

		/* TBD.. support 64 bit in the DT? */
		p_arc->start = (u64)be32_to_cpu(prop_reg[j++]);
		p_arc->end = (u64)be32_to_cpu(prop_reg[j++]);

		for (k = 0; k < MAX_ARC; k++)
			p_arc->read[k] = be32_to_cpu(prop_reg[j++]);

		for (k = 0; k < MAX_ARC; k++)
			p_arc->write[k] = be32_to_cpu(prop_reg[j++]);
	}

	return 0;
}
#else
static inline int brcm_ba_of_map_arc(struct device_node *node)
{
	return 0;
}
#endif

/**
* brcm_ba_of_map_v2() - v2.x DT parsing.
* return 0 if ok, 1 if some information is missing, and -1 on error

* @node: Device node
*/
static int brcm_ba_of_map_v2(struct device_node *node)
{
	int ret = 0;
	int len;
	const __be32 *prop_reg;
	struct ba_map *map = info->map;

	prop_reg = (u32 *)of_get_property(node, "stb", &len);
	if ((prop_reg) && (len == sizeof(u32) * 5)) {
		map->stb.vm.start = be32_to_cpu(prop_reg[0]);
		if (map->stb.vm.start == 0xFFFFFFFF)
			map->stb.vm.start = ~0;
		map->stb.vm.size = be32_to_cpu(prop_reg[1]);
		map->stb.vm.cpus = (u8)be32_to_cpu(prop_reg[2]);
		map->stb.vm.priority = be32_to_cpu(prop_reg[3]);
		map->stb.vm.latency = be32_to_cpu(prop_reg[4]);
		MSG_TRACEI("Obtained STB VM info from DT\n");
	} else {
		ret |= 1;
	}

	prop_reg = (u32 *)of_get_property(node, "rg", &len);
	if ((prop_reg) && (len == sizeof(u32) * 5)) {
		map->rg.vm.start = be32_to_cpu(prop_reg[0]);
		if (map->rg.vm.start == 0xFFFFFFFF)
			map->rg.vm.start = ~0;
		map->rg.vm.size = be32_to_cpu(prop_reg[1]);
		map->rg.vm.cpus = (u8)be32_to_cpu(prop_reg[2]);
		map->rg.vm.priority = be32_to_cpu(prop_reg[3]);
		map->rg.vm.latency = be32_to_cpu(prop_reg[4]);
		MSG_TRACEI("Obtained RG VM info from DT\n");
	} else {
		ret |= 1;
	}

	prop_reg = (u32 *)of_get_property(node, "tp1", &len);
	if ((prop_reg) && (len == sizeof(u32) * 2)) {
		map->docsis.tp1.start = be32_to_cpu(prop_reg[0]);
		map->docsis.tp1.size = be32_to_cpu(prop_reg[1]);
		MSG_TRACEI("Obtained TP1 info from DT\n");
	} else {
		ret |= 1;
	}

	prop_reg = (u32 *)of_get_property(node, "leap", &len);
	if ((prop_reg) && (len == sizeof(u32) * 2)) {
		/* TBD... use the start at some point... for now
		* it is hard coded */
		map->docsis.leap.size = be32_to_cpu(prop_reg[1]);
		MSG_TRACEI("Obtained LEAP info from DT\n");
	} else {
		ret |= 1;
	}

	prop_reg = (u32 *)of_get_property(node, "arm_boot_rom", &len);
	if ((prop_reg) && (len == sizeof(u32) * 2)) {
		map->arm.boot_rom.start = be32_to_cpu(prop_reg[0]);
		map->arm.boot_rom.size = be32_to_cpu(prop_reg[1]);
		MSG_TRACEI("Obtained ARM boot ROM info from DT\n");
	} else {
		ret |= 1;
	}

	prop_reg = (u32 *)of_get_property(node, "btrace_atw", &len);
	if ((prop_reg) && (len == sizeof(u32) * 8)) {
		map->arm.btrace_atw.index = (u8)be32_to_cpu(prop_reg[0]);
		map->arm.btrace_atw.source = be32_to_cpu(prop_reg[1]);
		map->arm.btrace_atw.size = be32_to_cpu(prop_reg[2]);
		map->arm.btrace_atw.dest = be32_to_cpu(prop_reg[3]);
		if (map->arm.btrace_atw.dest == 0xFFFFFFFF)
			map->arm.btrace_atw.dest = ~0;
		map->arm.btrace_atw.clients[0] = be32_to_cpu(prop_reg[4]);
		map->arm.btrace_atw.clients[1] = be32_to_cpu(prop_reg[5]);
		map->arm.btrace_atw.clients[2] = be32_to_cpu(prop_reg[6]);
		map->arm.btrace_atw.clients[3] = be32_to_cpu(prop_reg[7]);
		MSG_TRACEI("Obtained DEBUG ATW info from DT\n");
	} else {
		/* ignore */
		MSG_TRACEI("Did not get DEBUG ATW info from DT\n");
		map->arm.btrace_atw.index = (u8)0xff;
		map->arm.btrace_atw.size = 0;
	}

	prop_reg = (u32 *)of_get_property(node, "cm_xfer", &len);
	if ((prop_reg) && (len == sizeof(u32) * 3)) {
		map->docsis.xfer.start = be32_to_cpu(prop_reg[0]);
		map->docsis.xfer.size = be32_to_cpu(prop_reg[1]);
		map->docsis.xfer.count = (u8)be32_to_cpu(prop_reg[2]);
		MSG_TRACEI("Obtained CM/Xfer info from DT\n");
	} else {
		ret |= 1;
	}

	prop_reg = (u32 *)of_get_property(node, "cm_atw", &len);
	if ((prop_reg) && (len == sizeof(u32) * 8)) {
		map->docsis.cm.cm_atw.index = (u8)be32_to_cpu(prop_reg[0]);
		map->docsis.cm.cm_atw.source = be32_to_cpu(prop_reg[1]);
		map->docsis.cm.cm_atw.size = be32_to_cpu(prop_reg[2]);
		map->docsis.cm.cm_atw.dest = be32_to_cpu(prop_reg[3]);
		if (map->docsis.cm.cm_atw.dest == 0xFFFFFFFF)
			map->docsis.cm.cm_atw.dest = ~0;
		map->docsis.cm.cm_atw.clients[0] = be32_to_cpu(prop_reg[4]);
		map->docsis.cm.cm_atw.clients[1] = be32_to_cpu(prop_reg[5]);
		map->docsis.cm.cm_atw.clients[2] = be32_to_cpu(prop_reg[6]);
		map->docsis.cm.cm_atw.clients[3] = be32_to_cpu(prop_reg[7]);
		MSG_TRACEI("Obtained CM ATW info from DT\n");
	} else {
		ret |= 1;
	}

	prop_reg = (u32 *)of_get_property(node, "cm_atw_extra", &len);
	if ((prop_reg) && (len == sizeof(u32) * 8)) {
		map->docsis.cm.cm_atw_extra.index = (u8)be32_to_cpu(prop_reg[0]);
		map->docsis.cm.cm_atw_extra.source = be32_to_cpu(prop_reg[1]);
		map->docsis.cm.cm_atw_extra.size = be32_to_cpu(prop_reg[2]);
		map->docsis.cm.cm_atw_extra.dest = be32_to_cpu(prop_reg[3]);
		if (map->docsis.cm.cm_atw_extra.dest == 0xFFFFFFFF)
			map->docsis.cm.cm_atw_extra.dest = ~0;
		map->docsis.cm.cm_atw_extra.clients[0] = be32_to_cpu(prop_reg[4]);
		map->docsis.cm.cm_atw_extra.clients[1] = be32_to_cpu(prop_reg[5]);
		map->docsis.cm.cm_atw_extra.clients[2] = be32_to_cpu(prop_reg[6]);
		map->docsis.cm.cm_atw_extra.clients[3] = be32_to_cpu(prop_reg[7]);
		MSG_TRACEI("Obtained CM ATW extra info from DT\n");
		MSG_TRACEI("index %d source %08x size %08x dest %08x clients %08x\n",
			(int)map->docsis.cm.cm_atw_extra.index,
			(u32)map->docsis.cm.cm_atw_extra.source,
			map->docsis.cm.cm_atw_extra.size,
			(u32)map->docsis.cm.cm_atw_extra.dest,
			map->docsis.cm.cm_atw_extra.clients[0]);
	} else {
		/* ignore */
		map->docsis.cm.cm_atw_extra.index = (u8)0xff;
		map->docsis.cm.cm_atw_extra.size = 0;
	}

	prop_reg = (u32 *)of_get_property(node, "cm_boot_atw", &len);
	if ((prop_reg) && (len == sizeof(u32) * 8)) {
		map->docsis.cm.boot_atw.index = (u8)be32_to_cpu(prop_reg[0]);
		map->docsis.cm.boot_atw.source = be32_to_cpu(prop_reg[1]);
		map->docsis.cm.boot_atw.size = be32_to_cpu(prop_reg[2]);
		map->docsis.cm.boot_atw.dest = be32_to_cpu(prop_reg[3]);
		if (map->docsis.cm.boot_atw.dest == 0xFFFFFFFF)
			map->docsis.cm.boot_atw.dest = ~0;
		map->docsis.cm.boot_atw.clients[0] = be32_to_cpu(prop_reg[4]);
		map->docsis.cm.boot_atw.clients[1] = be32_to_cpu(prop_reg[5]);
		map->docsis.cm.boot_atw.clients[2] = be32_to_cpu(prop_reg[6]);
		map->docsis.cm.boot_atw.clients[3] = be32_to_cpu(prop_reg[7]);
		MSG_TRACEI("Obtained CM boot ATW info from DT\n");
	} else {
		ret |= 1;
	}

	prop_reg = (u32 *)of_get_property(node, "cm_boot_atw_extra", &len);
	if ((prop_reg) && (len == sizeof(u32) * 8)) {
		map->docsis.cm.boot_atw_extra.index = (u8)be32_to_cpu(prop_reg[0]);
		map->docsis.cm.boot_atw_extra.source = be32_to_cpu(prop_reg[1]);
		map->docsis.cm.boot_atw_extra.size = be32_to_cpu(prop_reg[2]);
		map->docsis.cm.boot_atw_extra.dest = be32_to_cpu(prop_reg[3]);
		if (map->docsis.cm.boot_atw_extra.dest == 0xFFFFFFFF)
			map->docsis.cm.boot_atw_extra.dest = ~0;
		map->docsis.cm.boot_atw_extra.clients[0] = be32_to_cpu(prop_reg[4]);
		map->docsis.cm.boot_atw_extra.clients[1] = be32_to_cpu(prop_reg[5]);
		map->docsis.cm.boot_atw_extra.clients[2] = be32_to_cpu(prop_reg[6]);
		map->docsis.cm.boot_atw_extra.clients[3] = be32_to_cpu(prop_reg[7]);
		MSG_TRACEI("Obtained CM boot ATW extra info from DT\n");
	} else {
		/* ignore */
		map->docsis.cm.boot_atw_extra.index = (u8)0xff;
		map->docsis.cm.boot_atw_extra.size = 0;
	}
	prop_reg = (u32 *)of_get_property(node, "cm_dsp_atw", &len);
	if ((prop_reg) && (len == sizeof(u32) * 8)) {
		map->docsis.dsp.atw.index = (u8)be32_to_cpu(prop_reg[0]);
		map->docsis.dsp.atw.source = be32_to_cpu(prop_reg[1]);
		map->docsis.dsp.atw.size = be32_to_cpu(prop_reg[2]);
		map->docsis.dsp.atw.dest = be32_to_cpu(prop_reg[3]);
		if (map->docsis.dsp.atw.dest == 0xFFFFFFFF)
			map->docsis.dsp.atw.dest = ~0;
		map->docsis.dsp.atw.clients[0] = be32_to_cpu(prop_reg[4]);
		map->docsis.dsp.atw.clients[1] = be32_to_cpu(prop_reg[5]);
		map->docsis.dsp.atw.clients[2] = be32_to_cpu(prop_reg[6]);
		map->docsis.dsp.atw.clients[3] = be32_to_cpu(prop_reg[7]);
		MSG_TRACEI("Obtained CM DSP ATW info from DT\n");
	} else {
		ret |= 1;
	}

	brcm_ba_of_map_arc(node); /* Ok to igore return value */

	return ret;
}


/**
* brcm_ba_of_map() - Get memory mapping information from OF if present
*
* Check the device tree for any mapping information. This will not apply if
* CONFIG_BCM_BA_HARDCODE_MAP is defined. Available settings are
*
* allow-user-map either defined or not. By default mapping from user space IS
* allowed, add this setting to have a more secure build (user space will NOT be
* able to specify MBOX block size, count, or any mapping configuration). Note
* that if this is set, the OF MUST contain sufficient mapping information
* otherwise the system will NOT work.
*
* mbox-cm-size - u32 size of CM to STB mailbox xfer
*
* mbox-cm-count - u32 number of CM to STB mailbox buffers (i.e. total size is
* size*count)
*
* map - BINARY DATA of struct ba_map
*/
static int brcm_ba_of_map(struct device_node *node)
{
	int ret = 0;
	int len;
	const __be32 *prop_reg;
	struct device_node *dnode;
	uint8_t map_valid_mask = 0;
	u32 version = 0;
	u64 tmp64;
	struct resource mem;

	FUNC_TRACE(1);

	if (!node)
		MSG_TRACEE("Invalid device node\n");

	prop_reg = (u32 *)of_get_property(node, "version", &len);
	if (prop_reg) {
		version = be32_to_cpu(*prop_reg);
		MSG_TRACEI("BA DT version 0x%x\n", version);
	}
	if (brcm_ba_version_check(version)) {
		MSG_TRACEE("==============================\n");
		MSG_TRACEE("Device tree version # missing or mismatch!\n");
		MSG_TRACEE("BOOT ASSIST MAY FAIL\n");
		MSG_TRACEE("PLEASE UPDATE DEVICE-TREE\n");
		MSG_TRACEE("EXPECTED VERSION IS 0x%x\n\n", BA_KVERSION);
	}

	if (of_get_property(node, "allow-user-map", &len)) {
		MSG_TRACEI("Allow the user mapping!\n");
			info->allow_user_map = true;
	} else {
		MSG_TRACEI("User space CAN NOT specify mapping\n");
			info->allow_user_map = false;
	}
	write_lock(&map_lock);
	/* At this point, will most likely get some mapping info from DT */
	info->map = (struct ba_map *)devm_kzalloc(&info->pdev->dev,
		     sizeof(struct ba_map), GFP_KERNEL);
	if (!info->map) {
		MSG_TRACEE("Failed to allocate memory!\n");
		write_unlock(&map_lock);
		return -ENOMEM;
	}
	/* "invalid" for ARC's is all FF's */
	memset(info->map->arc, 0xFF, sizeof(info->map->arc));
	write_unlock(&map_lock);

#if defined(CONFIG_BCM7145)
	if (BA_KVER_MAJOR(version) < 2)
		map_valid_mask = brcm_ba_of_map_v1(node);
	else
#endif
		map_valid_mask = brcm_ba_of_map_v2(node);

	if (map_valid_mask == (u8)-1)
		return -EINVAL;

	/* Get FPM size from FPM DT */
	dnode = of_find_node_by_name(NULL, "fpm");
	if (dnode == NULL) {
		MSG_TRACEE("Can't get FPM size use default\n");
	} else {
		ret = of_address_to_resource(dnode, 0, &mem);
		if (ret != 0) {
			MSG_TRACEE("Can't get FPM size use default\n");
		} else {
			ba_fpm_size = resource_size(&mem);
			ba_fpm_start = mem.start;
		}

		of_node_put(dnode);
	}
	MSG_TRACEI("FPM size is %lu\n", ba_fpm_size);

	/* Get MEMC0 size from DT */
	dnode = of_find_node_by_name(NULL, "memory");
	if (dnode == NULL) {
		MSG_TRACEE("Can't get MEMC0 size, defaulting to %lu\n",
			   ba_memc0_size);
	} else {
		prop_reg = of_get_property(dnode, "reg", NULL);
		if (prop_reg) {
			of_get_address(dnode, 0, &tmp64, NULL);
			ba_memc0_size = (uint32_t)tmp64;
			MSG_TRACEN("Obtained MEMC0 size from DT\n");
		} else {
			MSG_TRACEE("missing reg prop\n");
			ret = -EFAULT;
		}
		of_node_put(dnode);
	}
	MSG_TRACEI("MEMC0 size is %lu\n", ba_memc0_size);

	/* Map only valid if size, dest, and arc/atw all specified */
	if (map_valid_mask == 0) {
		info->flags |= MAP_VALID;
	} else {
		if (!info->allow_user_map) {
			MSG_TRACEE("SYSTEM INFO MISSING!\n");
			MSG_TRACEE("SYSTEM MAY NOT FUNCTION!\n");
		}
	}

	FUNC_TRACE(0);
	return ret;
}
#endif

static int cm_prep(void)
{
	FUNC_TRACE(1);

	FUNC_TRACE(0);
	return 0;
}

static const struct file_operations ba_fops = {
	.owner = THIS_MODULE,
	.open = ba_open,
	.release = ba_release,
	.read = ba_read,
	.write = ba_write,
	.unlocked_ioctl = ba_ioctl,
};

#ifdef ARC_SUPPORT /* Not supported for all chips */
/**
* brcm_ba_irq() - MEMC0/1 interrupt handling (for ARC violations)
*/
static irqreturn_t brcm_ba_irq(int irq, void *dev_id)
{
	u32 status, start, end, cmd;
	u32 id, nmb, type;
	int i;
	void __iomem *base = NULL;

	if ((dev_id != (void *)info) || (!info))
		return IRQ_NONE;

	if (irq == info->memc_irq[0])
		base = info->memc0;
#if MAX_ARC_MEMC > 1
	else if (irq == info->memc_irq[1])
		base = info->memc1;
#endif
	if (!base) {
		MSG_TRACEE("Unknown IRQ!\n");
		return IRQ_NONE;
	}

	R32(status, MC_INTR2_STATUS(base));

	if (!ARC_INTEREST(status))
		return IRQ_NONE;

	for (i = 0; i < ARC_MAX; i++) {
		R32(start, SCB_VIOLATION_START(base, i));
		if (!start)
			continue;
		R32(end, SCB_VIOLATION_END(base, i));
		start = start << 3;
		end = end << 3;
		R32(cmd, SCB_VIOLATION_CMD(base, i));
		id = (cmd & SCB_VIOLATION_CMD_CLIENTID_MASK) >>
		     SCB_VIOLATION_CMD_CLIENTID_SHIFT;
		nmb = (cmd & SCB_VIOLATION_CMD_NMB_MASK) >>
		      SCB_VIOLATION_CMD_NMB_SHIFT;
		type = (cmd & SCB_VIOLATION_CMD_TYPE_MASK) >>
		       SCB_VIOLATION_CMD_TYPE_SHIFT;
		MSG_TRACEE("\nSCB ARC %d VIOLATION: 0x%08x-0x%08x\n"
			   "\t Client=0x%x, NMB=0x%x, Request Type=0x%x\n",
			   i, start, end, id, nmb, type);

		/* Clear violation information */
		W32(SCB_VIOLATION_CLEAR(base, i), 1);
		W32(SCB_VIOLATION_CLEAR(base, i), 0);
	}

/* Depends on memory controller */
#if defined(UBUS_VIOLATION_START)
	for (i = 0; i < ARC_MAX; i++) {
		R32(start, UBUS_VIOLATION_START(base, i));
		if (!start)
			continue;
		R32(end, UBUS_VIOLATION_END(base, i));
		start = start << 3;
		end = end << 3;
		R32(cmd, UBUS_VIOLATION_CMD(base, i));
		id = (cmd & UBUS_VIOLATION_CMD_SRC_MASK) >>
		     UBUS_VIOLATION_CMD_SRC_SHIFT;
		nmb = (cmd & UBUS_VIOLATION_CMD_PKT_MASK) >>
		      UBUS_VIOLATION_CMD_PKT_SHIFT;
		type = (cmd & UBUS_VIOLATION_CMD_TYPE_MASK) >>
		       UBUS_VIOLATION_CMD_TYPE_SHIFT;
		MSG_TRACEE("\nUBUS ARC %d VIOLATION: 0x%08x-0x%08x\n"
			   "\t Source ID=0x%x, PKT=0x%x, Trans Type=0x%x\n",
			   i, start, end, id, nmb, type);

		/* Clear violation information */
		W32(UBUS_VIOLATION_CLEAR(base, i), 1);
		W32(UBUS_VIOLATION_CLEAR(base, i), 0);
	}
#endif

	/* Clear interrupts */
	MC_INT_CLEAR(base);

	return IRQ_HANDLED;
}
#else
static irqreturn_t brcm_ba_irq(int irq, void *dev_id)
{
	return IRQ_NONE;
}
#endif

/**
* brcm_ba_cm_mbox_event() - CM to RG mbox RG power event handler
*
* Called by the notifier chain when the RG to CM mbox is written.
* Currently handles one notification from the CM when requesting
* that the RG power island be immediately turned off.
*/
static int brcm_ba_cm_mbox_event(
		struct notifier_block *this,
		unsigned long event, void *ptr)
{
	int status = NOTIFY_OK;
	struct brcm_mbox_info *states = ptr;

	switch (event) {
	case MBOX_CHANGE_EVENT:
	{
		if ((states->mbox[MBOX_CM]) & CM_RG_POWERDOWN_REQ) {
			brcm_mbox_update_rg_powerdown();
			WF32(AON_CTRL_PM_CTRL, PM_SET_FORCE_BATTERY_MODE, 1);
			return status;
		}
		break;
	}
	default:
		status = NOTIFY_DONE;
		goto done;
		break;
	}

done:
	return status;
}

static struct notifier_block cm_mbox_notifier = {
	.notifier_call  = brcm_ba_cm_mbox_event,
};

static void ba_poweroff(void) {
	struct pwr_supply *supply;

	if (!list_empty(&board_supplies))
		list_for_each_entry(supply, &board_supplies, list)
			regulator_force_disable(supply->regulator);

	mdelay(1000);

	/*
	 * If we got here then the board did not have SW controllable power. So
	 * we have to set a flag that won't be cleared on reset and reboot. On
	 * reboot the BOLT DT selection algorithm will select a minimal DT to
	 * keep power consumption to a minimum. When the init below is run it
	 * will prevent the CM from booting and switch off the RG power plane
	 * to put the board into the lowest possible power consumption state.
	 */
	WF32(AON_CTRL_UNCLEARED_SCRATCH, PARK_ON_REBOOT, 1);
	machine_restart(NULL);
}

/**
* brcm_ba_probe()
*
* Get resources, add debugfs files (if so configured), get current power state,
* register IRQ (if so configured), create device node, check OF for settings.
*/
static int brcm_ba_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct device *dev = &pdev->dev;
	void __iomem *mem;
	int irq0 = -1, irq1 = -1;
	const char *str = NULL;
	const char *pwr_str = NULL;
	bool park_soc;
	int i, supplies;
	const char *name;
	struct pwr_supply *supply;
	struct device_node *dnode = NULL;

	FUNC_TRACE(1);

#if !defined(CONFIG_OF)
	MSG_TRACEE("Boot assist requires device tree support\n");
	return -1;
#endif

#ifdef CONFIG_BCM_BA_PERF
	memset(&metrics, 0, sizeof(metrics));
#endif

	METRIC_START;

	info = (struct ba_dev_info *)devm_kzalloc(dev,
		sizeof(struct ba_dev_info), GFP_KERNEL);
	if (!info) {
		MSG_TRACEE("Failed to allocate memory!\n");
		ret = -ENOMEM;
		goto ERROR_EXIT;
	}

	info->allow_user_map = true;
	info->memc_irq[0] = -1;
	info->memc_irq[1] = -1;

	rwlock_init(&map_lock);

	platform_set_drvdata(pdev, (void *)info);
	info->pdev = pdev;

#ifdef ARC_SUPPORT /* Not supported for all chips */
	/* Get IRQ's */
	irq0 = platform_get_irq(pdev, 0);
	if (irq0 < 0)
		MSG_TRACEE("Missing IRQ resource. MC0 ARC not supported\n");

	irq1 = platform_get_irq(pdev, 1);
	if (irq1 < 0)
		MSG_TRACEE("Missing IRQ resource. MC1 ARC not supported\n");
#endif

	ret = ba_map_res(pdev, NULL, "mem", 0);
	ret |= ba_map_res(pdev, &info->memc0, "mc0", 0);
#if MAX_ARC_MEMC > 1
	ba_map_res(pdev, &info->memc1, "mc1", 0); /* optional */
#endif
	ret |= ba_map_res(pdev, &info->rg_top, "rgtop", 1);
	ret |= ba_map_res(pdev, &info->cm_top, "cmtop", 1);
	ret |= ba_map_res(pdev, &info->aon, "aon", 0);

	if (unlikely(ret != 0)) {
		MSG_TRACEE("Error getting IO resource(s)\n");
		ret = -ENXIO;
		goto ERROR_EXIT;
	}

#ifdef CONFIG_BCM_BA_HARDCODE_MAP
	if (ba_hardcode_map(info) != 0) {
		MSG_TRACEE("Error in hard coded mapping information!\n");
		goto ERROR_EXIT;
	}
	ba_fpm_size = BA_FPM_SIZE;
#else
	/* Memory config possibly specified in device tree, go see */
	if (brcm_ba_of_map(dev->of_node) != 0) {
		MSG_TRACEE("Error in OF configuration!\n");
		goto ERROR_EXIT;
	}
#endif
	dnode = of_find_node_by_name(NULL, "power");
	if (dnode == NULL) {
		MSG_TRACEE("Can't find power node\n");
		goto ERROR_EXIT;
	}
	/* Just need 2 values from OTP */
	if (unlikely(ba_map_res(pdev, &mem, "otp", 0) != 0)) {
		MSG_TRACEE("Error getting IO resource\n");
		ret = -ENXIO;
		goto ERROR_EXIT;
	}
    memcpy(info->otp_regs, mem, sizeof(info->otp_regs));
    MSG_TRACEN("OTP[0]: 0x%08x, OTP[1]: 0x%08x\n", info->otp_regs[0], info->otp_regs[1]);
	iounmap(mem);

	/* Map the LED registers */
	if (unlikely(ba_map_res(pdev, &mem, "leds", 0) != 0)) {
		MSG_TRACEE("Error getting IO resource\n");
		ret = -ENXIO;
		goto ERROR_EXIT;
	}
	info->leds = mem;

	of_property_read_string(dev->of_node, "rg-batt-mode", &str);
	if (str) {
		if (!strcmp(str, "off")) {
			of_property_read_string(dnode, "boot-state", &pwr_str);
			if (pwr_str) {
				if (!strcmp(pwr_str, "battery")) {
					rg_batt_mode = RG_CPU_OFF_BATTERY;
				} else {
					if (of_property_read_bool(dnode, "delay-battery-mode"))
						rg_batt_mode = RG_CPU_OFF_UTILITY_DELAYED;
					else
						rg_batt_mode = RG_CPU_OFF_UTILITY;
				}
			}
		} else if (!strcmp(str, "cpu-on")) {
			of_property_read_string(dnode, "boot-state", &pwr_str);
			if (pwr_str) {
				if (!strcmp(pwr_str, "battery"))
					rg_batt_mode = RG_CPU_ON_BATTERY;
				else
					rg_batt_mode = RG_CPU_ON_UTILITY;
			}
		} else {
			MSG_TRACEE("Invalid RG battery mode specified!\n");
			ret = -EINVAL;
			goto ERROR_EXIT;
		}
	}
	park_soc = RF32(AON_CTRL_UNCLEARED_SCRATCH, PARK_ON_REBOOT);
	WF32(AON_CTRL_UNCLEARED_SCRATCH, PARK_ON_REBOOT, 0);
	if (park_soc) {
		pr_crit("\n\n");
		pr_crit("*************************************************\n");
		pr_crit("*      Parking SOC to minimize power.           *\n");
		pr_crit("*         Reset required to reboot.             *\n");
		pr_crit("*************************************************\n");
		WF32(AON_CTRL_PM_CTRL, PM_SET_FORCE_BATTERY_MODE, 1);
	}

	INIT_LIST_HEAD(&board_supplies);
	supplies = of_property_count_strings(dev->of_node,
			"board-pwr-supply-names");
	if (supplies <= 0)
		supplies = 0;

	for (i = 0; i < supplies; i++) {
		if (of_property_read_string_index(dev->of_node,
			"board-pwr-supply-names", i, &name))
			continue;
		supply = devm_kzalloc(&pdev->dev, sizeof(*supply), GFP_KERNEL);
		if (!supply) {
			MSG_TRACEE("Unable to alloc supply memory!\n");
			ret = -ENOMEM;
			goto ERROR_EXIT;
		}
		strlcpy(supply->name, name, sizeof(supply->name));
		supply->name[sizeof(supply->name) - 1] = '\0';
		supply->regulator = devm_regulator_get(&pdev->dev, name);
		if (IS_ERR(supply->regulator)) {
			MSG_TRACEE("Unable to get %s supply, err=%d\n",
				name, (int)PTR_ERR(supply->regulator));
			ret = (int)PTR_ERR(supply->regulator);
			goto ERROR_EXIT;
		}
		if (regulator_enable(supply->regulator))
			MSG_TRACEE("Unable to enable %s supply.\n",
				name);
		list_add_tail(&supply->list, &board_supplies);
	}
	pm_power_off = ba_poweroff;

	/* Allocate device nodes */
	MSG_TRACEN("Allocating device node(s)\n");
	ret = alloc_chrdev_region(&info->ba_dev, 0, 1, "boot assist");
	if (ret != 0) {
		MSG_TRACEE("Error allocating character device!\n");
		goto ERROR_EXIT;
	}
	MSG_TRACEI("Broadcom Boot Assist MAJOR is %d\n", MAJOR(info->ba_dev));

	info->ba_class = class_create(THIS_MODULE, "boot assist");
	if (!info->ba_class) {
		unregister_chrdev_region(info->ba_dev, 1);
		MSG_TRACEE("Failed to create class\n");
		goto ERROR_EXIT;
	}

	cdev_init(&info->ba_cdev, &ba_fops);
	ret = cdev_add(&info->ba_cdev, info->ba_dev, 1);
	if (ret != 0) {
		MSG_TRACEE("Error adding cdev\n");
		goto ERROR_EXIT;
	}

	MSG_TRACEN("Creating device nodes\n");
	device_create(info->ba_class, NULL, info->ba_dev, NULL, "ba");

	/* Add debug debugfs files */
	if (ba_config_dbgfs(true) != 0) {
		device_destroy(info->ba_class, info->ba_dev);
		cdev_del(&info->ba_cdev);
		goto ERROR_EXIT;
	}

	/* Hook into PM notifications */
	/* register_pm_notifier */ /* TBD */
	register_reboot_notifier(&brcm_ba_notifier);

	/* Register notifier callback for CM power requests */
	brcm_mbox_register_notifier(&cm_mbox_notifier);

	/* Any prep for RG and CM... TBD.. move to user space?
	* requires register access though.. */
	cm_prep();

	if (irq0 >= 0) {
		MSG_TRACEN("Enable MBOX IRQ %d\n", irq0);
		ret = devm_request_irq(dev, irq0, &brcm_ba_irq, IRQF_SHARED,
				       "brcm-ba", (void *)info);
		if (ret < 0) {
			MSG_TRACEE("error requesting IRQ #%d\n", irq0);
			ret = -ENXIO;
			goto ERROR_EXIT;
		}
		info->memc_irq[0] = irq0;
	}

	if (irq1 >= 0) {
		MSG_TRACEN("Enable MBOX IRQ %d\n", irq1);
		ret = devm_request_irq(dev, irq1, &brcm_ba_irq, IRQF_SHARED,
				       "brcm-ba", (void *)info);
		if (ret < 0) {
			MSG_TRACEE("error requesting IRQ #%d\n", irq1);
			ret = -ENXIO;
			goto ERROR_EXIT;
		}
		info->memc_irq[1] = irq1;
	}

	FUNC_TRACE(0);
	return 0;

ERROR_EXIT:

	if (info) {
		if (info->memc0)
			iounmap(info->memc0);
#if MAX_ARC_MEMC > 1
		if (info->memc1)
			iounmap(info->memc1);
#endif
		if (info->rg_top)
			iounmap(info->rg_top);
		if (info->cm_top)
			iounmap(info->cm_top);
		if (info->aon)
			iounmap(info->aon);
		if (info->leds)
			iounmap(info->leds);
		if (info->memc_irq[0] >= 0)
			devm_free_irq(dev, info->memc_irq[0], (void *)info);
		if (info->memc_irq[1] >= 0)
			devm_free_irq(dev, info->memc_irq[1], (void *)info);
		if (info->ba_class) {
			unregister_chrdev_region(info->ba_dev, 1);
			class_destroy(info->ba_class);
		}
		if (info->map) {
			write_lock(&map_lock);
			devm_kfree(&info->pdev->dev, info->map);
			info->map = NULL;
			write_unlock(&map_lock);
		}
		devm_kfree(dev, (void *)info);
		info = NULL;
	}

	FUNC_TRACE(0);
	return ret;
}

/**
* ba_notify_reboot() - Last ditch BA hook for power down
*
* At the moment, this is used to complete the C2 process when on battery.
* It is expected, that should this API be called, and system is in battery mode,
* that C2 init ioctl has already been called, and this will function as the
* completion of the C2 process, i.e. it will pull the plug.
*/
static int ba_notify_reboot(struct notifier_block *this,
			      unsigned long code, void *x)
{
	u32 tmp;

	FUNC_TRACE(1);

	/* C2 "complete" is only for battery mode */
/* 	if (ba_c2_batt == BS_UNKNOWN)	   */
/* 		ba_c2_batt = ba_boot_batt; */

	if (ba_c2_batt == BS_AC || ba_c2_batt == BS_UNKNOWN) {
		FUNC_TRACE(0);
		return NOTIFY_DONE;
	}

	/* At this point, STB should have satisfied any wake
	* on battery interrupt should it have occurred.*/

	if (RF32(AON_PM_BBM_L2_CPU_STATUS, DCS_HEADEND_INTR)) {
		/* CM has woken STB for some proxy service... */
		/* TBD... can this happen while on AC? If so, this code
		* is NOT right! */
		tmp = 0;
		SWF32(tmp, DCS_HEADEND_INTR, 1);
		W32(AON_PM_BBM_L2_CPU_CLEAR, tmp);

		/* TBD.. indicate to CM it's ok to assert future
		ubus_software_intr wakeup interrupts */
	}

	/* TBD... AVS COMMUNICATION to DISENGAGE */
	/* TBD... AVS COMMUNICATION wait for ACK */

	if (code != SYS_RESTART) {
		/* Force BBM if not trying to reboot */
		WF32(AON_CTRL_PM_CTRL, PM_SET_FORCE_BATTERY_MODE, 1);
	}

	FUNC_TRACE(0);
	return NOTIFY_DONE;
}

/**
* brcm_ba_remove()
*/
static int brcm_ba_remove(struct platform_device *pdev)
{
	FUNC_TRACE(1);

	ba_config_dbgfs(false);

	if (info) {
		if (info->memc_irq[0] >= 0) {
			if ((info->arc_enabled[0] > 0) && info->memc0)
				MC_INT_DISABLE(info->memc0);
			devm_free_irq(&info->pdev->dev, info->memc_irq[0],
				      (void *)info);
		}
#if MAX_ARC_MEMC > 1
		if (info->memc_irq[1] >= 0) {
			if ((info->arc_enabled[1] > 0) && info->memc1)
				MC_INT_DISABLE(info->memc1);
			devm_free_irq(&info->pdev->dev, info->memc_irq[1],
				      (void *)info);
		}
#endif
		if (info->leds)
			iounmap(info->leds);
		if (info->memc0)
			iounmap(info->memc0);
#if MAX_ARC_MEMC > 1
		if (info->memc1)
			iounmap(info->memc1);
#endif
		if (info->rg_top)
			iounmap(info->rg_top);
		if (info->cm_top)
			iounmap(info->cm_top);
		if (info->aon)
			iounmap(info->aon);
		if (info->ba_class)
			device_destroy(info->ba_class, info->ba_dev);

		cdev_del(&info->ba_cdev);
		class_destroy(info->ba_class);
		unregister_chrdev_region(info->ba_dev, 1);

		if (info->map)
			devm_kfree(&pdev->dev, (void *)info->map);

		devm_kfree(&pdev->dev, (void *)info);
		info = NULL;
	}

	unregister_reboot_notifier(&brcm_ba_notifier);
	brcm_mbox_unregister_notifier(&cm_mbox_notifier);

	FUNC_TRACE(0);
	return 0;
}

static struct platform_driver brcm_ba_driver = {
	.probe = brcm_ba_probe,
	.remove = brcm_ba_remove,
	.driver = {
		.name = "brcm-ba",
		.owner = THIS_MODULE,
		.of_match_table = of_platform_brcmba_table,
	},
};

static int __init brcm_ba_init(void)
{
	pr_info("%s driver v%s\n", MODULE_NAME, MODULE_VER);
	return platform_driver_register(&brcm_ba_driver);
}
module_init(brcm_ba_init);

static void __exit brcm_ba_exit(void)
{
	platform_driver_unregister(&brcm_ba_driver);
}
module_exit(brcm_ba_exit);

/**
* Returns the configured RG behavior when the system is running
* on battery power. Available to other kernel modules.
*/
enum rg_battery_mode brcm_ba_rg_batt_mode(void)
{
	return rg_batt_mode;
}
EXPORT_SYMBOL(brcm_ba_rg_batt_mode);

MODULE_LICENSE("GPL");
