 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/of_net.h>
#include <linux/of_mdio.h>
#include "dqnet_dt.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "dqnet_qmsg.h"

static int dqnet_read_prop_u32(struct device_node *of_node,
			       const char *propname, u32 *dst)
{
	int status = 0;

	if (!of_property_read_u32(of_node, propname, dst)) {
		pr_debug("%s = 0x%08x\n", propname, *dst);
	} else {
		pr_debug("Missing %s property!\n", propname);
		status = -EINVAL;
	}

	return status;
}

static int dqnet_read_prop_string(struct device_node *of_node,
				  const char *propname, char **dst)
{
	int status = 0;

	if (!of_property_read_string(of_node, propname, (const char **)dst)) {
		pr_debug("%s = %s\n", propname, *dst);
	} else {
		pr_debug("Missing %s property!\n", propname);
		status = -EINVAL;
	}

	return status;
}

static int dqnet_read_prop_string_index(struct device_node *of_node,
					const char *propname, int index,
					char **dst)
{
	int status = 0;

	status = of_property_read_string_index(of_node, propname, index,
					   (const char **)dst);
	if (!status) {
		pr_debug("%s = %s\n", propname, *dst);
	} else {
		pr_debug("Error reading %s property!\n", propname);
	}

	return status;
}

static int dqnet_read_prop_u32_array(struct device_node *of_node,
				     const char *propname, int cols, int *rows,
				     u32 *dst)
{
	int status = 0;
	struct property *prop;
	int num_entries;
	int i, j;
	const __be32 *ptr;

	prop = of_find_property(of_node, propname, NULL);
	if (!prop) {
		pr_debug("Unable to find %s property!\n", propname);
		status = -EINVAL;
		goto done;
	}

	if ((prop->length % cols*sizeof(u32)) != 0) {
		pr_err("%s property is malformed!\n", propname);
		status = -EINVAL;
		goto done;
	}

	num_entries = prop->length / sizeof(u32) / cols;
	if (num_entries > *rows) {
		pr_err("Array provided for %s not large enough!\n", propname);
		status = -EINVAL;
		goto done;
	}
	*rows = num_entries;

	ptr = prop->value;
	for (i = 0; i < *rows; i++)
		for (j = 0; j < cols; j++)
			dst[i*cols + j] = be32_to_cpup(ptr++);

done:
	return status;
}

static int dqnet_chan_qs_type_1(struct device_node *node,
				struct dqnet_channel *chan)
{
	int status = 0;
	char *str;
	int q_count;
	u32 *qs;
	int i;

	/*
	 * Format is:
	 * "dqm" specifies the DQM device
	 * "tx-q" specifies a list of Q's ordered from highest priority to lowest
	 * "rx-q" specifies a list of Q's ordered from highest priority to lowest
	 * Max # of Q's that can be specified for tx-q or rx-q is fixed at Q_PRIO_MAX
	 */
	status = dqnet_read_prop_string(node, "dqm", &str);
	if (status)
		goto done;

	q_count = Q_PRIO_MAX;
	qs = kmalloc(q_count*sizeof(u32), GFP_KERNEL);
	if (!qs) {
		pr_err("Unable to allocate q # array for channel %s.\n",
		    chan->name);
		status = -ENOMEM;
		goto done;
	}

	memset(qs, 0, q_count*sizeof(u32));
	chan->tx_disable = of_property_read_bool(node, "tx-disable");
	if (chan->tx_disable)
		goto do_rx_q_info;

	status = dqnet_read_prop_u32_array(node, "tx-q", 1,
					       &q_count, qs);
	if (status || q_count > Q_PRIO_MAX) {
		if (q_count > Q_PRIO_MAX) {
			pr_err("Too many TX Q's specified for channel %s.\n",
			    chan->name);
		}
		status = -EINVAL;
		goto err_free_qs;
	}

	for (i = 0; i < q_count; i++) {
		chan->tx_q_info[i].q_num = qs[i];
		strncpy(chan->tx_q_info[i].dev, str,
			sizeof(chan->tx_q_info[i].dev));
		chan->tx_q_info[i].dev[sizeof(chan->tx_q_info[i].dev)-1] =
			'\0';
		chan->tx_q_info[i].priority = i;
		chan->tx_q_info[i].q_type = DQNET_QUEUE_US;
	}
	chan->tx_q_count = q_count;

do_rx_q_info:
	status = dqnet_read_prop_u32_array(node, "rx-q", 1,
					       &q_count, qs);
	if (status || q_count > Q_PRIO_MAX) {
		if (q_count > Q_PRIO_MAX) {
			pr_err("Too many RX Q's specified for channel %s.\n",
			    chan->name);
		}
		status = -EINVAL;
		goto err_free_qs;
	}
	for (i = 0; i < q_count; i++) {
		chan->rx_q_info[i].q_num = qs[i];
		strncpy(chan->rx_q_info[i].dev, str,
			sizeof(chan->rx_q_info[i].dev));
		chan->rx_q_info[i].dev[sizeof(chan->rx_q_info[i].dev)-1] =
			'\0';
		chan->rx_q_info[i].priority = i;
		chan->rx_q_info[i].q_type = DQNET_QUEUE_US;
	}
	chan->rx_q_count = q_count;

err_free_qs:
	kfree(qs);
done:
	return status;
}

static int __dqnet_chan_qs_type_2(struct device_node *node,
				  enum dqnet_tx_rx tx_rx,
				  struct dqnet_channel *chan)
{
	int status = 0;
	char *propname;
	struct dqnet_q_info *q_info;
	int *count;
	char *str;
	int cells, col, row;

	/*
	 * Format is:
	 * #tx-q-cells/#rx-q-cells specifies # of parameters specified for each Q
	 * tx-q/rx-q is a list of Q's with parameters specified for each Q as follows:
	 * DQM-device, Q#, priority, downstream/upstream (tx-q only)
	 * The first 3 parameters must be present for both tx-q and rx-q, but
	 * downstream/upstream can only be present for tx-q.
	 */
	if (tx_rx == DQNET_TX) {
		propname = "tx-q";
		q_info = chan->tx_q_info;
		count = &chan->tx_q_count;
		status = dqnet_read_prop_u32(node, "#tx-q-cells", &cells);
		if (status)
			goto done;
	} else {
		propname = "rx-q";
		q_info = chan->rx_q_info;
		count = &chan->rx_q_count;
		status = dqnet_read_prop_u32(node, "#rx-q-cells", &cells);
		if (status)
			goto done;
	}

	for (row = 0, col = 0;
	     !status && row < Q_PRIO_MAX * Q_US_DS_MAX;
	     row++, col = 0) {
		status = dqnet_read_prop_string_index(node, propname,
						      row * cells + col++,
						      &str);
		if (status) {
			if (status == -ENODATA) {
				status = 0;
				break;
			} else {
				pr_err("Unable to find DQM device name ");
				pr_err("for channel %s\n", chan->name);
				goto done;
			}
		}
		strncpy(q_info[row].dev, str, sizeof(q_info[row].dev));
		q_info[row].dev[sizeof(q_info[row].dev)-1] = '\0';
		status = dqnet_read_prop_string_index(node, propname,
						      row * cells + col++,
						      &str);
		if (status) {
			pr_err("Unable to find Q # ");
			pr_err("for channel %s\n", chan->name);
			goto done;
		}
		status = kstrtoint(str, 10, &q_info[row].q_num);
		if (status) {
			pr_err("Unable to translate Q # ");
			pr_err("for channel %s\n", chan->name);
			goto done;
		}
		status = dqnet_read_prop_string_index(node, propname,
						      row * cells + col++,
						      &str);
		if (status) {
			pr_err("Unable to find Q priority ");
			pr_err("for channel %s\n", chan->name);
			goto done;
		}
		status = kstrtoint(str, 10, &q_info[row].priority);
		if (status) {
			pr_err("Unable to translate Q priority ");
			pr_err("for channel %s\n", chan->name);
			goto done;
		}
		q_info[row].q_type = DQNET_QUEUE_US;
		if (cells > 3) {
			status = dqnet_read_prop_string_index(node, propname,
					row * cells + col++,
					&str);
			if (status) {
				pr_err("Unable to find Q US/DS ");
				pr_err("for channel %s\n", chan->name);
				goto done;
			}
			if (!strncmp(str, Q_US_STRING, sizeof(Q_US_STRING)))
				q_info[row].q_type = DQNET_QUEUE_US;
			else if (!strncmp(str, Q_DS_STRING, sizeof(Q_DS_STRING)))
				q_info[row].q_type = DQNET_QUEUE_DS;
			else if (!strncmp(str, Q_CTL_STRING, sizeof(Q_CTL_STRING)))
				q_info[row].q_type = DQNET_QUEUE_CTL;
			else if (!strncmp(str, Q_EXP_STRING, sizeof(Q_EXP_STRING)))
				q_info[row].q_type = DQNET_QUEUE_EXP;
			else {
				pr_err("Invalid value specified for Q US/DS ");
				pr_err("for channel %s\n", chan->name);
				status = -EINVAL;
				goto done;
			}
		}
	}
	if (row >= Q_PRIO_MAX * Q_US_DS_MAX) {
		pr_err("Too many Q's for channel %s\n", chan->name);
		status = -EINVAL;
		goto done;
	}
	*count = row;

done:
	return status;
}


static int dqnet_chan_qs_type_2(struct device_node *node,
				struct dqnet_channel *chan)
{
	int status = 0;

	status = __dqnet_chan_qs_type_2(node, DQNET_TX, chan);
	if (status)
		goto done;
	status = __dqnet_chan_qs_type_2(node, DQNET_RX, chan);

done:
	return status;
}

int dqnet_parse_dt_node(struct platform_device *pdev)
{
	int status = 0;
	struct dqnet_netdev *ndev = pdev->dev.platform_data;
	struct dqnet_channel *chan = ndev->chan;
	struct device_node *of_node = pdev->dev.of_node;
	struct device_node *phan_node = NULL;
	char *str;
	const char *mac_addr;

	status = dqnet_read_prop_string(of_node, "dev-name", &ndev->name);
	if (status)
		goto done;

	ndev->dt_name = ndev->name;

	mac_addr = of_get_mac_address(of_node);
	if (!IS_ERR(mac_addr) && mac_addr)
		memcpy(ndev->mac_addr, mac_addr, sizeof(ndev->mac_addr));

	status = dqnet_read_prop_u32(of_node, "mac-id", &ndev->if_id);
	if (status)
	status = dqnet_read_prop_u32(of_node, "if-id",
				     &ndev->if_id);
	if (status)
		ndev->if_id = 0;
	status = dqnet_read_prop_u32(of_node, "mac-sub-id",
				     &ndev->if_sub_id);
	if (status)
	status = dqnet_read_prop_u32(of_node, "if-sub-id",
				     &ndev->if_sub_id);
	if (status)
		ndev->if_sub_id = 0;

	status = of_property_count_u32_elems(of_node, "tx-qm-q");
	if (status > 0) {
		ndev->tx_qm_q_count = status;
		ndev->tx_qm_q = kmalloc(status*sizeof(u32), GFP_KERNEL);
		of_property_read_u32_array(of_node, "tx-qm-q", ndev->tx_qm_q,
					   ndev->tx_qm_q_count);
	} else {
		status = dqnet_read_prop_u32(of_node, "tx-if-id",
				&ndev->tx_if_id);
		if (status)
			ndev->tx_if_id = -1;
	}

	status = dqnet_read_prop_string(of_node, "demux", &str);
	if (status)
		goto done;
	if (!strncmp(str, DEMUX_FAP_IFID_SUBID_STRING,
		     sizeof(DEMUX_FAP_IFID_SUBID_STRING)))
		ndev->demux_type = DQNET_DEMUX_TYPE_IFID_SUBID;
	else if (!strncmp(str, DEMUX_FAP_IFID_SF2_STRING,
			  sizeof(DEMUX_FAP_IFID_SF2_STRING)) ||
		 !strncmp(str, DEMUX_FAP_IFID_ARL_STRING,
			  sizeof(DEMUX_FAP_IFID_ARL_STRING)))
		ndev->demux_type = DQNET_DEMUX_TYPE_IFID_ARL;
	else if (!strncmp(str, DEMUX_NONE_STRING, sizeof(DEMUX_NONE_STRING)))
		ndev->demux_type = DQNET_DEMUX_TYPE_NONE;
	else {
		pr_err("Invalid value specified for demux for device %s.\n",
		    ndev->name);
		status = -EINVAL;
		goto done;
	}

	ndev->brcm_tag = of_property_read_bool(of_node, "brcm-tag");

	phan_node = of_parse_phandle(of_node, "channel", 0);
	if (!phan_node) {
		pr_err("Unable to retrieve channel phandle for device %s.\n",
		    ndev->name);
		status = -EINVAL;
		goto done;
	}

	status = dqnet_read_prop_string(phan_node, "dev-name", &str);
	if (status)
		goto err_put_node;
	strncpy(chan->name, str, sizeof(chan->name));
	chan->name[sizeof(chan->name)-1] = '\0';

	status = dqnet_chan_qs_type_1(phan_node, chan);
	if (status)
		status = dqnet_chan_qs_type_2(phan_node, chan);
	if (status)
		goto err_put_node;

	status = dqnet_read_prop_string(phan_node, "type", &str);
	if (status)
		goto err_put_node;
	if (!strncmp(str, CHAN_TYPE_FAP_EXCEPT_STRING,
		     sizeof(CHAN_TYPE_FAP_EXCEPT_STRING)))
		chan->type = DQNET_CHAN_TYPE_FAP_EXCEPT;
	else if (!strncmp(str, CHAN_TYPE_FAP_HOST_STRING,
		     sizeof(CHAN_TYPE_FAP_HOST_STRING)))
		chan->type = DQNET_CHAN_TYPE_FAP_HOST;
	else if (!strncmp(str, CHAN_TYPE_POINT_TO_POINT_STRING,
		     sizeof(CHAN_TYPE_POINT_TO_POINT_STRING)))
		chan->type = DQNET_CHAN_TYPE_POINT_TO_POINT;
	else if (!strncmp(str, CHAN_TYPE_POINT_TO_POINT_SWAP_STRING,
		     sizeof(CHAN_TYPE_POINT_TO_POINT_SWAP_STRING)))
		chan->type = DQNET_CHAN_TYPE_POINT_TO_POINT_SWAP;
	else if (!strncmp(str, CHAN_TYPE_SPOOFED_STRING,
		     sizeof(CHAN_TYPE_SPOOFED_STRING)))
		chan->type = DQNET_CHAN_TYPE_SPOOFED;
	else {
		pr_err("Invalid value specified for type for channel %s.\n",
		    chan->name);
		status = -EINVAL;
		goto err_put_node;
	}

	status = dqnet_read_prop_string(phan_node, "q-msg-fmt", &str);
	if (status)
		chan->qmsg_fmt = DQNET_QMSG_FMT_GFAP_FPM;
	else if (!strncmp(str, QMSG_FMT_GFAP_FPM_STRING,
			  sizeof(QMSG_FMT_GFAP_FPM_STRING)))
		chan->qmsg_fmt = DQNET_QMSG_FMT_GFAP_FPM;
	else if (!strncmp(str, QMSG_FMT_GFAP_SKB_STRING,
			  sizeof(QMSG_FMT_GFAP_SKB_STRING)))
		chan->qmsg_fmt = DQNET_QMSG_FMT_GFAP_SKB;
	else if (!strncmp(str, QMSG_FMT_GFAP3_FPM_STRING,
			  sizeof(QMSG_FMT_GFAP3_FPM_STRING)))
		chan->qmsg_fmt = DQNET_QMSG_FMT_GFAP3_FPM;
	else if (!strncmp(str, QMSG_FMT_GFAP3_SKB_STRING,
			  sizeof(QMSG_FMT_GFAP3_SKB_STRING)))
		chan->qmsg_fmt = DQNET_QMSG_FMT_GFAP3_SKB;
#ifdef CONFIG_BCM_RUNNER
	else if (!strncmp(str, QMSG_FMT_RFAP_FPM_STRING,
			  sizeof(QMSG_FMT_RFAP_FPM_STRING)))
		chan->qmsg_fmt = DQNET_QMSG_FMT_RFAP_FPM;
	else if (!strncmp(str, QMSG_FMT_RFAP_SKB_STRING,
			  sizeof(QMSG_FMT_RFAP_SKB_STRING)))
		chan->qmsg_fmt = DQNET_QMSG_FMT_RFAP_SKB;
#endif
	else {
		pr_err("Invalid value specified for q-msg-fmt for channel %s.\n",
		    chan->name);
		status = -EINVAL;
		goto err_put_node;
	}
	of_node_put(phan_node);

	status = dqnet_read_prop_string(of_node, "link-type", &str);
	if (status)
		goto done;
	if (!strncmp(str, LINK_TYPE_RPC_STRING,
		     sizeof(LINK_TYPE_RPC_STRING)))
		ndev->link_type = DQNET_LINK_TYPE_RPC;
	else if (!strncmp(str, LINK_TYPE_PHY_STRING,
		     sizeof(LINK_TYPE_PHY_STRING)))
		ndev->link_type = DQNET_LINK_TYPE_PHY;
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	else if (!strncmp(str, LINK_TYPE_SWITCH_STRING,
		     sizeof(LINK_TYPE_SWITCH_STRING)))
		ndev->link_type = DQNET_LINK_TYPE_SWITCH;
#endif
	else if (!strncmp(str, LINK_TYPE_FIXED_STRING,
		     sizeof(LINK_TYPE_FIXED_STRING)))
		ndev->link_type = DQNET_LINK_TYPE_FIXED;
	else {
		pr_err("Invalid value specified for link-type for device %s.\n",
		    ndev->name);
		status = -EINVAL;
		goto done;
	}

	if (ndev->link_type == DQNET_LINK_TYPE_RPC) {
		phan_node = of_parse_phandle(of_node, "rpc-channel", 0);
		if (!phan_node) {
			pr_err("Unable to retrieve rpc-channel phandle for ");
			pr_err("device %s.\n", ndev->name);
			status = -EINVAL;
			goto done;
		}

		status = dqnet_read_prop_string(phan_node, "dev-name",
						    &str);
		if (status)
			goto err_put_node;
		strncpy(ndev->rpc_name, str, sizeof(ndev->rpc_name));
		ndev->rpc_name[sizeof(ndev->rpc_name)-1] = '\0';
		of_node_put(phan_node);
	}

	if (ndev->link_type == DQNET_LINK_TYPE_PHY) {
		hal_init(&pdev->dev, &ndev->hal);

		phan_node = of_parse_phandle(of_node, "mac-handle", 0);
		if (phan_node) {
			if (!dqnet_read_prop_string(phan_node, "status", &str)) {
				if(!strncmp(str, "disabled", 8)) {
					pr_info("%s device: mac-handle status = %s\n", ndev->name,
							str);
					status = -ENODEV;
					goto err_put_node;
				}
			}
			of_node_put(phan_node);
		}

		if (ndev->hal.macdev)
			pr_info("%s: %s: mac device %s\n", MODULE_NAME,
					ndev->name, ndev->hal.macdev->name);

		if (ndev->hal.miidev)
			pr_info("%s: %s: mii device %s\n", MODULE_NAME,
					ndev->name, ndev->hal.miidev->name);

		if (of_phy_is_fixed_link(of_node)) {
			/*
			 * In the case of a fixed PHY, the DT node associated
			 * to the PHY is the net device DT node.
			 */
			status = of_phy_register_fixed_link(of_node);
			if (!status)
				ndev->phy_dn = of_node;
		} else {
			phan_node = of_parse_phandle(of_node, "phy-handle", 0);
			ndev->phy_dn = phan_node;
		}
		if (!ndev->phy_dn) {
			pr_err("Unable to retrieve phy phandle for ");
			pr_err("device %s.\n", ndev->name);
			status = -EINVAL;
			goto done;
		}

		ndev->phy_mode = of_get_phy_mode(ndev->phy_dn);
		if (ndev->phy_mode < 0) {
			pr_err("Unable to retrieve phy mode for ");
			pr_err("device %s.\n", ndev->name);
			status = ndev->phy_mode;
			goto err_put_node;
		}
	}

	if (ndev->link_type == DQNET_LINK_TYPE_FIXED) {
		//phan_node = of_parse_phandle(of_node, "iperf-channel", 0);
		//if (!phan_node) {
		//	pr_err("Unable to retrieve iperf-channel phandle for ");
		//	pr_err("device %s.\n", ndev->name);
		//	status = -EINVAL;
		//	goto done;
		//}

		status = dqnet_read_prop_string(phan_node, "dev-name",
						    &str);
		if (status)
			goto err_put_node;
		strncpy(ndev->iperf_name, str, sizeof(ndev->iperf_name));
		ndev->iperf_name[sizeof(ndev->rpc_name)-1] = '\0';
		of_node_put(phan_node);
	}


	goto done;

err_put_node:
	if (phan_node)
		of_node_put(phan_node);

done:
	return status;
}
