 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/proc_fs.h>

#include <bcmnethooks.h>

#include "dqnet.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "dqnet_procfs.h"
#include "dqnet_fap.h"
#include "dqnet_nethooks.h"

#define PROC_DIR		"driver/dqnet"
#define NETHOOKS_PROC_FILE	"nethooks"
#define CMD_PROC_FILE		"cmd"

static int dqnet_show_stats(struct seq_file *seq);

static void *dqnet_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (!*pos)
		return SEQ_START_TOKEN;
	return 0;
}

static void *dqnet_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	(*pos)++;
	return 0;
}

static void dqnet_seq_stop(struct seq_file *seq, void *v)
{
}

static int stats_seq_show(struct seq_file *seq, void *v)
{
	if (!v)
		return -1;
	dqnet_show_stats(seq);
	return 0;
}

static const struct seq_operations stats_seq_ops = {
	.start	= dqnet_seq_start,
	.next	= dqnet_seq_next,
	.stop	= dqnet_seq_stop,
	.show	= stats_seq_show,
};

static int cmd_statsclear(int argc, char *argv[])
{
	err_stats.exceeded_budget = 0;
	err_stats.dqm_rx = 0;
	err_stats.bad_q_msg = 0;
	err_stats.if_id_lookup = 0;
	err_stats.if_sub_id_lookup = 0;
	err_stats.no_eth_hdr = 0;
	err_stats.switch_arl_lookup = 0;
	err_stats.brcm_tag_lookup = 0;
	err_stats.drop_if_down = 0;
	err_stats.drop_carrier_off = 0;
	err_stats.drop_tx_rebroadcast = 0;
	err_stats.drop_switch_to_switch = 0;
	err_stats.dqm_tx_busy = 0;
	err_stats.dqm_tx_failed = 0;
	err_stats.fpm_alloc_failed = 0;
	err_stats.wd_timeout = 0;
	err_stats.rpc_dev_unknown = 0;
	err_stats.rpc_msg_bad = 0;
	err_stats.wlan_conversion_failed = 0;
#ifdef FPM_IN_SKB
	atomic_set(&err_stats.fpm_in_skb, 0);
	atomic_set(&err_stats.fpm_in_skb_free, 0);
	atomic_set(&err_stats.fpm_in_skb_shinfo_head, 0);
#endif
	return 0;
}

static void cmd_statsclear_help(char *str)
{
	pr_alert("%s clear: Clear driver common stats\n", str);
}

static struct proc_cmd_ops stats_command_entries[] = {
	PROC_CMD_INIT("clear", cmd_statsclear),
};

struct proc_cmd_table stats_command_table = {
	.module_name = "DQNET",
	.size = ARRAY_SIZE(stats_command_entries),
	.data_seq_read = (void *) &stats_seq_ops,
	.ops = stats_command_entries
};

static int dqnet_show_swport(struct seq_file *seq)
{
	u8 i, j;
	pr_seq(seq, "DQNET Switch port forwarding table (Row: From and Col: To):\n");
	pr_seq(seq, "   ");
	for (i = 0; i < MAX_PHY_PORTS; i++) {
		pr_seq_cont(seq, " %d", i);
	}
	pr_seq(seq, "\n   ");
	for (i = 0; i < MAX_PHY_PORTS; i++) {
		pr_seq_cont(seq, "__");
	}
	for (i = 0; i < MAX_PHY_PORTS; i++) {
		pr_seq(seq, "\n%d |", i);
		for (j = 0; j < MAX_PHY_PORTS; j++) {
			pr_seq_cont(seq, " %d", dqnet_swport_fwd[i][j]);
		}
	}
	pr_seq(seq, "\n\n");
	return 0;
}

static int swport_seq_show(struct seq_file *seq, void *v)
{
	if (!v)
		return -1;
	dqnet_show_swport(seq);
	return 0;
}

static const struct seq_operations swport_seq_ops = {
	.start	= dqnet_seq_start,
	.next	= dqnet_seq_next,
	.stop	= dqnet_seq_stop,
	.show	= swport_seq_show,
};

static void *dqnet_nethooks_proc_start(struct seq_file *seq, loff_t *pos)
{
	struct dqnet_netdev *ndev = NULL;
	struct dqnet_netdev *tmp;
	loff_t off = 0;

	pr_debug("-->\n");
	if (*pos == 0) {
#if defined (CONFIG_ARM64)
		seq_printf(seq, " Inter-  |                    | Hook  | Ena- |  Pass      |  Drop      |  Skip      |  Consume   | Pri-  |\n");
		seq_printf(seq, " face    |      Hook ID       | Point | bled |  Count     |  Count     |  Count     |  Count     | ority | Name\n");
#else
		seq_printf(seq, " Inter-  |            | Hook  | Ena- |  Pass      |  Drop      |  Skip      |  Consume   | Pri-  |\n");
		seq_printf(seq, " face    |  Hook ID   | Point | bled |  Count     |  Count     |  Count     |  Count     | ority | Name\n");
#endif
	}

	rcu_read_lock();
	list_for_each_entry_rcu(tmp, &ndevs.list, list) {
		if (off++ == *pos) {
			ndev = tmp;
			break;
		}
	}
	pr_debug("<--\n");
	return ndev;
}

static void dqnet_nethooks_proc_stop(struct seq_file *seq, void *v)
{
	pr_debug("-->\n");
	rcu_read_unlock();
	pr_debug("<--\n");
}

static void *dqnet_nethooks_proc_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	struct dqnet_netdev *ndev = NULL;
	struct dqnet_netdev *cur = v;
	struct list_head *next;

	pr_debug("-->\n");
	(*pos)++;
	next = list_next_rcu(&cur->list);
	if (next != &ndevs.list)
		ndev = list_entry_rcu(next, struct dqnet_netdev, list);
	pr_debug("<--\n");
	return ndev;
}

static int dqnet_nethooks_proc_show(struct seq_file *seq, void *v)
{
	int status = 0;
	int found = 0;
	struct dqnet_netdev *ndev = v;
	enum bcm_nethook_type type;
	struct bcm_nethook *nethook;

	pr_debug("-->\n");
	rcu_read_lock();
	for (type = 0; type < BCM_NETHOOK_TYPE_MAX; type++) {
		list_for_each_entry_rcu(nethook,
			&ndev->nethooks.hooks[type].list, list) {
			seq_printf(seq, "%8s   ", ndev->dev->name);
			seq_printf(seq, "0x%px   ", nethook->hook);
			switch (type) {
			case BCM_NETHOOK_RX_SKB:
				seq_printf(seq, "RxSKB");
				break;
			case BCM_NETHOOK_TX_SKB:
				seq_printf(seq, "TxSKB");
				break;
			case BCM_NETHOOK_RX_FPM:
				seq_printf(seq, "RxFPM");
				break;
			case BCM_NETHOOK_TX_FPM:
				seq_printf(seq, "TxFPM");
				break;
			/* coverity [dead_error_begin] */
			default:
				seq_printf(seq, "???????");
				break;
			}
			seq_printf(seq, "   ");
			seq_printf(seq, "%4s   ", nethook->enabled ?
				   "Y " : "N ");
			seq_printf(seq, "0x%08x   ", nethook->count[BCM_NETHOOK_PASS]);
			seq_printf(seq, "0x%08x   ", nethook->count[BCM_NETHOOK_DROP]);
			seq_printf(seq, "0x%08x   ", nethook->count[BCM_NETHOOK_SKIP]);
			seq_printf(seq, "0x%08x   ", nethook->count[BCM_NETHOOK_CONSUMED]);
			seq_printf(seq, "%4d    ", nethook->priority);
			seq_printf(seq, "%s", nethook->name);
			seq_printf(seq, "\n");
			found = 1;
		}
	}
	if (!found)
		seq_printf(seq, "%8s   No-hooks\n", ndev->dev->name);
	rcu_read_unlock();
	pr_debug("<--\n");
	return status;
}

static const struct seq_operations nethooks_seq_ops = {
	.start	= dqnet_nethooks_proc_start,
	.stop	= dqnet_nethooks_proc_stop,
	.next	= dqnet_nethooks_proc_next,
	.show	= dqnet_nethooks_proc_show,
};

static void dqnet_proc_cmd_nethook_enable_help(char *str)
{
	pr_alert("%s enable: Enable/disable specific nethook or all nethooks\n", str);
	pr_alert("%s  nethook <1|0> <ifname> <hook_id> <hook_point>\n", str);
	pr_alert("%s   ifname     interface name (e.g. eth0) or \"all\"\n", str);
	pr_alert("%s   hook_id    ID of hook (e.g. 0x12345678) or \"all\"\n", str);
	pr_alert("%s   hook_point hook point (e.g. TxSKB) or \"all\"\n", str);
}

static int dqnet_proc_cmd_nethook_enable(int argc, char *argv[])
{
	int status = 0;
	bool enable;
	struct dqnet_netdev *ndev;
#if defined (CONFIG_ARM64)
	u64 hook_id;
#else
	unsigned long hook_id;
#endif
	enum bcm_nethook_type type, type_min, type_max;
	struct bcm_nethook *nethook;
	bool found = false;

	pr_debug("-->\n");
	if (argc != 5)
		goto help;
	if (!strncmp(argv[1], "1", 2))
		enable = true;
	else if (!strncmp(argv[1], "0", 15))
		enable = false;
	else
		goto help;
	if (!strncmp(argv[3], "all", 3)) {
		hook_id = 0;
#if defined (CONFIG_ARM64)
	} else if (kstrtou64(argv[3], 0, &hook_id)) {
#else
	} else if (kstrtoul(argv[3], 0, &hook_id)) {
#endif
		pr_err("Invalid hook ID, %s.\n", argv[3]);
		goto help;
	}
	if (!strncmp(argv[4], "RxFPM", 5)) {
		type_min = BCM_NETHOOK_RX_FPM;
		type_max = type_min + 1;
	} else if (!strncmp(argv[4], "RxSKB", 5)) {
		type_min = BCM_NETHOOK_RX_SKB;
		type_max = type_min + 1;
	} else if (!strncmp(argv[4], "TxSKB", 5)) {
		type_min = BCM_NETHOOK_TX_SKB;
		type_max = type_min + 1;
	} else if (!strncmp(argv[4], "TxFPM", 5)) {
		type_min = BCM_NETHOOK_TX_FPM;
		type_max = type_min + 1;
	} else if (!strncmp(argv[4], "all", 3)) {
		type_min = 0;
		type_max = BCM_NETHOOK_TYPE_MAX;
	} else {
		pr_err("Invalid hook type, %s.\n", argv[4]);
		goto help;
	}
	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (strncmp(ndev->dev->name, argv[2], IFNAMSIZ) &&
		    strncmp(argv[2], "all", 3))
			continue;
		for (type = type_min; type < type_max; type++) {
			list_for_each_entry_rcu(nethook,
				&ndev->nethooks.hooks[type].list, list) {
				if (nethook->hook == (bcm_nethook_fn)hook_id ||
				    hook_id == 0) {
					found = true;
					nethook->enabled = enable;
				}
			}
		}
	}
	rcu_read_unlock();
	if (!found)
		pr_err("Unable to find specified nethook.\n");
	goto done;

help:
	dqnet_proc_cmd_nethook_enable_help("");
	status = -EINVAL;

done:
	pr_debug("<--\n");
	return status;
}

static int dqnet_show_stats(struct seq_file *seq)
{
	pr_seq(seq, "Error Stats:\n");
	pr_seq(seq, " exceeded_budget       = %d\n", err_stats.exceeded_budget);
	pr_seq(seq, " dqm_rx                = %d\n", err_stats.dqm_rx);
	pr_seq(seq, " bad_q_msg             = %d\n", err_stats.bad_q_msg);
	pr_seq(seq, " if_id_lookup          = %d\n", err_stats.if_id_lookup);
	pr_seq(seq, " if_sub_id_lookup      = %d\n", err_stats.if_sub_id_lookup);
	pr_seq(seq, " no_eth_hdr            = %d\n", err_stats.no_eth_hdr);
	pr_seq(seq, " switch_arl_lookup     = %d\n", err_stats.switch_arl_lookup);
	pr_seq(seq, " brcm_tag_lookup       = %d\n", err_stats.brcm_tag_lookup);
	pr_seq(seq, " drop_if_down          = %d\n", err_stats.drop_if_down);
	pr_seq(seq, " drop_carrier_off      = %d\n", err_stats.drop_carrier_off);
	pr_seq(seq, " drop_tx_rebroadcast   = %d\n", err_stats.drop_tx_rebroadcast);
	pr_seq(seq, " drop_switch_to_switch = %d\n", err_stats.drop_switch_to_switch);
	pr_seq(seq, " dqm_tx_busy           = %d\n", err_stats.dqm_tx_busy);
	pr_seq(seq, " dqm_tx_failed         = %d\n", err_stats.dqm_tx_failed);
	pr_seq(seq, " fpm_alloc_failed      = %d\n", err_stats.fpm_alloc_failed);
	pr_seq(seq, " wd_timeout            = %d\n", err_stats.wd_timeout);
	pr_seq(seq, " rpc_dev_unknown       = %d\n", err_stats.rpc_dev_unknown);
	pr_seq(seq, " rpc_msg_bad           = %d\n", err_stats.rpc_msg_bad);
	pr_seq(seq, " wlan_conversion_failed= %d\n", err_stats.wlan_conversion_failed);
#ifdef FPM_IN_SKB
	pr_seq(seq, " fpm_in_skb            = %d\n", atomic_read(&err_stats.fpm_in_skb));
	pr_seq(seq, " fpm_in_skb_free       = %d\n", atomic_read(&err_stats.fpm_in_skb_free));
	pr_seq(seq, " fpm_in_skb_shinfo     = %d\n", atomic_read(&err_stats.fpm_in_skb_shinfo_head));
	pr_seq(seq, "\n");
#endif
	return 0;
}

static void dqnet_proc_cmd_stats_help(char *str)
{
	pr_alert("%s show: Show driver common stats\n", str);
}

static int dqnet_proc_cmd_stats(int argc, char *argv[])
{
	dqnet_show_stats(NULL);
	return 0;
}

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
static void dqnet_proc_cmd_brcmtag_opcode_help(char *str)
{
	pr_alert("%s brcmtag_opcode: Set brcmtag opcode for tx unicast or multicast packet\n", str);
	pr_alert("%s   brcmtag_opcode <mc|uc> <opcode> <ifname>\n", str);
	pr_alert("%s    mc     multicast packets\n", str);
	pr_alert("%s    uc     unicast packets\n", str);
	pr_alert("%s    opcode 0 or 1\n", str);
	pr_alert("%s    ifname interface name (e.g. eth0) or \"all\"\n", str);
}

static int dqnet_proc_cmd_brcmtag_opcode(int argc, char *argv[])
{
	int status = 0;
	struct dqnet_netdev *ndev;
	u32 opcode;
	char *ifname = "all";

	pr_debug("-->\n");
	if (argc < 3)
		goto help;

	if (kstrtouint(argv[2], 0, &opcode))
		goto help;

	if ((opcode != 0) && (opcode != 1))
		goto help;

	if (argc == 4)
		ifname = argv[3];

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (strncmp(ndev->dev->name, ifname, sizeof(IFNAMSIZ)) &&
		    strncmp(ifname, "all", 3))
			continue;

		if (!strncmp(argv[1], "mc", 2)) {
			ndev->brcmtag_opc_mc = opcode;
		} else if (!strncmp(argv[1], "uc", 2)) {
			ndev->brcmtag_opc_uc = opcode;
		}
	}
	rcu_read_unlock();
	goto done;

help:
	dqnet_proc_cmd_brcmtag_opcode_help("");
done:
	pr_debug("<--\n");
	return status;
}

static void dqnet_proc_cmd_swport_forward_help(char *str)
{
	pr_alert("%s swport_forward: Enable/disable forwarding packets between switch ports\n", str);
	pr_alert("%s  Display Forwarding Table\n", str);
	pr_alert("%s   swport_forward\n", str);
	pr_alert("%s  Enable/Disable Rx Port -> Tx Port\n", str);
	pr_alert("%s   swport_forward <rx port> <tx port> <enable>\n", str);
	pr_alert("%s  Enable/Disable Port <-> All Port\n", str);
	pr_alert("%s   swport_forward <port> <enable>\n", str);
	pr_alert("%s    port    0,1,2..Max Switch Port-1\n", str);
	pr_alert("%s    enable  1 to Enable 0 to Disable\n", str);
}

static int dqnet_proc_cmd_swport_forward(int argc, char *argv[])
{
	u8 i, j, e;
	pr_debug("-->\n");
	if (argc == 1) {
		dqnet_show_swport(NULL);
	} else if (argc == 4) {
		if (kstrtou8(argv[1], 0, &i))
			goto help;
		if (kstrtou8(argv[2], 0, &j))
			goto help;
		if (kstrtou8(argv[3], 0, &e))
			goto help;
		if (i >= MAX_PHY_PORTS)
			goto help;
		if (j >= MAX_PHY_PORTS)
			goto help;
		dqnet_swport_fwd[i][j] = e;
	} else if (argc == 3) {
		if (kstrtou8(argv[1], 0, &i))
			goto help;
		if (kstrtou8(argv[2], 0, &e))
			goto help;
		if (i >= MAX_PHY_PORTS)
			goto help;
		for (j = 0; j < MAX_PHY_PORTS; j++) {
			if (i == j)
				continue;
			dqnet_swport_fwd[i][j] = e;
			dqnet_swport_fwd[j][i] = e;
		}
	}
	goto done;
help:
	dqnet_proc_cmd_swport_forward_help("");
done:
	pr_debug("<--\n");
	return 0;
}
#endif

static void dqnet_proc_cmd_swport_wan_help(char *str)
{
	pr_alert("%s swport_wan: Enable/disable WAN configuration on switch port\n", str);
	pr_alert("%s  swport_wan <ifname> <1|0>\n", str);
	pr_alert("%s   ifname interface name (e.g. eth0)\n", str);
}

static int dqnet_proc_cmd_swport_wan(int argc, char *argv[])
{
	int status = 0;
	int enable;
	char *ifname;

	pr_debug("-->\n");
	if (argc == 3) {
		ifname = argv[1];
		if (kstrtouint(argv[2], 0, &enable))
			goto help;
		dqnet_set_wan(ifname, enable);
		goto done;
	}

help:
	dqnet_proc_cmd_swport_wan_help("");
done:
	pr_debug("<--\n");
	return status;
}

static void dqnet_proc_cmd_napi_weight_help(char *str)
{
	pr_alert("%s napi_weight: Show/set NAPI weight per interface\n", str);
	pr_alert("%s  Show/Set NAPI weight per interface or all interface\n", str);
	pr_alert("%s   napi_weight <ifname|all>\n", str);
	pr_alert("%s  Set NAPI weight per interface or all interface\n", str);
	pr_alert("%s   napi_weight <ifname|all> <napi_weigh>t\n", str);
	pr_alert("%s    ifname interface name (e.g. eth0) or \"all\"\n", str);
}

static int dqnet_proc_cmd_napi_weight(int argc, char *argv[])
{
	int status = 0;
	struct dqnet_netdev *ndev;
	int napi_weight = 0;
	char *ifname = "all";

	pr_debug("-->\n");

	if (argc > 1) {
		if (!strncmp(argv[1], "help", sizeof("help")))
			goto help;
		ifname = argv[1];
	}

	if (argc > 2)
		if (kstrtouint(argv[2], 0, &napi_weight))
			goto help;

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (strncmp(ndev->dev->name, ifname, sizeof(IFNAMSIZ)) &&
		    strncmp(ifname, "all", 3))
			continue;

		if (argc > 2) {
			if (napi_weight > ndev->chan->max_msg) {
				napi_weight = ndev->chan->max_msg;
				pr_alert("%s: napi_weight clampped to max %d\n",
					ndev->dev->name,
					napi_weight);
			}
			ndev->chan->napi.weight = napi_weight;
		}
		pr_alert("%s: napi_weight %d\n", ndev->dev->name,
			ndev->chan->napi.weight);

	}
	rcu_read_unlock();
	goto done;

help:
	dqnet_proc_cmd_napi_weight_help("");
done:
	pr_debug("<--\n");
	return status;
}

static void dqnet_proc_cmd_rxqueue_cpu_def_help(char *str)
{
	pr_alert("%s rxqueue_cpu_def: Set Rx queue CPU affinity to default\n", str);
	pr_alert("%s  Rx Downstream queue type (3390 only) is affine to CPU 0\n", str);
	pr_alert("%s  All other queues are affine to all CPU\n", str);
}

static int dqnet_proc_cmd_rxqueue_cpu_def(int argc, char *argv[])
{
	int status = 0;
	struct dqnet_netdev *ndev;

	pr_debug("-->\n");
	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		dqnet_rxqueue_cpu_affinity_default(ndev);
	}
	rcu_read_unlock();
	pr_debug("<--\n");
	return status;
}

static void dqnet_proc_cmd_qm_q_help(char *str)
{
	pr_alert("%s qm_q: Show QM Queue assigned to interface\n", str);
	pr_alert("%s  qm_q <ifname>\n", str);
	pr_alert("%s   ifname interface name (e.g. eth0) or \"all\"\n", str);
}

static int dqnet_proc_cmd_qm_q(int argc, char *argv[])
{
	int status = 0, i;
	struct dqnet_netdev *ndev;
	char *ifname = "all";

	pr_debug("-->\n");
	if (argc == 2) {
		ifname = argv[1];
		rcu_read_lock();
		list_for_each_entry_rcu(ndev, &ndevs.list, list) {
			if (strncmp(ndev->dev->name, ifname, sizeof(IFNAMSIZ)) &&
			    strncmp(ifname, "all", 3))
				continue;
			pr_info("%s: ", ndev->dev->name);
			for (i = 0; i < ndev->tx_qm_q_count; i++)
				pr_cont(" %d", ndev->tx_qm_q[i]);
			pr_info("\n");

		}
		rcu_read_unlock();
		goto done;
	}
	dqnet_proc_cmd_qm_q_help("");
done:
	pr_debug("<--\n");
	return status;
}

static struct proc_cmd_ops command_entries[] = {
	PROC_CMD_INIT("stats", dqnet_proc_cmd_stats),
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	PROC_CMD_INIT("brcmtag_opcode", dqnet_proc_cmd_brcmtag_opcode),
	PROC_CMD_INIT("swport_forward", dqnet_proc_cmd_swport_forward),
#endif
	PROC_CMD_INIT("swport_wan", dqnet_proc_cmd_swport_wan),
	PROC_CMD_INIT("napi_weight", dqnet_proc_cmd_napi_weight),
	PROC_CMD_INIT("rxqueue_cpu_def", dqnet_proc_cmd_rxqueue_cpu_def),
	PROC_CMD_INIT("qm_q", dqnet_proc_cmd_qm_q),
};

struct proc_cmd_table dqnet_command_table = {
	.module_name = "DQNET",
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
static struct proc_cmd_ops swport_command_entries[] = {
	PROC_CMD_INIT("forward", dqnet_proc_cmd_swport_forward),
	PROC_CMD_INIT("wan", dqnet_proc_cmd_swport_wan),
};

struct proc_cmd_table swport_command_table = {
	.module_name = "DQNET",
	.size = ARRAY_SIZE(swport_command_entries),
	.data_seq_read = (void *) &swport_seq_ops,
	.ops = swport_command_entries
};
#endif

static struct proc_cmd_ops nethooks_command_entries[] = {
	PROC_CMD_INIT("enable", dqnet_proc_cmd_nethook_enable),
};

struct proc_cmd_table nethooks_command_table = {
	.module_name = "DQNET",
	.size = ARRAY_SIZE(nethooks_command_entries),
	.data_seq_read = (void *) &nethooks_seq_ops,
	.ops = nethooks_command_entries
};

static struct proc_dir_entry *proc_dir;
static struct proc_dir_entry *nethooks_proc_file;
static struct proc_dir_entry *cmd_proc_file;
static struct proc_dir_entry *stats_proc_file;
static struct proc_dir_entry *swport_proc_file;

int __init dqnet_proc_init(void)
{
	int status = 0;

	pr_debug("-->\n");
	proc_dir = proc_mkdir(PROC_DIR, NULL);
	if (!proc_dir) {
		pr_err("Failed to create PROC directory %s.\n",
		       PROC_DIR);
		status = -EIO;
		goto done;
	}
	nethooks_proc_file = proc_create_cmd(NETHOOKS_PROC_FILE, proc_dir,
					     &nethooks_command_table);
	if (!nethooks_proc_file) {
		pr_err("Failed to create %s\n", NETHOOKS_PROC_FILE);
		status = -EIO;
		dqnet_proc_exit();
		goto done;
	}
	cmd_proc_file = proc_create_cmd(CMD_PROC_FILE, proc_dir,
					&dqnet_command_table);
	if (!cmd_proc_file) {
		pr_err("Failed to create %s\n", CMD_PROC_FILE);
		status = -EIO;
		dqnet_proc_exit();
		goto done;
	}
	pr_err("Created %s file\n", CMD_PROC_FILE);
	stats_proc_file = proc_create_cmd("stats", proc_dir,
					  &stats_command_table);
	if (!stats_proc_file) {
		pr_err("Failed to create %s\n", "stats");
		status = -EIO;
		dqnet_proc_exit();
		goto done;
	}

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	swport_proc_file = proc_create_cmd("swport", proc_dir,
					   &swport_command_table);
	if (!swport_proc_file) {
		pr_err("Failed to create %s\n", "swport");
		status = -EIO;
		dqnet_proc_exit();
		goto done;
	}
#endif
done:
	pr_debug("<--\n");
	return status;
}

void dqnet_proc_exit(void)
{
	pr_debug("-->\n");
	if (cmd_proc_file) {
		remove_proc_entry(CMD_PROC_FILE, proc_dir);
		cmd_proc_file = NULL;
	}
	if (stats_proc_file) {
		remove_proc_entry("stats", proc_dir);
		cmd_proc_file = NULL;
	}
	if (swport_proc_file) {
		remove_proc_entry("swport", proc_dir);
		cmd_proc_file = NULL;
	}
	if (nethooks_proc_file) {
		remove_proc_entry(NETHOOKS_PROC_FILE, proc_dir);
		nethooks_proc_file = NULL;
	}
	if (proc_dir) {
		remove_proc_entry(PROC_DIR, NULL);
		proc_dir = NULL;
	}
	pr_debug("<--\n");
}
