 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/of.h>
#include <linux/of_mdio.h>
#include <linux/phy.h>
#include <linux/brcmphy.h>
#include <linux/mdio.h>

#include "ethsw.h"
#include "ethsw_core.h"
#include "ethsw_priv.h"

/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_api"

/*
 * Local Variables
 */
static struct ethsw_port_config port_config_default = {
	.auto_neg    = true,
	.full_duplex = true,
	.speed       = PORT_SPEED_1000M,
};

static struct ethsw_power_config power_config_default = {
	.auto_power            = true,
	.auto_dll              = true,
	.afe_pwrdn             = false,
	.eee_params.enable     = true,
	.eee_params.advertised = MDIO_EEE_100TX | MDIO_EEE_1000T,
	.eee_params.sleep_timer_g  = CREG_EEE_SLEEP_TIMER_G_DFLT,
	.eee_params.sleep_timer_h  = CREG_EEE_SLEEP_TIMER_H_DFLT,
	.eee_params.min_lp_timer_g = CREG_EEE_MIN_LP_TIMER_G_DFLT,
	.eee_params.min_lp_timer_h = CREG_EEE_MIN_LP_TIMER_H_DFLT,
	.eee_params.wake_timer_g   = CREG_EEE_WAKE_TIMER_G_DFLT,
	.eee_params.wake_timer_h   = CREG_EEE_WAKE_TIMER_H_DFLT,
};

/*
 * Ether Switch API Functions
 */

/* Initailly dqnet does not know the actul port_id. Instead it uses 0 based
 * index to connect. ethsw returns the actual port_id. */
int ethsw_set_phy_state(struct ethsw_port *port, int state)
{
	struct phy_device *phydev;

	if (!port) {
		LOG_ERR("Ethsw Port not initialized\n");
		return -EINVAL;
	}
	if (!port->phy.dev) {
		LOG_ERR("Phy device not initialized\n");
		return -EINVAL;
	}
	phydev = port->phy.dev;

	switch (state) {
	case ETHSW_PHY_STOP:
		if ((atomic_read(&port->phy.state) != ETHSW_PHY_STOP) &&
				(phy_is_started(port->phy.dev) ||
				port->phy.dev->state == PHY_DOWN)) {
			phy_stop(phydev);
			atomic_set(&port->phy.state, ETHSW_PHY_STOP);
		}
		break;

	case ETHSW_PHY_START:
		if ((atomic_read(&port->phy.state) != ETHSW_PHY_START) &&
				(phydev->state == PHY_READY ||
				phydev->state == PHY_HALTED)) {
			atomic_set(&port->phy.state, ETHSW_PHY_START);
			phy_start(phydev);
		}
		break;

	default:
		break;
	}

	return 0;
}

int ethsw_port_connect(struct net_device *net_dev,
		int *port_id,
		const char *port_name,
		ethsw_port_cb_t cb,
		void *cb_priv)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(*port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", *port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	/* Convert from 0 based id to the actual port ID. */
	*port_id = swdev->ethsw_port_idx_to_port_id(swdev, *port_id);

	port = &swdev->port[*port_id - swdev->subid_port_offset];
	port->cb      = cb;
	port->cb_priv = cb_priv;
	port->name = port_name;

	if (swdev == ethsw_dev[INTERNAL_SW]) {
		port->phy.dev = of_phy_connect(net_dev, port->phy.node,
				ethsw_phy_link_cb_sf2, 0xE000, port->phy.mode);
	} else
		port->phy.dev = of_phy_connect(net_dev, port->phy.node,
				ethsw_phy_link_cb_531xx, 0, port->phy.mode);

	if (port->phy.dev) {
		LOG_DEBUG("Device %s connected to port %d\n",
				port_name, *port_id);
		phy_suspend(port->phy.dev);
		phy_support_asym_pause(port->phy.dev);
	} else
		LOG_ERR("Failed to connect device %s to port %d\n",
				port->name, *port_id);

	FUNC_LEAVE();

	return ret;
}
EXPORT_SYMBOL(ethsw_port_connect);

int ethsw_port_disconnect(int port_id)
{
	int ret = 0;
	struct ethsw_port *port;

	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	/* disable port */
	ethsw_port_disable(port_id);

	port = &swdev->port[port_id - swdev->subid_port_offset];

	if (port->phy.dev) {
		phy_disconnect(port->phy.dev);
		LOG_DEBUG("Device %s disconnected from port %d\n",
				port->name, port_id);
	} else {
		LOG_ERR("Failed to disconnect device %s from port %d\n,",
				port->name, port_id);
	}

	port->cb      = NULL;
	port->cb_priv = NULL;
	port->name    = NULL;

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_disconnect);

int ethsw_port_enable(int port_id)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];

	/* Enable RX controls for the port */
	ethsw_port_rx_enable(swdev, port_id, 1);

	switch (port->type) {
	case MII_PORT:
		if (port->phy.dev)
			ethsw_set_phy_state(port, ETHSW_PHY_START);
		/* MII port are always up */
		else if (port->cb) {
			struct ethsw_port_status new_status;

			/* fill in port status with new link status */
			new_status.link_up     = true;
			new_status.full_duplex = true;
			new_status.speed       = PORT_SPEED_1000M;

			/* update port status */
			port->status = new_status;

			/* callback with port status */
			(*port->cb)(port->cb_priv, &port->status);
		}
		break;

	case PHY_PORT:
		if (port->phy.dev) {
			swdev->ethsw_led_powercntrl(swdev, port_id, LED_ON);
			LOG_DEBUG("port %i Led on \n", port_id);
			if (port->ethsw_phy_set_afe_power_state) {
				port->ethsw_phy_set_afe_power_state(swdev,
						port_id - swdev->subid_port_offset, false);
				LOG_DEBUG("phy afe power up for port %d\n", port_id);
			}
			else {
				ethsw_set_phy_state(port, ETHSW_PHY_START);
				LOG_DEBUG("phy start for port %d\n", port_id);
			}
		} else {
			LOG_ERR("phy start failed for port %d\n", port_id);
		}
		break;

	default:
		break;
	}

	/* Enable TX controls for the port */
	ethsw_port_tx_enable(swdev, port_id, 1);

	/* save admin state */
	port->enable = true;

	LOG_DEBUG("Port %d enabled\n", port_id);

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_enable);

int ethsw_port_disable(
		int port_id)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];

	/* Disable TX controls for the port */
	ethsw_port_tx_enable(swdev, port_id, 0);

	switch (port->type) {
	case MII_PORT:
		if (port->phy.dev) {
			ethsw_set_phy_state(port, ETHSW_PHY_STOP);
			LOG_DEBUG("phy stop for port %i\n", port_id);
			if(!port->phy.dev->link) {
				phy_suspend(port->phy.dev);
				LOG_DEBUG("phy suspend for port %i\n", port_id);
			}
		}
		/* MII port are always up */
		else if (port->cb) {
			struct ethsw_port_status new_status;

			/* fill in port status with new link status */
			new_status.link_up     = false;
			new_status.full_duplex = true;
			new_status.speed       = PORT_SPEED_1000M;

			/* update port status */
			port->status = new_status;

			/* callback with port status */
			(*port->cb)(port->cb_priv, &port->status);
		}
		break;

	case PHY_PORT:
		if (port->phy.dev) {
			if (port->ethsw_phy_set_afe_power_state) {
				port->ethsw_phy_set_afe_power_state(swdev,
						port_id - swdev->subid_port_offset, true);
				LOG_DEBUG("phy afe power down for port %d\n", port_id);
			}
			else {
				ethsw_set_phy_state(port, ETHSW_PHY_STOP);
				LOG_DEBUG("phy stop for port %i\n", port_id);
				if(!port->phy.dev->link) {
					phy_suspend(port->phy.dev);
					LOG_DEBUG("phy suspend for port %i\n", port_id);
				}
			}
			swdev->ethsw_led_powercntrl(swdev, port_id, LED_OFF);
			LOG_DEBUG("port %i Led off \n", port_id);
		}
		break;

	default:
		break;
	}

	/* Disable RX controls for the port */
	ethsw_port_rx_enable(swdev, port_id, 0);

	/* save admin state */
	port->enable = false;

	LOG_DEBUG("Port %d disabled\n", port_id);

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_disable);

int ethsw_port_config_set(
		int port_id,
		struct ethsw_port_config *port_config)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];
	if (!port_config) {
		LOG_DEBUG("Port %d config reset\n", port_id);
		port_config_default.speed =
			(port->phy.mode == PHY_INTERFACE_MODE_SGMII) ?
			PORT_SPEED_2500M : PORT_SPEED_1000M;
		port_config = &port_config_default;
	}

	/* save port config */
	port->config = *port_config;
	LOG_DEBUG("Port %d config set: auto_neg = %d, full_duplex = %d, speed = %d\n",
			port_id,
			port_config->auto_neg,
			port_config->full_duplex,
			port_config->speed);

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_config_set);

int ethsw_port_config_get(
		int port_id,
		struct ethsw_port_config *port_config)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];
	if (port_config)
		*port_config = port->config;

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_config_get);

int ethsw_port_status_get(
		int port_id,
		struct ethsw_port_status *port_status)
{
	int ret = 0;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX ||
			port_status == NULL) {
		ret = -EINVAL;
		goto EXIT;
	}

	ret = swdev->ethsw_port_status_get(swdev,port_id - swdev->subid_port_offset, port_status);

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_status_get);

int ethsw_port_status_get_sf2(
		struct ethsw_device *swdev,
		int port_id,
		struct ethsw_port_status *port_status)
{
	int ret = 0;
	struct ethsw_port *port;

	FUNC_ENTER();

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX ||
			port_status == NULL) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id];
	if (port->phy.dev) {
		port_status->link_up = port->phy.dev->link;
		port_status->full_duplex = port->phy.dev->duplex;
		port_status->speed = 0;
		switch (port->phy.dev->speed) {
		case SPEED_10:
			port_status->speed = PORT_SPEED_10M;
			break;
		case SPEED_100:
			port_status->speed = PORT_SPEED_100M;
			break;
		case SPEED_1000:
			port_status->speed = PORT_SPEED_1000M;
			break;
		case SPEED_2500:
			port_status->speed = PORT_SPEED_2500M;
			break;
		}

		if (!port_status->link_up)
			port_status->speed =
				(port->phy.mode == PHY_INTERFACE_MODE_SGMII) ?
				PORT_SPEED_2500M : PORT_SPEED_1000M;

		goto EXIT;
	}

	switch (port->type) {
	case IMP_PORT:
	case MII_PORT:
	case PHY_PORT:
		/* get port status via switch */
		port_status->link_up     = ethsw_switch_port_link_up(swdev, port_id);

		if (port_status->link_up)
			port_status->speed       = ethsw_switch_port_speed(port_id);
		else
			port_status->speed       = PORT_SPEED_1000M;

		port_status->full_duplex = ethsw_switch_port_full_duplex(swdev, port_id);
		break;

	default:
		break;
	}

EXIT:
	FUNC_LEAVE();
	return ret;
}
int ethsw_port_status_get_531xx(
		struct ethsw_device *swdev,
		int port_id,
		struct ethsw_port_status *port_status)
{
	int ret = 0;
	u16 link_status =0;
	u32 link_speed = 0;
	u16 link_duplex = 0;

	FUNC_ENTER();

	if (port_id < 0 || port_id >= ETHSW_PHY_PORT_MAX) {
		port_status = NULL;
		ret = -EINVAL;
		goto EXIT;
	}

	if (!(swdev->phy_port_mask & (1 << port_id)))
		goto EXIT;

	ret = ethsw_creg_read(swdev, 1, 0, (u8 *) &link_status, sizeof(link_status));
	if (ret) {
		LOG_ERR("Switch Failed to read the link status register.\n");
		goto EXIT;
	}

	if (link_status & (1 << port_id)) {
		port_status->link_up = true;
		ret = ethsw_creg_read(swdev, 1, 8, (u8 *) &link_duplex, sizeof(link_duplex));
		if (ret) {
			LOG_ERR("Switch Failed to read the link duplex register.\n");
			goto EXIT;
		}
		port_status->full_duplex = (link_duplex >> (port_id)) & 0x1;
	}
	else {
		port_status->link_up = false;
		port_status->full_duplex = false;
	}
	ret = ethsw_creg_read(swdev, 1, 4, (u8 *) &link_speed, sizeof(link_speed));
	if (ret) {
		LOG_ERR("Switch Failed to read the link speed register.\n");
		goto EXIT;
	}
	port_status->speed = (link_speed >> (2 * port_id)) & 0x3;

EXIT:
	FUNC_LEAVE();
	return ret;
}

bool ethsw_port_link_up(int port_id)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;
	bool link_up = false;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];
	switch (port->type) {
	case IMP_PORT:
	case MII_PORT:
	case PHY_PORT:
		/* get port status via switch */
		link_up = ethsw_switch_port_link_up(swdev, port_id);
		break;

	default:
		break;
	}

EXIT:
	FUNC_LEAVE();
	return link_up;
}

bool ethsw_port_full_duplex(int port_id)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;
	bool full_duplex = false;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];
	switch (port->type) {
	case IMP_PORT:
	case MII_PORT:
		/* get port status via switch */
		full_duplex = ethsw_switch_port_full_duplex(swdev, port_id);
		break;
	case PHY_PORT:
		if (port->phy.dev) {
			if (port->phy.dev->duplex == DUPLEX_FULL) {
				full_duplex = true;
			} else {
				full_duplex = false;
			}
		} else {
			LOG_DEBUG("phy set half duplex for port %i by default\n", port_id);
		}
		break;
	default:
		break;
	}

EXIT:
	FUNC_LEAVE();
	return full_duplex;
}
EXPORT_SYMBOL(ethsw_port_full_duplex);

int ethsw_port_power_config_set(
		int port_id,
		struct ethsw_power_config *power_config)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	if (!power_config) {
		LOG_DEBUG("Port %d power config reset\n", port_id);
		power_config = &power_config_default;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];

	/* PHY power config */
	switch (port->type) {
	case MII_PORT:
		/* AFE power config can also be supported on MII ports */
		if (port->ethsw_phy_set_afe_power_state)
			port->power_config.afe_pwrdn = power_config->afe_pwrdn;
		break;

	case PHY_PORT:
		if (swdev->eee_support) {
			/* set power config via switch core */
			CHECK_EXIT(ethsw_switch_port_eee_config_set(
						port_id,
						power_config));
		}

		/* save port power config */
		port->power_config = *power_config;

		LOG_DEBUG("Port %d power config set: eee_enable = %d\n",
				port_id,
				power_config->eee_params.enable);
		break;

	default:
		break;
	}

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_power_config_set);

int ethsw_port_power_config_get(
		int port_id,
		struct ethsw_power_config *power_config)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];

	if (power_config)
		*power_config = port->power_config;

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_power_config_get);

int ethsw_port_power_status_get(
		int port_id,
		struct ethsw_power_status *power_status)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX ||
			power_status == NULL) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];

	switch (port->type) {
	case IMP_PORT:
	case MII_PORT:
		power_status->power_up = port->enable;
		power_status->dll_up   = port->enable;
		if (port->ethsw_phy_get_afe_power_state)
			power_status->afe_pwrdn =
				port->ethsw_phy_get_afe_power_state(swdev,
						port_id
						- swdev->subid_port_offset);

		power_status->eee_capable  = false;
		power_status->eee_assert   = false;
		power_status->eee_indicate = false;
		power_status->eee_supported = 0;
		break;

	case PHY_PORT:
		power_status->eee_assert =
			(!port->power_config.eee_params.enable) ? false :
			ethsw_switch_port_eee_assert(port_id);

		power_status->eee_indicate =
			(!port->power_config.eee_params.enable) ? false :
			ethsw_switch_port_eee_indicate(port_id);

		if (port->ethsw_phy_get_afe_power_state)
			power_status->afe_pwrdn =
				port->ethsw_phy_get_afe_power_state(swdev,
						port_id
						- swdev->subid_port_offset);

		if (!port->phy.dev || !port->phy.dev->mdio.bus)
			break;

		power_status->eee_supported = 0;
		ret = phy_read_mmd(port->phy.dev, MDIO_MMD_PCS, MDIO_PCS_EEE_ABLE);
		if (ret >= 0)
			power_status->eee_supported = ret;

		power_status->eee_lp_advertised = 0;
		ret = phy_read_mmd(port->phy.dev, MDIO_MMD_AN, MDIO_AN_EEE_LPABLE);
		if (ret >= 0)
			power_status->eee_lp_advertised = ret;

		power_status->eee_active = false;
		if (power_status->eee_lp_advertised) {
			ret = phy_read_mmd(port->phy.dev, MDIO_MMD_AN, BRCM_CL45VEN_EEE_CONTROL);
			if (ret >= 0)
				power_status->eee_active = (ret & LPI_FEATURE_EN) ? true : false;
		}

		ret = 0;
		break;

	default:
		break;
	}

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_power_status_get);

bool ethsw_port_eee_assert(int port_id)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;
	bool eee_assert = false;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];

	switch (port->type) {
	case IMP_PORT:
	case MII_PORT:
		eee_assert = false;
		break;

	case PHY_PORT:
		/* get EEE via switch core */
		eee_assert =
			(!port->power_config.eee_params.enable) ? false :
			ethsw_switch_port_eee_assert(port_id);
		break;

	default:
		break;
	}

EXIT:
	FUNC_LEAVE();
	return eee_assert;
}

bool ethsw_port_eee_indicate(int port_id)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;
	bool eee_indicate = false;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];

	switch (port->type) {
	case IMP_PORT:
	case MII_PORT:
		eee_indicate = false;
		break;

	case PHY_PORT:
		/* get EEE via switch core */
		eee_indicate =
			(!port->power_config.eee_params.enable) ? false :
			ethsw_switch_port_eee_indicate(port_id);
		break;

	default:
		break;
	}

EXIT:
	FUNC_LEAVE();
	return eee_indicate;
}

int ethsw_port_set_stp_state(
		int port_id,
		unsigned int state)
{
	int ret = 0;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	ret = ethsw_switch_set_port_stp_state(port_id, state);

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_set_stp_state);

int ethsw_pwr_down_531xx(struct ethsw_device *swdev)
{
	int ret = 0;

	FUNC_ENTER();

	if (!swdev->ethsw_pwr_status)
		return -EIO;

#if VNET_IF_SUPPORT
	/* exit vnet device */
	ethsw_vnet_exit(swdev);
#endif

#if CDEV_IF_SUPPORT
	/* exit cdev device */
	ethsw_cdev_exit(swdev);
#endif
	/* exit switch PHY */
	ethsw_phy_exit(swdev);

	/* exit switch core */
	ethsw_core_exit(swdev);

	/* exit ethsw device, platformat dependent */
	if (swdev->ethsw_swdev_board_exit) {
		swdev->ethsw_swdev_board_exit(swdev);
	}

	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_pwr_down_531xx);

int ethsw_pwr_up_531xx(struct ethsw_device *swdev)
{
	int ret = 0;
	int i;

	FUNC_ENTER();

	if (swdev->ethsw_pwr_status)
		return -EIO;

	/* init ethsw device, platform dependent */
	ret = swdev->ethsw_swdev_board_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw device board init failed\n");
		goto ERROR;
	}

	/* get PHY port params */
	ethsw_phy_port_init_ids(swdev);

	ethsw_acb_enable(swdev);

	/* init switch core */
	ret = ethsw_core_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw core init failed\n");
		goto ERROR;
	}

	/* init switch PHY */
	ret = ethsw_phy_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw PHY init failed\n");
		goto ERROR;
	}

	/* reset config and disable all ports */
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		if (!((swdev->phy_port_mask | swdev->mii_port_mask) & (1 << i)))
			continue;

		ethsw_port_config_set(i, NULL);
		ethsw_port_power_config_set(i, NULL);
		ethsw_port_disable(i);
	}
#if CDEV_IF_SUPPORT
	/* init cdev device */
	ret = ethsw_cdev_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw cdev init failed\n");
		goto ERROR;
	}
#endif

#if VNET_IF_SUPPORT
	/* init vnet device */
	ret = ethsw_vnet_init(swdev);
	if (ret) {
		LOG_CRIT("Ethsw vnet init failed\n");
		goto ERROR;
	}
#endif

	FUNC_LEAVE();
	return ret;

ERROR:
	/* remove module properly */
	ethsw_pwr_down_531xx(swdev);

	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_pwr_up_531xx);

int ethsw_set_mtu(u16 mtu)
{
	int ret = 0;
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];

	if (!swdev)
		swdev = ethsw_dev[EXTERNAL_SPI_SW];

	if (swdev)
		ret = CREG_WR(PAGE_JUMBO_CTRL,CREG_MIB_GD_FM_MAX_SIZE,&mtu);

	return ret;
}
EXPORT_SYMBOL(ethsw_set_mtu);

int ethsw_alt_ethwan(int imp_port, int lan_port, bool enable)
{
	struct ethsw_device *swdev = subid_to_swdev_table[lan_port];

	if (!swdev)
		swdev = ethsw_dev[INTERNAL_SW];

	return ethsw_alt_ethwan_core(swdev, imp_port, lan_port, enable);
}
EXPORT_SYMBOL(ethsw_alt_ethwan);

int ethsw_port_is_enabled(int port_id)
{
	int ret = 0;
	struct ethsw_port *port;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		return -EINVAL;
	}

	if (!swdev->ethsw_pwr_status)
		return -EIO;

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}

	port = &swdev->port[port_id - swdev->subid_port_offset];
	ret = port->enable;

EXIT:
	FUNC_LEAVE();
	return ret;
}
EXPORT_SYMBOL(ethsw_port_is_enabled);
