 /****************************************************************************
 *
 * Copyright (c) 2020 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <net/netfilter/nf_conntrack_core.h>
#include <net/netfilter/nf_conntrack_offload.h>
#include "proc_cmd.h"
#include "flowmgr.h"

enum flow_tuple_match {
	FLOW_M_DST_INF_BIT,
	FLOW_M_SRC_INF_BIT,
	FLOW_M_ANY_INF_BIT,
	FLOW_M_DST_MAC_BIT,
	FLOW_M_SRC_MAC_BIT,
	FLOW_M_ANY_MAC_BIT,
	FLOW_M_DST_IP_BIT,
	FLOW_M_SRC_IP_BIT,
	FLOW_M_ANY_IP_BIT,
	FLOW_M_IP_PROTO_BIT,
	FLOW_M_DST_PORT_BIT,
	FLOW_M_SRC_PORT_BIT,
	FLOW_M_ANY_PORT_BIT,
	FLOW_M_MAPT_BIT,
};

#define __FLOW_M_BIT(bit)	(1 << (bit))
#define __FLOW_M(name)		__FLOW_M_BIT(FLOW_M_##name##_BIT)
#define FLOW_M_DST_INF		__FLOW_M(DST_INF)
#define FLOW_M_SRC_INF		__FLOW_M(SRC_INF)
#define FLOW_M_ANY_INF		__FLOW_M(ANY_INF)
#define FLOW_M_DST_MAC		__FLOW_M(DST_MAC)
#define FLOW_M_SRC_MAC		__FLOW_M(SRC_MAC)
#define FLOW_M_ANY_MAC		__FLOW_M(ANY_MAC)
#define FLOW_M_DST_IP		__FLOW_M(DST_IP)
#define FLOW_M_SRC_IP		__FLOW_M(SRC_IP)
#define FLOW_M_ANY_IP		__FLOW_M(ANY_IP)
#define FLOW_M_IP_PROTO		__FLOW_M(IP_PROTO)
#define FLOW_M_DST_PORT		__FLOW_M(DST_PORT)
#define FLOW_M_SRC_PORT		__FLOW_M(SRC_PORT)
#define FLOW_M_ANY_PORT		__FLOW_M(ANY_PORT)
#define FLOW_M_MAPT			__FLOW_M(MAPT)

#define FLOW_M_L2		(FLOW_M_DST_MAC | \
				 FLOW_M_SRC_MAC | \
				 FLOW_M_ANY_MAC)
#define FLOW_M_L3		(FLOW_M_DST_IP | \
				 FLOW_M_SRC_IP | \
				 FLOW_M_ANY_IP | \
				 FLOW_M_IP_PROTO)
#define FLOW_M_L4		(FLOW_M_DST_PORT | \
				 FLOW_M_SRC_PORT | \
				 FLOW_M_ANY_PORT)

struct flowmgr_match {
	int enable;
	u32 mask;
	struct net_device *in;
	struct net_device *out;
	struct flowmgr_db_entry *idb;
	struct flowmgr_db_entry *odb;
	struct nf_conntrack_tuple tuple;
};

void flowmgr_tuple_dump(struct seq_file *s,
			struct nf_conntrack_tuple *tuple,
			char *label)
{
	if (tuple->src.l3num == PF_INET6) {
		pr_seq(s, "%s: tuple src=%pI6 dst=%pI6 protonum=%d sport %d dport %d\n", label,
		       tuple->src.u3.ip6, tuple->dst.u3.ip6, tuple->dst.protonum,
		       ntohs(tuple->src.u.tcp.port), ntohs(tuple->dst.u.tcp.port));
	} else {
		pr_seq(s, "%s: tuple src=%pI4 dst=%pI4 protonum=%d sport=%d dport=%d\n", label,
		       &tuple->src.u3.ip, &tuple->dst.u3.ip, tuple->dst.protonum,
		       ntohs(tuple->src.u.tcp.port), ntohs(tuple->dst.u.tcp.port));
	}
}

void flowmgr_match_show(struct seq_file *seq, struct flowmgr_match *match)
{
	struct nf_conntrack_tuple *tuple;
	tuple = &match->tuple;
	pr_seq(seq, "Enable : %d\n",   match->enable);
	pr_seq(seq, "Mask   : 0x%x\n", match->mask);
	if ((match->mask & FLOW_M_DST_INF) && match->out)
		pr_seq(seq, "       : dst inf %s\n", match->out->name);
	if ((match->mask & FLOW_M_SRC_INF) && match->in)
		pr_seq(seq, "       : src inf %s\n", match->in->name);
	if ((match->mask & FLOW_M_ANY_INF) && match->in)
		pr_seq(seq, "       : any inf %s\n", match->in->name);
	if ((match->mask & FLOW_M_DST_MAC) && match->odb)
		pr_seq(seq, "       : dst mac %pM\n", match->odb->addr);
	if ((match->mask & FLOW_M_SRC_MAC) && match->idb)
		pr_seq(seq, "       : src mac %pM\n", match->idb->addr);
	if ((match->mask & FLOW_M_ANY_MAC) && match->idb)
		pr_seq(seq, "       : any mac %pM\n", match->idb->addr);
	if (match->mask & FLOW_M_DST_IP) {
		if (tuple->src.l3num == PF_INET6) {
			pr_seq(seq, "       : dst ip %pI6c\n",
			       tuple->dst.u3.ip6);
		} else {
			pr_seq(seq, "       : dst ip %pI4\n",
			       &tuple->dst.u3.ip);
		}
	}
	if (match->mask & FLOW_M_SRC_IP) {
		if (tuple->src.l3num == PF_INET6) {
			pr_seq(seq, "       : src ip %pI6c\n",
			       tuple->src.u3.ip6);
		} else {
			pr_seq(seq, "       : src ip %pI4\n",
			       &tuple->src.u3.ip);
		}
	}
	if (match->mask & FLOW_M_ANY_IP) {
		if (tuple->src.l3num == PF_INET6) {
			pr_seq(seq, "       : any ip %pI6c\n",
			       tuple->src.u3.ip6);
		} else {
			pr_seq(seq, "       : any ip %pI4\n",
			       &tuple->src.u3.ip);
		}
	}
	if (match->mask & FLOW_M_IP_PROTO)
		pr_seq(seq, "       : protonum %d\n",
		       tuple->dst.protonum);
	if (match->mask & FLOW_M_DST_PORT)
		pr_seq(seq, "       : dst port %d\n",
		       ntohs(tuple->dst.u.tcp.port));
	if (match->mask & FLOW_M_SRC_PORT)
		pr_seq(seq, "       : src port %d\n",
		       ntohs(tuple->src.u.tcp.port));
	if (match->mask & FLOW_M_ANY_PORT)
		pr_seq(seq, "       : any port %d\n",
		       ntohs(tuple->src.u.tcp.port));
	if (match->mask & FLOW_M_MAPT)
		pr_seq(seq, "       : mapt\n");
}

static void *flowmgr_match_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (!*pos)
		return SEQ_START_TOKEN;
	return 0;
}

static void *flowmgr_match_seq_next(struct seq_file *seq, void *v,
				       loff_t *pos)
{
	(*pos)++;
	return 0;
}

static void flowmgr_match_seq_stop(struct seq_file *seq, void *v)
{
}

static int flowmgr_match_seq_show(struct seq_file *seq, void *v)
{
	struct flowmgr_match *match;
	struct proc_cmd_table *table;
	if (!v)
		return -1;
	table = (struct proc_cmd_table *) seq->private;
	match = (struct flowmgr_match *) table->data;
	flowmgr_match_show(seq, match);
	return 0;

}

static const struct seq_operations flowmgr_match_seq_ops = {
	.start	= flowmgr_match_seq_start,
	.next	= flowmgr_match_seq_next,
	.stop	= flowmgr_match_seq_stop,
	.show	= flowmgr_match_seq_show,
};

static int match_inf(struct flowmgr_match *match, char *type, char *dev_name)
{
	struct net_device *dev;
	dev = __dev_get_by_name(&init_net,
				dev_name);
	if (!dev) {
		pr_err("%s Invalid interface name %s\n", __func__, dev_name);
	} else if (!dev->group) {
		pr_err("%s Interface %s not accelerated\n", __func__, dev_name);
	} else if (strcasecmp(type, "dst") == 0) {
		match->out = dev;
		match->mask &= ~(FLOW_M_ANY_INF);
		match->mask |= FLOW_M_DST_INF;
		return 0;
	} else if (strcasecmp(type, "src") == 0) {
		match->in = dev;
		match->mask &= ~(FLOW_M_ANY_INF);
		match->mask |= FLOW_M_SRC_INF;
		return 0;
	} else if (strcasecmp(type, "any") == 0) {
		match->in = dev;
		match->mask &= ~(FLOW_M_DST_INF | FLOW_M_SRC_INF);
		match->mask |= FLOW_M_ANY_INF;
		return 0;
	}
	return -1;
}

static void cmd_inf_help(char *str)
{
	pr_alert("%s inf: Match interface name\n", str);
	pr_alert("%s  inf <dev_name>\n", str);
	pr_alert("%s   dev_name - eth0, cm0 or any\n", str);
}

static int cmd_inf(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	if (argc == 2) {
		if (match_inf(match, "any", argv[1]) == 0)
			goto done;
	}
/* help */
	cmd_inf_help("");
done:
	return 0;
}

static int match_mac(struct flowmgr_match *match, char *type, char *addr_str)
{
	struct flowmgr_db_entry *db;
	__u8 mac[6];
	if (mac_pton(addr_str, mac) == 0) {
		pr_err("%s Invalid mac address %s\n", __func__, addr_str);
	} else if (strcasecmp(type, "dst") == 0) {
		db = flowmgr_db_find(mac);
		if (db) {
			match->odb = db;
			match->mask &= ~(FLOW_M_ANY_MAC);
			match->mask |= FLOW_M_DST_MAC;
			return 0;
		}
		pr_err("%s mac address not found in db %s\n", __func__, addr_str);
	} else if (strcasecmp(type, "src") == 0) {
		db = flowmgr_db_find(mac);
		if (db) {
			match->mask &= ~(FLOW_M_ANY_MAC);
			match->mask |= FLOW_M_SRC_MAC;
			match->idb = db;
			return 0;
		}
		pr_err("%s mac address not found in db %s\n", __func__, addr_str);
	} else if (strcasecmp(type, "any") == 0) {
		db = flowmgr_db_find(mac);
		if (db) {
			match->mask &= ~(FLOW_M_DST_MAC | FLOW_M_SRC_MAC);
			match->mask |= FLOW_M_ANY_MAC;
			match->idb = db;
			return 0;
		}
		pr_err("%s mac address not found in db %s\n", __func__, addr_str);
	}
	return -1;
}

static void cmd_mac_help(char *str)
{
	pr_alert("%s mac: Match mac address\n", str);
	pr_alert("%s  mac <mac>\n", str);
	pr_alert("%s   mac - valid mac address or any\n", str);
}

static int cmd_mac(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	if (argc == 2) {
		if (match_mac(match, "any", argv[1]) == 0)
			goto done;
	}
/* help */
	cmd_mac_help("");
done:
	return 0;
}

static int match_host(struct flowmgr_match *match, char *type, char *ip_str)
{
	__be32 ip;
	__be32 ip6[4];
	if (in4_pton(ip_str, strlen(ip_str),
		    (u8 *)&ip,
		     -1, NULL) != 0) {
		if (strcasecmp(type, "dst") == 0) {
			match->tuple.dst.u3.ip = ip;
			match->mask &= ~(FLOW_M_ANY_IP);
			match->mask |= FLOW_M_DST_IP;
			if (match->tuple.src.l3num == PF_INET6) {
				match->tuple.src.u3.ip = 0;
				match->mask &= ~(FLOW_M_SRC_IP);
				pr_alert("Setting and Clearing SRC IP to IPV4\n");
			}
			match->tuple.src.l3num = PF_INET;
			return 0;
		} else if (strcasecmp(type, "src") == 0) {
			match->tuple.src.u3.ip = ip;
			match->mask &= ~(FLOW_M_ANY_IP);
			match->mask |= FLOW_M_SRC_IP;
			match->tuple.src.l3num = PF_INET;
			return 0;
		} else if (strcasecmp(type, "any") == 0) {
			match->tuple.src.u3.ip = ip;
			match->mask &= ~(FLOW_M_DST_IP | FLOW_M_SRC_IP);
			match->mask |= FLOW_M_ANY_IP;
			match->tuple.src.l3num = PF_INET;
			return 0;
		}
	} else if (in6_pton(ip_str, strlen(ip_str),
		    (u8 *)ip6,
		     -1, NULL) != 0) {
		if (strcasecmp(type, "dst") == 0) {
			memcpy(match->tuple.dst.u3.ip6, ip6, 16);
			match->mask &= ~(FLOW_M_ANY_IP);
			match->mask |= FLOW_M_DST_IP;
			if (match->tuple.src.l3num == PF_INET) {
				memset(match->tuple.src.u3.ip6, 0, 16);
				match->mask &= ~(FLOW_M_SRC_IP);
				pr_alert("Setting and Clearing SRC IP to IPV6\n");
			}
			match->tuple.src.l3num = PF_INET6;
			return 0;
		} else if (strcasecmp(type, "src") == 0) {
			memcpy(match->tuple.src.u3.ip6, ip6, 16);
			match->mask &= ~(FLOW_M_ANY_IP);
			match->mask |= FLOW_M_SRC_IP;
			match->tuple.src.l3num = PF_INET6;
			return 0;
		} else if (strcasecmp(type, "any") == 0) {
			memcpy(match->tuple.src.u3.ip6, ip6, 16);
			match->mask &= ~(FLOW_M_DST_IP | FLOW_M_SRC_IP);
			match->mask |= FLOW_M_ANY_IP;
			match->tuple.src.l3num = PF_INET6;
			return 0;
		}
	} else {
		pr_err("%s Invalid ip address %s\n", __func__, ip_str);
	}
	return -1;
}

static void cmd_host_help(char *str)
{
	pr_alert("%s host: Match host ip address\n", str);
	pr_alert("%s  host <ip>\n", str);
	pr_alert("%s   ip - valid ip address or any\n", str);
}

static int cmd_host(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	if (argc == 2) {
		if (match_host(match, "any", argv[1]) == 0)
			goto done;
	}
/* help */
	cmd_host_help("");
done:
	return 0;
}

static u8 ip_proto_from_string(const char *str)
{
	u8 prot = 0;
	if (kstrtou8(str, 0, &prot) != 0) {
		if (strcasecmp(str, "udp") == 0)
			return IPPROTO_UDP;
		else if (strcasecmp(str, "tcp") == 0)
			return IPPROTO_TCP;
		else if (strcasecmp(str, "esp") == 0)
			return IPPROTO_ESP;
		else if (strcasecmp(str, "ah") == 0)
			return IPPROTO_AH;
	}
	if (prot == IPPROTO_UDP)
		return prot;
	else if (prot == IPPROTO_TCP)
		return prot;
	else if (prot == IPPROTO_ESP)
		return prot;
	else if (prot == IPPROTO_AH)
		return prot;
	else
		return 0;
}

static void cmd_proto_help(char *str)
{
	pr_alert("%s proto: Match protocol name or num\n", str);
	pr_alert("%s  proto <proto>\n", str);
	pr_alert("%s   proto - tcp|udp|esp|ah|protonum\n", str);
}

static int cmd_proto(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	if (argc == 2) {
		u_int8_t protonum;
		protonum = ip_proto_from_string(argv[1]);
		match->tuple.dst.protonum = protonum;
		match->mask |= FLOW_M_IP_PROTO;
		goto done;
	}
/* help */
	cmd_proto_help("");
done:
	return 0;
}

static void cmd_tcp_help(char *str)
{
	pr_alert("%s tcp: Match TCP proto\n", str);
}

static int cmd_tcp(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	match->tuple.dst.protonum = IPPROTO_TCP;
	match->mask |= FLOW_M_IP_PROTO;
	return 0;
}

static void cmd_udp_help(char *str)
{
	pr_alert("%s udp: Match UDP proto\n", str);
}

static int cmd_udp(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	match->tuple.dst.protonum = IPPROTO_UDP;
	match->mask |= FLOW_M_IP_PROTO;
	return 0;
}

static int match_port(struct flowmgr_match *match, char *type, char *port_str)
{
	__be16 port;
	if (kstrtou16(port_str, 0, &port) != 0) {
		pr_err("%s Invalid Port %s\n", __func__, port_str);
	} else if (strcasecmp(type, "dst") == 0) {
		match->tuple.dst.u.tcp.port = htons(port);
		match->mask &= ~(FLOW_M_ANY_PORT);
		match->mask |= FLOW_M_DST_PORT;
		return 0;
	} else if (strcasecmp(type, "src") == 0) {
		match->tuple.src.u.tcp.port = htons(port);
		match->mask &= ~(FLOW_M_ANY_PORT);
		match->mask |= FLOW_M_SRC_PORT;
		return 0;
	} else if (strcasecmp(type, "any") == 0) {
		match->tuple.src.u.tcp.port = htons(port);
		match->mask &= ~(FLOW_M_DST_PORT | FLOW_M_SRC_PORT);
		match->mask |= FLOW_M_ANY_PORT;
		return 0;
	}
	return -1;
}

static void cmd_port_help(char *str)
{
	pr_alert("%s port: Match port number\n", str);
	pr_alert("%s  port <num>\n", str);
	pr_alert("%s   num - valid port number\n", str);
}

static int cmd_port(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	if (argc == 2) {
		if (match_port(match, "any", argv[1]) == 0)
			goto done;
	}
/* help */
	cmd_port_help("");
done:
	return 0;
}

static void cmd_clear_help(char *str)
{
	pr_alert("%s clear: Clear all matching rules\n", str);
}

static int cmd_clear(void *data, int argc, char *argv[])
{
	int enable;
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	enable = match->enable;
	memset(match, 0, sizeof(struct flowmgr_match));
	match->enable = enable;
	return 0;
}

static void cmd_src_help(char *str)
{
	pr_alert("%s src: Match src inf, mac, host, or port\n", str);
	pr_alert("%s  src <inf|mac|net|port> <value>\n", str);
}

static int cmd_src(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	if (argc == 3) {
		if (strcasecmp(argv[1], "inf") == 0) {
			if (match_inf(match, "src", argv[2]) == 0)
				goto done;
		} else if (strcasecmp(argv[1], "mac") == 0) {
			if (match_mac(match, "src", argv[2]) == 0)
				goto done;
		} else if (strcasecmp(argv[1], "host") == 0) {
			if (match_host(match, "src", argv[2]) == 0)
				goto done;
		} else if (strcasecmp(argv[1], "port") == 0) {
			if (match_port(match, "src", argv[2]) == 0)
				goto done;
		}
	}
/* help */
	cmd_src_help("");
done:
	return 0;
}

static void cmd_dst_help(char *str)
{
	pr_alert("%s dst: Match dst inf, mac, host, or port\n", str);
	pr_alert("%s  dst <inf|mac|net|port> <value>\n", str);
}

static int cmd_dst(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	if (argc == 3) {
		if (strcasecmp(argv[1], "inf") == 0) {
			if (match_inf(match, "dst", argv[2]) == 0)
				goto done;
		} else if (strcasecmp(argv[1], "mac") == 0) {
			if (match_mac(match, "dst", argv[2]) == 0)
				goto done;
		} else if (strcasecmp(argv[1], "host") == 0) {
			if (match_host(match, "dst", argv[2]) == 0)
				goto done;
		} else if (strcasecmp(argv[1], "port") == 0) {
			if (match_port(match, "dst", argv[2]) == 0)
				goto done;
		}
	}
/* help */
	cmd_dst_help("");
done:
	return 0;
}

static void cmd_enable_help(char *str)
{
	pr_alert("%s 1: Enable match\n", str);
}

static int cmd_enable(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	match->enable = 1;
	flowmgr_match_show(NULL, match);
	return 0;
}

static void cmd_disable_help(char *str)
{
	pr_alert("%s 1: Disable match\n", str);
}

static int cmd_disable(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	match->enable = 0;
	return 0;
}

static void cmd_mapt_help(char *str)
{
	pr_alert("%s matp: Match mapt flows\n", str);
}

static int cmd_mapt(void *data, int argc, char *argv[])
{
	struct flowmgr_match *match;
	match = (struct flowmgr_match *) data;
	match->mask |= FLOW_M_MAPT;
	return 0;
}

static struct proc_cmd_ops match_command_entries[] = {
	PROC_CMD_DATA_INIT("inf",   cmd_inf),
	PROC_CMD_DATA_INIT("mac",   cmd_mac),
	PROC_CMD_DATA_INIT("host",  cmd_host),
	PROC_CMD_DATA_INIT("proto", cmd_proto),
	PROC_CMD_DATA_INIT("tcp",   cmd_tcp),
	PROC_CMD_DATA_INIT("udp",   cmd_udp),
	PROC_CMD_DATA_INIT("port",  cmd_port),
	PROC_CMD_DATA_INIT("dst",   cmd_dst),
	PROC_CMD_DATA_INIT("src",   cmd_src),
	PROC_CMD_DATA_INIT("clear", cmd_clear),
	PROC_CMD_DATA_INIT("1",     cmd_enable),
	PROC_CMD_DATA_INIT("0",     cmd_disable),
	PROC_CMD_DATA_INIT("mapt",  cmd_mapt),
};

static struct flowmgr_match ctdebug;

struct proc_cmd_table ctdebug_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(match_command_entries),
	.data = &ctdebug,
	.data_seq_read = (void *) &flowmgr_match_seq_ops,
	.ops = match_command_entries
};

static struct flowmgr_match cttrace;

struct proc_cmd_table cttrace_command_table = {
	.module_name = "FLOWMGR",
	.size = ARRAY_SIZE(match_command_entries),
	.data = &cttrace,
	.data_seq_read = (void *) &flowmgr_match_seq_ops,
	.ops = match_command_entries
};

int flowmgr_match_check(struct flowmgr_match *match,
			struct nf_conntrack_tuple *ct_tuple,
			struct offload_info *ct_offload_info)
{
	struct nf_conntrack_tuple *match_tuple;
	int checks = 0;
	int checks_match = 0;
	match_tuple = &match->tuple;
	if ((match->mask & FLOW_M_ANY_INF) && match->in) {
		checks++;
		if (ct_offload_info->iif == match->in->ifindex)
			checks_match++;
		else if (ct_offload_info->oif == match->in->ifindex)
			checks_match++;
		else
			goto done;
	}
	if ((match->mask & FLOW_M_DST_INF) && match->out) {
		checks++;
		if (ct_offload_info->oif == match->out->ifindex)
			checks_match++;
		else
			goto done;
	}
	if ((match->mask & FLOW_M_SRC_INF) && match->in) {
		checks++;
		if (ct_offload_info->iif == match->in->ifindex)
			checks_match++;
		else
			goto done;
	}
	if ((match->mask & FLOW_M_ANY_MAC) && match->idb) {
		checks++;
		if (ct_offload_info->idb == match->idb)
			checks_match++;
		else if (ct_offload_info->odb == match->idb)
			checks_match++;
		else
			goto done;
	}
	if ((match->mask & FLOW_M_DST_MAC) && match->odb) {
		checks++;
		if (ct_offload_info->odb == match->odb)
			checks_match++;
		else
			goto done;
	}
	if ((match->mask & FLOW_M_SRC_MAC) && match->idb) {
		checks++;
		if (ct_offload_info->idb == match->idb)
			checks_match++;
		else
			goto done;
	}
	if (match->mask & FLOW_M_ANY_IP) {
		checks++;
		if (match_tuple->src.l3num == PF_INET6) {
			if (ct_tuple->src.l3num == PF_INET)
				goto done;
			if (memcmp(match_tuple->src.u3.ip6,
			       ct_tuple->src.u3.ip6, 16) == 0)
				checks_match++;
			else if (memcmp(match_tuple->src.u3.ip6,
			       ct_tuple->dst.u3.ip6, 16) == 0)
				checks_match++;
			else
				goto done;
		} else if (ct_tuple->src.l3num == PF_INET) {
			if (match_tuple->src.u3.ip ==
			   ct_tuple->src.u3.ip)
				checks_match++;
			else if (match_tuple->src.u3.ip ==
				 ct_tuple->dst.u3.ip)
				 checks_match++;
		} else
			goto done;
	}
	if (match->mask & FLOW_M_DST_IP) {
		checks++;
		if (match_tuple->src.l3num == PF_INET6) {
			if (ct_tuple->src.l3num == PF_INET)
				goto done;
			if (memcmp(match_tuple->dst.u3.ip6,
			       ct_tuple->dst.u3.ip6, 16) == 0)
				checks_match++;
			else
				goto done;
		} else if (ct_tuple->src.l3num == PF_INET) {
			if (match_tuple->dst.u3.ip ==
			    ct_tuple->dst.u3.ip)
				checks_match++;
		} else
			goto done;
	}
	if (match->mask & FLOW_M_SRC_IP) {
		checks++;
		if (match_tuple->src.l3num == PF_INET6) {
			if (ct_tuple->src.l3num == PF_INET)
				goto done;
			if (memcmp(match_tuple->src.u3.ip6,
			       ct_tuple->src.u3.ip6, 16) == 0)
				checks_match++;
			else
				goto done;
		} else if (ct_tuple->src.l3num == PF_INET) {
			if (match_tuple->src.u3.ip ==
			    ct_tuple->src.u3.ip)
				checks_match++;
		} else
			goto done;
	}
	if (match->mask & FLOW_M_IP_PROTO) {
		checks++;
		if (match_tuple->dst.protonum ==
		    ct_tuple->dst.protonum)
			checks_match++;
		else
			goto done;
	}
	if (match->mask & FLOW_M_DST_PORT) {
		checks++;
		if (match_tuple->dst.u.tcp.port ==
		    ct_tuple->dst.u.tcp.port)
			checks_match++;
		else
			goto done;
	}
	if (match->mask & FLOW_M_ANY_PORT) {
		checks++;
		if (match_tuple->src.u.tcp.port ==
		    ct_tuple->src.u.tcp.port)
			checks_match++;
		else if (match_tuple->src.u.tcp.port ==
			    ct_tuple->dst.u.tcp.port)
				checks_match++;
		else
			goto done;
	}
	if (match->mask & FLOW_M_SRC_PORT) {
		checks++;
		if (match_tuple->src.u.tcp.port ==
		    ct_tuple->src.u.tcp.port)
			checks_match++;
		else
			goto done;
	}
	net_dbg_ratelimited("FLOWMGR CTDGB: #checks=%d #match=%d\n",
		 checks, checks_match);
	if (checks == checks_match)
		return 1;
done:
	return 0;
}

int flowmgr_ctdebug_check(struct nf_conn *ct,
			  void *offload,
			  int dir)
{
	struct offload_info *ct_offload_info;
	struct flowmgr_match *match = &ctdebug;
	struct nf_conntrack_tuple *match_tuple;
	struct nf_conntrack_tuple *ct_tuple;
	struct nf_conn_offload *ct_offload;

	if (!offload || !ct)
		return 0;
	/* If ctdebug is not enabled then return */
	if (!match->enable)
		return 0;
	ct_offload = (struct nf_conn_offload *) offload;
	ct_offload_info = &ct_offload->info[dir];
	/* If debug is initialized for conntrack then return */
	if (ct_offload_info->debug)
		return 0;
	/* If ctdebug is enabled but no mask */
	if (!match->mask) {
		ct_offload_orig.debug = 1;
		ct_offload_repl.debug = 1;
		return 0;
	}
	/* Match mapt */
	if (ct_offload_info->map) {
		if (match->mask & FLOW_M_MAPT) {
			ct_offload_info->debug = 1;
			return 0;
		}
	}
	/* Match */
	match_tuple = &match->tuple;
	ct_tuple = nf_ct_tuple(ct, dir);
	nf_ct_tuple_dump(net_dbg_ratelimited, ct_tuple, "FLOWMGR CTDGB");
	nf_ct_offload_dump(net_dbg_ratelimited, ct_offload_info, "FLOWMGR CTDGB");
	if (flowmgr_match_check(match, ct_tuple, ct_offload_info))
		ct_offload_info->debug = 1;
	return 0;
}

int flowmgr_cttrace_check(void *tuple, void *info)
{
	struct nf_conntrack_tuple *ct_tuple;
	struct offload_info *ct_offload_info;
	struct flowmgr_match *match = &cttrace;
	struct nf_conntrack_tuple *match_tuple;
	if (!tuple || !info)
		return 0;
	/* If cttrace is not enabled then return 1 */
	if (!match->enable)
		return 1;
	ct_tuple = (struct nf_conntrack_tuple *) tuple;
	ct_offload_info = (struct offload_info *) info;
	/* If cttrace is enabled but no mask */
	if (!match->mask) {
		return 1;
	}
	/* Match */
	match_tuple = &match->tuple;
	if (flowmgr_match_check(match, ct_tuple, ct_offload_info))
		return 1;
	return 0;
}

int flowmgr_debug_init(void)
{
	proc_create_cmd("ctdebug", flowmgr.proc_dir, &ctdebug_command_table);
	proc_create_cmd("cttrace", flowmgr.proc_dir, &cttrace_command_table);
	return 0;
}

void flowmgr_debug_exit(void)
{
	if (flowmgr.proc_dir) {
		remove_proc_entry("ctdebug", flowmgr.proc_dir);
		remove_proc_entry("cttrace", flowmgr.proc_dir);
	}
}
