 /****************************************************************************
 *
 * Copyright (c) 2020 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Broadcom BCM9339x Gigabit Ethernet internal GMII transceiver.
 *
 * Author: Ravi Patel <ravi.patel@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/phy.h>
#include <linux/delay.h>
#include <linux/brcmphy.h>
#include "bcm-phy-lib.h"

#define BCM9339X_PHYS_MAX 2

struct bcm9339x_phys {
	struct phy_device *phydev[BCM9339X_PHYS_MAX];
	u32 phyad;
	u32 probe;
	u32 power;
	u32 alloc;
};

static int bcm9339x_probe(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct bcm9339x_phys *phys = NULL;
	struct phy_device *phy;
	u32 i, addr;

	addr = phydev->mdio.addr & ~(BCM9339X_PHYS_MAX - 1);
	for (i = addr; i < addr + BCM9339X_PHYS_MAX; i++) {
		phy = mdiobus_get_phy(phydev->mdio.bus, i);
		if (!phy)
			continue;

		phys = phy->priv;
		if (phys)
			break;
	}

	if (!phys) {
		phys = kzalloc(sizeof(*phys), GFP_KERNEL);
		if (!phys)
			return -ENOMEM;

		phys->phyad = addr;
	}

	i = phydev->mdio.addr - phys->phyad;
	phys->probe |= (1 << i);
	phys->phydev[i] = phydev;
	phydev->priv = phys;
	dev_info(dev, "probed\n");

	return 0;
}

static int bcm9339x_config_init(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct bcm9339x_phys *phys = phydev->priv;
	int i, val, addr, cnt = phys->probe;
	int rcal_code_lp, rcal_code_11, cal_config2;
	int rcal_code_11d2, txcfgch0, txcfg2;

	if (phys->power)
		goto _advertise_modes;

	if (!phys->phydev[0]) {
		phys->phydev[0] = kzalloc(sizeof(*phydev), GFP_KERNEL);
		if (!phys->phydev[0])
			return -ENOMEM;

		phys->phydev[0]->mdio.bus = phydev->mdio.bus;
		phys->phydev[0]->mdio.addr = phys->phyad;
		phys->alloc = 1;
	}

	addr = phys->phydev[0]->mdio.addr;

	/* Put all PHYs in Power Down mode */
	for (i = 0; i < 2; i++)
		phy_write(phys->phydev[i], 0x00, 0x1940);

	/* Reset PHY to default state */
	for (i = 0; i < 2; i++) {
		if (!(cnt & (1 << i)))
			continue;

		phy_write(phys->phydev[i], 0x00, 0x9140);
	}

	/* Reset AFE and PLL */
	for (i = 0; i < 2; i++) {
		if (!(cnt & (1 << i)))
			continue;

		bcm_phy_write_exp(phys->phydev[i], 0x0003, 0x0006);
		bcm_phy_write_exp(phys->phydev[i], 0x0003, 0x0000);
	}

	/* Write PLL/AFE control registers */
	/* Config PLL (54MHz xtal) for all PHYs! */
	/* Reset PLL */
	bcm_phy_write_misc(phys->phydev[0], 0x0030, 0x0001, 0x0000);
	/* PLL_NDIV_2 ndiv_integer */
	bcm_phy_write_misc(phys->phydev[0], 0x0031, 0x0000, 0x044a);
	/* PLL_MANUAL_CONFIG pdiv=1 */
	bcm_phy_write_misc(phys->phydev[0], 0x0033, 0x0002, 0x0001);
	/* PLL_NDIV ndiv_fraction */
	bcm_phy_write_misc(phys->phydev[0], 0x0031, 0x0001, 0x2F68);
	/* PLL_NUDGE frequency nudge factor */
	bcm_phy_write_misc(phys->phydev[0], 0x0031, 0x0002, 0x0000);
	/* PLLREF_REFERENCE reference frequency=54MHz, Mode 0 */
	bcm_phy_write_misc(phys->phydev[0], 0x0030, 0x0003, 0x0036);
	/* PLL_CONTROLS_2 Init bypass mode - default setting */
	bcm_phy_write_misc(phys->phydev[0], 0x0032, 0x0003, 0x0000);
	/* PLL_CONTROLS Bypass code - Default, vcoclk enabled */
	bcm_phy_write_misc(phys->phydev[0], 0x0033, 0x0000, 0x0002);
	/* PLL_LDO_CONTROL LDOs at default setting */
	bcm_phy_write_misc(phys->phydev[0], 0x0030, 0x0002, 0x01C0);
	/* 'PLL_LDO_CONTROL Release PLL reset */
	bcm_phy_write_misc(phys->phydev[0], 0x0030, 0x0001, 0x0001);

	/* Config BIAS for all PHYs! */
	/* AFE_BG_CONFIG Bandgap curvature correction to correct default -- Erol */
	bcm_phy_write_misc(phys->phydev[0], 0x0038, 0x0000, 0x0010);

	/* Run RCAL */
	/* AFE_CAL_CONFIG_2, enable max averaging (B0) */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0003, 0x0038);
	/* AFE_CAL_CONFIG_2, no reset, analog powerup */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0003, 0x003B);
	/* 2us wait */
	udelay(2);
	/* AFE_CAL_CONFIG_2, start cal */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0003, 0x003F);
	/* 5ms - wait if you want to be sure it is finished before start testing */
	mdelay(5);

	/* Run RCCAL VREF = 1000 TARGET = 10 */
	/* AFE_CAL_CONFIG_0, Vref=1000, Target=10, averaging enabled */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0001, 0x1C82);
	/* AFE_CAL_CONFIG_0, no reset and analog powerup */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0001, 0x9E82);
	/* 2us wait */
	udelay(2);
	/* AFE_CAL_CONFIG_0, start calibration */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0001, 0x9F82);
	/* 100us wait */
	udelay(100);
	/* AFE_CAL_CONFIG_0, clear start calibration, set HiBW */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0001, 0x9E86);
	/* 2us wait */
	udelay(2);
	/* AFE_CAL_CONFIG_0, start calibration with hi BW mode set */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0001, 0x9F86);
	/* 100us - wait if you want to be sure it is finished before start testing */
	udelay(100);

	/* SELECT EXTERNAL RESISTOR - COMMENTED out! */
	/* AFE_CAL_CONFIG_2 select Rext */
	/* bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0003, 0x007F); */

	/* TX AMPLITUDE FINETUNE */
	/* AFE_BIAS_CONFIG_0 Adjust 10BT amplitude additional +7% and 100BT +2% */
	bcm_phy_write_misc(phys->phydev[0], 0x0038, 0x0001, 0xE7EA);
	/* AFE_BIAS_CONFIG_1 Adjust 1G mode amplitude and 1G testmode1 */
	bcm_phy_write_misc(phys->phydev[0], 0x0038, 0x0002, 0xEDE0);

	/* Adjust 10BT bias and RCAL settings */
	/* read CORE_EXPA9 */
	val = bcm_phy_read_exp(phys->phydev[0], 0x00A9);
	dev_dbg(dev, "CORE_EXPA9 0x%04x\n", val);
	/* expA9<6:1> is rcalcode<5:0> */
	val = (val & 0x007E) / 2;
	/* correct RCAL code +1 is -1% rprog LP: +16 */
	rcal_code_lp = val + 16;
	dev_dbg(dev, "rcal_code_lp 0x%04x\n", rcal_code_lp);
	/* correct RCAL code +1 is -1% rprog 11: +10 */
	rcal_code_11 = val + 10;
	dev_dbg(dev, "rcal_code_11 0x%04x\n", rcal_code_11);
	/* saturate RCAL code if necessary */
	if (rcal_code_lp > 0x003f) {
		rcal_code_lp = 0x003f;
		dev_dbg(dev, "RCALNEWCODE LP saturated to max value 0x3f\n");
	}

	/* saturate RCAL code if necessary */
	if (rcal_code_11 > 0x003f) {
		rcal_code_11 = 0x003f;
		dev_dbg(dev, "RCALNEWCODE 1111 saturated to max value 0x3f\n");
	}

	/* AFE_BIAS_CONFIG_0 */
	/* REXT=1 BYP=1 RCAL_st1<5:0>=new rcal code */
	cal_config2 = 0x00F8 + (rcal_code_lp << 8);
	/* AFE_CAL_CONFIG_2 */
	bcm_phy_write_misc(phys->phydev[0], 0x0039, 0x0003, cal_config2);
	/* AFE_BIAS_CONFIG_0 10BT bias code Bias: EA */
	bcm_phy_write_misc(phys->phydev[0], 0x0038, 0x0001, 0xE7EA);

	for (i = 0; i < 2; i++) {
		if (!(cnt & (1 << i)))
			continue;

		/* Config individual AFE */
		/* AFE_RXCONFIG_3 invert adc clock output and 'adc refp ldo current To correct default */
		bcm_phy_write_misc(phys->phydev[i], 0x003B, 0x0000, 0x8002);

		/* AFE_TX_CONFIG_1 100BT stair case, high BW, 1G stair case, alternate encode */
		bcm_phy_write_misc(phys->phydev[i], 0x003C, 0x0003, 0xF882);

		/* AFE_TX_CONFIG_2 1000BT DAC transition method per Erol, bits[32], DAC Shuffle sequence 1 + 10BT imp adjust bits */
		bcm_phy_write_misc(phys->phydev[i], 0x003D, 0x0000, 0x3201);

		/* AFE_RXCONFIG_1 some non-overlap fix per Erol */
		bcm_phy_write_misc(phys->phydev[i], 0x003A, 0x0002, 0x0C00);

		/* RX Full scale calibration */
		/* pwdb override (rxconfig<5>) to turn on RX LDO indpendent of pwdb controls from DSP */
		/* AFE_RXCONFIG_0 */
		bcm_phy_write_misc(phys->phydev[i], 0x003A, 0x0001, 0x0020);

		/* Specify target offset for each channel */
		/* AFE_RXCONFIG_CH0 */
		bcm_phy_write_misc(phys->phydev[i], 0x003B, 0x0002, 0x0000);
		/* AFE_RXCONFIG_CH1 */
		bcm_phy_write_misc(phys->phydev[i], 0x003B, 0x0003, 0x0000);
		/* AFE_RXCONFIG_CH2 */
		bcm_phy_write_misc(phys->phydev[i], 0x003C, 0x0000, 0x0000);
		/* AFE_RXCONFIG_CH3 */
		bcm_phy_write_misc(phys->phydev[i], 0x003C, 0x0001, 0x0000);

		/* Set cal_bypassb bit rxconfig<43> */
		/* AFE_RXCONFIG_2 */
		bcm_phy_write_misc(phys->phydev[i], 0x003A, 0x0003, 0x0800);

		/* At least 2us delay needed  before this line is executed. */
		udelay(2);

		/* Revert pwdb_override (rxconfig<5>) to 0 so that the RX pwr is controlled by DSP. */
		/* AFE_RXCONFIG_0 */
		bcm_phy_write_misc(phys->phydev[i], 0x003A, 0x0001, 0x0000);

		/* Adjust 10BT bias and RCAL settings */
		/* AFE_TX_CONFIG_2 */
		/* Drop LSB */
		rcal_code_11d2 = (rcal_code_11 & 0xFFFE) / 2;
		/* read AFE_TX_CONFIG_CH0 */
		txcfgch0 = bcm_phy_read_misc(phys->phydev[i], 0x003D, 0x0001);
		/* clear bits <11:5> */
		txcfgch0 = (txcfgch0 & ~0x0FE0);
		/* set txcfg_ch0<5>=1 (enable + set local rcal) */
		txcfgch0 = (txcfgch0 | 0x0020) | (rcal_code_11d2 << 6);
		/* write AFE_TX_CONFIG_CH0 */
		bcm_phy_write_misc(phys->phydev[i], 0x003D, 0x0001, txcfgch0);
		/* AFE_TX_CONFIG_CH0 */
		/* read AFE_TX_CONFIG_2 */
		txcfg2 = bcm_phy_read_misc(phys->phydev[i], 0x003D, 0x0000);
		/* set txcfg<45:44>=11 (enable Rextra + invert fullscaledetect) */
		txcfg2 = (txcfg2 & ~0x3000) | 0x3000;
		/* write AFE_TX_CONFIG_2 */
		bcm_phy_write_misc(phys->phydev[i], 0x003D, 0x0000, txcfg2);
	}

	/* read AFE_BG_CONFIG */
	val = bcm_phy_read_misc(phys->phydev[0], 0x0038, 0x0000);
	dev_dbg(dev, "PHY 0x%02x AFE_BG_CONFIG 0x%04x\n", addr, val);

	/* read AFE_BIAS_CONFIG_0 */
	val = bcm_phy_read_misc(phys->phydev[0], 0x0038, 0x0001);
	dev_dbg(dev, "PHY 0x%02x AFE_BIAS_CONFIG_0 0x%04x\n", addr, val);

	/* read AFE_BIAS_CONFIG_1 */
	val = bcm_phy_read_misc(phys->phydev[0], 0x0038, 0x0002);
	dev_dbg(dev, "PHY 0x%02x AFE_BIAS_CONFIG_1 0x%04x\n", addr, val);

	/* read AFE_BIAS_CONFIG_2 */
	val = bcm_phy_read_misc(phys->phydev[0], 0x0038, 0x0003);
	dev_dbg(dev, "PHY 0x%02x AFE_BIAS_CONFIG_2 0x%04x\n", addr, val);

	/* read AFE_BIAS_CONFIG_3 */
	val = bcm_phy_read_misc(phys->phydev[0], 0x0039, 0x0000);
	dev_dbg(dev, "PHY 0x%02x AFE_BIAS_CONFIG_3 0x%04x\n", addr, val);

	/* read AFE_CAL_CONFIG_0 */
	val = bcm_phy_read_misc(phys->phydev[0], 0x0039, 0x0001);
	dev_dbg(dev, "PHY 0x%02x AFE_CAL_CONFIG_0 0x%04x\n", addr, val);

	/* read AFE_CAL_CONFIG_1 */
	val = bcm_phy_read_misc(phys->phydev[0], 0x0039, 0x0002);
	dev_dbg(dev, "PHY 0x%02x AFE_CAL_CONFIG_1 0x%04x\n", addr, val);

	/* read AFE_CAL_CONFIG_2 */
	val = bcm_phy_read_misc(phys->phydev[0], 0x0039, 0x0003);
	dev_dbg(dev, "PHY 0x%02x AFE_CAL_CONFIG_2 0x%04x\n", addr, val);

	/* read AFE_CAL_CONFIG_3 */
	val = bcm_phy_read_misc(phys->phydev[0], 0x003A, 0x0000);
	dev_dbg(dev, "PHY 0x%02x AFE_CAL_CONFIG_3 0x%04x\n", addr, val);

	for (i = 0; i < 2; i++) {
		if (!(cnt & (1 << i)))
			continue;

		/* read PLL_status */
		val = bcm_phy_read_misc(phys->phydev[i], 0x0034, 0x0001);
		dev_dbg(dev, "PHY 0x%02x PLL Status 0x%04x\n", addr, val);

		/* read AFE_RXCONFIG_0 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003A, 0x0001);
		dev_dbg(dev, "PHY 0x%02x AFE_RXCONFIG_0 0x%04x\n", addr, val);

		/* read AFE_RXCONFIG_1 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003A, 0x0002);
		dev_dbg(dev, "PHY 0x%02x AFE_RXCONFIG_1 0x%04x\n", addr, val);

		/* read AFE_RXCONFIG_2 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003A, 0x0003);
		dev_dbg(dev, "PHY 0x%02x AFE_RXCONFIG_2 0x%04x\n", addr, val);

		/* read AFE_RXCONFIG_3 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003B, 0x0000);
		dev_dbg(dev, "PHY 0x%02x AFE_RXCONFIG_3 0x%04x\n", addr, val);

		/* read AFE_RXCONFIG_CH0 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003B, 0x0002);
		dev_dbg(dev, "PHY 0x%02x AFE_RXCONFIG_CH0 0x%04x\n", addr, val);

		/* read AFE_RXCONFIG_CH1 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003B, 0x0003);
		dev_dbg(dev, "PHY 0x%02x AFE_RXCONFIG_CH1 0x%04x\n", addr, val);

		/* read AFE_RXCONFIG_CH2 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003C, 0x0000);
		dev_dbg(dev, "PHY 0x%02x AFE_RXCONFIG_CH2 0x%04x\n", addr, val);

		/* read AFE_RXCONFIG_CH3 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003C, 0x0001);
		dev_dbg(dev, "PHY 0x%02x AFE_RXCONFIG_CH3 0x%04x\n", addr, val);

		/* read AFE_TX_CONFIG_1 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003C, 0x0003);
		dev_dbg(dev, "PHY 0x%02x AFE_TX_CONFIG_1 0x%04x\n", addr, val);

		/* read AFE_TX_CONFIG_2 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003D, 0x0000);
		dev_dbg(dev, "PHY 0x%02x AFE_TX_CONFIG_2 0x%04x\n", addr, val);

		/* read AFE_TX_CONFIG_CH0 */
		val = bcm_phy_read_misc(phys->phydev[i], 0x003D, 0x0001);
		dev_dbg(dev, "PHY 0x%02x AFE_TX_CONFIG_CH0 0x%04x\n", addr, val);
		addr++;
	}

	phys->power = phys->probe;

_advertise_modes:
	/* advertise 10/100 */
	phy_write(phydev, MII_ADVERTISE, ADVERTISE_ALL|ADVERTISE_CSMA);
	/* advertise 1000 */
	phy_write(phydev, MII_CTRL1000, ADVERTISE_1000FULL|ADVERTISE_1000HALF);
	/* advertise autoneg */
	phy_write(phydev, MII_BMCR, BMCR_ANENABLE|BMCR_ANRESTART|BMCR_FULLDPLX|BMCR_SPEED1000);

	return 0;
}

static void bcm9339x_remove(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct bcm9339x_phys *phys = phydev->priv;

	if (!phys)
		return;

	if (dev->type->pm)
		dev->type->pm->suspend(dev);

	phys->power &= ~(1 << (phydev->mdio.addr - phys->phyad));
	if (!phys->power) {
		if (phys->alloc) {
			memset(phys->phydev[0], 0, sizeof(struct phy_device));
			kfree(phys->phydev[0]);
		}

		memset(phys, 0, sizeof(struct bcm9339x_phys));
		kfree(phys);
	}

	phydev->priv = NULL;
	dev_info(dev, "removed\n");
}

static void bcm9339x_shutdown(struct device *dev)
{
	struct phy_device *phydev = to_phy_device(dev);

	bcm9339x_remove(phydev);
	dev_info(dev, "shutdown\n");
}

#define BCM9339X_GPHY(_oui, _name)		\
{						\
	.phy_id		= (_oui),		\
	.name		= _name,		\
	.phy_id_mask	= 0xfffffff0,		\
	/* .features       = PHY_GBIT_FEATURES |     */       \
	/*                   SUPPORTED_Pause |       */       \
	/*                   SUPPORTED_Asym_Pause,   */       \
	/* .flags          = PHY_HAS_MAGICANEG |     */       \
	/*                   PHY_HAS_INTERRUPT,      */       \
	.config_init	= bcm9339x_config_init,	\
	.probe		= bcm9339x_probe,	\
	.suspend	= genphy_suspend,	\
	.resume		= genphy_resume,	\
	.config_aneg	= genphy_config_aneg,	\
	.read_status	= genphy_read_status,	\
	.remove		= bcm9339x_remove,	\
	.mdiodrv.driver	= {			\
		.owner = THIS_MODULE,		\
		.shutdown = bcm9339x_shutdown,	\
	},					\
}

static struct phy_driver bcm9339x_drivers[] = {
	BCM9339X_GPHY(PHY_ID_BCM9339X, "Broadcom BCM9339X"),
	BCM9339X_GPHY(PHY_ID_BCM93392, "Broadcom BCM93392")
};

module_phy_driver(bcm9339x_drivers);

static struct mdio_device_id __maybe_unused bcm9339x_tbl[] = {
	{ PHY_ID_BCM9339X, 0xfffffff0, },
	{ PHY_ID_BCM93392, 0xfffffff0, },
	{ }
};

MODULE_DEVICE_TABLE(mdio, bcm9339x_tbl);

MODULE_DESCRIPTION("Broadcom BCM9339x PHY driver");
MODULE_AUTHOR("Ravi Patel <ravi.patel@broadcom.com>");
MODULE_LICENSE("GPL");
