 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/export.h>
#include <linux/swab.h>
#include <linux/string.h>
#include <linux/mdio.h>
#include <uapi/linux/if_bridge.h>

#include "ethsw.h"
#include "ethsw_priv.h"
#include "ethsw_core.h"
#include <brcm_mbox.h>

#define CORE_USE_HW_STAT        0

#define CORE_USE_VLAN           1
#define CORE_USE_BRCM_TAG       1
#define CORE_USE_TRUNK_GROUP    1
#define CORE_USE_SINGLE_IMP     1

/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_core"

struct ethsw_driver *ethsw_drv[ETHSW_MAX_SWITCHS];
struct ethsw_device *ethsw_dev[ETHSW_MAX_SWITCHS];
struct ethsw_device * subid_to_swdev_table[SYSTEM_MAX_PORTS + 1];

bool ethsw_vlan_disable = false;
bool hash_fix_enabled = false;
bool reserved_multicast_fix_enabled = false;
#define DEFAULT_MTU_SIZE	1522

/*
 * Ether Switch Core Functions
 */

int ethsw_core_init(struct ethsw_device *swdev)
{
	int ret = 0;
	u8  reg_val8;
	u16 reg_val16;
	int i;
	struct ethsw_core *swcore;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	/* alloc ethsw core control block */
	swcore = kzalloc(sizeof(struct ethsw_core), GFP_KERNEL);
	if (!swcore) {
		LOG_ERR("Ethsw core control block alloc failed\n");
		ret = -ENOMEM;
		goto ERROR;
	}

	swdev->swcore = swcore;

	/* init spinlocks */
	spin_lock_init(&swcore->creg_lock);
	spin_lock_init(&swcore->mib_snapshot_lock);

	/* init states/status */
	swcore->managed = true;
	swcore->cert = false;
	swcore->ethwan = false;
	swcore->ethwan_port = ETHSW_PORT_ID_INVALID;
	swcore->altwan_enble = false;
	swcore->altwan_impport = ETHSW_PORT_ID_INVALID;
	swcore->altwan_lanport = ETHSW_PORT_ID_INVALID;

	/* setup all port except IMP port 8 */
	for (i = 0; i < 8; i++) {
		if (swdev->port[i].type != INV_PORT) {
			/* disable spanning tree for all ports for now */
			reg_val8 =
				CREG_FLD_SHF(CREG_G_PCTL, G_MISTP_STATE, 0);

			CHECK_EXIT(CREG_IDX_WR(
				PAGE_CTLREG,
				CREG_G_PCTL, i,
				&reg_val8));

			/* always use override for MII ports */
			if (swdev->port[i].type == MII_PORT) {
				reg_val8 =
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, SW_OVERRIDE, 1) |
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, TXFLOW_CNTL, 1) |
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, RXFLOW_CNTL, 1) |
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, SPEED,       2) |
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, DUPLX_MODE,  1) |
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, LINK_STS,    1);

				if (swdev->pdev) {
					if (swdev->port[i].mii.speed == 2000) {
						reg_val8 |= CREG_FLD_SHF(
							CREG_STS_OVERRIDE_GMII, SPEED_UP_2G, 1);
					}
				}
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_CTLREG,
					CREG_STS_OVERRIDE_GMII, i,
					&reg_val8));
			}

			/* always use override for IMP ports */
			if (swdev->port[i].type == IMP_PORT) {
				reg_val8 =
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, SW_OVERRIDE, 1) |
					#ifdef CONFIG_BCM7145
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, TXFLOW_CNTL, 1) |
					#else
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, TXFLOW_CNTL, 0) |
					#endif
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, RXFLOW_CNTL, 1) |
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, SPEED,       2) |
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, DUPLX_MODE,  1) |
					CREG_FLD_SHF(CREG_STS_OVERRIDE_GMII, LINK_STS,    1);

				if (swdev->pdev) {
					if (swdev->port[i].imp.speed == 2000) {
						reg_val8 |= CREG_FLD_SHF(
							CREG_STS_OVERRIDE_GMII, SPEED_UP_2G, 1);
					}
				}
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_CTLREG,
					CREG_STS_OVERRIDE_GMII, i,
					&reg_val8));
			}
		}
	}

	/* always use override for IMP port 8 */
	if (swdev->port[8].type == IMP_PORT) {
		reg_val8 =
			CREG_FLD_SHF(CREG_STS_OVERRIDE_IMP, MII_SW_OR,   1) |
			#ifdef CONFIG_BCM7145
			CREG_FLD_SHF(CREG_STS_OVERRIDE_IMP, TXFLOW_CNTL, 1) |
			#else
			CREG_FLD_SHF(CREG_STS_OVERRIDE_IMP, TXFLOW_CNTL, 0) |
			#endif
			CREG_FLD_SHF(CREG_STS_OVERRIDE_IMP, RXFLOW_CNTL, 1) |
			CREG_FLD_SHF(CREG_STS_OVERRIDE_IMP, SPEED,       2) |
			CREG_FLD_SHF(CREG_STS_OVERRIDE_IMP, DUPLX_MODE,  1) |
			CREG_FLD_SHF(CREG_STS_OVERRIDE_IMP, LINK_STS,    1);

		if (swdev->pdev) {
			if (swdev->port[8].imp.speed == 2000) {
				reg_val8 |=
					CREG_FLD_SHF(CREG_STS_OVERRIDE_IMP, GMII_SPEED_UP_2G, 1);
			}
		}
		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_STS_OVERRIDE_IMP,
			&reg_val8));
	}

	if (swdev->sdev) {
		u8 data;
		/* IMP port setup */
		/* If we have both and internal and external switch, we must program */
		/* the internal one to correctly setup the IMP port. */
		if ((swdev == ethsw_dev[EXTERNAL_SPI_SW]) && ethsw_dev[INTERNAL_SW]) {
			data = 0x01;
		}
		else
			data = 0x03;
		CHECK_EXIT(ethsw_creg_write(swdev, 0, 0x60, &data, 1));
		data = 0x8b;
		CHECK_EXIT(ethsw_creg_write(swdev, 0, 0x0e, &data, 1));
	}

	/* set switch mode to managed */
	reg_val8 = 0x04 |
		CREG_FLD_SHF(CREG_SWMODE, SW_FWDG_EN,   1) |
		CREG_FLD_SHF(CREG_SWMODE, SW_FWDG_MODE, 1);

	CHECK_EXIT(CREG_WR(
		PAGE_CTLREG,
		CREG_SWMODE,
		&reg_val8));

	/* Configure the Broadcom header mode */
#if CORE_USE_BRCM_TAG
	reg_val8 = 0;
	if (swdev == ethsw_dev[INTERNAL_SW]) {
		reg_val8 |= swdev->port[7].type == IMP_PORT ? 4 : 0;
		reg_val8 |= swdev->port[5].type == IMP_PORT ? 2 : 0;
		reg_val8 |= swdev->port[8].type == IMP_PORT ? 1 : 0;
	}
#else
	reg_val8 = 0;
#endif
	CHECK_EXIT(CREG_WR(
		PAGE_MNGMODE,
		CREG_BRCM_HDR_CTRL,
		&reg_val8));

	/* Enable BRCM headers for RX and TX port paths */
#if CORE_USE_BRCM_TAG
	if (swdev == ethsw_dev[INTERNAL_SW]) {
		reg_val16 = CREG_BRCM_HDR_RX_DIS_PHY_PORT_MASK;
		if (swdev->port[5].type != IMP_PORT)
			reg_val16 |= (1 << 5);
		if (swdev->port[7].type != IMP_PORT)
			reg_val16 |= (1 << 7);
		if (swdev->port[8].type != IMP_PORT)
			reg_val16 |= (1 << 8);

		CHECK_EXIT(CREG_WR(
			PAGE_MNGMODE,
			CREG_BRCM_HDR_RX_DIS,
			&reg_val16));
		reg_val16 = CREG_BRCM_HDR_TX_DIS_PHY_PORT_MASK;
		CHECK_EXIT(CREG_WR(
			PAGE_MNGMODE,
			CREG_BRCM_HDR_TX_DIS,
			&reg_val16));
	}
#endif

#if CORE_USE_SINGLE_IMP
		/* set IMP port control to enable forwarding */
		reg_val8 =
			CREG_FLD_SHF(CREG_IMP_CTL, RX_UCST_EN, 1) |
			CREG_FLD_SHF(CREG_IMP_CTL, RX_MCST_EN, 1) |
			CREG_FLD_SHF(CREG_IMP_CTL, RX_BCST_EN, 1);

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_IMP_CTL,
			&reg_val8));
#endif

#if CORE_USE_TRUNK_GROUP
	/* Trunk group does NOT work for IMP port in 7145A0. We are reverting
	 * to independent P5/P8 ports, with P8's forwarding flags turned off.
	 */
	/* use trunk group if there are 2 (or) 3 IMP ports */
	if (swdev->imp_port_count == 2 || swdev->imp_port_count == 3) {
		/* enable trunk groups */
		reg_val8 =
			CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, EN_TRUNK_LOCAL, 1) |
			CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, HASH_SEL,       0);

		CHECK_EXIT(CREG_WR(
			PAGE_TRUNK,
			CREG_MAC_TRUNK_CTL,
			&reg_val8));

		/* assign all IMP ports to trunk group 0 */
		reg_val16 = swdev->imp_port_mask;

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_TRUNK,
			CREG_TRUNK_GRP_CTL, 0,
			&reg_val16));

		CHECK_EXIT(CREG_IDX_WR(
	        PAGE_TRUNK,
	        CREG_IMP0_GRP_CTL, 0,
	        &reg_val16));
	}
#endif

    /* disable cross talk among IMP ports */
    for (i = 0; i < ETHSW_PORT_MAX; i++) {
       if (swdev->port[i].type == IMP_PORT) {
          /* exclude all IMP ports as egress ports */
          reg_val16 = 0x1FF ^ swdev->imp_port_mask;
          CHECK_EXIT(CREG_IDX_WR(
             PAGE_VLAN,
             CREG_PORT_VLAN_CTL, i,
             &reg_val16));
       }
    }
   
	/* set global management config for single-IMP mode */
	reg_val8 =
		CREG_FLD_SHF(CREG_GMNGCFG, FRM_MNGP, 2);
	reg_val8 |=
		CREG_FLD_SHF(CREG_GMNGCFG, RXBPDU_EN, 1);

	CHECK_EXIT(CREG_WR(
		PAGE_MNGMODE,
		CREG_GMNGCFG,
		&reg_val8));

	/* set IMP ports to aggregation mode */
	reg_val8 =
		((swdev->port[5].type == IMP_PORT) ?
		 CREG_FLD_SHF(CREG_QOS_GLOBAL_CTRL, P5_AGGREGATION_MODE, 1) : 0) |
		((swdev->port[8].type == IMP_PORT) ?
		 CREG_FLD_SHF(CREG_QOS_GLOBAL_CTRL, P8_AGGREGATION_MODE, 1) : 0);

	CHECK_EXIT(CREG_WR(
		PAGE_QOS,
		CREG_QOS_GLOBAL_CTRL,
		&reg_val8));

	/* turn on learning of unicast src addresses in multicast frames */
	reg_val8 =
		CREG_FLD_SHF(CREG_RSV_MCAST_CTRL, EN_RES_MUL_LEARN, 1) |
		CREG_FLD_SHF(CREG_RSV_MCAST_CTRL, EN_MUL_1, 1); /* default */

	CHECK_EXIT(CREG_WR(
		PAGE_CTLREG,
		CREG_RSV_MCAST_CTRL,
		&reg_val8));
#if CORE_USE_VLAN
#if defined (CONFIG_BRCM_CERTIFICATION)
	/* turn OFF 802.1Q VLAN by default */
	reg_val8 =
		CREG_FLD_SHF(CREG_VLAN_CTRL0, VLAN_EN,             0) |
		CREG_FLD_SHF(CREG_VLAN_CTRL0, VLAN_LEARN_MODE,     3) |
		CREG_FLD_SHF(CREG_VLAN_CTRL0, CHANGE_1P_VID_OUTER, 1) |
		CREG_FLD_SHF(CREG_VLAN_CTRL0, CHANGE_1P_VID_INNER, 1);
#else
	/* turn ON 802.1Q VLAN by default */
	reg_val8 =
		CREG_FLD_SHF(CREG_VLAN_CTRL0, VLAN_EN,
					                swdev->vlan_default_mode) |
		CREG_FLD_SHF(CREG_VLAN_CTRL0, VLAN_LEARN_MODE,     3) |
		CREG_FLD_SHF(CREG_VLAN_CTRL0, CHANGE_1P_VID_OUTER, 1) |
		CREG_FLD_SHF(CREG_VLAN_CTRL0, CHANGE_1P_VID_INNER, 1);
#endif

	CHECK_EXIT(CREG_WR(
		PAGE_BCM8021Q,
		CREG_VLAN_CTRL0,
		&reg_val8));

	/* check reserved multicast frames against VLAN fwd and untag maps */
	reg_val8 = 0x02 |
		CREG_FLD_SHF(CREG_VLAN_CTRL1, EN_RSV_MCAST_UNTAG,  1) |
		CREG_FLD_SHF(CREG_VLAN_CTRL1, EN_RSV_MCAST_FWDMAP, 1);

	CHECK_EXIT(CREG_WR(
		PAGE_BCM8021Q,
		CREG_VLAN_CTRL1,
		&reg_val8));

	/* disable source VLAN membership checking */
	reg_val8 =
		CREG_FLD_SHF(CREG_VLAN_CTRL4, INGR_VID_CHK, 2);

	CHECK_EXIT(CREG_WR(
		PAGE_BCM8021Q,
		CREG_VLAN_CTRL4,
		&reg_val8));

	/* always add VID 1 (default port VID) in VLAN table */
	CHECK_EXIT(ethsw_vlan_table_add(
		swdev,
		1,
		(1 << ETHSW_PORT_MAX) - 1,
		(1 << ETHSW_PORT_MAX) - 1));

#if !CORE_USE_BRCM_TAG
	/* add VID FFx for each non-IMP port in VLAN table */
	for (i = 0; i < 8; i++) {
		if (swdev->port[i].type != IMP_PORT) {
			CHECK_EXIT(ethsw_vlan_table_add(
				swdev,
				0xFF0 + i,
				1 << i,
				1 << i));
		}
	}
#endif
#endif

	/* turn off EEE by default */
	reg_val16 = 0;
	CHECK_EXIT(CREG_WR(
		PAGE_EEE_CTL,
		CREG_EEE_EN_CTRL,
		&reg_val16));

#if defined(CONFIG_BCM_MBOX)
	/* Register the GPHY Powerdown funtion with the MBOX ISR */
	if (swdev == ethsw_dev[INTERNAL_SW])
		swdev->nb.notifier_call  = ethsw_phy_mbox_cb_sf2;
	else
		swdev->nb.notifier_call  = ethsw_phy_mbox_cb_531xx;

	if(brcm_mbox_register_notifier(&swdev->nb)!=0) {
		LOG_ERR("Ethsw core failed to regiter MBOX ISR callback");
		ret = -EINVAL;
		goto ERROR;
	}
#endif

	/* Default MTU to 1500 */
	ethsw_set_mtu(DEFAULT_MTU_SIZE);

	LOG_DEBUG("Ethsw core initialized\n");

 EXIT:
	FUNC_LEAVE();
	return(ret);

 ERROR:
	/* exit core properly */
	ethsw_core_exit(swdev);

	FUNC_LEAVE();
	return(ret);
}

void ethsw_core_exit(struct ethsw_device *swdev)
{
	struct ethsw_core *swcore;

	FUNC_ENTER();
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized\n");
		goto EXIT;
	}

#if defined(CONFIG_BCM_MBOX)
	/* Unegister the GPHY Powerdown funtion with the MBOX ISR */
	brcm_mbox_unregister_notifier(&swdev->nb);
#endif

	swcore = swdev->swcore;
	if (!swcore) {
		goto EXIT;
	}

	/* free ethsw core control block */
	kfree(swcore);
	swdev->swcore = NULL;

	LOG_DEBUG("Ethsw core exited\n");

 EXIT:
	FUNC_LEAVE();
}

/****************************************************************************
  Control Functions
****************************************************************************/

/* switch control functions */
int ethsw_switch_reset(void)
{
	int i, ret = 0;
	u16  reg_val16;
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];


	FUNC_ENTER();
	/* reset the switch chip */

	LOG_INFO("Resetting Switch Chip to defaults\n");
	if (swdev) {
		VALIDATE_ETH_PWR_STATUS(swdev);
		// Set lan ports to their defaults for talking to other ports
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type == PHY_PORT || swdev->port[i].type == MII_PORT) {
				if (!(swdev->phy_port_mask & (1 << i)) &&
						!(swdev->mii_port_mask & (1 << i)))
					continue;
				/* exclude all IMP ports as egress ports */
				reg_val16 = 0x1FF;
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_VLAN,
					CREG_PORT_VLAN_CTL, i,
					&reg_val16));
			}
		}

	}
	swdev = ethsw_dev[EXTERNAL_SPI_SW];
	if (swdev) {
		VALIDATE_ETH_PWR_STATUS(swdev);
		// Set lan ports to their defaults for talking to other ports
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type == PHY_PORT || swdev->port[i].type == MII_PORT) {
				if (!(swdev->phy_port_mask & (1 << i)) &&
						!(swdev->mii_port_mask & (1 << i)))
					continue;
				/* exclude all IMP ports as egress ports */
				reg_val16 = 0x1FF;
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_VLAN,
					CREG_PORT_VLAN_CTL, i,
					&reg_val16));
			}
		}
	}
EXIT:
	FUNC_LEAVE();
	return (0);
}

int ethsw_switch_mode_set(bool managed)
{
	int ret = 0;

	FUNC_ENTER();
	ethsw_switch_mode_set_sf2(managed);
	ethsw_switch_mode_set_531xx(managed);

	FUNC_LEAVE();
	return(ret);
}


int ethsw_switch_mode_set_sf2(bool managed)
{
	int ret = 0;
	u8  reg_val8;
	u16 reg_val16;
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];
	struct ethsw_core *swcore = ethsw_dev[INTERNAL_SW]->swcore;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);

	VALIDATE_ETH_PWR_STATUS(swdev);
	VALIDATE_SWCORE(swdev);

	if (managed) {
		/* set switch mode */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

		reg_val8 = CREG_FLD_SET(
			CREG_SWMODE,
			SW_FWDG_MODE,
			reg_val8,
			1);

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

#if CORE_USE_SINGLE_IMP
		/* set IMP port control */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_IMP_CTL,
			&reg_val8));

		reg_val8 |= (
			CREG_FLD_SHF(CREG_IMP_CTL, RX_UCST_EN, 1) |
			CREG_FLD_SHF(CREG_IMP_CTL, RX_MCST_EN, 1) |
			CREG_FLD_SHF(CREG_IMP_CTL, RX_BCST_EN, 1));

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_IMP_CTL,
			&reg_val8));
#endif
	}
	else {
		/* set switch mode */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

		reg_val8 = CREG_FLD_SET(
			CREG_SWMODE,
			SW_FWDG_MODE,
			reg_val8,
			0);

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

		/* set switch control */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_SWITCH_CTRL,
			&reg_val16));

		reg_val16 = CREG_FLD_SET(
			CREG_SWITCH_CTRL,
			MII_DUMB_FWDG_EN,
			reg_val16,
			1);

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_SWITCH_CTRL,
			&reg_val16));
	}

	swcore->managed = managed;
	LOG_INFO("Internal Switch mode set, managed = %d\n", managed);

 EXIT:
	FUNC_LEAVE();
	return(ret);
}


int ethsw_switch_mode_set_531xx(bool managed)
{
	int ret = 0;
	u8  reg_val8;
	u16 reg_val16;
	struct ethsw_device *swdev = ethsw_dev[EXTERNAL_SPI_SW];
	struct ethsw_core *swcore = ethsw_dev[EXTERNAL_SPI_SW]->swcore;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);

	VALIDATE_ETH_PWR_STATUS(swdev);
	VALIDATE_SWCORE(swdev);

	if (managed) {
		/* set switch mode */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

		reg_val8 = CREG_FLD_SET(
			CREG_SWMODE,
			SW_FWDG_MODE,
			reg_val8,
			1);

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

		/* set IMP port control */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_IMP_CTL,
			&reg_val8));

		reg_val8 |= (
			CREG_FLD_SHF(CREG_IMP_CTL, RX_UCST_EN, 1) |
			CREG_FLD_SHF(CREG_IMP_CTL, RX_MCST_EN, 1) |
			CREG_FLD_SHF(CREG_IMP_CTL, RX_BCST_EN, 1));

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_IMP_CTL,
			&reg_val8));
	}
	else {
		/* set switch mode */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

		reg_val8 = CREG_FLD_SET(
			CREG_SWMODE,
			SW_FWDG_MODE,
			reg_val8,
			0);

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

		/* set switch control */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_SWITCH_CTRL,
			&reg_val16));

		reg_val16 = CREG_FLD_SET(
			CREG_SWITCH_CTRL,
			MII_DUMB_FWDG_EN,
			reg_val16,
			1);

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_SWITCH_CTRL,
			&reg_val16));
	}

	swcore->managed = managed;
	LOG_INFO("531xx Switch mode set, managed = %d\n", managed);

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_switch_mode_get(bool *managed)
{
	int ret = 0;
	/*ZACK*/
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

#if CORE_USE_HW_STAT
	if (managed) {
		u8  reg_val8;

		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_SWMODE,
			&reg_val8));

		*managed = CREG_FLD_GET(
			CREG_SWMODE, SW_FWDG_MODE, reg_val8);
	}

 EXIT:
#else
	if (managed) {
		struct ethsw_core *swcore = swdev->swcore;
		VALIDATE_SWCORE(swdev);

		*managed = swcore->managed;
	}
#endif

	FUNC_LEAVE();
	return(ret);
}

int ethsw_switch_cert_set(bool enable)
{
	int ret = 0;
	u16 reg_val16;
	u32 reg_val32;
	int i;
	/*ZACK*/
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];
	struct ethsw_core *swcore = ethsw_dev[INTERNAL_SW]->swcore;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);
	VALIDATE_SWCORE(swdev);

	if (enable) {
		/* set IMP ports to enable PCP (trusted port) */
		CHECK_EXIT(CREG_RD(
			PAGE_QOS,
			CREG_QOS_1P_EN,
			&reg_val16));

		reg_val16 |= (
			((swdev->port[5].type == IMP_PORT) ? (1 << 5) : 0) |
			((swdev->port[8].type == IMP_PORT) ? (1 << 8) : 0));

		CHECK_EXIT(CREG_WR(
			PAGE_QOS,
			CREG_QOS_1P_EN,
			&reg_val16));

		/* set IMP ports to use PCP for TC mapping */
		/* PCP is only present in IP packets with trusted tag, */
		/* i.e. for index 3 and 7 */
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type == IMP_PORT) {
				CHECK_EXIT(CREG_IDX_RD(
					PAGE_QOS,
					CREG_TC_SEL_TABLE, i,
					&reg_val16));

				reg_val16 &= ~(
					CREG_FLD_MSK(CREG_TC_SEL_TABLE, TC_SEL_3) |
					CREG_FLD_MSK(CREG_TC_SEL_TABLE, TC_SEL_7));

				reg_val16 |= (
					CREG_FLD_SHF(CREG_TC_SEL_TABLE, TC_SEL_3, 1) |
					CREG_FLD_SHF(CREG_TC_SEL_TABLE, TC_SEL_7, 1));

				CHECK_EXIT(CREG_IDX_WR(
					PAGE_QOS,
					CREG_TC_SEL_TABLE, i,
					&reg_val16));
			}
		}

		/* set all ports to use 2 TX queues based on TC */
		reg_val32 = (
			CREG_FLD_SHF(CREG_TC2COS_MAP, PRT001_TO_QID, 1) |
			CREG_FLD_SHF(CREG_TC2COS_MAP, PRT010_TO_QID, 1) |
			CREG_FLD_SHF(CREG_TC2COS_MAP, PRT011_TO_QID, 1) |
			CREG_FLD_SHF(CREG_TC2COS_MAP, PRT100_TO_QID, 1) |
			CREG_FLD_SHF(CREG_TC2COS_MAP, PRT101_TO_QID, 1) |
			CREG_FLD_SHF(CREG_TC2COS_MAP, PRT110_TO_QID, 1) |
			CREG_FLD_SHF(CREG_TC2COS_MAP, PRT111_TO_QID, 1));

		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type != INV_PORT) {
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_QOS,
					CREG_TC2COS_MAP, i,
					&reg_val32));
			}
		}

		/* disable non-IMP port TX based back-pressure (pause) */
		CHECK_EXIT(CREG_RD(
			PAGE_FLOWCON,
			CREG_FC_PAUSE_DROP_CTRL,
			&reg_val16));

		reg_val16 &= ~(
			CREG_FLD_MSK(CREG_FC_PAUSE_DROP_CTRL, TX_TOTAL_PAUSE_EN) |
			CREG_FLD_MSK(CREG_FC_PAUSE_DROP_CTRL, TX_TXQ_PAUSE_EN));

		CHECK_EXIT(CREG_WR(
			PAGE_FLOWCON,
			CREG_FC_PAUSE_DROP_CTRL,
			&reg_val16));
	}
	else {
		/* reset IMP ports to disable PCP (trusted port) */
		CHECK_EXIT(CREG_RD(
			PAGE_QOS,
			CREG_QOS_1P_EN,
			&reg_val16));

		reg_val16 &= ~(
			((swdev->port[5].type == IMP_PORT) ? (1 << 5) : 0) |
			((swdev->port[8].type == IMP_PORT) ? (1 << 8) : 0));

		CHECK_EXIT(CREG_WR(
			PAGE_QOS,
			CREG_QOS_1P_EN,
			&reg_val16));

		/* reset IMP ports NOT to use PCP for TC mapping */
		/* PCP is only present in IP packets with trusted tag, */
		/* i.e. for index 3 and 7. */
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type == IMP_PORT) {
				CHECK_EXIT(CREG_IDX_RD(
					PAGE_QOS,
					CREG_TC_SEL_TABLE, i,
					&reg_val16));

				reg_val16 &= ~(
					CREG_FLD_MSK(CREG_TC_SEL_TABLE, TC_SEL_3) |
					CREG_FLD_MSK(CREG_TC_SEL_TABLE, TC_SEL_7));

				CHECK_EXIT(CREG_IDX_WR(
					PAGE_QOS,
					CREG_TC_SEL_TABLE, i,
					&reg_val16));
			}
		}

		/* reset all ports to use single TX queues */
		reg_val32 = 0;

		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type != INV_PORT) {
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_QOS,
					CREG_TC2COS_MAP, i,
					&reg_val32));
			}
		}

		/* enable non-IMP port TX based back-pressure (pause) */
		CHECK_EXIT(CREG_RD(
			PAGE_FLOWCON,
			CREG_FC_PAUSE_DROP_CTRL,
			&reg_val16));

		reg_val16 |= (
			CREG_FLD_SHF(CREG_FC_PAUSE_DROP_CTRL, TX_TOTAL_PAUSE_EN, 1) |
			CREG_FLD_SHF(CREG_FC_PAUSE_DROP_CTRL, TX_TXQ_PAUSE_EN,   1));

		CHECK_EXIT(CREG_WR(
			PAGE_FLOWCON,
			CREG_FC_PAUSE_DROP_CTRL,
			&reg_val16));
	}

	swcore->cert = enable;
	LOG_INFO("Switch certification mode set, enable = %d\n", enable);

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_switch_cert_get(bool *enable)
{
	int ret = 0;
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

#if CORE_USE_HW_STAT
	if (enable) {
		u16 reg_val16;
		int tx_total_pause, tx_txq_pause;

		/* check non-IMP port TX based back-pressure (pause) */
		CHECK_EXIT(CREG_RD(
			PAGE_FLOWCON,
			CREG_FC_PAUSE_DROP_CTRL,
			&reg_val16));

		tx_total_pause = CREG_FLD_GET(
			CREG_FC_PAUSE_DROP_CTRL, TX_TOTAL_PAUSE_EN, reg_val16);

		tx_txq_pause = CREG_FLD_GET(
			CREG_FC_PAUSE_DROP_CTRL, TX_TXQ_PAUSE_EN, reg_val16);

		*enable = ~(tx_total_pause || tx_txq_pause);
	}

 EXIT:
#else
	if (enable) {
		struct ethsw_core *swcore = swdev->swcore;
		VALIDATE_SWCORE(swdev);

		*enable = swcore->cert;
	}
#endif

	FUNC_LEAVE();
	return(ret);
}

int ethsw_switch_ethwan_set(bool enable, int port_id)
{
	int ret = 0;
	u8  reg_val8;
	u16 reg_val16;
	struct ethsw_device *swdev;
	struct ethsw_core *swcore;

	FUNC_ENTER();
	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);
    swcore = swdev->swcore;
	VALIDATE_SWCORE(swdev);

	if (port_id < 0 || port_id >= ETHSW_PORT_MAX) {
		ret = -EINVAL;
		goto EXIT;
	}
	if((int)(port_id - swdev->subid_port_offset) < 0)
	{
		ret = -EINVAL;
		goto EXIT;
	}
	if (swdev->port[port_id - swdev->subid_port_offset].type != PHY_PORT &&
		swdev->port[port_id - swdev->subid_port_offset].type != MII_PORT) {
		ret = -EINVAL;
		goto EXIT;
	}

	if (enable) {
		/* set global management config */
		CHECK_EXIT(CREG_RD(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&reg_val8));

		reg_val8 = CREG_FLD_SET(
			CREG_GMNGCFG,
			FRM_MNGP,
			reg_val8,
			3);

		CHECK_EXIT(CREG_WR(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&reg_val8));

		/* set WAN port select */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_WAN_PORT_SEL,
			&reg_val16));

		reg_val16 = CREG_FLD_SET(
			CREG_WAN_PORT_SEL,
			WAN_SELECT,
			reg_val16,
			(1 << (port_id- swdev->subid_port_offset)));

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_WAN_PORT_SEL,
			&reg_val16));
	}
	else {

		/* set global management config */
		CHECK_EXIT(CREG_RD(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&reg_val8));

		reg_val8 = CREG_FLD_SET(
			CREG_GMNGCFG,
			FRM_MNGP,
			reg_val8,
			2);

		CHECK_EXIT(CREG_WR(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&reg_val8));

		/* clear WAN port select */
		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_WAN_PORT_SEL,
			&reg_val16));

		reg_val16 = CREG_FLD_SET(
			CREG_WAN_PORT_SEL,
			WAN_SELECT,
			reg_val16,
			0);

		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_WAN_PORT_SEL,
			&reg_val16));
	}

	swcore->ethwan = enable;
	swcore->ethwan_port = port_id;
	LOG_INFO("Switch ethernet WAN set, enable = %d, port_id = %d port=%d\n",
			 enable, port_id, swdev->subid_port_offset);

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_switch_ethwan_get(bool *enable, int *port_id)
{
	int ret = 0;
	struct ethsw_device *swdev;
#if CORE_USE_HW_STAT
	u16 reg_val16;
	int port_map, i;
#else
	struct ethsw_core *swcore;
#endif

	FUNC_ENTER();
	if (!port_id)
		goto EXIT;

	swdev = subid_to_swdev(*port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

#if CORE_USE_HW_STAT
	if (enable) {
		u8  reg_val8;
		int imp_mode;

		CHECK_EXIT(CREG_RD(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&reg_val8));

		imp_mode = CREG_FLD_GET(
			CREG_GMNGCFG, FRM_MNGP, reg_val8);

		*enable = (imp_mode == 3);
	}

	CHECK_EXIT(CREG_RD(
		PAGE_CTLREG,
		CREG_WAN_PORT_SEL,
		&reg_val16));

	port_map = CREG_FLD_GET(
		CREG_WAN_PORT_SEL, WAN_SELECT, reg_val16);

	*port_id = ETHSW_PORT_ID_INVALID;
	for (i = 0; i < ETHSW_PORT_MAX; i++) {
		if (port_map & (1 << i)) {
			*port_id = i;
			break;
		}
	}

#else
	swcore = swdev->swcore;
	VALIDATE_SWCORE(swdev);
	if (enable) {
		*enable = swcore->ethwan;
	}

	*port_id = swcore->ethwan_port;
#endif

EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_switch_alt_ethwan_get_sf2(struct ethsw_device *swdev, int *imp_port, int *lan_port,  bool *enable)
{
	int ret = 0;
    struct ethsw_core *swcore;

	VALIDATE_SWDEV(swdev);
	swcore = swdev->swcore;
	VALIDATE_SWCORE(swdev);
#if CORE_USE_HW_STAT
    // Need to be Updated
#else
    if (enable)
        *enable =  swcore->altwan_enble;
    if (imp_port)
        *imp_port = swcore->altwan_impport;
    if (lan_port)
        *lan_port = swcore->altwan_lanport;
#endif

	FUNC_LEAVE();
	return(ret);
}

int ethsw_switch_alt_ethwan_get_53xxx(struct ethsw_device *swdev, int *imp_port, int *lan_port,  bool *enable)
{
	int ret = 0;
    struct ethsw_core *swcore;

	VALIDATE_SWDEV(swdev);
	swcore = swdev->swcore;
	VALIDATE_SWCORE(swdev);
#if CORE_USE_HW_STAT
    // Need to be Updated
#else
    if (enable)
        *enable =  swcore->altwan_enble;
    if (imp_port)
        *imp_port = swcore->altwan_impport;
    if (lan_port)
        *lan_port = swcore->altwan_lanport;
#endif

	FUNC_LEAVE();
	return(ret);
}

int ethsw_switch_alt_ethwan_get(struct ethsw_device *swdev, int *imp_port, int *lan_port,  bool *enable)
{
	int ret = 0;

	FUNC_ENTER();
	if (swdev != ethsw_dev[INTERNAL_SW])
		ret = ethsw_switch_alt_ethwan_get_53xxx(swdev, imp_port, lan_port, enable);
	else
		ret = ethsw_switch_alt_ethwan_get_sf2(swdev, imp_port, lan_port, enable);

	FUNC_LEAVE();
	return(ret);
}

/****************************************************************************
  Status Functions
****************************************************************************/

bool ethsw_switch_port_link_up(struct ethsw_device *swdev, int port_id)
{
	int ret = 0;
	u16 reg_val16;
	bool link_up = false;
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		goto EXIT;
	}
	if (!swdev->ethsw_pwr_status)
		goto EXIT;

	CHECK_EXIT(CREG_RD(
		PAGE_STSREG,
		CREG_LNKSTS,
		&reg_val16));

	link_up = (reg_val16 >> (port_id - swdev->subid_port_offset)) & 0x1;

 EXIT:
	return(link_up);
}

bool ethsw_switch_port_full_duplex(struct ethsw_device *swdev,int port_id)
{
	int ret = 0;
	u16 reg_val16;
	bool full_duplex = false;
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		goto EXIT;
	}
	if (!swdev->ethsw_pwr_status)
		goto EXIT;

	CHECK_EXIT(CREG_RD(
		PAGE_STSREG,
		CREG_DUPSTS,
		&reg_val16));

	full_duplex = (reg_val16 >> (port_id - swdev->subid_port_offset)) & 0x1;

 EXIT:
	return(full_duplex);
}

int ethsw_switch_port_speed(int port_id)
{
	int ret = 0;
	u32 reg_val32;
	int port_speed = 0;
	struct ethsw_device *swdev;

	swdev = subid_to_swdev(port_id);
	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized for port: %d\n", port_id);
		goto EXIT;
	}
	if (!swdev->ethsw_pwr_status)
		goto EXIT;

	CHECK_EXIT(CREG_RD(
		PAGE_STSREG,
		CREG_SPDSTS,
		&reg_val32));

	port_speed = (reg_val32 >> ((port_id - swdev->subid_port_offset) * 2)) & 0x3;

 EXIT:
	return(port_speed);
}
EXPORT_SYMBOL(ethsw_switch_port_speed);

/****************************************************************************
  EEE Functions
****************************************************************************/

int ethsw_switch_port_eee_config_set(
	int port_id,
	struct ethsw_power_config *power_config)
{
	int ret = 0;
	u16 reg_val16;
	u32 reg_val32;
	struct ethsw_device *swdev;
	struct ethsw_port *port;
	int eee_lp = 1;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	port = &swdev->port[port_id];

	if (port->phy.dev && port->phy.dev->mdio.bus) {
		eee_lp = phy_read_mmd(port->phy.dev, MDIO_MMD_AN, MDIO_AN_EEE_LPABLE);
	}

	CHECK_EXIT(CREG_RD(
		PAGE_EEE_CTL,
		CREG_EEE_EN_CTRL,
		&reg_val16));

	if (power_config->eee_params.enable && eee_lp > 0) {
		reg_val16 |= (1 << (port_id - swdev->subid_port_offset));
	}
	else {
		reg_val16 &= ~(1 << (port_id - swdev->subid_port_offset));
	}

	CHECK_EXIT(CREG_WR(
		PAGE_EEE_CTL,
		CREG_EEE_EN_CTRL,
		&reg_val16));

	if (power_config->eee_params.enable && eee_lp > 0) {
		struct ethsw_eee_params *eee_params =
			&power_config->eee_params;

		reg_val32 = (eee_params->sleep_timer_g) ?
			eee_params->sleep_timer_g :
			CREG_EEE_SLEEP_TIMER_G_DFLT;

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_EEE_CTL,
			CREG_EEE_SLEEP_TIMER_G, port_id - swdev->subid_port_offset,
			&reg_val32));

		reg_val32 = (eee_params->sleep_timer_h) ?
			eee_params->sleep_timer_h :
			CREG_EEE_SLEEP_TIMER_H_DFLT;

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_EEE_CTL,
			CREG_EEE_SLEEP_TIMER_H, port_id - swdev->subid_port_offset,
			&reg_val32));

		reg_val32 = (eee_params->min_lp_timer_g) ?
			eee_params->min_lp_timer_g :
			CREG_EEE_MIN_LP_TIMER_G_DFLT;

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_EEE_CTL,
			CREG_EEE_MIN_LP_TIMER_G, port_id - swdev->subid_port_offset,
			&reg_val32));

		reg_val32 = (eee_params->min_lp_timer_h) ?
			eee_params->min_lp_timer_h :
			CREG_EEE_MIN_LP_TIMER_H_DFLT;

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_EEE_CTL,
			CREG_EEE_MIN_LP_TIMER_H, port_id - swdev->subid_port_offset,
			&reg_val32));

		reg_val16 = (eee_params->wake_timer_g) ?
			eee_params->wake_timer_g :
			CREG_EEE_WAKE_TIMER_G_DFLT;

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_EEE_CTL,
			CREG_EEE_WAKE_TIMER_G, port_id - swdev->subid_port_offset,
			&reg_val16));

		reg_val16 = (eee_params->wake_timer_h) ?
			eee_params->wake_timer_h :
			CREG_EEE_WAKE_TIMER_H_DFLT;

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_EEE_CTL,
			CREG_EEE_WAKE_TIMER_H, port_id - swdev->subid_port_offset,
			&reg_val16));
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

bool ethsw_switch_port_eee_assert(int port_id)
{
	int ret = 0;
	u16 reg_val16;
	bool eee_assert = false;
	struct ethsw_device *swdev;

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	CHECK_EXIT(CREG_RD(
		PAGE_EEE_CTL,
		CREG_EEE_LPI_ASSERT,
		&reg_val16));

	eee_assert = (reg_val16 >> (port_id - swdev->subid_port_offset)) & 0x1;

 EXIT:
	return(eee_assert);
}

bool ethsw_switch_port_eee_indicate(int port_id)
{
	int ret = 0;
	u16 reg_val16;
	bool eee_indicate = false;
	struct ethsw_device *swdev;

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	CHECK_EXIT(CREG_RD(
		PAGE_EEE_CTL,
		CREG_EEE_LPI_INDICATE,
		&reg_val16));

	eee_indicate = (reg_val16 >> (port_id - swdev->subid_port_offset)) & 0x1;

 EXIT:
	return(eee_indicate);
}


int ethsw_alt_ethwan_core(struct ethsw_device *swdev, int imp_port, int lan_port, bool enable)
{
    struct ethsw_core *swcore;
    int ret = 0;

	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);
	swcore = swdev->swcore;
	VALIDATE_SWCORE(swdev);
	
	if (swdev != ethsw_dev[INTERNAL_SW])
		ret = ethsw_alt_ethwan_53xxx(swdev, imp_port, lan_port - swdev->subid_port_offset, enable);
	else {
		ret = ethsw_alt_ethwan_sf2(swdev, imp_port, lan_port - swdev->subid_port_offset, enable);
	}
    // Update status if set is successful
    if (!ret) {
		swcore->altwan_enble = enable;
		swcore->altwan_impport = imp_port;
		swcore->altwan_lanport = lan_port;
    }
    return(ret);
}

int ethsw_alt_ethwan_53xxx(struct ethsw_device *swdev, int imp_port, int lan_port, bool enable)
{
	struct ethsw_device *swdev_sf2 = ethsw_dev[INTERNAL_SW];
	int ret = 0;
	int i;

	if (enable) {
		/* Turn off the other phys on the external switch not used for WAN*/
		for (i = 0; i < ETHSW_PHY_PORT_MAX; i++) {
			if (i != lan_port) {
				if (!(swdev->phy_port_mask & (1 << i)))
					continue;
				ethsw_port_disable(i + swdev->subid_port_offset);
			}
		}
	}
	else {
		/* Turn on the other phys on the external */
		for (i = 0; i < ETHSW_PHY_PORT_MAX; i++) {
	      if (!(swdev->phy_port_mask & (1 << i)))
				continue;
			ethsw_port_enable(i + swdev->subid_port_offset);
		}
	}

	ret = ethsw_alt_ethwan_sf2(swdev_sf2,imp_port, ETHSW_SF2_TO_EXT_SWITCH_PORT, enable);

	return ret;
}

int ethsw_alt_ethwan_sf2(struct ethsw_device *swdev, int imp_port, int lan_port, bool enable)
{
	u16 reg_val16;
	int ret = 0;
	int i;

	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	if (enable) {
		// To enable alt ethwan we first must make sure the
		// imp port is removed from the trunk group.
		//swdev->imp_port_mask &= ~(1 << imp_port);
		reg_val16 = swdev->imp_port_mask & ~(1 << imp_port);

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_TRUNK,
			CREG_TRUNK_GRP_CTL, 0,
			&reg_val16));

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_TRUNK,
			CREG_IMP0_GRP_CTL, 0,
			&reg_val16));

		// we must create a port based vlan for the two ports
		reg_val16 = (1 << imp_port) | (1 << lan_port);
		CHECK_EXIT(CREG_IDX_WR(
			PAGE_VLAN,
			CREG_PORT_VLAN_CTL, imp_port,
			&reg_val16));

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_VLAN,
			CREG_PORT_VLAN_CTL, lan_port,
			&reg_val16));

		//remove it from ALL the others also....
		reg_val16 = 0x1FF ^ (1 << imp_port);
		reg_val16 ^= (1 << lan_port);

		for (i = 0; i < 9; i++) {
			if ((i != lan_port) && (i != imp_port)) {
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_VLAN,
					CREG_PORT_VLAN_CTL, i,
					&reg_val16));
			}
			if (i == lan_port || i == imp_port) {

				// we must turn off arl learning for these ports.
				ethsw_arl_swlearn_enable(i,false);
				ethsw_arl_hwlearn_enable(i,false);
				ethsw_arl_fast_age_set(i);
			}
		}
	}
	else {

		// Add IMP port back to trunking group
		swdev->imp_port_mask |= (1 << imp_port);
		reg_val16 = swdev->imp_port_mask;
		CHECK_EXIT(CREG_IDX_WR(
			PAGE_TRUNK,
			CREG_TRUNK_GRP_CTL, 0,
			&reg_val16));

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_TRUNK,
			CREG_IMP0_GRP_CTL, 0,
			&reg_val16));

		/* disable cross talk among IMP ports */
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type == IMP_PORT) {
				/* exclude all IMP ports as egress ports */
				reg_val16 = 0x1FF ^ swdev->imp_port_mask;
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_VLAN,
					CREG_PORT_VLAN_CTL, i,
					&reg_val16));
			}
		}
		// Set lan ports back to their defaults
		reg_val16 = 0x1ff;
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type != IMP_PORT) {
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_VLAN,
					CREG_PORT_VLAN_CTL, i,
					&reg_val16));
			}
		}

		for (i = 0; i < 9; i++) {
			if (i == lan_port || i == imp_port)
				ethsw_arl_hwlearn_enable(i,true);
		}
	}

EXIT:
   return(0);

}



/****************************************************************************
  ARL Control Registers
****************************************************************************/

int ethsw_arl_config_set(struct ethsw_arl_config *arl_cfg)
{
	int ret = 0;

	FUNC_ENTER();

	FUNC_LEAVE();
	return(ret);
}

int ethsw_arl_config_get(struct ethsw_arl_config *arl_cfg)
{
	int ret = 0;

	FUNC_ENTER();

	FUNC_LEAVE();
	return(ret);
}

/****************************************************************************
  ARL/VTBL Access Functions
****************************************************************************/

/* read ARL table, find bucket with 4 entries */
static int ethsw_arl_table_read(struct ethsw_device *swdev, u8 *mac, u16 vid)
{
	int ret = 0;
	u8  rwctl;
	u64 mac48;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	pack_mac_to_u48(&mac48, mac);

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_MAC,
		&mac48));

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_VID,
		&vid));

	rwctl =
		CREG_FLD_SHF(CREG_ARLA_RWCTL, ARL_STRTDN, 1) |
		CREG_FLD_SHF(CREG_ARLA_RWCTL, ARL_RW,     1);

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_RWCTL,
		&rwctl));

	do {
		udelay(10);

		CHECK_EXIT(CREG_RD(
			PAGE_ARLACCS,
			CREG_ARLA_RWCTL,
			&rwctl));

	} while (CREG_FLD_GET(
				 CREG_ARLA_RWCTL,
				 ARL_STRTDN,
				 rwctl));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

/* write ARL table, update bucket with 4 entries */
static int ethsw_arl_table_write(struct ethsw_device *swdev)
{
	int ret = 0;
	u8  rwctl;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	rwctl =
		CREG_FLD_SHF(CREG_ARLA_RWCTL, ARL_STRTDN, 1) |
		CREG_FLD_SHF(CREG_ARLA_RWCTL, ARL_RW,     0);

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_RWCTL,
		&rwctl));

	do {
		udelay(10);

		CHECK_EXIT(CREG_RD(
			PAGE_ARLACCS,
			CREG_ARLA_RWCTL,
			&rwctl));

	} while (CREG_FLD_GET(
				 CREG_ARLA_RWCTL,
				 ARL_STRTDN,
				 rwctl));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_arl_table_find(u8 *mac, u16 vid, u16 *port)
{
	u16 query_vid = 0;

	if(vid == 0)
		query_vid = 1;
	else
		query_vid = vid;

	return ethsw_arl_cache_table_find(mac, query_vid, port);
}

int ethsw_arl_switch_table_find(struct ethsw_device *swdev, u8 *mac, u16 vid, u16 *port)
{
	int ret = 0;
	int i;
	u64 macvid_new, macvid_entry;
	u32 fwd_entry;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	/* if we dont have vlans turned on then dont use it as a lookup*/
	if (swdev->vlan_default_mode == 0) {
		vid = 0;
	}

	LOG_DEBUG("ARL port lookup for %pM vid=%u\n", mac, vid);

	/* find ARL table bucket */
	CHECK_EXIT(ethsw_arl_table_read(swdev, mac, vid));

	pack_macvid_to_u64(&macvid_new, mac, vid);

	for (i = 0; i < 4; i++) {
		CHECK_EXIT(CREG_IDX_RD(
			PAGE_ARLACCS,
			CREG_ARLA_FWD_ENTRY, i,
			&fwd_entry));

		if (CREG_FLD_GET(
				CREG_ARLA_FWD_ENTRY,
				ARL_VALID,
				fwd_entry)) {
			/* entry is valid */
			CHECK_EXIT(CREG_IDX_RD(
				PAGE_ARLACCS,
				CREG_ARLA_MACVID_ENTRY, i,
				&macvid_entry));

			if (macvid_entry == macvid_new) {
				/* matching entry found */
				*port = CREG_FLD_GET(
					CREG_ARLA_FWD_ENTRY,
					PORTID,
					fwd_entry);

				*port += swdev->subid_port_offset;
				break;
			}
		}
	}

	if (i == 4) {
		/* entry not found, return invalid */
		*port = ETHSW_PORT_MAP_INVALID;
	}

	LOG_DEBUG("port found %u\n", *port);
 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_arl_table_find);

int ethsw_arl_table_add(u8 *mac, u16 vid, u16 port)
{
	return ethsw_arl_cache_table_add( mac, vid, port);
}

int ethsw_arl_table_mcast_add(struct ethsw_device *swdev, u8 *mac, u16 vid, u16 port)
{
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	return ethsw_arl_switch_table_add( swdev, mac, vid, port);
}
EXPORT_SYMBOL(ethsw_arl_table_mcast_add);


int ethsw_arl_switch_table_add(struct ethsw_device *swdev, u8 *mac, u16 vid, u16 port)
{
	int ret = 0;
	int i;
	u64 macvid_new, macvid_entry;
	u32 fwd_new, fwd_entry;
	int entry = 0;
	bool update = false;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	/* read ARL table bucket */
	CHECK_EXIT(ethsw_arl_table_read(swdev, mac, vid));

	pack_macvid_to_u64(&macvid_new, mac, vid);

	for (i = 0; i < 4; i++) {
		CHECK_EXIT(CREG_IDX_RD(
			PAGE_ARLACCS,
			CREG_ARLA_FWD_ENTRY, i,
			&fwd_entry));

		if (CREG_FLD_GET(
				CREG_ARLA_FWD_ENTRY,
				ARL_VALID,
				fwd_entry)) {
			/* entry is valid */
			CHECK_EXIT(CREG_IDX_RD(
				PAGE_ARLACCS,
				CREG_ARLA_MACVID_ENTRY, i,
				&macvid_entry));

			if (macvid_entry == macvid_new) {
				/* matching entry found */
				if (CREG_FLD_GET(
						CREG_ARLA_FWD_ENTRY,
						ARL_STATIC,
						fwd_entry) != 1 ||
					CREG_FLD_GET(
						CREG_ARLA_FWD_ENTRY,
						PORTID,
						fwd_entry) != port) {
					/* update needed */
					entry = i;
					update = true;
				}
				break;
			}
		}
		else {
			/* entry is empty */
			if (!update) {
				/* remember the first empty entry */
				entry = i;
				update = true;
			}
		}
	}

	if (update) {
		fwd_new =
			CREG_FLD_SHF(CREG_ARLA_FWD_ENTRY, ARL_VALID,  1) |
			CREG_FLD_SHF(CREG_ARLA_FWD_ENTRY, ARL_STATIC, 1) |
			CREG_FLD_SHF(CREG_ARLA_FWD_ENTRY, PORTID,     port);

		/* only need to write MAC/VID if different */
		if (macvid_entry != macvid_new) {
			CHECK_EXIT(CREG_IDX_WR(
				PAGE_ARLACCS,
				CREG_ARLA_MACVID_ENTRY, entry,
				&macvid_new));
		}

		CHECK_EXIT(CREG_IDX_WR(
			PAGE_ARLACCS,
			CREG_ARLA_FWD_ENTRY, entry,
			&fwd_new));

		/* write ARL table bucket */
		CHECK_EXIT(ethsw_arl_table_write(swdev));
	}
	else if (i == 4) {
		/* no empty entry */
		ret = -ENOMEM;
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_arl_table_add);

int ethsw_arl_table_delete(u8 *mac, u16 vid)
{
	return ethsw_arl_cache_table_delete(mac, vid);
  //  return ethsw_arl_switch_table_delete(mac, vid);
}


int ethsw_arl_table_mcast_delete(struct ethsw_device *swdev, u8 *mac, u16 vid)
{
	return ethsw_arl_switch_table_delete(swdev, mac, vid);
}

int ethsw_arl_switch_table_delete(struct ethsw_device *swdev, u8 *mac, u16 vid)
{
	int ret = 0;
	int i;
	u64 macvid_new, macvid_entry;
	u32 fwd_new, fwd_entry;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	/* read ARL table bucket */
	CHECK_EXIT(ethsw_arl_table_read(swdev, mac, vid));

	pack_macvid_to_u64(&macvid_new, mac, vid);

	for (i = 0; i < 4; i++) {
		CHECK_EXIT(CREG_IDX_RD(
			PAGE_ARLACCS,
			CREG_ARLA_FWD_ENTRY, i,
			&fwd_entry));

		if (CREG_FLD_GET(
				CREG_ARLA_FWD_ENTRY,
				ARL_VALID,
				fwd_entry)) {
			/* entry is valid */
			CHECK_EXIT(CREG_IDX_RD(
				PAGE_ARLACCS,
				CREG_ARLA_MACVID_ENTRY, i,
				&macvid_entry));

			if (macvid_entry == macvid_new) {
				/* matching entry found */
				fwd_new = 0;
				CHECK_EXIT(CREG_IDX_WR(
					PAGE_ARLACCS,
					CREG_ARLA_FWD_ENTRY, i,
					&fwd_new));

				/* write ARL table bucket */
				CHECK_EXIT(ethsw_arl_table_write(swdev));
				break;
			}
		}
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_arl_table_delete);

int ethsw_arl_fast_age_set(int port_id)
{
	int ret = 0;
	u8  reg_val8_rd, reg_val8_wr;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	port_id -= swdev->subid_port_offset;

	/* set port for fast age */
	CHECK_EXIT(CREG_WR(
	   PAGE_CTLREG,
	   CREG_FAST_AGE_PORT,
	   (u8 *)&port_id));

	CHECK_EXIT(CREG_RD(
	   PAGE_CTLREG,
	   CREG_FAST_AGEING,
	   &reg_val8_rd));
	reg_val8_wr = reg_val8_rd;

	/* enable port fast age */
	reg_val8_wr = CREG_FLD_SET(
	   CREG_FAST_AGEING,
	   EN_AGE_PORT,
	   reg_val8_wr,
	   1);

	/* start fast age */
	reg_val8_wr = CREG_FLD_SET(
	   CREG_FAST_AGEING,
	   FAST_AGE_STR_DONE,
	   reg_val8_wr,
	   1);

	CHECK_EXIT(CREG_WR(
	   PAGE_CTLREG,
	   CREG_FAST_AGEING,
	   &reg_val8_wr));

	/* wait for command done */
	do {
		udelay(10);

		CHECK_EXIT(CREG_RD(
			PAGE_CTLREG,
			CREG_FAST_AGEING,
			&reg_val8_wr));

	} while (CREG_FLD_GET(
				 CREG_FAST_AGEING,
				 FAST_AGE_STR_DONE,
				 reg_val8_wr));

	/* clear fast age */
	CHECK_EXIT(CREG_WR(
	   PAGE_CTLREG,
	   CREG_FAST_AGEING,
	   &reg_val8_rd));

	reg_val8_wr = 0;
	CHECK_EXIT(CREG_WR(
	   PAGE_CTLREG,
	   CREG_FAST_AGE_PORT,
	   &reg_val8_wr));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_arl_fast_age_set);

#define SMP_ARL_ACCESS_REGISTER_PAGE     0x05
#define ARL_TABLE_SEARCH_CNTRL           0x50
#define ARL_TABLE_MAC_VID_RESULT0        0x60
#define ARL_TABLE_MAC_VID_RESULT1        0x70
#define ARL_TABLE_SEARCH_DATA_RESULT0    0x68
#define ARL_TABLE_SEARCH_DATA_RESULT1    0x78

#ifdef __BIG_ENDIAN

typedef struct  __attribute__((__packed__)) {
	u64 reserved :4;
	u64 vid      :12;
	u64 mac      :48;
} ethsw_arl_table_mac_vid_result;

typedef struct  __attribute__((__packed__)) {
	u32 rsrvd1   :15;
	u32 valid    :1;
	u32 stat     :1;
	u32 age      :1;
	u32 tc       :3;
	u32 rsrvd2   :2;
	u32 port     :9;
} ethsw_arl_table_data_result;

typedef struct __attribute__((__packed__)) {
	u8 start   :1;
	u8 rsrvd   :6;
	u8 valid   :1;
} ethsw_arl_table_search_ctrl;

#else

typedef struct __attribute__((__packed__)) {
	u64 mac      :48;
	u64 vid      :12;
	u64 reserved :4;
} ethsw_arl_table_mac_vid_result;

typedef struct  __attribute__((__packed__)) {
	u32 port     :9;
	u32 rsrvd2   :2;
	u32 tc       :3;
	u32 age      :1;
	u32 stat     :1;
	u32 valid    :1;
	u32 rsrvd1   :15;
} ethsw_arl_table_data_result;

typedef struct __attribute__((__packed__)) {
	u8 valid   :1;
	u8 rsrvd   :6;
	u8 start   :1;
} ethsw_arl_table_search_ctrl;

#endif

int ethsw_arl_table_get_next(struct ethsw_device *swdev,
			     ethsw_arl_table_entry *tbl_entry)
{
	ethsw_arl_table_mac_vid_result mac_vid_res;
	ethsw_arl_table_data_result data_res;
	ethsw_arl_table_search_ctrl search_ctl;
	u8  mac_vid_result_regs[ETHSW_ARL_TBL_ENTRIES_PER_READ] =
		{ARL_TABLE_MAC_VID_RESULT0,
		 ARL_TABLE_MAC_VID_RESULT1};
	u8  data_result_regs[ETHSW_ARL_TBL_ENTRIES_PER_READ] =
		{ARL_TABLE_SEARCH_DATA_RESULT0,
		 ARL_TABLE_SEARCH_DATA_RESULT1};
	static int search_in_progress = false;
	u64 tmp64;
	int i, ret;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	memset(tbl_entry, 0, (ETHSW_ARL_TBL_ENTRIES_PER_READ *
			      sizeof(ethsw_arl_table_entry)));

	if(search_in_progress == false) {
		// start the search
		search_ctl.start = 1;
		search_ctl.rsrvd = 0;
		search_ctl.valid = 0;
		ret = ethsw_creg_write(swdev, SMP_ARL_ACCESS_REGISTER_PAGE,
				       ARL_TABLE_SEARCH_CNTRL,
				       (u8 *)&search_ctl, sizeof(search_ctl));
		if(ret) {
			LOG_ERR("Failed to start ARL table dump\n");
			search_ctl.start = 0;
			goto EXIT;
		}
		search_in_progress = true;
	}


	ret = ethsw_creg_read(swdev, SMP_ARL_ACCESS_REGISTER_PAGE,
			      ARL_TABLE_SEARCH_CNTRL,
			      (u8 *)&search_ctl,
			      sizeof(search_ctl));
	if(ret) {
		LOG_ERR("Failed to ascertain if ARL table entry is valid \n");
		goto EXIT;
	}

	if (search_ctl.valid) {
		for(i = 0; i < ETHSW_ARL_TBL_ENTRIES_PER_READ; i ++) {
			ret = ethsw_creg_read(swdev, SMP_ARL_ACCESS_REGISTER_PAGE,
					      mac_vid_result_regs[i],
					      (u8 *)&mac_vid_res, sizeof(mac_vid_res));
			if(ret) {
				LOG_ERR("Failed to read ARL table mac vid result reg\n");
				goto EXIT;
			}

			ret = ethsw_creg_read(swdev, SMP_ARL_ACCESS_REGISTER_PAGE,
					      data_result_regs[i],
					      (u8 *)&data_res, sizeof(data_res));
			if(ret) {
				LOG_ERR("Failed to read ARL table data result reg\n");
				goto EXIT;
			}

			if (data_res.valid) {
				tbl_entry[i].valid = 1;
				tbl_entry[i].port = data_res.port;
				tbl_entry[i].stat = data_res.stat;
				tbl_entry[i].vid = mac_vid_res.vid;
				tmp64 = mac_vid_res.mac;
#ifdef __LITTLE_ENDIAN
				swab64s(&tmp64);
				tmp64 >>= 16;
#else
				tmp64 <<= 16;
#endif
				memcpy((void *) &tbl_entry[i].mac,
				       (void *) &tmp64, 6);
				LOG_DEBUG("%d valid %u, mac=%pM port=%u "
					  "vid=%04u static=%u\n", i,
					  tbl_entry[i].valid,
					  tbl_entry[i].mac,
					  tbl_entry[i].port,
					  tbl_entry[i].vid,
					  tbl_entry[i].stat);
			}
		}
	}

EXIT:
	FUNC_LEAVE();
	if (search_ctl.start) {
		return( ! ETHSW_ARL_TBL_LAST_ENTRY);
	} else {
		search_in_progress = false;
		return(ETHSW_ARL_TBL_LAST_ENTRY);
	}
}


static int ethsw_multiport_get_idx(int idx, u64 *mac64, u16 *etype, u16 *port)
{
	int ret = 0;
	u16 mpctl, mpmode;
	u64 mpaddr;
	u32 mpvec;
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];

	FUNC_ENTER();

	if (!swdev)
		swdev = ethsw_dev[EXTERNAL_SPI_SW];

	if (swdev) {
		VALIDATE_ETH_PWR_STATUS(swdev);
		CHECK_EXIT(CREG_RD(
			PAGE_ARLCTL,
			CREG_MULTI_PORT_CTL,
			&mpctl));

		mpmode = CREG_FLD_IDX_GET(
			CREG_MULTI_PORT_CTL,
			MPORT_CTRL, idx,
			mpctl);

		if (mpmode != 0) {
			/* entry is enabled */
			CHECK_EXIT(CREG_IDX_RD(
				PAGE_ARLCTL,
				CREG_MULTIPORT_ADDR, idx,
				&mpaddr));

			if (port) {
				CHECK_EXIT(CREG_IDX_RD(
					PAGE_ARLCTL,
					CREG_MPORTVEC, idx,
					&mpvec));
			}
		}
		else {
			/* entry is disabled */
			mpaddr = 0;
			mpvec  = ETHSW_PORT_MAP_INVALID;
		}

		if (mac64) *mac64 = mpaddr & ((1LL << 48) - 1);
		if (etype) *etype = mpaddr >> 48;
		if (port)  *port  = mpvec; /* mp vec is 32-bit */
	}
 EXIT:
	FUNC_LEAVE();
	return(ret);
}

static int ethsw_multiport_set_idx(int idx, u64 mac64, u16 etype, u16 port)
{
	int ret = 0;
	u16 mpctl, mpmode;
	u64 mpaddr;
	u32 mpvec;
	/*ZACK*/
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];


	FUNC_ENTER();
	if (!swdev)
		swdev = ethsw_dev[EXTERNAL_SPI_SW];

	if (swdev) {
		VALIDATE_ETH_PWR_STATUS(swdev);
		mpaddr = mac64 | ((u64)etype << 48);
		mpvec  = port; /* mp vec is 32-bit */
		mpmode =
			(mpaddr == 0) ? 0 : /* disabled */
			(mac64  == 0) ? 1 : /* etype matching only */
			(etype  == 0) ? 2 : /* mac matching only */
			3;                  /* etype + mac matching */

		CHECK_EXIT(CREG_RD(
			PAGE_ARLCTL,
			CREG_MULTI_PORT_CTL,
			&mpctl));

		mpctl = CREG_FLD_IDX_SET(
			CREG_MULTI_PORT_CTL,
			MPORT_CTRL, idx,
			mpctl,
			mpmode);

		CHECK_EXIT(CREG_WR(
			PAGE_ARLCTL,
			CREG_MULTI_PORT_CTL,
			&mpctl));

		if (mpaddr != 0) {
			CHECK_EXIT(CREG_IDX_WR(
				PAGE_ARLCTL,
				CREG_MULTIPORT_ADDR, idx,
				&mpaddr));

			CHECK_EXIT(CREG_IDX_WR(
				PAGE_ARLCTL,
				CREG_MPORTVEC, idx,
				&mpvec));
		}
	}
 EXIT:
	FUNC_LEAVE();
	return(ret);
}

static int ethsw_multiport_find_idx(u64 mac64, u16 etype, int *idx)
{
	int ret = 0;
	int i;
	u16 mpctl, mpmode;
	u64 mpaddr, mpaddr_entry;
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];


	FUNC_ENTER();

	if (!swdev)
		swdev = ethsw_dev[EXTERNAL_SPI_SW];

	if (swdev) {
		VALIDATE_ETH_PWR_STATUS(swdev);
		mpaddr = mac64 | ((u64)etype << 48);

		CHECK_EXIT(CREG_RD(
			PAGE_ARLCTL,
			CREG_MULTI_PORT_CTL,
			&mpctl));

		for (i = 0; i < 6; i++) {
			mpmode = CREG_FLD_IDX_GET(
				CREG_MULTI_PORT_CTL,
				MPORT_CTRL, i,
				mpctl);

			if (mpmode != 0) {
				/* entry is enabled */
				CHECK_EXIT(CREG_IDX_RD(
					PAGE_ARLCTL,
					CREG_MULTIPORT_ADDR, i,
					&mpaddr_entry));

				if (mpaddr_entry == mpaddr) {
					if (idx) *idx = i;
					break;
				}
			}
			else {
				/* entry is disabled */
				if (mpaddr == 0) {
					if (idx) *idx = i;
					break;
				}
			}
		}

		if (i == 6) {
			/* return invalid if no entry found */
			if (idx) *idx = -1;
		}
	}
 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_multiport_find(u8 *mac, u16 etype, u16 *port)
{
	int ret = 0;
	int idx;
	u64 mac64;

	FUNC_ENTER();

	if (mac == NULL) {
		ret = -EINVAL;
		goto EXIT;
	}

	pack_mac_to_u64(&mac64, mac);

	CHECK_EXIT(ethsw_multiport_find_idx(mac64, etype, &idx));

	if (idx != -1) {
		/* entry found */
		CHECK_EXIT(ethsw_multiport_get_idx(idx, NULL, NULL, port));
	}
	else {
		/* entry not found, return invalid */
		*port = ETHSW_PORT_MAP_INVALID;
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_multiport_add(u8 *mac, u16 etype, u16 port)
{
	int ret = 0;
	int idx;
	u64 mac64;

	FUNC_ENTER();

	if (mac == NULL) {
		ret = -EINVAL;
		goto EXIT;
	}

	pack_mac_to_u64(&mac64, mac);

	CHECK_EXIT(ethsw_multiport_find_idx(mac64, etype, &idx));

	if (idx == -1) {
		/* entry not found, find 1st available entry */
		CHECK_EXIT(ethsw_multiport_find_idx(0, 0, &idx));
	}

	if (idx == -1) {
		/* no available entry */
		ret = -ENOMEM;
		goto EXIT;
	}

	/* set entry */
	CHECK_EXIT(ethsw_multiport_set_idx(idx, mac64, etype, port));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_multiport_delete(u8 *mac, u16 etype)
{
	int ret = 0;
	int idx;
	u64 mac64;

	FUNC_ENTER();

	if (mac == NULL) {
		ret = -EINVAL;
		goto EXIT;
	}

	pack_mac_to_u64(&mac64, mac);

	CHECK_EXIT(ethsw_multiport_find_idx(mac64, etype, &idx));

	if (idx != -1) {
		/* entry found, disable entry */
		CHECK_EXIT(ethsw_multiport_set_idx(idx, 0, 0, 0));
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_vlan_table_find(struct ethsw_device *swdev, u16 vid, u16 *untag_map, u16 *fwd_map)
{
	int ret = 0;
	u8  rwctl;
	u32 entry;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_VTBL_ADDR,
		&vid));

	rwctl =
		CREG_FLD_SHF(CREG_ARLA_VTBL_RWCTRL, VTBL_STDN,   1) |
		CREG_FLD_SHF(CREG_ARLA_VTBL_RWCTRL, VTBL_RW_CLR, 1);

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_VTBL_RWCTRL,
		&rwctl));

	do {
		udelay(10);

		CHECK_EXIT(CREG_RD(
			PAGE_ARLACCS,
			CREG_ARLA_VTBL_RWCTRL,
			&rwctl));

	} while (CREG_FLD_GET(
				 CREG_ARLA_VTBL_RWCTRL,
				 VTBL_STDN,
				 rwctl));


	CHECK_EXIT(CREG_RD(
		PAGE_ARLACCS,
		CREG_ARLA_VTBL_ENTRY,
		&entry));

	if (untag_map) {
		*untag_map = CREG_FLD_GET(
			CREG_ARLA_VTBL_ENTRY,
			UNTAG_MAP,
			entry);
	}

	if (fwd_map) {
		*fwd_map = CREG_FLD_GET(
			CREG_ARLA_VTBL_ENTRY,
			FWD_MAP,
			entry);
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_vlan_table_add( struct ethsw_device *swdev, u16 vid, u16 untag_map, u16 fwd_map)
{
	int ret = 0;
	u8  rwctl;
	u32 entry;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_VTBL_ADDR,
		&vid));

	entry =
		CREG_FLD_SHF(CREG_ARLA_VTBL_ENTRY, UNTAG_MAP, untag_map) |
		CREG_FLD_SHF(CREG_ARLA_VTBL_ENTRY, FWD_MAP,   fwd_map);

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_VTBL_ENTRY,
		&entry));

	rwctl =
		CREG_FLD_SHF(CREG_ARLA_VTBL_RWCTRL, VTBL_STDN,   1) |
		CREG_FLD_SHF(CREG_ARLA_VTBL_RWCTRL, VTBL_RW_CLR, 0);

	CHECK_EXIT(CREG_WR(
		PAGE_ARLACCS,
		CREG_ARLA_VTBL_RWCTRL,
		&rwctl));

	do {
		udelay(10);

		CHECK_EXIT(CREG_RD(
			PAGE_ARLACCS,
			CREG_ARLA_VTBL_RWCTRL,
			&rwctl));

	} while (CREG_FLD_GET(
				 CREG_ARLA_VTBL_RWCTRL,
				 VTBL_STDN,
				 rwctl));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_vlan_table_delete(struct ethsw_device *swdev, u16 vid)
{
	int ret = 0;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	CHECK_EXIT(ethsw_vlan_table_add(swdev, vid, 0, 0));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_vlan_port_tag_set(int port_id, u16 vid, u8 pcp)
{
	int ret = 0;
	u16 port_tag;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	port_tag =
		CREG_FLD_SHF(CREG_DEFAULT_1Q_TAG, VID, vid) |
		CREG_FLD_SHF(CREG_DEFAULT_1Q_TAG, PRI, pcp);

	CHECK_EXIT(CREG_IDX_WR(
		PAGE_BCM8021Q,
		CREG_DEFAULT_1Q_TAG, port_id - swdev->subid_port_offset,
		&port_tag));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_vlan_port_tag_set);

int ethsw_vlan_port_tag_get(int port_id, u16 *vid, u8 *pcp)
{
	int ret = 0;
	u16 port_tag;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	CHECK_EXIT(CREG_IDX_RD(
		PAGE_BCM8021Q,
		CREG_DEFAULT_1Q_TAG, port_id - swdev->subid_port_offset,
		&port_tag));

	if (vid) {
		*vid = CREG_FLD_GET(CREG_DEFAULT_1Q_TAG, VID, port_tag);
	}

	if (pcp) {
		*pcp = CREG_FLD_GET(CREG_DEFAULT_1Q_TAG, PRI, port_tag);
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_vlan_port_tag_get);


int ethsw_vlan_enable(u8 vlan_enable)
{
	int ret = 0;
	u8 reg_val8;
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];

	/* if we disable VLAN make sure that we save the disable value */
	/* and if anything else tries to setup VLANS that we do not do */
	/* the setup. This is for customers who want to do vconfig for Linux */
	/* but don't want the switch to be vlan configured. */
	if (vlan_enable == 0) {
		ethsw_vlan_disable = true;
	}
	if (swdev) {
		FUNC_ENTER();
		VALIDATE_ETH_PWR_STATUS(swdev);

		CHECK_EXIT(CREG_RD(
			PAGE_BCM8021Q,
			CREG_VLAN_CTRL0,
			&reg_val8));

		reg_val8 =CREG_FLD_SET(CREG_VLAN_CTRL0, VLAN_EN,	reg_val8, vlan_enable);

		CHECK_EXIT(CREG_WR(
			PAGE_BCM8021Q,
			CREG_VLAN_CTRL0,
			&reg_val8));
	}

	swdev = ethsw_dev[EXTERNAL_SPI_SW];
	if (swdev) {
		FUNC_ENTER();
		VALIDATE_ETH_PWR_STATUS(swdev);

		CHECK_EXIT(CREG_RD(
			PAGE_BCM8021Q,
			CREG_VLAN_CTRL0,
			&reg_val8));

		reg_val8 =CREG_FLD_SET(CREG_VLAN_CTRL0, VLAN_EN,	reg_val8, vlan_enable);

		CHECK_EXIT(CREG_WR(
			PAGE_BCM8021Q,
			CREG_VLAN_CTRL0,
			&reg_val8));
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_vlan_enable);

int ethsw_vlan_port_vid_add(int port_id, u16 vid, bool untag)
{
	int ret = 0;
	u16 untag_map = 0, fwd_map = 0;
	struct ethsw_device *swdev;

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	if (vid == 0 ||
		vid == 0xFFF ||
		port_id > ETHSW_PORT_MAX) {
		LOG_ERR("vid add Invalid parameters %d\n",port_id - swdev->subid_port_offset);
		ret = -EINVAL;
		goto EXIT;
	}

	CHECK_EXIT(ethsw_vlan_table_find(swdev, vid, &untag_map, &fwd_map));

	if (fwd_map & (1 << (port_id - swdev->subid_port_offset))) {
		goto EXIT;
	}
	else if (fwd_map == 0) {
		/* always include IMP ports */
		fwd_map   = swdev->imp_port_mask;
		untag_map = 0;
	}

	fwd_map   |= (1 << (port_id - swdev->subid_port_offset));
	if (untag)
		untag_map |= (1 << (port_id - swdev->subid_port_offset));

	CHECK_EXIT(ethsw_vlan_table_add(swdev, vid, untag_map, fwd_map));

	/* if we have turned OFF the vlans by command then you must first turn on VLANs gain */
	/* via the command  */
	if (!ethsw_vlan_disable) {
		/* If the vlan is anything other than the default vlan 1, then we need to turn on VLANS*/
		if (vid != 1) {
			ethsw_vlan_enable(1);
		}
	}
	else {
		LOG_INFO("ethsw_vlan_port_vid_add: MUST turn on VLANs using vlan command first for VLANS to take effect\n");
	}
 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_vlan_port_vid_add);

int ethsw_vlan_port_vid_delete(int port_id, u16 vid, bool untag)
{
	int ret = 0;
	u16 untag_map = 0, fwd_map = 0;
	struct ethsw_device *swdev;

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	if (vid == 0 ||
		vid == 0xFFF ||
		port_id > ETHSW_PORT_MAX) {
		LOG_ERR("vid delete Invalid parameters port=%d\n",port_id - swdev->subid_port_offset);
		ret = -EINVAL;
		goto EXIT;
	}

	CHECK_EXIT(ethsw_vlan_table_find(swdev, vid, &untag_map, &fwd_map));

	if ((fwd_map & (1 << (port_id- swdev->subid_port_offset))) == 0) {
		goto EXIT;
	}

	fwd_map   &= ~(1 << (port_id - swdev->subid_port_offset));
	if (untag)
		untag_map &= ~(1 << (port_id - swdev->subid_port_offset));

	if (vid != 1 &&
		fwd_map == swdev->imp_port_mask) {
		CHECK_EXIT(ethsw_vlan_table_delete(swdev, vid));
	}
	else {
		CHECK_EXIT(ethsw_vlan_table_add(swdev, vid, untag_map, fwd_map));
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_vlan_port_vid_delete);

bool ethsw_ipv6_mc_flood_disabled(void)
{
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];

	if (!swdev)
		swdev = ethsw_dev[EXTERNAL_SPI_SW];

	VALIDATE_SWDEV(swdev);

	return swdev->vlan_ipv6_mc_nf_mode_enabled;
}
EXPORT_SYMBOL(ethsw_ipv6_mc_flood_disabled);

int ethsw_vlan_enabled(void)
{
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];

	if (!swdev)
		swdev = ethsw_dev[EXTERNAL_SPI_SW];

	VALIDATE_SWDEV(swdev);


	return swdev->vlan_default_mode;
}
EXPORT_SYMBOL(ethsw_vlan_enabled);

int ethsw_mib_counters_clear(int port_id)
{
	int ret = 0;
	struct ethsw_device *swdev;


	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	if (swdev->port[port_id - swdev->subid_port_offset].type == INV_PORT) {
		ret = -EINVAL;
		goto EXIT;
	}

	ret = ethsw_mib_counters_reset(1<<(port_id - swdev->subid_port_offset));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_mib_counters_clear);

bool ethsw_drop_packet(int port, int start_port)
{
	/* NEED To run through all of the ports to see if one is enabled  */
	/* before it and if there was, then we return true to drop pkt*/
	struct ethsw_device *swdev = ethsw_dev[EXTERNAL_SPI_SW];
	int j;
	bool link_status;

	if (!swdev) {
		LOG_ERR("Ethsw switch device not initialized\n");
		return false;
	}

	for (j = start_port; j < port; j++) {
		link_status = swdev->port[j].status.link_up;
		if ((j != port)  && link_status) {
			return true;
		}
		if (j == port)
			return false;
	}
	return false;
}
EXPORT_SYMBOL(ethsw_drop_packet);

/****************************************************************************
  Port MIB Functions
****************************************************************************/

static int ethsw_mib_map[MIB_COUNTER_MAX] = {
	/* transmit counters */
	CREG_TX_OCTETS,
	0,
	CREG_TX_DROP_PKTS,
	CREG_TX_BROADCAST_PKTS,
	CREG_TX_MULTICAST_PKTS,
	CREG_TX_UNICAST_PKTS,
	CREG_TX_PAUSE_PKTS,
	CREG_TX_COLLISIONS,
	CREG_TX_SINGLE_COLLISION,
	CREG_TX_MULTIPLE_COLLISION,
	CREG_TX_LATE_COLLISION,
	CREG_TX_EXCESSIVE_COLLISION,
	CREG_TX_DEFERRED_TRANSMIT,
	CREG_TX_FRAME_IN_DISC,

	CREG_TX_Q0_PKT,
	CREG_TX_Q1_PKT,
	CREG_TX_Q2_PKT,
	CREG_TX_Q3_PKT,
	CREG_TX_Q4_PKT,
	CREG_TX_Q5_PKT,

	CREG_TX_Q6_PKT,
	CREG_TX_Q7_PKT,
	CREG_TX_PKTS_64_OCTETS,
	CREG_TX_PKTS_65TO127_OCTETS,
	CREG_TX_PKTS_128TO255_OCTETS,
	CREG_TX_PKTS_256TO511_OCTETS,
	CREG_TX_PKTS_512TO1023_OCTETS,
	CREG_TX_PKTS_1024TOMAX_OCTETS,

	/* receive counters */
	CREG_RX_OCTETS,
	0,
	CREG_RX_GOOD_OCTETS,
	0,
	CREG_RX_DROP_PKTS,
	CREG_RX_BROADCAST_PKTS,
	CREG_RX_MULTICAST_PKTS,
	CREG_RX_UNICAST_PKTS,
	CREG_RX_PAUSE_PKTS,
	CREG_RX_SA_CHANGES,

	CREG_RX_UNDERSIZE_PKTS,
	CREG_RX_PKTS_64_OCTETS,
	CREG_RX_PKTS_65TO127_OCTETS,
	CREG_RX_PKTS_128TO255_OCTETS,
	CREG_RX_PKTS_256TO511_OCTETS,
	CREG_RX_PKTS_512TO1023_OCTETS,
	CREG_RX_PKTS_1024TOMAX_OCTETS,
	CREG_RX_OVERSIZE_PKTS,
	CREG_RX_JUMBO_PKTS,

	CREG_RX_FRAGMENTS,
	CREG_RX_JABBERS,
	CREG_RX_DISCARD,
	CREG_RX_ALIGNMENT_ERRORS,
	CREG_RX_FCS_ERRORS,
	CREG_RX_SYMBOL_ERRORS,
	CREG_RX_IN_RANGE_ERRORS,
	CREG_RX_OUT_OF_RANGE_ERRORS,

	/* EEE counters */
	CREG_EEE_LPI_EVENT,
	CREG_EEE_LPI_DURATION
};

int ethsw_mib_counters_reset(u16 port_map)
{
	int ret = 0;
	u8  cfg;
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];

	FUNC_ENTER();

	if (swdev) {
		VALIDATE_ETH_PWR_STATUS(swdev);
		if (swdev->pdev) {
			/* only reset MIB counters for indicated ports */
			CHECK_EXIT(CREG_WR(
				PAGE_MNGMODE,
				CREG_RST_MIB_CNT_EN,
				&port_map));
		}
		CHECK_EXIT(CREG_RD(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&cfg));

		cfg = CREG_FLD_SET(
			CREG_GMNGCFG,
			RST_MIB_CNT,
			cfg,
			1);

		CHECK_EXIT(CREG_WR(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&cfg));

		cfg = CREG_FLD_SET(
			CREG_GMNGCFG,
			RST_MIB_CNT,
			cfg,
			0);

		CHECK_EXIT(CREG_WR(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&cfg));
	}
	swdev = ethsw_dev[EXTERNAL_SPI_SW];
	if (swdev) {
		VALIDATE_ETH_PWR_STATUS(swdev);
		if (swdev->sdev) {
			/* only reset MIB counters for indicated ports */
			CHECK_EXIT(CREG_WR(
				PAGE_MNGMODE,
				CREG_RST_MIB_CNT_EN,
				&port_map));
		}
		CHECK_EXIT(CREG_RD(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&cfg));

		cfg = CREG_FLD_SET(
			CREG_GMNGCFG,
			RST_MIB_CNT,
			cfg,
			1);

		CHECK_EXIT(CREG_WR(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&cfg));

		cfg = CREG_FLD_SET(
			CREG_GMNGCFG,
			RST_MIB_CNT,
			cfg,
			0);

		CHECK_EXIT(CREG_WR(
			PAGE_MNGMODE,
			CREG_GMNGCFG,
			&cfg));
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}

int ethsw_mib_counter32_get(int port_id, int mib_counter, u32 *value)
{
	int ret = 0;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	if (port_id > ETHSW_PORT_MAX ||
		mib_counter >= MIB_COUNTER_MAX) {
		LOG_ERR("mib 32 get Invalid parameters port=%d\n",port_id - swdev->subid_port_offset);
		ret = -EINVAL;
		goto EXIT;
	}

	if (!value) {
		LOG_ERR("Data pointer is NULL\n");
		ret = -EINVAL;
		goto EXIT;
	}

	if (mib_counter == MIB_TX_OCTETS ||
		mib_counter == MIB_RX_OCTETS ||
		mib_counter == MIB_RX_GOOD_OCTETS) {
		LOG_ERR("MIB counter is 64-bit wide\n");
		ret = -EINVAL;
		goto EXIT;
	}

	ret = ethsw_creg_read(
		swdev,
		PAGE_MIB_PORT_0 + port_id - swdev->subid_port_offset,
		ethsw_mib_map[mib_counter],
		(u8 *)value, 4);

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_mib_counter32_get);

int ethsw_mib_counter64_get(int port_id, int mib_counter, u64 *value)
{
	int ret = 0;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	if (port_id > ETHSW_PORT_MAX ||
		mib_counter >= MIB_COUNTER_MAX) {
		LOG_ERR("mib 64 get Invalid parameters port=%d\n",port_id - swdev->subid_port_offset);
		ret = -EINVAL;
		goto EXIT;
	}


	if (!value) {
		LOG_ERR("Data pointer is NULL\n");
		ret = -EINVAL;
		goto EXIT;
	}

	if (mib_counter != MIB_TX_OCTETS &&
		mib_counter != MIB_RX_OCTETS &&
		mib_counter != MIB_RX_GOOD_OCTETS) {
		u32 value_lo;

		/* 32-bit counter */
		ret = ethsw_creg_read(
			swdev,
			PAGE_MIB_PORT_0 + port_id - swdev->subid_port_offset,
			ethsw_mib_map[mib_counter],
			(u8 *)&value_lo, 4);

		/* copy to 64-bit output */
		*value = value_lo;
	}
	else {
		/* 64-bit counter */
		ret = ethsw_creg_read(
			swdev,
			PAGE_MIB_PORT_0 + port_id - swdev->subid_port_offset,
			ethsw_mib_map[mib_counter],
			(u8 *)value, 8);
	}

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_mib_counter64_get);

int ethsw_mib_counters_get(int port_id, int start, int count, u32 *values)
{
	struct ethsw_device *swdev;

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	return ethsw_mib_counters_get_base(swdev, port_id - swdev->subid_port_offset, start, count, values);
}

int ethsw_mib_counters_get_base(struct ethsw_device *swdev, int port_id, int start, int count, u32 *values)
{
	int ret = 0;
	u8  ctl;
	int i;
	uint max_count = 0;
	struct ethsw_core *swcore;

	FUNC_ENTER();
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);
	swcore = swdev->swcore;
	VALIDATE_SWCORE(swdev);

	if (port_id > ETHSW_PORT_MAX ||
		start + count > MIB_COUNTER_MAX) {
		LOG_ERR("mib get Invalid parameters port=%d\n",port_id);
		ret = -EINVAL;
		goto EXIT2;
	}

	if (!values) {
		LOG_ERR("Data pointer is NULL\n");
		ret = -EINVAL;
		goto EXIT2;
	}

	/* lock MIB snapshot access */
	spin_lock(&swcore->mib_snapshot_lock);

	/* start snapshot */
	ctl =
		CREG_FLD_SHF(CREG_MIB_SNAPSHOT_CTL, SNAPSHOT_STDONE, 1) |
		CREG_FLD_SHF(CREG_MIB_SNAPSHOT_CTL, SNAPSHOT_PORT,   port_id);

	CHECK_EXIT(CREG_WR(
		PAGE_MIB_SNAPSHOT,
		CREG_MIB_SNAPSHOT_CTL,
		&ctl));

	/* wait for snapshot done */
	do {
		udelay(10);

		CHECK_EXIT(CREG_RD(
			PAGE_MIB_SNAPSHOT,
			CREG_MIB_SNAPSHOT_CTL,
			&ctl));

		max_count++;
		if (max_count > 100) {
			LOG_ERR("MIB SNAPSHOT Timeout Error\n");
			goto EXIT;
		}
	} while (CREG_FLD_GET(
				 CREG_MIB_SNAPSHOT_CTL,
				 SNAPSHOT_STDONE,
				 ctl));

	/* read snapshot MIB counters */
	for (i = start; i < start + count; i++) {
		if (i == MIB_TX_OCTETS_RSV ||
			i == MIB_RX_OCTETS_RSV ||
			i == MIB_RX_GOOD_OCTETS_RSV) {
			/* reserved for 64-bit counter */
			values++;
			continue;
		} else if (i == MIB_TX_OCTETS ||
				 i == MIB_RX_OCTETS ||
				 i == MIB_RX_GOOD_OCTETS) {
			/* 64-bit counter */
			ret = ethsw_creg_read(
				swdev,
				PAGE_MIB_SNAPSHOT_PORT,
				ethsw_mib_map[i],
				(u8 *)(values++), 8);
		} else {
			/* 32-bit counter */
			ret = ethsw_creg_read(
				swdev,
				PAGE_MIB_SNAPSHOT_PORT,
				ethsw_mib_map[i],
				(u8 *)(values++), 4);
		}

		if (ret) {
			goto EXIT;
		}
	}

 EXIT:
	spin_unlock(&swcore->mib_snapshot_lock);

 EXIT2:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_mib_counters_get);

int ethsw_arl_learning_enable(bool enable)
{
	int i, ret = 0;
	u16 learn;
	u32 ageing;
	u16  reg_val16;
	u8  reg_val8;
	u8  imp_ports;
	u8  imp_idx;
	u16 imp_port_mask[ETHSW_PORT_MAX] = {0};
	struct ethsw_device *swdev = ethsw_dev[INTERNAL_SW];
	FUNC_ENTER();

	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	if (enable) {
		learn = 0x0;
		ageing = 300; /* 300s ageing */

		// Set lan ports to their defaults for talking to other ports
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			/* exclude all IMP ports as egress ports */
			if (swdev->port[i].type != IMP_PORT) {
				reg_val16 = 0x1FF;
				CHECK_EXIT(CREG_IDX_WR(
				   PAGE_VLAN,
				   CREG_PORT_VLAN_CTL, i,
				   &reg_val16));
			}
		}

		/* use trunk group if there are more than 2 IMP ports */
		if (swdev->imp_port_mask & 0xFF) {
			/* enable trunk groups */
			reg_val8 =
				CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, EN_TRUNK_LOCAL, 1) |
				CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, HASH_SEL,       0);

			CHECK_EXIT(CREG_WR(
				PAGE_TRUNK,
				CREG_MAC_TRUNK_CTL,
				&reg_val8));
		}

	} else {
		/* Disable learning on all ports except IMP ports */
		learn = 0x1FF ^ swdev->imp_port_mask;
		ageing = 1;   /* 1s ageing */

		if (swdev->imp_port_mask & 0xFF) {
			/* disable trunk groups */
			reg_val8 =
				CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, EN_TRUNK_LOCAL, 0) |
				CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, HASH_SEL,       0);

			CHECK_EXIT(CREG_WR(
				PAGE_TRUNK,
				CREG_MAC_TRUNK_CTL,
				&reg_val8));

			imp_ports = 0;
			for (i = 0; i < ETHSW_PORT_MAX; i++) {
				if (swdev->port[i].type == IMP_PORT) {
					imp_port_mask[imp_ports] = 1 << i;
					imp_ports++;
				}
			}

			imp_idx = 0;
			for (i = 0; i < ETHSW_PORT_MAX; i++) {
				if ((swdev->port[i].type != INV_PORT) &&
				    (swdev->port[i].type != IMP_PORT)) {
					reg_val16 = imp_port_mask[imp_idx];
					CHECK_EXIT(CREG_IDX_WR(
					   PAGE_VLAN,
					   CREG_PORT_VLAN_CTL, i,
					   &reg_val16));
					imp_idx++;
					if (imp_idx >= imp_ports)
						imp_idx = 0;
				}
			}
		} else {
			for (i = 0; i < ETHSW_PORT_MAX; i++) {
				/* Port -> Imp 8 */
				if (swdev->port[i].type != IMP_PORT) {
					reg_val16 = 1 << 8;
					CHECK_EXIT(CREG_IDX_WR(
					   PAGE_VLAN,
					   CREG_PORT_VLAN_CTL, i,
					   &reg_val16));
				}
			}
		}
	}
	CHECK_EXIT(CREG_WR(
	   PAGE_CTLREG,
	   CREG_HW_LEARN,
	   &learn));
	CHECK_EXIT(CREG_WR(
	   PAGE_CTLREG,
	   CREG_SW_LEARN,
	   &learn));
	CHECK_EXIT(CREG_WR(
	   PAGE_MNGMODE,
	   CREG_SPTAGT,
	   &ageing));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_arl_learning_enable);

int ethsw_arl_swlearn_enable(u16 port, bool enable)
{
	int ret = 0;
	u16 learn;
	struct ethsw_device *swdev;
	FUNC_ENTER();

	swdev = subid_to_swdev(port);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	CHECK_EXIT(CREG_RD(
	   PAGE_CTLREG,
	   CREG_SW_LEARN,
	   &learn));

	if (enable) {
		learn |= (1 << port);
	} else {
		learn &= ~(1 << port);
	}
	CHECK_EXIT(CREG_WR(
	   PAGE_CTLREG,
	   CREG_SW_LEARN,
	   &learn));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_arl_swlearn_enable);

int ethsw_arl_hwlearn_enable(u16 port, bool enable)
{
	int ret = 0;
	u16 learn;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	CHECK_EXIT(CREG_RD(
	   PAGE_CTLREG,
	   CREG_HW_LEARN,
	   &learn));

	if (enable) {
		learn &= ~(1 << port);
	} else {
		learn |= (1 << port);
	}
	CHECK_EXIT(CREG_WR(
	   PAGE_CTLREG,
	   CREG_HW_LEARN,
	   &learn));

 EXIT:
	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_arl_hwlearn_enable);

int ethsw_switch_set_port_stp_state(
	int port_id,
	unsigned int state)
{
	int ret = 0;
	struct ethsw_device *swdev;

	FUNC_ENTER();

	swdev = subid_to_swdev(port_id);
	VALIDATE_SWDEV(swdev);
	VALIDATE_ETH_PWR_STATUS(swdev);

	/* Call platform specific call just in case there are */
	/* differences between hardware */
	if (swdev->ethsw_switch_set_port_stp_state_specific) {
		ret = swdev->ethsw_switch_set_port_stp_state_specific(swdev, port_id - swdev->subid_port_offset, state);
	}

	FUNC_LEAVE();
	return(ret);
}
EXPORT_SYMBOL(ethsw_switch_set_port_stp_state);

int ethsw_creg_write(struct ethsw_device *swdev,
		     int page_num, int reg_addr, u8 *data, int size)
{
	return (swdev->ethsw_creg_write)(swdev, page_num, reg_addr, data, size);
}

int ethsw_creg_read(struct ethsw_device *swdev,
		    int page_num, int reg_addr, u8 *data, int size)
{
	return (swdev->ethsw_creg_read)(swdev, page_num, reg_addr, data, size);
}

bool hash_collision_fix_enabled(void) {
	return hash_fix_enabled;
}
int hash_collision_fix_enable(struct ethsw_device *swdev, bool enable) {

	int ret = 0;
	int i;
	u8  reg_val8;
	u16 reg_val16;

	if (swdev->imp_port_count == 2) {

		/* Disable HW Learning on All ports */
		reg_val16 = 0x1FF;
		CHECK_EXIT(CREG_WR(
			PAGE_CTLREG,
			CREG_HW_LEARN,
			&reg_val16));
		/* Flush ARL Table */
		for (i = 0; i < ETHSW_PORT_MAX; i++)
			ethsw_arl_fast_age_set(i);

		if (enable && swdev->swcore->altwan_enble == false) {
			hash_fix_enabled = true;

			/* Disable trunk groups */
			reg_val8 =
				CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, EN_TRUNK_LOCAL, 0) |
				CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, HASH_SEL,       0);

			CHECK_EXIT(CREG_WR(
			   PAGE_TRUNK,
			   CREG_MAC_TRUNK_CTL,
			   &reg_val8));

			/* Remove all IMP ports from trunk group 0 */
			reg_val16 = 0x0;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_TRUNK,
			   CREG_TRUNK_GRP_CTL, 0,
			   &reg_val16));

			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_TRUNK,
			   CREG_IMP0_GRP_CTL, 0,
			   &reg_val16));

			for (i = 0; i < 5; i++) {
			   /* Include IMP port 5 (skip 8) */
			   reg_val16 = 0x1FF ^ (1 << 8);
			   CHECK_EXIT(CREG_IDX_WR(
				  PAGE_VLAN,
				  CREG_PORT_VLAN_CTL, i,
				  &reg_val16));
			}
			/* setup IMP port 7 port based VLAN*/
			/* Include IMP port 8 (skip 5) */
			reg_val16 = 0x1FF ^ (1 << 5);
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, 7,
			   &reg_val16));

			/* disable cross talk among IMP ports */
			for (i = 0; i < ETHSW_PORT_MAX; i++) {
			   if (swdev->port[i].type == IMP_PORT) {
				  /* exclude all IMP ports as egress ports */
				  reg_val16 = 0x1FF ^ swdev->imp_port_mask;
				  CHECK_EXIT(CREG_IDX_WR(
					 PAGE_VLAN,
					 CREG_PORT_VLAN_CTL, i,
					 &reg_val16));
			   }
			}

			/* DISABLE HW learning only for IMP port 5 and 8*/
			reg_val16 = 0x120;
			CHECK_EXIT(CREG_WR(
				PAGE_CTLREG,
				CREG_HW_LEARN,
				&reg_val16));

			 /* setup to forward unicast and multicast packets when ARL MISS*/
			 reg_val8 = (CREG_NEW_CTRL_MC_FWD_EN_MASK | CREG_NEW_CTRL_UC_FWD_EN_MASK | 1);
			 CHECK_EXIT(CREG_WR(
				CREG_G_PCTL,
				CREG_NEW_CTRL,
				&reg_val8));
			 /* Send unicast to ports 5 and 8 */
			reg_val16 = swdev->imp_port_mask;
			CHECK_EXIT(CREG_WR(
				CREG_G_PCTL,
				CREG_ULF_DROP_MAP,
				&reg_val16));

			 /* Send mutlicast to all ports */
			 reg_val16 = 0x1ff;
			 CHECK_EXIT(CREG_WR(
				CREG_G_PCTL,
				CREG_MLF_DROP_MAP,
				&reg_val16));

			 CHECK_EXIT(CREG_WR(
				CREG_G_PCTL,
				CREG_MLF_IPMC_FWD_MAP,
				&reg_val16));
      } else {
		hash_fix_enabled = false;

	   	/* enable trunk groups */
	   	reg_val8 =
	   		CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, EN_TRUNK_LOCAL, 1) |
	   		CREG_FLD_SHF(CREG_MAC_TRUNK_CTL, HASH_SEL,       0);

	   	CHECK_EXIT(CREG_WR(
	   		PAGE_TRUNK,
	   		CREG_MAC_TRUNK_CTL,
	   		&reg_val8));

	   	/* assign all IMP ports to trunk group 0 */
	   	reg_val16 = swdev->imp_port_mask;

	   	CHECK_EXIT(CREG_IDX_WR(
	   		PAGE_TRUNK,
	   		CREG_TRUNK_GRP_CTL, 0,
	   		&reg_val16));

	   	CHECK_EXIT(CREG_IDX_WR(
		   PAGE_TRUNK,
		   CREG_IMP0_GRP_CTL, 0,
		   &reg_val16));

		// Set lan ports to their defaults for talking to other ports
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			/* exclude all IMP ports as egress ports */
			if (swdev->port[i].type != IMP_PORT) {
				reg_val16 = 0x1FF;
				CHECK_EXIT(CREG_IDX_WR(
				   PAGE_VLAN,
				   CREG_PORT_VLAN_CTL, i,
				   &reg_val16));
			}
		}
		/* disable cross talk among IMP ports */
		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			if (swdev->port[i].type == IMP_PORT) {
				/* exclude all IMP ports as egress ports */
				reg_val16 = 0x1FF ^ swdev->imp_port_mask;
				CHECK_EXIT(CREG_IDX_WR(
				   PAGE_VLAN,
				   CREG_PORT_VLAN_CTL, i,
				   &reg_val16));
			}
		}

		/* enable HW learning for all ports*/
		reg_val16 = 0x0;
		CHECK_EXIT(CREG_WR(
		   PAGE_CTLREG,
		   CREG_HW_LEARN,
		   &reg_val16));

		/* reset uniicast drop map back to default value */
		reg_val16 = 0x0;
			CHECK_EXIT(CREG_WR(
			CREG_G_PCTL,
			CREG_ULF_DROP_MAP,
			&reg_val16));

		/* reset mutlicast drop map back to default value */
		 reg_val16 = 0x0;
		 CHECK_EXIT(CREG_WR(
			CREG_G_PCTL,
			CREG_MLF_DROP_MAP,
			&reg_val16));

		 /* reset MFL_IPMC back to default value*/
		reg_val16 = 0x0;
		CHECK_EXIT(CREG_WR(
		   CREG_G_PCTL,
		   CREG_MLF_IPMC_FWD_MAP,
		   &reg_val16));

		/* setup to drop unicast and multicast packets when ARL MISS*/
		reg_val8 =  1;
		CHECK_EXIT(CREG_WR(
		   CREG_G_PCTL,
		   CREG_NEW_CTRL,
		   &reg_val8));
	  }
	}
	EXIT:
	return(ret);
}

bool reserved_multicast_enabled(void) {
	return reserved_multicast_fix_enabled;
}
int reserved_multicast_enable(struct ethsw_device *swdev, bool enable) {


	int ret = 0;
    u8  reg_val8;
    u16 reg_val16;

	if (enable) {
		reserved_multicast_fix_enabled = true;

		reg_val16 = 0x121;
		CHECK_EXIT(CREG_IDX_WR(
		   PAGE_VLAN,
		   CREG_PORT_VLAN_CTL, CREG_VLAN_P0,
		   &reg_val16));

		reg_val16 = 0x122;
		CHECK_EXIT(CREG_IDX_WR(
		   PAGE_VLAN,
		   CREG_PORT_VLAN_CTL, CREG_VLAN_P1,
		   &reg_val16));

		reg_val16 = 0x124;
		CHECK_EXIT(CREG_IDX_WR(
		   PAGE_VLAN,
		   CREG_PORT_VLAN_CTL, CREG_VLAN_P2,
		   &reg_val16));

		reg_val16 = 0x128;
		CHECK_EXIT(CREG_IDX_WR(
		   PAGE_VLAN,
		   CREG_PORT_VLAN_CTL, CREG_VLAN_P3,
		   &reg_val16));

		reg_val16 = 0x130;
		CHECK_EXIT(CREG_IDX_WR(
		   PAGE_VLAN,
		   CREG_PORT_VLAN_CTL, CREG_VLAN_P4,
		   &reg_val16));

		reg_val16 = 0x5f;
		CHECK_EXIT(CREG_IDX_WR(
		   PAGE_VLAN,
		   CREG_PORT_VLAN_CTL, CREG_VLAN_P5,
		   &reg_val16));

		reg_val16 = 0x5f;
		CHECK_EXIT(CREG_IDX_WR(
		   PAGE_VLAN,
		   CREG_PORT_VLAN_CTL, CREG_VLAN_IMP,
		   &reg_val16));

		reg_val8 = 0x8;
		CHECK_EXIT(CREG_WR(
			PAGE_TRUNK,
			CREG_MAC_TRUNK_CTL,
			&reg_val8));

		if (swdev->imp_port_count == 2) {

			reg_val16 = 0x120;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P7,
			   &reg_val16));


			reg_val16 = 0x120;
			CHECK_EXIT(CREG_IDX_WR(
				PAGE_TRUNK,
				CREG_TRUNK_GRP_CTL, 0,
				&reg_val16));

		} else {

		  	reg_val16 = 0x5f;
		  	CHECK_EXIT(CREG_IDX_WR(
		  	   PAGE_VLAN,
		  	   CREG_PORT_VLAN_CTL, CREG_VLAN_P7,
		  	   &reg_val16));

		  	reg_val16 = 0x1a0;
		  	CHECK_EXIT(CREG_IDX_WR(
		  		PAGE_TRUNK,
		  		CREG_TRUNK_GRP_CTL, 0,
		  		&reg_val16));
		}
	}
	else {
		reserved_multicast_fix_enabled = false;

		if (swdev->imp_port_count == 2) {

			reg_val16 = 0x20;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P0,
			   &reg_val16));

			reg_val16 = 0xff;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P1,
			   &reg_val16));


			reg_val16 = 0x100;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P2,
			   &reg_val16));

			reg_val16 = 0xff;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P3,
			   &reg_val16));

			reg_val16 = 0x20;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P4,
			   &reg_val16));

			reg_val16 = 0xdf;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P5,
			   &reg_val16));

			reg_val16 = 0x100;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P7,
			   &reg_val16));

			reg_val16 = 0xdf;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_IMP,
			   &reg_val16));

			reg_val8 = 0x0;
			CHECK_EXIT(CREG_WR(
				PAGE_TRUNK,
				CREG_MAC_TRUNK_CTL,
				&reg_val8));

			reg_val16 = 0x000;

			CHECK_EXIT(CREG_IDX_WR(
			  PAGE_TRUNK,
			  CREG_TRUNK_GRP_CTL, 0,
			  &reg_val16));


		} else {
			/* non 2.5G lan port */
			reg_val16 = 0x20;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P0,
			   &reg_val16));

			reg_val16 = 0x80;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P1,
			   &reg_val16));

			reg_val16 = 0x100;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P2,
			   &reg_val16));

			reg_val16 = 0x20;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P3,
			   &reg_val16));

			reg_val16 = 0x1ff;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P4,
			   &reg_val16));

			reg_val16 = 0x5f;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_P5,
			   &reg_val16));

		  	reg_val16 = 0x5f;
		  	CHECK_EXIT(CREG_IDX_WR(
		  	   PAGE_VLAN,
		  	   CREG_PORT_VLAN_CTL, CREG_VLAN_P7,
		  	   &reg_val16));

			reg_val16 = 0x5f;
			CHECK_EXIT(CREG_IDX_WR(
			   PAGE_VLAN,
			   CREG_PORT_VLAN_CTL, CREG_VLAN_IMP,
			   &reg_val16));

			reg_val8 = 0x00;
			CHECK_EXIT(CREG_WR(
				PAGE_TRUNK,
				CREG_MAC_TRUNK_CTL,
				&reg_val8));

		  	reg_val16 = 0x1a0;
		  	CHECK_EXIT(CREG_IDX_WR(
		  		PAGE_TRUNK,
		  		CREG_TRUNK_GRP_CTL, 0,
		  		&reg_val16));
		}
	}
EXIT:
	return ret;
}

