 /****************************************************************************
 *
 * Copyright (c) 2020 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/module.h>
#include <net/switchdev.h>
#include <linux/if_bridge.h>
#include <linux/log2.h>
#include <linux/etherdevice.h>
#include <linux/if_vlan.h>
#include "dqnet_priv.h"
#include "dqnet_fap.h"

static struct workqueue_struct *swdev_wq;

static int set_stp(struct net_device *dev, uint32_t state,
		   struct switchdev_trans *trans)
{

	pr_debug("%s %d\n", dev->name, state);
	if (switchdev_trans_ph_prepare(trans)) {
		return 0;
	}

	return dqnet_fap_set_stp(dev, state);
}

static int set_ageing(struct net_device *dev,
		      clock_t ageing,
		      struct switchdev_trans *trans)
{
	pr_debug("%s %ld\n", dev->name, ageing);
	if (switchdev_trans_ph_prepare(trans)) {
		return 0;
	}

	return dqnet_fap_set_ageing(dev, ageing);
}

static int fdb_add(struct net_device *dev,
		   struct switchdev_notifier_fdb_info *info)
{
	return dqnet_fap_fdb_add(dev, info->vid, info->addr);
}

static int fdb_del(struct net_device *dev,
		   struct switchdev_notifier_fdb_info *info)
{
	return dqnet_fap_fdb_del(dev, info->vid, info->addr);
}

static int port_attr_set(struct net_device *dev,
			 const struct switchdev_attr *attr,
			 struct switchdev_trans *trans)
{
	int err = 0;

	pr_debug("%s %d\n", dev->name, attr->id);
	switch (attr->id) {
	case SWITCHDEV_ATTR_ID_PORT_STP_STATE:
		err = set_stp(dev, attr->u.stp_state, trans);
		break;
	case SWITCHDEV_ATTR_ID_PORT_BRIDGE_FLAGS:
		break;
	case SWITCHDEV_ATTR_ID_BRIDGE_AGEING_TIME:
		err = set_ageing(dev, attr->u.ageing_time, trans);
		break;
	default:
		err = -EOPNOTSUPP;
		break;
	}

	return err;
}

static int
swdev_port_attr_set_event(struct net_device *netdev,
			  struct switchdev_notifier_port_attr_info *port_attr_info)
{
	int err;

	err = port_attr_set(netdev, port_attr_info->attr,
			    port_attr_info->trans);

	port_attr_info->handled = true;
	return notifier_from_errno(err);
}

struct swdev_event_work {
	struct work_struct work;
	struct switchdev_notifier_fdb_info fdb_info;
	struct net_device *dev;
	unsigned long event;
};

static void
fdb_offload_notify(struct net_device *netdev,
		   struct switchdev_notifier_fdb_info *recv_info)
{
	struct switchdev_notifier_fdb_info info;

	info.addr = recv_info->addr;
	info.vid = recv_info->vid;
	info.offloaded = true;
	call_switchdev_notifiers(SWITCHDEV_FDB_OFFLOADED,
				 netdev, &info.info, NULL);
}

static void swdev_event_work(struct work_struct *work)
{
	struct swdev_event_work *swdev_work =
		container_of(work, struct swdev_event_work, work);
	struct net_device *dev = swdev_work->dev;
	struct switchdev_notifier_fdb_info *fdb_info;
	int err;

	rtnl_lock();
	switch (swdev_work->event) {
	case SWITCHDEV_FDB_ADD_TO_DEVICE:
		fdb_info = &swdev_work->fdb_info;

		/* Do not add interface own MAC address */
		if (ether_addr_equal(dev->dev_addr, fdb_info->addr))
			break;

		err = fdb_add(dev, fdb_info);
		if (err) {
			netdev_dbg(dev, "fdb add failed err=%d\n", err);
			break;
		}
		fdb_offload_notify(dev, fdb_info);
		break;
	case SWITCHDEV_FDB_DEL_TO_DEVICE:
		fdb_info = &swdev_work->fdb_info;
		if (!fdb_info->offloaded)
			break;
		err = fdb_del(dev, fdb_info);
		if (err)
			netdev_dbg(dev, "fdb del failed err=%d\n", err);
		break;
	}
	rtnl_unlock();

	kfree(swdev_work->fdb_info.addr);
	kfree(swdev_work);
	dev_put(dev);
}

/* called under rcu_read_lock() */
static int swdev_event(struct notifier_block *unused,
		       unsigned long event, void *ptr)
{
	struct net_device *dev = switchdev_notifier_info_to_dev(ptr);
	struct swdev_event_work *swdev_work;
	struct switchdev_notifier_fdb_info *fdb_info = ptr;

	if (!dev->dev_port || is_vlan_dev(dev) || netif_is_bond_master(dev))
		return NOTIFY_DONE;

	netdev_dbg(dev, "event %ld\n", event);

	if (event == SWITCHDEV_PORT_ATTR_SET)
		return swdev_port_attr_set_event(dev, ptr);

	swdev_work = kzalloc(sizeof(*swdev_work), GFP_ATOMIC);
	if (WARN_ON(!swdev_work))
		return NOTIFY_BAD;

	INIT_WORK(&swdev_work->work, swdev_event_work);
	swdev_work->dev = dev;
	swdev_work->event = event;

	switch (event) {
	case SWITCHDEV_FDB_ADD_TO_DEVICE: /* fall through */
	case SWITCHDEV_FDB_DEL_TO_DEVICE:
		memcpy(&swdev_work->fdb_info, ptr,
		       sizeof(swdev_work->fdb_info));
		swdev_work->fdb_info.addr = kzalloc(ETH_ALEN, GFP_ATOMIC);
		if (unlikely(!swdev_work->fdb_info.addr)) {
			kfree(swdev_work);
			return NOTIFY_BAD;
		}

		ether_addr_copy((u8 *)swdev_work->fdb_info.addr,
				fdb_info->addr);
		/* Take a reference on the device */
		dev_hold(dev);
		break;
	default:
		kfree(swdev_work);
		return NOTIFY_DONE;
	}

	queue_work(swdev_wq, &swdev_work->work);
	return NOTIFY_DONE;
}

static int swdev_blocking_event(struct notifier_block *unused,
				unsigned long event, void *ptr)
{
	struct net_device *dev = switchdev_notifier_info_to_dev(ptr);

	if (!dev->dev_port || is_vlan_dev(dev) || netif_is_bond_master(dev))
		return NOTIFY_DONE;

	netdev_dbg(dev, "event %ld\n", event);

	switch (event) {
	case SWITCHDEV_PORT_ATTR_SET:
		return swdev_port_attr_set_event(dev, ptr);
	}

	return NOTIFY_DONE;
}

static struct notifier_block swdev_notifier = {
	.notifier_call = swdev_event,
};

static struct notifier_block swdev_blocking_notifier = {
	.notifier_call = swdev_blocking_event,
};

int dqnet_swdev_init(void)
{
	swdev_wq = alloc_ordered_workqueue("dqnet_swdev", WQ_MEM_RECLAIM);
	register_switchdev_notifier(&swdev_notifier);
	register_switchdev_blocking_notifier(&swdev_blocking_notifier);
	return 0;
}

void dqnet_swdev_exit(void)
{
	unregister_switchdev_notifier(&swdev_notifier);
	unregister_switchdev_notifier(&swdev_blocking_notifier);
	destroy_workqueue(swdev_wq);
}
