/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2017 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Broadcom virtual flash block IO RPC interface.
 *
 * Author: Tim Ross <tim.ross@broadcom.com>
 *****************************************************************************/
#ifndef _VFBIO_RPC_H_
#define _VFBIO_RPC_H_

#ifdef __KERNEL__	/* Linux kernel */
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#endif

#define VFBIO_LUN_MAX		(1 << 8)
#define VFBIO_LUN_INFO_NAME_MAX	16
#define SYS_FUNC_MAX	2

enum vfbio_func_idx {
	VFBIO_FUNC_ALLOC = 0,
	VFBIO_FUNC_FREE,
	VFBIO_FUNC_LUN_ID,
	VFBIO_FUNC_LUN_INFO,
	VFBIO_FUNC_READ,
	VFBIO_FUNC_WRITE,
	VFBIO_FUNC_DISCARD,
	VFBIO_FUNC_FLUSH,
	VFBIO_FUNC_SG_READ,
	VFBIO_FUNC_SG_WRITE,
	VFBIO_FUNC_ASYNC_DISCARD,
	VFBIO_FUNC_ASYNC_FLUSH,
	VFBIO_FUNC_ASYNC_COMPLETE,
	VFBIO_FUNC_DEVICE_HEALTH,
	VFBIO_FUNC_EMMC_STANDARD_INFO,
	VFBIO_FUNC_EMMC_PROP_INFO,
	VFBIO_FUNC_GET_ATTR,
	VFBIO_FUNC_SET_ATTR,
	VFBIO_FUNC_MAX
};

enum sys_func_idx {
	SYS_FUNC_SET_LUN_NEXT_BOOT_STATE = 23,
	SYS_FUNC_GET_LUN_NEXT_BOOT_STATE = 24
};

/*
 * Sys service, LUN state info
 * W1: | 31..24 | 23..9 |   8    | 7..0 |
 *     |   RC   | RSVD  | State  |  LUN |
 *
 * W2: | 31..0 |
 *     |  RSVD |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * State:	0: Operational, 1: Backup
 * RSVD:	Reserved
 * LUN:		Logical unit #
 */
static inline bool sys_msg_get_lun_state(rpc_msg *msg)
{
	return ((msg->data[0] >> 8) & 0x1)?true:false;
}

/*
 * Sys service, LUN state info
 * W1: | 31..24 |  10  |   9  |   8    | 7..0 |
 *     |   RC   |FIRST | LAST | State  |  LUN |
 *
 * W2: | 31..0 |
 *     |  RSVD |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * State:	0: Operational, 1: Backup
 * First:	Flag to specific as First message
 *			(Discard Previous LUN state message if set)
 * Last:	Flag to specific as Last message(Write to Non-Vol if set)
 * RSVD:	Reserved
 * LUN:		Logical unit #
 */
static inline void sys_msg_set_lunswap_flag(rpc_msg *msg, bool first, bool last)
{
	msg->data[0] = (msg->data[0] & ~(0x3 << 9)) |
		((last | (first << 1)) << 9);
}

static inline void sys_msg_set_lunswap_state(rpc_msg *msg, bool v)
{
	msg->data[0] = (msg->data[0] & ~(0x1 << 8)) | (v << 8);
}

static inline u8 vfbio_msg_get_retcode(rpc_msg *msg)
{
	return ((msg->data[0] >> 24) & 0xff);
}
static inline void vfbio_msg_set_retcode(rpc_msg *msg, u8 v)
{
	msg->data[0] = (msg->data[0] & ~(0xff << 24)) | (v << 24);
}

static inline u8 vfbio_msg_get_lun(rpc_msg *msg)
{
	return (msg->data[0] & 0xff);
}
static inline void vfbio_msg_set_lun(rpc_msg *msg, u8 v)
{
	msg->data[0] = (msg->data[0] & ~0xff) | v;
}
static inline void vfbio_msg_set_begin(rpc_msg *msg, bool v)
{
	msg->data[0] = (msg->data[0] & ~0x01) | v;
}
static inline void vfbio_msg_set_lock(rpc_msg *msg, bool v)
{
	msg->data[0] = (msg->data[0] & ~0x100) | (v << 8);
}

/*
 * vfbio_open, vfbio_close
 * request & reply encoding
 * W1: |     31    | 30..8 | 7..0 |
 *     | Exclusive | RSVD  |  LUN |
 *     |(open only)|       |      |
 *
 * W2: | 31..0 |
 *     |  RSVD |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * Exclusive:	Must be 1
 * RSVD:	Reserved
 * LUN:		Logical unit #
 */
static inline u8 vfbio_msg_get_exclusive(rpc_msg *msg)
{
	return (msg->data[0] >> 31) & 0x1;
}
static inline void vfbio_msg_set_exclusive(rpc_msg *msg, u8 v)
{
	msg->data[0] = (msg->data[0] & ~(0x1 << 31)) | (v << 31);
}

/*
 * vfbio_lun_info
 * request encoding
 * W1: |       31..24        | 23..8 | 7..0 |
 *     | LUN_INFO_BUF[39:32] | RSVD  |  LUN |
 *
 * W2: |         31..0       |
 *     |  LUN_INFO_BUF[31:0] |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * reply encoding
 * W1: | 31..24 | 23..8  | 7..0 |
 *     |   RC   |  RSVD  |  LUN |
 *
 * W2: | 31..0  |
 *     |  RSVD  |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * LUN_INFO_BUF[39..32]
 *	Upper 8 bits of 40-bit reply buffer physical address
 * LUN_INFO_BUF[31..0]
 *	Lower 32 bits of 40-bit reply buffer physical address
 * RSVD:	Reserved
 * LUN:		Logical unit #
 */
struct vfbio_lun_info {
	/*
	 * W0: |  31..8 |  7..0  |
	 *     |  RSVD  | BLK_SZ |
	 *
	 * W1: | 31..0  |
	 *     | N_BLKS |
	 *
	 * W2..W5: | 31..24  | 23..16  | 15..8   |  7..0   |
	 *         | NAME[3] | NAME[2] | NAME[1] | NAME[0] |
	 */
	/*
	 * blk_sz:
	 * 0 = 512B
	 * 1 = 1KB
	 * 2 = 2KB
	 * 3 = 4KB
	 * 4..7 = RSVD
	 */
	u8 blk_sz;
	u8 rsvd1[3];
	u32 n_blks;
	u8 name[VFBIO_LUN_INFO_NAME_MAX]; /* NULL-terminated ASCII string */
};
static inline u8 vfbio_lun_info_get_blk_sz(struct vfbio_lun_info *info)
{
	return info->blk_sz & 0xf;
}
static inline u8 vfbio_lun_info_set_blk_sz(struct vfbio_lun_info *info, u8 v)
{
	return info->blk_sz = (info->blk_sz & ~0xf) | v;
}

/*
 * vfbio_read, vfbio_write
 * request encoding
 * W1: |       31..24     | 23..8  | 7..0 |
 *     | DMA ADDR[39..32] | N_BLKS |  LUN |
 *
 * W2: |      31..0      |
 *     | DMA ADDR[31..0] |
 *
 * W3: | 31..0 |
 *     |  BLK  |
 *
 * DMA ADDR[39..32]:	Upper 8 bits of 40-bit DMA address
 * N_BLKS:		# of blocks to read/write
 * LUN:			Logical unit #
 * DMA ADDR[31..0]:	Lower 32 bits of 40-bit DMA address
 * BLK:			Starting block in LUN
 *
 * reply encoding same as request except
 * N_BLKS:	# of blocks actually read/written
 */
static inline u32 vfbio_msg_get_n_blks(rpc_msg *msg)
{
	return (msg->data[0] >> 8) & 0xffff;
}
static inline void vfbio_msg_set_n_blks(rpc_msg *msg, u32 v)
{
	msg->data[0] = (msg->data[0] & ~(0xffff << 8)) |
		((v & 0xffff) << 8);
}
static inline u64 vfbio_msg_get_addr(rpc_msg *msg)
{
	return ((u64)(msg->data[0] & (0xff << 24)) << 8) | msg->data[1];
}
static inline void vfbio_msg_set_addr(rpc_msg *msg, u64 v)
{
	msg->data[0] = (msg->data[0] & ~(0xff << 24)) |
		(u32)((v >> 8) & (0xff << 24));
	msg->data[1] = (u32)(v & 0xffffffff);
}
static inline u32 vfbio_msg_get_blk(rpc_msg *msg)
{
	return msg->data[2];
}
static inline void vfbio_msg_set_blk(rpc_msg *msg, u32 v)
{
	msg->data[2]  = v;
}

/*
 * vfbio_discard
 * request encoding
 * W1: | 31..24 | 23..8  | 7..0 |
 *     | RSVD   | N_BLKS |  LUN |
 *
 * W2: | 31..0 |
 *     | RSVD  |
 *
 * W3: | 31..0 |
 *     |  BLK  |
 *
 * RSVD:		Reserved
 * N_BLKS:		# of blocks to discard
 * LUN:			Logical unit #
 * BLK:			Starting block in LUN
 *
 * reply encoding same as request except
 * N_BLKS:	# of blocks actually discarded
 */

/*
 * vfbio_sg_read, vfbio_sg_write
 * message encoding
 * W1: | 31..16 |       15..8      | 7..0 |
 *     | RSVD   | DMA ADDR[39..32] |  LUN |
 *
 * W2: |      31..0      |
 *     | DMA ADDR[31..0] |
 *
 * W3: | 31..0 |
 *     |  RSVD |
 *
 * RSVD:		Reserved
 * DMA ADDR[39..32]:	Upper 8 bits of 40-bit DMA address
 * LUN:			Logical unit #
 * DMA ADDR[31..0]:	Lower 32 bits of 40-bit DMA address
 */
/*
 * See vfbio_msg_get_addr() and vfbio_msg_set_addr() above for
 * manipulation of msg.
 */
struct vfbio_sg_segment {
	/*
	 * Each SG segment encoding
	 * W0: |      31..24      |  23..8  | 7..0 |
	 *     | DMA ADDR[39..32] | N_BLKS  | RSVD |
	 *
	 * W1: |       31..0     |
	 *     | DMA ADDR[31..0] |
	 *
	 * DMA ADDR[39..32]:	Upper 8 bits of 40-bit DMA address
	 * N_BLKS:		# of blocks to transfer
	 * RSVD:		Reserved
	 * DMA ADDR[31..0]:	Lower 32 bits of 40-bit DMA address
	 */
	u32 w[2];
};
#define VFBIO_SG_SEGMENT_MAX_BLKS	(1 << 16)

/*
 * It is imperative that this struct size remains a multiple of the
 * cache line size. This buffer is passed to the SMC so we must
 * perform cache operations on it. To keep these operations from
 * becoming complex we must ensure the buffer starts and ends on cache
 * line boundaries.
 */
#define VFBIO_SG_LIST_SEGMENTS	30	/* struct size = 256B */
struct vfbio_hw_sgl {
	/*
	 * SG list header encoding
	 * W0: | 31..16 | 7..0  |
	 *     |  RSVD  | N_SEG |
	 *
	 * W1: | 31..0 |
	 *     |  BLK  |
	 *
	 * W2: | 31..0 |
	 *     |  TAG  |
	 *
	 * W2: | 31..0 |
	 *     | RSVD  |
	 *
	 * RSVD:	Reserved
	 * N_SEG:	# of SG segments
	 * BLK:		starting block #
	 * TAG:		32-bit unique transaction ID
	 */
	u32 hdr[4];
	struct vfbio_sg_segment seg[VFBIO_SG_LIST_SEGMENTS];
};
static inline u8 vfbio_sgl_get_n_segs(struct vfbio_hw_sgl *sgl)
{
	return sgl->hdr[0] & 0xff;
}
static inline void vfbio_sgl_set_n_segs(struct vfbio_hw_sgl *sgl, u8 v)
{
	sgl->hdr[0] = (sgl->hdr[0] & ~0xff) | v;
}
static inline u32 vfbio_sgl_get_blk(struct vfbio_hw_sgl *sgl)
{
	return sgl->hdr[1];
}
static inline void vfbio_sgl_set_blk(struct vfbio_hw_sgl *sgl, u32 v)
{
	sgl->hdr[1] = v;
}
static inline u32 vfbio_sgl_get_tag(struct vfbio_hw_sgl *sgl)
{
	return sgl->hdr[2];
}
static inline void vfbio_sgl_set_tag(struct vfbio_hw_sgl *sgl, u32 v)
{
	sgl->hdr[2] = v;
}
static inline u16 vfbio_sgl_seg_get_n_blks(struct vfbio_hw_sgl *sgl, u8 seg)
{
	return (sgl->seg[seg].w[0] >> 8) & 0xffff;
}
static inline void vfbio_sgl_seg_set_n_blks(struct vfbio_hw_sgl *sgl, u8 seg,
					    u16 v)
{
	sgl->seg[seg].w[0] = (sgl->seg[seg].w[0] & ~(0xffff << 8)) | (v << 8);
}
static inline u64 vfbio_sgl_seg_get_addr(struct vfbio_hw_sgl *sgl, u8 seg)
{
	return (((u64)(sgl->seg[seg].w[0] & (0xff << 24))) << 8) |
		sgl->seg[seg].w[1];
}
static inline void vfbio_sgl_seg_set_addr(struct vfbio_hw_sgl *sgl,
						u8 seg, u64 v)
{
	sgl->seg[seg].w[0] = (sgl->seg[seg].w[0] & ~(0xff << 24)) |
		(u32)((v >> 8) & (0xff << 24));
	sgl->seg[seg].w[1] = (u32)(v & 0xffffffff);
}

/*
 * vfbio_async_discard
 * request encoding
 * W1: | 31..24 | 23..8  | 7..0 |
 *     | RSVD   | N_BLKS |  LUN |
 *
 * W2: | 31..0 |
 *     | TAG   |
 *
 * W3: | 31..0 |
 *     |  BLK  |
 *
 * RSVD:		Reserved
 * N_BLKS:		# of blocks to discard
 * LUN:			Logical unit #
 * TAG:			32-bit tag for use only by client
 * BLK:			Starting block in LUN
 *
 */
static inline u32 vfbio_msg_get_tag(rpc_msg *msg)
{
	return msg->data[1];
}
static inline void vfbio_msg_set_tag(rpc_msg *msg, u32 v)
{
	msg->data[1] = v;
}

/*
 * vfbio_get_device_health
 * request encoding
 * W1: | 31..24 | 23..1  |  0  |
 *     |   RC   |  RSVD  | END |
 *
 * W2: | 31..24 | 23..16 | 15..8 | 7..0  |
 *     |  RSVD  |  EST-B | EST-A | P-EOL |
 *
 * W3: | 31..0  |
 *     |  RSVD  |
 *
 * RSVD:		Reserved
 * EST-B:		Device Life Time estimate(EST-B)
 * EST-A:		Device Life Time estimate(EST-A)
 * P-EOL:		Pre EOL Life Time Estimation Values
 *
 */
static inline bool vfbio_msg_get_dh_end(rpc_msg *msg)
{
	return (msg->data[0]) & 0x1;
}
static inline bool vfbio_msg_get_dh_pre_eol(rpc_msg *msg)
{
	return (msg->data[1]) & 0xFF;
}
static inline bool vfbio_msg_get_dh_est_a(rpc_msg *msg)
{
	return (msg->data[1] >> 8) & 0xFF;
}
static inline bool vfbio_msg_get_dh_est_b(rpc_msg *msg)
{
	return (msg->data[1] >> 16) & 0xFF;
}

#endif
