/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 ***************************************************************************/

#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/irqreturn.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/hash.h>
#include <linux/types.h>
#include <linux/kthread.h>
#include "mdqm_dev.h"

#define MDQMDEV_HASHBITS    8
#define MDQMDEV_HASHENTRIES (1 << MDQMDEV_HASHBITS)

struct list_head	mdqmdev_base_head;
struct hlist_head	*mdqmdev_name_head;
struct hlist_head	*mdqmdev_index_head;
DEFINE_RWLOCK(mdqmdev_base_lock);

static inline struct hlist_head *mdqmdev_name_hash(const char *name)
{
	unsigned int hash = full_name_hash(NULL, name, strnlen(name, IFNAMSIZ));

	return &mdqmdev_name_head[hash_32(hash, MDQMDEV_HASHBITS)];
}

static inline struct hlist_head *mdqmdev_index_hash(int index)
{
	return &mdqmdev_index_head[index & (MDQMDEV_HASHENTRIES - 1)];
}

struct mdqm_dev *mdqmdev_get_by_name(char *name)
{
	struct mdqm_dev *dev;
	struct hlist_head *head = mdqmdev_name_hash(name);

	rcu_read_lock();
	hlist_for_each_entry_rcu(dev, head, name_hlist)
		if (!strncmp(dev->name, name, MDQM_DEV_NAME))
			return dev;
	rcu_read_unlock();

	return NULL;
}

static irqreturn_t rx_isr(int q_id, void *context, u32 flags)
{
	struct mdqm_q_info *qinfo;
	struct mdqm_dev *mdev;
	int i;
	qinfo = (struct mdqm_q_info *) context;
	mdev = qinfo->mdev;

	for (i = 0; i < mdev->q_count; i++)
		dqm_disable_rx_cb(mdev->q[i].handle);

	if (mdev->type == MDEV_TYPE_RX_KTHREAD) {
		atomic_set(&mdev->pkt_avail, 1);
		wake_up_interruptible(&mdev->wq);
	} else if (mdev->type == MDEV_TYPE_RX_NAPI) {
		napi_schedule(&mdev->napi);
	}

	qinfo->count++;
	return IRQ_HANDLED;
}

int mdqm_register(char *mdqmdev_name, struct mdqm_cb *cb)
{
	struct mdqm_dev *mdev;
	struct dqm_cb dqmcb = {};
	int i, cb_id = -1;

	if (!mdqmdev_name || !cb) {
		dev_err(NULL, "Invalid Parameters\n");
		return -EINVAL;
	}

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev) {
		dev_err(NULL, "Invalid Device\n");
		return -EINVAL;
	}

	if (mdev->refcnt >= MDQM_MAX_USER) {
		dev_err(&mdev->pdev->dev, "Exceeds Max User\n");
		return -EMFILE;
	}

	write_lock_bh(&mdqmdev_base_lock);
	for (i = 0; i < MDQM_MAX_USER; i++) {
		if (!mdev->cb[i].name) {
			cb_id = i;
			break;
		}
	}

	dev_info(&mdev->pdev->dev, "cb_id = %d\n", cb_id);

	if (mdev->refcnt == 0) {
		for (i = 0; i < mdev->q_count; i++) {
			if (mdev->type == MDEV_TYPE_RX_KTHREAD || mdev->type == MDEV_TYPE_RX_NAPI) {
				dqmcb.fn = (dqm_isr_callback)rx_isr;
				dqmcb.context = (void *)&mdev->q[i];
				dqmcb.name = (char *) cb->name;
				mdev->q[i].handle = dqm_register(mdev->dqm_dev,
								 mdev->q[i].num,
								 &dqmcb,
								 &(mdev->q[i].tok_size),
								 DQM_F_RX);
			} else {
				dqmcb.fn = NULL;
				dqmcb.context = NULL;
				dqmcb.name = (char *) cb->name;
				mdev->q[i].handle = dqm_register(mdev->dqm_dev,
								 mdev->q[i].num,
								 &dqmcb,
								 &(mdev->q[i].tok_size),
								 DQM_F_TX);
			}
			if (!mdev->q[i].handle) {
				pr_err("%s unable to acquire DQM %d on device %s\n",
				       __func__, mdev->q[i].num, mdev->dqm_dev);
				cb_id = -1;
			}
		}
	}

	if (cb_id >= 0) {
		memcpy(&(mdev->cb[cb_id]), cb,
		       sizeof(struct mdqm_cb));
		mdev->refcnt++;
	}

	write_unlock_bh(&mdqmdev_base_lock);

	return cb_id;
}
EXPORT_SYMBOL(mdqm_register);

int mdqm_unregister(char *mdqmdev_name, int cb_id)
{
	struct mdqm_dev *mdev;
	int i, status;

	if (!mdqmdev_name) {
		dev_err(NULL, "Invalid Parameters\n");
		return -EINVAL;
	}

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev) {
		dev_err(NULL, "Invalid Device\n");
		return -EINVAL;
	}

	if (cb_id >= MDQM_MAX_USER) {
		dev_err(&mdev->pdev->dev, "Exceeds Max User\n");
		return -EINVAL;
	}

	if (!mdev->cb[cb_id].name) {
		dev_err(&mdev->pdev->dev, "ID %d Already freed\n",
			cb_id);
		return -EINVAL;
	}

	dev_info(&mdev->pdev->dev, "cb_id = %d\n", cb_id);

	write_lock_bh(&mdqmdev_base_lock);
	memset(&(mdev->cb[cb_id]), 0,
	       sizeof(struct mdqm_cb));
	mdev->refcnt--;

	if (mdev->refcnt == 0) {
		for (i = 0; i < mdev->q_count; i++) {
			if (mdev->type == MDEV_TYPE_RX_KTHREAD ||
			    mdev->type == MDEV_TYPE_RX_NAPI) {
				status = dqm_release(mdev->q[i].handle,
						     DQM_F_RX);
			} else {
				status = dqm_release(mdev->q[i].handle,
						     DQM_F_TX);
			}

			if (status) {
				pr_err("%s unable to release DQM %d\n",
				       __func__, mdev->q[i].num);
			}
		}
	}
	write_unlock_bh(&mdqmdev_base_lock);

	return 0;
}
EXPORT_SYMBOL(mdqm_unregister);

int mdqm_send(char *mdqmdev_name, u32 *msgdata)
{
	struct mdqm_dev *mdev;

	if (!mdqmdev_name) {
		dev_err(NULL, "Invalid Parameters\n");
		return -EINVAL;
	}

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev) {
		dev_err(NULL, "Invalid Device\n");
		return -EINVAL;
	}

	if (mdev->type == MDEV_TYPE_TX)
		return dqm_tx(mdev->q[0].handle, 1, mdev->q[0].tok_size, msgdata);

	return -EINVAL;
}
EXPORT_SYMBOL(mdqm_send);

void *mdqm_to_dqm_h(char *mdqmdev_name, int num)
{
	struct mdqm_dev *mdev;

	if (!mdqmdev_name) {
		dev_err(NULL, "Invalid Parameters\n");
		return NULL;
	}

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev) {
		dev_err(NULL, "Invalid Device\n");
		return NULL;
	}

	if (num >= MDQM_MAX_Q) {
		dev_err(NULL, "Queue numner %d > %d\n",
			num, MDQM_MAX_Q);
		return NULL;
	}

	return (mdev->q[num].handle);
}
EXPORT_SYMBOL(mdqm_to_dqm_h);

int mdqm_send_priority(char *mdqmdev_name, int priority, u32 *msgdata)
{
	return -EOPNOTSUPP;
}
EXPORT_SYMBOL(mdqm_send_priority);
int mdqm_enable(char *mdqmdev_name)
{
	int i;
	struct mdqm_dev *mdev;

	if (!mdqmdev_name) {
		dev_err(NULL, "Invalid Parameters\n");
		return -EINVAL;
	}

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev) {
		dev_err(NULL, "Invalid Device\n");
		return -EINVAL;
	}

	atomic_inc(&mdev->enable);

	if (atomic_read(&mdev->enable) > 1)
		return 0;

	if (mdev->type == MDEV_TYPE_RX_KTHREAD || mdev->type == MDEV_TYPE_RX_NAPI)
		for (i = 0; i < mdev->q_count; i++)
			dqm_enable_rx_cb(mdev->q[i].handle);

	return 0;
}
EXPORT_SYMBOL(mdqm_enable);

int mdqm_disable(char *mdqmdev_name)
{
	int i;
	struct mdqm_dev *mdev;

	if (!mdqmdev_name) {
		dev_err(NULL, "Invalid Parameters\n");
		return -EINVAL;
	}

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev) {
		dev_err(NULL, "Invalid Device\n");
		return -EINVAL;
	}

	atomic_dec(&mdev->enable);

	if (atomic_read(&mdev->enable) > 0)
		return 0;

	if (mdev->type == MDEV_TYPE_RX_KTHREAD || mdev->type == MDEV_TYPE_RX_NAPI)
		for (i = 0; i < mdev->q_count; i++)
			dqm_disable_rx_cb(mdev->q[i].handle);

	return 0;
}
EXPORT_SYMBOL(mdqm_disable);

int mdqm_enable_id(char *mdqmdev_name, int cb_id)
{
	int i;
	struct mdqm_dev *mdev;
	int int_mask;

	if (!mdqmdev_name) {
		dev_err(NULL, "Invalid Parameters\n");
		return -EINVAL;
	}

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev) {
		dev_err(NULL, "Invalid Device\n");
		return -EINVAL;
	}

	int_mask = mdev->int_mask;
	mdev->int_mask |= (1<<cb_id);
	if (int_mask)
		return 0;

	if (mdev->type == MDEV_TYPE_RX_KTHREAD || mdev->type == MDEV_TYPE_RX_NAPI)
		for (i = 0; i < mdev->q_count; i++)
			dqm_enable_rx_cb(mdev->q[i].handle);

	return 0;
}
EXPORT_SYMBOL(mdqm_enable_id);

int mdqm_disable_id(char *mdqmdev_name, int cb_id)
{
	int i;
	struct mdqm_dev *mdev;

	if (!mdqmdev_name) {
		dev_err(NULL, "Invalid Parameters\n");
		return -EINVAL;
	}

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev) {
		dev_err(NULL, "Invalid Device\n");
		return -EINVAL;
	}

	mdev->int_mask &= ~(1<<cb_id);
	if (mdev->int_mask)
		return 0;

	if (mdev->type == MDEV_TYPE_RX_KTHREAD || mdev->type == MDEV_TYPE_RX_NAPI)
		for (i = 0; i < mdev->q_count; i++)
			dqm_disable_rx_cb(mdev->q[i].handle);

	return 0;
}
EXPORT_SYMBOL(mdqm_disable_id);

/* Device list insertion */
static void list_mdqmdev(struct mdqm_dev *dev)
{
	write_lock_bh(&mdqmdev_base_lock);
	list_add_tail_rcu(&dev->dev_list, &mdqmdev_base_head);
	hlist_add_head_rcu(&dev->name_hlist, mdqmdev_name_hash(dev->name));
	hlist_add_head_rcu(&dev->index_hlist,
			   mdqmdev_index_hash(dev->id));
	write_unlock_bh(&mdqmdev_base_lock);
}

static void unlist_mdqmdev(struct mdqm_dev *dev)
{
	write_lock_bh(&mdqmdev_base_lock);
	list_del_rcu(&dev->dev_list);
	hlist_del_rcu(&dev->name_hlist);
	hlist_del_rcu(&dev->index_hlist);
	write_unlock_bh(&mdqmdev_base_lock);
}

static struct hlist_head *mdqmdev_create_hash(void)
{
	int i;
	struct hlist_head *hash;

	hash = kmalloc(sizeof(*hash) * MDQMDEV_HASHENTRIES, GFP_KERNEL);
	if (hash != NULL)
		for (i = 0; i < MDQMDEV_HASHENTRIES; i++)
			INIT_HLIST_HEAD(&hash[i]);

	return hash;
}

static int mdqm_priority_queue(struct mdqm_dev *mdev, int budget)
{
	int status = 0, i, j, ret;
	int count = 0;
	u32 *msgdata = mdev->msgdata;

	for (i = 0; i < mdev->q_count; i++) {
		status = dqm_rx(mdev->q[i].handle,
				budget-count,
				mdev->q[i].tok_size,
				&msgdata[count * DQM_MAX_MSGSZ]);
		if (status < 0) {
			pr_err("Error receiving DQM %d\n on %s.\n",
			       mdev->q[i].num, mdev->name);
		} else
			count += status;

		if (count >= budget)
			break;
	}

	for (i = 0; i < count; i++) {
		for (j = 0; j < MDQM_MAX_USER; j++) {
			if (!mdev->cb[j].handler)
				continue;

			ret = mdev->cb[j].handler(mdev->id,
						  &msgdata[i * DQM_MAX_MSGSZ],
						  mdev->cb[j].ctx);
			if (ret == 0)
				break;
		}
	}

	return count;
}

#if defined(MDQM_ROUND_ROBIN)
static int mdqm_round_robin(struct mdqm_dev *mdev, int budget)
{
	int status[mdev->q_count], i, j, k, ret;
	int count[mdev->q_count];
	int present_count = 0;
	msgdata[mdev->q_count][budget * DQM_MAX_MSGSZ];

	memset(status, 0, sizeof(status));
	memset(count, 0, sizeof(count));

	for (i = 0; i < mdev->q_count; i++) {
		status[i] = dqm_rx(mdev->q[i].handke,
				   budget,
				   mdev->q[i].tok_size,
				   &msgdata[i][0]);
		if (status[i] < 0) {
			pr_err("Error receiving DQM ID %d\n on %s.\n",
					mdev->q[i].num, mdev->name);
			continue;
		} else if (status[i]) {
			count[i] = status[i];
			if (count[i] > present_count)
				present_count = count[i];
		}
	}

	for (j = 0; j < present_count; j++) {
		for (i = 0; i < mdev->q_count && j < count[i]; i++) {
			for (k = 0; k < MDQM_MAX_USER; k++) {
				if (!mdev->cb[k].handler)
					continue;

				ret = mdev->cb[k].handler(mdev->id,
						&msgdata[i][j * mdev->q[i].tok_size],
						mdev->cb[k].ctx);
				if (ret == 0)
					break;
			}
		}
	}

	return present_count;
}
#endif

static int mdqm_poll_napi(struct napi_struct *napi, int budget)
{
	struct mdqm_dev *mdev =
		container_of(napi, struct mdqm_dev, napi);
	int count = mdqm_priority_queue(mdev, budget);
	int i;

	if (count < budget) {
		napi_complete(napi);
		for (i = 0; i < mdev->q_count; i++)
			dqm_enable_rx_cb(mdev->q[i].handle);
	}

	return count;
}

static int mdqm_poll_kthread(void *data)
{
	struct mdqm_dev *mdev = (struct mdqm_dev *)data;
	int budget = def_napi_weight;
	int count, i;

	while (!mdev->stop_thread) {
		wait_event_interruptible(mdev->wq, (atomic_read(&mdev->pkt_avail) == 1));
		spin_lock_bh(&mdev->lock);
		atomic_set(&mdev->pkt_avail, 0);
		count = mdqm_priority_queue(mdev, budget);

		if (count >= budget) {
			atomic_set(&mdev->pkt_avail, 1);
			wake_up_interruptible(&mdev->wq);
		} else {
			for (i = 0; i < mdev->q_count; i++)
				dqm_enable_rx_cb(mdev->q[i].handle);
		}

		spin_unlock_bh(&mdev->lock);
		schedule();
	}

	return 0;
}

int mdqm_probe(struct platform_device *pdev)
{
	int status = 0;
	struct mdqm_dev *mdev;

	mdev = kmalloc(sizeof(struct mdqm_dev), GFP_KERNEL);
	if (!mdev) {
		status = -ENOMEM;
		goto done;
	}
	memset(mdev, 0, sizeof(struct mdqm_dev));

	pdev->dev.platform_data = mdev;
	mdev->pdev = pdev;
	status = mdqm_parse_dt_node(pdev);
	if (status)
		goto err_free_dt;

	if (mdev->type == MDEV_TYPE_RX_KTHREAD) {
		atomic_set(&mdev->pkt_avail, 0);
		mdev->task = kthread_create(mdqm_poll_kthread, mdev, mdev->name);
		init_waitqueue_head(&mdev->wq);
		if (mdev->task)
			wake_up_process(mdev->task);
	} else if (mdev->type == MDEV_TYPE_RX_NAPI) {
		init_dummy_netdev(&mdev->napi_dev);
		netif_napi_add(&mdev->napi_dev, &mdev->napi, mdqm_poll_napi,
			       def_napi_weight);
		napi_enable(&mdev->napi);
	}

	list_mdqmdev(mdev);
	if (mdqm_procfs_init(mdev))
		goto err_free_dt;

	goto done;

err_free_dt:
	kfree(mdev);

done:
	return status;
}

int mdqm_remove(struct platform_device *pdev)
{
	struct mdqm_dev *mdev;
	mdev = pdev->dev.platform_data;

	if (mdev->type == MDEV_TYPE_RX_KTHREAD) {
		kthread_stop(mdev->task);
	} else if (mdev->type == MDEV_TYPE_RX_NAPI) {
		napi_disable(&mdev->napi);
		netif_napi_del(&mdev->napi);
	}

	mdqm_procfs_exit(mdev);
	unlist_mdqmdev(mdev);
	kfree(mdev);
	return 0;
}

int mdqm_create(char *mdqmdev_name, char *dqmdev_name, char *qtype, u32 qcnt, u32 *qs)
{
	int status = 0, i;
	struct mdqm_dev *mdev;
	struct platform_device *pdev;

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (mdev)
		return -EEXIST;

	pdev = platform_device_alloc(MDQM_MODULE_NAME, -1);
	if (!pdev)
		return -ENOMEM;

	pdev->dev.init_name = mdqmdev_name;

	mdev = kmalloc(sizeof(struct mdqm_dev), GFP_KERNEL);
	if (!mdev) {
		kfree(pdev);
		return -ENOMEM;
	}
	memset(mdev, 0, sizeof(struct mdqm_dev));
	mdev->pdev = pdev;

	strncpy(mdev->name, mdqmdev_name, sizeof(mdev->name));
	mdev->name[sizeof(mdev->name)-1] = '\0';

	strncpy(mdev->dqm_dev, dqmdev_name, sizeof(mdev->dqm_dev));
	mdev->dqm_dev[sizeof(mdev->dqm_dev)-1] = '\0';

	if (!strncmp(qtype, "rx", sizeof("rx")))
		mdev->type = MDEV_TYPE_RX_KTHREAD;
	else if (!strncmp(qtype, "rx-napi", sizeof("rx-napi")))
		mdev->type = MDEV_TYPE_RX_NAPI;
	else if (!strncmp(qtype, "tx", sizeof("tx")))
		mdev->type = MDEV_TYPE_TX;

	mdev->q_count = qcnt;

	for (i = 0; i < mdev->q_count; i++) {
		mdev->q[i].priority = i;
		mdev->q[i].num = qs[i];
		mdev->q[i].mdev = mdev;
	}

	if (mdev->type == MDEV_TYPE_RX_KTHREAD) {
		atomic_set(&mdev->pkt_avail, 0);
		mdev->task = kthread_create(mdqm_poll_kthread, mdev, mdev->name);
		init_waitqueue_head(&mdev->wq);
		if (mdev->task)
			wake_up_process(mdev->task);
	} else if (mdev->type == MDEV_TYPE_RX_NAPI) {
		init_dummy_netdev(&mdev->napi_dev);
		netif_napi_add(&mdev->napi_dev, &mdev->napi, mdqm_poll_napi,
			       def_napi_weight);
		napi_enable(&mdev->napi);
	}

	list_mdqmdev(mdev);
	if (mdqm_procfs_init(mdev))
		goto err_free_dt;

	goto done;

err_free_dt:
	kfree(mdev);

done:
	return status;
}
EXPORT_SYMBOL(mdqm_create);

int mdqm_delete(char *mdqmdev_name)
{
	int refcnt;
	struct mdqm_dev *mdev;

	mdev = mdqmdev_get_by_name(mdqmdev_name);
	if (!mdev)
		return -ENODEV;

	write_lock_bh(&mdqmdev_base_lock);
	refcnt = mdev->refcnt;
	write_unlock_bh(&mdqmdev_base_lock);
	if (refcnt)
		return -EBUSY;

	if (mdev->type == MDEV_TYPE_RX_KTHREAD) {
		mdev->stop_thread = 1;
		atomic_set(&mdev->pkt_avail, 1);
		wake_up_interruptible(&mdev->wq);
		kthread_stop(mdev->task);
	} else if (mdev->type == MDEV_TYPE_RX_NAPI) {
		napi_disable(&mdev->napi);
		netif_napi_del(&mdev->napi);
	}

	mdqm_procfs_exit(mdev);
	unlist_mdqmdev(mdev);
	kfree(mdev->pdev);
	kfree(mdev);

	return 0;
}
EXPORT_SYMBOL(mdqm_delete);

static const struct of_device_id mdqm_of_match[] = {
	{.compatible = "brcm,mdqm"},
	{}
};
MODULE_DEVICE_TABLE(of, mdqm_of_match);
static struct platform_driver mdqm_driver = {
	.probe	= mdqm_probe,
	.remove	= mdqm_remove,
	.driver	= {
		.name		= MDQM_MODULE_NAME,
		.owner		= THIS_MODULE,
		.of_match_table	= mdqm_of_match
	}
};


static int __init mdqm_init(void)
{
	INIT_LIST_HEAD(&mdqmdev_base_head);
	mdqmdev_name_head = mdqmdev_create_hash();
	if (mdqmdev_name_head == NULL)
		goto err_name;

	mdqmdev_index_head = mdqmdev_create_hash();
	if (mdqmdev_index_head == NULL)
		goto err_idx;

	platform_driver_register(&mdqm_driver);
	pr_info("%s_Init\n", MDQM_MODULE_NAME);
	return 0;

err_idx:
	kfree(mdqmdev_name_head);
err_name:
	return -ENOMEM;
}

static void mdqm_exit(void)
{
	kfree(mdqmdev_name_head);
	kfree(mdqmdev_index_head);
	platform_driver_unregister(&mdqm_driver);
	pr_info("%s_Exit\n", MDQM_MODULE_NAME);
}

module_init(mdqm_init);
module_exit(mdqm_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("mdqm");
