/****************************************************************************
 *
 * Broadcom Proprietary and Confidential. (c) 2016 Broadcom.  All rights reserved.
 * The term Broadcom refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 *
 * Filename: oamhook_core.c
 *
 ****************************************************************************
 * Description: Create pon0 port
 *                   Handle OAM packets
 *                   1. reply ping packets
 *                   2. handle vCM Link Status packets
 *                   3. pass other packets to pon0
 ****************************************************************************/

#include <linux/module.h>
#include <linux/etherdevice.h>
#include <linux/rtnetlink.h>
#include "bcmnethooks.h"
#include "oamhook.h"

#define EPON_OAM_NAME					"EPON OAM/Unicast"

static u16 ping_correction_tag;
static int vcm_link_state;
static char *fixedpon_port;
static struct net_device *fixedpon_dev;
static struct net_device *pon0_dev;


static bool check_skbuff_length(struct sk_buff *skb, u32 *packet_length)
{
	bool status = true;
	struct ethhdr *hdr = eth_hdr(skb);
	u8 *packet = (u8 *)hdr;

	if (skb->len > (OAM_LENGTH_OFF + 1)) {
		*packet_length = ((((packet[OAM_LENGTH_OFF]&0xff)<<8)
			|(packet[OAM_LENGTH_OFF + 1]&0xff))&0xffff)
			+ OAM_PKT_HDR_LEN;
		if (skb->len <
				(*packet_length - OAM_PKT_LLC_HDR_LEN)) {
			status = false;
		}
	} else {
		status = false;
	}

	return status;
}

static int respond_ping(struct sk_buff *skb)
{
	struct ethhdr *hdr = eth_hdr(skb);
	u16 ping_flag = 0;
	u8 *packet = (u8 *)hdr;
	u32 packet_length;
	int status = BCM_NETHOOK_CONSUMED;

	if (!check_skbuff_length(skb, &packet_length)) {
		pr_err("Error ping packet!\n");
		status = BCM_NETHOOK_DROP;
		goto done;
	}

	ping_flag = (((packet[OAM_PING_FLAG_OFF]&0xff)<<8)
		|(packet[OAM_PING_FLAG_OFF + 1]&0xff))&0xffff;

	/* swap saddr and daddr */
	ether_addr_copy(hdr->h_dest, hdr->h_source);
	ether_addr_copy(hdr->h_source, skb->dev->dev_addr);

	packet[OAM_CORRECTION_TAG_OFF] = (ping_correction_tag>>8)&0xff;
	packet[OAM_CORRECTION_TAG_OFF + 1] = ping_correction_tag&0xff;
	ping_correction_tag++;

	if (ping_flag == OAM_PING_UNSATISFIED) {
		pr_debug("OAM_PING_UNSATISFIED\n");
		packet[OAM_PING_FLAG_OFF] =
			(OAM_PING_UNDEFINE>>8)&0xff;
		packet[OAM_PING_FLAG_OFF + 1] =
			OAM_PING_UNDEFINE&0xff;
	} else if (ping_flag == OAM_PING_SATISFIED) {
		pr_debug("OAM_PING_SATISFIED\n");
		packet[OAM_PING_FLAG_OFF] =
			(OAM_PING_PAUSE>>8)&0xff;
		packet[OAM_PING_FLAG_OFF + 1] =
			OAM_PING_PAUSE&0xff;
	} else {
		pr_debug("Received Private Ping OAM, flag is unknown - %d\n",
			ping_flag);
	}

	/* Push - Reverse of what is done in eth_type_trans */
	skb_push(skb, ETH_HLEN);
	skb->dev->netdev_ops->ndo_start_xmit(skb, skb->dev);

done:
	return status;
}

static int parse_oampducodeanddata(struct sk_buff *skb)
{
	u8 opcode, vcm_link_flag0, vcm_link_flag1, vcm_link_flag2;
	u8 vcm_link_flag3, vcm_link_flag4, vcm_link_flag5;
	u32 packet_length;
	struct ethhdr *hdr = eth_hdr(skb);
	u8 *packet = (u8 *)hdr;
	int status = OAM_OAMPDUCODEANDDATA_CONTENT_NORMAL;

	if (!check_skbuff_length(skb, &packet_length)) {
		pr_err("Error oampducodeanddata packet!\n");
		status = OAM_OAMPDUCODEANDDATA_CONTENT_UNKNOWN;
		goto done;
	}

	opcode = packet[OAM_OPCODE_OFF]&0xff;

	/* it's ONU notification message */
	if (opcode == 1) {
		vcm_link_flag0 = packet[OAM_VCM_LINK_FLAG_OFF]&0xff;
		vcm_link_flag1 = packet[OAM_VCM_LINK_FLAG_OFF + 1]&0xff;
		vcm_link_flag2 = packet[OAM_VCM_LINK_FLAG_OFF + 2]&0xff;
		vcm_link_flag3 = packet[OAM_VCM_LINK_FLAG_OFF + 3]&0xff;
		vcm_link_flag4 = packet[OAM_VCM_LINK_FLAG_OFF + 4]&0xff;
		vcm_link_flag5 = packet[OAM_VCM_LINK_FLAG_OFF + 5]&0xff;

		/* if all match, it's vCM Link status notification message */
		if ((vcm_link_flag0 == OAM_VCM_LINK_FLAG0)
			&& (vcm_link_flag1 == OAM_VCM_LINK_FLAG1)
			&& (vcm_link_flag2 == OAM_VCM_LINK_FLAG2)
			&& (vcm_link_flag3 == OAM_VCM_LINK_FLAG3)
			&& (vcm_link_flag4 == OAM_VCM_LINK_FLAG4)
			&& (vcm_link_flag5 == OAM_VCM_LINK_FLAG5)) {
			vcm_link_state = packet[OAM_VCM_LINK_STATE_OFF]&0xff;
			status = OAM_OAMPDUCODEANDDATA_CONTENT_VCMLINKNOTIFY;
		}
	}
done:
	return status;
}

static int respond_oampducodeanddata(struct sk_buff *skb)
{
	int type;
	bool status;

	type = parse_oampducodeanddata(skb);

	switch (type) {
	/* it's vCM link state notify */
	case OAM_OAMPDUCODEANDDATA_CONTENT_VCMLINKNOTIFY:
		if (vcm_link_state) {
			pr_debug("vCM Link up\n");
            netif_carrier_on(pon0_dev);
		} else {
			pr_debug("vCM Link down\n");
            netif_carrier_off(pon0_dev);
		}
		status = BCM_NETHOOK_DROP;
		break;
	default:
		status = BCM_NETHOOK_PASS;
		break;
	}

	return status;
}

enum bcm_nethook_result epon_oam_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result status = BCM_NETHOOK_PASS;
	struct sk_buff *skb = buf;
	struct ethhdr *hdr = eth_hdr(skb);
	u8 *packet = (u8 *)hdr;
	u16 oam_proto_type = 0;
	u8  sub_type = 0;

	if (skb->len < (OAM_PROTOCOL_TYPE_OFF + 1)) {
		pr_err("Error packet!\n");
		status = BCM_NETHOOK_DROP;
		goto done;
	}

	/* parse oam header */
	if (hdr->h_proto == OAM_ETHER_TYPE)
		sub_type = packet[OAM_SUB_TYPE_OFF]&0xff;

	if (sub_type == OAM_SUB_TYPE)
		oam_proto_type = (((packet[OAM_PROTOCOL_TYPE_OFF]&0xff)<<8)
			|(packet[OAM_PROTOCOL_TYPE_OFF + 1]&0xff))&0xffff;

	/* not OAM, pass */
	if ((hdr->h_proto != OAM_ETHER_TYPE)
		|| (sub_type != OAM_SUB_TYPE)) {
		skb->dev = pon0_dev;
        pon0_dev->stats.rx_packets ++;
        pon0_dev->stats.rx_bytes += skb->len;
        netif_receive_skb(skb);
		status = BCM_NETHOOK_CONSUMED;
		goto done;
	}

	switch (oam_proto_type) {
	case OAM_PROTO_PING:
		status = respond_ping(skb);
		break;
	case OAM_PROTO_OAMPDUCODEANDDATA:
		status = respond_oampducodeanddata(skb);
		break;
	default:
        status = BCM_NETHOOK_PASS;
        break;
	}

done:
	return status;
}

static netdev_tx_t pon0_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
    skb->dev = fixedpon_dev;
    pon0_dev->stats.tx_packets ++;
    pon0_dev->stats.tx_bytes += skb->len;
    fixedpon_dev->netdev_ops->ndo_start_xmit(skb, fixedpon_dev);

	return NETDEV_TX_OK;
}

static int	pon0_set_mac_address(struct net_device *dev, void *p)
{
	int status = 0;
	struct sockaddr *addr = (struct sockaddr *)p;

	if (netif_running(dev)) {
		netdev_err(dev, "Device busy.\n");
		status = -EBUSY;
		goto done;
	}

	memcpy(dev->dev_addr, addr->sa_data, dev->addr_len);

done:
	return status;
}

static const struct net_device_ops pon0_netdev_ops = {
	.ndo_init		= NULL,
	.ndo_uninit		= NULL,
	.ndo_open		= NULL,
	.ndo_stop		= NULL,
	.ndo_start_xmit = pon0_start_xmit,
	.ndo_set_mac_address = pon0_set_mac_address,
};

static const struct device_type pon0_type = {
	.name = "pon0",
};

void pon0_setup(struct net_device *pon0_dev)
{
	unsigned char mfg_id[3] = {0x02, 0x10, 0x18};

	/* Initialize the device entry points */
	ether_setup(pon0_dev);
	pon0_dev->netdev_ops = &pon0_netdev_ops;

	SET_NETDEV_DEVTYPE(pon0_dev, &pon0_type);

	/* Initialize the device options */
	pon0_dev->tx_queue_len = 0;
	pon0_dev->flags |= IFF_MULTICAST;

    /* Ensure valid dev_addr */
    if (is_zero_ether_addr(pon0_dev->dev_addr) &&
        pon0_dev->addr_assign_type == NET_ADDR_PERM) {
        eth_hw_addr_random(pon0_dev);
        memcpy(pon0_dev->dev_addr, mfg_id, 3);
    }
}


int pon0_create(struct net *net)
{
	int res = 0;

	rtnl_lock();

	pon0_dev = alloc_netdev_mq(0,
				   "pon0", NET_NAME_UNKNOWN,
				   pon0_setup, 1);
	if (!pon0_dev) {
		pr_err("pon0: eek! can't alloc netdev!\n");
		rtnl_unlock();
		return -ENOMEM;
	}

	dev_net_set(pon0_dev, net);

	res = register_netdevice(pon0_dev);

	netif_carrier_off(pon0_dev);

	rtnl_unlock();
	if (res < 0)
        free_netdev(pon0_dev);
	return res;
}

module_param(fixedpon_port, charp, 0000);
MODULE_PARM_DESC(fixedpon_port, "pon port name string");

static int __init oamhook_init(void)
{
	int status = 0;

	if (!fixedpon_port) {
		pr_err("please input fixedpon_port=\n");
		status = -EINVAL;
		goto done;
	}

	/* get device by name */
	fixedpon_dev = dev_get_by_name(&init_net, fixedpon_port);
	if (!fixedpon_dev) {
		pr_err("Failed to get %s device\n", fixedpon_port);
		status = -ENODEV;
		goto done;
	}

    /* create pon0 port */
    status = pon0_create(dev_net(fixedpon_dev));
    if (status) {
        pr_err("Failed to create pon0 port\n");
        goto done;
    }

	/* register and enable oam hook on pon0 rx */
	status = bcm_nethook_register_hook(fixedpon_dev,
		BCM_NETHOOK_RX_SKB,
		RX_SKB_PRIO_EPON_OAM,
		EPON_OAM_NAME,
		epon_oam_hook);
	if (status) {
		pr_err("Failed to register epon oam nethook\n");
		goto done;
	}
	status = bcm_nethook_enable_hook(fixedpon_dev,
		BCM_NETHOOK_RX_SKB,
		epon_oam_hook,
		true);
	if (status) {
		pr_err("Failed to enable epon oam nethook\n");
		bcm_nethook_unregister_hook(fixedpon_dev,
			BCM_NETHOOK_RX_SKB,
			epon_oam_hook);
		goto done;
	}

	pr_debug("OAMHOOK Init:\n");
done:
	return status;
}

static void __exit oamhook_exit(void)
{
	/* unregister hooks */
	bcm_nethook_unregister_hook(fixedpon_dev,
		BCM_NETHOOK_RX_SKB,
		epon_oam_hook);

	pr_debug("OAMHOOK Exit:\n");
}

module_init(oamhook_init);
module_exit(oamhook_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("oamhook");
