/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2017 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * vFlash block IO proc driver
 *
 * Author: Tim Ross <tim.ross@broadcom.com>
 * Author: Venky Selvaraj <venky.selvaraj@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <proc_cmd.h>
#include "vfbio_priv.h"
#include "vfbio_proc.h"

#define VFBIO_PROC_DIR "driver/vfbio"
#define VFBIO_DH_PROC_FILE	"device_health"
#define VFBIO_CMD_PROC_FILE	"cmd"
#define EST_MAX 12
#define PRE_EOL_MAX 5
#define TUNNEL_NAME "rg-smc"

char *est[EST_MAX] = {
	"Not Defined",
	"0% - 10% device lifetime used",
	"10% - 20% device lifetime used",
	"20% - 30% device lifetime used",
	"30% - 40% device lifetime used",
	"40% - 50% device lifetime used",
	"50% - 60% device lifetime used",
	"60% - 70% device lifetime used",
	"70% - 80% device lifetime used",
	"80% - 90% device lifetime used",
	"90% - 100% device lifetime used",
	"Exceeded its maximum estimated device lifetime"
};

char *pre_eol[PRE_EOL_MAX] = {
	"Not Defined",
	"Normal",
	"Warning",
	"Urgent",
	"Read Only"
};

struct dh_res {
	bool end;
	u8 est_a;
	u8 est_b;
	u8 pre_eol;
};

static struct proc_dir_entry *vfbio_proc_dir;
static struct proc_dir_entry *vfbio_dh_proc_file;
static bool vbio_dh_end;
static bool vbio_dh_begin;
static u32 device_cnt;
static const char lum_swap_delim[] = ",";

static struct vfbio_device *vfbio_get_dev_info(char *lun_name)
{
	struct list_head *pos;
	struct vfbio_device *vfdev = NULL;
	bool dev_found = false;

	if (!lun_name) {
		pr_err("LUN Name NULL\n");
		return NULL;
	}

	list_for_each(pos, &vfdevs_list) {
		vfdev = list_entry(pos, struct vfbio_device, list);
		if (!strcmp(lun_name, vfdev->name)) {
			pr_debug("LUN Name %s found\n", lun_name);
			dev_found = true;
			break;
		}
	}

	if (!dev_found) {
		pr_err("LUN Name %s not found\n", lun_name);
		return NULL;
	}

	return vfdev;
}

static int vfbio_lunswap(char *lun_name, struct vfbio_device *vfdev,
			bool first, bool last)
{
	int status = 0;
	rpc_msg msg;
	bool state;

	if (!vfdev) {
		vfdev = vfbio_get_dev_info(lun_name);
		if (!vfdev) {
			pr_err("Getting LUN ID failed\n");
			return -EINVAL;
		}
	}
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		SYS_FUNC_GET_LUN_NEXT_BOOT_STATE, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);
	status = rpc_send_request(vfdev->tunnel, &msg);
	if (unlikely(status)) {
		pr_err(RED("rpc_send_request failure (%d) on get LUN boot state\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}
	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		pr_err(RED("invalid response %d on get LUN boot state\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}
	state = sys_msg_get_lun_state(&msg);

	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		SYS_FUNC_SET_LUN_NEXT_BOOT_STATE, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);
	sys_msg_set_lunswap_flag(&msg, first, last);
	sys_msg_set_lunswap_state(&msg, !state);
	status = rpc_send_request(vfdev->tunnel, &msg);
	if (unlikely(status)) {
		pr_err(RED("rpc_send_request failure (%d) on set LUN boot state\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}
	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		pr_err(RED("invalid response %d on set LUN boot state\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}

	return status;
}

static int vfbio_lunstat(char *lun_name, struct vfbio_device *vfdev,
			bool *state)
{
	int status = 0;
	rpc_msg msg;

	if (!vfdev) {
		vfdev = vfbio_get_dev_info(lun_name);
		if (!vfdev) {
			pr_debug("Getting LUN ID failed\n");
			return -EINVAL;
		}
	}
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		SYS_FUNC_GET_LUN_NEXT_BOOT_STATE, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);
	status = rpc_send_request(vfdev->tunnel, &msg);
	if (unlikely(status)) {
		pr_err(RED("rpc_send_request failure (%d) on get LUN\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}
	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		pr_err(RED("invalid response %d on get LUN\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}
	*state = sys_msg_get_lun_state(&msg);

	return 0;
};


static int vfbio_lunlock(char *lun_name, struct vfbio_device *vfdev, bool lock)
{
	int status = 0;
	rpc_msg msg;

	if (!vfdev) {
		vfdev = vfbio_get_dev_info(lun_name);
		if (!vfdev) {
			pr_debug("Getting LUN ID failed\n");
			return -EINVAL;
		}
	}

	rpc_msg_init(&msg, RPC_SERVICE_VFBIO,
		VFBIO_FUNC_SET_ATTR, 0,
		0, 0, 0);
	vfbio_msg_set_lun(&msg, vfdev->lun);
	vfbio_msg_set_lock(&msg, lock);

	status = rpc_send_request(rpc_get_fifo_tunnel_id("rg-smc"), &msg);
	if (unlikely(status)) {
		pr_err(RED("rpc_send_request failure (%d) on lock LUN\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}

	if (vfbio_msg_get_retcode(&msg) == 0xff) {
		pr_err(RED("invalid response %d on lock LUN\n"),
			status);
		rpc_dump_msg(&msg);
		return -EINVAL;
	}

	return 0;
};


static inline int vfbio_get_device_health(int tunnel,
			struct dh_res *res, bool begin)
{
	int status = 0;
	rpc_msg msg;

	BUG_ON(!res);

	rpc_msg_init(&msg, RPC_SERVICE_VFBIO, VFBIO_FUNC_DEVICE_HEALTH, 0,
		     0, 0, 0);
	vfbio_msg_set_begin(&msg, begin);
	status = rpc_send_request(tunnel, &msg);
	if (unlikely(status)) {
		pr_err(RED("rpc_send_request failure (%d)\n"),
			status);
		rpc_dump_msg(&msg);
		return status;
	}
	status = vfbio_msg_get_retcode(&msg);
	if (unlikely(status)) {
		pr_err(RED("vfbio msg retcode %d\n"), (s8)status);
		rpc_dump_msg(&msg);
		status = -EIO;
	}
	res->end = vfbio_msg_get_dh_end(&msg);
	res->est_a = vfbio_msg_get_dh_est_a(&msg);
	res->est_b = vfbio_msg_get_dh_est_b(&msg);
	res->pre_eol = vfbio_msg_get_dh_pre_eol(&msg);

	return status;
}

static void *vfbio_dh_proc_start(struct seq_file *seq, loff_t *pos)
{
	struct vfbio_device *vfdev = NULL;

	pr_debug("-->\n");

	if (*pos == 0) {
		device_cnt = 0;
		vbio_dh_begin = true;
		vbio_dh_end = false;
		seq_puts(seq, "---------- Flash devices health report ----------\n");
	} else
		vbio_dh_begin = false;

	if (vbio_dh_end)
		return NULL;

	vfdev = list_first_entry(&vfdevs_list, struct vfbio_device, list);

	pr_debug("<--\n");
	return vfdev;
}

static void vfbio_dh_proc_stop(struct seq_file *seq, void *v)
{
	pr_debug("-->\n");
	pr_debug("<--\n");
}

static void *vfbio_dh_proc_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	struct vfbio_device *vfdev = v;

	pr_debug("-->\n");

	(*pos)++;
	if (vbio_dh_end)
		return NULL;

	pr_debug("<--\n");
	return vfdev;
}

static int vfbio_dh_proc_show(struct seq_file *seq, void *v)
{
	struct vfbio_device *vfbio = v;
	struct dh_res res;
	int status = 0;

	pr_debug("-->\n");

	if (!v)
		return -EINVAL;

	memset(&res, 0, sizeof(struct dh_res));
	device_cnt++;
	status = vfbio_get_device_health(vfbio->tunnel, &res, vbio_dh_begin);
	if (unlikely(status))
		return -EIO;
	seq_printf(seq, "Device Number: %u\n", device_cnt);
	seq_printf(seq, "Device Lifetime estimate(EST-A) : Value as per JESD84-B51 standard - %d (%s)\n",
		res.est_a, (res.est_a < EST_MAX) ?
		est[res.est_a] : "Reserved");
	seq_printf(seq, "Device Lifetime estimate(EST-B) : Value as per JESD84-B51 standard - %d (%s)\n",
		res.est_b, (res.est_b < EST_MAX) ?
		est[res.est_b] : "Reserved");
	seq_printf(seq, "Pre EOL Lifetime Estimation(Pre EOL) : Value as per JESD84-B51 standard - %d (%s)\n",
		res.pre_eol, (res.pre_eol < PRE_EOL_MAX) ?
		pre_eol[res.pre_eol] : "Reserved");
	seq_puts(seq, "\n");
	vbio_dh_begin = false;
	vbio_dh_end = res.end;
	pr_debug("<--\n");
	return 0;
}

static const struct seq_operations vfbio_dh_seq_ops = {
	.start = vfbio_dh_proc_start,
	.stop = vfbio_dh_proc_stop,
	.next = vfbio_dh_proc_next,
	.show = vfbio_dh_proc_show,
};

static int vfbio_dh_proc_open(struct inode *inode, struct file *file)
{
	int status;

	pr_debug("-->\n");

	status = seq_open(file, &vfbio_dh_seq_ops);
	if (!status) {
		struct seq_file *sf = file->private_data;

		sf->private = PDE_DATA(inode);
	}

	pr_debug("<--\n");
	return status;
}

static const struct file_operations vfbio_dh_fops = {
	.owner = THIS_MODULE,
	.open = vfbio_dh_proc_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = seq_release,
};

static void vfbio_proc_cmd_lunswap_help(char *str)
{
	pr_alert("%s lunswap: Swap specified LUN (or) all LUNs\n", str);
	pr_alert("%s  lunswap {LUN name(s) seperated by ',' | all}\n", str);
}

static void vfbio_proc_cmd_lunstat_help(char *str)
{
	pr_alert("%s lunstat: Provide status of specified LUN (or) all LUNs\n",
			 str);
	pr_alert("%s  lunstat  {LUN name(s) seperated by ',' | all}\n", str);
}

static void vfbio_proc_cmd_lunlock_help(char *str)
{
	pr_alert("%s lunlock: Lock/unlock specified LUN\n", str);
	pr_alert("%s  lunlock {LUN name} {0 | 1}\n", str);
}

static void vfbio_proc_cmd_help_help(char *str)
{
	pr_alert("%s help: Help on individual command\n", str);
}

static int vfbio_proc_cmd_help(int argc, char *argv[])
{
	int status = 0;

	if (argc != 2) {
		pr_err(RED("Invalid # of arguments.\n"));
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(argv[1], "lunswap", sizeof("lunswap")))
		vfbio_proc_cmd_lunswap_help("");
	else if (!strncmp(argv[1], "lunstat", sizeof("lunstat")))
		vfbio_proc_cmd_lunstat_help("");
	else if (!strncmp(argv[1], "lunlock", sizeof("lunlock")))
		vfbio_proc_cmd_lunlock_help("");
	else {
		pr_err(RED("Unrecognized command: %s\n"),
			argv[1]);
		status = -EINVAL;
	}

done:
	return status;
}

static int vfbio_proc_cmd_lunswap(int argc, char *argv[])
{
	int status = 0;
	struct list_head *pos;
	struct vfbio_device *vfdev = NULL;
	int cnt = 0, first_vflash = -1, last_vflash = 0;
	char *lun_name = NULL, *tmp = NULL, *tmp_bak = NULL;

	if (argc != 2) {
		pr_err("Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(argv[1], "all", strlen("all"))) {
		list_for_each(pos, &vfdevs_list) {
			vfdev = list_entry(pos, struct vfbio_device, list);
			if (vfdev->name[strlen(vfdev->name)-1] == 'b') {
				if (first_vflash < 0)
					first_vflash = cnt;
				last_vflash = cnt;
			}
			cnt++;
		}
		cnt = 0;

		list_for_each(pos, &vfdevs_list) {
			vfdev = list_entry(pos, struct vfbio_device, list);
			if (vfdev->name[strlen(vfdev->name)-1] == 'b') {
				status = vfbio_lunswap(NULL, vfdev,
					cnt == first_vflash,
					cnt == last_vflash);
				if (status < 0) {
					pr_err("Lun Swap Failed for %s.\n",
						vfdev->name);
					goto done;
				}
			}
			cnt++;
		}
	} else {
		tmp = kmalloc(strlen(argv[1])+1, GFP_KERNEL);
		if (!tmp) {
			pr_err("Unable to allocate memory to copy LUN names.\n");
			status = -ENOMEM;
			goto done;
		}
		tmp_bak = tmp;
		memset(tmp_bak, 0, strlen(argv[1])+1);
		memcpy(tmp_bak, argv[1], strlen(argv[1]));
		while ((lun_name = strsep(&tmp_bak, lum_swap_delim))) {
			if (first_vflash < 0)
				first_vflash = cnt;
			last_vflash = cnt;
			cnt++;
		}
		cnt = 0;
		tmp_bak = tmp;
		memset(tmp_bak, 0, strlen(argv[1])+1);
		memcpy(tmp_bak, argv[1], strlen(argv[1]));
		while ((lun_name = strsep(&tmp_bak, lum_swap_delim))) {
			status = vfbio_lunswap(lun_name, NULL,
				cnt == first_vflash,
				cnt == last_vflash);
			if (status < 0) {
				pr_err("Lun Swap Failed for %s.\n",
					lun_name);
				goto done;
			}
			cnt++;
		}

	}

done:
	if (tmp)
		kfree(tmp);
	return status;
}

static int vfbio_proc_cmd_lunstat(int argc, char *argv[])
{
	int status = 0;
	struct list_head *pos;
	struct vfbio_device *vfdev = NULL;
	char *lun_name;
	bool state;

	if (argc != 2) {
		pr_err("Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(argv[1], "all", strlen("all"))) {
		list_for_each(pos, &vfdevs_list) {
			vfdev = list_entry(pos, struct vfbio_device, list);
			if (vfdev->name[strlen(vfdev->name)-1] == 'o') {
				status = vfbio_lunstat(NULL, vfdev, &state);
				if (status < 0) {
					pr_err("Getting Lun state Failed for %s.\n",
						   vfdev->name);
					goto done;
				}
				pr_info("%s - LUN %d State : %s\n",
					vfdev->name, vfdev->lun,
					(state ? "backup  " : "operational"));
			}
		}
	} else {
		while ((lun_name = strsep(&argv[1], lum_swap_delim))) {
			status = vfbio_lunstat(lun_name, NULL, &state);
			if (status < 0) {
				pr_err("Get Lun stat Failed for %s.\n",
					lun_name);
				goto done;
			}
			pr_info("%s - State : %s\n",
				lun_name, (state ? "backup  " : "operational"));
		}
	}

done:
	return status;
}

static int vfbio_proc_cmd_lunlock(int argc, char *argv[])
{
	int status = 0;
	struct vfbio_device *vfdev = NULL;
	char *lun_name;
	char *lun_lock;
	bool lock = 0;

	if (argc != 3) {
		pr_err("Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	lun_name = argv[1];
	lun_lock = argv[2];

	if(!strcmp(lun_lock,"0")) {
		lock = false;
	}
	else if (!strcmp(lun_lock,"1")) {
		lock = true;
	}
	else {
		pr_err("Invalid lock/unlock flag argument %s.\n", lun_lock);
		status = -EINVAL;
		goto done;
	}

	status = vfbio_lunlock(lun_name, vfdev, lock);
	if (status < 0) {
		pr_err("Lun lock/unlock Failed for %s.\n",
			lun_name);
		goto done;
	}

done:
	if(status) {
		pr_info("Lun lock/unlock Failed for %s. Locked: %d.\n", lun_name, lock);
	}
	else {
		pr_info("Lun lock/unlock Passed for %s. Locked: %d.\n", lun_name, lock);
	}

	return status;
}

static struct proc_cmd_ops command_entries[] = {
	PROC_CMD_INIT("help", vfbio_proc_cmd_help),
	PROC_CMD_INIT("lunswap", vfbio_proc_cmd_lunswap),
	PROC_CMD_INIT("lunstat", vfbio_proc_cmd_lunstat),
	PROC_CMD_INIT("lunlock", vfbio_proc_cmd_lunlock),
};

struct proc_cmd_table vfbio_command_table = {
	.module_name = MODULE_NAME,
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

static struct proc_dir_entry *vfbio_cmd_proc_file;

int __init vfbio_proc_init(void)
{
	int status = 0;

	pr_debug("-->\n");
	vfbio_proc_dir = proc_mkdir(VFBIO_PROC_DIR, NULL);
	if (!vfbio_proc_dir) {
		pr_err("Failed to create PROC directory %s.\n",
		       VFBIO_PROC_DIR);
		status = -EIO;
		goto done;
	}
	vfbio_dh_proc_file = proc_create(VFBIO_DH_PROC_FILE, 0444,
					 vfbio_proc_dir, &vfbio_dh_fops);
	if (!vfbio_dh_proc_file) {
		pr_err("Failed to create %s\n", VFBIO_DH_PROC_FILE);
		status = -EIO;
		vfbio_proc_exit();
		goto done;
	}
	vfbio_cmd_proc_file = proc_create_cmd(VFBIO_CMD_PROC_FILE,
		vfbio_proc_dir, &vfbio_command_table);
	if (!vfbio_cmd_proc_file) {
		pr_err("Failed to create %s\n", VFBIO_CMD_PROC_FILE);
		status = -EIO;
		vfbio_proc_exit();
		goto done;
	}

done:
	pr_debug("<--\n");
	return status;
}

void vfbio_proc_exit(void)
{
	pr_debug("-->\n");
	if (vfbio_cmd_proc_file) {
		remove_proc_entry(VFBIO_CMD_PROC_FILE, vfbio_proc_dir);
		vfbio_cmd_proc_file = NULL;
	}
	if (vfbio_dh_proc_file) {
		remove_proc_entry(VFBIO_DH_PROC_FILE, vfbio_proc_dir);
		vfbio_dh_proc_file = NULL;
	}
	if (vfbio_proc_dir) {
		remove_proc_entry(VFBIO_PROC_DIR, NULL);
		vfbio_proc_dir = NULL;
	}

	pr_debug("<--\n");
}
