 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/phy.h>
#include <linux/regulator/consumer.h>

#include "ethsw.h"
#include "ethsw_priv.h"

/*
 * Log Utilities
 */
/* define log module */
#define LOG_MODULE "ethsw_531xx"

int ethsw_mbus_write(struct mii_bus *mbus, int phy_id, int reg_addr, u16 reg_val);
int ethsw_mbus_read(struct mii_bus *mbus, int phy_id, int reg_addr);

/*
 * Ethsw Device Functions
 */


/*
 * Switch PHY Functions
 */


int ethsw_swdev_board_init_531xx(struct ethsw_device *swdev)
{
	int ret = 0;
	u8 data;
	struct spi_device *spi;
	int max_waittime = 100;

	FUNC_ENTER();

	swdev->vreg = devm_regulator_get(&swdev->sdev->dev, "pwr");
	if (IS_ERR(swdev->vreg)) {
		LOG_ERR("Failed to get power regulator.\n");
		ret = PTR_ERR(swdev->vreg);
		goto ERROR;
	}
	/*
	 * Bounce power to switch to ensure it will work during battery
	 * transitions. This is being done because of historically observed
	 * bugs requiring this solution.
	 */
	ret = regulator_enable(swdev->vreg);
	if (ret) {
		LOG_ERR("Failed to enable power.\n");
		goto ERROR;
	}
	/*
	 * 531xx switches require 350ms after reset line is released to
	 * guarantee SPI interface is operational. There is a 100ms
	 * delay circuit for the reset line that starts as soon as the
	 * regulator power is applied. The DT node for this regulator
	 * has the startup-delay-us property set to 1ms which is just enough
	 * to ensure the power is fully up. So required delay is
	 * 100ms + 350ms = 450ms. Use 500ms to ensure enough delay.
	 */
	mdelay(500);
	ret = regulator_disable(swdev->vreg);
	if (ret) {
		LOG_ERR("Failed to disable power.\n");
		goto ERROR;
	}
	/*
	 * Delay again to ensure any capacitance is bled off and switch
	 * is fully powered down. 1ms should be enough.
	 */
	mdelay(1);
	ret = regulator_enable(swdev->vreg);
	if (ret) {
		LOG_ERR("Failed to enable power.\n");
		goto ERROR;
	}
	mdelay(500);

	/* Check the regulator status and update it */
	if (regulator_is_enabled(swdev->vreg)) {
		LOG_DEBUG("Enabled regulator(pwr).\n");
		swdev->ethsw_pwr_status = true;
	} else {
		swdev->ethsw_pwr_status = false;
		LOG_INFO("Unable to enable regulator(pwr).\n");
		goto ERROR;
	}

	spi = swdev->sdev;
	spi->mode = SPI_MODE_3;
	spi->max_speed_hz = swdev->spi_freq;
	spi_setup(spi);

	/* Reset the switch */
	data = 0xD0;
	ret = ethsw_creg_write(swdev, 0, 0x79, &data, 1);
	if (ret) {
		LOG_ERR("Switch Failed to reset.\n");
		goto ERROR;
	}

	do {
		mdelay(10);
		ret = ethsw_creg_read(swdev, 0x0, 0x79, &data, 1);
		if (ret) {
			LOG_ERR("Switch Failed to reset.\n");
			goto ERROR;
		}
		max_waittime--;
	} while (data != 0 && max_waittime);

	if (max_waittime == 0) {
		LOG_ERR("Switch Failed to reset.\n");
		goto ERROR;
	}
	ret = ethsw_creg_read(swdev, 0x2, 0x30, (u8 *) &swdev->chip_id, 4);
	if (ret) {
		LOG_ERR("Switch Failed to reset.\n");
		goto ERROR;
	}
	LOG_DEBUG("Device ID %x\n", swdev->chip_id);

	/* MII port tx/rx clock delay */
	if (swdev->port[5].type == MII_PORT) {
		data = 0x01;
		ret = ethsw_creg_write(swdev, 0, 0x65, &data, 1);
	}

ERROR:
	FUNC_LEAVE();
	return(ret);
}

void ethsw_swdev_board_exit_531xx(struct ethsw_device *swdev)
{
	FUNC_ENTER();
    swdev->ethsw_pwr_status = false;

	if (swdev->vreg) {
		regulator_disable(swdev->vreg);
		/* Check the regulator status and update it */
		if (!regulator_is_enabled(swdev->vreg)) {
			swdev->ethsw_pwr_status = false;
			LOG_DEBUG("Disabled regulator(pwr).\n");
		} else {
			swdev->ethsw_pwr_status = true;
			LOG_INFO("Unable to Disable regulator(pwr).\n");
		}
		regulator_put(swdev->vreg);
	}

	FUNC_LEAVE();
}

/* Port index is alway 0 based. Port IDs however depend on the switch model. For
 * some switches port IDs are 0 based for others 1 based. The same applies to
 * phy IDs. */
#define ETHSW_53134_PHY_ID_OFFSET 0
#define ETHSW_53124_PHY_ID_OFFSET 1
int ethsw_port_idx_to_phy_id_531xx(struct ethsw_device *swdev, int port_idx)
{
	if(ethsw_isswitch_53134(swdev->chip_id)) {
		return (port_idx + ETHSW_53134_PHY_ID_OFFSET);
	} else if(swdev->chip_id == ETHSW_53124_SWITCH) {
		return (port_idx + ETHSW_53124_PHY_ID_OFFSET);
	} else {
		return ETHSW_PORT_ID_INVALID;
	}
}


int ethsw_port_idx_to_port_id_531xx(struct ethsw_device *swdev, int port_idx)
{
	if(ethsw_isswitch_53134(swdev->chip_id)) {
		return (port_idx + ETHSW_53134_PORT_OFFSET);
	} else if(swdev->chip_id == ETHSW_53124_SWITCH) {
		return (port_idx + ETHSW_53124_PORT_OFFSET);
	} else {
		return ETHSW_PORT_ID_INVALID;
	}
}

int ethsw_port_id_to_port_idx_531xx(struct ethsw_device *swdev, int port_id)
{
	if(ethsw_isswitch_53134(swdev->chip_id)) {
		return (port_id - ETHSW_53134_PORT_OFFSET);
	} else if(swdev->chip_id == ETHSW_53124_SWITCH) {
		return (port_id - ETHSW_53124_PORT_OFFSET);
	} else {
		return ETHSW_PORT_ID_INVALID;
	}
}

/* the 53134 switch chip can have three different chipids */
bool ethsw_isswitch_53134(int chip_id)
{
	if ((chip_id == ETHSW_53134_SWITCH)
		|| (chip_id == ETHSW_53134_SWITCH_ALT1)
		|| (chip_id == ETHSW_53134_SWITCH_ALT2)) {
		return true;
	} else {
		return false;
	}
}

int ethsw_phy_board_init_531xx(struct ethsw_device *swdev)
{
	int ret = 0;
	int page;
	u16 data = 0xDE1;

	FUNC_ENTER();

	// 53134 seems to have bad defaults for Auto-Negotiation Advertisement
	// Register
	for(page = 0x11; page < 0x15; page ++) {
		ret = ethsw_creg_write(swdev, page, 0x8, (u8 *)&data, 2);
		if (ret) {
			LOG_ERR("Failed ethsw_phy_board_init_531xx\n");
			break;
		}
	}

	FUNC_LEAVE();
	return(ret);
}

int ethsw_phy_mbox_cb_531xx(struct notifier_block *this,
		      unsigned long event, void *ptr)
{
	int ret = NOTIFY_DONE;
	return ret;
}
int ethsw_led_powercntrl_531xx(struct ethsw_device *swdev, int port_id, bool state)
{
	/* STUB function*/
	return 0;
}

