 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/jiffies.h>
#include <linux/debugfs.h>
#include <linux/sched.h>
#include <linux/phy.h>
#include <linux/interrupt.h>
#include <linux/kmod.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/platform_device.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/of.h>
#include <linux/io.h>
#include <linux/of_fdt.h>
#include <linux/reboot.h>
#include <brcm_mbox.h>

#include "brcm_mbox_priv.h"

/**
* DOC: MBOX Overview
*
* This kernel module is responsible for providing user space access to the
* MBOX registers. This is used for tracking states of other CPU's, setting
* the current CPU's state, as well as some basic IPC
*/

static const struct of_device_id of_platform_brcm_mbox_table[] = {
	{ .compatible = "brcm,brcm-mbox", },
	{},
};

#define MODULE_NAME	"brcm-mbox"
#define MODULE_VER	"1.0"

static struct brcm_mbox_dev_info *info;
static ATOMIC_NOTIFIER_HEAD(brcm_mbox_chain);

int brcm_mbox_register_notifier(struct notifier_block *nb)
{
	return atomic_notifier_chain_register(&brcm_mbox_chain, nb);
}
EXPORT_SYMBOL(brcm_mbox_register_notifier);

int brcm_mbox_unregister_notifier(struct notifier_block *nb)
{
	return atomic_notifier_chain_unregister(&brcm_mbox_chain, nb);
}
EXPORT_SYMBOL(brcm_mbox_unregister_notifier);

static int brcm_mbox_notifier_call_chain(
				unsigned long val, void *v)
{
	return atomic_notifier_call_chain(&brcm_mbox_chain, val, v);
}

#ifdef CONFIG_BCM_MBOX_PERF

static struct mbox_metric metrics;
static unsigned int irqcnt;

static void describe_irq(u32 old, u32 new, u8 mbox, char *buf, int *cnt)
{
	if (old == new)
		return;

	*cnt += sprintf(&buf[*cnt], " | %s IRQ [0x%08x to 0x%08x].",
			mbox_to_string(mbox), old, new);

	if ((mbox >= MBOX_CM) && (mbox <= MBOX_STB)) {
		if (POWER(old) != POWER(new))
			*cnt += sprintf(&buf[*cnt], " {%s}",
					power_to_string(POWER(new)));
	}

	switch (mbox) {
	case MBOX_BMU:
		if (BATT_STATE(old) != BATT_STATE(new))
			*cnt += sprintf(&buf[*cnt], " {%s}",
					bmustate_to_string(BATT_STATE(new)));
		break;
	case MBOX_CM:
		if (STATE(old) != STATE(new))
			*cnt += sprintf(&buf[*cnt], " {%s}",
					cmstate_to_string(STATE(new)));
		break;
	case MBOX_RG:
		if (STATE(old) != STATE(new))
			*cnt += sprintf(&buf[*cnt], " {%s}",
					rgstate_to_string(STATE(new)));
		break;
	case MBOX_STB:
		if (STATE(old) != STATE(new))
			*cnt += sprintf(&buf[*cnt], " {%s}",
					stbstate_to_string(STATE(new)));
		break;
	case MBOX_SVM:
		if (STATE(old) != STATE(new))
			*cnt += sprintf(&buf[*cnt], " {%s}",
					svmstate_to_string(STATE(new)));
		break;
	default:
		break;
	}

	return;
}

/**
* mbox_dbgfs_metrics_read() - Provide a dump of collected metrics to user.
* Note: very inefficient for small access, will generate full text each time
* it is called, regardless of the size requested. It's usage/nature doesn't
* really call for improvement
*/
static int mbox_dbgfs_metrics_read(struct file *fp, char __user *user_buffer,
				   size_t count, loff_t *position)
{
	char *l_buf;
	int ret = 0;
	int i = 0, j;
	int cnt = 0;
	struct mbox_met_irq irq;

	/* No error checking on this size! */
	/* Assume max 100 chars per line for max collected irq's
	 * plus 1 header line */
	l_buf = kmalloc((MAX_IRQ_METRIC+1)*100, GFP_KERNEL);
	if (!l_buf)
		return -ENOMEM;

	/* assure that first pass will be "different" */
	memset(&irq, 0xFF, sizeof(irq));

	cnt += sprintf(&l_buf[cnt], "Start jiffy: 0x%lx\n", metrics.start_jiff);
	cnt += sprintf(&l_buf[cnt], "jiffies, ms from start, description\n");

	while (i < irqcnt) {
		cnt += sprintf(&l_buf[cnt], "0x%lx, %u,", metrics.irq[i].jiff,
			       jiffies_to_msecs(metrics.irq[i].jiff-
			       metrics.start_jiff));

		/* Now describe what happened */
		for (j = 0; j <= max_mbox; j++) {
			describe_irq(irq.mbox[j], metrics.irq[i].mbox[j], j,
				     l_buf, &cnt);
		}
		cnt += sprintf(&l_buf[cnt], "\n");

		memcpy(&irq, &metrics.irq[i], sizeof(irq));
		i++;
	}

	ret = simple_read_from_buffer(user_buffer, count, position, l_buf, cnt);
	kfree(l_buf);

	return ret;
}

static const struct file_operations fops_dbg_metric = {
	.read = mbox_dbgfs_metrics_read,
};

/**
* ba_config_dbgfs() - Create/Free debugfs files
* @create: If true, create, else free
*
* For debugging purposes, various debugfs files can be created for performance
* metrics of the CPU's in the system (should be used in conjunction w/
* CONFIG_BCM_BA_PERF metrics
*
*/
static int mbox_config_dbgfs(bool create)
{
	static struct dentry *parent;
	static struct dentry *metric_dentry;

	FUNC_TRACE(1);

	if (create) {
		parent = debugfs_create_dir(DEBUG_FS_DIR, NULL);
		if ((!parent) || (parent == ERR_PTR(-ENODEV))) {
			MSG_TRACEE("Failed to create debugfs directory\n");
			FUNC_TRACE(0);
			parent = NULL;
			return -1;
		}

		metric_dentry = debugfs_create_file("metrics", S_IRUGO, parent,
				NULL, &fops_dbg_metric);
	} else {
		if ((metric_dentry) && (metric_dentry != ERR_PTR(-ENODEV))) {
			debugfs_remove(metric_dentry);
			metric_dentry = NULL;
		}

		if ((parent) && (parent == ERR_PTR(-ENODEV))) {
			debugfs_remove(parent);
			parent = NULL;
		}
	}

	FUNC_TRACE(0);
	return 0;
}
#endif

/**
* brcm_mbox_irq() - Mailbox IRQ handler
*
* This will handle any interrupt from registered mailbox's (RG/CM/STB/BMU)
*/
static irqreturn_t brcm_mbox_irq(int irq, void *dev_id)
{
	u32 stat;
	struct brcm_mbox_info *states;
	unsigned long flags;
	int i;

	if ((dev_id != (void *)info) || (!info))
		return IRQ_NONE;

	states = &info->states;

	spin_lock_irqsave(&info->state_lock, flags);

	R32(stat, info->reg_status);

	/* Only get/check mbox's we're allowed to see */
	stat &= info->read_mask;
	if (!stat) {
		spin_unlock_irqrestore(&info->state_lock, flags);
		return IRQ_NONE; /* Not an IRQ we requested */
	}

	/* Clear the status. Must be done before reading */
	W32(info->reg_status, stat);

	if (info->override_bmu_state)
		stat &= ~MBOX_BIT(MBOX_BMU);

	for (i = 0; i < MAX_NUM_MBOX; i++) {
		if (stat & MBOX_BIT(i)) {
			R32(states->mbox[i], MBOX(i));
			MSG_TRACEN("MBOX %d (%s): 0x%08x\n", i,
				   mbox_to_string(i), states->mbox[i]);
			if (i == 1 && ((states->mbox[i] & 0x0f)
					== CM_RUN_STATE_EXCEPTION_SHUTDOWN_REQ) &&
						  (states->mbox[i] & (0x1<<19))) {
				MSG_TRACEE("********************************************************* \n");
				MSG_TRACEE("RESET CAUSE : cm_watchdog_timer (1 of 32 possible causes) \n");
				MSG_TRACEE("********************************************************* \n");
			}
			else if (i == 1 && ((states->mbox[i] & 0x0f)
					== CM_RUN_STATE_SHUTDOWN_REQUESTED) &&
						  (states->mbox[i] & (0x1<<19))) {			
				MSG_TRACEE("*********************************** \n");
				MSG_TRACEE("RESET CAUSE : CM Request Reset... \n");
				MSG_TRACEE("*********************************** \n");
			}
		}
	}
	states->mask = stat;

	/* Update which MBOX's have new data */
	for (i = 0; i < BRCM_MBOX_MAX_INST; i++)
		info->inst[i].mask |= stat;

	if (stat)
		METRIC_IRQ;

	brcm_mbox_notifier_call_chain(MBOX_CHANGE_EVENT, states);
	spin_unlock_irqrestore(&info->state_lock, flags);

	/* Ok to do this outside of state_lock? TBD */
	if (stat)
		for (i = 0; i < BRCM_MBOX_MAX_INST; i++) {
			if (info->inst[i].mask)
				wake_up(&info->inst[i].wait);
		}

	return IRQ_HANDLED;
}


/**
* brcm_mbox_open() - Device node opening
*
* Allocate per instance info. Each "open" will have it's own mask of what
* MBOX data is "new" as well as it's own wait queue
*/
static int brcm_mbox_open(struct inode *inode, struct file *file)
{
	unsigned long flags;
	int i, ret = 0;

	FUNC_TRACE(1);

	/* Overload on state_lock.. TBD */
	spin_lock_irqsave(&info->state_lock, flags);
	for (i = 0; i < BRCM_MBOX_MAX_INST; i++) {
		if (info->inst_mask & (1<<i))
			continue;
		file->private_data = (void *)&info->inst[i];
		info->inst_mask |= (1<<i);
		break;
	}
	spin_unlock_irqrestore(&info->state_lock, flags);


	if (!file->private_data) {
		MSG_TRACEE("Exceeded supported # opens\n");
		ret = -EBUSY;
	}

	FUNC_TRACE(0);
	return ret;
}

/**
* brcm_mbox_release() - Device node closing
*/
static int brcm_mbox_release(struct inode *inode, struct file *file)
{
	unsigned long flags;
	int i, ret = 0;

	FUNC_TRACE(1);

	/* Overload on state_lock.. TBD */
	spin_lock_irqsave(&info->state_lock, flags);
	for (i = 0; i < BRCM_MBOX_MAX_INST; i++) {
		if (file->private_data != (void *)&info->inst[i])
			continue;
		info->inst[i].mask = 0;
		wake_up(&info->inst[i].wait); /* TBD? */
		file->private_data = NULL;
		info->inst_mask &= ~(1<<i);
		break;
	}
	spin_unlock_irqrestore(&info->state_lock, flags);

	if (file->private_data) {
		MSG_TRACEE("Invalid release request\n");
		ret = -ENXIO;
	}

	FUNC_TRACE(0);
	return ret;
}

/**
* brcm_mbox_read() - TBD for future development
*/
static ssize_t brcm_mbox_read(struct file *file, char __user *buf, size_t size,
			loff_t *ppos)
{
	FUNC_TRACE(1);

	FUNC_TRACE(0);
	return size;
}

/**
* brcm_mbox_get_state() - Get current state of the system
*
* Will query BMU state (AC power, Battery power, LOW battery power)
* CM state, RG state, STB state, as well as limited additional MBOX info
* (CM requests, etc.)
*/
int brcm_mbox_get_state(struct brcm_mbox_info *state)
{
	unsigned long flags;
	int i;

	FUNC_TRACE(1);

	spin_lock_irqsave(&info->state_lock, flags);

	for (i = 0; i < MAX_NUM_MBOX; i++)
		if (info->read_mask & MBOX_BIT(i))
			R32(state->mbox[i], MBOX(i));

#ifdef CONFIG_BRCM_IKOS
	SWF32(state->mbox[0], BMUPOWER_STATE, BS_AC);
#endif

	if (info->override_bmu_state)
		state->mbox[MBOX_BMU] = BATT_STATE_AC;

	spin_unlock_irqrestore(&info->state_lock, flags);

	FUNC_TRACE(0);
	return 0;
}
EXPORT_SYMBOL(brcm_mbox_get_state);

/**
* brcm_mbox_ioctl() - ioctls for broadcom MBOX user/kernel interaction
* @arg cmd: Ioctl/command
* @arg arg: data from/to user space
*
* Supported ioctls are
* BRCM_MBOX_IOCTL_STATE - Request MBOX states from user space
* BRCM_MBOX_IOCTL_SET - Set a MBOX to a value
*/
static long brcm_mbox_ioctl(struct file *file, unsigned int cmd,
			    unsigned long arg)
{
	struct brcm_mbox_inst *pinst;
	struct brcm_mbox_set set;
	struct brcm_mbox_perm perm;
	int ret = 0;
	unsigned long flags;

	FUNC_TRACE(1);

	pinst = (struct brcm_mbox_inst *)file->private_data;

	/* Shut down much of this until basic requirements are met */
	if (!pinst) {
		FUNC_TRACE(0);
		return -EAGAIN;
	}

	switch (cmd) {
	case BRCM_MBOX_IOCTL_STATE:
		/* Give states info to user space */
		MSG_TRACEN("Sending states to userspace\n");

		spin_lock_irqsave(&info->state_lock, flags);
		info->states.mask = pinst->mask;
		ret = copy_to_user((void *)arg, &info->states,
				   sizeof(struct brcm_mbox_info));
		if (ret == 0)
			pinst->mask = 0;

		spin_unlock_irqrestore(&info->state_lock, flags);
		break;
	case BRCM_MBOX_IOCTL_SET:
		/* Set a MBOX */
		ret = copy_from_user(&set, (void *)arg, sizeof(set));
		if (ret != 0) {
			MSG_TRACEE("Error getting MBOX set info\n");
			break;
		}

		if ((info->write_mask & MBOX_BIT(set.mbox)) == 0) {
			MSG_TRACEE("Invalid MBOX write req (%d)!\n", set.mbox);
			break;
		}

		MSG_TRACEI("Setting MBOX %d to 0x%x\n", set.mbox, set.value);
		W32(MBOX(set.mbox), set.value);
		break;
	case BRCM_MBOX_IOCTL_PERM:
		MSG_TRACEN("Sending permissions to userspace\n");

		perm.read = GET_CPU_COMM_MBOX_READ_MASK(info->read_mask);
		perm.write = info->write_mask;

		ret = copy_to_user((void *)arg, &perm,
				   sizeof(struct brcm_mbox_perm));
		if (ret != 0)
			MSG_TRACEE("Failed to write PERM info to user space\n");
		break;
	default:
		MSG_TRACEE("Unsupported IOCTL %d\n", cmd);
		ret = -EINVAL;
		break;
	}

	FUNC_TRACE(0);
	return ret;
}

/**
* brcm_mbox_poll() - Used by user space to poll for when MBOX data has changed.
*/
static unsigned int brcm_mbox_poll(struct file *file, poll_table *wait)
{
	struct brcm_mbox_inst *pinst;
	unsigned int ret = 0;

	FUNC_TRACE(1);

	pinst = (struct brcm_mbox_inst *)file->private_data;

	poll_wait(file, &pinst->wait, wait);

	if (pinst->mask)
		ret = (POLLPRI | POLLOUT);

	FUNC_TRACE(0);
	return ret;
}

#if defined(CONFIG_OF)
/**
* brcm_mbox_of_map() - Get info from device tree.
*/
static int brcm_mbox_of_map(struct device_node *node)
{
	const __be32 *prop_reg;
	int len;

	FUNC_TRACE(1);

	if (!node)
		MSG_TRACEE("Invalid device node\n");

	/* Get read/write masks */
	prop_reg = (u32 *)of_get_property(node, "read", &len);
	if (prop_reg)
		info->read_mask = (u16)be32_to_cpu(*prop_reg);

	prop_reg = (u32 *)of_get_property(node, "write", &len);
	if (prop_reg)
		info->write_mask = (u16)be32_to_cpu(*prop_reg);

	/*
	 * Some boards that do not support battery do not have Vsys
	 * wired to the BMU so it reports an incorrect battery state. We have
	 * to override the BMU battery state MBOX bits that we read so that
	 * they always indicate we are on AC power.
	 */
	info->override_bmu_state =
		of_property_read_bool(node, "override-bmu-state");

	FUNC_TRACE(0);
	return 0;
}
#else
static int brcm_mbox_of_map(struct device_node *node)
{
	return -1;
}
#endif

static const struct file_operations mbox_fops = {
	.owner = THIS_MODULE,
	.open = brcm_mbox_open,
	.release = brcm_mbox_release,
	.read = brcm_mbox_read,
	.unlocked_ioctl = brcm_mbox_ioctl,
	.poll = brcm_mbox_poll,
};

/**
* mbox_notify_reboot() - Last ditch mbox hook for power down
*
* In case "normal" shutdown path is not used (i.e. via user space), this will
* "catch" shutdown and do the minimum required MBOX operations (i.e. set
* state to no power/not running).
* It will NOT do ANY sync with other CPU's.
* Note that by the time this code is running, there is no more user space
* component which is still alive
*/
static int mbox_notify_reboot(struct notifier_block *this,
			      unsigned long code, void *x)
{
	u32 tmp;
	unsigned long flags;

	FUNC_TRACE(1);

	MSG_TRACEE("MBOX shutting down!\n");

	spin_lock_irqsave(&info->state_lock, flags);

	if (info->write_mask & (1<<MBOX_STB)) {
		R32(tmp, MBOX(MBOX_STB));
		SWF32(tmp, POWER_STATE, POWER_NONE);
		if (code == SYS_RESTART)
			SWF32(tmp, STB_SVMRESTART_STB, 1);
		SWF32(tmp, RUN_STATE, STB_SHUTDOWN_READY);
		W32(MBOX(MBOX_STB), tmp);
		info->states.mbox[MBOX_STB] = tmp;
	}

	if (info->write_mask & (1<<MBOX_RG)) {
		R32(tmp, MBOX(MBOX_RG));
		SWF32(tmp, POWER_STATE, POWER_NONE);
		if (code == SYS_RESTART) {
			/* Don't think there's harm in setting both? */
			SWF32(tmp, RG_STBRESTART_RG, 1);
			SWF32(tmp, RG_SVMRESTART_RG, 1);
		}
		SWF32(tmp, RUN_STATE, RG_SHUTDOWN_READY);
		W32(MBOX(MBOX_RG), tmp);
		info->states.mbox[MBOX_RG] = tmp;
	}

	if (info->write_mask & (1<<MBOX_SVM)) {
		R32(tmp, MBOX(MBOX_SVM));
		SWF32(tmp, POWER_STATE, POWER_NONE);
		SWF32(tmp, RUN_STATE, SVM_SHUTDOWN_READY);
		W32(MBOX(MBOX_SVM), tmp);
		info->states.mbox[MBOX_SVM] = tmp;
	}

	spin_unlock_irqrestore(&info->state_lock, flags);

	FUNC_TRACE(0);
	return NOTIFY_DONE;
}

static struct notifier_block mbox_notifier = {
	.notifier_call  = mbox_notify_reboot,
	.next           = NULL,
	.priority       = 0,
};

/**
* mbox_update_rg_powerdown() - Update RG mailbox to reflect powerdown state
*
* This will be called from the immediate powerdown request handler and
* do the minimum required MBOX operations (i.e. set state to no
* power/not running). It will NOT do ANY sync with other CPU's.
*/
void brcm_mbox_update_rg_powerdown (void)
{
	u32 tmp;

	R32(tmp, MBOX(MBOX_RG));

	/**
	 *  Set any bits that tell other systems that the RG is
	 *  now in a powerdown state
	 */
	SWF32(tmp, POWER_STATE, POWER_NONE);
	SWF32(tmp, RG_STBRESTART_RG, 1);
	SWF32(tmp, RG_SVMRESTART_RG, 1);
	SWF32(tmp, RUN_STATE, RG_SHUTDOWN_READY);
	W32(MBOX(MBOX_RG), tmp);

	/**
	*  Do a "dummy" read here to make sure that the UBUS register
	*  is properly updated before proceeding
	*/
	R32(tmp, MBOX(MBOX_RG));

	return;
}
EXPORT_SYMBOL(brcm_mbox_update_rg_powerdown);

static inline void brcm_mbox_lock(u8 idx, bool lock)
{
	u32 tmp = 0;

	MSG_TRACEI("%s MBOX(%d) (%s)\n", lock ? "LOCK" : "UNLOCK", idx,
		   mbox_to_string(idx));

	/* Assume 3390 (and later?) will just always lock for STB? */
	if (lock) {
		/* TBD... assume only STB can lock? */
		tmp = ((1&MBOX_LOCK_MASK)<<MBOX_LOCK_SHIFT);
		tmp |= ((0&MBOX_CLR_ON_READ_MASK) << MBOX_CLR_ON_READ_SHIFT);
		tmp |= ((MBOX_LOCK_OWNER_STB & MBOX_LOCK_SRC_MASK) <<
			MBOX_LOCK_SRC_SHIFT);
	} else {
		tmp = 0;
	}

	if (idx > MAX_LOCKED_MBOX)
		W32(MBOX_CFG(idx), tmp);
}

/**
* brcm_mbox_probe()
*
* Get resources, get current power state,
* register IRQ (if so configured), create device node, check OF for settings.
*/
static int brcm_mbox_probe(struct platform_device *pdev)
{
	int ret = -1;
	u32 tmp;
	struct resource *res;
	struct device *dev = &pdev->dev;
	int i;
	struct device_node *dnode = NULL;

	FUNC_TRACE(1);

#ifdef CONFIG_BCM_MBOX_PERF
	memset(&metrics, 0, sizeof(metrics));
#endif

	METRIC_START;

	info = (struct brcm_mbox_dev_info *)devm_kzalloc(dev,
		sizeof(struct brcm_mbox_dev_info), GFP_KERNEL);
	if (!info) {
		MSG_TRACEE("Failed to allocate memory!\n");
		ret = -ENOMEM;
		goto ERROR_EXIT;
	}

	info->irq = -1;
	spin_lock_init(&info->state_lock);

	platform_set_drvdata(pdev, (void *)info);
	info->pdev = pdev;

	res = platform_get_resource(pdev, IORESOURCE_MEM, IO_MEM_MBOX);
	/* Exclusive access to MBOX registers */
	info->mbox = devm_ioremap_resource(dev, res);
	if ((!res) || !info->mbox) {
		MSG_TRACEE("Error getting MBOX IO resource\n");
		ret = -EADDRNOTAVAIL;
		goto ERROR_EXIT;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, IO_MEM_COMM);
	if (!res) {
		MSG_TRACEE("Error getting COMM IO resource\n");
		ret = -EADDRNOTAVAIL;
		goto ERROR_EXIT;
	}
	/* Do NOT get exclusive access to CPU COMM registers */
	info->comm = devm_ioremap(dev, res->start, resource_size(res));
	if (!info->comm) {
		MSG_TRACEE("Error getting COMM IO resource\n");
		ret = -EADDRNOTAVAIL;
		goto ERROR_EXIT;
	}

	/* Open up access to all of the MBOX related regs */
	W32(CPU_COMM_REGS_CPUC_HV_MODE, HV_MODE);

	/* Init wait queue */
	for (i = 0; i < BRCM_MBOX_MAX_INST; i++)
		init_waitqueue_head(&info->inst[i].wait);

	/* Allocate device nodes */
	MSG_TRACEN("Allocating device node(s)\n");
	ret = alloc_chrdev_region(&info->mbox_dev, 0, 1, "brcm mbox");
	if (ret != 0) {
		MSG_TRACEE("Error allocating character device!\n");
		goto ERROR_EXIT;
	}
	MSG_TRACEI("Broadcom MBOX MAJOR is %d\n", MAJOR(info->mbox_dev));

	info->mbox_class = class_create(THIS_MODULE, "brcm mbox");
	if (!info->mbox_class) {
		unregister_chrdev_region(info->mbox_dev, 1);
		MSG_TRACEE("Failed to create class\n");
		goto ERROR_EXIT;
	}

	cdev_init(&info->mbox_cdev, &mbox_fops);
	ret = cdev_add(&info->mbox_cdev, info->mbox_dev, 1);
	if (ret != 0) {
		MSG_TRACEE("Error adding cdev\n");
		goto ERROR_EXIT;
	}

	MSG_TRACEN("Creating device nodes\n");
	device_create(info->mbox_class, NULL, info->mbox_dev, NULL, "mbox");

	/* Collect any settings from device tree */
	if (brcm_mbox_of_map(dev->of_node) != 0) {
		MSG_TRACEE("Error in OF configuration!\n");
		goto ERROR_EXIT;
	}

	/* Request and register IRQ */
	info->irq = platform_get_irq(pdev, 0);
	if (info->irq < 0) {
		MSG_TRACEE("Missing IRQ resource\n");
		ret = -ENXIO;
		goto ERROR_EXIT;
	}

	/* Register for any interrupt specified in DT */
	info->read_mask = SET_CPU_COMM_MBOX_READ_MASK(info->read_mask);

	/* Update state(s) and grab ownership (where needed) */
	for (i = 0; i < MAX_NUM_MBOX; i++) {
		if (info->write_mask & MBOX_BIT(i)) {
			tmp = 0; /* Default, clear any info */
			switch (i) {
			case MBOX_RG:
				SWF32(tmp, RUN_STATE, RG_KERNEL);
				SWF32(tmp, POWER_STATE, POWER_HIGH);
				break;
			case MBOX_STB:
				R32(tmp, MBOX(i)); /* Set by BOLT, update! */
				SWF32(tmp, RUN_STATE, STB_KERNEL);
				/* Bolt will set power state for STB...*/
				/* TBD... move to SVM? */
				/*WF32(tmp, POWER_STATE, POWER_HIGH);*/
				break;
			case MBOX_SVM:
				/* At least in A0/B0, no hard coding for SVM */
				brcm_mbox_lock(i, true);
				SWF32(tmp, RUN_STATE, SVM_KERNEL);
				SWF32(tmp, POWER_STATE, POWER_HIGH);
				break;
			default:
				brcm_mbox_lock(i, true);
				continue;
			}
			W32(MBOX(i), tmp);
		}
	}

	/* Which interrupt registers to use */
	switch (info->irq) {
	case CPUC_IRQ_ARM0:
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM0_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM0_MASK;
		break;
	case CPUC_IRQ_ARM1:
	default:
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM1_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM1_MASK;
		break;
	case CPUC_IRQ_ARM2:
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM2_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM2_MASK;
		break;
	case CPUC_IRQ_ARM3:
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM3_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM3_MASK;
		break;
	}
	MSG_TRACEN("Using A15S Registers\n");

	/* Get the current states of power/cpu's */
	/* Clear status that we care about, then check all to prevent
	 * unexpected interrupts from before driver comes on-line */
	W32(info->reg_status, info->read_mask);
	brcm_mbox_get_state(&info->states);

	MSG_TRACEN("Enable MBOX IRQ %d\n", info->irq);
	ret = devm_request_irq(dev, info->irq, &brcm_mbox_irq, IRQF_SHARED,
			"brcm-mbox", (void *)info);
	if (ret < 0) {
		MSG_TRACEE("error requesting IRQ #%d\n", info->irq);
		info->irq = -1; /* So that cleanup code works correctly */
		ret = -ENXIO;
		goto ERROR_EXIT;
	}
	dnode = of_find_node_by_name(NULL, "power");
	if (dnode == NULL) {
		info->states.delay_battery_mode = 0;
	} else {
		/* Check delay battery mode status from device tree */
		if (of_property_read_bool(dnode, "delay-battery-mode"))
			info->states.delay_battery_mode = 1;
		else
			info->states.delay_battery_mode = 0;
	}
	/* Enable MBOX interrupts.. TBD */
	MSG_TRACEN("Enable MBOX interrupt mask\n");
	R32(tmp, info->reg_mask);
	W32(info->reg_mask, info->read_mask | tmp);

	/* Hook into PM notifications */
	/* register_pm_notifier */ /* TBD */
	register_reboot_notifier(&mbox_notifier);

#ifdef CONFIG_BCM_MBOX_PERF
	mbox_config_dbgfs(true);
#endif

	FUNC_TRACE(0);
	return 0;

ERROR_EXIT:

	if (info) {
		if (info->mbox)
			iounmap(info->mbox);
		if (info->comm)
			iounmap(info->comm);
		if (info->irq >= 0)
			devm_free_irq(dev, info->irq, (void *)info);
		if (info->mbox_class) {
			unregister_chrdev_region(info->mbox_dev, 1);
			class_destroy(info->mbox_class);
		}

		devm_kfree(dev, (void *)info);
		info = NULL;
	}

	FUNC_TRACE(0);
	return ret;
}

/**
* brcm_mbox_remove()
*/
static int brcm_mbox_remove(struct platform_device *pdev)
{
	u32 tmp;
	int i;
	struct brcm_mbox_dev_info *pinfo;

	FUNC_TRACE(1);

	pinfo = (struct brcm_mbox_dev_info *)platform_get_drvdata(pdev);
	if (!pinfo) {
		FUNC_TRACE(0);
		return 0;
	}

	unregister_reboot_notifier(&mbox_notifier);

#ifdef CONFIG_BCM_MBOX_PERF
	mbox_config_dbgfs(false);
#endif

	/* Disable MBOX interrupts */
	MSG_TRACEN("Enable MBOX interrupt mask\n");
	R32(tmp, info->reg_mask);
	tmp &= ~info->read_mask;
	W32(info->reg_mask, tmp);

	/* Free any locked MBOX's */
	for (i = 0; i < MAX_NUM_MBOX; i++) {
		if (info->write_mask & MBOX_BIT(i)) {
			switch (i) {
			case MBOX_RG:
			case MBOX_STB:
				break;
			case MBOX_SVM:
				brcm_mbox_lock(i, false);
				break;
			default:
				brcm_mbox_lock(i, false);
				break;
			}
		}
	}

	if (info->mbox)
		iounmap(info->mbox);
	if (info->comm)
		iounmap(info->comm);
	if (info->irq >= 0)
		devm_free_irq(&pdev->dev, info->irq, (void *)info);
	if (info->mbox_class)
		device_destroy(info->mbox_class, info->mbox_dev);

	cdev_del(&info->mbox_cdev);
	class_destroy(info->mbox_class);
	unregister_chrdev_region(info->mbox_dev, 1);

	devm_kfree(&pdev->dev, (void *)info);
	pinfo = NULL;

	FUNC_TRACE(0);
	return 0;
}

static struct platform_driver brcm_mbox_driver = {
	.probe = brcm_mbox_probe,
	.remove = brcm_mbox_remove,
	.driver = {
		.name = "brcm-mbox",
		.owner = THIS_MODULE,
		.of_match_table = of_platform_brcm_mbox_table,
	},
};

static int __init brcm_mbox_init(void)
{
	pr_info("%s driver v%s\n", MODULE_NAME, MODULE_VER);
	return platform_driver_register(&brcm_mbox_driver);
}
arch_initcall(brcm_mbox_init);

static void __exit brcm_mbox_exit(void)
{
	platform_driver_unregister(&brcm_mbox_driver);
}
module_exit(brcm_mbox_exit);

MODULE_LICENSE("GPL");
