 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/rculist.h>
#include <linux/spinlock.h>
#include <linux/times.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/jhash.h>
#include <linux/random.h>
#include <linux/slab.h>
#include <linux/atomic.h>
#include <asm/unaligned.h>
#include <linux/proc_fs.h>
#include "ebt_dhcpsnoop.h"

/* Client DB Structure */
struct ebt_dhcpsnoop_db_entry {
	struct hlist_node	hlist;

	struct rcu_head		rcu;
	unsigned long		created;
	__u8			iif;             /* Input Interface Index  */
	atomic_t		use;             /* Use count              */
	__u8			addr[6];         /* MAC address            */
	__u8			hostname[255];   /* Hostname               */
	__be32			ipv4addr;        /* IPv4 Address           */
	__u8			status;
	__u32			leasetime;
};

/* Following code is derived from br_fdb.c implementation */

#define EBT_DHCPSNOOP_HASH_BITS 8
#define EBT_DHCPSNOOP_HASH_SIZE (1 << EBT_DHCPSNOOP_HASH_BITS)

static spinlock_t			hash_lock;
static struct hlist_head		hash[EBT_DHCPSNOOP_HASH_SIZE];

static struct kmem_cache *ebt_dhcpsnoop_db_cache __read_mostly;

static u32 db_salt __read_mostly;

#define EBT_DHCPSNOOP_PROC_DIR_NAME	"driver/ebt_dhcpsnoop"

static struct
proc_dir_entry *ebt_dhcpsnoop_proc_dir;

static inline int ebt_dhcpsnoop_mac_hash(const unsigned char *mac)
{
	/* use 1 byte of OUI cnd 3 bytes of NIC */
	u32 key = get_unaligned((u32 *)(mac + 2));
	return jhash_1word(key, db_salt) & (EBT_DHCPSNOOP_HASH_SIZE - 1);
}

static void db_rcu_free(struct rcu_head *head)
{
	struct ebt_dhcpsnoop_db_entry *ent
		= container_of(head, struct ebt_dhcpsnoop_db_entry, rcu);
	kmem_cache_free(ebt_dhcpsnoop_db_cache, ent);
}

static void db_delete(struct ebt_dhcpsnoop_db_entry *f)
{
	if (f && atomic_dec_and_test(&f->use)) {
		hlist_del_rcu(&f->hlist);
		call_rcu(&f->rcu, db_rcu_free);
	}
}

/* Completely flush all dynamic entries in client database */
void ebt_dhcpsnoop_db_flush(void)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < EBT_DHCPSNOOP_HASH_SIZE; i++) {
		struct ebt_dhcpsnoop_db_entry *f;
		struct hlist_node *n;
		hlist_for_each_entry_safe(f, n, &hash[i], hlist) {
			db_delete(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Flush all entries referring to a specific port.
 * if do_all is set also flush static entries
 */
void ebt_dhcpsnoop_db_delete_by_port(int iif)
{
	int i;

	spin_lock_bh(&hash_lock);
	for (i = 0; i < EBT_DHCPSNOOP_HASH_SIZE; i++) {
		struct hlist_node *h, *g;

		hlist_for_each_safe(h, g, &hash[i]) {
			struct ebt_dhcpsnoop_db_entry *f;
			f = hlist_entry(h, struct ebt_dhcpsnoop_db_entry,
					hlist);
			if (f->iif != iif)
				continue;

			db_delete(f);
		}
	}
	spin_unlock_bh(&hash_lock);
}

/* Internal Function: Find entry based on mac address in specific list */
static struct ebt_dhcpsnoop_db_entry *db_find(struct hlist_head *head,
					const unsigned char *addr)
{
	struct ebt_dhcpsnoop_db_entry *db;

	hlist_for_each_entry(db, head, hlist) {
		if (ether_addr_equal(db->addr, addr))
			return db;
	}
	return NULL;
}

/* Find entry based on mac address */
struct ebt_dhcpsnoop_db_entry *ebt_dhcpsnoop_db_find(const unsigned char *addr)
{
	struct hlist_head *head = &hash[ebt_dhcpsnoop_mac_hash(addr)];
	struct ebt_dhcpsnoop_db_entry *db = NULL;

	spin_lock_bh(&hash_lock);
	db = db_find(head, addr);
	spin_unlock_bh(&hash_lock);
	return db;
}

/* Internal Function: Create entry based on input interface and mac address
   in specific list */
static struct ebt_dhcpsnoop_db_entry *db_create(struct hlist_head *head,
					  int iif,
					  const unsigned char *addr)
{
	struct ebt_dhcpsnoop_db_entry *db;

	db = kmem_cache_alloc(ebt_dhcpsnoop_db_cache, GFP_ATOMIC);
	if (db) {
		memset(db, 0, sizeof(struct ebt_dhcpsnoop_db_entry));
		memcpy(db->addr, addr, ETH_ALEN);
		db->iif = iif;
		atomic_set(&db->use, 1);
		db->created = jiffies;
		hlist_add_head_rcu(&db->hlist, head);
	}
	return db;
}

/* Internal Function: Delete entry based on mac address */
static int db_delete_by_addr(const u8 *addr)
{
	struct hlist_head *head = &hash[ebt_dhcpsnoop_mac_hash(addr)];
	struct ebt_dhcpsnoop_db_entry *db;

	db = db_find(head, addr);
	if (!db)
		return -ENOENT;

	db_delete(db);
	return 0;
}

/* Delete entry based on mac address */
int ebt_dhcpsnoop_db_delete(const unsigned char *addr)
{
	int err;

	spin_lock_bh(&hash_lock);
	err = db_delete_by_addr(addr);
	spin_unlock_bh(&hash_lock);
	return err;
}

/* Internal Function: Update entry based on input interface and mac address */
static int db_update(int iif, const unsigned char *addr,
		     const __u8 pkt_type,
		     const __be32 *ipaddr,
		     const __u8 *hostname,
		     const __u32 *leasetime)
{
	struct hlist_head *head = &hash[ebt_dhcpsnoop_mac_hash(addr)];
	struct ebt_dhcpsnoop_db_entry *db;

	if (!is_valid_ether_addr(addr))
		return -EINVAL;

	db = db_find(head, addr);
	if (!db)
		db = db_create(head, iif, addr);
	if (!db)
		return -ENOMEM;
	db->iif = iif;
	db->created = jiffies;
	db->status = pkt_type;
	if (leasetime)
		db->leasetime = ntohl(*leasetime);
	if (ipaddr)
		db->ipv4addr = *ipaddr;
	if (hostname)
		strncpy(db->hostname, hostname, sizeof(db->hostname)-1);
	return 0;
}

/* Update entry based on input interface and mac address */
int ebt_dhcpsnoop_db_update(const struct net_device *in,
			    const unsigned char *addr,
			    const __u8 pkt_type,
			    const __be32 *ipaddr,
			    const __u8 *hostname,
			    const __u32 *leasetime)
{
	int ret;

	spin_lock_bh(&hash_lock);
	ret = db_update(in->ifindex, addr, pkt_type,
			ipaddr, hostname, leasetime);
	spin_unlock_bh(&hash_lock);
	return ret;
}
EXPORT_SYMBOL(ebt_dhcpsnoop_db_update);

/* Get first entry in specific list */
struct hlist_node  *ebt_dhcpsnoop_db_get_first(int *hashid)
{
	struct hlist_node *h;
	int i;

	for (i = 0; i < EBT_DHCPSNOOP_HASH_SIZE; i++) {
		h = rcu_dereference(hlist_first_rcu(&hash[i]));
		if (h) {
			*hashid = i;
			return h;
		}
	}
	return NULL;
}

/* Get next entry in specific list */
struct hlist_node *ebt_dhcpsnoop_db_get_next(int *hashid,
				       struct hlist_node *head)
{
	head = rcu_dereference(hlist_next_rcu(head));
	while (!head) {
		if (++*hashid >= EBT_DHCPSNOOP_HASH_SIZE)
			return NULL;
		head = rcu_dereference(
				hlist_first_rcu(
				   &hash[*hashid]));
	}
	return head;
}

/* Get index in specific list */
struct hlist_node *ebt_dhcpsnoop_db_get_idx(int *hashid, loff_t pos)
{
	struct hlist_node *head = ebt_dhcpsnoop_db_get_first(hashid);

	if (head)
		while (pos && (head = ebt_dhcpsnoop_db_get_next(hashid, head)))
			pos--;
	return pos ? NULL : head;
}

static void *ebt_dhcpsnoop_db_seq_start(struct seq_file *seq, loff_t *pos)
	__acquires(RCU)
{
	rcu_read_lock();
	seq->private = 0;
	return ebt_dhcpsnoop_db_get_idx((int *)&seq->private, *pos);
}

static void *ebt_dhcpsnoop_db_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	(*pos)++;
	return ebt_dhcpsnoop_db_get_next((int *)&seq->private, v);
}

static void ebt_dhcpsnoop_db_seq_stop(struct seq_file *seq, void *v)
	__releases(RCU)
{
	rcu_read_unlock();
}

static int ebt_dhcpsnoop_db_seq_show(struct seq_file *seq, void *v)
{
	struct ebt_dhcpsnoop_db_entry *db = v;
	struct net_device *dev;
	struct timeval tv;

	if (!v)
		return -1;
	jiffies_to_timeval(jiffies - db->created, &tv);

	dev = __dev_get_by_index(&init_net, db->iif);
	seq_printf(seq,
		   " mac=%pM inf=%s status=%d ip=%pI4"
		   " hostname=%s lease=%d life=%ld\n",
		   db->addr, dev->name, db->status, &db->ipv4addr,
		   db->hostname, db->leasetime, tv.tv_sec);
	return 0;

}

static const struct seq_operations ebt_dhcpsnoop_db_seq_ops = {
	.start	= ebt_dhcpsnoop_db_seq_start,
	.next	= ebt_dhcpsnoop_db_seq_next,
	.stop	= ebt_dhcpsnoop_db_seq_stop,
	.show	= ebt_dhcpsnoop_db_seq_show,
};

static int ebt_dhcpsnoop_db_seq_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &ebt_dhcpsnoop_db_seq_ops);
	return ret;
};

#define PROCFS_BUFF_MAX 256

ssize_t ebt_dhcpsnoop_db_seq_write(struct file *file, const char __user *buffer,
				   size_t count, loff_t *off)
{
	u32 buf_size = count;
	__u8 addr[6];
	char proc_fs_buffer[PROCFS_BUFF_MAX];

	memset(proc_fs_buffer, 0, sizeof(proc_fs_buffer));
	if (buf_size > 0) {
		if (buf_size > PROCFS_BUFF_MAX)
			buf_size = PROCFS_BUFF_MAX;

		if (copy_from_user(proc_fs_buffer, buffer, buf_size))
			return -EFAULT;
		proc_fs_buffer[buf_size-1] = '\0';
		if (strncmp(proc_fs_buffer, "flush", strlen("flush")) == 0) {
			ebt_dhcpsnoop_db_flush();
		} else if (strncmp(proc_fs_buffer, "del", strlen("del")) == 0) {
			if ((strlen("del")+1) < strlen(proc_fs_buffer)) {
				/* coverity [tainted_data] */
				if (mac_pton(&proc_fs_buffer[strlen("del")+1], addr)) {
					ebt_dhcpsnoop_db_delete(addr);
				}
			}
		}
	}
	return buf_size;
}

static const struct file_operations ebt_dhcpsnoop_proc_db_fops = {
	.owner    = THIS_MODULE,
	.open     = ebt_dhcpsnoop_db_seq_open,
	.read     = seq_read,
	.write    = ebt_dhcpsnoop_db_seq_write,
	.llseek   = seq_lseek,
	.release  = seq_release,
};

void ebt_dhcpsnoop_procfs_init(void)
{
	ebt_dhcpsnoop_proc_dir = proc_mkdir(EBT_DHCPSNOOP_PROC_DIR_NAME, NULL);
	if (ebt_dhcpsnoop_proc_dir == NULL) {
		pr_warn("EBT_DHCPSNOOP Warning: cannot create /proc/%s\n",
			EBT_DHCPSNOOP_PROC_DIR_NAME);
		return;
	}
	proc_create_data("db", S_IRUGO, ebt_dhcpsnoop_proc_dir,
			 &ebt_dhcpsnoop_proc_db_fops, NULL);
}

void ebt_dhcpsnoop_procfs_fini(void)
{
	if (ebt_dhcpsnoop_proc_dir) {
		remove_proc_entry("db", ebt_dhcpsnoop_proc_dir);
		remove_proc_entry(EBT_DHCPSNOOP_PROC_DIR_NAME, NULL);
		ebt_dhcpsnoop_proc_dir = NULL;
	}
}

static int use_count;
int ebt_dhcpsnoop_db_init(void)
{
	use_count++;
	if (use_count > 1)
		return 0;
	ebt_dhcpsnoop_db_flush();
	ebt_dhcpsnoop_procfs_init();
	return 0;
}
EXPORT_SYMBOL(ebt_dhcpsnoop_db_init);

void ebt_dhcpsnoop_db_fini(void)
{
	use_count--;
	if (use_count > 0)
		return;
	ebt_dhcpsnoop_db_flush();
	ebt_dhcpsnoop_procfs_fini();
}
EXPORT_SYMBOL(ebt_dhcpsnoop_db_fini);

static int __init ebt_dhcpsnoop_db_mod_init(void)
{
	ebt_dhcpsnoop_db_cache =
		kmem_cache_create("ebt_dhcpsnoop_db_cache",
				  sizeof(struct ebt_dhcpsnoop_db_entry),
				  0,
				  SLAB_HWCACHE_ALIGN, NULL);
	if (!ebt_dhcpsnoop_db_cache)
		return -ENOMEM;

	get_random_bytes(&db_salt, sizeof(db_salt));
	return 0;
}

static void __exit ebt_dhcpsnoop_db_mod_fini(void)
{
	kmem_cache_destroy(ebt_dhcpsnoop_db_cache);
}
module_init(ebt_dhcpsnoop_db_mod_init);
module_exit(ebt_dhcpsnoop_db_mod_fini);
MODULE_DESCRIPTION("Ebtables: DHCP Snoop Client DB");
MODULE_LICENSE("GPL");
