 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2018 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/io.h>

#include <proc_cmd.h>
#include "vfbioss_priv.h"
#include "vfbioss_proc.h"

#define PROC_DIR		"driver/vfbioss"
#define CMD_PROC_FILE		"cmd"
#define DUMP_PROC_FILE		"dump"

struct seq_buf {
	u8	*buf;
	u32	remain;
	loff_t	offset;
	u32	count;
	u8	line[10 + 32 * 3 + 2 + 32 + 1];
};
static struct seq_buf sbuf;
static int lun;

static int vfbioss_proc_cmd_help(int argc, char *argv[])
{
	int status = 0;

	if (argc != 2) {
		pr_err("Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	if (!strncmp(argv[1], "dump", sizeof("dump"))) {
		pr_info("dump [lun]\n");
		pr_info("\tlun    set the LUN # to dump\n");
	} else {
		pr_err("Unrecognized command: %s\n", argv[1]);
		status = -EINVAL;
	}

done:
	return status;
}

static int vfbioss_proc_cmd_dump(int argc, char *argv[])
{
	int status = 0;

	if (argc != 2) {
		pr_err("Invalid # of arguments.\n");
		status = -EINVAL;
		goto done;
	}

	status = kstrtou32(argv[1], 0, &lun);
	if (status)
		pr_err("Bad value: %s\n", argv[1]);

done:
	return status;
}

static void *vfbioss_dump_proc_start(struct seq_file *seq, loff_t *pos)
{
	struct seq_buf *sb = &sbuf;

	if (*pos == 0) {
		sb->offset = 0;
		sb->buf = (u8 *)luns[lun].addr + sb->offset;
		sb->remain = luns[lun].n_blks * luns[lun].blk_sz;
		sb->count = 0;
	}
	if (!sb->remain)
		return NULL;
	else
		return sb;
}

static void vfbioss_buf_proc_stop(struct seq_file *seq, void *v)
{
}

static void *vfbioss_buf_proc_next(struct seq_file *seq, void *v,
			       loff_t *pos)
{
	struct seq_buf *sb = v;

	if (!sb->remain) {
		sb = NULL;
	}

	return sb;
}

static int vfbioss_buf_proc_show(struct seq_file *seq, void *v)
{
	struct seq_buf *sb = v;
	int len;
	bool repeat;
	static bool print_repeat = true;

	len = sb->remain < 16 ? sb->remain : 16;
	repeat = sb->count >= 16 ?
		memcmp(sb->buf, sb->buf - 16, len) == 0 : false;
	if (!repeat || sb->remain == len) {
		snprintf(sb->line, 11, "%8llx: ", sb->offset);
		hex_dump_to_buffer(sb->buf, len, 16, 1,
				   &sb->line[10], sizeof(sb->line)-10, true);
		seq_printf(seq, "%s\n", sb->line);
		print_repeat = true;
	} else if (print_repeat) {
		seq_printf(seq, "*\n");
		print_repeat = false;
	}
	sb->remain -= len;
	sb->buf += len;
	sb->offset += len;
	sb->count += len;

	return 0;
}

static const struct seq_operations dump_seq_ops = {
	.start	= vfbioss_dump_proc_start,
	.stop	= vfbioss_buf_proc_stop,
	.next	= vfbioss_buf_proc_next,
	.show	= vfbioss_buf_proc_show,
};
static int vfbioss_dump_proc_open(struct inode *inode, struct file *file)
{
	int fid;
	fid = seq_open(file, &dump_seq_ops);
	return fid;
}
static const struct file_operations dump_fops = {
	.owner		= THIS_MODULE,
	.open		= vfbioss_dump_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= seq_release,
};

static struct proc_cmd_ops command_entries[] = {
	{ .name = "help", .do_command	= vfbioss_proc_cmd_help},
	{ .name = "dump", .do_command	= vfbioss_proc_cmd_dump},
};

struct proc_cmd_table vfbioss_command_table = {
	.module_name = "VFBIOSS",
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

static struct proc_dir_entry *proc_dir;
static struct proc_dir_entry *cmd_proc_file;
static struct proc_dir_entry *dump_proc_file;

void vfbioss_proc_exit(void)
{
	if (cmd_proc_file) {
		remove_proc_entry(CMD_PROC_FILE, proc_dir);
		cmd_proc_file = NULL;
	}
	if (dump_proc_file) {
		remove_proc_entry(DUMP_PROC_FILE, proc_dir);
		dump_proc_file = NULL;
	}
	if (proc_dir) {
		remove_proc_entry(PROC_DIR, NULL);
		proc_dir = NULL;
	}
}

int vfbioss_proc_init(void)
{
	int status = 0;

	proc_dir = proc_mkdir(PROC_DIR, NULL);
	if (!proc_dir) {
		pr_err("Failed to create PROC directory %s.\n",
		       PROC_DIR);
		status = -EIO;
		goto done;
	}
	dump_proc_file = proc_create(DUMP_PROC_FILE, S_IRUGO,
				    proc_dir, &dump_fops);
	if (!dump_proc_file) {
		pr_err("Failed to create %s\n", DUMP_PROC_FILE);
		status = -EIO;
		vfbioss_proc_exit();
		goto done;
	}
	cmd_proc_file = proc_create_cmd(CMD_PROC_FILE, proc_dir,
					&vfbioss_command_table);
	if (!cmd_proc_file) {
		pr_err("Failed to create %s\n", CMD_PROC_FILE);
		status = -EIO;
		vfbioss_proc_exit();
		goto done;
	}

done:
	return status;
}
