/*
 * BCM SPU crypto testing module based on tcrypt and testmgr code
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 */
#ifndef _CRYPTO_TCRYPT_H
#define _CRYPTO_TCRYPT_H

struct cipher_speed_template {
	const char *key;
	unsigned int klen;
};

struct aead_speed_template {
	const char *key;
	unsigned int klen;
};

struct hash_speed {
	unsigned int blen;	/* buffer length */
	unsigned int plen;	/* per-update length */
	unsigned int klen;	/* key length */
};

/*
 * DES test vectors.
 */
#define DES3_SPEED_VECTORS	1

static struct cipher_speed_template des3_speed_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\x55\x55\x55\x55\x55\x55\x55\x55"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 24,
	}
};

/*
 * Cipher speed tests
 */
static u8 speed_template_8[] = {8, 0};
static u8 speed_template_24[] = {24, 0};
static u8 speed_template_16_24_32[] = {16, 24, 32, 0};
static u8 speed_template_32_48_64[] = {32, 48, 64, 0};

/*
 * AEAD speed tests
 */
static u8 aead_speed_template_19[] = {19, 0};
static u8 aead_speed_template_20[] = {20, 0};

/*
 * Digest speed tests
 */
static struct hash_speed generic_hash_speed_template[] = {
	{ .blen = 16,	.plen = 16, },
	{ .blen = 64,	.plen = 16, },
	{ .blen = 64,	.plen = 64, },
	{ .blen = 256,	.plen = 16, },
	{ .blen = 256,	.plen = 64, },
	{ .blen = 256,	.plen = 256, },
	{ .blen = 1024,	.plen = 16, },
	{ .blen = 1024,	.plen = 256, },
	{ .blen = 1024,	.plen = 1024, },
	{ .blen = 2048,	.plen = 16, },
	{ .blen = 2048,	.plen = 256, },
	{ .blen = 2048,	.plen = 1024, },
	{ .blen = 2048,	.plen = 2048, },
	{ .blen = 4096,	.plen = 16, },
	{ .blen = 4096,	.plen = 256, },
	{ .blen = 4096,	.plen = 1024, },
	{ .blen = 4096,	.plen = 4096, },
	{ .blen = 8192,	.plen = 16, },
	{ .blen = 8192,	.plen = 256, },
	{ .blen = 8192,	.plen = 1024, },
	{ .blen = 8192,	.plen = 4096, },
	{ .blen = 8192,	.plen = 8192, },

	/* End marker */
	{  .blen = 0,	.plen = 0, }
};

static struct hash_speed hash_speed_template_16[] = {
	{ .blen = 16,	.plen = 16,	.klen = 16, },
	{ .blen = 64,	.plen = 16,	.klen = 16, },
	{ .blen = 64,	.plen = 64,	.klen = 16, },
	{ .blen = 256,	.plen = 16,	.klen = 16, },
	{ .blen = 256,	.plen = 64,	.klen = 16, },
	{ .blen = 256,	.plen = 256,	.klen = 16, },
	{ .blen = 1024,	.plen = 16,	.klen = 16, },
	{ .blen = 1024,	.plen = 256,	.klen = 16, },
	{ .blen = 1024,	.plen = 1024,	.klen = 16, },
	{ .blen = 2048,	.plen = 16,	.klen = 16, },
	{ .blen = 2048,	.plen = 256,	.klen = 16, },
	{ .blen = 2048,	.plen = 1024,	.klen = 16, },
	{ .blen = 2048,	.plen = 2048,	.klen = 16, },
	{ .blen = 4096,	.plen = 16,	.klen = 16, },
	{ .blen = 4096,	.plen = 256,	.klen = 16, },
	{ .blen = 4096,	.plen = 1024,	.klen = 16, },
	{ .blen = 4096,	.plen = 4096,	.klen = 16, },
	{ .blen = 8192,	.plen = 16,	.klen = 16, },
	{ .blen = 8192,	.plen = 256,	.klen = 16, },
	{ .blen = 8192,	.plen = 1024,	.klen = 16, },
	{ .blen = 8192,	.plen = 4096,	.klen = 16, },
	{ .blen = 8192,	.plen = 8192,	.klen = 16, },

	/* End marker */
	{  .blen = 0,	.plen = 0,	.klen = 0, }
};

#define MAX_DIGEST_SIZE		64
#define MAX_TAP			8

#define MAX_KEYLEN		160
#define MAX_IVLEN		32

struct hash_testvec {
	/* only used with keyed hash algorithms */
	char *key;
	char *plaintext;
	char *digest;
	unsigned char tap[MAX_TAP];
	unsigned short psize;
	unsigned char np;
	unsigned char ksize;
};

/*
 * cipher_testvec:	structure to describe a cipher test
 * @key:	A pointer to a key used by the test
 * @klen:	The length of @key
 * @iv:		A pointer to the IV used by the test
 * @input:	A pointer to data used as input
 * @ilen	The length of data in @input
 * @result:	A pointer to what the test need to produce
 * @rlen:	The length of data in @result
 * @fail:	If set to one, the test need to fail
 * @wk:		Does the test need CRYPTO_TFM_REQ_WEAK_KEY
 * 		( e.g. test needs to fail due to a weak key )
 * @np: 	numbers of SG to distribute data in (from 1 to MAX_TAP)
 * @tap:	How to distribute data in @np SGs
 * @also_non_np: 	if set to 1, the test will be also done without
 * 			splitting data in @np SGs
 * @fips_skip:	Skip the test vector in FIPS mode
 */

struct cipher_testvec {
	char *key;
	char *iv;
	char *iv_out;
	char *input;
	char *result;
	unsigned short tap[MAX_TAP];
	int np;
	unsigned char also_non_np;
	bool fail;
	unsigned char wk; /* weak key flag */
	unsigned char klen;
	unsigned short ilen;
	unsigned short rlen;
	bool fips_skip;
};

struct aead_testvec {
	char *key;
	char *iv;
	char *input;
	char *assoc;
	char *result;
	unsigned char tap[MAX_TAP];
	unsigned char atap[MAX_TAP];
	int np;
	int anp;
	bool fail;
	unsigned char novrfy;	/* ccm dec verification failure expected */
	unsigned char wk; /* weak key flag */
	unsigned char klen;
	unsigned short ilen;
	unsigned short alen;
	unsigned short rlen;
};

struct akcipher_testvec {
	unsigned char *key;
	unsigned char *m;
	unsigned char *c;
	unsigned int key_len;
	unsigned int m_size;
	unsigned int c_size;
	bool public_key_vec;
};

static char zeroed_string[48];
#if 0
#define SHA3_224_TEST_VECTORS	3
static struct hash_testvec sha3_224_tv_template[] = {
	{
		.plaintext = "",
		.digest	= "\x6b\x4e\x03\x42\x36\x67\xdb\xb7"
				"\x3b\x6e\x15\x45\x4f\x0e\xb1\xab"
				"\xd4\x59\x7f\x9a\x1b\x07\x8e\x3f"
				"\x5b\x5a\x6b\xc7",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x9e\x86\xff\x69\x55\x7c\xa9\x5f"
				"\x40\x5f\x08\x12\x69\x68\x5b\x38"
				"\xe3\xa8\x19\xb3\x09\xee\x94\x2f"
				"\x48\x2b\x6a\x8b",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkl"
				"jklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x8a\x24\x10\x8b\x15\x4a\xda\x21"
				"\xc9\xfd\x55\x74\x49\x44\x79\xba"
				"\x5c\x7e\x7a\xb7\x6e\xf2\x64\xea"
				"\xd0\xfc\xce\x33",
	},
};

#define SHA3_256_TEST_VECTORS	3
static struct hash_testvec sha3_256_tv_template[] = {
	{
		.plaintext = "",
		.digest	= "\xa7\xff\xc6\xf8\xbf\x1e\xd7\x66"
				"\x51\xc1\x47\x56\xa0\x61\xd6\x62"
				"\xf5\x80\xff\x4d\xe4\x3b\x49\xfa"
				"\x82\xd8\x0a\x4b\x80\xf8\x43\x4a",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x80\x08\x4b\xf2\xfb\xa0\x24\x75"
				"\x72\x6f\xeb\x2c\xab\x2d\x82\x15"
				"\xea\xb1\x4b\xc6\xbd\xd8\xbf\xb2"
				"\xc8\x15\x12\x57\x03\x2e\xcd\x8b",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkl"
			     "jklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x41\xc0\xdb\xa2\xa9\xd6\x24\x08"
				"\x49\x10\x03\x76\xa8\x23\x5e\x2c"
				"\x82\xe1\xb9\x99\x8a\x99\x9e\x21"
				"\xdb\x32\xdd\x97\x49\x6d\x33\x76",
	},
};


#define SHA3_384_TEST_VECTORS	3
static struct hash_testvec sha3_384_tv_template[] = {
	{
		.plaintext = "",
		.digest	= "\x0c\x63\xa7\x5b\x84\x5e\x4f\x7d"
				"\x01\x10\x7d\x85\x2e\x4c\x24\x85"
				"\xc5\x1a\x50\xaa\xaa\x94\xfc\x61"
				"\x99\x5e\x71\xbb\xee\x98\x3a\x2a"
				"\xc3\x71\x38\x31\x26\x4a\xdb\x47"
				"\xfb\x6b\xd1\xe0\x58\xd5\xf0\x04",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x18\x15\xf7\x74\xf3\x20\x49\x1b"
				"\x48\x56\x9e\xfe\xc7\x94\xd2\x49"
				"\xee\xb5\x9a\xae\x46\xd2\x2b\xf7"
				"\x7d\xaf\xe2\x5c\x5e\xdc\x28\xd7"
				"\xea\x44\xf9\x3e\xe1\x23\x4a\xa8"
				"\x8f\x61\xc9\x19\x12\xa4\xcc\xd9",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkl"
			     "jklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x99\x1c\x66\x57\x55\xeb\x3a\x4b"
				"\x6b\xbd\xfb\x75\xc7\x8a\x49\x2e"
				"\x8c\x56\xa2\x2c\x5c\x4d\x7e\x42"
				"\x9b\xfd\xbc\x32\xb9\xd4\xad\x5a"
				"\xa0\x4a\x1f\x07\x6e\x62\xfe\xa1"
				"\x9e\xef\x51\xac\xd0\x65\x7c\x22",
	},
};


#define SHA3_512_TEST_VECTORS	3
static struct hash_testvec sha3_512_tv_template[] = {
	{
		.plaintext = "",
		.digest	= "\xa6\x9f\x73\xcc\xa2\x3a\x9a\xc5"
				"\xc8\xb5\x67\xdc\x18\x5a\x75\x6e"
				"\x97\xc9\x82\x16\x4f\xe2\x58\x59"
				"\xe0\xd1\xdc\xc1\x47\x5c\x80\xa6"
				"\x15\xb2\x12\x3a\xf1\xf5\xf9\x4c"
				"\x11\xe3\xe9\x40\x2c\x3a\xc5\x58"
				"\xf5\x00\x19\x9d\x95\xb6\xd3\xe3"
				"\x01\x75\x85\x86\x28\x1d\xcd\x26",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x69\x7f\x2d\x85\x61\x72\xcb\x83"
				"\x09\xd6\xb8\xb9\x7d\xac\x4d\xe3"
				"\x44\xb5\x49\xd4\xde\xe6\x1e\xdf"
				"\xb4\x96\x2d\x86\x98\xb7\xfa\x80"
				"\x3f\x4f\x93\xff\x24\x39\x35\x86"
				"\xe2\x8b\x5b\x95\x7a\xc3\xd1\xd3"
				"\x69\x42\x0c\xe5\x33\x32\x71\x2f"
				"\x99\x7b\xd3\x36\xd0\x9a\xb0\x2a",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkl"
			     "jklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x04\xa3\x71\xe8\x4e\xcf\xb5\xb8"
				"\xb7\x7c\xb4\x86\x10\xfc\xa8\x18"
				"\x2d\xd4\x57\xce\x6f\x32\x6a\x0f"
				"\xd3\xd7\xec\x2f\x1e\x91\x63\x6d"
				"\xee\x69\x1f\xbe\x0c\x98\x53\x02"
				"\xba\x1b\x0d\x8d\xc7\x8c\x08\x63"
				"\x46\xb5\x33\xb4\x9c\x03\x0d\x99"
				"\xa2\x7d\xaf\x11\x39\xd6\xe7\x5e",
	},
};
#endif

/*
 * MD5 test vectors from RFC1321
 */
#define MD5_TEST_VECTORS	7

static struct hash_testvec md5_tv_template[] = {
	{
		.digest	= "\xd4\x1d\x8c\xd9\x8f\x00\xb2\x04"
			  "\xe9\x80\x09\x98\xec\xf8\x42\x7e",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x0c\xc1\x75\xb9\xc0\xf1\xb6\xa8"
			  "\x31\xc3\x99\xe2\x69\x77\x26\x61",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\x90\x01\x50\x98\x3c\xd2\x4f\xb0"
			  "\xd6\x96\x3f\x7d\x28\xe1\x7f\x72",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\xf9\x6b\x69\x7d\x7c\xb7\x93\x8d"
			  "\x52\x5a\x2f\x31\xaa\xf1\x61\xd0",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xc3\xfc\xd3\xd7\x61\x92\xe4\x00"
			  "\x7d\xfb\x49\x6c\xca\x67\xe1\x3b",
		.np	= 2,
		.tap	= {13, 13}
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\xd1\x74\xab\x98\xd2\x77\xd9\xf5"
			  "\xa5\x61\x1c\x2c\x9f\x41\x9d\x9f",
	}, {
		.plaintext = "12345678901234567890123456789012345678901234567890123456789012"
			   "345678901234567890",
		.psize	= 80,
		.digest	= "\x57\xed\xf4\xa2\x2b\xe3\xc9\x55"
			  "\xac\x49\xda\x2e\x21\x07\xb6\x7a",
	}

};

/*
 * SHA1 test vectors  from from FIPS PUB 180-1
 * Long vector from CAVS 5.0
 */
#define SHA1_TEST_VECTORS	6

static struct hash_testvec sha1_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= "\xda\x39\xa3\xee\x5e\x6b\x4b\x0d\x32\x55"
			  "\xbf\xef\x95\x60\x18\x90\xaf\xd8\x07\x09",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xa9\x99\x3e\x36\x47\x06\x81\x6a\xba\x3e"
			  "\x25\x71\x78\x50\xc2\x6c\x9c\xd0\xd8\x9d",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x84\x98\x3e\x44\x1c\x3b\xd2\x6e\xba\xae"
			  "\x4a\xa1\xf9\x51\x29\xe5\xe5\x46\x70\xf1",
		.np	= 2,
		.tap	= { 28, 28 }
	}, {
		.plaintext = "\xec\x29\x56\x12\x44\xed\xe7\x06"
			     "\xb6\xeb\x30\xa1\xc3\x71\xd7\x44"
			     "\x50\xa1\x05\xc3\xf9\x73\x5f\x7f"
			     "\xa9\xfe\x38\xcf\x67\xf3\x04\xa5"
			     "\x73\x6a\x10\x6e\x92\xe1\x71\x39"
			     "\xa6\x81\x3b\x1c\x81\xa4\xf3\xd3"
			     "\xfb\x95\x46\xab\x42\x96\xfa\x9f"
			     "\x72\x28\x26\xc0\x66\x86\x9e\xda"
			     "\xcd\x73\xb2\x54\x80\x35\x18\x58"
			     "\x13\xe2\x26\x34\xa9\xda\x44\x00"
			     "\x0d\x95\xa2\x81\xff\x9f\x26\x4e"
			     "\xcc\xe0\xa9\x31\x22\x21\x62\xd0"
			     "\x21\xcc\xa2\x8d\xb5\xf3\xc2\xaa"
			     "\x24\x94\x5a\xb1\xe3\x1c\xb4\x13"
			     "\xae\x29\x81\x0f\xd7\x94\xca\xd5"
			     "\xdf\xaf\x29\xec\x43\xcb\x38\xd1"
			     "\x98\xfe\x4a\xe1\xda\x23\x59\x78"
			     "\x02\x21\x40\x5b\xd6\x71\x2a\x53"
			     "\x05\xda\x4b\x1b\x73\x7f\xce\x7c"
			     "\xd2\x1c\x0e\xb7\x72\x8d\x08\x23"
			     "\x5a\x90\x11",
		.psize	= 163,
		.digest	= "\x97\x01\x11\xc4\xe7\x7b\xcc\x88\xcc\x20"
			  "\x45\x9c\x02\xb6\x9b\x4a\xa8\xf5\x82\x17",
		.np	= 4,
		.tap	= { 63, 64, 31, 5 }
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = "\xc8\x71\xf6\x9a\x63\xcc\xa9\x84\x84\x82"
			  "\x64\xe7\x79\x95\x5d\xd7\x19\x41\x7c\x91",
	}, {
		.plaintext = "\x08\x9f\x13\xaa\x41\xd8\x4c\xe3"
			     "\x7a\x11\x85\x1c\xb3\x27\xbe\x55"
			     "\xec\x60\xf7\x8e\x02\x99\x30\xc7"
			     "\x3b\xd2\x69\x00\x74\x0b\xa2\x16"
			     "\xad\x44\xdb\x4f\xe6\x7d\x14\x88"
			     "\x1f\xb6\x2a\xc1\x58\xef\x63\xfa"
			     "\x91\x05\x9c\x33\xca\x3e\xd5\x6c"
			     "\x03\x77\x0e\xa5\x19\xb0\x47\xde"
			     "\x52\xe9\x80\x17\x8b\x22\xb9\x2d"
			     "\xc4\x5b\xf2\x66\xfd\x94\x08\x9f"
			     "\x36\xcd\x41\xd8\x6f\x06\x7a\x11"
			     "\xa8\x1c\xb3\x4a\xe1\x55\xec\x83"
			     "\x1a\x8e\x25\xbc\x30\xc7\x5e\xf5"
			     "\x69\x00\x97\x0b\xa2\x39\xd0\x44"
			     "\xdb\x72\x09\x7d\x14\xab\x1f\xb6"
			     "\x4d\xe4\x58\xef\x86\x1d\x91\x28"
			     "\xbf\x33\xca\x61\xf8\x6c\x03\x9a"
			     "\x0e\xa5\x3c\xd3\x47\xde\x75\x0c"
			     "\x80\x17\xae\x22\xb9\x50\xe7\x5b"
			     "\xf2\x89\x20\x94\x2b\xc2\x36\xcd"
			     "\x64\xfb\x6f\x06\x9d\x11\xa8\x3f"
			     "\xd6\x4a\xe1\x78\x0f\x83\x1a\xb1"
			     "\x25\xbc\x53\xea\x5e\xf5\x8c\x00"
			     "\x97\x2e\xc5\x39\xd0\x67\xfe\x72"
			     "\x09\xa0\x14\xab\x42\xd9\x4d\xe4"
			     "\x7b\x12\x86\x1d\xb4\x28\xbf\x56"
			     "\xed\x61\xf8\x8f\x03\x9a\x31\xc8"
			     "\x3c\xd3\x6a\x01\x75\x0c\xa3\x17"
			     "\xae\x45\xdc\x50\xe7\x7e\x15\x89"
			     "\x20\xb7\x2b\xc2\x59\xf0\x64\xfb"
			     "\x92\x06\x9d\x34\xcb\x3f\xd6\x6d"
			     "\x04\x78\x0f\xa6\x1a\xb1\x48\xdf"
			     "\x53\xea\x81\x18\x8c\x23\xba\x2e"
			     "\xc5\x5c\xf3\x67\xfe\x95\x09\xa0"
			     "\x37\xce\x42\xd9\x70\x07\x7b\x12"
			     "\xa9\x1d\xb4\x4b\xe2\x56\xed\x84"
			     "\x1b\x8f\x26\xbd\x31\xc8\x5f\xf6"
			     "\x6a\x01\x98\x0c\xa3\x3a\xd1\x45"
			     "\xdc\x73\x0a\x7e\x15\xac\x20\xb7"
			     "\x4e\xe5\x59\xf0\x87\x1e\x92\x29"
			     "\xc0\x34\xcb\x62\xf9\x6d\x04\x9b"
			     "\x0f\xa6\x3d\xd4\x48\xdf\x76\x0d"
			     "\x81\x18\xaf\x23\xba\x51\xe8\x5c"
			     "\xf3\x8a\x21\x95\x2c\xc3\x37\xce"
			     "\x65\xfc\x70\x07\x9e\x12\xa9\x40"
			     "\xd7\x4b\xe2\x79\x10\x84\x1b\xb2"
			     "\x26\xbd\x54\xeb\x5f\xf6\x8d\x01"
			     "\x98\x2f\xc6\x3a\xd1\x68\xff\x73"
			     "\x0a\xa1\x15\xac\x43\xda\x4e\xe5"
			     "\x7c\x13\x87\x1e\xb5\x29\xc0\x57"
			     "\xee\x62\xf9\x90\x04\x9b\x32\xc9"
			     "\x3d\xd4\x6b\x02\x76\x0d\xa4\x18"
			     "\xaf\x46\xdd\x51\xe8\x7f\x16\x8a"
			     "\x21\xb8\x2c\xc3\x5a\xf1\x65\xfc"
			     "\x93\x07\x9e\x35\xcc\x40\xd7\x6e"
			     "\x05\x79\x10\xa7\x1b\xb2\x49\xe0"
			     "\x54\xeb\x82\x19\x8d\x24\xbb\x2f"
			     "\xc6\x5d\xf4\x68\xff\x96\x0a\xa1"
			     "\x38\xcf\x43\xda\x71\x08\x7c\x13"
			     "\xaa\x1e\xb5\x4c\xe3\x57\xee\x85"
			     "\x1c\x90\x27\xbe\x32\xc9\x60\xf7"
			     "\x6b\x02\x99\x0d\xa4\x3b\xd2\x46"
			     "\xdd\x74\x0b\x7f\x16\xad\x21\xb8"
			     "\x4f\xe6\x5a\xf1\x88\x1f\x93\x2a"
			     "\xc1\x35\xcc\x63\xfa\x6e\x05\x9c"
			     "\x10\xa7\x3e\xd5\x49\xe0\x77\x0e"
			     "\x82\x19\xb0\x24\xbb\x52\xe9\x5d"
			     "\xf4\x8b\x22\x96\x2d\xc4\x38\xcf"
			     "\x66\xfd\x71\x08\x9f\x13\xaa\x41"
			     "\xd8\x4c\xe3\x7a\x11\x85\x1c\xb3"
			     "\x27\xbe\x55\xec\x60\xf7\x8e\x02"
			     "\x99\x30\xc7\x3b\xd2\x69\x00\x74"
			     "\x0b\xa2\x16\xad\x44\xdb\x4f\xe6"
			     "\x7d\x14\x88\x1f\xb6\x2a\xc1\x58"
			     "\xef\x63\xfa\x91\x05\x9c\x33\xca"
			     "\x3e\xd5\x6c\x03\x77\x0e\xa5\x19"
			     "\xb0\x47\xde\x52\xe9\x80\x17\x8b"
			     "\x22\xb9\x2d\xc4\x5b\xf2\x66\xfd"
			     "\x94\x08\x9f\x36\xcd\x41\xd8\x6f"
			     "\x06\x7a\x11\xa8\x1c\xb3\x4a\xe1"
			     "\x55\xec\x83\x1a\x8e\x25\xbc\x30"
			     "\xc7\x5e\xf5\x69\x00\x97\x0b\xa2"
			     "\x39\xd0\x44\xdb\x72\x09\x7d\x14"
			     "\xab\x1f\xb6\x4d\xe4\x58\xef\x86"
			     "\x1d\x91\x28\xbf\x33\xca\x61\xf8"
			     "\x6c\x03\x9a\x0e\xa5\x3c\xd3\x47"
			     "\xde\x75\x0c\x80\x17\xae\x22\xb9"
			     "\x50\xe7\x5b\xf2\x89\x20\x94\x2b"
			     "\xc2\x36\xcd\x64\xfb\x6f\x06\x9d"
			     "\x11\xa8\x3f\xd6\x4a\xe1\x78\x0f"
			     "\x83\x1a\xb1\x25\xbc\x53\xea\x5e"
			     "\xf5\x8c\x00\x97\x2e\xc5\x39\xd0"
			     "\x67\xfe\x72\x09\xa0\x14\xab\x42"
			     "\xd9\x4d\xe4\x7b\x12\x86\x1d\xb4"
			     "\x28\xbf\x56\xed\x61\xf8\x8f\x03"
			     "\x9a\x31\xc8\x3c\xd3\x6a\x01\x75"
			     "\x0c\xa3\x17\xae\x45\xdc\x50\xe7"
			     "\x7e\x15\x89\x20\xb7\x2b\xc2\x59"
			     "\xf0\x64\xfb\x92\x06\x9d\x34\xcb"
			     "\x3f\xd6\x6d\x04\x78\x0f\xa6\x1a"
			     "\xb1\x48\xdf\x53\xea\x81\x18\x8c"
			     "\x23\xba\x2e\xc5\x5c\xf3\x67\xfe"
			     "\x95\x09\xa0\x37\xce\x42\xd9\x70"
			     "\x07\x7b\x12\xa9\x1d\xb4\x4b\xe2"
			     "\x56\xed\x84\x1b\x8f\x26\xbd\x31"
			     "\xc8\x5f\xf6\x6a\x01\x98\x0c\xa3"
			     "\x3a\xd1\x45\xdc\x73\x0a\x7e\x15"
			     "\xac\x20\xb7\x4e\xe5\x59\xf0\x87"
			     "\x1e\x92\x29\xc0\x34\xcb\x62\xf9"
			     "\x6d\x04\x9b\x0f\xa6\x3d\xd4\x48"
			     "\xdf\x76\x0d\x81\x18\xaf\x23\xba"
			     "\x51\xe8\x5c\xf3\x8a\x21\x95\x2c"
			     "\xc3\x37\xce\x65\xfc\x70\x07\x9e"
			     "\x12\xa9\x40\xd7\x4b\xe2\x79\x10"
			     "\x84\x1b\xb2\x26\xbd\x54\xeb\x5f"
			     "\xf6\x8d\x01\x98\x2f\xc6\x3a\xd1"
			     "\x68\xff\x73\x0a\xa1\x15\xac\x43"
			     "\xda\x4e\xe5\x7c\x13\x87\x1e\xb5"
			     "\x29\xc0\x57\xee\x62\xf9\x90\x04"
			     "\x9b\x32\xc9\x3d\xd4\x6b\x02\x76"
			     "\x0d\xa4\x18\xaf\x46\xdd\x51\xe8"
			     "\x7f\x16\x8a\x21\xb8\x2c\xc3\x5a"
			     "\xf1\x65\xfc\x93\x07\x9e\x35\xcc"
			     "\x40\xd7\x6e\x05\x79\x10\xa7\x1b"
			     "\xb2\x49\xe0\x54\xeb\x82\x19\x8d"
			     "\x24\xbb\x2f\xc6\x5d\xf4\x68\xff"
			     "\x96\x0a\xa1\x38\xcf\x43\xda\x71"
			     "\x08\x7c\x13\xaa\x1e\xb5\x4c",
		.psize     = 1023,
		.digest    = "\xb8\xe3\x54\xed\xc5\xfc\xef\xa4"
			     "\x55\x73\x4a\x81\x99\xe4\x47\x2a"
			     "\x30\xd6\xc9\x85",
	}
};


/*
 * SHA224 test vectors from from FIPS PUB 180-2
 */
#define SHA224_TEST_VECTORS     5

static struct hash_testvec sha224_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= "\xd1\x4a\x02\x8c\x2a\x3a\x2b\xc9"
			  "\x47\x61\x02\xbb\x28\x82\x34\xc4"
			  "\x15\xa2\xb0\x1f\x82\x8e\xa6\x2a"
			  "\xc5\xb3\xe4\x2f",
	}, {
		.plaintext = "abc",
		.psize  = 3,
		.digest = "\x23\x09\x7D\x22\x34\x05\xD8\x22"
			  "\x86\x42\xA4\x77\xBD\xA2\x55\xB3"
			  "\x2A\xAD\xBC\xE4\xBD\xA0\xB3\xF7"
			  "\xE3\x6C\x9D\xA7",
	}, {
		.plaintext =
		"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize  = 56,
		.digest = "\x75\x38\x8B\x16\x51\x27\x76\xCC"
			  "\x5D\xBA\x5D\xA1\xFD\x89\x01\x50"
			  "\xB0\xC6\x45\x5C\xB4\xF5\x8B\x19"
			  "\x52\x52\x25\x25",
		.np     = 2,
		.tap    = { 28, 28 }
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = "\xc4\xdb\x2b\x3a\x58\xc3\x99\x01"
			  "\x42\xfd\x10\x92\xaa\x4e\x04\x08"
			  "\x58\xbb\xbb\xe8\xf8\x14\xa7\x0c"
			  "\xef\x3b\xcb\x0e",
	}, {
		.plaintext = "\x08\x9f\x13\xaa\x41\xd8\x4c\xe3"
			     "\x7a\x11\x85\x1c\xb3\x27\xbe\x55"
			     "\xec\x60\xf7\x8e\x02\x99\x30\xc7"
			     "\x3b\xd2\x69\x00\x74\x0b\xa2\x16"
			     "\xad\x44\xdb\x4f\xe6\x7d\x14\x88"
			     "\x1f\xb6\x2a\xc1\x58\xef\x63\xfa"
			     "\x91\x05\x9c\x33\xca\x3e\xd5\x6c"
			     "\x03\x77\x0e\xa5\x19\xb0\x47\xde"
			     "\x52\xe9\x80\x17\x8b\x22\xb9\x2d"
			     "\xc4\x5b\xf2\x66\xfd\x94\x08\x9f"
			     "\x36\xcd\x41\xd8\x6f\x06\x7a\x11"
			     "\xa8\x1c\xb3\x4a\xe1\x55\xec\x83"
			     "\x1a\x8e\x25\xbc\x30\xc7\x5e\xf5"
			     "\x69\x00\x97\x0b\xa2\x39\xd0\x44"
			     "\xdb\x72\x09\x7d\x14\xab\x1f\xb6"
			     "\x4d\xe4\x58\xef\x86\x1d\x91\x28"
			     "\xbf\x33\xca\x61\xf8\x6c\x03\x9a"
			     "\x0e\xa5\x3c\xd3\x47\xde\x75\x0c"
			     "\x80\x17\xae\x22\xb9\x50\xe7\x5b"
			     "\xf2\x89\x20\x94\x2b\xc2\x36\xcd"
			     "\x64\xfb\x6f\x06\x9d\x11\xa8\x3f"
			     "\xd6\x4a\xe1\x78\x0f\x83\x1a\xb1"
			     "\x25\xbc\x53\xea\x5e\xf5\x8c\x00"
			     "\x97\x2e\xc5\x39\xd0\x67\xfe\x72"
			     "\x09\xa0\x14\xab\x42\xd9\x4d\xe4"
			     "\x7b\x12\x86\x1d\xb4\x28\xbf\x56"
			     "\xed\x61\xf8\x8f\x03\x9a\x31\xc8"
			     "\x3c\xd3\x6a\x01\x75\x0c\xa3\x17"
			     "\xae\x45\xdc\x50\xe7\x7e\x15\x89"
			     "\x20\xb7\x2b\xc2\x59\xf0\x64\xfb"
			     "\x92\x06\x9d\x34\xcb\x3f\xd6\x6d"
			     "\x04\x78\x0f\xa6\x1a\xb1\x48\xdf"
			     "\x53\xea\x81\x18\x8c\x23\xba\x2e"
			     "\xc5\x5c\xf3\x67\xfe\x95\x09\xa0"
			     "\x37\xce\x42\xd9\x70\x07\x7b\x12"
			     "\xa9\x1d\xb4\x4b\xe2\x56\xed\x84"
			     "\x1b\x8f\x26\xbd\x31\xc8\x5f\xf6"
			     "\x6a\x01\x98\x0c\xa3\x3a\xd1\x45"
			     "\xdc\x73\x0a\x7e\x15\xac\x20\xb7"
			     "\x4e\xe5\x59\xf0\x87\x1e\x92\x29"
			     "\xc0\x34\xcb\x62\xf9\x6d\x04\x9b"
			     "\x0f\xa6\x3d\xd4\x48\xdf\x76\x0d"
			     "\x81\x18\xaf\x23\xba\x51\xe8\x5c"
			     "\xf3\x8a\x21\x95\x2c\xc3\x37\xce"
			     "\x65\xfc\x70\x07\x9e\x12\xa9\x40"
			     "\xd7\x4b\xe2\x79\x10\x84\x1b\xb2"
			     "\x26\xbd\x54\xeb\x5f\xf6\x8d\x01"
			     "\x98\x2f\xc6\x3a\xd1\x68\xff\x73"
			     "\x0a\xa1\x15\xac\x43\xda\x4e\xe5"
			     "\x7c\x13\x87\x1e\xb5\x29\xc0\x57"
			     "\xee\x62\xf9\x90\x04\x9b\x32\xc9"
			     "\x3d\xd4\x6b\x02\x76\x0d\xa4\x18"
			     "\xaf\x46\xdd\x51\xe8\x7f\x16\x8a"
			     "\x21\xb8\x2c\xc3\x5a\xf1\x65\xfc"
			     "\x93\x07\x9e\x35\xcc\x40\xd7\x6e"
			     "\x05\x79\x10\xa7\x1b\xb2\x49\xe0"
			     "\x54\xeb\x82\x19\x8d\x24\xbb\x2f"
			     "\xc6\x5d\xf4\x68\xff\x96\x0a\xa1"
			     "\x38\xcf\x43\xda\x71\x08\x7c\x13"
			     "\xaa\x1e\xb5\x4c\xe3\x57\xee\x85"
			     "\x1c\x90\x27\xbe\x32\xc9\x60\xf7"
			     "\x6b\x02\x99\x0d\xa4\x3b\xd2\x46"
			     "\xdd\x74\x0b\x7f\x16\xad\x21\xb8"
			     "\x4f\xe6\x5a\xf1\x88\x1f\x93\x2a"
			     "\xc1\x35\xcc\x63\xfa\x6e\x05\x9c"
			     "\x10\xa7\x3e\xd5\x49\xe0\x77\x0e"
			     "\x82\x19\xb0\x24\xbb\x52\xe9\x5d"
			     "\xf4\x8b\x22\x96\x2d\xc4\x38\xcf"
			     "\x66\xfd\x71\x08\x9f\x13\xaa\x41"
			     "\xd8\x4c\xe3\x7a\x11\x85\x1c\xb3"
			     "\x27\xbe\x55\xec\x60\xf7\x8e\x02"
			     "\x99\x30\xc7\x3b\xd2\x69\x00\x74"
			     "\x0b\xa2\x16\xad\x44\xdb\x4f\xe6"
			     "\x7d\x14\x88\x1f\xb6\x2a\xc1\x58"
			     "\xef\x63\xfa\x91\x05\x9c\x33\xca"
			     "\x3e\xd5\x6c\x03\x77\x0e\xa5\x19"
			     "\xb0\x47\xde\x52\xe9\x80\x17\x8b"
			     "\x22\xb9\x2d\xc4\x5b\xf2\x66\xfd"
			     "\x94\x08\x9f\x36\xcd\x41\xd8\x6f"
			     "\x06\x7a\x11\xa8\x1c\xb3\x4a\xe1"
			     "\x55\xec\x83\x1a\x8e\x25\xbc\x30"
			     "\xc7\x5e\xf5\x69\x00\x97\x0b\xa2"
			     "\x39\xd0\x44\xdb\x72\x09\x7d\x14"
			     "\xab\x1f\xb6\x4d\xe4\x58\xef\x86"
			     "\x1d\x91\x28\xbf\x33\xca\x61\xf8"
			     "\x6c\x03\x9a\x0e\xa5\x3c\xd3\x47"
			     "\xde\x75\x0c\x80\x17\xae\x22\xb9"
			     "\x50\xe7\x5b\xf2\x89\x20\x94\x2b"
			     "\xc2\x36\xcd\x64\xfb\x6f\x06\x9d"
			     "\x11\xa8\x3f\xd6\x4a\xe1\x78\x0f"
			     "\x83\x1a\xb1\x25\xbc\x53\xea\x5e"
			     "\xf5\x8c\x00\x97\x2e\xc5\x39\xd0"
			     "\x67\xfe\x72\x09\xa0\x14\xab\x42"
			     "\xd9\x4d\xe4\x7b\x12\x86\x1d\xb4"
			     "\x28\xbf\x56\xed\x61\xf8\x8f\x03"
			     "\x9a\x31\xc8\x3c\xd3\x6a\x01\x75"
			     "\x0c\xa3\x17\xae\x45\xdc\x50\xe7"
			     "\x7e\x15\x89\x20\xb7\x2b\xc2\x59"
			     "\xf0\x64\xfb\x92\x06\x9d\x34\xcb"
			     "\x3f\xd6\x6d\x04\x78\x0f\xa6\x1a"
			     "\xb1\x48\xdf\x53\xea\x81\x18\x8c"
			     "\x23\xba\x2e\xc5\x5c\xf3\x67\xfe"
			     "\x95\x09\xa0\x37\xce\x42\xd9\x70"
			     "\x07\x7b\x12\xa9\x1d\xb4\x4b\xe2"
			     "\x56\xed\x84\x1b\x8f\x26\xbd\x31"
			     "\xc8\x5f\xf6\x6a\x01\x98\x0c\xa3"
			     "\x3a\xd1\x45\xdc\x73\x0a\x7e\x15"
			     "\xac\x20\xb7\x4e\xe5\x59\xf0\x87"
			     "\x1e\x92\x29\xc0\x34\xcb\x62\xf9"
			     "\x6d\x04\x9b\x0f\xa6\x3d\xd4\x48"
			     "\xdf\x76\x0d\x81\x18\xaf\x23\xba"
			     "\x51\xe8\x5c\xf3\x8a\x21\x95\x2c"
			     "\xc3\x37\xce\x65\xfc\x70\x07\x9e"
			     "\x12\xa9\x40\xd7\x4b\xe2\x79\x10"
			     "\x84\x1b\xb2\x26\xbd\x54\xeb\x5f"
			     "\xf6\x8d\x01\x98\x2f\xc6\x3a\xd1"
			     "\x68\xff\x73\x0a\xa1\x15\xac\x43"
			     "\xda\x4e\xe5\x7c\x13\x87\x1e\xb5"
			     "\x29\xc0\x57\xee\x62\xf9\x90\x04"
			     "\x9b\x32\xc9\x3d\xd4\x6b\x02\x76"
			     "\x0d\xa4\x18\xaf\x46\xdd\x51\xe8"
			     "\x7f\x16\x8a\x21\xb8\x2c\xc3\x5a"
			     "\xf1\x65\xfc\x93\x07\x9e\x35\xcc"
			     "\x40\xd7\x6e\x05\x79\x10\xa7\x1b"
			     "\xb2\x49\xe0\x54\xeb\x82\x19\x8d"
			     "\x24\xbb\x2f\xc6\x5d\xf4\x68\xff"
			     "\x96\x0a\xa1\x38\xcf\x43\xda\x71"
			     "\x08\x7c\x13\xaa\x1e\xb5\x4c",
		.psize     = 1023,
		.digest    = "\x98\x43\x07\x63\x75\xe0\xa7\x1c"
			     "\x78\xb1\x8b\xfd\x04\xf5\x2d\x91"
			     "\x20\x48\xa4\x28\xff\x55\xb1\xd3"
			     "\xe6\xf9\x4f\xcc",
	}
};

/*
 * SHA256 test vectors from from NIST
 */
#define SHA256_TEST_VECTORS	5

static struct hash_testvec sha256_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= "\xe3\xb0\xc4\x42\x98\xfc\x1c\x14"
			  "\x9a\xfb\xf4\xc8\x99\x6f\xb9\x24"
			  "\x27\xae\x41\xe4\x64\x9b\x93\x4c"
			  "\xa4\x95\x99\x1b\x78\x52\xb8\x55",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xba\x78\x16\xbf\x8f\x01\xcf\xea"
			  "\x41\x41\x40\xde\x5d\xae\x22\x23"
			  "\xb0\x03\x61\xa3\x96\x17\x7a\x9c"
			  "\xb4\x10\xff\x61\xf2\x00\x15\xad",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x24\x8d\x6a\x61\xd2\x06\x38\xb8"
			  "\xe5\xc0\x26\x93\x0c\x3e\x60\x39"
			  "\xa3\x3c\xe4\x59\x64\xff\x21\x67"
			  "\xf6\xec\xed\xd4\x19\xdb\x06\xc1",
		.np	= 2,
		.tap	= { 28, 28 }
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = "\xb5\xfe\xad\x56\x7d\xff\xcb\xa4"
			  "\x2c\x32\x29\x32\x19\xbb\xfb\xfa"
			  "\xd6\xff\x94\xa3\x72\x91\x85\x66"
			  "\x3b\xa7\x87\x77\x58\xa3\x40\x3a",
	}, {
		.plaintext = "\x08\x9f\x13\xaa\x41\xd8\x4c\xe3"
			     "\x7a\x11\x85\x1c\xb3\x27\xbe\x55"
			     "\xec\x60\xf7\x8e\x02\x99\x30\xc7"
			     "\x3b\xd2\x69\x00\x74\x0b\xa2\x16"
			     "\xad\x44\xdb\x4f\xe6\x7d\x14\x88"
			     "\x1f\xb6\x2a\xc1\x58\xef\x63\xfa"
			     "\x91\x05\x9c\x33\xca\x3e\xd5\x6c"
			     "\x03\x77\x0e\xa5\x19\xb0\x47\xde"
			     "\x52\xe9\x80\x17\x8b\x22\xb9\x2d"
			     "\xc4\x5b\xf2\x66\xfd\x94\x08\x9f"
			     "\x36\xcd\x41\xd8\x6f\x06\x7a\x11"
			     "\xa8\x1c\xb3\x4a\xe1\x55\xec\x83"
			     "\x1a\x8e\x25\xbc\x30\xc7\x5e\xf5"
			     "\x69\x00\x97\x0b\xa2\x39\xd0\x44"
			     "\xdb\x72\x09\x7d\x14\xab\x1f\xb6"
			     "\x4d\xe4\x58\xef\x86\x1d\x91\x28"
			     "\xbf\x33\xca\x61\xf8\x6c\x03\x9a"
			     "\x0e\xa5\x3c\xd3\x47\xde\x75\x0c"
			     "\x80\x17\xae\x22\xb9\x50\xe7\x5b"
			     "\xf2\x89\x20\x94\x2b\xc2\x36\xcd"
			     "\x64\xfb\x6f\x06\x9d\x11\xa8\x3f"
			     "\xd6\x4a\xe1\x78\x0f\x83\x1a\xb1"
			     "\x25\xbc\x53\xea\x5e\xf5\x8c\x00"
			     "\x97\x2e\xc5\x39\xd0\x67\xfe\x72"
			     "\x09\xa0\x14\xab\x42\xd9\x4d\xe4"
			     "\x7b\x12\x86\x1d\xb4\x28\xbf\x56"
			     "\xed\x61\xf8\x8f\x03\x9a\x31\xc8"
			     "\x3c\xd3\x6a\x01\x75\x0c\xa3\x17"
			     "\xae\x45\xdc\x50\xe7\x7e\x15\x89"
			     "\x20\xb7\x2b\xc2\x59\xf0\x64\xfb"
			     "\x92\x06\x9d\x34\xcb\x3f\xd6\x6d"
			     "\x04\x78\x0f\xa6\x1a\xb1\x48\xdf"
			     "\x53\xea\x81\x18\x8c\x23\xba\x2e"
			     "\xc5\x5c\xf3\x67\xfe\x95\x09\xa0"
			     "\x37\xce\x42\xd9\x70\x07\x7b\x12"
			     "\xa9\x1d\xb4\x4b\xe2\x56\xed\x84"
			     "\x1b\x8f\x26\xbd\x31\xc8\x5f\xf6"
			     "\x6a\x01\x98\x0c\xa3\x3a\xd1\x45"
			     "\xdc\x73\x0a\x7e\x15\xac\x20\xb7"
			     "\x4e\xe5\x59\xf0\x87\x1e\x92\x29"
			     "\xc0\x34\xcb\x62\xf9\x6d\x04\x9b"
			     "\x0f\xa6\x3d\xd4\x48\xdf\x76\x0d"
			     "\x81\x18\xaf\x23\xba\x51\xe8\x5c"
			     "\xf3\x8a\x21\x95\x2c\xc3\x37\xce"
			     "\x65\xfc\x70\x07\x9e\x12\xa9\x40"
			     "\xd7\x4b\xe2\x79\x10\x84\x1b\xb2"
			     "\x26\xbd\x54\xeb\x5f\xf6\x8d\x01"
			     "\x98\x2f\xc6\x3a\xd1\x68\xff\x73"
			     "\x0a\xa1\x15\xac\x43\xda\x4e\xe5"
			     "\x7c\x13\x87\x1e\xb5\x29\xc0\x57"
			     "\xee\x62\xf9\x90\x04\x9b\x32\xc9"
			     "\x3d\xd4\x6b\x02\x76\x0d\xa4\x18"
			     "\xaf\x46\xdd\x51\xe8\x7f\x16\x8a"
			     "\x21\xb8\x2c\xc3\x5a\xf1\x65\xfc"
			     "\x93\x07\x9e\x35\xcc\x40\xd7\x6e"
			     "\x05\x79\x10\xa7\x1b\xb2\x49\xe0"
			     "\x54\xeb\x82\x19\x8d\x24\xbb\x2f"
			     "\xc6\x5d\xf4\x68\xff\x96\x0a\xa1"
			     "\x38\xcf\x43\xda\x71\x08\x7c\x13"
			     "\xaa\x1e\xb5\x4c\xe3\x57\xee\x85"
			     "\x1c\x90\x27\xbe\x32\xc9\x60\xf7"
			     "\x6b\x02\x99\x0d\xa4\x3b\xd2\x46"
			     "\xdd\x74\x0b\x7f\x16\xad\x21\xb8"
			     "\x4f\xe6\x5a\xf1\x88\x1f\x93\x2a"
			     "\xc1\x35\xcc\x63\xfa\x6e\x05\x9c"
			     "\x10\xa7\x3e\xd5\x49\xe0\x77\x0e"
			     "\x82\x19\xb0\x24\xbb\x52\xe9\x5d"
			     "\xf4\x8b\x22\x96\x2d\xc4\x38\xcf"
			     "\x66\xfd\x71\x08\x9f\x13\xaa\x41"
			     "\xd8\x4c\xe3\x7a\x11\x85\x1c\xb3"
			     "\x27\xbe\x55\xec\x60\xf7\x8e\x02"
			     "\x99\x30\xc7\x3b\xd2\x69\x00\x74"
			     "\x0b\xa2\x16\xad\x44\xdb\x4f\xe6"
			     "\x7d\x14\x88\x1f\xb6\x2a\xc1\x58"
			     "\xef\x63\xfa\x91\x05\x9c\x33\xca"
			     "\x3e\xd5\x6c\x03\x77\x0e\xa5\x19"
			     "\xb0\x47\xde\x52\xe9\x80\x17\x8b"
			     "\x22\xb9\x2d\xc4\x5b\xf2\x66\xfd"
			     "\x94\x08\x9f\x36\xcd\x41\xd8\x6f"
			     "\x06\x7a\x11\xa8\x1c\xb3\x4a\xe1"
			     "\x55\xec\x83\x1a\x8e\x25\xbc\x30"
			     "\xc7\x5e\xf5\x69\x00\x97\x0b\xa2"
			     "\x39\xd0\x44\xdb\x72\x09\x7d\x14"
			     "\xab\x1f\xb6\x4d\xe4\x58\xef\x86"
			     "\x1d\x91\x28\xbf\x33\xca\x61\xf8"
			     "\x6c\x03\x9a\x0e\xa5\x3c\xd3\x47"
			     "\xde\x75\x0c\x80\x17\xae\x22\xb9"
			     "\x50\xe7\x5b\xf2\x89\x20\x94\x2b"
			     "\xc2\x36\xcd\x64\xfb\x6f\x06\x9d"
			     "\x11\xa8\x3f\xd6\x4a\xe1\x78\x0f"
			     "\x83\x1a\xb1\x25\xbc\x53\xea\x5e"
			     "\xf5\x8c\x00\x97\x2e\xc5\x39\xd0"
			     "\x67\xfe\x72\x09\xa0\x14\xab\x42"
			     "\xd9\x4d\xe4\x7b\x12\x86\x1d\xb4"
			     "\x28\xbf\x56\xed\x61\xf8\x8f\x03"
			     "\x9a\x31\xc8\x3c\xd3\x6a\x01\x75"
			     "\x0c\xa3\x17\xae\x45\xdc\x50\xe7"
			     "\x7e\x15\x89\x20\xb7\x2b\xc2\x59"
			     "\xf0\x64\xfb\x92\x06\x9d\x34\xcb"
			     "\x3f\xd6\x6d\x04\x78\x0f\xa6\x1a"
			     "\xb1\x48\xdf\x53\xea\x81\x18\x8c"
			     "\x23\xba\x2e\xc5\x5c\xf3\x67\xfe"
			     "\x95\x09\xa0\x37\xce\x42\xd9\x70"
			     "\x07\x7b\x12\xa9\x1d\xb4\x4b\xe2"
			     "\x56\xed\x84\x1b\x8f\x26\xbd\x31"
			     "\xc8\x5f\xf6\x6a\x01\x98\x0c\xa3"
			     "\x3a\xd1\x45\xdc\x73\x0a\x7e\x15"
			     "\xac\x20\xb7\x4e\xe5\x59\xf0\x87"
			     "\x1e\x92\x29\xc0\x34\xcb\x62\xf9"
			     "\x6d\x04\x9b\x0f\xa6\x3d\xd4\x48"
			     "\xdf\x76\x0d\x81\x18\xaf\x23\xba"
			     "\x51\xe8\x5c\xf3\x8a\x21\x95\x2c"
			     "\xc3\x37\xce\x65\xfc\x70\x07\x9e"
			     "\x12\xa9\x40\xd7\x4b\xe2\x79\x10"
			     "\x84\x1b\xb2\x26\xbd\x54\xeb\x5f"
			     "\xf6\x8d\x01\x98\x2f\xc6\x3a\xd1"
			     "\x68\xff\x73\x0a\xa1\x15\xac\x43"
			     "\xda\x4e\xe5\x7c\x13\x87\x1e\xb5"
			     "\x29\xc0\x57\xee\x62\xf9\x90\x04"
			     "\x9b\x32\xc9\x3d\xd4\x6b\x02\x76"
			     "\x0d\xa4\x18\xaf\x46\xdd\x51\xe8"
			     "\x7f\x16\x8a\x21\xb8\x2c\xc3\x5a"
			     "\xf1\x65\xfc\x93\x07\x9e\x35\xcc"
			     "\x40\xd7\x6e\x05\x79\x10\xa7\x1b"
			     "\xb2\x49\xe0\x54\xeb\x82\x19\x8d"
			     "\x24\xbb\x2f\xc6\x5d\xf4\x68\xff"
			     "\x96\x0a\xa1\x38\xcf\x43\xda\x71"
			     "\x08\x7c\x13\xaa\x1e\xb5\x4c",
		.psize     = 1023,
		.digest    = "\xc5\xce\x0c\xca\x01\x4f\x53\x3a"
			     "\x32\x32\x17\xcc\xd4\x6a\x71\xa9"
			     "\xf3\xed\x50\x10\x64\x8e\x06\xbe"
			     "\x9b\x4a\xa6\xbb\x05\x89\x59\x51",
	}
};

/*
 * SHA384 test vectors from from NIST and kerneli
 */
#define SHA384_TEST_VECTORS	6

static struct hash_testvec sha384_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= "\x38\xb0\x60\xa7\x51\xac\x96\x38"
			  "\x4c\xd9\x32\x7e\xb1\xb1\xe3\x6a"
			  "\x21\xfd\xb7\x11\x14\xbe\x07\x43"
			  "\x4c\x0c\xc7\xbf\x63\xf6\xe1\xda"
			  "\x27\x4e\xde\xbf\xe7\x6f\x65\xfb"
			  "\xd5\x1a\xd2\xf1\x48\x98\xb9\x5b",
	}, {
		.plaintext= "abc",
		.psize	= 3,
		.digest	= "\xcb\x00\x75\x3f\x45\xa3\x5e\x8b"
			  "\xb5\xa0\x3d\x69\x9a\xc6\x50\x07"
			  "\x27\x2c\x32\xab\x0e\xde\xd1\x63"
			  "\x1a\x8b\x60\x5a\x43\xff\x5b\xed"
			  "\x80\x86\x07\x2b\xa1\xe7\xcc\x23"
			  "\x58\xba\xec\xa1\x34\xc8\x25\xa7",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x33\x91\xfd\xdd\xfc\x8d\xc7\x39"
			  "\x37\x07\xa6\x5b\x1b\x47\x09\x39"
			  "\x7c\xf8\xb1\xd1\x62\xaf\x05\xab"
			  "\xfe\x8f\x45\x0d\xe5\xf3\x6b\xc6"
			  "\xb0\x45\x5a\x85\x20\xbc\x4e\x6f"
			  "\x5f\xe9\x5b\x1f\xe3\xc8\x45\x2b",
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
			   "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= "\x09\x33\x0c\x33\xf7\x11\x47\xe8"
			  "\x3d\x19\x2f\xc7\x82\xcd\x1b\x47"
			  "\x53\x11\x1b\x17\x3b\x3b\x05\xd2"
			  "\x2f\xa0\x80\x86\xe3\xb0\xf7\x12"
			  "\xfc\xc7\xc7\x1a\x55\x7e\x2d\xb9"
			  "\x66\xc3\xe9\xfa\x91\x74\x60\x39",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcd"
			   "efghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz",
		.psize	= 104,
		.digest	= "\x3d\x20\x89\x73\xab\x35\x08\xdb"
			  "\xbd\x7e\x2c\x28\x62\xba\x29\x0a"
			  "\xd3\x01\x0e\x49\x78\xc1\x98\xdc"
			  "\x4d\x8f\xd0\x14\xe5\x82\x82\x3a"
			  "\x89\xe1\x6f\x9b\x2a\x7b\xbc\x1a"
			  "\xc9\x38\xe2\xd1\x99\xe8\xbe\xa4",
		.np	= 4,
		.tap	= { 26, 26, 26, 26 }
	}, {
		.plaintext = "\x08\x9f\x13\xaa\x41\xd8\x4c\xe3"
			     "\x7a\x11\x85\x1c\xb3\x27\xbe\x55"
			     "\xec\x60\xf7\x8e\x02\x99\x30\xc7"
			     "\x3b\xd2\x69\x00\x74\x0b\xa2\x16"
			     "\xad\x44\xdb\x4f\xe6\x7d\x14\x88"
			     "\x1f\xb6\x2a\xc1\x58\xef\x63\xfa"
			     "\x91\x05\x9c\x33\xca\x3e\xd5\x6c"
			     "\x03\x77\x0e\xa5\x19\xb0\x47\xde"
			     "\x52\xe9\x80\x17\x8b\x22\xb9\x2d"
			     "\xc4\x5b\xf2\x66\xfd\x94\x08\x9f"
			     "\x36\xcd\x41\xd8\x6f\x06\x7a\x11"
			     "\xa8\x1c\xb3\x4a\xe1\x55\xec\x83"
			     "\x1a\x8e\x25\xbc\x30\xc7\x5e\xf5"
			     "\x69\x00\x97\x0b\xa2\x39\xd0\x44"
			     "\xdb\x72\x09\x7d\x14\xab\x1f\xb6"
			     "\x4d\xe4\x58\xef\x86\x1d\x91\x28"
			     "\xbf\x33\xca\x61\xf8\x6c\x03\x9a"
			     "\x0e\xa5\x3c\xd3\x47\xde\x75\x0c"
			     "\x80\x17\xae\x22\xb9\x50\xe7\x5b"
			     "\xf2\x89\x20\x94\x2b\xc2\x36\xcd"
			     "\x64\xfb\x6f\x06\x9d\x11\xa8\x3f"
			     "\xd6\x4a\xe1\x78\x0f\x83\x1a\xb1"
			     "\x25\xbc\x53\xea\x5e\xf5\x8c\x00"
			     "\x97\x2e\xc5\x39\xd0\x67\xfe\x72"
			     "\x09\xa0\x14\xab\x42\xd9\x4d\xe4"
			     "\x7b\x12\x86\x1d\xb4\x28\xbf\x56"
			     "\xed\x61\xf8\x8f\x03\x9a\x31\xc8"
			     "\x3c\xd3\x6a\x01\x75\x0c\xa3\x17"
			     "\xae\x45\xdc\x50\xe7\x7e\x15\x89"
			     "\x20\xb7\x2b\xc2\x59\xf0\x64\xfb"
			     "\x92\x06\x9d\x34\xcb\x3f\xd6\x6d"
			     "\x04\x78\x0f\xa6\x1a\xb1\x48\xdf"
			     "\x53\xea\x81\x18\x8c\x23\xba\x2e"
			     "\xc5\x5c\xf3\x67\xfe\x95\x09\xa0"
			     "\x37\xce\x42\xd9\x70\x07\x7b\x12"
			     "\xa9\x1d\xb4\x4b\xe2\x56\xed\x84"
			     "\x1b\x8f\x26\xbd\x31\xc8\x5f\xf6"
			     "\x6a\x01\x98\x0c\xa3\x3a\xd1\x45"
			     "\xdc\x73\x0a\x7e\x15\xac\x20\xb7"
			     "\x4e\xe5\x59\xf0\x87\x1e\x92\x29"
			     "\xc0\x34\xcb\x62\xf9\x6d\x04\x9b"
			     "\x0f\xa6\x3d\xd4\x48\xdf\x76\x0d"
			     "\x81\x18\xaf\x23\xba\x51\xe8\x5c"
			     "\xf3\x8a\x21\x95\x2c\xc3\x37\xce"
			     "\x65\xfc\x70\x07\x9e\x12\xa9\x40"
			     "\xd7\x4b\xe2\x79\x10\x84\x1b\xb2"
			     "\x26\xbd\x54\xeb\x5f\xf6\x8d\x01"
			     "\x98\x2f\xc6\x3a\xd1\x68\xff\x73"
			     "\x0a\xa1\x15\xac\x43\xda\x4e\xe5"
			     "\x7c\x13\x87\x1e\xb5\x29\xc0\x57"
			     "\xee\x62\xf9\x90\x04\x9b\x32\xc9"
			     "\x3d\xd4\x6b\x02\x76\x0d\xa4\x18"
			     "\xaf\x46\xdd\x51\xe8\x7f\x16\x8a"
			     "\x21\xb8\x2c\xc3\x5a\xf1\x65\xfc"
			     "\x93\x07\x9e\x35\xcc\x40\xd7\x6e"
			     "\x05\x79\x10\xa7\x1b\xb2\x49\xe0"
			     "\x54\xeb\x82\x19\x8d\x24\xbb\x2f"
			     "\xc6\x5d\xf4\x68\xff\x96\x0a\xa1"
			     "\x38\xcf\x43\xda\x71\x08\x7c\x13"
			     "\xaa\x1e\xb5\x4c\xe3\x57\xee\x85"
			     "\x1c\x90\x27\xbe\x32\xc9\x60\xf7"
			     "\x6b\x02\x99\x0d\xa4\x3b\xd2\x46"
			     "\xdd\x74\x0b\x7f\x16\xad\x21\xb8"
			     "\x4f\xe6\x5a\xf1\x88\x1f\x93\x2a"
			     "\xc1\x35\xcc\x63\xfa\x6e\x05\x9c"
			     "\x10\xa7\x3e\xd5\x49\xe0\x77\x0e"
			     "\x82\x19\xb0\x24\xbb\x52\xe9\x5d"
			     "\xf4\x8b\x22\x96\x2d\xc4\x38\xcf"
			     "\x66\xfd\x71\x08\x9f\x13\xaa\x41"
			     "\xd8\x4c\xe3\x7a\x11\x85\x1c\xb3"
			     "\x27\xbe\x55\xec\x60\xf7\x8e\x02"
			     "\x99\x30\xc7\x3b\xd2\x69\x00\x74"
			     "\x0b\xa2\x16\xad\x44\xdb\x4f\xe6"
			     "\x7d\x14\x88\x1f\xb6\x2a\xc1\x58"
			     "\xef\x63\xfa\x91\x05\x9c\x33\xca"
			     "\x3e\xd5\x6c\x03\x77\x0e\xa5\x19"
			     "\xb0\x47\xde\x52\xe9\x80\x17\x8b"
			     "\x22\xb9\x2d\xc4\x5b\xf2\x66\xfd"
			     "\x94\x08\x9f\x36\xcd\x41\xd8\x6f"
			     "\x06\x7a\x11\xa8\x1c\xb3\x4a\xe1"
			     "\x55\xec\x83\x1a\x8e\x25\xbc\x30"
			     "\xc7\x5e\xf5\x69\x00\x97\x0b\xa2"
			     "\x39\xd0\x44\xdb\x72\x09\x7d\x14"
			     "\xab\x1f\xb6\x4d\xe4\x58\xef\x86"
			     "\x1d\x91\x28\xbf\x33\xca\x61\xf8"
			     "\x6c\x03\x9a\x0e\xa5\x3c\xd3\x47"
			     "\xde\x75\x0c\x80\x17\xae\x22\xb9"
			     "\x50\xe7\x5b\xf2\x89\x20\x94\x2b"
			     "\xc2\x36\xcd\x64\xfb\x6f\x06\x9d"
			     "\x11\xa8\x3f\xd6\x4a\xe1\x78\x0f"
			     "\x83\x1a\xb1\x25\xbc\x53\xea\x5e"
			     "\xf5\x8c\x00\x97\x2e\xc5\x39\xd0"
			     "\x67\xfe\x72\x09\xa0\x14\xab\x42"
			     "\xd9\x4d\xe4\x7b\x12\x86\x1d\xb4"
			     "\x28\xbf\x56\xed\x61\xf8\x8f\x03"
			     "\x9a\x31\xc8\x3c\xd3\x6a\x01\x75"
			     "\x0c\xa3\x17\xae\x45\xdc\x50\xe7"
			     "\x7e\x15\x89\x20\xb7\x2b\xc2\x59"
			     "\xf0\x64\xfb\x92\x06\x9d\x34\xcb"
			     "\x3f\xd6\x6d\x04\x78\x0f\xa6\x1a"
			     "\xb1\x48\xdf\x53\xea\x81\x18\x8c"
			     "\x23\xba\x2e\xc5\x5c\xf3\x67\xfe"
			     "\x95\x09\xa0\x37\xce\x42\xd9\x70"
			     "\x07\x7b\x12\xa9\x1d\xb4\x4b\xe2"
			     "\x56\xed\x84\x1b\x8f\x26\xbd\x31"
			     "\xc8\x5f\xf6\x6a\x01\x98\x0c\xa3"
			     "\x3a\xd1\x45\xdc\x73\x0a\x7e\x15"
			     "\xac\x20\xb7\x4e\xe5\x59\xf0\x87"
			     "\x1e\x92\x29\xc0\x34\xcb\x62\xf9"
			     "\x6d\x04\x9b\x0f\xa6\x3d\xd4\x48"
			     "\xdf\x76\x0d\x81\x18\xaf\x23\xba"
			     "\x51\xe8\x5c\xf3\x8a\x21\x95\x2c"
			     "\xc3\x37\xce\x65\xfc\x70\x07\x9e"
			     "\x12\xa9\x40\xd7\x4b\xe2\x79\x10"
			     "\x84\x1b\xb2\x26\xbd\x54\xeb\x5f"
			     "\xf6\x8d\x01\x98\x2f\xc6\x3a\xd1"
			     "\x68\xff\x73\x0a\xa1\x15\xac\x43"
			     "\xda\x4e\xe5\x7c\x13\x87\x1e\xb5"
			     "\x29\xc0\x57\xee\x62\xf9\x90\x04"
			     "\x9b\x32\xc9\x3d\xd4\x6b\x02\x76"
			     "\x0d\xa4\x18\xaf\x46\xdd\x51\xe8"
			     "\x7f\x16\x8a\x21\xb8\x2c\xc3\x5a"
			     "\xf1\x65\xfc\x93\x07\x9e\x35\xcc"
			     "\x40\xd7\x6e\x05\x79\x10\xa7\x1b"
			     "\xb2\x49\xe0\x54\xeb\x82\x19\x8d"
			     "\x24\xbb\x2f\xc6\x5d\xf4\x68\xff"
			     "\x96\x0a\xa1\x38\xcf\x43\xda\x71"
			     "\x08\x7c\x13\xaa\x1e\xb5\x4c",
		.psize     = 1023,
		.digest    = "\x4d\x97\x23\xc8\xea\x7a\x7c\x15"
			     "\xb8\xff\x97\x9c\xf5\x13\x4f\x31"
			     "\xde\x67\xf7\x24\x73\xcd\x70\x1c"
			     "\x03\x4a\xba\x8a\x87\x49\xfe\xdc"
			     "\x75\x29\x62\x83\xae\x3f\x17\xab"
			     "\xfd\x10\x4d\x8e\x17\x1c\x1f\xca",
	}
};

/*
 * SHA512 test vectors from from NIST and kerneli
 */
#define SHA512_TEST_VECTORS	6

static struct hash_testvec sha512_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= "\xcf\x83\xe1\x35\x7e\xef\xb8\xbd"
			  "\xf1\x54\x28\x50\xd6\x6d\x80\x07"
			  "\xd6\x20\xe4\x05\x0b\x57\x15\xdc"
			  "\x83\xf4\xa9\x21\xd3\x6c\xe9\xce"
			  "\x47\xd0\xd1\x3c\x5d\x85\xf2\xb0"
			  "\xff\x83\x18\xd2\x87\x7e\xec\x2f"
			  "\x63\xb9\x31\xbd\x47\x41\x7a\x81"
			  "\xa5\x38\x32\x7a\xf9\x27\xda\x3e",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xdd\xaf\x35\xa1\x93\x61\x7a\xba"
			  "\xcc\x41\x73\x49\xae\x20\x41\x31"
			  "\x12\xe6\xfa\x4e\x89\xa9\x7e\xa2"
			  "\x0a\x9e\xee\xe6\x4b\x55\xd3\x9a"
			  "\x21\x92\x99\x2a\x27\x4f\xc1\xa8"
			  "\x36\xba\x3c\x23\xa3\xfe\xeb\xbd"
			  "\x45\x4d\x44\x23\x64\x3c\xe8\x0e"
			  "\x2a\x9a\xc9\x4f\xa5\x4c\xa4\x9f",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x20\x4a\x8f\xc6\xdd\xa8\x2f\x0a"
			  "\x0c\xed\x7b\xeb\x8e\x08\xa4\x16"
			  "\x57\xc1\x6e\xf4\x68\xb2\x28\xa8"
			  "\x27\x9b\xe3\x31\xa7\x03\xc3\x35"
			  "\x96\xfd\x15\xc1\x3b\x1b\x07\xf9"
			  "\xaa\x1d\x3b\xea\x57\x78\x9c\xa0"
			  "\x31\xad\x85\xc7\xa7\x1d\xd7\x03"
			  "\x54\xec\x63\x12\x38\xca\x34\x45",
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
			   "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= "\x8e\x95\x9b\x75\xda\xe3\x13\xda"
			  "\x8c\xf4\xf7\x28\x14\xfc\x14\x3f"
			  "\x8f\x77\x79\xc6\xeb\x9f\x7f\xa1"
			  "\x72\x99\xae\xad\xb6\x88\x90\x18"
			  "\x50\x1d\x28\x9e\x49\x00\xf7\xe4"
			  "\x33\x1b\x99\xde\xc4\xb5\x43\x3a"
			  "\xc7\xd3\x29\xee\xb6\xdd\x26\x54"
			  "\x5e\x96\xe5\x5b\x87\x4b\xe9\x09",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcd"
			   "efghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz",
		.psize	= 104,
		.digest	= "\x93\x0d\x0c\xef\xcb\x30\xff\x11"
			  "\x33\xb6\x89\x81\x21\xf1\xcf\x3d"
			  "\x27\x57\x8a\xfc\xaf\xe8\x67\x7c"
			  "\x52\x57\xcf\x06\x99\x11\xf7\x5d"
			  "\x8f\x58\x31\xb5\x6e\xbf\xda\x67"
			  "\xb2\x78\xe6\x6d\xff\x8b\x84\xfe"
			  "\x2b\x28\x70\xf7\x42\xa5\x80\xd8"
			  "\xed\xb4\x19\x87\x23\x28\x50\xc9",
		.np	= 4,
		.tap	= { 26, 26, 26, 26 }
	}, {
		.plaintext = "\x08\x9f\x13\xaa\x41\xd8\x4c\xe3"
			     "\x7a\x11\x85\x1c\xb3\x27\xbe\x55"
			     "\xec\x60\xf7\x8e\x02\x99\x30\xc7"
			     "\x3b\xd2\x69\x00\x74\x0b\xa2\x16"
			     "\xad\x44\xdb\x4f\xe6\x7d\x14\x88"
			     "\x1f\xb6\x2a\xc1\x58\xef\x63\xfa"
			     "\x91\x05\x9c\x33\xca\x3e\xd5\x6c"
			     "\x03\x77\x0e\xa5\x19\xb0\x47\xde"
			     "\x52\xe9\x80\x17\x8b\x22\xb9\x2d"
			     "\xc4\x5b\xf2\x66\xfd\x94\x08\x9f"
			     "\x36\xcd\x41\xd8\x6f\x06\x7a\x11"
			     "\xa8\x1c\xb3\x4a\xe1\x55\xec\x83"
			     "\x1a\x8e\x25\xbc\x30\xc7\x5e\xf5"
			     "\x69\x00\x97\x0b\xa2\x39\xd0\x44"
			     "\xdb\x72\x09\x7d\x14\xab\x1f\xb6"
			     "\x4d\xe4\x58\xef\x86\x1d\x91\x28"
			     "\xbf\x33\xca\x61\xf8\x6c\x03\x9a"
			     "\x0e\xa5\x3c\xd3\x47\xde\x75\x0c"
			     "\x80\x17\xae\x22\xb9\x50\xe7\x5b"
			     "\xf2\x89\x20\x94\x2b\xc2\x36\xcd"
			     "\x64\xfb\x6f\x06\x9d\x11\xa8\x3f"
			     "\xd6\x4a\xe1\x78\x0f\x83\x1a\xb1"
			     "\x25\xbc\x53\xea\x5e\xf5\x8c\x00"
			     "\x97\x2e\xc5\x39\xd0\x67\xfe\x72"
			     "\x09\xa0\x14\xab\x42\xd9\x4d\xe4"
			     "\x7b\x12\x86\x1d\xb4\x28\xbf\x56"
			     "\xed\x61\xf8\x8f\x03\x9a\x31\xc8"
			     "\x3c\xd3\x6a\x01\x75\x0c\xa3\x17"
			     "\xae\x45\xdc\x50\xe7\x7e\x15\x89"
			     "\x20\xb7\x2b\xc2\x59\xf0\x64\xfb"
			     "\x92\x06\x9d\x34\xcb\x3f\xd6\x6d"
			     "\x04\x78\x0f\xa6\x1a\xb1\x48\xdf"
			     "\x53\xea\x81\x18\x8c\x23\xba\x2e"
			     "\xc5\x5c\xf3\x67\xfe\x95\x09\xa0"
			     "\x37\xce\x42\xd9\x70\x07\x7b\x12"
			     "\xa9\x1d\xb4\x4b\xe2\x56\xed\x84"
			     "\x1b\x8f\x26\xbd\x31\xc8\x5f\xf6"
			     "\x6a\x01\x98\x0c\xa3\x3a\xd1\x45"
			     "\xdc\x73\x0a\x7e\x15\xac\x20\xb7"
			     "\x4e\xe5\x59\xf0\x87\x1e\x92\x29"
			     "\xc0\x34\xcb\x62\xf9\x6d\x04\x9b"
			     "\x0f\xa6\x3d\xd4\x48\xdf\x76\x0d"
			     "\x81\x18\xaf\x23\xba\x51\xe8\x5c"
			     "\xf3\x8a\x21\x95\x2c\xc3\x37\xce"
			     "\x65\xfc\x70\x07\x9e\x12\xa9\x40"
			     "\xd7\x4b\xe2\x79\x10\x84\x1b\xb2"
			     "\x26\xbd\x54\xeb\x5f\xf6\x8d\x01"
			     "\x98\x2f\xc6\x3a\xd1\x68\xff\x73"
			     "\x0a\xa1\x15\xac\x43\xda\x4e\xe5"
			     "\x7c\x13\x87\x1e\xb5\x29\xc0\x57"
			     "\xee\x62\xf9\x90\x04\x9b\x32\xc9"
			     "\x3d\xd4\x6b\x02\x76\x0d\xa4\x18"
			     "\xaf\x46\xdd\x51\xe8\x7f\x16\x8a"
			     "\x21\xb8\x2c\xc3\x5a\xf1\x65\xfc"
			     "\x93\x07\x9e\x35\xcc\x40\xd7\x6e"
			     "\x05\x79\x10\xa7\x1b\xb2\x49\xe0"
			     "\x54\xeb\x82\x19\x8d\x24\xbb\x2f"
			     "\xc6\x5d\xf4\x68\xff\x96\x0a\xa1"
			     "\x38\xcf\x43\xda\x71\x08\x7c\x13"
			     "\xaa\x1e\xb5\x4c\xe3\x57\xee\x85"
			     "\x1c\x90\x27\xbe\x32\xc9\x60\xf7"
			     "\x6b\x02\x99\x0d\xa4\x3b\xd2\x46"
			     "\xdd\x74\x0b\x7f\x16\xad\x21\xb8"
			     "\x4f\xe6\x5a\xf1\x88\x1f\x93\x2a"
			     "\xc1\x35\xcc\x63\xfa\x6e\x05\x9c"
			     "\x10\xa7\x3e\xd5\x49\xe0\x77\x0e"
			     "\x82\x19\xb0\x24\xbb\x52\xe9\x5d"
			     "\xf4\x8b\x22\x96\x2d\xc4\x38\xcf"
			     "\x66\xfd\x71\x08\x9f\x13\xaa\x41"
			     "\xd8\x4c\xe3\x7a\x11\x85\x1c\xb3"
			     "\x27\xbe\x55\xec\x60\xf7\x8e\x02"
			     "\x99\x30\xc7\x3b\xd2\x69\x00\x74"
			     "\x0b\xa2\x16\xad\x44\xdb\x4f\xe6"
			     "\x7d\x14\x88\x1f\xb6\x2a\xc1\x58"
			     "\xef\x63\xfa\x91\x05\x9c\x33\xca"
			     "\x3e\xd5\x6c\x03\x77\x0e\xa5\x19"
			     "\xb0\x47\xde\x52\xe9\x80\x17\x8b"
			     "\x22\xb9\x2d\xc4\x5b\xf2\x66\xfd"
			     "\x94\x08\x9f\x36\xcd\x41\xd8\x6f"
			     "\x06\x7a\x11\xa8\x1c\xb3\x4a\xe1"
			     "\x55\xec\x83\x1a\x8e\x25\xbc\x30"
			     "\xc7\x5e\xf5\x69\x00\x97\x0b\xa2"
			     "\x39\xd0\x44\xdb\x72\x09\x7d\x14"
			     "\xab\x1f\xb6\x4d\xe4\x58\xef\x86"
			     "\x1d\x91\x28\xbf\x33\xca\x61\xf8"
			     "\x6c\x03\x9a\x0e\xa5\x3c\xd3\x47"
			     "\xde\x75\x0c\x80\x17\xae\x22\xb9"
			     "\x50\xe7\x5b\xf2\x89\x20\x94\x2b"
			     "\xc2\x36\xcd\x64\xfb\x6f\x06\x9d"
			     "\x11\xa8\x3f\xd6\x4a\xe1\x78\x0f"
			     "\x83\x1a\xb1\x25\xbc\x53\xea\x5e"
			     "\xf5\x8c\x00\x97\x2e\xc5\x39\xd0"
			     "\x67\xfe\x72\x09\xa0\x14\xab\x42"
			     "\xd9\x4d\xe4\x7b\x12\x86\x1d\xb4"
			     "\x28\xbf\x56\xed\x61\xf8\x8f\x03"
			     "\x9a\x31\xc8\x3c\xd3\x6a\x01\x75"
			     "\x0c\xa3\x17\xae\x45\xdc\x50\xe7"
			     "\x7e\x15\x89\x20\xb7\x2b\xc2\x59"
			     "\xf0\x64\xfb\x92\x06\x9d\x34\xcb"
			     "\x3f\xd6\x6d\x04\x78\x0f\xa6\x1a"
			     "\xb1\x48\xdf\x53\xea\x81\x18\x8c"
			     "\x23\xba\x2e\xc5\x5c\xf3\x67\xfe"
			     "\x95\x09\xa0\x37\xce\x42\xd9\x70"
			     "\x07\x7b\x12\xa9\x1d\xb4\x4b\xe2"
			     "\x56\xed\x84\x1b\x8f\x26\xbd\x31"
			     "\xc8\x5f\xf6\x6a\x01\x98\x0c\xa3"
			     "\x3a\xd1\x45\xdc\x73\x0a\x7e\x15"
			     "\xac\x20\xb7\x4e\xe5\x59\xf0\x87"
			     "\x1e\x92\x29\xc0\x34\xcb\x62\xf9"
			     "\x6d\x04\x9b\x0f\xa6\x3d\xd4\x48"
			     "\xdf\x76\x0d\x81\x18\xaf\x23\xba"
			     "\x51\xe8\x5c\xf3\x8a\x21\x95\x2c"
			     "\xc3\x37\xce\x65\xfc\x70\x07\x9e"
			     "\x12\xa9\x40\xd7\x4b\xe2\x79\x10"
			     "\x84\x1b\xb2\x26\xbd\x54\xeb\x5f"
			     "\xf6\x8d\x01\x98\x2f\xc6\x3a\xd1"
			     "\x68\xff\x73\x0a\xa1\x15\xac\x43"
			     "\xda\x4e\xe5\x7c\x13\x87\x1e\xb5"
			     "\x29\xc0\x57\xee\x62\xf9\x90\x04"
			     "\x9b\x32\xc9\x3d\xd4\x6b\x02\x76"
			     "\x0d\xa4\x18\xaf\x46\xdd\x51\xe8"
			     "\x7f\x16\x8a\x21\xb8\x2c\xc3\x5a"
			     "\xf1\x65\xfc\x93\x07\x9e\x35\xcc"
			     "\x40\xd7\x6e\x05\x79\x10\xa7\x1b"
			     "\xb2\x49\xe0\x54\xeb\x82\x19\x8d"
			     "\x24\xbb\x2f\xc6\x5d\xf4\x68\xff"
			     "\x96\x0a\xa1\x38\xcf\x43\xda\x71"
			     "\x08\x7c\x13\xaa\x1e\xb5\x4c",
		.psize     = 1023,
		.digest    = "\x76\xc9\xd4\x91\x7a\x5f\x0f\xaa"
			     "\x13\x39\xf3\x01\x7a\xfa\xe5\x41"
			     "\x5f\x0b\xf8\xeb\x32\xfc\xbf\xb0"
			     "\xfa\x8c\xcd\x17\x83\xe2\xfa\xeb"
			     "\x1c\x19\xde\xe2\x75\xdc\x34\x64"
			     "\x5f\x35\x9c\x61\x2f\x10\xf9\xec"
			     "\x59\xca\x9d\xcc\x25\x0c\x43\xba"
			     "\x85\xa8\xf8\xfe\xb5\x24\xb2\xee",
	}
};


/*
 * HMAC-MD5 test vectors from RFC2202
 * (These need to be fixed to not use strlen).
 */
#define HMAC_MD5_TEST_VECTORS	7

static struct hash_testvec hmac_md5_tv_template[] =
{
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
		.ksize	= 16,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x92\x94\x72\x7a\x36\x38\xbb\x1c"
			  "\x13\xf4\x8e\xf8\x15\x8b\xfc\x9d",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x75\x0c\x78\x3e\x6a\xb0\xb5\x03"
			  "\xea\xa8\x6e\x31\x0a\x5d\xb7\x38",
		.np	= 2,
		.tap	= {14, 14}
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
		.ksize	= 16,
		.plaintext = "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",
		.psize	= 50,
		.digest	= "\x56\xbe\x34\x52\x1d\x14\x4c\x88"
			  "\xdb\xb8\xc7\x33\xf0\xe8\xb3\xf6",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18\x19",
		.ksize	= 25,
		.plaintext = "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd",
		.psize	= 50,
		.digest	= "\x69\x7e\xaf\x0a\xca\x3a\x3a\xea"
			  "\x3a\x75\x16\x47\x46\xff\xaa\x79",
	}, {
		.key	= "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c",
		.ksize	= 16,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= "\x56\x46\x1e\xf2\x34\x2e\xdc\x00"
			  "\xf9\xba\xb9\x95\x69\x0e\xfd\x4c",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= "\x6b\x1a\xb7\xfe\x4b\xd7\xbf\x8f"
			  "\x0b\x62\xe6\xce\x61\xb9\xd0\xcd",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			   "Block-Size Data",
		.psize	= 73,
		.digest	= "\x6f\x63\x0f\xad\x67\xcd\xa0\xee"
			  "\x1f\xb1\xf5\x62\xdb\x3a\xa5\x3e",
	},
};

/*
 * HMAC-SHA1 test vectors from RFC2202
 */
#define HMAC_SHA1_TEST_VECTORS	7

static struct hash_testvec hmac_sha1_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\xb6\x17\x31\x86\x55\x05\x72\x64"
			  "\xe2\x8b\xc0\xb6\xfb\x37\x8c\x8e\xf1"
			  "\x46\xbe",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\xef\xfc\xdf\x6a\xe5\xeb\x2f\xa2\xd2\x74"
			  "\x16\xd5\xf1\x84\xdf\x9c\x25\x9a\x7c\x79",
		.np	= 2,
		.tap	= { 14, 14 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
		.ksize	= 20,
		.plaintext = "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",
		.psize	= 50,
		.digest	= "\x12\x5d\x73\x42\xb9\xac\x11\xcd\x91\xa3"
			  "\x9a\xf4\x8a\xa1\x7b\x4f\x63\xf1\x75\xd3",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18\x19",
		.ksize	= 25,
		.plaintext = "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd",
		.psize	= 50,
		.digest	= "\x4c\x90\x07\xf4\x02\x62\x50\xc6\xbc\x84"
			  "\x14\xf9\xbf\x50\xc8\x6c\x2d\x72\x35\xda",
	}, {
		.key	= "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c",
		.ksize	= 20,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= "\x4c\x1a\x03\x42\x4b\x55\xe0\x7f\xe7\xf2"
			  "\x7b\xe1\xd5\x8b\xb9\x32\x4a\x9a\x5a\x04",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= "\xaa\x4a\xe5\xe1\x52\x72\xd0\x0e\x95\x70"
			  "\x56\x37\xce\x8a\x3b\x55\xed\x40\x21\x12",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			   "Block-Size Data",
		.psize	= 73,
		.digest	= "\xe8\xe9\x9d\x0f\x45\x23\x7d\x78\x6d\x6b"
			  "\xba\xa7\x96\x5c\x78\x08\xbb\xff\x1a\x91",
	},
};


/*
 * SHA224 HMAC test vectors from RFC4231
 */
#define HMAC_SHA224_TEST_VECTORS    4

static struct hash_testvec hmac_sha224_tv_template[] = {
	{
		.key    = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			"\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			"\x0b\x0b\x0b\x0b",
		.ksize  = 20,
		/*  ("Hi There") */
		.plaintext = "\x48\x69\x20\x54\x68\x65\x72\x65",
		.psize  = 8,
		.digest = "\x89\x6f\xb1\x12\x8a\xbb\xdf\x19"
			"\x68\x32\x10\x7c\xd4\x9d\xf3\x3f"
			"\x47\xb4\xb1\x16\x99\x12\xba\x4f"
			"\x53\x68\x4b\x22",
	}, {
		.key    = "Jefe",
		.ksize  = 4,
		/* ("what do ya want for nothing?") */
		.plaintext = "\x77\x68\x61\x74\x20\x64\x6f\x20"
			"\x79\x61\x20\x77\x61\x6e\x74\x20"
			"\x66\x6f\x72\x20\x6e\x6f\x74\x68"
			"\x69\x6e\x67\x3f",
		.psize  = 28,
		.digest = "\xa3\x0e\x01\x09\x8b\xc6\xdb\xbf"
			"\x45\x69\x0f\x3a\x7e\x9e\x6d\x0f"
			"\x8b\xbe\xa2\xa3\x9e\x61\x48\x00"
			"\x8f\xd0\x5e\x44",
		.np = 4,
		.tap    = { 7, 7, 7, 7 }
	}, {
		.key    = "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa",
		.ksize  = 131,
		/* ("Test Using Larger Than Block-Size Key - Hash Key First") */
		.plaintext = "\x54\x65\x73\x74\x20\x55\x73\x69"
			"\x6e\x67\x20\x4c\x61\x72\x67\x65"
			"\x72\x20\x54\x68\x61\x6e\x20\x42"
			"\x6c\x6f\x63\x6b\x2d\x53\x69\x7a"
			"\x65\x20\x4b\x65\x79\x20\x2d\x20"
			"\x48\x61\x73\x68\x20\x4b\x65\x79"
			"\x20\x46\x69\x72\x73\x74",
		.psize  = 54,
		.digest = "\x95\xe9\xa0\xdb\x96\x20\x95\xad"
			"\xae\xbe\x9b\x2d\x6f\x0d\xbc\xe2"
			"\xd4\x99\xf1\x12\xf2\xd2\xb7\x27"
			"\x3f\xa6\x87\x0e",
	}, {
		.key    = "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa",
		.ksize  = 131,
		/* ("This is a test using a larger than block-size key and a")
		(" larger than block-size data. The key needs to be")
			(" hashed before being used by the HMAC algorithm.") */
		.plaintext = "\x54\x68\x69\x73\x20\x69\x73\x20"
			"\x61\x20\x74\x65\x73\x74\x20\x75"
			"\x73\x69\x6e\x67\x20\x61\x20\x6c"
			"\x61\x72\x67\x65\x72\x20\x74\x68"
			"\x61\x6e\x20\x62\x6c\x6f\x63\x6b"
			"\x2d\x73\x69\x7a\x65\x20\x6b\x65"
			"\x79\x20\x61\x6e\x64\x20\x61\x20"
			"\x6c\x61\x72\x67\x65\x72\x20\x74"
			"\x68\x61\x6e\x20\x62\x6c\x6f\x63"
			"\x6b\x2d\x73\x69\x7a\x65\x20\x64"
			"\x61\x74\x61\x2e\x20\x54\x68\x65"
			"\x20\x6b\x65\x79\x20\x6e\x65\x65"
			"\x64\x73\x20\x74\x6f\x20\x62\x65"
			"\x20\x68\x61\x73\x68\x65\x64\x20"
			"\x62\x65\x66\x6f\x72\x65\x20\x62"
			"\x65\x69\x6e\x67\x20\x75\x73\x65"
			"\x64\x20\x62\x79\x20\x74\x68\x65"
			"\x20\x48\x4d\x41\x43\x20\x61\x6c"
			"\x67\x6f\x72\x69\x74\x68\x6d\x2e",
		.psize  = 152,
		.digest = "\x3a\x85\x41\x66\xac\x5d\x9f\x02"
			"\x3f\x54\xd5\x17\xd0\xb3\x9d\xbd"
			"\x94\x67\x70\xdb\x9c\x2b\x95\xc9"
			"\xf6\xf5\x65\xd1",
	},
};

/*
 * HMAC-SHA256 test vectors from
 * draft-ietf-ipsec-ciph-sha-256-01.txt
 */
#define HMAC_SHA256_TEST_VECTORS	10

static struct hash_testvec hmac_sha256_tv_template[] = {
	{
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18"
			  "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20",
		.ksize	= 32,
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xa2\x1b\x1f\x5d\x4c\xf4\xf7\x3a"
			  "\x4d\xd9\x39\x75\x0f\x7a\x06\x6a"
			  "\x7f\x98\xcc\x13\x1c\xb1\x6a\x66"
			  "\x92\x75\x90\x21\xcf\xab\x81\x81",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18"
			  "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20",
		.ksize	= 32,
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x10\x4f\xdc\x12\x57\x32\x8f\x08"
			  "\x18\x4b\xa7\x31\x31\xc5\x3c\xae"
			  "\xe6\x98\xe3\x61\x19\x42\x11\x49"
			  "\xea\x8c\x71\x24\x56\x69\x7d\x30",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18"
			  "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20",
		.ksize	= 32,
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
			   "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 112,
		.digest	= "\x47\x03\x05\xfc\x7e\x40\xfe\x34"
			  "\xd3\xee\xb3\xe7\x73\xd9\x5a\xab"
			  "\x73\xac\xf0\xfd\x06\x04\x47\xa5"
			  "\xeb\x45\x95\xbf\x33\xa9\xd1\xa3",
	}, {
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			"\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			"\x0b\x0b\x0b\x0b\x0b\x0b",
		.ksize	= 32,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x19\x8a\x60\x7e\xb4\x4b\xfb\xc6"
			  "\x99\x03\xa0\xf1\xcf\x2b\xbd\xc5"
			  "\xba\x0a\xa3\xf3\xd9\xae\x3c\x1c"
			  "\x7a\x3b\x16\x96\xa0\xb6\x8c\xf7",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x5b\xdc\xc1\x46\xbf\x60\x75\x4e"
			  "\x6a\x04\x24\x26\x08\x95\x75\xc7"
			  "\x5a\x00\x3f\x08\x9d\x27\x39\x83"
			  "\x9d\xec\x58\xb9\x64\xec\x38\x43",
		.np	= 2,
		.tap	= { 14, 14 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
		.ksize	= 32,
		.plaintext = "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",
		.psize	= 50,
		.digest	= "\xcd\xcb\x12\x20\xd1\xec\xcc\xea"
			  "\x91\xe5\x3a\xba\x30\x92\xf9\x62"
			  "\xe5\x49\xfe\x6c\xe9\xed\x7f\xdc"
			  "\x43\x19\x1f\xbd\xe4\x5c\x30\xb0",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18"
			  "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"
			  "\x21\x22\x23\x24\x25",
		.ksize	= 37,
		.plaintext = "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd",
		.psize	= 50,
		.digest	= "\xd4\x63\x3c\x17\xf6\xfb\x8d\x74"
			  "\x4c\x66\xde\xe0\xf8\xf0\x74\x55"
			  "\x6e\xc4\xaf\x55\xef\x07\x99\x85"
			  "\x41\x46\x8e\xb4\x9b\xd2\xe9\x17",
	}, {
		.key	= "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c"
			"\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c"
			"\x0c\x0c\x0c\x0c\x0c\x0c",
		.ksize	= 32,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= "\x75\x46\xaf\x01\x84\x1f\xc0\x9b"
			  "\x1a\xb9\xc3\x74\x9a\x5f\x1c\x17"
			  "\xd4\xf5\x89\x66\x8a\x58\x7b\x27"
			  "\x00\xa9\xc9\x7c\x11\x93\xcf\x42",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= "\x69\x53\x02\x5e\xd9\x6f\x0c\x09"
			  "\xf8\x0a\x96\xf7\x8e\x65\x38\xdb"
			  "\xe2\xe7\xb8\x20\xe3\xdd\x97\x0e"
			  "\x7d\xdd\x39\x09\x1b\x32\x35\x2f",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than "
			   "One Block-Size Data",
		.psize	= 73,
		.digest	= "\x63\x55\xac\x22\xe8\x90\xd0\xa3"
			  "\xc8\x48\x1a\x5c\xa4\x82\x5b\xc8"
			  "\x84\xd3\xe7\xa1\xff\x98\xa2\xfc"
			  "\x2a\xc7\xd8\xe0\x64\xc3\xb2\xe6",
	},
};
#if 0
#define CMAC_AES_TEST_VECTORS 6

static struct hash_testvec aes_cmac128_tv_template[] = {
	{ /* From NIST Special Publication 800-38B, AES-128 */
		.key		= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
				  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.plaintext	= zeroed_string,
		.digest		= "\xbb\x1d\x69\x29\xe9\x59\x37\x28"
				  "\x7f\xa3\x7d\x12\x9b\x75\x67\x46",
		.psize		= 0,
		.ksize		= 16,
	}, {
		.key		= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
				  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a",
		.digest		= "\x07\x0a\x16\xb4\x6b\x4d\x41\x44"
				  "\xf7\x9b\xdd\x9d\xd0\x4a\x28\x7c",
		.psize		= 16,
		.ksize		= 16,
	}, {
		.key		= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
				  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
				  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
				  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11",
		.digest		= "\xdf\xa6\x67\x47\xde\x9a\xe6\x30"
				  "\x30\xca\x32\x61\x14\x97\xc8\x27",
		.psize		= 40,
		.ksize		= 16,
	}, {
		.key		= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
				  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
				  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
				  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
				  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
				  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
				  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.digest		= "\x51\xf0\xbe\xbf\x7e\x3b\x9d\x92"
				  "\xfc\x49\x74\x17\x79\x36\x3c\xfe",
		.psize		= 64,
		.ksize		= 16,
	}, { /* From NIST Special Publication 800-38B, AES-256 */
		.key		= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
				  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
				  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
				  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.plaintext	= zeroed_string,
		.digest		= "\x02\x89\x62\xf6\x1b\x7b\xf8\x9e"
				  "\xfc\x6b\x55\x1f\x46\x67\xd9\x83",
		.psize		= 0,
		.ksize		= 32,
	}, {
		.key		= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
				  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
				  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
				  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
				  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
				  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
				  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
				  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
				  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.digest		= "\xe1\x99\x21\x90\x54\x9f\x6e\xd5"
				  "\x69\x6a\x2c\x05\x6c\x31\x54\x10",
		.psize		= 64,
		.ksize		= 32,
	}
};

#define CMAC_DES3_EDE_TEST_VECTORS 4

static struct hash_testvec des3_ede_cmac64_tv_template[] = {
/*
 * From NIST Special Publication 800-38B, Three Key TDEA
 * Corrected test vectors from:
 *  http://csrc.nist.gov/publications/nistpubs/800-38B/Updated_CMAC_Examples.pdf
 */
	{
		.key		= "\x8a\xa8\x3b\xf8\xcb\xda\x10\x62"
				  "\x0b\xc1\xbf\x19\xfb\xb6\xcd\x58"
				  "\xbc\x31\x3d\x4a\x37\x1c\xa8\xb5",
		.plaintext	= zeroed_string,
		.digest		= "\xb7\xa6\x88\xe1\x22\xff\xaf\x95",
		.psize		= 0,
		.ksize		= 24,
	}, {
		.key		= "\x8a\xa8\x3b\xf8\xcb\xda\x10\x62"
				  "\x0b\xc1\xbf\x19\xfb\xb6\xcd\x58"
				  "\xbc\x31\x3d\x4a\x37\x1c\xa8\xb5",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96",
		.digest		= "\x8e\x8f\x29\x31\x36\x28\x37\x97",
		.psize		= 8,
		.ksize		= 24,
	}, {
		.key		= "\x8a\xa8\x3b\xf8\xcb\xda\x10\x62"
				  "\x0b\xc1\xbf\x19\xfb\xb6\xcd\x58"
				  "\xbc\x31\x3d\x4a\x37\x1c\xa8\xb5",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57",
		.digest		= "\x74\x3d\xdb\xe0\xce\x2d\xc2\xed",
		.psize		= 20,
		.ksize		= 24,
	}, {
		.key		= "\x8a\xa8\x3b\xf8\xcb\xda\x10\x62"
				  "\x0b\xc1\xbf\x19\xfb\xb6\xcd\x58"
				  "\xbc\x31\x3d\x4a\x37\x1c\xa8\xb5",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
				  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51",
		.digest		= "\x33\xe6\xb1\x09\x24\x00\xea\xe5",
		.psize		= 32,
		.ksize		= 24,
	}
};
#endif
#define XCBC_AES_TEST_VECTORS 6

static struct hash_testvec aes_xcbc128_tv_template[] = {
	{
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = zeroed_string,
		.digest = "\x75\xf0\x25\x1d\x52\x8a\xc0\x1c"
			  "\x45\x73\xdf\xd5\x84\xd7\x9f\x29",
		.psize	= 0,
		.ksize	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02",
		.digest	= "\x5b\x37\x65\x80\xae\x2f\x19\xaf"
			  "\xe7\x21\x9c\xee\xf1\x72\x75\x6f",
		.psize	= 3,
		.ksize	= 16,
	} , {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02\x03\x04\x05\x06\x07"
			     "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.digest = "\xd2\xa2\x46\xfa\x34\x9b\x68\xa7"
			  "\x99\x98\xa4\x39\x4f\xf7\xa2\x63",
		.psize	= 16,
		.ksize	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02\x03\x04\x05\x06\x07"
			     "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			     "\x10\x11\x12\x13",
		.digest = "\x47\xf5\x1b\x45\x64\x96\x62\x15"
			  "\xb8\x98\x5c\x63\x05\x5e\xd3\x08",
		.tap	= { 10, 10 },
		.psize	= 20,
		.np	= 2,
		.ksize	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02\x03\x04\x05\x06\x07"
			     "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			     "\x10\x11\x12\x13\x14\x15\x16\x17"
			     "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.digest = "\xf5\x4f\x0e\xc8\xd2\xb9\xf3\xd3"
			  "\x68\x07\x73\x4b\xd5\x28\x3f\xd4",
		.psize	= 32,
		.ksize	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02\x03\x04\x05\x06\x07"
			     "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			     "\x10\x11\x12\x13\x14\x15\x16\x17"
			     "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			     "\x20\x21",
		.digest = "\xbe\xcb\xb3\xbc\xcd\xb5\x18\xa3"
			  "\x06\x77\xd5\x48\x1f\xb6\xb4\xd8",
		.tap	= { 17, 17 },
		.psize	= 34,
		.np	= 2,
		.ksize	= 16,
	}
};

/*
 * SHA384 HMAC test vectors from RFC4231
 */

#define HMAC_SHA384_TEST_VECTORS	4

static struct hash_testvec hmac_sha384_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\xaf\xd0\x39\x44\xd8\x48\x95\x62"
			  "\x6b\x08\x25\xf4\xab\x46\x90\x7f"
			  "\x15\xf9\xda\xdb\xe4\x10\x1e\xc6"
			  "\x82\xaa\x03\x4c\x7c\xeb\xc5\x9c"
			  "\xfa\xea\x9e\xa9\x07\x6e\xde\x7f"
			  "\x4a\xf1\x52\xe8\xb2\xfa\x9c\xb6",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\xaf\x45\xd2\xe3\x76\x48\x40\x31"
			  "\x61\x7f\x78\xd2\xb5\x8a\x6b\x1b"
			  "\x9c\x7e\xf4\x64\xf5\xa0\x1b\x47"
			  "\xe4\x2e\xc3\x73\x63\x22\x44\x5e"
			  "\x8e\x22\x40\xca\x5e\x69\xe2\xc7"
			  "\x8b\x32\x39\xec\xfa\xb2\x16\x49",
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "Test Using Larger Than Block-Siz"
			   "e Key - Hash Key First",
		.psize	= 54,
		.digest	= "\x4e\xce\x08\x44\x85\x81\x3e\x90"
			  "\x88\xd2\xc6\x3a\x04\x1b\xc5\xb4"
			  "\x4f\x9e\xf1\x01\x2a\x2b\x58\x8f"
			  "\x3c\xd1\x1f\x05\x03\x3a\xc4\xc6"
			  "\x0c\x2e\xf6\xab\x40\x30\xfe\x82"
			  "\x96\x24\x8d\xf1\x63\xf4\x49\x52",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "This is a test u"
			   "sing a larger th"
			   "an block-size ke"
			   "y and a larger t"
			   "han block-size d"
			   "ata. The key nee"
			   "ds to be hashed "
			   "before being use"
			   "d by the HMAC al"
			   "gorithm.",
		.psize	= 152,
		.digest	= "\x66\x17\x17\x8e\x94\x1f\x02\x0d"
			  "\x35\x1e\x2f\x25\x4e\x8f\xd3\x2c"
			  "\x60\x24\x20\xfe\xb0\xb8\xfb\x9a"
			  "\xdc\xce\xbb\x82\x46\x1e\x99\xc5"
			  "\xa6\x78\xcc\x31\xe7\x99\x17\x6d"
			  "\x38\x60\xe6\x11\x0c\x46\x52\x3e",
	},
};

/*
 * SHA512 HMAC test vectors from RFC4231
 */

#define HMAC_SHA512_TEST_VECTORS	4

static struct hash_testvec hmac_sha512_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x87\xaa\x7c\xde\xa5\xef\x61\x9d"
			  "\x4f\xf0\xb4\x24\x1a\x1d\x6c\xb0"
			  "\x23\x79\xf4\xe2\xce\x4e\xc2\x78"
			  "\x7a\xd0\xb3\x05\x45\xe1\x7c\xde"
			  "\xda\xa8\x33\xb7\xd6\xb8\xa7\x02"
			  "\x03\x8b\x27\x4e\xae\xa3\xf4\xe4"
			  "\xbe\x9d\x91\x4e\xeb\x61\xf1\x70"
			  "\x2e\x69\x6c\x20\x3a\x12\x68\x54",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x16\x4b\x7a\x7b\xfc\xf8\x19\xe2"
			  "\xe3\x95\xfb\xe7\x3b\x56\xe0\xa3"
			  "\x87\xbd\x64\x22\x2e\x83\x1f\xd6"
			  "\x10\x27\x0c\xd7\xea\x25\x05\x54"
			  "\x97\x58\xbf\x75\xc0\x5a\x99\x4a"
			  "\x6d\x03\x4f\x65\xf8\xf0\xe6\xfd"
			  "\xca\xea\xb1\xa3\x4d\x4a\x6b\x4b"
			  "\x63\x6e\x07\x0a\x38\xbc\xe7\x37",
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "Test Using Large"
			   "r Than Block-Siz"
			   "e Key - Hash Key"
			   " First",
		.psize	= 54,
		.digest	= "\x80\xb2\x42\x63\xc7\xc1\xa3\xeb"
			"\xb7\x14\x93\xc1\xdd\x7b\xe8\xb4"
			"\x9b\x46\xd1\xf4\x1b\x4a\xee\xc1"
			"\x12\x1b\x01\x37\x83\xf8\xf3\x52"
			"\x6b\x56\xd0\x37\xe0\x5f\x25\x98"
			"\xbd\x0f\xd2\x21\x5d\x6a\x1e\x52"
			"\x95\xe6\x4f\x73\xf6\x3f\x0a\xec"
			"\x8b\x91\x5a\x98\x5d\x78\x65\x98",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext =
			  "This is a test u"
			  "sing a larger th"
			  "an block-size ke"
			  "y and a larger t"
			  "han block-size d"
			  "ata. The key nee"
			  "ds to be hashed "
			  "before being use"
			  "d by the HMAC al"
			  "gorithm.",
		.psize	= 152,
		.digest	= "\xe3\x7b\x6a\x77\x5d\xc8\x7d\xba"
			"\xa4\xdf\xa9\xf9\x6e\x5e\x3f\xfd"
			"\xde\xbd\x71\xf8\x86\x72\x89\x86"
			"\x5d\xf5\xa3\x2d\x20\xcd\xc9\x44"
			"\xb6\x02\x2c\xac\x3c\x49\x82\xb1"
			"\x0d\x5e\xeb\x55\xc3\xe4\xde\x15"
			"\x13\x46\x76\xfb\x6d\xe0\x44\x60"
			"\x65\xc9\x74\x40\xfa\x8c\x6a\x58",
	},
};
#if 0
#define HMAC_SHA3_224_TEST_VECTORS	4

static struct hash_testvec hmac_sha3_224_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x3b\x16\x54\x6b\xbc\x7b\xe2\x70"
			  "\x6a\x03\x1d\xca\xfd\x56\x37\x3d"
			  "\x98\x84\x36\x76\x41\xd8\xc5\x9a"
			  "\xf3\xc8\x60\xf7",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x7f\xdb\x8d\xd8\x8b\xd2\xf6\x0d"
			  "\x1b\x79\x86\x34\xad\x38\x68\x11"
			  "\xc2\xcf\xc8\x5b\xfa\xf5\xd5\x2b"
			  "\xba\xce\x5e\x66",
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "Test Using Large"
			   "r Than Block-Siz"
			   "e Key - Hash Key"
			   " First",
		.psize	= 54,
		.digest = "\xb4\xa1\xf0\x4c\x00\x28\x7a\x9b"
			  "\x7f\x60\x75\xb3\x13\xd2\x79\xb8"
			  "\x33\xbc\x8f\x75\x12\x43\x52\xd0"
			  "\x5f\xb9\x99\x5f",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext =
			  "This is a test u"
			  "sing a larger th"
			  "an block-size ke"
			  "y and a larger t"
			  "han block-size d"
			  "ata. The key nee"
			  "ds to be hashed "
			  "before being use"
			  "d by the HMAC al"
			  "gorithm.",
		.psize	= 152,
		.digest	= "\x05\xd8\xcd\x6d\x00\xfa\xea\x8d"
			  "\x1e\xb6\x8a\xde\x28\x73\x0b\xbd"
			  "\x3c\xba\xb6\x92\x9f\x0a\x08\x6b"
			  "\x29\xcd\x62\xa0",
	},
};

#define HMAC_SHA3_256_TEST_VECTORS	4

static struct hash_testvec hmac_sha3_256_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\xba\x85\x19\x23\x10\xdf\xfa\x96"
			  "\xe2\xa3\xa4\x0e\x69\x77\x43\x51"
			  "\x14\x0b\xb7\x18\x5e\x12\x02\xcd"
			  "\xcc\x91\x75\x89\xf9\x5e\x16\xbb",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\xc7\xd4\x07\x2e\x78\x88\x77\xae"
			  "\x35\x96\xbb\xb0\xda\x73\xb8\x87"
			  "\xc9\x17\x1f\x93\x09\x5b\x29\x4a"
			  "\xe8\x57\xfb\xe2\x64\x5e\x1b\xa5",
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "Test Using Large"
			   "r Than Block-Siz"
			   "e Key - Hash Key"
			   " First",
		.psize	= 54,
		.digest = "\xed\x73\xa3\x74\xb9\x6c\x00\x52"
			  "\x35\xf9\x48\x03\x2f\x09\x67\x4a"
			  "\x58\xc0\xce\x55\x5c\xfc\x1f\x22"
			  "\x3b\x02\x35\x65\x60\x31\x2c\x3b",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext =
			  "This is a test u"
			  "sing a larger th"
			  "an block-size ke"
			  "y and a larger t"
			  "han block-size d"
			  "ata. The key nee"
			  "ds to be hashed "
			  "before being use"
			  "d by the HMAC al"
			  "gorithm.",
		.psize	= 152,
		.digest	= "\x65\xc5\xb0\x6d\x4c\x3d\xe3\x2a"
			  "\x7a\xef\x87\x63\x26\x1e\x49\xad"
			  "\xb6\xe2\x29\x3e\xc8\xe7\xc6\x1e"
			  "\x8d\xe6\x17\x01\xfc\x63\xe1\x23",
	},
};

#define HMAC_SHA3_384_TEST_VECTORS	4

static struct hash_testvec hmac_sha3_384_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x68\xd2\xdc\xf7\xfd\x4d\xdd\x0a"
			  "\x22\x40\xc8\xa4\x37\x30\x5f\x61"
			  "\xfb\x73\x34\xcf\xb5\xd0\x22\x6e"
			  "\x1b\xc2\x7d\xc1\x0a\x2e\x72\x3a"
			  "\x20\xd3\x70\xb4\x77\x43\x13\x0e"
			  "\x26\xac\x7e\x3d\x53\x28\x86\xbd",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\xf1\x10\x1f\x8c\xbf\x97\x66\xfd"
			  "\x67\x64\xd2\xed\x61\x90\x3f\x21"
			  "\xca\x9b\x18\xf5\x7c\xf3\xe1\xa2"
			  "\x3c\xa1\x35\x08\xa9\x32\x43\xce"
			  "\x48\xc0\x45\xdc\x00\x7f\x26\xa2"
			  "\x1b\x3f\x5e\x0e\x9d\xf4\xc2\x0a",
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "Test Using Large"
			   "r Than Block-Siz"
			   "e Key - Hash Key"
			   " First",
		.psize	= 54,
		.digest = "\x0f\xc1\x95\x13\xbf\x6b\xd8\x78"
			  "\x03\x70\x16\x70\x6a\x0e\x57\xbc"
			  "\x52\x81\x39\x83\x6b\x9a\x42\xc3"
			  "\xd4\x19\xe4\x98\xe0\xe1\xfb\x96"
			  "\x16\xfd\x66\x91\x38\xd3\x3a\x11"
			  "\x05\xe0\x7c\x72\xb6\x95\x3b\xcc",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext =
			  "This is a test u"
			  "sing a larger th"
			  "an block-size ke"
			  "y and a larger t"
			  "han block-size d"
			  "ata. The key nee"
			  "ds to be hashed "
			  "before being use"
			  "d by the HMAC al"
			  "gorithm.",
		.psize	= 152,
		.digest	= "\x02\x6f\xdf\x6b\x50\x74\x1e\x37"
			  "\x38\x99\xc9\xf7\xd5\x40\x6d\x4e"
			  "\xb0\x9f\xc6\x66\x56\x36\xfc\x1a"
			  "\x53\x00\x29\xdd\xf5\xcf\x3c\xa5"
			  "\xa9\x00\xed\xce\x01\xf5\xf6\x1e"
			  "\x2f\x40\x8c\xdf\x2f\xd3\xe7\xe8",
	},
};

#define HMAC_SHA3_512_TEST_VECTORS	4

static struct hash_testvec hmac_sha3_512_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\xeb\x3f\xbd\x4b\x2e\xaa\xb8\xf5"
			  "\xc5\x04\xbd\x3a\x41\x46\x5a\xac"
			  "\xec\x15\x77\x0a\x7c\xab\xac\x53"
			  "\x1e\x48\x2f\x86\x0b\x5e\xc7\xba"
			  "\x47\xcc\xb2\xc6\xf2\xaf\xce\x8f"
			  "\x88\xd2\x2b\x6d\xc6\x13\x80\xf2"
			  "\x3a\x66\x8f\xd3\x88\x8b\xb8\x05"
			  "\x37\xc0\xa0\xb8\x64\x07\x68\x9e",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x5a\x4b\xfe\xab\x61\x66\x42\x7c"
			  "\x7a\x36\x47\xb7\x47\x29\x2b\x83"
			  "\x84\x53\x7c\xdb\x89\xaf\xb3\xbf"
			  "\x56\x65\xe4\xc5\xe7\x09\x35\x0b"
			  "\x28\x7b\xae\xc9\x21\xfd\x7c\xa0"
			  "\xee\x7a\x0c\x31\xd0\x22\xa9\x5e"
			  "\x1f\xc9\x2b\xa9\xd7\x7d\xf8\x83"
			  "\x96\x02\x75\xbe\xb4\xe6\x20\x24",
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "Test Using Large"
			   "r Than Block-Siz"
			   "e Key - Hash Key"
			   " First",
		.psize	= 54,
		.digest = "\x00\xf7\x51\xa9\xe5\x06\x95\xb0"
			  "\x90\xed\x69\x11\xa4\xb6\x55\x24"
			  "\x95\x1c\xdc\x15\xa7\x3a\x5d\x58"
			  "\xbb\x55\x21\x5e\xa2\xcd\x83\x9a"
			  "\xc7\x9d\x2b\x44\xa3\x9b\xaf\xab"
			  "\x27\xe8\x3f\xde\x9e\x11\xf6\x34"
			  "\x0b\x11\xd9\x91\xb1\xb9\x1b\xf2"
			  "\xee\xe7\xfc\x87\x24\x26\xc3\xa4",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext =
			  "This is a test u"
			  "sing a larger th"
			  "an block-size ke"
			  "y and a larger t"
			  "han block-size d"
			  "ata. The key nee"
			  "ds to be hashed "
			  "before being use"
			  "d by the HMAC al"
			  "gorithm.",
		.psize	= 152,
		.digest	= "\x38\xa4\x56\xa0\x04\xbd\x10\xd3"
			  "\x2c\x9a\xb8\x33\x66\x84\x11\x28"
			  "\x62\xc3\xdb\x61\xad\xcc\xa3\x18"
			  "\x29\x35\x5e\xaf\x46\xfd\x5c\x73"
			  "\xd0\x6a\x1f\x0d\x13\xfe\xc9\xa6"
			  "\x52\xfb\x38\x11\xb5\x77\xb1\xb1"
			  "\xd1\xb9\x78\x9f\x97\xae\x5b\x83"
			  "\xc6\xf4\x4d\xfc\xf1\xd6\x7e\xba",
	},
};
#endif
/*
 * DES test vectors.
 */
#define DES_ENC_TEST_VECTORS		11
#define DES_DEC_TEST_VECTORS		5
#define DES_CBC_ENC_TEST_VECTORS	6
#define DES_CBC_DEC_TEST_VECTORS	5
#define DES_CTR_ENC_TEST_VECTORS	2
#define DES_CTR_DEC_TEST_VECTORS	2
#define DES3_EDE_ENC_TEST_VECTORS	4
#define DES3_EDE_DEC_TEST_VECTORS	4
#define DES3_EDE_CBC_ENC_TEST_VECTORS	2
#define DES3_EDE_CBC_DEC_TEST_VECTORS	2
#define DES3_EDE_CTR_ENC_TEST_VECTORS	2
#define DES3_EDE_CTR_DEC_TEST_VECTORS	2

static struct cipher_testvec des_enc_tv_template[] = {
	{ /* From Applied Cryptography */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7",
		.ilen	= 8,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d",
		.rlen	= 8,
	}, { /* Same key, different plaintext block */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x22\x33\x44\x55\x66\x77\x88\x99",
		.ilen	= 8,
		.result	= "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b",
		.rlen	= 8,
	}, { /* Sbox test from NBS */
		.key	= "\x7c\xa1\x10\x45\x4a\x1a\x6e\x57",
		.klen	= 8,
		.input	= "\x01\xa1\xd6\xd0\x39\x77\x67\x42",
		.ilen	= 8,
		.result	= "\x69\x0f\x5b\x0d\x9a\x26\x93\x9b",
		.rlen	= 8,
	}, { /* Three blocks */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xca\xfe\xba\xbe\xfe\xed\xbe\xef",
		.ilen	= 24,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b"
			  "\xb4\x99\x26\xf7\x1f\xe1\xd4\x90",
		.rlen	= 24,
	}, { /* Weak key */
		.fail	= true,
		.wk	= 1,
		.key	= "\x01\x01\x01\x01\x01\x01\x01\x01",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7",
		.ilen	= 8,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d",
		.rlen	= 8,
	}, { /* Two blocks -- for testing encryption across pages */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99",
		.ilen	= 16,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b",
		.rlen	= 16,
		.np	= 2,
		.tap	= { 8, 8 }
	}, { /* Four blocks -- for testing encryption with chunking */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xca\xfe\xba\xbe\xfe\xed\xbe\xef"
			  "\x22\x33\x44\x55\x66\x77\x88\x99",
		.ilen	= 32,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b"
			  "\xb4\x99\x26\xf7\x1f\xe1\xd4\x90"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b",
		.rlen	= 32,
		.np	= 3,
		.tap	= { 14, 10, 8 }
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xca\xfe\xba\xbe\xfe\xed\xbe\xef",
		.ilen	= 24,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b"
			  "\xb4\x99\x26\xf7\x1f\xe1\xd4\x90",
		.rlen	= 24,
		.np	= 4,
		.tap	= { 2, 1, 3, 18 }
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99",
		.ilen	= 16,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b",
		.rlen	= 16,
		.np	= 5,
		.tap	= { 2, 2, 2, 2, 8 }
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7",
		.ilen	= 8,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d",
		.rlen	= 8,
		.np	= 8,
		.tap	= { 1, 1, 1, 1, 1, 1, 1, 1 }
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.ilen	= 248,
		.result	= "\x88\xCB\x1F\xAB\x2F\x2A\x49\x57"
			  "\x92\xB9\x77\xFF\x2F\x47\x58\xDD"
			  "\xD7\x8A\x91\x95\x26\x33\x78\xB2"
			  "\x33\xBA\xB2\x3E\x02\xF5\x1F\xEF"
			  "\x98\xC5\xA6\xD2\x7D\x79\xEC\xB3"
			  "\x45\xF3\x4C\x61\xAC\x6C\xC2\x55"
			  "\xE5\xD3\x06\x58\x8A\x42\x3E\xDD"
			  "\x3D\x20\x45\xE9\x6F\x0D\x25\xA8"
			  "\xA5\xC7\x69\xCE\xD5\x3B\x7B\xC9"
			  "\x9E\x65\xE7\xA3\xF2\xE4\x18\x94"
			  "\xD2\x81\xE9\x33\x2B\x2D\x49\xC4"
			  "\xFE\xDA\x7F\xE2\xF2\x8C\x9C\xDC"
			  "\x73\x58\x11\x1F\x81\xD7\x21\x1A"
			  "\x80\xD0\x0D\xE8\x45\xD6\xD8\xD5"
			  "\x2E\x51\x16\xCA\x09\x89\x54\x62"
			  "\xF7\x04\x3D\x75\xB9\xA3\x84\xF4"
			  "\x62\xF0\x02\x58\x83\xAF\x30\x87"
			  "\x85\x3F\x01\xCD\x8E\x58\x42\xC4"
			  "\x41\x73\xE0\x15\x0A\xE6\x2E\x80"
			  "\x94\xF8\x5B\x3A\x4E\xDF\x51\xB2"
			  "\x9D\xE4\xC4\x9D\xF7\x3F\xF8\x8E"
			  "\x37\x22\x4D\x00\x2A\xEF\xC1\x0F"
			  "\x14\xA0\x66\xAB\x79\x39\xD0\x8E"
			  "\xE9\x95\x61\x74\x12\xED\x07\xD7"
			  "\xDD\x95\xDC\x7B\x57\x25\x27\x9C"
			  "\x51\x96\x16\xF7\x94\x61\xB8\x87"
			  "\xF0\x21\x1B\x32\xFB\x07\x0F\x29"
			  "\x56\xBD\x9D\x22\xA2\x9F\xA2\xB9"
			  "\x46\x31\x4C\x5E\x2E\x95\x61\xEF"
			  "\xE1\x58\x39\x09\xB4\x8B\x40\xAC"
			  "\x5F\x62\xC7\x72\xD9\xFC\xCB\x9A",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 248 - 10, 2, 8 },
	},
};

static struct cipher_testvec des_dec_tv_template[] = {
	{ /* From Applied Cryptography */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xe7",
		.rlen	= 8,
	}, { /* Sbox test from NBS */
		.key	= "\x7c\xa1\x10\x45\x4a\x1a\x6e\x57",
		.klen	= 8,
		.input	= "\x69\x0f\x5b\x0d\x9a\x26\x93\x9b",
		.ilen	= 8,
		.result	= "\x01\xa1\xd6\xd0\x39\x77\x67\x42",
		.rlen	= 8,
	}, { /* Two blocks, for chunking test */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\x69\x0f\x5b\x0d\x9a\x26\x93\x9b",
		.ilen	= 16,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\xa3\x99\x7b\xca\xaf\x69\xa0\xf5",
		.rlen	= 16,
		.np	= 2,
		.tap	= { 8, 8 }
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\x69\x0f\x5b\x0d\x9a\x26\x93\x9b",
		.ilen	= 16,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\xa3\x99\x7b\xca\xaf\x69\xa0\xf5",
		.rlen	= 16,
		.np	= 3,
		.tap	= { 3, 12, 1 }
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.input	= "\x88\xCB\x1F\xAB\x2F\x2A\x49\x57"
			  "\x92\xB9\x77\xFF\x2F\x47\x58\xDD"
			  "\xD7\x8A\x91\x95\x26\x33\x78\xB2"
			  "\x33\xBA\xB2\x3E\x02\xF5\x1F\xEF"
			  "\x98\xC5\xA6\xD2\x7D\x79\xEC\xB3"
			  "\x45\xF3\x4C\x61\xAC\x6C\xC2\x55"
			  "\xE5\xD3\x06\x58\x8A\x42\x3E\xDD"
			  "\x3D\x20\x45\xE9\x6F\x0D\x25\xA8"
			  "\xA5\xC7\x69\xCE\xD5\x3B\x7B\xC9"
			  "\x9E\x65\xE7\xA3\xF2\xE4\x18\x94"
			  "\xD2\x81\xE9\x33\x2B\x2D\x49\xC4"
			  "\xFE\xDA\x7F\xE2\xF2\x8C\x9C\xDC"
			  "\x73\x58\x11\x1F\x81\xD7\x21\x1A"
			  "\x80\xD0\x0D\xE8\x45\xD6\xD8\xD5"
			  "\x2E\x51\x16\xCA\x09\x89\x54\x62"
			  "\xF7\x04\x3D\x75\xB9\xA3\x84\xF4"
			  "\x62\xF0\x02\x58\x83\xAF\x30\x87"
			  "\x85\x3F\x01\xCD\x8E\x58\x42\xC4"
			  "\x41\x73\xE0\x15\x0A\xE6\x2E\x80"
			  "\x94\xF8\x5B\x3A\x4E\xDF\x51\xB2"
			  "\x9D\xE4\xC4\x9D\xF7\x3F\xF8\x8E"
			  "\x37\x22\x4D\x00\x2A\xEF\xC1\x0F"
			  "\x14\xA0\x66\xAB\x79\x39\xD0\x8E"
			  "\xE9\x95\x61\x74\x12\xED\x07\xD7"
			  "\xDD\x95\xDC\x7B\x57\x25\x27\x9C"
			  "\x51\x96\x16\xF7\x94\x61\xB8\x87"
			  "\xF0\x21\x1B\x32\xFB\x07\x0F\x29"
			  "\x56\xBD\x9D\x22\xA2\x9F\xA2\xB9"
			  "\x46\x31\x4C\x5E\x2E\x95\x61\xEF"
			  "\xE1\x58\x39\x09\xB4\x8B\x40\xAC"
			  "\x5F\x62\xC7\x72\xD9\xFC\xCB\x9A",
		.ilen	= 248,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 248 - 10, 2, 8 },
	},
};

static struct cipher_testvec des_cbc_enc_tv_template[] = {
	{ /* From OpenSSL */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.input	= "\x37\x36\x35\x34\x33\x32\x31\x20"
			  "\x4e\x6f\x77\x20\x69\x73\x20\x74"
			  "\x68\x65\x20\x74\x69\x6d\x65\x20",
		.ilen	= 24,
		.result	= "\xcc\xd1\x73\xff\xab\x20\x39\xf4"
			  "\xac\xd8\xae\xfd\xdf\xd8\xa1\xeb"
			  "\x46\x8e\x91\x15\x78\x88\xba\x68",
		.rlen	= 24,
	}, { /* FIPS Pub 81 */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x12\x34\x56\x78\x90\xab\xcd\xef",
		.input	= "\x4e\x6f\x77\x20\x69\x73\x20\x74",
		.ilen	= 8,
		.result	= "\xe5\xc7\xcd\xde\x87\x2b\xf2\x7c",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\xe5\xc7\xcd\xde\x87\x2b\xf2\x7c",
		.input	= "\x68\x65\x20\x74\x69\x6d\x65\x20",
		.ilen	= 8,
		.result	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.input	= "\x66\x6f\x72\x20\x61\x6c\x6c\x20",
		.ilen	= 8,
		.result	= "\x68\x37\x88\x49\x9a\x7c\x05\xf6",
		.rlen	= 8,
	}, { /* Copy of openssl vector for chunk testing */
	     /* From OpenSSL */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.input	= "\x37\x36\x35\x34\x33\x32\x31\x20"
			  "\x4e\x6f\x77\x20\x69\x73\x20\x74"
			  "\x68\x65\x20\x74\x69\x6d\x65\x20",
		.ilen	= 24,
		.result	= "\xcc\xd1\x73\xff\xab\x20\x39\xf4"
			  "\xac\xd8\xae\xfd\xdf\xd8\xa1\xeb"
			  "\x46\x8e\x91\x15\x78\x88\xba\x68",
		.rlen	= 24,
		.np	= 2,
		.tap	= { 13, 11 }
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.ilen	= 248,
		.result	= "\x71\xCC\x56\x1C\x87\x2C\x43\x20"
			  "\x1C\x20\x13\x09\xF9\x2B\x40\x47"
			  "\x99\x10\xD1\x1B\x65\x33\x33\xBA"
			  "\x88\x0D\xA2\xD1\x86\xFF\x4D\xF4"
			  "\x5A\x0C\x12\x96\x32\x57\xAA\x26"
			  "\xA7\xF4\x32\x8D\xBC\x10\x31\x9E"
			  "\x81\x72\x74\xDE\x30\x19\x69\x49"
			  "\x54\x9C\xC3\xEB\x0B\x97\xDD\xD1"
			  "\xE8\x6D\x0D\x05\x83\xA5\x12\x08"
			  "\x47\xF8\x88\x03\x86\x51\x3C\xEF"
			  "\xE7\x11\x73\x4D\x44\x2B\xE2\x16"
			  "\xE8\xA5\x06\x50\x66\x70\x0E\x14"
			  "\xBA\x21\x3B\xD5\x23\x5B\xA7\x8F"
			  "\x56\xB6\xA7\x44\xDB\x86\xAB\x69"
			  "\x33\x3C\xBE\x64\xC4\x22\xD3\xFE"
			  "\x49\x90\x88\x6A\x09\x8F\x76\x59"
			  "\xCB\xB7\xA0\x2D\x79\x75\x92\x8A"
			  "\x82\x1D\xC2\xFE\x09\x1F\x78\x6B"
			  "\x2F\xD6\xA4\x87\x1E\xC4\x53\x63"
			  "\x80\x02\x61\x2F\xE3\x46\xB6\xB5"
			  "\xAA\x95\xF4\xEE\xA7\x64\x2B\x4F"
			  "\x20\xCF\xD2\x47\x4E\x39\x65\xB3"
			  "\x11\x87\xA2\x6C\x49\x7E\x36\xC7"
			  "\x62\x8B\x48\x0D\x6A\x64\x00\xBD"
			  "\x71\x91\x8C\xE9\x70\x19\x01\x4F"
			  "\x4E\x68\x23\xBA\xDA\x24\x2E\x45"
			  "\x02\x14\x33\x21\xAE\x58\x4B\xCF"
			  "\x3B\x4B\xE8\xF8\xF6\x4F\x34\x93"
			  "\xD7\x07\x8A\xD7\x18\x92\x36\x8C"
			  "\x82\xA9\xBD\x6A\x31\x91\x39\x11"
			  "\xC6\x4A\xF3\x55\xC7\x29\x2E\x63",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 248 - 10, 2, 8 },
	},
};

static struct cipher_testvec des_cbc_dec_tv_template[] = {
	{ /* FIPS Pub 81 */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x12\x34\x56\x78\x90\xab\xcd\xef",
		.input	= "\xe5\xc7\xcd\xde\x87\x2b\xf2\x7c",
		.ilen	= 8,
		.result	= "\x4e\x6f\x77\x20\x69\x73\x20\x74",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\xe5\xc7\xcd\xde\x87\x2b\xf2\x7c",
		.input	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.ilen	= 8,
		.result	= "\x68\x65\x20\x74\x69\x6d\x65\x20",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.input	= "\x68\x37\x88\x49\x9a\x7c\x05\xf6",
		.ilen	= 8,
		.result	= "\x66\x6f\x72\x20\x61\x6c\x6c\x20",
		.rlen	= 8,
	}, { /* Copy of above, for chunk testing */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.input	= "\x68\x37\x88\x49\x9a\x7c\x05\xf6",
		.ilen	= 8,
		.result	= "\x66\x6f\x72\x20\x61\x6c\x6c\x20",
		.rlen	= 8,
		.np	= 2,
		.tap	= { 4, 4 }
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47",
		.input	= "\x71\xCC\x56\x1C\x87\x2C\x43\x20"
			  "\x1C\x20\x13\x09\xF9\x2B\x40\x47"
			  "\x99\x10\xD1\x1B\x65\x33\x33\xBA"
			  "\x88\x0D\xA2\xD1\x86\xFF\x4D\xF4"
			  "\x5A\x0C\x12\x96\x32\x57\xAA\x26"
			  "\xA7\xF4\x32\x8D\xBC\x10\x31\x9E"
			  "\x81\x72\x74\xDE\x30\x19\x69\x49"
			  "\x54\x9C\xC3\xEB\x0B\x97\xDD\xD1"
			  "\xE8\x6D\x0D\x05\x83\xA5\x12\x08"
			  "\x47\xF8\x88\x03\x86\x51\x3C\xEF"
			  "\xE7\x11\x73\x4D\x44\x2B\xE2\x16"
			  "\xE8\xA5\x06\x50\x66\x70\x0E\x14"
			  "\xBA\x21\x3B\xD5\x23\x5B\xA7\x8F"
			  "\x56\xB6\xA7\x44\xDB\x86\xAB\x69"
			  "\x33\x3C\xBE\x64\xC4\x22\xD3\xFE"
			  "\x49\x90\x88\x6A\x09\x8F\x76\x59"
			  "\xCB\xB7\xA0\x2D\x79\x75\x92\x8A"
			  "\x82\x1D\xC2\xFE\x09\x1F\x78\x6B"
			  "\x2F\xD6\xA4\x87\x1E\xC4\x53\x63"
			  "\x80\x02\x61\x2F\xE3\x46\xB6\xB5"
			  "\xAA\x95\xF4\xEE\xA7\x64\x2B\x4F"
			  "\x20\xCF\xD2\x47\x4E\x39\x65\xB3"
			  "\x11\x87\xA2\x6C\x49\x7E\x36\xC7"
			  "\x62\x8B\x48\x0D\x6A\x64\x00\xBD"
			  "\x71\x91\x8C\xE9\x70\x19\x01\x4F"
			  "\x4E\x68\x23\xBA\xDA\x24\x2E\x45"
			  "\x02\x14\x33\x21\xAE\x58\x4B\xCF"
			  "\x3B\x4B\xE8\xF8\xF6\x4F\x34\x93"
			  "\xD7\x07\x8A\xD7\x18\x92\x36\x8C"
			  "\x82\xA9\xBD\x6A\x31\x91\x39\x11"
			  "\xC6\x4A\xF3\x55\xC7\x29\x2E\x63",
		.ilen	= 248,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 248 - 10, 2, 8 },
	},
};
#if  0
static struct cipher_testvec des_ctr_enc_tv_template[] = {
	{ /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.ilen	= 248,
		.result	= "\x2F\x96\x06\x0F\x50\xC9\x68\x03"
			  "\x0F\x31\xD4\x64\xA5\x29\x77\x35"
			  "\xBC\x7A\x9F\x19\xE7\x0D\x33\x3E"
			  "\x12\x0B\x8C\xAE\x48\xAE\xD9\x02"
			  "\x0A\xD4\xB0\xD6\x37\xB2\x65\x1C"
			  "\x4B\x65\xEB\x24\xB5\x8E\xAD\x47"
			  "\x0D\xDA\x79\x77\xA0\x29\xA0\x2B"
			  "\xC8\x0F\x85\xDC\x03\x13\xA9\x04"
			  "\x19\x40\xBE\xBE\x5C\x49\x4A\x69"
			  "\xED\xE8\xE1\x9E\x14\x43\x74\xDE"
			  "\xEC\x6E\x11\x3F\x36\xEF\x7B\xFB"
			  "\xBE\x4C\x91\x43\x22\x65\x72\x48"
			  "\xE2\x12\xED\x88\xAC\xA7\xC9\x91"
			  "\x14\xA2\x36\x1C\x29\xFF\xC8\x4F"
			  "\x72\x5C\x4B\xB0\x1E\x93\xC2\xFA"
			  "\x9D\x53\x86\xA0\xAE\xC6\xB7\x3C"
			  "\x59\x0C\xD0\x8F\xA6\xD8\xA4\x31"
			  "\xB7\x30\x1C\x21\x38\xFB\x68\x8C"
			  "\x2E\xF5\x6E\x73\xC3\x16\x5F\x12"
			  "\x0C\x33\xB9\x1E\x7B\x70\xDE\x86"
			  "\x32\xB3\xC1\x16\xAB\xD9\x49\x0B"
			  "\x96\x28\x72\x6B\xF3\x30\xA9\xEB"
			  "\x69\xE2\x1E\x58\x46\xA2\x8E\xC7"
			  "\xC0\xEF\x07\xB7\x77\x2C\x00\x05"
			  "\x46\xBD\xFE\x53\x81\x8B\xA4\x03"
			  "\x20\x0F\xDB\x78\x0B\x1F\x53\x04"
			  "\x4C\x60\x4C\xC3\x2A\x86\x86\x7E"
			  "\x13\xD2\x26\xED\x5D\x3E\x9C\xF2"
			  "\x5C\xC4\x15\xC9\x9A\x21\xC5\xCD"
			  "\x19\x7F\x99\x19\x53\xCE\x1D\x14"
			  "\x69\x74\xA1\x06\x46\x0F\x4E\x75",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 248 - 10, 2, 8 },
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82",
		.ilen	= 247,
		.result	= "\x62\xE5\xF4\xDC\x99\xE7\x89\xE3"
			  "\xF4\x10\xCC\x21\x99\xEB\xDC\x15"
			  "\x19\x13\x93\x27\x9D\xB6\x6F\x45"
			  "\x17\x55\x61\x72\xC8\xD3\x7F\xA5"
			  "\x32\xD0\xD3\x02\x15\xA4\x05\x23"
			  "\x9C\x23\x61\x60\x77\x7B\x6C\x95"
			  "\x26\x49\x42\x2E\xF3\xC1\x8C\x6D"
			  "\xC8\x47\xD5\x94\xE7\x53\xC8\x23"
			  "\x1B\xA5\x0B\xCB\x12\xD3\x7A\x12"
			  "\xA4\x42\x15\x34\xF7\x5F\xDC\x58"
			  "\x5B\x58\x4C\xAD\xD1\x33\x8E\xE6"
			  "\xE5\xA0\xDA\x4D\x94\x3D\x63\xA8"
			  "\x02\x82\xBB\x16\xB8\xDC\xB5\x58"
			  "\xC3\x2D\x79\xE4\x25\x79\x43\xF9"
			  "\x6D\xD3\xCA\xC0\xE8\x12\xD4\x7E"
			  "\x04\x25\x79\xFD\x27\xFB\xC4\xEA"
			  "\x32\x94\x48\x92\xF3\x68\x1A\x7F"
			  "\x36\x33\x43\x79\xF7\xCA\xC2\x38"
			  "\xC0\x68\xD4\x53\xA9\xCC\x43\x0C"
			  "\x40\x57\x3E\xED\x00\x9F\x22\x6E"
			  "\x80\x99\x0B\xCC\x40\x63\x46\x8A"
			  "\xE8\xC4\x9B\x6D\x7A\x08\x6E\xA9"
			  "\x6F\x84\xBC\xB3\xF4\x95\x0B\x2D"
			  "\x6A\xBA\x37\x50\xC3\xCF\x9F\x7C"
			  "\x59\x5E\xDE\x0B\x30\xFA\x34\x8A"
			  "\xF8\xD1\xA2\xF8\x4E\xBD\x5D\x5E"
			  "\x7D\x71\x99\xE0\xF6\xE5\x7C\xE0"
			  "\x6D\xEE\x82\x89\x92\xD4\xF5\xD7"
			  "\xDF\x85\x2D\xE1\xB2\xD6\xAB\x94"
			  "\xA5\xA6\xE7\xB0\x51\x36\x52\x37"
			  "\x91\x45\x05\x3E\x58\xBF\x32",
		.rlen	= 247,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 247 - 8, 8 },
	},
};

static struct cipher_testvec des_ctr_dec_tv_template[] = {
	{ /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x2F\x96\x06\x0F\x50\xC9\x68\x03"
			  "\x0F\x31\xD4\x64\xA5\x29\x77\x35"
			  "\xBC\x7A\x9F\x19\xE7\x0D\x33\x3E"
			  "\x12\x0B\x8C\xAE\x48\xAE\xD9\x02"
			  "\x0A\xD4\xB0\xD6\x37\xB2\x65\x1C"
			  "\x4B\x65\xEB\x24\xB5\x8E\xAD\x47"
			  "\x0D\xDA\x79\x77\xA0\x29\xA0\x2B"
			  "\xC8\x0F\x85\xDC\x03\x13\xA9\x04"
			  "\x19\x40\xBE\xBE\x5C\x49\x4A\x69"
			  "\xED\xE8\xE1\x9E\x14\x43\x74\xDE"
			  "\xEC\x6E\x11\x3F\x36\xEF\x7B\xFB"
			  "\xBE\x4C\x91\x43\x22\x65\x72\x48"
			  "\xE2\x12\xED\x88\xAC\xA7\xC9\x91"
			  "\x14\xA2\x36\x1C\x29\xFF\xC8\x4F"
			  "\x72\x5C\x4B\xB0\x1E\x93\xC2\xFA"
			  "\x9D\x53\x86\xA0\xAE\xC6\xB7\x3C"
			  "\x59\x0C\xD0\x8F\xA6\xD8\xA4\x31"
			  "\xB7\x30\x1C\x21\x38\xFB\x68\x8C"
			  "\x2E\xF5\x6E\x73\xC3\x16\x5F\x12"
			  "\x0C\x33\xB9\x1E\x7B\x70\xDE\x86"
			  "\x32\xB3\xC1\x16\xAB\xD9\x49\x0B"
			  "\x96\x28\x72\x6B\xF3\x30\xA9\xEB"
			  "\x69\xE2\x1E\x58\x46\xA2\x8E\xC7"
			  "\xC0\xEF\x07\xB7\x77\x2C\x00\x05"
			  "\x46\xBD\xFE\x53\x81\x8B\xA4\x03"
			  "\x20\x0F\xDB\x78\x0B\x1F\x53\x04"
			  "\x4C\x60\x4C\xC3\x2A\x86\x86\x7E"
			  "\x13\xD2\x26\xED\x5D\x3E\x9C\xF2"
			  "\x5C\xC4\x15\xC9\x9A\x21\xC5\xCD"
			  "\x19\x7F\x99\x19\x53\xCE\x1D\x14"
			  "\x69\x74\xA1\x06\x46\x0F\x4E\x75",
		.ilen	= 248,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 248 - 10, 2, 8 },
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47",
		.input	= "\x62\xE5\xF4\xDC\x99\xE7\x89\xE3"
			  "\xF4\x10\xCC\x21\x99\xEB\xDC\x15"
			  "\x19\x13\x93\x27\x9D\xB6\x6F\x45"
			  "\x17\x55\x61\x72\xC8\xD3\x7F\xA5"
			  "\x32\xD0\xD3\x02\x15\xA4\x05\x23"
			  "\x9C\x23\x61\x60\x77\x7B\x6C\x95"
			  "\x26\x49\x42\x2E\xF3\xC1\x8C\x6D"
			  "\xC8\x47\xD5\x94\xE7\x53\xC8\x23"
			  "\x1B\xA5\x0B\xCB\x12\xD3\x7A\x12"
			  "\xA4\x42\x15\x34\xF7\x5F\xDC\x58"
			  "\x5B\x58\x4C\xAD\xD1\x33\x8E\xE6"
			  "\xE5\xA0\xDA\x4D\x94\x3D\x63\xA8"
			  "\x02\x82\xBB\x16\xB8\xDC\xB5\x58"
			  "\xC3\x2D\x79\xE4\x25\x79\x43\xF9"
			  "\x6D\xD3\xCA\xC0\xE8\x12\xD4\x7E"
			  "\x04\x25\x79\xFD\x27\xFB\xC4\xEA"
			  "\x32\x94\x48\x92\xF3\x68\x1A\x7F"
			  "\x36\x33\x43\x79\xF7\xCA\xC2\x38"
			  "\xC0\x68\xD4\x53\xA9\xCC\x43\x0C"
			  "\x40\x57\x3E\xED\x00\x9F\x22\x6E"
			  "\x80\x99\x0B\xCC\x40\x63\x46\x8A"
			  "\xE8\xC4\x9B\x6D\x7A\x08\x6E\xA9"
			  "\x6F\x84\xBC\xB3\xF4\x95\x0B\x2D"
			  "\x6A\xBA\x37\x50\xC3\xCF\x9F\x7C"
			  "\x59\x5E\xDE\x0B\x30\xFA\x34\x8A"
			  "\xF8\xD1\xA2\xF8\x4E\xBD\x5D\x5E"
			  "\x7D\x71\x99\xE0\xF6\xE5\x7C\xE0"
			  "\x6D\xEE\x82\x89\x92\xD4\xF5\xD7"
			  "\xDF\x85\x2D\xE1\xB2\xD6\xAB\x94"
			  "\xA5\xA6\xE7\xB0\x51\x36\x52\x37"
			  "\x91\x45\x05\x3E\x58\xBF\x32",
		.ilen	= 247,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82",
		.rlen	= 247,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 247 - 8, 8 },
	},
};
#endif
static struct cipher_testvec des3_ede_enc_tv_template[] = {
	{ /* These are from openssl */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\x55\x55\x55\x55\x55\x55\x55\x55"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 24,
		.input	= "\x73\x6f\x6d\x65\x64\x61\x74\x61",
		.ilen	= 8,
		.result	= "\x18\xd7\x48\xe5\x63\x62\x05\x72",
		.rlen	= 8,
	}, {
		.key	= "\x03\x52\x02\x07\x67\x20\x82\x17"
			  "\x86\x02\x87\x66\x59\x08\x21\x98"
			  "\x64\x05\x6a\xbd\xfe\xa9\x34\x57",
		.klen	= 24,
		.input	= "\x73\x71\x75\x69\x67\x67\x6c\x65",
		.ilen	= 8,
		.result	= "\xc0\x7d\x2a\x0f\xa5\x66\xfa\x30",
		.rlen	= 8,
	}, {
		.key	= "\x10\x46\x10\x34\x89\x98\x80\x20"
			  "\x91\x07\xd0\x15\x89\x19\x01\x01"
			  "\x19\x07\x92\x10\x98\x1a\x01\x01",
		.klen	= 24,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\xe1\xef\x62\xc3\x32\xfe\x82\x5b",
		.rlen	= 8,
	}, { /* Generated with Crypto++ */
		.key	= "\xF3\x9C\xD6\xF3\x9C\xB9\x5A\x67"
			  "\x00\x5A\x67\x00\x2D\xCE\xEB\x2D"
			  "\xCE\xEB\xB4\x51\x72\xB4\x51\x72",
		.klen	= 24,
		.input	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.ilen	= 496,
		.result	= "\x4E\x9A\x40\x3D\x61\x7D\x17\xFA"
			  "\x16\x86\x88\x0B\xD8\xAE\xF8\xE4"
			  "\x81\x01\x04\x00\x76\xFA\xED\xD3"
			  "\x44\x7E\x21\x9D\xF0\xFB\x2B\x64"
			  "\xCA\x4E\x90\xE0\xC0\x63\x28\x92"
			  "\xF3\x1F\xA4\x53\x2C\x77\xCC\x77"
			  "\x69\x56\xD0\x19\xAD\x00\x2D\x97"
			  "\xBC\xDE\x49\x6A\x82\xBC\x16\xE2"
			  "\x2F\x3E\x72\xEE\xD1\xCE\xFC\x1B"
			  "\xEA\x32\x56\xE4\x0B\xAF\x27\x36"
			  "\xAF\x08\xB9\x61\xB7\x48\x23\x27"
			  "\xEE\x4D\xC8\x79\x56\x06\xEB\xC7"
			  "\x5B\xCA\x0A\xC6\x5E\x5C\xCB\xB6"
			  "\x9D\xDA\x04\x59\xE2\x09\x48\x7E"
			  "\x6B\x37\xC6\xFE\x92\xA9\x1E\x6E"
			  "\x0D\x19\xFA\x33\x0F\xEE\x36\x68"
			  "\x11\xBB\xF9\x5A\x73\xAB\x3A\xEA"
			  "\xAC\x28\xD8\xD5\x27\xE8\x6B\x16"
			  "\x45\x86\x50\x01\x70\x35\x99\x92"
			  "\xDF\x0C\x07\x88\x8B\x7F\x9E\x4B"
			  "\xD2\x04\x84\x90\xC4\x27\xDF\x0A"
			  "\x49\xA8\xA7\x1A\x6D\x78\x16\xCA"
			  "\xB3\x18\x5C\xC3\x93\x63\x5A\x68"
			  "\x77\x02\xBA\xED\x62\x71\xB1\xD9"
			  "\x5E\xE5\x6F\x1A\xCC\x1D\xBE\x2E"
			  "\x11\xF3\xA6\x97\xCA\x8E\xBF\xB4"
			  "\x56\xA1\x36\x6B\xB1\x0A\x3E\x70"
			  "\xEA\xD7\xCD\x72\x7B\x79\xC8\xAD"
			  "\x6B\xFE\xFB\xBA\x64\xAE\x19\xC1"
			  "\x82\xCF\x8A\xA1\x50\x17\x7F\xB2"
			  "\x6F\x7B\x0F\x52\xC5\x3E\x4A\x52"
			  "\x3F\xD9\x3F\x01\xA6\x41\x1A\xB3"
			  "\xB3\x7A\x0E\x8E\x75\xB2\xB1\x5F"
			  "\xDB\xEA\x84\x13\x26\x6C\x85\x4E"
			  "\xAE\x6B\xDC\xE7\xE7\xAD\xB0\x06"
			  "\x5C\xBA\x92\xD0\x30\xBB\x8D\xD2"
			  "\xAE\x4C\x70\x85\xA0\x07\xE3\x2C"
			  "\xD1\x27\x9C\xCF\xDB\x13\xB7\xE5"
			  "\xF9\x6A\x02\xD0\x39\x9D\xB6\xE7"
			  "\xD1\x17\x25\x08\xF9\xA9\xA6\x67"
			  "\x38\x80\xD1\x22\xAB\x1A\xD7\x26"
			  "\xAD\xCA\x19\x1B\xFA\x18\xA7\x57"
			  "\x31\xEC\xC9\xED\xDB\x79\xC0\x48"
			  "\xAC\x31\x9F\x03\x8B\x62\x5B\x7E"
			  "\x0E\xA6\xD0\x64\xEE\xEA\x00\xFC"
			  "\x58\xC8\xDE\x51\x4E\x17\x15\x11"
			  "\x66\x58\xB6\x90\xDC\xDF\xA1\x49"
			  "\xCA\x79\xE9\x31\x31\x42\xDC\x56"
			  "\x0B\xCD\xB6\x0D\xC7\x64\xF7\x19"
			  "\xD9\x42\x05\x7F\xBC\x2F\xFC\x90"
			  "\xAE\x29\x86\xAA\x43\x7A\x4F\x6B"
			  "\xCE\xEA\xBC\x31\x8D\x65\x9D\x46"
			  "\xEA\x77\xB4\xF9\x58\xEA\x5D\x84"
			  "\xE4\xDC\x14\xBB\xBD\x15\x0E\xDA"
			  "\xD8\xE4\xA4\x5D\x61\xF9\x58\x0F"
			  "\xE4\x82\x77\xCE\x87\xC0\x09\xF0"
			  "\xD6\x10\x9E\x34\xE1\x0C\x67\x55"
			  "\x7B\x6D\xD5\x51\x4B\x00\xEE\xBA"
			  "\xF2\x7B\xBE\x75\x07\x42\x9D\x99"
			  "\x12\xE1\x71\x4A\xF9\x2A\xF5\xF6"
			  "\x93\x03\xD7\x51\x09\xFA\xBE\x68"
			  "\xD8\x45\xFF\x33\xBA\xBB\x2B\x63",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	},
};

static struct cipher_testvec des3_ede_dec_tv_template[] = {
	{ /* These are from openssl */
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\x55\x55\x55\x55\x55\x55\x55\x55"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 24,
		.input	= "\x18\xd7\x48\xe5\x63\x62\x05\x72",
		.ilen	= 8,
		.result	= "\x73\x6f\x6d\x65\x64\x61\x74\x61",
		.rlen	= 8,
	}, {
		.key	= "\x03\x52\x02\x07\x67\x20\x82\x17"
			  "\x86\x02\x87\x66\x59\x08\x21\x98"
			  "\x64\x05\x6a\xbd\xfe\xa9\x34\x57",
		.klen	= 24,
		.input	= "\xc0\x7d\x2a\x0f\xa5\x66\xfa\x30",
		.ilen	= 8,
		.result	= "\x73\x71\x75\x69\x67\x67\x6c\x65",
		.rlen	= 8,
	}, {
		.key	= "\x10\x46\x10\x34\x89\x98\x80\x20"
			  "\x91\x07\xd0\x15\x89\x19\x01\x01"
			  "\x19\x07\x92\x10\x98\x1a\x01\x01",
		.klen	= 24,
		.input	= "\xe1\xef\x62\xc3\x32\xfe\x82\x5b",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, { /* Generated with Crypto++ */
		.key	= "\xF3\x9C\xD6\xF3\x9C\xB9\x5A\x67"
			  "\x00\x5A\x67\x00\x2D\xCE\xEB\x2D"
			  "\xCE\xEB\xB4\x51\x72\xB4\x51\x72",
		.klen	= 24,
		.input	= "\x4E\x9A\x40\x3D\x61\x7D\x17\xFA"
			  "\x16\x86\x88\x0B\xD8\xAE\xF8\xE4"
			  "\x81\x01\x04\x00\x76\xFA\xED\xD3"
			  "\x44\x7E\x21\x9D\xF0\xFB\x2B\x64"
			  "\xCA\x4E\x90\xE0\xC0\x63\x28\x92"
			  "\xF3\x1F\xA4\x53\x2C\x77\xCC\x77"
			  "\x69\x56\xD0\x19\xAD\x00\x2D\x97"
			  "\xBC\xDE\x49\x6A\x82\xBC\x16\xE2"
			  "\x2F\x3E\x72\xEE\xD1\xCE\xFC\x1B"
			  "\xEA\x32\x56\xE4\x0B\xAF\x27\x36"
			  "\xAF\x08\xB9\x61\xB7\x48\x23\x27"
			  "\xEE\x4D\xC8\x79\x56\x06\xEB\xC7"
			  "\x5B\xCA\x0A\xC6\x5E\x5C\xCB\xB6"
			  "\x9D\xDA\x04\x59\xE2\x09\x48\x7E"
			  "\x6B\x37\xC6\xFE\x92\xA9\x1E\x6E"
			  "\x0D\x19\xFA\x33\x0F\xEE\x36\x68"
			  "\x11\xBB\xF9\x5A\x73\xAB\x3A\xEA"
			  "\xAC\x28\xD8\xD5\x27\xE8\x6B\x16"
			  "\x45\x86\x50\x01\x70\x35\x99\x92"
			  "\xDF\x0C\x07\x88\x8B\x7F\x9E\x4B"
			  "\xD2\x04\x84\x90\xC4\x27\xDF\x0A"
			  "\x49\xA8\xA7\x1A\x6D\x78\x16\xCA"
			  "\xB3\x18\x5C\xC3\x93\x63\x5A\x68"
			  "\x77\x02\xBA\xED\x62\x71\xB1\xD9"
			  "\x5E\xE5\x6F\x1A\xCC\x1D\xBE\x2E"
			  "\x11\xF3\xA6\x97\xCA\x8E\xBF\xB4"
			  "\x56\xA1\x36\x6B\xB1\x0A\x3E\x70"
			  "\xEA\xD7\xCD\x72\x7B\x79\xC8\xAD"
			  "\x6B\xFE\xFB\xBA\x64\xAE\x19\xC1"
			  "\x82\xCF\x8A\xA1\x50\x17\x7F\xB2"
			  "\x6F\x7B\x0F\x52\xC5\x3E\x4A\x52"
			  "\x3F\xD9\x3F\x01\xA6\x41\x1A\xB3"
			  "\xB3\x7A\x0E\x8E\x75\xB2\xB1\x5F"
			  "\xDB\xEA\x84\x13\x26\x6C\x85\x4E"
			  "\xAE\x6B\xDC\xE7\xE7\xAD\xB0\x06"
			  "\x5C\xBA\x92\xD0\x30\xBB\x8D\xD2"
			  "\xAE\x4C\x70\x85\xA0\x07\xE3\x2C"
			  "\xD1\x27\x9C\xCF\xDB\x13\xB7\xE5"
			  "\xF9\x6A\x02\xD0\x39\x9D\xB6\xE7"
			  "\xD1\x17\x25\x08\xF9\xA9\xA6\x67"
			  "\x38\x80\xD1\x22\xAB\x1A\xD7\x26"
			  "\xAD\xCA\x19\x1B\xFA\x18\xA7\x57"
			  "\x31\xEC\xC9\xED\xDB\x79\xC0\x48"
			  "\xAC\x31\x9F\x03\x8B\x62\x5B\x7E"
			  "\x0E\xA6\xD0\x64\xEE\xEA\x00\xFC"
			  "\x58\xC8\xDE\x51\x4E\x17\x15\x11"
			  "\x66\x58\xB6\x90\xDC\xDF\xA1\x49"
			  "\xCA\x79\xE9\x31\x31\x42\xDC\x56"
			  "\x0B\xCD\xB6\x0D\xC7\x64\xF7\x19"
			  "\xD9\x42\x05\x7F\xBC\x2F\xFC\x90"
			  "\xAE\x29\x86\xAA\x43\x7A\x4F\x6B"
			  "\xCE\xEA\xBC\x31\x8D\x65\x9D\x46"
			  "\xEA\x77\xB4\xF9\x58\xEA\x5D\x84"
			  "\xE4\xDC\x14\xBB\xBD\x15\x0E\xDA"
			  "\xD8\xE4\xA4\x5D\x61\xF9\x58\x0F"
			  "\xE4\x82\x77\xCE\x87\xC0\x09\xF0"
			  "\xD6\x10\x9E\x34\xE1\x0C\x67\x55"
			  "\x7B\x6D\xD5\x51\x4B\x00\xEE\xBA"
			  "\xF2\x7B\xBE\x75\x07\x42\x9D\x99"
			  "\x12\xE1\x71\x4A\xF9\x2A\xF5\xF6"
			  "\x93\x03\xD7\x51\x09\xFA\xBE\x68"
			  "\xD8\x45\xFF\x33\xBA\xBB\x2B\x63",
		.ilen	= 496,
		.result	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	},
};

static struct cipher_testvec des3_ede_cbc_enc_tv_template[] = {
	{ /* Generated from openssl */
		.key	= "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
			  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
			  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
			  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
			  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
			  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
			  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
			  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
			  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
			  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19",
		.rlen	= 128,
	}, { /* Generated with Crypto++ */
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xB2\xD7\x48\xED\x06\x44\xF9\x12"
			  "\xB7\x28\x4D\x83\x24\x59\xF2\x17",
		.input	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.ilen	= 496,
		.result	= "\xF8\xF6\xB5\x60\x5C\x5A\x75\x84"
			  "\x87\x81\x53\xBA\xC9\x6F\xEC\xD5"
			  "\x1E\x68\x8E\x85\x12\x86\x1D\x38"
			  "\x1C\x91\x40\xCC\x69\x6A\xD5\x35"
			  "\x0D\x7C\xB5\x07\x7C\x7B\x2A\xAF"
			  "\x32\xBC\xA1\xB3\x84\x31\x1B\x3C"
			  "\x0A\x2B\xFA\xD3\x9F\xB0\x8C\x37"
			  "\x8F\x9D\xA7\x6D\x6C\xFA\xD7\x90"
			  "\xE3\x69\x54\xED\x3A\xC4\xF1\x6B"
			  "\xB1\xCC\xFB\x7D\xD8\x8E\x17\x0B"
			  "\x9C\xF6\x4C\xD6\xFF\x03\x4E\xD9"
			  "\xE6\xA5\xAD\x25\xE6\x17\x69\x63"
			  "\x11\x35\x61\x94\x88\x7B\x1C\x48"
			  "\xF1\x24\x20\x29\x6B\x93\x1A\x8E"
			  "\x43\x03\x89\xD8\xB1\xDA\x47\x7B"
			  "\x79\x3A\x83\x76\xDA\xAE\xC6\xBB"
			  "\x22\xF8\xE8\x3D\x9A\x65\x54\xD8"
			  "\x4C\xE9\xE7\xE4\x63\x2F\x5C\x73"
			  "\x5A\xC3\xAE\x46\xA8\xCD\x57\xE6"
			  "\x67\x88\xA5\x20\x6F\x5F\x97\xC7"
			  "\xCC\x15\xA2\x0A\x93\xEA\x33\xE7"
			  "\x03\x5F\xEC\x64\x30\x6F\xEE\xD7"
			  "\x7E\xDF\xD6\xE9\x6F\x3F\xD6\x1E"
			  "\xBE\x67\x6C\x5B\x97\xA0\x09\xE6"
			  "\xEE\xFE\x55\xA3\x29\x65\xE0\x12"
			  "\xA1\x6A\x8A\x6F\xF2\xE6\xF1\x96"
			  "\x87\xFB\x9C\x05\xDD\x80\xEC\xFF"
			  "\xC5\xED\x50\xFE\xFC\x91\xCD\xCE"
			  "\x25\x2C\x5F\xD9\xAD\x95\x7D\x99"
			  "\xF0\x05\xC4\x71\x46\x5F\xF9\x0D"
			  "\xD2\x63\xDF\x9B\x96\x2E\x2B\xA6"
			  "\x2B\x1C\xD5\xFB\x96\x24\x60\x60"
			  "\x54\x40\xB8\x62\xA4\xF8\x46\x95"
			  "\x73\x28\xA3\xA6\x16\x2B\x17\xE7"
			  "\x7A\xF8\x62\x54\x3B\x64\x69\xE1"
			  "\x71\x34\x29\x5B\x4E\x05\x9B\xFA"
			  "\x5E\xF1\x96\xB7\xCE\x16\x9B\x59"
			  "\xF1\x1A\x4C\x51\x26\xFD\x79\xE2"
			  "\x3B\x8E\x71\x69\x6A\x91\xB6\x65"
			  "\x32\x09\xB8\xE4\x09\x1F\xEA\x39"
			  "\xCE\x20\x65\x9F\xD6\xD1\xC7\xF0"
			  "\x73\x50\x08\x56\x20\x9B\x94\x23"
			  "\x14\x39\xB7\x2B\xB1\x2D\x6D\x6F"
			  "\x41\x5B\xCC\xE2\x18\xAE\x62\x89"
			  "\x78\x8E\x67\x23\xD0\xFB\x2B\xE5"
			  "\x25\xC9\x48\x97\xB5\xD3\x17\xD5"
			  "\x6A\x9F\xA7\x48\x0C\x2B\x73\x3B"
			  "\x57\x08\xAE\x91\xF2\xB7\x57\x89"
			  "\xF4\xD0\xB0\x07\xB0\x42\x6C\xAF"
			  "\x98\x1A\xE7\xD1\xAC\x1E\xB5\x02"
			  "\xD4\x56\x42\x79\x79\x7F\x2A\x77"
			  "\x25\xE9\x7D\xC1\x88\x19\x2B\x49"
			  "\x6F\x46\x59\xAB\x56\x1F\x61\xE0"
			  "\x0C\x24\x9C\xC9\x5B\x63\xA9\x12"
			  "\xCF\x88\x96\xB6\xA8\x24\xC6\xA8"
			  "\x21\x85\x1A\x62\x7E\x34\xBB\xEB"
			  "\xBD\x02\x2A\xC7\xD8\x89\x80\xC5"
			  "\xB1\xBB\x60\xA5\x22\xFC\x6F\x38"
			  "\x02\x80\xA3\x28\x22\x75\xE1\xE9"
			  "\x90\xE9\xFA\x4B\x00\x10\xAC\x58"
			  "\x83\x70\xFF\x86\xE6\xAA\x0F\x1F"
			  "\x95\x63\x73\xA2\x44\xAC\xF8\xA5",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	},
};

static struct cipher_testvec des3_ede_cbc_dec_tv_template[] = {
	{ /* Generated from openssl */
		.key	= "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
			  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.input	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
			  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
			  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
			  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
			  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
			  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
			  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
			  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
			  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19",
		.ilen	= 128,
		.result	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.rlen	= 128,
	}, { /* Generated with Crypto++ */
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xB2\xD7\x48\xED\x06\x44\xF9\x12"
			  "\xB7\x28\x4D\x83\x24\x59\xF2\x17",
		.input	= "\xF8\xF6\xB5\x60\x5C\x5A\x75\x84"
			  "\x87\x81\x53\xBA\xC9\x6F\xEC\xD5"
			  "\x1E\x68\x8E\x85\x12\x86\x1D\x38"
			  "\x1C\x91\x40\xCC\x69\x6A\xD5\x35"
			  "\x0D\x7C\xB5\x07\x7C\x7B\x2A\xAF"
			  "\x32\xBC\xA1\xB3\x84\x31\x1B\x3C"
			  "\x0A\x2B\xFA\xD3\x9F\xB0\x8C\x37"
			  "\x8F\x9D\xA7\x6D\x6C\xFA\xD7\x90"
			  "\xE3\x69\x54\xED\x3A\xC4\xF1\x6B"
			  "\xB1\xCC\xFB\x7D\xD8\x8E\x17\x0B"
			  "\x9C\xF6\x4C\xD6\xFF\x03\x4E\xD9"
			  "\xE6\xA5\xAD\x25\xE6\x17\x69\x63"
			  "\x11\x35\x61\x94\x88\x7B\x1C\x48"
			  "\xF1\x24\x20\x29\x6B\x93\x1A\x8E"
			  "\x43\x03\x89\xD8\xB1\xDA\x47\x7B"
			  "\x79\x3A\x83\x76\xDA\xAE\xC6\xBB"
			  "\x22\xF8\xE8\x3D\x9A\x65\x54\xD8"
			  "\x4C\xE9\xE7\xE4\x63\x2F\x5C\x73"
			  "\x5A\xC3\xAE\x46\xA8\xCD\x57\xE6"
			  "\x67\x88\xA5\x20\x6F\x5F\x97\xC7"
			  "\xCC\x15\xA2\x0A\x93\xEA\x33\xE7"
			  "\x03\x5F\xEC\x64\x30\x6F\xEE\xD7"
			  "\x7E\xDF\xD6\xE9\x6F\x3F\xD6\x1E"
			  "\xBE\x67\x6C\x5B\x97\xA0\x09\xE6"
			  "\xEE\xFE\x55\xA3\x29\x65\xE0\x12"
			  "\xA1\x6A\x8A\x6F\xF2\xE6\xF1\x96"
			  "\x87\xFB\x9C\x05\xDD\x80\xEC\xFF"
			  "\xC5\xED\x50\xFE\xFC\x91\xCD\xCE"
			  "\x25\x2C\x5F\xD9\xAD\x95\x7D\x99"
			  "\xF0\x05\xC4\x71\x46\x5F\xF9\x0D"
			  "\xD2\x63\xDF\x9B\x96\x2E\x2B\xA6"
			  "\x2B\x1C\xD5\xFB\x96\x24\x60\x60"
			  "\x54\x40\xB8\x62\xA4\xF8\x46\x95"
			  "\x73\x28\xA3\xA6\x16\x2B\x17\xE7"
			  "\x7A\xF8\x62\x54\x3B\x64\x69\xE1"
			  "\x71\x34\x29\x5B\x4E\x05\x9B\xFA"
			  "\x5E\xF1\x96\xB7\xCE\x16\x9B\x59"
			  "\xF1\x1A\x4C\x51\x26\xFD\x79\xE2"
			  "\x3B\x8E\x71\x69\x6A\x91\xB6\x65"
			  "\x32\x09\xB8\xE4\x09\x1F\xEA\x39"
			  "\xCE\x20\x65\x9F\xD6\xD1\xC7\xF0"
			  "\x73\x50\x08\x56\x20\x9B\x94\x23"
			  "\x14\x39\xB7\x2B\xB1\x2D\x6D\x6F"
			  "\x41\x5B\xCC\xE2\x18\xAE\x62\x89"
			  "\x78\x8E\x67\x23\xD0\xFB\x2B\xE5"
			  "\x25\xC9\x48\x97\xB5\xD3\x17\xD5"
			  "\x6A\x9F\xA7\x48\x0C\x2B\x73\x3B"
			  "\x57\x08\xAE\x91\xF2\xB7\x57\x89"
			  "\xF4\xD0\xB0\x07\xB0\x42\x6C\xAF"
			  "\x98\x1A\xE7\xD1\xAC\x1E\xB5\x02"
			  "\xD4\x56\x42\x79\x79\x7F\x2A\x77"
			  "\x25\xE9\x7D\xC1\x88\x19\x2B\x49"
			  "\x6F\x46\x59\xAB\x56\x1F\x61\xE0"
			  "\x0C\x24\x9C\xC9\x5B\x63\xA9\x12"
			  "\xCF\x88\x96\xB6\xA8\x24\xC6\xA8"
			  "\x21\x85\x1A\x62\x7E\x34\xBB\xEB"
			  "\xBD\x02\x2A\xC7\xD8\x89\x80\xC5"
			  "\xB1\xBB\x60\xA5\x22\xFC\x6F\x38"
			  "\x02\x80\xA3\x28\x22\x75\xE1\xE9"
			  "\x90\xE9\xFA\x4B\x00\x10\xAC\x58"
			  "\x83\x70\xFF\x86\xE6\xAA\x0F\x1F"
			  "\x95\x63\x73\xA2\x44\xAC\xF8\xA5",
		.ilen	= 496,
		.result	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	},
};
#if 0
static struct cipher_testvec des3_ede_ctr_enc_tv_template[] = {
	{ /* Generated with Crypto++ */
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.ilen	= 496,
		.result	= "\x07\xC2\x08\x20\x72\x1F\x49\xEF"
			  "\x19\xCD\x6F\x32\x53\x05\x22\x15"
			  "\xA2\x85\x2B\xDB\x85\xD2\xD8\xB9"
			  "\xDD\x0D\x1B\x45\xCB\x69\x11\xD4"
			  "\xEA\xBE\xB2\x45\x5D\x0C\xAE\xBE"
			  "\xA0\xC1\x27\xAC\x65\x9F\x53\x7E"
			  "\xAF\xC2\x1B\xB5\xB8\x6D\x36\x0C"
			  "\x25\xC0\xF8\x6D\x0B\x29\x01\xDA"
			  "\x13\x78\xDC\x89\x12\x12\x43\xFA"
			  "\xF6\x12\xEF\x8D\x87\x62\x78\x83"
			  "\xE2\xBE\x41\x20\x4C\x6D\x35\x1B"
			  "\xD1\x0C\x30\xCF\xE2\xDE\x2B\x03"
			  "\xBF\x45\x73\xD4\xE5\x59\x95\xD1"
			  "\xB3\x9B\x27\x62\x97\xBD\xDE\x7F"
			  "\xA4\xD2\x39\x80\xAA\x50\x23\xF0"
			  "\x74\x88\x3D\xA8\x6A\x18\x79\x3B"
			  "\xC4\x96\x6C\x8D\x22\x40\x92\x6E"
			  "\xD6\xAD\x2A\x1F\xDE\x63\xC0\xE7"
			  "\x07\xF7\x2D\xF7\xB5\xF3\xF0\xCC"
			  "\x01\x7C\x2A\x9B\xC2\x10\xCA\xAA"
			  "\xFD\x2B\x3F\xC5\xF3\xF6\xFC\x9B"
			  "\x45\xDB\x53\xE4\x5B\xF3\xC9\x7B"
			  "\x8E\x52\xFF\xC8\x02\xB8\xAC\x9D"
			  "\xA1\x00\x39\xDA\x3D\x2D\x0E\x01"
			  "\x09\x7D\x8D\x5E\xBE\x53\xB9\xB0"
			  "\x8E\xE7\xE2\x96\x6A\xB2\x78\xEA"
			  "\xDE\x23\x8B\xA5\xFA\x5C\xE3\xDA"
			  "\xBF\x8E\x31\x6A\x55\xD1\x6A\xB2"
			  "\xB5\x46\x6F\xA5\xF0\xEE\xBA\x1F"
			  "\x9F\x98\xB0\x66\x4F\xD0\x3F\xA9"
			  "\xDF\x5F\x58\xC4\xF4\xFF\x75\x5C"
			  "\x40\x3A\x09\x7E\x6E\x1C\x97\xD4"
			  "\xCC\xE7\xE7\x71\xCF\x0B\x15\x08"
			  "\x71\xFA\x07\x97\xCD\xE6\xCA\x1D"
			  "\x14\x28\x0C\xCF\x99\x13\x7A\xF1"
			  "\xEB\xFA\xFA\x92\x07\xDE\x1D\xA1"
			  "\xD3\x36\x69\xFE\x51\x4D\x9F\x2E"
			  "\x83\x37\x4F\x1F\x48\x30\xED\x04"
			  "\x4D\xA4\xEF\x3A\xCA\x76\xF4\x1C"
			  "\x41\x8F\x63\x37\x78\x2F\x86\xA6"
			  "\xEF\x41\x7E\xD2\xAF\x88\xAB\x67"
			  "\x52\x71\xC3\x8E\xF8\x26\x93\x72"
			  "\xAA\xD6\x0E\xE7\x0B\x46\xB1\x3A"
			  "\xB4\x08\xA9\xA8\xA0\xCF\x20\x0C"
			  "\x52\xBC\x8B\x05\x56\xB2\xBC\x31"
			  "\x9B\x74\xB9\x29\x29\x96\x9A\x50"
			  "\xDC\x45\xDC\x1A\xEB\x0C\x64\xD4"
			  "\xD3\x05\x7E\x59\x55\xC3\xF4\x90"
			  "\xC2\xAB\xF8\x9B\x8A\xDA\xCE\xA1"
			  "\xC3\xF4\xAD\x77\xDD\x44\xC8\xAC"
			  "\xA3\xF1\xC9\xD2\x19\x5C\xB0\xCA"
			  "\xA2\x34\xC1\xF7\x6C\xFD\xAC\x65"
			  "\x32\xDC\x48\xC4\xF2\x00\x6B\x77"
			  "\xF1\x7D\x76\xAC\xC0\x31\x63\x2A"
			  "\xA5\x3A\x62\xC8\x91\xB1\x03\x65"
			  "\xCB\x43\xD1\x06\xDF\xC3\x67\xBC"
			  "\xDC\xE0\xCD\x35\xCE\x49\x65\xA0"
			  "\x52\x7B\xA7\x0D\x07\xA9\x1B\xB0"
			  "\x40\x77\x72\xC2\xEA\x0E\x3A\x78"
			  "\x46\xB9\x91\xB6\xE7\x3D\x51\x42"
			  "\xFD\x51\xB0\xC6\x2C\x63\x13\x78"
			  "\x5C\xEE\xFC\xCF\xC4\x70\x00\x34",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	}, { /* Generated with Crypto++ */
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xB2\xD7\x48\xED\x06\x44\xF9\x12"
			  "\xB7\x28\x4D\x83\x24\x59\xF2\x17",
		.input	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47"
			  "\x2E\xB1\x18",
		.ilen	= 499,
		.result	= "\x23\xFF\x5C\x99\x75\xBB\x1F\xD4"
			  "\xBC\x27\x9D\x36\x60\xA9\xC9\xF7"
			  "\x94\x9D\x1B\xFF\x8E\x95\x57\x89"
			  "\x8C\x2E\x33\x70\x43\x61\xE6\xD2"
			  "\x82\x33\x63\xB6\xC4\x34\x5E\xF8"
			  "\x96\x07\xA7\xD2\x3B\x8E\xC9\xAA"
			  "\x7C\xA0\x55\x89\x2E\xE1\x85\x25"
			  "\x14\x04\xDA\x6B\xE0\xEE\x56\xCF"
			  "\x08\x2E\x69\xD4\x54\xDE\x22\x84"
			  "\x69\xA6\xA7\xD3\x3A\x9A\xE8\x05"
			  "\x63\xDB\xBF\x46\x3A\x26\x2E\x0F"
			  "\x58\x5C\x46\xEA\x07\x40\xDA\xE1"
			  "\x14\x1D\xCD\x4F\x06\xC0\xCA\x54"
			  "\x1E\xC9\x45\x85\x67\x7C\xC2\xB5"
			  "\x97\x5D\x61\x78\x2E\x46\xEC\x6A"
			  "\x53\xF4\xD0\xAE\xFA\xB4\x86\x29"
			  "\x9F\x17\x33\x24\xD8\xB9\xB2\x05"
			  "\x93\x88\xEA\xF7\xA0\x70\x69\x49"
			  "\x88\x6B\x73\x40\x41\x8D\xD9\xD9"
			  "\x7E\x78\xE9\xBE\x6C\x14\x22\x7A"
			  "\x66\xE1\xDA\xED\x10\xFF\x69\x1D"
			  "\xB9\xAA\xF2\x56\x72\x1B\x23\xE2"
			  "\x45\x54\x8B\xA3\x70\x23\xB4\x5E"
			  "\x8E\x96\xC9\x05\x00\xB3\xB6\xC2"
			  "\x2A\x02\x43\x7A\x62\xD5\xC8\xD2"
			  "\xC2\xD0\xE4\x78\xA1\x7B\x3E\xE8"
			  "\x9F\x7F\x7D\x40\x54\x30\x3B\xC0"
			  "\xA5\x54\xFD\xCA\x25\xEC\x44\x3E"
			  "\x1A\x54\x7F\x88\xD0\xE1\xFE\x71"
			  "\xCE\x05\x49\x89\xBA\xD6\x72\xE7"
			  "\xD6\x5D\x3F\xA2\xD9\xAB\xC5\x02"
			  "\xD6\x43\x22\xAF\xA2\xE4\x80\x85"
			  "\xD7\x87\xB9\xEA\x43\xDB\xC8\xEF"
			  "\x5C\x82\x2E\x98\x0D\x30\x41\x6B"
			  "\x08\x48\x8D\xF0\xF8\x60\xD7\x9D"
			  "\xE9\xDE\x40\xAD\x0D\xAD\x0D\x58"
			  "\x2A\x98\x35\xFE\xF7\xDD\x4B\x40"
			  "\xDE\xB0\x05\xD9\x7B\x09\x4D\xBC"
			  "\x42\xC0\xF1\x15\x0B\xFA\x26\x6B"
			  "\xC6\x12\x13\x4F\xCB\x35\xBA\x35"
			  "\xDD\x7A\x36\x9C\x12\x57\x55\x83"
			  "\x78\x58\x09\xD0\xB0\xCF\x7C\x5C"
			  "\x38\xCF\xBD\x79\x5B\x13\x4D\x97"
			  "\xC1\x85\x6F\x97\xC9\xE8\xC2\xA4"
			  "\x98\xE2\xBD\x77\x6B\x53\x39\x1A"
			  "\x28\x10\xE7\xE0\xE7\xDE\x9D\x69"
			  "\x78\x6F\x8E\xD2\xD9\x5D\xD2\x15"
			  "\x9E\xB5\x4D\x8C\xC0\x78\x22\x2F"
			  "\x17\x11\x2E\x99\xD7\xE3\xA4\x4F"
			  "\x65\xA5\x6B\x03\x2C\x35\x6F\xDA"
			  "\x8A\x19\x08\xE1\x08\x48\x59\x51"
			  "\x53\x4B\xD1\xDF\xDA\x14\x50\x5F"
			  "\xDF\xB5\x8C\xDF\xC6\xFD\x85\xFA"
			  "\xD4\xF9\x64\x45\x65\x0D\x7D\xF4"
			  "\xC8\xCD\x3F\x32\xAF\xDD\x30\xED"
			  "\x7B\xAA\xAC\xF0\xDA\x7F\xDF\x75"
			  "\x1C\xA4\xF1\xCB\x5E\x4F\x0B\xB4"
			  "\x97\x73\x28\xDE\xCF\xAF\x82\xBD"
			  "\xC4\xBA\xB4\x9C\x0D\x16\x77\x42"
			  "\x42\x39\x7C\x53\xA4\xD4\xDD\x40"
			  "\x5C\x60\x1F\x6E\xA7\xE2\xDC\xE7"
			  "\x32\x0F\x05\x2F\xF2\x4C\x95\x3B"
			  "\xF2\x79\xD9",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec des3_ede_ctr_dec_tv_template[] = {
	{ /* Generated with Crypto++ */
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x07\xC2\x08\x20\x72\x1F\x49\xEF"
			  "\x19\xCD\x6F\x32\x53\x05\x22\x15"
			  "\xA2\x85\x2B\xDB\x85\xD2\xD8\xB9"
			  "\xDD\x0D\x1B\x45\xCB\x69\x11\xD4"
			  "\xEA\xBE\xB2\x45\x5D\x0C\xAE\xBE"
			  "\xA0\xC1\x27\xAC\x65\x9F\x53\x7E"
			  "\xAF\xC2\x1B\xB5\xB8\x6D\x36\x0C"
			  "\x25\xC0\xF8\x6D\x0B\x29\x01\xDA"
			  "\x13\x78\xDC\x89\x12\x12\x43\xFA"
			  "\xF6\x12\xEF\x8D\x87\x62\x78\x83"
			  "\xE2\xBE\x41\x20\x4C\x6D\x35\x1B"
			  "\xD1\x0C\x30\xCF\xE2\xDE\x2B\x03"
			  "\xBF\x45\x73\xD4\xE5\x59\x95\xD1"
			  "\xB3\x9B\x27\x62\x97\xBD\xDE\x7F"
			  "\xA4\xD2\x39\x80\xAA\x50\x23\xF0"
			  "\x74\x88\x3D\xA8\x6A\x18\x79\x3B"
			  "\xC4\x96\x6C\x8D\x22\x40\x92\x6E"
			  "\xD6\xAD\x2A\x1F\xDE\x63\xC0\xE7"
			  "\x07\xF7\x2D\xF7\xB5\xF3\xF0\xCC"
			  "\x01\x7C\x2A\x9B\xC2\x10\xCA\xAA"
			  "\xFD\x2B\x3F\xC5\xF3\xF6\xFC\x9B"
			  "\x45\xDB\x53\xE4\x5B\xF3\xC9\x7B"
			  "\x8E\x52\xFF\xC8\x02\xB8\xAC\x9D"
			  "\xA1\x00\x39\xDA\x3D\x2D\x0E\x01"
			  "\x09\x7D\x8D\x5E\xBE\x53\xB9\xB0"
			  "\x8E\xE7\xE2\x96\x6A\xB2\x78\xEA"
			  "\xDE\x23\x8B\xA5\xFA\x5C\xE3\xDA"
			  "\xBF\x8E\x31\x6A\x55\xD1\x6A\xB2"
			  "\xB5\x46\x6F\xA5\xF0\xEE\xBA\x1F"
			  "\x9F\x98\xB0\x66\x4F\xD0\x3F\xA9"
			  "\xDF\x5F\x58\xC4\xF4\xFF\x75\x5C"
			  "\x40\x3A\x09\x7E\x6E\x1C\x97\xD4"
			  "\xCC\xE7\xE7\x71\xCF\x0B\x15\x08"
			  "\x71\xFA\x07\x97\xCD\xE6\xCA\x1D"
			  "\x14\x28\x0C\xCF\x99\x13\x7A\xF1"
			  "\xEB\xFA\xFA\x92\x07\xDE\x1D\xA1"
			  "\xD3\x36\x69\xFE\x51\x4D\x9F\x2E"
			  "\x83\x37\x4F\x1F\x48\x30\xED\x04"
			  "\x4D\xA4\xEF\x3A\xCA\x76\xF4\x1C"
			  "\x41\x8F\x63\x37\x78\x2F\x86\xA6"
			  "\xEF\x41\x7E\xD2\xAF\x88\xAB\x67"
			  "\x52\x71\xC3\x8E\xF8\x26\x93\x72"
			  "\xAA\xD6\x0E\xE7\x0B\x46\xB1\x3A"
			  "\xB4\x08\xA9\xA8\xA0\xCF\x20\x0C"
			  "\x52\xBC\x8B\x05\x56\xB2\xBC\x31"
			  "\x9B\x74\xB9\x29\x29\x96\x9A\x50"
			  "\xDC\x45\xDC\x1A\xEB\x0C\x64\xD4"
			  "\xD3\x05\x7E\x59\x55\xC3\xF4\x90"
			  "\xC2\xAB\xF8\x9B\x8A\xDA\xCE\xA1"
			  "\xC3\xF4\xAD\x77\xDD\x44\xC8\xAC"
			  "\xA3\xF1\xC9\xD2\x19\x5C\xB0\xCA"
			  "\xA2\x34\xC1\xF7\x6C\xFD\xAC\x65"
			  "\x32\xDC\x48\xC4\xF2\x00\x6B\x77"
			  "\xF1\x7D\x76\xAC\xC0\x31\x63\x2A"
			  "\xA5\x3A\x62\xC8\x91\xB1\x03\x65"
			  "\xCB\x43\xD1\x06\xDF\xC3\x67\xBC"
			  "\xDC\xE0\xCD\x35\xCE\x49\x65\xA0"
			  "\x52\x7B\xA7\x0D\x07\xA9\x1B\xB0"
			  "\x40\x77\x72\xC2\xEA\x0E\x3A\x78"
			  "\x46\xB9\x91\xB6\xE7\x3D\x51\x42"
			  "\xFD\x51\xB0\xC6\x2C\x63\x13\x78"
			  "\x5C\xEE\xFC\xCF\xC4\x70\x00\x34",
		.ilen	= 496,
		.result	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	}, { /* Generated with Crypto++ */
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xB2\xD7\x48\xED\x06\x44\xF9\x12"
			  "\xB7\x28\x4D\x83\x24\x59\xF2\x17",
		.input	= "\x23\xFF\x5C\x99\x75\xBB\x1F\xD4"
			  "\xBC\x27\x9D\x36\x60\xA9\xC9\xF7"
			  "\x94\x9D\x1B\xFF\x8E\x95\x57\x89"
			  "\x8C\x2E\x33\x70\x43\x61\xE6\xD2"
			  "\x82\x33\x63\xB6\xC4\x34\x5E\xF8"
			  "\x96\x07\xA7\xD2\x3B\x8E\xC9\xAA"
			  "\x7C\xA0\x55\x89\x2E\xE1\x85\x25"
			  "\x14\x04\xDA\x6B\xE0\xEE\x56\xCF"
			  "\x08\x2E\x69\xD4\x54\xDE\x22\x84"
			  "\x69\xA6\xA7\xD3\x3A\x9A\xE8\x05"
			  "\x63\xDB\xBF\x46\x3A\x26\x2E\x0F"
			  "\x58\x5C\x46\xEA\x07\x40\xDA\xE1"
			  "\x14\x1D\xCD\x4F\x06\xC0\xCA\x54"
			  "\x1E\xC9\x45\x85\x67\x7C\xC2\xB5"
			  "\x97\x5D\x61\x78\x2E\x46\xEC\x6A"
			  "\x53\xF4\xD0\xAE\xFA\xB4\x86\x29"
			  "\x9F\x17\x33\x24\xD8\xB9\xB2\x05"
			  "\x93\x88\xEA\xF7\xA0\x70\x69\x49"
			  "\x88\x6B\x73\x40\x41\x8D\xD9\xD9"
			  "\x7E\x78\xE9\xBE\x6C\x14\x22\x7A"
			  "\x66\xE1\xDA\xED\x10\xFF\x69\x1D"
			  "\xB9\xAA\xF2\x56\x72\x1B\x23\xE2"
			  "\x45\x54\x8B\xA3\x70\x23\xB4\x5E"
			  "\x8E\x96\xC9\x05\x00\xB3\xB6\xC2"
			  "\x2A\x02\x43\x7A\x62\xD5\xC8\xD2"
			  "\xC2\xD0\xE4\x78\xA1\x7B\x3E\xE8"
			  "\x9F\x7F\x7D\x40\x54\x30\x3B\xC0"
			  "\xA5\x54\xFD\xCA\x25\xEC\x44\x3E"
			  "\x1A\x54\x7F\x88\xD0\xE1\xFE\x71"
			  "\xCE\x05\x49\x89\xBA\xD6\x72\xE7"
			  "\xD6\x5D\x3F\xA2\xD9\xAB\xC5\x02"
			  "\xD6\x43\x22\xAF\xA2\xE4\x80\x85"
			  "\xD7\x87\xB9\xEA\x43\xDB\xC8\xEF"
			  "\x5C\x82\x2E\x98\x0D\x30\x41\x6B"
			  "\x08\x48\x8D\xF0\xF8\x60\xD7\x9D"
			  "\xE9\xDE\x40\xAD\x0D\xAD\x0D\x58"
			  "\x2A\x98\x35\xFE\xF7\xDD\x4B\x40"
			  "\xDE\xB0\x05\xD9\x7B\x09\x4D\xBC"
			  "\x42\xC0\xF1\x15\x0B\xFA\x26\x6B"
			  "\xC6\x12\x13\x4F\xCB\x35\xBA\x35"
			  "\xDD\x7A\x36\x9C\x12\x57\x55\x83"
			  "\x78\x58\x09\xD0\xB0\xCF\x7C\x5C"
			  "\x38\xCF\xBD\x79\x5B\x13\x4D\x97"
			  "\xC1\x85\x6F\x97\xC9\xE8\xC2\xA4"
			  "\x98\xE2\xBD\x77\x6B\x53\x39\x1A"
			  "\x28\x10\xE7\xE0\xE7\xDE\x9D\x69"
			  "\x78\x6F\x8E\xD2\xD9\x5D\xD2\x15"
			  "\x9E\xB5\x4D\x8C\xC0\x78\x22\x2F"
			  "\x17\x11\x2E\x99\xD7\xE3\xA4\x4F"
			  "\x65\xA5\x6B\x03\x2C\x35\x6F\xDA"
			  "\x8A\x19\x08\xE1\x08\x48\x59\x51"
			  "\x53\x4B\xD1\xDF\xDA\x14\x50\x5F"
			  "\xDF\xB5\x8C\xDF\xC6\xFD\x85\xFA"
			  "\xD4\xF9\x64\x45\x65\x0D\x7D\xF4"
			  "\xC8\xCD\x3F\x32\xAF\xDD\x30\xED"
			  "\x7B\xAA\xAC\xF0\xDA\x7F\xDF\x75"
			  "\x1C\xA4\xF1\xCB\x5E\x4F\x0B\xB4"
			  "\x97\x73\x28\xDE\xCF\xAF\x82\xBD"
			  "\xC4\xBA\xB4\x9C\x0D\x16\x77\x42"
			  "\x42\x39\x7C\x53\xA4\xD4\xDD\x40"
			  "\x5C\x60\x1F\x6E\xA7\xE2\xDC\xE7"
			  "\x32\x0F\x05\x2F\xF2\x4C\x95\x3B"
			  "\xF2\x79\xD9",
		.ilen	= 499,
		.result	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47"
			  "\x2E\xB1\x18",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};
#endif
/*
 * AES test vectors.
 */
#define AES_ENC_TEST_VECTORS 4
#define AES_DEC_TEST_VECTORS 4
#define AES_CBC_ENC_TEST_VECTORS 5
#define AES_CBC_DEC_TEST_VECTORS 5
#define HMAC_MD5_ECB_CIPHER_NULL_ENC_TEST_VECTORS 2
#define HMAC_MD5_ECB_CIPHER_NULL_DEC_TEST_VECTORS 2
#define HMAC_SHA1_ECB_CIPHER_NULL_ENC_TEST_VEC 2
#define HMAC_SHA1_ECB_CIPHER_NULL_DEC_TEST_VEC 2
#define HMAC_SHA1_AES_CBC_ENC_TEST_VEC 7
#define HMAC_SHA256_AES_CBC_ENC_TEST_VEC 7
#define HMAC_SHA512_AES_CBC_ENC_TEST_VEC 7
#define AES_LRW_ENC_TEST_VECTORS 8
#define AES_LRW_DEC_TEST_VECTORS 8
#define AES_XTS_ENC_TEST_VECTORS 5
#define AES_XTS_DEC_TEST_VECTORS 5
#define AES_CTR_ENC_TEST_VECTORS 5
#define AES_CTR_DEC_TEST_VECTORS 5
#define AES_OFB_ENC_TEST_VECTORS 1
#define AES_OFB_DEC_TEST_VECTORS 1
#define AES_CTR_3686_ENC_TEST_VECTORS 7
#define AES_CTR_3686_DEC_TEST_VECTORS 6
#define AES_GCM_ENC_TEST_VECTORS 9
#define AES_GCM_DEC_TEST_VECTORS 8
#define AES_GCM_4106_ENC_TEST_VECTORS 23
#define AES_GCM_4106_DEC_TEST_VECTORS 23
#define AES_GCM_4543_ENC_TEST_VECTORS 1
#define AES_GCM_4543_DEC_TEST_VECTORS 2
#define AES_CCM_ENC_TEST_VECTORS 8
#define AES_CCM_DEC_TEST_VECTORS 7
#define AES_CCM_4309_ENC_TEST_VECTORS 7
#define AES_CCM_4309_DEC_TEST_VECTORS 10

static struct cipher_testvec aes_enc_tv_template[] = {
	{ /* From FIPS-197 */
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\x69\xc4\xe0\xd8\x6a\x7b\x04\x30"
			  "\xd8\xcd\xb7\x80\x70\xb4\xc5\x5a",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17",
		.klen	= 24,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\xdd\xa9\x7c\xa4\x86\x4c\xdf\xe0"
			  "\x6e\xaf\x70\xa0\xec\x0d\x71\x91",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\x8e\xa2\xb7\xca\x51\x67\x45\xbf"
			  "\xea\xfc\x49\x90\x4b\x49\x60\x89",
		.rlen	= 16,
	}, { /* Generated with Crypto++ */
		.key	= "\xA6\xC9\x83\xA6\xC9\xEC\x0F\x32"
			  "\x55\x0F\x32\x55\x78\x9B\xBE\x78"
			  "\x9B\xBE\xE1\x04\x27\xE1\x04\x27"
			  "\x4A\x6D\x90\x4A\x6D\x90\xB3\xD6",
		.klen	= 32,
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\x71\x73\xF7\xDB\x24\x93\x21\x6D"
			  "\x61\x1E\xBB\x63\x42\x79\xDB\x64"
			  "\x6F\x82\xC0\xCA\xA3\x9B\xFA\x0B"
			  "\xD9\x08\xC7\x4A\x90\xAE\x8F\x5F"
			  "\x5E\x06\xF0\x5F\x31\x51\x18\x37"
			  "\x45\xD7\xCA\x3A\xFD\x6C\x3F\xE1"
			  "\xDD\x8D\x22\x65\x2B\x00\x50\xCE"
			  "\xBA\x28\x67\xD7\xCE\x0E\x0D\xEA"
			  "\x78\x69\x7F\xAE\x8F\x8B\x69\x37"
			  "\x75\xE0\xDC\x96\xE0\xB7\xF4\x09"
			  "\xCB\x6D\xA2\xFB\xDA\xAF\x09\xF8"
			  "\x81\x82\x27\xFA\x45\x9C\x29\xA4"
			  "\x22\x8B\x78\x69\x5B\x46\xF9\x39"
			  "\x1B\xCC\xF9\x1D\x09\xEB\xBC\x5C"
			  "\x41\x72\x51\x97\x1D\x07\x49\xA0"
			  "\x1B\x8E\x65\x4B\xB2\x6A\x12\x03"
			  "\x6A\x60\x95\xAC\xBD\xAC\x1A\x64"
			  "\xDE\x5A\xA5\xF0\x83\x2F\xCB\xCA"
			  "\x22\x74\xA6\x6C\x9B\x73\xCE\x3F"
			  "\xE1\x8B\x22\x17\x59\x0C\x47\x89"
			  "\x33\xA1\xD6\x47\x03\x19\x4F\xA8"
			  "\x67\x69\xF0\x5B\xF0\x20\xAD\x06"
			  "\x27\x81\x92\xD8\xC5\xBA\x98\x12"
			  "\xBE\x24\xB5\x2F\x75\x02\xC2\xAD"
			  "\x12\x2F\x07\x32\xEE\x39\xAF\x64"
			  "\x05\x8F\xB3\xD4\xEB\x1B\x46\x6E"
			  "\xD9\x21\xF9\xC4\xB7\xC9\x45\x68"
			  "\xB4\xA1\x74\x9F\x82\x47\xEB\xCC"
			  "\xBD\x0A\x14\x95\x0F\x8B\xA8\x2F"
			  "\x4B\x1B\xA7\xBF\x82\xA6\x43\x0C"
			  "\xB9\x39\x4A\xA8\x10\x6F\x50\x7B"
			  "\x25\xFB\x26\x81\xE0\x2F\xF0\x96"
			  "\x8D\x8B\xAC\x92\x0F\xF6\xED\x64"
			  "\x63\x29\x4C\x8E\x18\x13\xC5\xBF"
			  "\xFC\xA0\xD9\xBF\x7C\x3A\x0E\x29"
			  "\x6F\xD1\x6C\x6F\xA5\xDA\xBF\xB1"
			  "\x30\xEA\x44\x2D\xC3\x8F\x16\xE1"
			  "\x66\xFA\xA3\x21\x3E\xFC\x13\xCA"
			  "\xF0\xF6\xF0\x59\xBD\x8F\x38\x50"
			  "\x31\xCB\x69\x3F\x96\x15\xD6\xF5"
			  "\xAE\xFF\xF6\xAA\x41\x85\x4C\x10"
			  "\x58\xE3\xF9\x44\xE6\x28\xDA\x9A"
			  "\xDC\x6A\x80\x34\x73\x97\x1B\xC5"
			  "\xCA\x26\x16\x77\x0E\x60\xAB\x89"
			  "\x0F\x04\x27\xBD\xCE\x3E\x71\xB4"
			  "\xA0\xD7\x22\x7E\xDB\xEB\x24\x70"
			  "\x42\x71\x51\x78\x70\xB3\xE0\x3D"
			  "\x84\x8E\x8D\x7B\xD0\x6D\xEA\x92"
			  "\x11\x08\x42\x4F\xE5\xAD\x26\x92"
			  "\xD2\x00\xAE\xA8\xE3\x4B\x37\x47"
			  "\x22\xC1\x95\xC1\x63\x7F\xCB\x03"
			  "\xF3\xE3\xD7\x9D\x60\xC7\xBC\xEA"
			  "\x35\xA2\xFD\x45\x52\x39\x13\x6F"
			  "\xC1\x53\xF3\x53\xDF\x33\x84\xD7"
			  "\xD2\xC8\x37\xB0\x75\xE3\x41\x46"
			  "\xB3\xC7\x83\x2E\x8A\xBB\xA4\xE5"
			  "\x7F\x3C\xFD\x8B\xEB\xEA\x63\xBD"
			  "\xB7\x46\xE7\xBF\x09\x9C\x0D\x0F"
			  "\x40\x86\x7F\x51\xE1\x11\x9C\xCB"
			  "\x88\xE6\x68\x47\xE3\x2B\xC5\xFF"
			  "\x09\x79\xA0\x43\x5C\x0D\x08\x58"
			  "\x17\xBB\xC0\x6B\x62\x3F\x56\xE9",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	},
};

static struct cipher_testvec aes_dec_tv_template[] = {
	{ /* From FIPS-197 */
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\x69\xc4\xe0\xd8\x6a\x7b\x04\x30"
			  "\xd8\xcd\xb7\x80\x70\xb4\xc5\x5a",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17",
		.klen	= 24,
		.input	= "\xdd\xa9\x7c\xa4\x86\x4c\xdf\xe0"
			  "\x6e\xaf\x70\xa0\xec\x0d\x71\x91",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\x8e\xa2\xb7\xca\x51\x67\x45\xbf"
			  "\xea\xfc\x49\x90\x4b\x49\x60\x89",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, { /* Generated with Crypto++ */
		.key	= "\xA6\xC9\x83\xA6\xC9\xEC\x0F\x32"
			  "\x55\x0F\x32\x55\x78\x9B\xBE\x78"
			  "\x9B\xBE\xE1\x04\x27\xE1\x04\x27"
			  "\x4A\x6D\x90\x4A\x6D\x90\xB3\xD6",
		.klen	= 32,
		.input	= "\x71\x73\xF7\xDB\x24\x93\x21\x6D"
			  "\x61\x1E\xBB\x63\x42\x79\xDB\x64"
			  "\x6F\x82\xC0\xCA\xA3\x9B\xFA\x0B"
			  "\xD9\x08\xC7\x4A\x90\xAE\x8F\x5F"
			  "\x5E\x06\xF0\x5F\x31\x51\x18\x37"
			  "\x45\xD7\xCA\x3A\xFD\x6C\x3F\xE1"
			  "\xDD\x8D\x22\x65\x2B\x00\x50\xCE"
			  "\xBA\x28\x67\xD7\xCE\x0E\x0D\xEA"
			  "\x78\x69\x7F\xAE\x8F\x8B\x69\x37"
			  "\x75\xE0\xDC\x96\xE0\xB7\xF4\x09"
			  "\xCB\x6D\xA2\xFB\xDA\xAF\x09\xF8"
			  "\x81\x82\x27\xFA\x45\x9C\x29\xA4"
			  "\x22\x8B\x78\x69\x5B\x46\xF9\x39"
			  "\x1B\xCC\xF9\x1D\x09\xEB\xBC\x5C"
			  "\x41\x72\x51\x97\x1D\x07\x49\xA0"
			  "\x1B\x8E\x65\x4B\xB2\x6A\x12\x03"
			  "\x6A\x60\x95\xAC\xBD\xAC\x1A\x64"
			  "\xDE\x5A\xA5\xF0\x83\x2F\xCB\xCA"
			  "\x22\x74\xA6\x6C\x9B\x73\xCE\x3F"
			  "\xE1\x8B\x22\x17\x59\x0C\x47\x89"
			  "\x33\xA1\xD6\x47\x03\x19\x4F\xA8"
			  "\x67\x69\xF0\x5B\xF0\x20\xAD\x06"
			  "\x27\x81\x92\xD8\xC5\xBA\x98\x12"
			  "\xBE\x24\xB5\x2F\x75\x02\xC2\xAD"
			  "\x12\x2F\x07\x32\xEE\x39\xAF\x64"
			  "\x05\x8F\xB3\xD4\xEB\x1B\x46\x6E"
			  "\xD9\x21\xF9\xC4\xB7\xC9\x45\x68"
			  "\xB4\xA1\x74\x9F\x82\x47\xEB\xCC"
			  "\xBD\x0A\x14\x95\x0F\x8B\xA8\x2F"
			  "\x4B\x1B\xA7\xBF\x82\xA6\x43\x0C"
			  "\xB9\x39\x4A\xA8\x10\x6F\x50\x7B"
			  "\x25\xFB\x26\x81\xE0\x2F\xF0\x96"
			  "\x8D\x8B\xAC\x92\x0F\xF6\xED\x64"
			  "\x63\x29\x4C\x8E\x18\x13\xC5\xBF"
			  "\xFC\xA0\xD9\xBF\x7C\x3A\x0E\x29"
			  "\x6F\xD1\x6C\x6F\xA5\xDA\xBF\xB1"
			  "\x30\xEA\x44\x2D\xC3\x8F\x16\xE1"
			  "\x66\xFA\xA3\x21\x3E\xFC\x13\xCA"
			  "\xF0\xF6\xF0\x59\xBD\x8F\x38\x50"
			  "\x31\xCB\x69\x3F\x96\x15\xD6\xF5"
			  "\xAE\xFF\xF6\xAA\x41\x85\x4C\x10"
			  "\x58\xE3\xF9\x44\xE6\x28\xDA\x9A"
			  "\xDC\x6A\x80\x34\x73\x97\x1B\xC5"
			  "\xCA\x26\x16\x77\x0E\x60\xAB\x89"
			  "\x0F\x04\x27\xBD\xCE\x3E\x71\xB4"
			  "\xA0\xD7\x22\x7E\xDB\xEB\x24\x70"
			  "\x42\x71\x51\x78\x70\xB3\xE0\x3D"
			  "\x84\x8E\x8D\x7B\xD0\x6D\xEA\x92"
			  "\x11\x08\x42\x4F\xE5\xAD\x26\x92"
			  "\xD2\x00\xAE\xA8\xE3\x4B\x37\x47"
			  "\x22\xC1\x95\xC1\x63\x7F\xCB\x03"
			  "\xF3\xE3\xD7\x9D\x60\xC7\xBC\xEA"
			  "\x35\xA2\xFD\x45\x52\x39\x13\x6F"
			  "\xC1\x53\xF3\x53\xDF\x33\x84\xD7"
			  "\xD2\xC8\x37\xB0\x75\xE3\x41\x46"
			  "\xB3\xC7\x83\x2E\x8A\xBB\xA4\xE5"
			  "\x7F\x3C\xFD\x8B\xEB\xEA\x63\xBD"
			  "\xB7\x46\xE7\xBF\x09\x9C\x0D\x0F"
			  "\x40\x86\x7F\x51\xE1\x11\x9C\xCB"
			  "\x88\xE6\x68\x47\xE3\x2B\xC5\xFF"
			  "\x09\x79\xA0\x43\x5C\x0D\x08\x58"
			  "\x17\xBB\xC0\x6B\x62\x3F\x56\xE9",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	},
};

static struct cipher_testvec aes_cbc_enc_tv_template[] = {
	{ /* From RFC 3602 */
		.key    = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 16,
		.iv	= "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input	= "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a",
		.rlen   = 16,
		.also_non_np = 1,
		.np	= 8,
		.tap	= { 3, 2, 3, 2, 3, 1, 1, 1 },
	}, {
		.key    = "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1",
		.rlen   = 32,
	}, { /* From NIST SP800-38A */
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b",
		.rlen	= 64,
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\xEA\x65\x8A\x19\xB0\x66\xC1\x3F"
			  "\xCE\xF1\x97\x75\xC1\xFD\xB5\xAF"
			  "\x52\x65\xF7\xFF\xBC\xD8\x2D\x9F"
			  "\x2F\xB9\x26\x9B\x6F\x10\xB7\xB8"
			  "\x26\xA1\x02\x46\xA2\xAD\xC6\xC0"
			  "\x11\x15\xFF\x6D\x1E\x82\x04\xA6"
			  "\xB1\x74\xD1\x08\x13\xFD\x90\x7C"
			  "\xF5\xED\xD3\xDB\x5A\x0A\x0C\x2F"
			  "\x0A\x70\xF1\x88\x07\xCF\x21\x26"
			  "\x40\x40\x8A\xF5\x53\xF7\x24\x4F"
			  "\x83\x38\x43\x5F\x08\x99\xEB\xE3"
			  "\xDC\x02\x64\x67\x50\x6E\x15\xC3"
			  "\x01\x1A\xA0\x81\x13\x65\xA6\x73"
			  "\x71\xA6\x3B\x91\x83\x77\xBE\xFA"
			  "\xDB\x71\x73\xA6\xC1\xAE\x43\xC3"
			  "\x36\xCE\xD6\xEB\xF9\x30\x1C\x4F"
			  "\x80\x38\x5E\x9C\x6E\xAB\x98\x2F"
			  "\x53\xAF\xCF\xC8\x9A\xB8\x86\x43"
			  "\x3E\x86\xE7\xA1\xF4\x2F\x30\x40"
			  "\x03\xA8\x6C\x50\x42\x9F\x77\x59"
			  "\x89\xA0\xC5\xEC\x9A\xB8\xDD\x99"
			  "\x16\x24\x02\x07\x48\xAE\xF2\x31"
			  "\x34\x0E\xC3\x85\xFE\x1C\x95\x99"
			  "\x87\x58\x98\x8B\xE7\xC6\xC5\x70"
			  "\x73\x81\x07\x7C\x56\x2F\xD8\x1B"
			  "\xB7\xB9\x2B\xAB\xE3\x01\x87\x0F"
			  "\xD8\xBB\xC0\x0D\xAC\x2C\x2F\x98"
			  "\x3C\x0B\xA2\x99\x4A\x8C\xF7\x04"
			  "\xE0\xE0\xCF\xD1\x81\x5B\xFE\xF5"
			  "\x24\x04\xFD\xB8\xDF\x13\xD8\xCD"
			  "\xF1\xE3\x3D\x98\x50\x02\x77\x9E"
			  "\xBC\x22\xAB\xFA\xC2\x43\x1F\x66"
			  "\x20\x02\x23\xDA\xDF\xA0\x89\xF6"
			  "\xD8\xF3\x45\x24\x53\x6F\x16\x77"
			  "\x02\x3E\x7B\x36\x5F\xA0\x3B\x78"
			  "\x63\xA2\xBD\xB5\xA4\xCA\x1E\xD3"
			  "\x57\xBC\x0B\x9F\x43\x51\x28\x4F"
			  "\x07\x50\x6C\x68\x12\x07\xCF\xFA"
			  "\x6B\x72\x0B\xEB\xF8\x88\x90\x2C"
			  "\x7E\xF5\x91\xD1\x03\xD8\xD5\xBD"
			  "\x22\x39\x7B\x16\x03\x01\x69\xAF"
			  "\x3D\x38\x66\x28\x0C\xBE\x5B\xC5"
			  "\x03\xB4\x2F\x51\x8A\x56\x17\x2B"
			  "\x88\x42\x6D\x40\x68\x8F\xD0\x11"
			  "\x19\xF9\x1F\x43\x79\x95\x31\xFA"
			  "\x28\x7A\x3D\xF7\x66\xEB\xEF\xAC"
			  "\x06\xB2\x01\xAD\xDB\x68\xDB\xEC"
			  "\x8D\x53\x6E\x72\x68\xA3\xC7\x63"
			  "\x43\x2B\x78\xE0\x04\x29\x8F\x72"
			  "\xB2\x2C\xE6\x84\x03\x30\x6D\xCD"
			  "\x26\x92\x37\xE1\x2F\xBB\x8B\x9D"
			  "\xE4\x4C\xF6\x93\xBC\xD9\xAD\x44"
			  "\x52\x65\xC7\xB0\x0E\x3F\x0E\x61"
			  "\x56\x5D\x1C\x6D\xA7\x05\x2E\xBC"
			  "\x58\x08\x15\xAB\x12\xAB\x17\x4A"
			  "\x5E\x1C\xF2\xCD\xB8\xA2\xAE\xFB"
			  "\x9B\x2E\x0E\x85\x34\x80\x0E\x3F"
			  "\x4C\xB8\xDB\xCE\x1C\x90\xA1\x61"
			  "\x6C\x69\x09\x35\x9E\xD4\xF4\xAD"
			  "\xBC\x06\x41\xE3\x01\xB4\x4E\x0A"
			  "\xE0\x1F\x91\xF8\x82\x96\x2D\x65"
			  "\xA3\xAA\x13\xCC\x50\xFF\x7B\x02",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	},
};

static struct cipher_testvec aes_cbc_dec_tv_template[] = {
	{ /* From RFC 3602 */
		.key    = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input  = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a",
		.ilen   = 16,
		.result = "Single block msg",
		.rlen   = 16,
		.also_non_np = 1,
		.np	= 8,
		.tap	= { 3, 2, 3, 2, 3, 1, 1, 1 },
	}, {
		.key    = "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1",
		.ilen   = 32,
		.result = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen   = 32,
	}, { /* From NIST SP800-38A */
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\xEA\x65\x8A\x19\xB0\x66\xC1\x3F"
			  "\xCE\xF1\x97\x75\xC1\xFD\xB5\xAF"
			  "\x52\x65\xF7\xFF\xBC\xD8\x2D\x9F"
			  "\x2F\xB9\x26\x9B\x6F\x10\xB7\xB8"
			  "\x26\xA1\x02\x46\xA2\xAD\xC6\xC0"
			  "\x11\x15\xFF\x6D\x1E\x82\x04\xA6"
			  "\xB1\x74\xD1\x08\x13\xFD\x90\x7C"
			  "\xF5\xED\xD3\xDB\x5A\x0A\x0C\x2F"
			  "\x0A\x70\xF1\x88\x07\xCF\x21\x26"
			  "\x40\x40\x8A\xF5\x53\xF7\x24\x4F"
			  "\x83\x38\x43\x5F\x08\x99\xEB\xE3"
			  "\xDC\x02\x64\x67\x50\x6E\x15\xC3"
			  "\x01\x1A\xA0\x81\x13\x65\xA6\x73"
			  "\x71\xA6\x3B\x91\x83\x77\xBE\xFA"
			  "\xDB\x71\x73\xA6\xC1\xAE\x43\xC3"
			  "\x36\xCE\xD6\xEB\xF9\x30\x1C\x4F"
			  "\x80\x38\x5E\x9C\x6E\xAB\x98\x2F"
			  "\x53\xAF\xCF\xC8\x9A\xB8\x86\x43"
			  "\x3E\x86\xE7\xA1\xF4\x2F\x30\x40"
			  "\x03\xA8\x6C\x50\x42\x9F\x77\x59"
			  "\x89\xA0\xC5\xEC\x9A\xB8\xDD\x99"
			  "\x16\x24\x02\x07\x48\xAE\xF2\x31"
			  "\x34\x0E\xC3\x85\xFE\x1C\x95\x99"
			  "\x87\x58\x98\x8B\xE7\xC6\xC5\x70"
			  "\x73\x81\x07\x7C\x56\x2F\xD8\x1B"
			  "\xB7\xB9\x2B\xAB\xE3\x01\x87\x0F"
			  "\xD8\xBB\xC0\x0D\xAC\x2C\x2F\x98"
			  "\x3C\x0B\xA2\x99\x4A\x8C\xF7\x04"
			  "\xE0\xE0\xCF\xD1\x81\x5B\xFE\xF5"
			  "\x24\x04\xFD\xB8\xDF\x13\xD8\xCD"
			  "\xF1\xE3\x3D\x98\x50\x02\x77\x9E"
			  "\xBC\x22\xAB\xFA\xC2\x43\x1F\x66"
			  "\x20\x02\x23\xDA\xDF\xA0\x89\xF6"
			  "\xD8\xF3\x45\x24\x53\x6F\x16\x77"
			  "\x02\x3E\x7B\x36\x5F\xA0\x3B\x78"
			  "\x63\xA2\xBD\xB5\xA4\xCA\x1E\xD3"
			  "\x57\xBC\x0B\x9F\x43\x51\x28\x4F"
			  "\x07\x50\x6C\x68\x12\x07\xCF\xFA"
			  "\x6B\x72\x0B\xEB\xF8\x88\x90\x2C"
			  "\x7E\xF5\x91\xD1\x03\xD8\xD5\xBD"
			  "\x22\x39\x7B\x16\x03\x01\x69\xAF"
			  "\x3D\x38\x66\x28\x0C\xBE\x5B\xC5"
			  "\x03\xB4\x2F\x51\x8A\x56\x17\x2B"
			  "\x88\x42\x6D\x40\x68\x8F\xD0\x11"
			  "\x19\xF9\x1F\x43\x79\x95\x31\xFA"
			  "\x28\x7A\x3D\xF7\x66\xEB\xEF\xAC"
			  "\x06\xB2\x01\xAD\xDB\x68\xDB\xEC"
			  "\x8D\x53\x6E\x72\x68\xA3\xC7\x63"
			  "\x43\x2B\x78\xE0\x04\x29\x8F\x72"
			  "\xB2\x2C\xE6\x84\x03\x30\x6D\xCD"
			  "\x26\x92\x37\xE1\x2F\xBB\x8B\x9D"
			  "\xE4\x4C\xF6\x93\xBC\xD9\xAD\x44"
			  "\x52\x65\xC7\xB0\x0E\x3F\x0E\x61"
			  "\x56\x5D\x1C\x6D\xA7\x05\x2E\xBC"
			  "\x58\x08\x15\xAB\x12\xAB\x17\x4A"
			  "\x5E\x1C\xF2\xCD\xB8\xA2\xAE\xFB"
			  "\x9B\x2E\x0E\x85\x34\x80\x0E\x3F"
			  "\x4C\xB8\xDB\xCE\x1C\x90\xA1\x61"
			  "\x6C\x69\x09\x35\x9E\xD4\xF4\xAD"
			  "\xBC\x06\x41\xE3\x01\xB4\x4E\x0A"
			  "\xE0\x1F\x91\xF8\x82\x96\x2D\x65"
			  "\xA3\xAA\x13\xCC\x50\xFF\x7B\x02",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	},
};

static struct aead_testvec hmac_sha1_aes_cbc_enc_tv_temp[] = {
	{ /* RFC 3602 Case 1 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00"
			  "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 8 + 20 + 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.assoc	= "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.alen	= 16,
		.input  = "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a"
			  "\x1b\x13\xcb\xaf\x89\x5e\xe1\x2c"
			  "\x13\xc5\x2e\xa3\xcc\xed\xdc\xb5"
			  "\x03\x71\xa2\x06",
		.rlen   = 16 + 20,
	}, { /* RFC 3602 Case 2 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33"
			  "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 8 + 20 + 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.assoc	= "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.alen	= 16,
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1"
			  "\xad\x9b\x4c\x5c\x85\xe1\xda\xae"
			  "\xee\x81\x4e\xd7\xdb\x74\xcf\x58"
			  "\x65\x39\xf8\xde",
		.rlen   = 32 + 20,
	}, { /* RFC 3602 Case 3 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"            /* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x6c\x3e\xa0\x47\x76\x30\xce\x21"
			  "\xa2\xce\x33\x4a\xa7\x46\xc2\xcd",
		.klen   = 8 + 20 + 16,
		.iv     = "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.assoc	= "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.alen	= 16,
		.input  = "This is a 48-byte message (exactly 3 AES blocks)",
		.ilen   = 48,
		.result = "\xd0\xa0\x2b\x38\x36\x45\x17\x53"
			  "\xd4\x93\x66\x5d\x33\xf0\xe8\x86"
			  "\x2d\xea\x54\xcd\xb2\x93\xab\xc7"
			  "\x50\x69\x39\x27\x67\x72\xf8\xd5"
			  "\x02\x1c\x19\x21\x6b\xad\x52\x5c"
			  "\x85\x79\x69\x5d\x83\xba\x26\x84"
			  "\xc2\xec\x0c\xf8\x7f\x05\xba\xca"
			  "\xff\xee\x4c\xd0\x93\xe6\x36\x7f"
			  "\x8d\x62\xf2\x1e",
		.rlen   = 48 + 20,
	}, { /* RFC 3602 Case 4 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x56\xe4\x7a\x38\xc5\x59\x89\x74"
			  "\xbc\x46\x90\x3d\xba\x29\x03\x49",
		.klen   = 8 + 20 + 16,
		.iv     = "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.assoc	= "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.alen	= 16,
		.input  = "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf",
		.ilen   = 64,
		.result = "\xc3\x0e\x32\xff\xed\xc0\x77\x4e"
			  "\x6a\xff\x6a\xf0\x86\x9f\x71\xaa"
			  "\x0f\x3a\xf0\x7a\x9a\x31\xa9\xc6"
			  "\x84\xdb\x20\x7e\xb0\xef\x8e\x4e"
			  "\x35\x90\x7a\xa6\x32\xc3\xff\xdf"
			  "\x86\x8b\xb7\xb2\x9d\x3d\x46\xad"
			  "\x83\xce\x9f\x9a\x10\x2e\xe9\x9d"
			  "\x49\xa5\x3e\x87\xf4\xc3\xda\x55"
			  "\x1c\x45\x57\xa9\x56\xcb\xa9\x2d"
			  "\x18\xac\xf1\xc7\x5d\xd1\xcd\x0d"
			  "\x1d\xbe\xc6\xe9",
		.rlen   = 64 + 20,
	}, { /* RFC 3602 Case 5 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"            /* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x90\xd3\x82\xb4\x10\xee\xba\x7a"
			  "\xd9\x38\xc4\x6c\xec\x1a\x82\xbf",
		.klen   = 8 + 20 + 16,
		.iv     = "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.alen   = 24,
		.input  = "\x08\x00\x0e\xbd\xa7\x0a\x00\x00"
			  "\x8e\x9c\x08\x3d\xb9\x5b\x07\x00"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0e\x01",
		.ilen   = 80,
		.result = "\xf6\x63\xc2\x5d\x32\x5c\x18\xc6"
			  "\xa9\x45\x3e\x19\x4e\x12\x08\x49"
			  "\xa4\x87\x0b\x66\xcc\x6b\x99\x65"
			  "\x33\x00\x13\xb4\x89\x8d\xc8\x56"
			  "\xa4\x69\x9e\x52\x3a\x55\xdb\x08"
			  "\x0b\x59\xec\x3a\x8e\x4b\x7e\x52"
			  "\x77\x5b\x07\xd1\xdb\x34\xed\x9c"
			  "\x53\x8a\xb5\x0c\x55\x1b\x87\x4a"
			  "\xa2\x69\xad\xd0\x47\xad\x2d\x59"
			  "\x13\xac\x19\xb7\xcf\xba\xd4\xa6"
			  "\x58\xc6\x84\x75\xe4\xe9\x6b\x0c"
			  "\xe1\xc5\x0b\x73\x4d\x82\x55\xa8"
			  "\x85\xe1\x59\xf7",
		.rlen   = 80 + 20,
       }, { /* NIST SP800-38A F.2.3 CBC-AES192.Encrypt */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"            /* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x18"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen   = 8 + 20 + 24,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.alen	= 16,
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd"
			  "\x73\xe3\x19\x3f\x8b\xc9\xc6\xf4"
			  "\x5a\xf1\x5b\xa8\x98\x07\xc5\x36"
			  "\x47\x4c\xfc\x36",
		.rlen   = 64 + 20,
	}, { /* NIST SP800-38A F.2.5 CBC-AES256.Encrypt */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x20"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen   = 8 + 20 + 32,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.alen	= 16,
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b"
			  "\xa3\xe8\x9b\x17\xe3\xf4\x7f\xde"
			  "\x1b\x9f\xc6\x81\x26\x43\x4a\x87"
			  "\x51\xee\xd6\x4e",
		.rlen   = 64 + 20,
	},
};

static struct aead_testvec hmac_sha256_aes_cbc_enc_tv_temp[] = {
	{ /* RFC 3602 Case 1 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 8 + 32 + 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.assoc	= "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.alen	= 16,
		.input  = "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a"
			  "\xcc\xde\x2d\x6a\xae\xf1\x0b\xcc"
			  "\x38\x06\x38\x51\xb4\xb8\xf3\x5b"
			  "\x5c\x34\xa6\xa3\x6e\x0b\x05\xe5"
			  "\x6a\x6d\x44\xaa\x26\xa8\x44\xa5",
		.rlen   = 16 + 32,
	}, { /* RFC 3602 Case 2 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 8 + 32 + 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.assoc	= "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.alen	= 16,
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1"
			  "\xf5\x33\x53\xf3\x68\x85\x2a\x99"
			  "\x0e\x06\x58\x8f\xba\xf6\x06\xda"
			  "\x49\x69\x0d\x5b\xd4\x36\x06\x62"
			  "\x35\x5e\x54\x58\x53\x4d\xdf\xbf",
		.rlen   = 32 + 32,
	}, { /* RFC 3602 Case 3 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"            /* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x6c\x3e\xa0\x47\x76\x30\xce\x21"
			  "\xa2\xce\x33\x4a\xa7\x46\xc2\xcd",
		.klen   = 8 + 32 + 16,
		.iv     = "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.assoc	= "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.alen	= 16,
		.input  = "This is a 48-byte message (exactly 3 AES blocks)",
		.ilen   = 48,
		.result = "\xd0\xa0\x2b\x38\x36\x45\x17\x53"
			  "\xd4\x93\x66\x5d\x33\xf0\xe8\x86"
			  "\x2d\xea\x54\xcd\xb2\x93\xab\xc7"
			  "\x50\x69\x39\x27\x67\x72\xf8\xd5"
			  "\x02\x1c\x19\x21\x6b\xad\x52\x5c"
			  "\x85\x79\x69\x5d\x83\xba\x26\x84"
			  "\x68\xb9\x3e\x90\x38\xa0\x88\x01"
			  "\xe7\xc6\xce\x10\x31\x2f\x9b\x1d"
			  "\x24\x78\xfb\xbe\x02\xe0\x4f\x40"
			  "\x10\xbd\xaa\xc6\xa7\x79\xe0\x1a",
		.rlen   = 48 + 32,
	}, { /* RFC 3602 Case 4 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x56\xe4\x7a\x38\xc5\x59\x89\x74"
			  "\xbc\x46\x90\x3d\xba\x29\x03\x49",
		.klen   = 8 + 32 + 16,
		.iv     = "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.assoc	= "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.alen	= 16,
		.input  = "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf",
		.ilen   = 64,
		.result = "\xc3\x0e\x32\xff\xed\xc0\x77\x4e"
			  "\x6a\xff\x6a\xf0\x86\x9f\x71\xaa"
			  "\x0f\x3a\xf0\x7a\x9a\x31\xa9\xc6"
			  "\x84\xdb\x20\x7e\xb0\xef\x8e\x4e"
			  "\x35\x90\x7a\xa6\x32\xc3\xff\xdf"
			  "\x86\x8b\xb7\xb2\x9d\x3d\x46\xad"
			  "\x83\xce\x9f\x9a\x10\x2e\xe9\x9d"
			  "\x49\xa5\x3e\x87\xf4\xc3\xda\x55"
			  "\x7a\x1b\xd4\x3c\xdb\x17\x95\xe2"
			  "\xe0\x93\xec\xc9\x9f\xf7\xce\xd8"
			  "\x3f\x54\xe2\x49\x39\xe3\x71\x25"
			  "\x2b\x6c\xe9\x5d\xec\xec\x2b\x64",
		.rlen   = 64 + 32,
	}, { /* RFC 3602 Case 5 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"            /* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x90\xd3\x82\xb4\x10\xee\xba\x7a"
			  "\xd9\x38\xc4\x6c\xec\x1a\x82\xbf",
		.klen   = 8 + 32 + 16,
		.iv     = "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.alen   = 24,
		.input  = "\x08\x00\x0e\xbd\xa7\x0a\x00\x00"
			  "\x8e\x9c\x08\x3d\xb9\x5b\x07\x00"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0e\x01",
		.ilen   = 80,
		.result = "\xf6\x63\xc2\x5d\x32\x5c\x18\xc6"
			  "\xa9\x45\x3e\x19\x4e\x12\x08\x49"
			  "\xa4\x87\x0b\x66\xcc\x6b\x99\x65"
			  "\x33\x00\x13\xb4\x89\x8d\xc8\x56"
			  "\xa4\x69\x9e\x52\x3a\x55\xdb\x08"
			  "\x0b\x59\xec\x3a\x8e\x4b\x7e\x52"
			  "\x77\x5b\x07\xd1\xdb\x34\xed\x9c"
			  "\x53\x8a\xb5\x0c\x55\x1b\x87\x4a"
			  "\xa2\x69\xad\xd0\x47\xad\x2d\x59"
			  "\x13\xac\x19\xb7\xcf\xba\xd4\xa6"
			  "\xbb\xd4\x0f\xbe\xa3\x3b\x4c\xb8"
			  "\x3a\xd2\xe1\x03\x86\xa5\x59\xb7"
			  "\x73\xc3\x46\x20\x2c\xb1\xef\x68"
			  "\xbb\x8a\x32\x7e\x12\x8c\x69\xcf",
		.rlen   = 80 + 32,
       }, { /* NIST SP800-38A F.2.3 CBC-AES192.Encrypt */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"            /* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x18"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen   = 8 + 32 + 24,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.alen   = 16,
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd"
			  "\x2f\xee\x5f\xdb\x66\xfe\x79\x09"
			  "\x61\x81\x31\xea\x5b\x3d\x8e\xfb"
			  "\xca\x71\x85\x93\xf7\x85\x55\x8b"
			  "\x7a\xe4\x94\xca\x8b\xba\x19\x33",
		.rlen   = 64 + 32,
	}, { /* NIST SP800-38A F.2.5 CBC-AES256.Encrypt */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x20"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen   = 8 + 32 + 32,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.alen   = 16,
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b"
			  "\x24\x29\xed\xc2\x31\x49\xdb\xb1"
			  "\x8f\x74\xbd\x17\x92\x03\xbe\x8f"
			  "\xf3\x61\xde\x1c\xe9\xdb\xcd\xd0"
			  "\xcc\xce\xe9\x85\x57\xcf\x6f\x5f",
		.rlen   = 64 + 32,
	},
};
#if 0
static struct aead_testvec hmac_sha512_aes_cbc_enc_tv_temp[] = {
	{ /* RFC 3602 Case 1 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 8 + 64 + 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.assoc	= "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.alen   = 16,
		.input  = "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a"
			  "\x3f\xdc\xad\x90\x03\x63\x5e\x68"
			  "\xc3\x13\xdd\xa4\x5c\x4d\x54\xa7"
			  "\x19\x6e\x03\x75\x2b\xa1\x62\xce"
			  "\xe0\xc6\x96\x75\xb2\x14\xca\x96"
			  "\xec\xbd\x50\x08\x07\x64\x1a\x49"
			  "\xe8\x9a\x7c\x06\x3d\xcb\xff\xb2"
			  "\xfa\x20\x89\xdd\x9c\xac\x9e\x16"
			  "\x18\x8a\xa0\x6d\x01\x6c\xa3\x3a",
		.rlen   = 16 + 64,
	}, { /* RFC 3602 Case 2 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 8 + 64 + 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.assoc	= "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.alen   = 16,
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1"
			  "\xda\xb2\x0c\xb2\x26\xc4\xd5\xef"
			  "\x60\x38\xa4\x5e\x9a\x8c\x1b\x41"
			  "\x03\x9f\xc4\x64\x7f\x01\x42\x9b"
			  "\x0e\x1b\xea\xef\xbc\x88\x19\x5e"
			  "\x31\x7e\xc2\x95\xfc\x09\x32\x0a"
			  "\x46\x32\x7c\x41\x9c\x59\x3e\xe9"
			  "\x8f\x9f\xd4\x31\xd6\x22\xbd\xf8"
			  "\xf7\x0a\x94\xe5\xa9\xc3\xf6\x9d",
		.rlen   = 32 + 64,
	}, { /* RFC 3602 Case 3 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"            /* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x6c\x3e\xa0\x47\x76\x30\xce\x21"
			  "\xa2\xce\x33\x4a\xa7\x46\xc2\xcd",
		.klen   = 8 + 64 + 16,
		.iv     = "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.assoc	= "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.alen   = 16,
		.input  = "This is a 48-byte message (exactly 3 AES blocks)",
		.ilen   = 48,
		.result = "\xd0\xa0\x2b\x38\x36\x45\x17\x53"
			  "\xd4\x93\x66\x5d\x33\xf0\xe8\x86"
			  "\x2d\xea\x54\xcd\xb2\x93\xab\xc7"
			  "\x50\x69\x39\x27\x67\x72\xf8\xd5"
			  "\x02\x1c\x19\x21\x6b\xad\x52\x5c"
			  "\x85\x79\x69\x5d\x83\xba\x26\x84"
			  "\x64\x19\x17\x5b\x57\xe0\x21\x0f"
			  "\xca\xdb\xa1\x26\x38\x14\xa2\x69"
			  "\xdb\x54\x67\x80\xc0\x54\xe0\xfd"
			  "\x3e\x91\xe7\x91\x7f\x13\x38\x44"
			  "\xb7\xb1\xd6\xc8\x7d\x48\x8d\x41"
			  "\x08\xea\x29\x6c\x74\x67\x3f\xb0"
			  "\xac\x7f\x5c\x1d\xf5\xee\x22\x66"
			  "\x27\xa6\xb6\x13\xba\xba\xf0\xc2",
		.rlen   = 48 + 64,
	}, { /* RFC 3602 Case 4 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x56\xe4\x7a\x38\xc5\x59\x89\x74"
			  "\xbc\x46\x90\x3d\xba\x29\x03\x49",
		.klen   = 8 + 64 + 16,
		.iv     = "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.assoc	= "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.alen   = 16,
		.input  = "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf",
		.ilen   = 64,
		.result = "\xc3\x0e\x32\xff\xed\xc0\x77\x4e"
			  "\x6a\xff\x6a\xf0\x86\x9f\x71\xaa"
			  "\x0f\x3a\xf0\x7a\x9a\x31\xa9\xc6"
			  "\x84\xdb\x20\x7e\xb0\xef\x8e\x4e"
			  "\x35\x90\x7a\xa6\x32\xc3\xff\xdf"
			  "\x86\x8b\xb7\xb2\x9d\x3d\x46\xad"
			  "\x83\xce\x9f\x9a\x10\x2e\xe9\x9d"
			  "\x49\xa5\x3e\x87\xf4\xc3\xda\x55"
			  "\x82\xcd\x42\x28\x21\x20\x15\xcc"
			  "\xb7\xb2\x48\x40\xc7\x64\x41\x3a"
			  "\x61\x32\x82\x85\xcf\x27\xed\xb4"
			  "\xe4\x68\xa2\xf5\x79\x26\x27\xb2"
			  "\x51\x67\x6a\xc4\xf0\x66\x55\x50"
			  "\xbc\x6f\xed\xd5\x8d\xde\x23\x7c"
			  "\x62\x98\x14\xd7\x2f\x37\x8d\xdf"
			  "\xf4\x33\x80\xeb\x8e\xb4\xa4\xda",
		.rlen   = 64 + 64,
	}, { /* RFC 3602 Case 5 */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"            /* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x10"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x90\xd3\x82\xb4\x10\xee\xba\x7a"
			  "\xd9\x38\xc4\x6c\xec\x1a\x82\xbf",
		.klen   = 8 + 64 + 16,
		.iv     = "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.alen   = 24,
		.input  = "\x08\x00\x0e\xbd\xa7\x0a\x00\x00"
			  "\x8e\x9c\x08\x3d\xb9\x5b\x07\x00"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0e\x01",
		.ilen   = 80,
		.result = "\xf6\x63\xc2\x5d\x32\x5c\x18\xc6"
			  "\xa9\x45\x3e\x19\x4e\x12\x08\x49"
			  "\xa4\x87\x0b\x66\xcc\x6b\x99\x65"
			  "\x33\x00\x13\xb4\x89\x8d\xc8\x56"
			  "\xa4\x69\x9e\x52\x3a\x55\xdb\x08"
			  "\x0b\x59\xec\x3a\x8e\x4b\x7e\x52"
			  "\x77\x5b\x07\xd1\xdb\x34\xed\x9c"
			  "\x53\x8a\xb5\x0c\x55\x1b\x87\x4a"
			  "\xa2\x69\xad\xd0\x47\xad\x2d\x59"
			  "\x13\xac\x19\xb7\xcf\xba\xd4\xa6"
			  "\x74\x84\x94\xe2\xd7\x7a\xf9\xbf"
			  "\x00\x8a\xa2\xd5\xb7\xf3\x60\xcf"
			  "\xa0\x47\xdf\x4e\x09\xf4\xb1\x7f"
			  "\x14\xd9\x3d\x53\x8e\x12\xb3\x00"
			  "\x4c\x0a\x4e\x32\x40\x43\x88\xce"
			  "\x92\x26\xc1\x76\x20\x11\xeb\xba"
			  "\x62\x4f\x9a\x62\x25\xc3\x75\x80"
			  "\xb7\x0a\x17\xf5\xd7\x94\xb4\x14",
		.rlen   = 80 + 64,
       }, { /* NIST SP800-38A F.2.3 CBC-AES192.Encrypt */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"            /* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x18"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen   = 8 + 64 + 24,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.alen   = 16,
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd"
			  "\x77\x4b\x69\x9d\x3a\x0d\xb4\x99"
			  "\x8f\xc6\x8e\x0e\x72\x58\xe3\x56"
			  "\xbb\x21\xd2\x7d\x93\x11\x17\x91"
			  "\xc4\x83\xfd\x0a\xea\x71\xfe\x77"
			  "\xae\x6f\x0a\xa5\xf0\xcf\xe1\x35"
			  "\xba\x03\xd5\x32\xfa\x5f\x41\x58"
			  "\x8d\x43\x98\xa7\x94\x16\x07\x02"
			  "\x0f\xb6\x81\x50\x28\x95\x2e\x75",
		.rlen   = 64 + 64,
	}, { /* NIST SP800-38A F.2.5 CBC-AES256.Encrypt */
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"            /* rta type */
#endif
			  "\x00\x00\x00\x20"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen   = 8 + 64 + 32,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.alen   = 16,
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b"
			  "\xb2\x27\x69\x7f\x45\x64\x79\x2b"
			  "\xb7\xb8\x4c\xd4\x75\x94\x68\x40"
			  "\x2a\xea\x91\xc7\x3f\x7c\xed\x7b"
			  "\x95\x2c\x9b\xa8\xf5\xe5\x52\x8d"
			  "\x6b\xe1\xae\xf1\x74\xfa\x0d\x0c"
			  "\xe3\x8d\x64\xc3\x8d\xff\x7c\x8c"
			  "\xdb\xbf\xa0\xb4\x01\xa2\xa8\xa2"
			  "\x2c\xb1\x62\x2c\x10\xca\xf1\x21",
		.rlen   = 64 + 64,
	},
};
#endif
#define HMAC_SHA1_DES_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha1_des_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
	.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x08"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
		  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24",
		.klen	= 8 + 20 + 8,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x70\xd6\xde\x64\x87\x17\xf1\xe8"
			  "\x54\x31\x85\x37\xed\x6b\x01\x8d"
			  "\xe3\xcc\xe0\x1d\x5e\xf3\xfe\xf1"
			  "\x41\xaa\x33\x91\xa7\x7d\x99\x88"
			  "\x4d\x85\x6e\x2f\xa3\x69\xf5\x82"
			  "\x3a\x6f\x25\xcb\x7d\x58\x1f\x9b"
			  "\xaa\x9c\x11\xd5\x76\x67\xce\xde"
			  "\x56\xd7\x5a\x80\x69\xea\x3a\x02"
			  "\xf0\xc7\x7c\xe3\xcb\x40\xe5\x52"
			  "\xd1\x10\x92\x78\x0b\x8e\x5b\xf1"
			  "\xe3\x26\x1f\xe1\x15\x41\xc7\xba"
			  "\x99\xdb\x08\x51\x1c\xd3\x01\xf4"
			  "\x87\x47\x39\xb8\xd2\xdd\xbd\xfb"
			  "\x66\x13\xdf\x1c\x01\x44\xf0\x7a"
			  "\x1a\x6b\x13\xf5\xd5\x0b\xb8\xba"
			  "\x53\xba\xe1\x76\xe3\x82\x07\x86"
			  "\x95\x16\x20\x09\xf5\x95\x19\xfd"
			  "\x3c\xc7\xe0\x42\xc0\x14\x69\xfa"
			  "\x5c\x44\xa9\x37",
			  .rlen	= 128 + 20,
	},
};

#define HMAC_SHA224_DES_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha224_des_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x08"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
		  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24",
		.klen	= 8 + 24 + 8,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x70\xd6\xde\x64\x87\x17\xf1\xe8"
			  "\x54\x31\x85\x37\xed\x6b\x01\x8d"
			  "\xe3\xcc\xe0\x1d\x5e\xf3\xfe\xf1"
			  "\x41\xaa\x33\x91\xa7\x7d\x99\x88"
			  "\x4d\x85\x6e\x2f\xa3\x69\xf5\x82"
			  "\x3a\x6f\x25\xcb\x7d\x58\x1f\x9b"
			  "\xaa\x9c\x11\xd5\x76\x67\xce\xde"
			  "\x56\xd7\x5a\x80\x69\xea\x3a\x02"
			  "\xf0\xc7\x7c\xe3\xcb\x40\xe5\x52"
			  "\xd1\x10\x92\x78\x0b\x8e\x5b\xf1"
			  "\xe3\x26\x1f\xe1\x15\x41\xc7\xba"
			  "\x99\xdb\x08\x51\x1c\xd3\x01\xf4"
		  "\x87\x47\x39\xb8\xd2\xdd\xbd\xfb"
			  "\x66\x13\xdf\x1c\x01\x44\xf0\x7a"
			  "\x1a\x6b\x13\xf5\xd5\x0b\xb8\xba"
			  "\x53\xba\xe1\x76\xe3\x82\x07\x86"
			  "\x9c\x2d\x7e\xee\x20\x34\x55\x0a"
			  "\xce\xb5\x4e\x64\x53\xe7\xbf\x91"
			  "\xab\xd4\xd9\xda\xc9\x12\xae\xf7",
		.rlen	= 128 + 24,
	},
};

#define HMAC_SHA256_DES_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha256_des_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x08"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24",
		.klen	= 8 + 32 + 8,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x70\xd6\xde\x64\x87\x17\xf1\xe8"
			  "\x54\x31\x85\x37\xed\x6b\x01\x8d"
			  "\xe3\xcc\xe0\x1d\x5e\xf3\xfe\xf1"
			  "\x41\xaa\x33\x91\xa7\x7d\x99\x88"
			  "\x4d\x85\x6e\x2f\xa3\x69\xf5\x82"
			  "\x3a\x6f\x25\xcb\x7d\x58\x1f\x9b"
			  "\xaa\x9c\x11\xd5\x76\x67\xce\xde"
			  "\x56\xd7\x5a\x80\x69\xea\x3a\x02"
			  "\xf0\xc7\x7c\xe3\xcb\x40\xe5\x52"
			  "\xd1\x10\x92\x78\x0b\x8e\x5b\xf1"
			  "\xe3\x26\x1f\xe1\x15\x41\xc7\xba"
		  "\x99\xdb\x08\x51\x1c\xd3\x01\xf4"
			  "\x87\x47\x39\xb8\xd2\xdd\xbd\xfb"
		  "\x66\x13\xdf\x1c\x01\x44\xf0\x7a"
		  "\x1a\x6b\x13\xf5\xd5\x0b\xb8\xba"
			  "\x53\xba\xe1\x76\xe3\x82\x07\x86"
			  "\xc6\x58\xa1\x60\x70\x91\x39\x36"
			  "\x50\xf6\x5d\xab\x4b\x51\x4e\x5e"
			  "\xde\x63\xde\x76\x52\xde\x9f\xba"
			  "\x90\xcf\x15\xf2\xbb\x6e\x84\x00",
		.rlen	= 128 + 32,
	},
};

#define HMAC_SHA384_DES_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha384_des_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x08"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24",
		.klen	= 8 + 48 + 8,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x70\xd6\xde\x64\x87\x17\xf1\xe8"
			  "\x54\x31\x85\x37\xed\x6b\x01\x8d"
			  "\xe3\xcc\xe0\x1d\x5e\xf3\xfe\xf1"
			  "\x41\xaa\x33\x91\xa7\x7d\x99\x88"
			  "\x4d\x85\x6e\x2f\xa3\x69\xf5\x82"
			  "\x3a\x6f\x25\xcb\x7d\x58\x1f\x9b"
			  "\xaa\x9c\x11\xd5\x76\x67\xce\xde"
			  "\x56\xd7\x5a\x80\x69\xea\x3a\x02"
			  "\xf0\xc7\x7c\xe3\xcb\x40\xe5\x52"
			  "\xd1\x10\x92\x78\x0b\x8e\x5b\xf1"
			  "\xe3\x26\x1f\xe1\x15\x41\xc7\xba"
			  "\x99\xdb\x08\x51\x1c\xd3\x01\xf4"
			  "\x87\x47\x39\xb8\xd2\xdd\xbd\xfb"
			  "\x66\x13\xdf\x1c\x01\x44\xf0\x7a"
			  "\x1a\x6b\x13\xf5\xd5\x0b\xb8\xba"
			  "\x53\xba\xe1\x76\xe3\x82\x07\x86"
			  "\xa8\x8e\x9c\x74\x8c\x2b\x99\xa0"
			  "\xc8\x8c\xef\x25\x07\x83\x11\x3a"
			  "\x31\x8d\xbe\x3b\x6a\xd7\x96\xfe"
			  "\x5e\x67\xb5\x74\xe7\xe7\x85\x61"
			  "\x6a\x95\x26\x75\xcc\x53\x89\xf3"
			  "\x74\xc9\x2a\x76\x20\xa2\x64\x62",
		.rlen	= 128 + 48,
	},
};

#define HMAC_SHA512_DES_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha512_des_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
		  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x08"	/* enc key length */
		  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24",
		.klen	= 8 + 64 + 8,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x70\xd6\xde\x64\x87\x17\xf1\xe8"
			  "\x54\x31\x85\x37\xed\x6b\x01\x8d"
			  "\xe3\xcc\xe0\x1d\x5e\xf3\xfe\xf1"
			  "\x41\xaa\x33\x91\xa7\x7d\x99\x88"
			  "\x4d\x85\x6e\x2f\xa3\x69\xf5\x82"
			  "\x3a\x6f\x25\xcb\x7d\x58\x1f\x9b"
			  "\xaa\x9c\x11\xd5\x76\x67\xce\xde"
		  "\x56\xd7\x5a\x80\x69\xea\x3a\x02"
			  "\xf0\xc7\x7c\xe3\xcb\x40\xe5\x52"
		  "\xd1\x10\x92\x78\x0b\x8e\x5b\xf1"
			  "\xe3\x26\x1f\xe1\x15\x41\xc7\xba"
			  "\x99\xdb\x08\x51\x1c\xd3\x01\xf4"
			  "\x87\x47\x39\xb8\xd2\xdd\xbd\xfb"
			  "\x66\x13\xdf\x1c\x01\x44\xf0\x7a"
		  "\x1a\x6b\x13\xf5\xd5\x0b\xb8\xba"
			  "\x53\xba\xe1\x76\xe3\x82\x07\x86"
			  "\xc6\x2c\x73\x88\xb0\x9d\x5f\x3e"
			  "\x5b\x78\xca\x0e\xab\x8a\xa3\xbb"
			  "\xd9\x1d\xc3\xe3\x05\xac\x76\xfb"
			  "\x58\x83\xda\x67\xfb\x21\x24\xa2"
			  "\xb1\xa7\xd7\x66\xa6\x8d\xa6\x93"
			  "\x97\xe2\xe3\xb8\xaa\x48\x85\xee"
			  "\x8c\xf6\x07\x95\x1f\xa6\x6c\x96"
			  "\x99\xc7\x5c\x8d\xd8\xb5\x68\x7b",
		.rlen	= 128 + 64,
	},
};

#define HMAC_SHA1_DES3_EDE_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha1_des3_ede_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x18"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
		  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
			  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 8 + 20 + 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
		  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
			  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
		  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
		  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
		  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
			  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
		  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
			  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
			  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19"
			  "\x67\x6d\xb1\xf5\xb8\x10\xdc\xc6"
			  "\x75\x86\x96\x6b\xb1\xc5\xe4\xcf"
			  "\xd1\x60\x91\xb3",
			  .rlen	= 128 + 20,
	},
};

#define HMAC_SHA224_DES3_EDE_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha224_des3_ede_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x18"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
			  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 8 + 24 + 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
		  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
		  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
			  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
			  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
			  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
		  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
			  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
		  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
		  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19"
			  "\x15\x24\x7f\x5a\x45\x4a\x66\xce"
			  "\x2b\x0b\x93\x99\x2f\x9d\x0c\x6c"
			  "\x56\x1f\xe1\xa6\x41\xb2\x4c\xd0",
			  .rlen	= 128 + 24,
	},
};

#define HMAC_SHA256_DES3_EDE_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha256_des3_ede_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x18"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
			  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 8 + 32 + 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
			  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
			  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
			  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
			  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
			  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
			  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
			  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
			  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19"
			  "\x73\xb0\xea\x9f\xe8\x18\x80\xd6"
			  "\x56\x38\x44\xc0\xdb\xe3\x4f\x71"
			  "\xf7\xce\xd1\xd3\xf8\xbd\x3e\x4f"
			  "\xca\x43\x95\xdf\x80\x61\x81\xa9",
		.rlen	= 128 + 32,
	},
};

#define HMAC_SHA384_DES3_EDE_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha384_des3_ede_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x18"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
			  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 8 + 48 + 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
			  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
			  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
			  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
			  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
			  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
			  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
			  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
			  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19"
			  "\x6d\x77\xfc\x80\x9d\x8a\x9c\xb7"
		  "\x70\xe7\x93\xbf\x73\xe6\x9f\x83"
			  "\x99\x62\x23\xe6\x5b\xd0\xda\x18"
			  "\xa4\x32\x8a\x0b\x46\xd7\xf0\x39"
			  "\x36\x5d\x13\x2f\x86\x10\x78\xd6"
			  "\xd6\xbe\x5c\xb9\x15\x89\xf9\x1b",
		.rlen	= 128 + 48,
	},
};

#define HMAC_SHA512_DES3_EDE_CBC_ENC_TEST_VEC	1

static struct aead_testvec hmac_sha512_des3_ede_cbc_enc_tv_temp[] = {
	{ /*Generated with cryptopp*/
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		/* rta length */
			  "\x01\x00"		/* rta type */
#else
		.key    = "\x00\x08"		/* rta length */
			  "\x00\x01"		/* rta type */
#endif
			  "\x00\x00\x00\x18"	/* enc key length */
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
		  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 8 + 64 + 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01"
			  "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.alen   = 16,
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
		  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
			  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
			  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
			  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
			  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
			  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
			  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
			  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
			  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19"
			  "\x41\xb5\x1f\xbb\xbd\x4e\xb8\x32"
			  "\x22\x86\x4e\x57\x1b\x2a\xd8\x6e"
			  "\xa9\xfb\xc8\xf3\xbf\x2d\xae\x2b"
			  "\x3b\xbc\x41\xe8\x38\xbb\xf1\x60"
			  "\x4c\x68\xa9\x4e\x8c\x73\xa7\xc0"
			  "\x2a\x74\xd4\x65\x12\xcb\x55\xf2"
			  "\xd5\x02\x6d\xe6\xaf\xc9\x2f\xf2"
			  "\x57\xaa\x85\xf7\xf3\x6a\xcb\xdb",
		.rlen	= 128 + 64,
	},
};

static struct cipher_testvec aes_xts_enc_tv_template[] = {
	/* http://grouper.ieee.org/groups/1619/email/pdf00086.pdf */
	{ /* XTS-AES 1 */
		.key    = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen   = 32,
		.fips_skip = 1,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen   = 32,
		.result = "\x91\x7c\xf6\x9e\xbd\x68\xb2\xec"
			  "\x9b\x9f\xe9\xa3\xea\xdd\xa6\x92"
			  "\xcd\x43\xd2\xf5\x95\x98\xed\x85"
			  "\x8c\x02\xc2\x65\x2f\xbf\x92\x2e",
		.rlen   = 32,
	}, { /* XTS-AES 2 */
		.key    = "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen   = 32,
		.iv     = "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen   = 32,
		.result = "\xc4\x54\x18\x5e\x6a\x16\x93\x6e"
			  "\x39\x33\x40\x38\xac\xef\x83\x8b"
			  "\xfb\x18\x6f\xff\x74\x80\xad\xc4"
			  "\x28\x93\x82\xec\xd6\xd3\x94\xf0",
		.rlen   = 32,
	}, { /* XTS-AES 3 */
		.key    = "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen   = 32,
		.iv     = "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen   = 32,
		.result = "\xaf\x85\x33\x6b\x59\x7a\xfc\x1a"
			  "\x90\x0b\x2e\xb2\x1e\xc9\x49\xd2"
			  "\x92\xdf\x4c\x04\x7e\x0b\x21\x53"
			  "\x21\x86\xa5\x97\x1a\x22\x7a\x89",
		.rlen   = 32,
	}, { /* XTS-AES 4 */
		.key    = "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen   = 512,
		.result = "\x27\xa7\x47\x9b\xef\xa1\xd4\x76"
			  "\x48\x9f\x30\x8c\xd4\xcf\xa6\xe2"
			  "\xa9\x6e\x4b\xbe\x32\x08\xff\x25"
			  "\x28\x7d\xd3\x81\x96\x16\xe8\x9c"
			  "\xc7\x8c\xf7\xf5\xe5\x43\x44\x5f"
			  "\x83\x33\xd8\xfa\x7f\x56\x00\x00"
			  "\x05\x27\x9f\xa5\xd8\xb5\xe4\xad"
			  "\x40\xe7\x36\xdd\xb4\xd3\x54\x12"
			  "\x32\x80\x63\xfd\x2a\xab\x53\xe5"
			  "\xea\x1e\x0a\x9f\x33\x25\x00\xa5"
			  "\xdf\x94\x87\xd0\x7a\x5c\x92\xcc"
			  "\x51\x2c\x88\x66\xc7\xe8\x60\xce"
			  "\x93\xfd\xf1\x66\xa2\x49\x12\xb4"
			  "\x22\x97\x61\x46\xae\x20\xce\x84"
			  "\x6b\xb7\xdc\x9b\xa9\x4a\x76\x7a"
			  "\xae\xf2\x0c\x0d\x61\xad\x02\x65"
			  "\x5e\xa9\x2d\xc4\xc4\xe4\x1a\x89"
			  "\x52\xc6\x51\xd3\x31\x74\xbe\x51"
			  "\xa1\x0c\x42\x11\x10\xe6\xd8\x15"
			  "\x88\xed\xe8\x21\x03\xa2\x52\xd8"
			  "\xa7\x50\xe8\x76\x8d\xef\xff\xed"
			  "\x91\x22\x81\x0a\xae\xb9\x9f\x91"
			  "\x72\xaf\x82\xb6\x04\xdc\x4b\x8e"
			  "\x51\xbc\xb0\x82\x35\xa6\xf4\x34"
			  "\x13\x32\xe4\xca\x60\x48\x2a\x4b"
			  "\xa1\xa0\x3b\x3e\x65\x00\x8f\xc5"
			  "\xda\x76\xb7\x0b\xf1\x69\x0d\xb4"
			  "\xea\xe2\x9c\x5f\x1b\xad\xd0\x3c"
			  "\x5c\xcf\x2a\x55\xd7\x05\xdd\xcd"
			  "\x86\xd4\x49\x51\x1c\xeb\x7e\xc3"
			  "\x0b\xf1\x2b\x1f\xa3\x5b\x91\x3f"
			  "\x9f\x74\x7a\x8a\xfd\x1b\x13\x0e"
			  "\x94\xbf\xf9\x4e\xff\xd0\x1a\x91"
			  "\x73\x5c\xa1\x72\x6a\xcd\x0b\x19"
			  "\x7c\x4e\x5b\x03\x39\x36\x97\xe1"
			  "\x26\x82\x6f\xb6\xbb\xde\x8e\xcc"
			  "\x1e\x08\x29\x85\x16\xe2\xc9\xed"
			  "\x03\xff\x3c\x1b\x78\x60\xf6\xde"
			  "\x76\xd4\xce\xcd\x94\xc8\x11\x98"
			  "\x55\xef\x52\x97\xca\x67\xe9\xf3"
			  "\xe7\xff\x72\xb1\xe9\x97\x85\xca"
			  "\x0a\x7e\x77\x20\xc5\xb3\x6d\xc6"
			  "\xd7\x2c\xac\x95\x74\xc8\xcb\xbc"
			  "\x2f\x80\x1e\x23\xe5\x6f\xd3\x44"
			  "\xb0\x7f\x22\x15\x4b\xeb\xa0\xf0"
			  "\x8c\xe8\x89\x1e\x64\x3e\xd9\x95"
			  "\xc9\x4d\x9a\x69\xc9\xf1\xb5\xf4"
			  "\x99\x02\x7a\x78\x57\x2a\xee\xbd"
			  "\x74\xd2\x0c\xc3\x98\x81\xc2\x13"
			  "\xee\x77\x0b\x10\x10\xe4\xbe\xa7"
			  "\x18\x84\x69\x77\xae\x11\x9f\x7a"
			  "\x02\x3a\xb5\x8c\xca\x0a\xd7\x52"
			  "\xaf\xe6\x56\xbb\x3c\x17\x25\x6a"
			  "\x9f\x6e\x9b\xf1\x9f\xdd\x5a\x38"
			  "\xfc\x82\xbb\xe8\x72\xc5\x53\x9e"
			  "\xdb\x60\x9e\xf4\xf7\x9c\x20\x3e"
			  "\xbb\x14\x0f\x2e\x58\x3c\xb2\xad"
			  "\x15\xb4\xaa\x5b\x65\x50\x16\xa8"
			  "\x44\x92\x77\xdb\xd4\x77\xef\x2c"
			  "\x8d\x6c\x01\x7d\xb7\x38\xb1\x8d"
			  "\xeb\x4a\x42\x7d\x19\x23\xce\x3f"
			  "\xf2\x62\x73\x57\x79\xa4\x18\xf2"
			  "\x0a\x28\x2d\xf9\x20\x14\x7b\xea"
			  "\xbe\x42\x1e\xe5\x31\x9d\x05\x68",
		.rlen   = 512,
	}, { /* XTS-AES 10, XTS-AES-256, data unit 512 bytes */
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\x1c\x3b\x3a\x10\x2f\x77\x03\x86"
			  "\xe4\x83\x6c\x99\xe3\x70\xcf\x9b"
			  "\xea\x00\x80\x3f\x5e\x48\x23\x57"
			  "\xa4\xae\x12\xd4\x14\xa3\xe6\x3b"
			  "\x5d\x31\xe2\x76\xf8\xfe\x4a\x8d"
			  "\x66\xb3\x17\xf9\xac\x68\x3f\x44"
			  "\x68\x0a\x86\xac\x35\xad\xfc\x33"
			  "\x45\xbe\xfe\xcb\x4b\xb1\x88\xfd"
			  "\x57\x76\x92\x6c\x49\xa3\x09\x5e"
			  "\xb1\x08\xfd\x10\x98\xba\xec\x70"
			  "\xaa\xa6\x69\x99\xa7\x2a\x82\xf2"
			  "\x7d\x84\x8b\x21\xd4\xa7\x41\xb0"
			  "\xc5\xcd\x4d\x5f\xff\x9d\xac\x89"
			  "\xae\xba\x12\x29\x61\xd0\x3a\x75"
			  "\x71\x23\xe9\x87\x0f\x8a\xcf\x10"
			  "\x00\x02\x08\x87\x89\x14\x29\xca"
			  "\x2a\x3e\x7a\x7d\x7d\xf7\xb1\x03"
			  "\x55\x16\x5c\x8b\x9a\x6d\x0a\x7d"
			  "\xe8\xb0\x62\xc4\x50\x0d\xc4\xcd"
			  "\x12\x0c\x0f\x74\x18\xda\xe3\xd0"
			  "\xb5\x78\x1c\x34\x80\x3f\xa7\x54"
			  "\x21\xc7\x90\xdf\xe1\xde\x18\x34"
			  "\xf2\x80\xd7\x66\x7b\x32\x7f\x6c"
			  "\x8c\xd7\x55\x7e\x12\xac\x3a\x0f"
			  "\x93\xec\x05\xc5\x2e\x04\x93\xef"
			  "\x31\xa1\x2d\x3d\x92\x60\xf7\x9a"
			  "\x28\x9d\x6a\x37\x9b\xc7\x0c\x50"
			  "\x84\x14\x73\xd1\xa8\xcc\x81\xec"
			  "\x58\x3e\x96\x45\xe0\x7b\x8d\x96"
			  "\x70\x65\x5b\xa5\xbb\xcf\xec\xc6"
			  "\xdc\x39\x66\x38\x0a\xd8\xfe\xcb"
			  "\x17\xb6\xba\x02\x46\x9a\x02\x0a"
			  "\x84\xe1\x8e\x8f\x84\x25\x20\x70"
			  "\xc1\x3e\x9f\x1f\x28\x9b\xe5\x4f"
			  "\xbc\x48\x14\x57\x77\x8f\x61\x60"
			  "\x15\xe1\x32\x7a\x02\xb1\x40\xf1"
			  "\x50\x5e\xb3\x09\x32\x6d\x68\x37"
			  "\x8f\x83\x74\x59\x5c\x84\x9d\x84"
			  "\xf4\xc3\x33\xec\x44\x23\x88\x51"
			  "\x43\xcb\x47\xbd\x71\xc5\xed\xae"
			  "\x9b\xe6\x9a\x2f\xfe\xce\xb1\xbe"
			  "\xc9\xde\x24\x4f\xbe\x15\x99\x2b"
			  "\x11\xb7\x7c\x04\x0f\x12\xbd\x8f"
			  "\x6a\x97\x5a\x44\xa0\xf9\x0c\x29"
			  "\xa9\xab\xc3\xd4\xd8\x93\x92\x72"
			  "\x84\xc5\x87\x54\xcc\xe2\x94\x52"
			  "\x9f\x86\x14\xdc\xd2\xab\xa9\x91"
			  "\x92\x5f\xed\xc4\xae\x74\xff\xac"
			  "\x6e\x33\x3b\x93\xeb\x4a\xff\x04"
			  "\x79\xda\x9a\x41\x0e\x44\x50\xe0"
			  "\xdd\x7a\xe4\xc6\xe2\x91\x09\x00"
			  "\x57\x5d\xa4\x01\xfc\x07\x05\x9f"
			  "\x64\x5e\x8b\x7e\x9b\xfd\xef\x33"
			  "\x94\x30\x54\xff\x84\x01\x14\x93"
			  "\xc2\x7b\x34\x29\xea\xed\xb4\xed"
			  "\x53\x76\x44\x1a\x77\xed\x43\x85"
			  "\x1a\xd7\x7f\x16\xf5\x41\xdf\xd2"
			  "\x69\xd5\x0d\x6a\x5f\x14\xfb\x0a"
			  "\xab\x1c\xbb\x4c\x15\x50\xbe\x97"
			  "\xf7\xab\x40\x66\x19\x3c\x4c\xaa"
			  "\x77\x3d\xad\x38\x01\x4b\xd2\x09"
			  "\x2f\xa7\x55\xc8\x24\xbb\x5e\x54"
			  "\xc4\xf3\x6f\xfd\xa9\xfc\xea\x70"
			  "\xb9\xc6\xe6\x93\xe1\x48\xc1\x51",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 512 - 20, 4, 16 },
	}
};

static struct cipher_testvec aes_xts_dec_tv_template[] = {
	/* http://grouper.ieee.org/groups/1619/email/pdf00086.pdf */
	{ /* XTS-AES 1 */
		.key    = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen   = 32,
		.fips_skip = 1,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input = "\x91\x7c\xf6\x9e\xbd\x68\xb2\xec"
			 "\x9b\x9f\xe9\xa3\xea\xdd\xa6\x92"
			 "\xcd\x43\xd2\xf5\x95\x98\xed\x85"
			 "\x8c\x02\xc2\x65\x2f\xbf\x92\x2e",
		.ilen   = 32,
		.result  = "\x00\x00\x00\x00\x00\x00\x00\x00"
			   "\x00\x00\x00\x00\x00\x00\x00\x00"
			   "\x00\x00\x00\x00\x00\x00\x00\x00"
			   "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen   = 32,
	}, { /* XTS-AES 2 */
		.key    = "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen   = 32,
		.iv     = "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\xc4\x54\x18\x5e\x6a\x16\x93\x6e"
			  "\x39\x33\x40\x38\xac\xef\x83\x8b"
			  "\xfb\x18\x6f\xff\x74\x80\xad\xc4"
			  "\x28\x93\x82\xec\xd6\xd3\x94\xf0",
		.ilen   = 32,
		.result = "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen   = 32,
	}, { /* XTS-AES 3 */
		.key    = "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen   = 32,
		.iv     = "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input = "\xaf\x85\x33\x6b\x59\x7a\xfc\x1a"
			  "\x90\x0b\x2e\xb2\x1e\xc9\x49\xd2"
			  "\x92\xdf\x4c\x04\x7e\x0b\x21\x53"
			  "\x21\x86\xa5\x97\x1a\x22\x7a\x89",
		.ilen   = 32,
		.result  = "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen   = 32,
	}, { /* XTS-AES 4 */
		.key    = "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x27\xa7\x47\x9b\xef\xa1\xd4\x76"
			  "\x48\x9f\x30\x8c\xd4\xcf\xa6\xe2"
			  "\xa9\x6e\x4b\xbe\x32\x08\xff\x25"
			  "\x28\x7d\xd3\x81\x96\x16\xe8\x9c"
			  "\xc7\x8c\xf7\xf5\xe5\x43\x44\x5f"
			  "\x83\x33\xd8\xfa\x7f\x56\x00\x00"
			  "\x05\x27\x9f\xa5\xd8\xb5\xe4\xad"
			  "\x40\xe7\x36\xdd\xb4\xd3\x54\x12"
			  "\x32\x80\x63\xfd\x2a\xab\x53\xe5"
			  "\xea\x1e\x0a\x9f\x33\x25\x00\xa5"
			  "\xdf\x94\x87\xd0\x7a\x5c\x92\xcc"
			  "\x51\x2c\x88\x66\xc7\xe8\x60\xce"
			  "\x93\xfd\xf1\x66\xa2\x49\x12\xb4"
			  "\x22\x97\x61\x46\xae\x20\xce\x84"
			  "\x6b\xb7\xdc\x9b\xa9\x4a\x76\x7a"
			  "\xae\xf2\x0c\x0d\x61\xad\x02\x65"
			  "\x5e\xa9\x2d\xc4\xc4\xe4\x1a\x89"
			  "\x52\xc6\x51\xd3\x31\x74\xbe\x51"
			  "\xa1\x0c\x42\x11\x10\xe6\xd8\x15"
			  "\x88\xed\xe8\x21\x03\xa2\x52\xd8"
			  "\xa7\x50\xe8\x76\x8d\xef\xff\xed"
			  "\x91\x22\x81\x0a\xae\xb9\x9f\x91"
			  "\x72\xaf\x82\xb6\x04\xdc\x4b\x8e"
			  "\x51\xbc\xb0\x82\x35\xa6\xf4\x34"
			  "\x13\x32\xe4\xca\x60\x48\x2a\x4b"
			  "\xa1\xa0\x3b\x3e\x65\x00\x8f\xc5"
			  "\xda\x76\xb7\x0b\xf1\x69\x0d\xb4"
			  "\xea\xe2\x9c\x5f\x1b\xad\xd0\x3c"
			  "\x5c\xcf\x2a\x55\xd7\x05\xdd\xcd"
			  "\x86\xd4\x49\x51\x1c\xeb\x7e\xc3"
			  "\x0b\xf1\x2b\x1f\xa3\x5b\x91\x3f"
			  "\x9f\x74\x7a\x8a\xfd\x1b\x13\x0e"
			  "\x94\xbf\xf9\x4e\xff\xd0\x1a\x91"
			  "\x73\x5c\xa1\x72\x6a\xcd\x0b\x19"
			  "\x7c\x4e\x5b\x03\x39\x36\x97\xe1"
			  "\x26\x82\x6f\xb6\xbb\xde\x8e\xcc"
			  "\x1e\x08\x29\x85\x16\xe2\xc9\xed"
			  "\x03\xff\x3c\x1b\x78\x60\xf6\xde"
			  "\x76\xd4\xce\xcd\x94\xc8\x11\x98"
			  "\x55\xef\x52\x97\xca\x67\xe9\xf3"
			  "\xe7\xff\x72\xb1\xe9\x97\x85\xca"
			  "\x0a\x7e\x77\x20\xc5\xb3\x6d\xc6"
			  "\xd7\x2c\xac\x95\x74\xc8\xcb\xbc"
			  "\x2f\x80\x1e\x23\xe5\x6f\xd3\x44"
			  "\xb0\x7f\x22\x15\x4b\xeb\xa0\xf0"
			  "\x8c\xe8\x89\x1e\x64\x3e\xd9\x95"
			  "\xc9\x4d\x9a\x69\xc9\xf1\xb5\xf4"
			  "\x99\x02\x7a\x78\x57\x2a\xee\xbd"
			  "\x74\xd2\x0c\xc3\x98\x81\xc2\x13"
			  "\xee\x77\x0b\x10\x10\xe4\xbe\xa7"
			  "\x18\x84\x69\x77\xae\x11\x9f\x7a"
			  "\x02\x3a\xb5\x8c\xca\x0a\xd7\x52"
			  "\xaf\xe6\x56\xbb\x3c\x17\x25\x6a"
			  "\x9f\x6e\x9b\xf1\x9f\xdd\x5a\x38"
			  "\xfc\x82\xbb\xe8\x72\xc5\x53\x9e"
			  "\xdb\x60\x9e\xf4\xf7\x9c\x20\x3e"
			  "\xbb\x14\x0f\x2e\x58\x3c\xb2\xad"
			  "\x15\xb4\xaa\x5b\x65\x50\x16\xa8"
			  "\x44\x92\x77\xdb\xd4\x77\xef\x2c"
			  "\x8d\x6c\x01\x7d\xb7\x38\xb1\x8d"
			  "\xeb\x4a\x42\x7d\x19\x23\xce\x3f"
			  "\xf2\x62\x73\x57\x79\xa4\x18\xf2"
			  "\x0a\x28\x2d\xf9\x20\x14\x7b\xea"
			  "\xbe\x42\x1e\xe5\x31\x9d\x05\x68",
		.ilen   = 512,
		.result = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen   = 512,
	}, { /* XTS-AES 10, XTS-AES-256, data unit 512 bytes */
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x1c\x3b\x3a\x10\x2f\x77\x03\x86"
			  "\xe4\x83\x6c\x99\xe3\x70\xcf\x9b"
			  "\xea\x00\x80\x3f\x5e\x48\x23\x57"
			  "\xa4\xae\x12\xd4\x14\xa3\xe6\x3b"
			  "\x5d\x31\xe2\x76\xf8\xfe\x4a\x8d"
			  "\x66\xb3\x17\xf9\xac\x68\x3f\x44"
			  "\x68\x0a\x86\xac\x35\xad\xfc\x33"
			  "\x45\xbe\xfe\xcb\x4b\xb1\x88\xfd"
			  "\x57\x76\x92\x6c\x49\xa3\x09\x5e"
			  "\xb1\x08\xfd\x10\x98\xba\xec\x70"
			  "\xaa\xa6\x69\x99\xa7\x2a\x82\xf2"
			  "\x7d\x84\x8b\x21\xd4\xa7\x41\xb0"
			  "\xc5\xcd\x4d\x5f\xff\x9d\xac\x89"
			  "\xae\xba\x12\x29\x61\xd0\x3a\x75"
			  "\x71\x23\xe9\x87\x0f\x8a\xcf\x10"
			  "\x00\x02\x08\x87\x89\x14\x29\xca"
			  "\x2a\x3e\x7a\x7d\x7d\xf7\xb1\x03"
			  "\x55\x16\x5c\x8b\x9a\x6d\x0a\x7d"
			  "\xe8\xb0\x62\xc4\x50\x0d\xc4\xcd"
			  "\x12\x0c\x0f\x74\x18\xda\xe3\xd0"
			  "\xb5\x78\x1c\x34\x80\x3f\xa7\x54"
			  "\x21\xc7\x90\xdf\xe1\xde\x18\x34"
			  "\xf2\x80\xd7\x66\x7b\x32\x7f\x6c"
			  "\x8c\xd7\x55\x7e\x12\xac\x3a\x0f"
			  "\x93\xec\x05\xc5\x2e\x04\x93\xef"
			  "\x31\xa1\x2d\x3d\x92\x60\xf7\x9a"
			  "\x28\x9d\x6a\x37\x9b\xc7\x0c\x50"
			  "\x84\x14\x73\xd1\xa8\xcc\x81\xec"
			  "\x58\x3e\x96\x45\xe0\x7b\x8d\x96"
			  "\x70\x65\x5b\xa5\xbb\xcf\xec\xc6"
			  "\xdc\x39\x66\x38\x0a\xd8\xfe\xcb"
			  "\x17\xb6\xba\x02\x46\x9a\x02\x0a"
			  "\x84\xe1\x8e\x8f\x84\x25\x20\x70"
			  "\xc1\x3e\x9f\x1f\x28\x9b\xe5\x4f"
			  "\xbc\x48\x14\x57\x77\x8f\x61\x60"
			  "\x15\xe1\x32\x7a\x02\xb1\x40\xf1"
			  "\x50\x5e\xb3\x09\x32\x6d\x68\x37"
			  "\x8f\x83\x74\x59\x5c\x84\x9d\x84"
			  "\xf4\xc3\x33\xec\x44\x23\x88\x51"
			  "\x43\xcb\x47\xbd\x71\xc5\xed\xae"
			  "\x9b\xe6\x9a\x2f\xfe\xce\xb1\xbe"
			  "\xc9\xde\x24\x4f\xbe\x15\x99\x2b"
			  "\x11\xb7\x7c\x04\x0f\x12\xbd\x8f"
			  "\x6a\x97\x5a\x44\xa0\xf9\x0c\x29"
			  "\xa9\xab\xc3\xd4\xd8\x93\x92\x72"
			  "\x84\xc5\x87\x54\xcc\xe2\x94\x52"
			  "\x9f\x86\x14\xdc\xd2\xab\xa9\x91"
			  "\x92\x5f\xed\xc4\xae\x74\xff\xac"
			  "\x6e\x33\x3b\x93\xeb\x4a\xff\x04"
			  "\x79\xda\x9a\x41\x0e\x44\x50\xe0"
			  "\xdd\x7a\xe4\xc6\xe2\x91\x09\x00"
			  "\x57\x5d\xa4\x01\xfc\x07\x05\x9f"
			  "\x64\x5e\x8b\x7e\x9b\xfd\xef\x33"
			  "\x94\x30\x54\xff\x84\x01\x14\x93"
			  "\xc2\x7b\x34\x29\xea\xed\xb4\xed"
			  "\x53\x76\x44\x1a\x77\xed\x43\x85"
			  "\x1a\xd7\x7f\x16\xf5\x41\xdf\xd2"
			  "\x69\xd5\x0d\x6a\x5f\x14\xfb\x0a"
			  "\xab\x1c\xbb\x4c\x15\x50\xbe\x97"
			  "\xf7\xab\x40\x66\x19\x3c\x4c\xaa"
			  "\x77\x3d\xad\x38\x01\x4b\xd2\x09"
			  "\x2f\xa7\x55\xc8\x24\xbb\x5e\x54"
			  "\xc4\xf3\x6f\xfd\xa9\xfc\xea\x70"
			  "\xb9\xc6\xe6\x93\xe1\x48\xc1\x51",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 512 - 20, 4, 16 },
	}
};


static struct cipher_testvec aes_ctr_enc_tv_template[] = {
	{ /* From NIST Special Publication 800-38A, Appendix F.5 */
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x87\x4d\x61\x91\xb6\x20\xe3\x26"
			  "\x1b\xef\x68\x64\x99\x0d\xb6\xce"
			  "\x98\x06\xf6\x6b\x79\x70\xfd\xff"
			  "\x86\x17\x18\x7b\xb9\xff\xfd\xff"
			  "\x5a\xe4\xdf\x3e\xdb\xd5\xd3\x5e"
			  "\x5b\x4f\x09\x02\x0d\xb0\x3e\xab"
			  "\x1e\x03\x1d\xda\x2f\xbe\x03\xd1"
			  "\x79\x21\x70\xa0\xf3\x00\x9c\xee",
		.rlen	= 64,
	}, {
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x1a\xbc\x93\x24\x17\x52\x1c\xa2"
			  "\x4f\x2b\x04\x59\xfe\x7e\x6e\x0b"
			  "\x09\x03\x39\xec\x0a\xa6\xfa\xef"
			  "\xd5\xcc\xc2\xc6\xf4\xce\x8e\x94"
			  "\x1e\x36\xb2\x6b\xd1\xeb\xc6\x70"
			  "\xd1\xbd\x1d\x66\x56\x20\xab\xf7"
			  "\x4f\x78\xa7\xf6\xd2\x98\x09\x58"
			  "\x5a\x97\xda\xec\x58\xc6\xb0\x50",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x60\x1e\xc3\x13\x77\x57\x89\xa5"
			  "\xb7\xa7\xf5\x04\xbb\xf3\xd2\x28"
			  "\xf4\x43\xe3\xca\x4d\x62\xb5\x9a"
			  "\xca\x84\xe9\x90\xca\xca\xf5\xc5"
			  "\x2b\x09\x30\xda\xa2\x3d\xe9\x4c"
			  "\xe8\x70\x17\xba\x2d\x84\x98\x8d"
			  "\xdf\xc9\xc5\x8d\xb6\x7a\xad\xa6"
			  "\x13\xc2\xdd\x08\x45\x79\x41\xa6",
		.rlen	= 64,
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\x04\xF3\xD3\x88\x17\xEF\xDC\xEF"
			  "\x8B\x04\xF8\x3A\x66\x8D\x1A\x53"
			  "\x57\x1F\x4B\x23\xE4\xA0\xAF\xF9"
			  "\x69\x95\x35\x98\x8D\x4D\x8C\xC1"
			  "\xF0\xB2\x7F\x80\xBB\x54\x28\xA2"
			  "\x7A\x1B\x9F\x77\xEC\x0E\x6E\xDE"
			  "\xF0\xEC\xB8\xE4\x20\x62\xEE\xDB"
			  "\x5D\xF5\xDD\xE3\x54\xFC\xDD\xEB"
			  "\x6A\xEE\x65\xA1\x21\xD6\xD7\x81"
			  "\x47\x61\x12\x4D\xC2\x8C\xFA\x78"
			  "\x1F\x28\x02\x01\xC3\xFC\x1F\xEC"
			  "\x0F\x10\x4F\xB3\x12\x45\xC6\x3B"
			  "\x7E\x08\xF9\x5A\xD0\x5D\x73\x2D"
			  "\x58\xA4\xE5\xCB\x1C\xB4\xCE\x74"
			  "\x32\x41\x1F\x31\x9C\x08\xA2\x5D"
			  "\x67\xEB\x72\x1D\xF8\xE7\x70\x54"
			  "\x34\x4B\x31\x69\x84\x66\x96\x44"
			  "\x56\xCC\x1E\xD9\xE6\x13\x6A\xB9"
			  "\x2D\x0A\x05\x45\x2D\x90\xCC\xDF"
			  "\x16\x5C\x5F\x79\x34\x52\x54\xFE"
			  "\xFE\xCD\xAD\x04\x2E\xAD\x86\x06"
			  "\x1F\x37\xE8\x28\xBC\xD3\x8F\x5B"
			  "\x92\x66\x87\x3B\x8A\x0A\x1A\xCC"
			  "\x6E\xAB\x9F\x0B\xFA\x5C\xE6\xFD"
			  "\x3C\x98\x08\x12\xEC\xAA\x9E\x11"
			  "\xCA\xB2\x1F\xCE\x5E\x5B\xB2\x72"
			  "\x9C\xCC\x5D\xC5\xE0\x32\xC0\x56"
			  "\xD5\x45\x16\xD2\xAF\x13\x66\xF7"
			  "\x8C\x67\xAC\x79\xB2\xAF\x56\x27"
			  "\x3F\xCC\xFE\xCB\x1E\xC0\x75\xF1"
			  "\xA7\xC9\xC3\x1D\x8E\xDD\xF9\xD4"
			  "\x42\xC8\x21\x08\x16\xF7\x01\xD7"
			  "\xAC\x8E\x3F\x1D\x56\xC1\x06\xE4"
			  "\x9C\x62\xD6\xA5\x6A\x50\x44\xB3"
			  "\x35\x1C\x82\xB9\x10\xF9\x42\xA1"
			  "\xFC\x74\x9B\x44\x4F\x25\x02\xE3"
			  "\x08\xF5\xD4\x32\x39\x08\x11\xE8"
			  "\xD2\x6B\x50\x53\xD4\x08\xD1\x6B"
			  "\x3A\x4A\x68\x7B\x7C\xCD\x46\x5E"
			  "\x0D\x07\x19\xDB\x67\xD7\x98\x91"
			  "\xD7\x17\x10\x9B\x7B\x8A\x9B\x33"
			  "\xAE\xF3\x00\xA6\xD4\x15\xD9\xEA"
			  "\x85\x99\x22\xE8\x91\x38\x70\x83"
			  "\x93\x01\x24\x6C\xFA\x9A\xB9\x07"
			  "\xEA\x8D\x3B\xD9\x2A\x43\x59\x16"
			  "\x2F\x69\xEE\x84\x36\x44\x76\x98"
			  "\xF3\x04\x2A\x7C\x74\x3D\x29\x2B"
			  "\x0D\xAD\x8F\x44\x82\x9E\x57\x8D"
			  "\xAC\xED\x18\x1F\x50\xA4\xF5\x98"
			  "\x1F\xBD\x92\x91\x1B\x2D\xA6\xD6"
			  "\xD2\xE3\x02\xAA\x92\x3B\xC6\xB3"
			  "\x1B\x39\x72\xD5\x26\xCA\x04\xE0"
			  "\xFC\x58\x78\xBB\xB1\x3F\xA1\x9C"
			  "\x42\x24\x3E\x2E\x22\xBB\x4B\xBA"
			  "\xF4\x52\x0A\xE6\xAE\x47\xB4\x7D"
			  "\x1D\xA8\xBE\x81\x1A\x75\xDA\xAC"
			  "\xA6\x25\x1E\xEF\x3A\xC0\x6C\x63"
			  "\xEF\xDC\xC9\x79\x10\x26\xE8\x61"
			  "\x29\xFC\xA4\x05\xDF\x7D\x5C\x63"
			  "\x10\x09\x9B\x46\x9B\xF2\x2C\x2B"
			  "\xFA\x3A\x05\x4C\xFA\xD1\xFF\xFE"
			  "\xF1\x4C\xE5\xB2\x91\x64\x0C\x51",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12"
			  "\x7B\xE4\x4D",
		.ilen	= 499,
		.result	= "\xDA\x4E\x3F\xBC\xE8\xB6\x3A\xA2"
			  "\xD5\x4D\x84\x4A\xA9\x0C\xE1\xA5"
			  "\xB8\x73\xBC\xF9\xBB\x59\x2F\x44"
			  "\x8B\xAB\x82\x6C\xB4\x32\x9A\xDE"
			  "\x5A\x0B\xDB\x7A\x6B\xF2\x38\x9F"
			  "\x06\xF7\xF7\xFF\xFF\xC0\x8A\x2E"
			  "\x76\xEA\x06\x32\x23\xF3\x59\x2E"
			  "\x75\xDE\x71\x86\x3C\x98\x23\x44"
			  "\x5B\xF2\xFA\x6A\x00\xBB\xC1\xAD"
			  "\x58\xBD\x3E\x6F\x2E\xB4\x19\x04"
			  "\x70\x8B\x92\x55\x23\xE9\x6A\x3A"
			  "\x78\x7A\x1B\x10\x85\x52\x9C\x12"
			  "\xE4\x55\x81\x21\xCE\x53\xD0\x3B"
			  "\x63\x77\x2C\x74\xD1\xF5\x60\xF3"
			  "\xA1\xDE\x44\x3C\x8F\x4D\x2F\xDD"
			  "\x8A\xFE\x3C\x42\x8E\xD3\xF2\x8E"
			  "\xA8\x28\x69\x65\x31\xE1\x45\x83"
			  "\xE4\x49\xC4\x9C\xA7\x28\xAA\x21"
			  "\xCD\x5D\x0F\x15\xB7\x93\x07\x26"
			  "\xB0\x65\x6D\x91\x90\x23\x7A\xC6"
			  "\xDB\x68\xB0\xA1\x8E\xA4\x76\x4E"
			  "\xC6\x91\x83\x20\x92\x4D\x63\x7A"
			  "\x45\x18\x18\x74\x19\xAD\x71\x01"
			  "\x6B\x23\xAD\x9D\x4E\xE4\x6E\x46"
			  "\xC9\x73\x7A\xF9\x02\x95\xF4\x07"
			  "\x0E\x7A\xA6\xC5\xAE\xFA\x15\x2C"
			  "\x51\x71\xF1\xDC\x22\xB6\xAC\xD8"
			  "\x19\x24\x44\xBC\x0C\xFB\x3C\x2D"
			  "\xB1\x50\x47\x15\x0E\xDB\xB6\xD7"
			  "\xE8\x61\xE5\x95\x52\x1E\x3E\x49"
			  "\x70\xE9\x66\x04\x4C\xE1\xAF\xBD"
			  "\xDD\x15\x3B\x20\x59\x24\xFF\xB0"
			  "\x39\xAA\xE7\xBF\x23\xA3\x6E\xD5"
			  "\x15\xF0\x61\x4F\xAE\x89\x10\x58"
			  "\x5A\x33\x95\x52\x2A\xB5\x77\x9C"
			  "\xA5\x43\x80\x40\x27\x2D\xAE\xD9"
			  "\x3F\xE0\x80\x94\x78\x79\xCB\x7E"
			  "\xAD\x12\x44\x4C\xEC\x27\xB0\xEE"
			  "\x0B\x05\x2A\x82\x99\x58\xBB\x7A"
			  "\x8D\x6D\x9D\x8E\xE2\x8E\xE7\x93"
			  "\x2F\xB3\x09\x8D\x06\xD5\xEE\x70"
			  "\x16\xAE\x35\xC5\x52\x0F\x46\x1F"
			  "\x71\xF9\x5E\xF2\x67\xDC\x98\x2F"
			  "\xA3\x23\xAA\xD5\xD0\x49\xF4\xA6"
			  "\xF6\xB8\x32\xCD\xD6\x85\x73\x60"
			  "\x59\x20\xE7\x55\x0E\x91\xE2\x0C"
			  "\x3F\x1C\xEB\x3D\xDF\x52\x64\xF2"
			  "\x7D\x8B\x5D\x63\x16\xB9\xB2\x5D"
			  "\x5E\xAB\xB2\x97\xAB\x78\x44\xE7"
			  "\xC6\x72\x20\xC5\x90\x9B\xDC\x5D"
			  "\xB0\xEF\x44\xEF\x87\x31\x8D\xF4"
			  "\xFB\x81\x5D\xF7\x96\x96\xD4\x50"
			  "\x89\xA7\xF6\xB9\x67\x76\x40\x9E"
			  "\x9D\x40\xD5\x2C\x30\xB8\x01\x8F"
			  "\xE4\x7B\x71\x48\xA9\xA0\xA0\x1D"
			  "\x87\x52\xA4\x91\xA9\xD7\xA9\x51"
			  "\xD9\x59\xF7\xCC\x63\x22\xC1\x8D"
			  "\x84\x7B\xD8\x22\x32\x5C\x6F\x1D"
			  "\x6E\x9F\xFA\xDD\x49\x40\xDC\x37"
			  "\x14\x8C\xE1\x80\x1B\xDD\x36\x2A"
			  "\xD0\xE9\x54\x99\x5D\xBA\x3B\x11"
			  "\xD8\xFE\xC9\x5B\x5C\x25\xE5\x76"
			  "\xFB\xF2\x3F",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec aes_ctr_dec_tv_template[] = {
	{ /* From NIST Special Publication 800-38A, Appendix F.5 */
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x87\x4d\x61\x91\xb6\x20\xe3\x26"
			  "\x1b\xef\x68\x64\x99\x0d\xb6\xce"
			  "\x98\x06\xf6\x6b\x79\x70\xfd\xff"
			  "\x86\x17\x18\x7b\xb9\xff\xfd\xff"
			  "\x5a\xe4\xdf\x3e\xdb\xd5\xd3\x5e"
			  "\x5b\x4f\x09\x02\x0d\xb0\x3e\xab"
			  "\x1e\x03\x1d\xda\x2f\xbe\x03\xd1"
			  "\x79\x21\x70\xa0\xf3\x00\x9c\xee",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x1a\xbc\x93\x24\x17\x52\x1c\xa2"
			  "\x4f\x2b\x04\x59\xfe\x7e\x6e\x0b"
			  "\x09\x03\x39\xec\x0a\xa6\xfa\xef"
			  "\xd5\xcc\xc2\xc6\xf4\xce\x8e\x94"
			  "\x1e\x36\xb2\x6b\xd1\xeb\xc6\x70"
			  "\xd1\xbd\x1d\x66\x56\x20\xab\xf7"
			  "\x4f\x78\xa7\xf6\xd2\x98\x09\x58"
			  "\x5a\x97\xda\xec\x58\xc6\xb0\x50",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x60\x1e\xc3\x13\x77\x57\x89\xa5"
			  "\xb7\xa7\xf5\x04\xbb\xf3\xd2\x28"
			  "\xf4\x43\xe3\xca\x4d\x62\xb5\x9a"
			  "\xca\x84\xe9\x90\xca\xca\xf5\xc5"
			  "\x2b\x09\x30\xda\xa2\x3d\xe9\x4c"
			  "\xe8\x70\x17\xba\x2d\x84\x98\x8d"
			  "\xdf\xc9\xc5\x8d\xb6\x7a\xad\xa6"
			  "\x13\xc2\xdd\x08\x45\x79\x41\xa6",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x04\xF3\xD3\x88\x17\xEF\xDC\xEF"
			  "\x8B\x04\xF8\x3A\x66\x8D\x1A\x53"
			  "\x57\x1F\x4B\x23\xE4\xA0\xAF\xF9"
			  "\x69\x95\x35\x98\x8D\x4D\x8C\xC1"
			  "\xF0\xB2\x7F\x80\xBB\x54\x28\xA2"
			  "\x7A\x1B\x9F\x77\xEC\x0E\x6E\xDE"
			  "\xF0\xEC\xB8\xE4\x20\x62\xEE\xDB"
			  "\x5D\xF5\xDD\xE3\x54\xFC\xDD\xEB"
			  "\x6A\xEE\x65\xA1\x21\xD6\xD7\x81"
			  "\x47\x61\x12\x4D\xC2\x8C\xFA\x78"
			  "\x1F\x28\x02\x01\xC3\xFC\x1F\xEC"
			  "\x0F\x10\x4F\xB3\x12\x45\xC6\x3B"
			  "\x7E\x08\xF9\x5A\xD0\x5D\x73\x2D"
			  "\x58\xA4\xE5\xCB\x1C\xB4\xCE\x74"
			  "\x32\x41\x1F\x31\x9C\x08\xA2\x5D"
			  "\x67\xEB\x72\x1D\xF8\xE7\x70\x54"
			  "\x34\x4B\x31\x69\x84\x66\x96\x44"
			  "\x56\xCC\x1E\xD9\xE6\x13\x6A\xB9"
			  "\x2D\x0A\x05\x45\x2D\x90\xCC\xDF"
			  "\x16\x5C\x5F\x79\x34\x52\x54\xFE"
			  "\xFE\xCD\xAD\x04\x2E\xAD\x86\x06"
			  "\x1F\x37\xE8\x28\xBC\xD3\x8F\x5B"
			  "\x92\x66\x87\x3B\x8A\x0A\x1A\xCC"
			  "\x6E\xAB\x9F\x0B\xFA\x5C\xE6\xFD"
			  "\x3C\x98\x08\x12\xEC\xAA\x9E\x11"
			  "\xCA\xB2\x1F\xCE\x5E\x5B\xB2\x72"
			  "\x9C\xCC\x5D\xC5\xE0\x32\xC0\x56"
			  "\xD5\x45\x16\xD2\xAF\x13\x66\xF7"
			  "\x8C\x67\xAC\x79\xB2\xAF\x56\x27"
			  "\x3F\xCC\xFE\xCB\x1E\xC0\x75\xF1"
			  "\xA7\xC9\xC3\x1D\x8E\xDD\xF9\xD4"
			  "\x42\xC8\x21\x08\x16\xF7\x01\xD7"
			  "\xAC\x8E\x3F\x1D\x56\xC1\x06\xE4"
			  "\x9C\x62\xD6\xA5\x6A\x50\x44\xB3"
			  "\x35\x1C\x82\xB9\x10\xF9\x42\xA1"
			  "\xFC\x74\x9B\x44\x4F\x25\x02\xE3"
			  "\x08\xF5\xD4\x32\x39\x08\x11\xE8"
			  "\xD2\x6B\x50\x53\xD4\x08\xD1\x6B"
			  "\x3A\x4A\x68\x7B\x7C\xCD\x46\x5E"
			  "\x0D\x07\x19\xDB\x67\xD7\x98\x91"
			  "\xD7\x17\x10\x9B\x7B\x8A\x9B\x33"
			  "\xAE\xF3\x00\xA6\xD4\x15\xD9\xEA"
			  "\x85\x99\x22\xE8\x91\x38\x70\x83"
			  "\x93\x01\x24\x6C\xFA\x9A\xB9\x07"
			  "\xEA\x8D\x3B\xD9\x2A\x43\x59\x16"
			  "\x2F\x69\xEE\x84\x36\x44\x76\x98"
			  "\xF3\x04\x2A\x7C\x74\x3D\x29\x2B"
			  "\x0D\xAD\x8F\x44\x82\x9E\x57\x8D"
			  "\xAC\xED\x18\x1F\x50\xA4\xF5\x98"
			  "\x1F\xBD\x92\x91\x1B\x2D\xA6\xD6"
			  "\xD2\xE3\x02\xAA\x92\x3B\xC6\xB3"
			  "\x1B\x39\x72\xD5\x26\xCA\x04\xE0"
			  "\xFC\x58\x78\xBB\xB1\x3F\xA1\x9C"
			  "\x42\x24\x3E\x2E\x22\xBB\x4B\xBA"
			  "\xF4\x52\x0A\xE6\xAE\x47\xB4\x7D"
			  "\x1D\xA8\xBE\x81\x1A\x75\xDA\xAC"
			  "\xA6\x25\x1E\xEF\x3A\xC0\x6C\x63"
			  "\xEF\xDC\xC9\x79\x10\x26\xE8\x61"
			  "\x29\xFC\xA4\x05\xDF\x7D\x5C\x63"
			  "\x10\x09\x9B\x46\x9B\xF2\x2C\x2B"
			  "\xFA\x3A\x05\x4C\xFA\xD1\xFF\xFE"
			  "\xF1\x4C\xE5\xB2\x91\x64\x0C\x51",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 3,
		.tap	= { 496 - 20, 4, 16 },
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\xDA\x4E\x3F\xBC\xE8\xB6\x3A\xA2"
			  "\xD5\x4D\x84\x4A\xA9\x0C\xE1\xA5"
			  "\xB8\x73\xBC\xF9\xBB\x59\x2F\x44"
			  "\x8B\xAB\x82\x6C\xB4\x32\x9A\xDE"
			  "\x5A\x0B\xDB\x7A\x6B\xF2\x38\x9F"
			  "\x06\xF7\xF7\xFF\xFF\xC0\x8A\x2E"
			  "\x76\xEA\x06\x32\x23\xF3\x59\x2E"
			  "\x75\xDE\x71\x86\x3C\x98\x23\x44"
			  "\x5B\xF2\xFA\x6A\x00\xBB\xC1\xAD"
			  "\x58\xBD\x3E\x6F\x2E\xB4\x19\x04"
			  "\x70\x8B\x92\x55\x23\xE9\x6A\x3A"
			  "\x78\x7A\x1B\x10\x85\x52\x9C\x12"
			  "\xE4\x55\x81\x21\xCE\x53\xD0\x3B"
			  "\x63\x77\x2C\x74\xD1\xF5\x60\xF3"
			  "\xA1\xDE\x44\x3C\x8F\x4D\x2F\xDD"
			  "\x8A\xFE\x3C\x42\x8E\xD3\xF2\x8E"
			  "\xA8\x28\x69\x65\x31\xE1\x45\x83"
			  "\xE4\x49\xC4\x9C\xA7\x28\xAA\x21"
			  "\xCD\x5D\x0F\x15\xB7\x93\x07\x26"
			  "\xB0\x65\x6D\x91\x90\x23\x7A\xC6"
			  "\xDB\x68\xB0\xA1\x8E\xA4\x76\x4E"
			  "\xC6\x91\x83\x20\x92\x4D\x63\x7A"
			  "\x45\x18\x18\x74\x19\xAD\x71\x01"
			  "\x6B\x23\xAD\x9D\x4E\xE4\x6E\x46"
			  "\xC9\x73\x7A\xF9\x02\x95\xF4\x07"
			  "\x0E\x7A\xA6\xC5\xAE\xFA\x15\x2C"
			  "\x51\x71\xF1\xDC\x22\xB6\xAC\xD8"
			  "\x19\x24\x44\xBC\x0C\xFB\x3C\x2D"
			  "\xB1\x50\x47\x15\x0E\xDB\xB6\xD7"
			  "\xE8\x61\xE5\x95\x52\x1E\x3E\x49"
			  "\x70\xE9\x66\x04\x4C\xE1\xAF\xBD"
			  "\xDD\x15\x3B\x20\x59\x24\xFF\xB0"
			  "\x39\xAA\xE7\xBF\x23\xA3\x6E\xD5"
			  "\x15\xF0\x61\x4F\xAE\x89\x10\x58"
			  "\x5A\x33\x95\x52\x2A\xB5\x77\x9C"
			  "\xA5\x43\x80\x40\x27\x2D\xAE\xD9"
			  "\x3F\xE0\x80\x94\x78\x79\xCB\x7E"
			  "\xAD\x12\x44\x4C\xEC\x27\xB0\xEE"
			  "\x0B\x05\x2A\x82\x99\x58\xBB\x7A"
			  "\x8D\x6D\x9D\x8E\xE2\x8E\xE7\x93"
			  "\x2F\xB3\x09\x8D\x06\xD5\xEE\x70"
			  "\x16\xAE\x35\xC5\x52\x0F\x46\x1F"
			  "\x71\xF9\x5E\xF2\x67\xDC\x98\x2F"
			  "\xA3\x23\xAA\xD5\xD0\x49\xF4\xA6"
			  "\xF6\xB8\x32\xCD\xD6\x85\x73\x60"
			  "\x59\x20\xE7\x55\x0E\x91\xE2\x0C"
			  "\x3F\x1C\xEB\x3D\xDF\x52\x64\xF2"
			  "\x7D\x8B\x5D\x63\x16\xB9\xB2\x5D"
			  "\x5E\xAB\xB2\x97\xAB\x78\x44\xE7"
			  "\xC6\x72\x20\xC5\x90\x9B\xDC\x5D"
			  "\xB0\xEF\x44\xEF\x87\x31\x8D\xF4"
			  "\xFB\x81\x5D\xF7\x96\x96\xD4\x50"
			  "\x89\xA7\xF6\xB9\x67\x76\x40\x9E"
			  "\x9D\x40\xD5\x2C\x30\xB8\x01\x8F"
			  "\xE4\x7B\x71\x48\xA9\xA0\xA0\x1D"
			  "\x87\x52\xA4\x91\xA9\xD7\xA9\x51"
			  "\xD9\x59\xF7\xCC\x63\x22\xC1\x8D"
			  "\x84\x7B\xD8\x22\x32\x5C\x6F\x1D"
			  "\x6E\x9F\xFA\xDD\x49\x40\xDC\x37"
			  "\x14\x8C\xE1\x80\x1B\xDD\x36\x2A"
			  "\xD0\xE9\x54\x99\x5D\xBA\x3B\x11"
			  "\xD8\xFE\xC9\x5B\x5C\x25\xE5\x76"
			  "\xFB\xF2\x3F",
		.ilen	= 499,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12"
			  "\x7B\xE4\x4D",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec aes_ofb_enc_tv_template[] = {
	 /* From NIST Special Publication 800-38A, Appendix F.5 */
	{
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result = "\x3b\x3f\xd9\x2e\xb7\x2d\xad\x20"
			  "\x33\x34\x49\xf8\xe8\x3c\xfb\x4a"
			  "\x77\x89\x50\x8d\x16\x91\x8f\x03\xf5"
			  "\x3c\x52\xda\xc5\x4e\xd8\x25"
			  "\x97\x40\x05\x1e\x9c\x5f\xec\xf6\x43"
			  "\x44\xf7\xa8\x22\x60\xed\xcc"
			  "\x30\x4c\x65\x28\xf6\x59\xc7\x78"
			  "\x66\xa5\x10\xd9\xc1\xd6\xae\x5e",
		.rlen	= 64,
	}
};

static struct cipher_testvec aes_ofb_dec_tv_template[] = {
	 /* From NIST Special Publication 800-38A, Appendix F.5 */
	{
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input = "\x3b\x3f\xd9\x2e\xb7\x2d\xad\x20"
			  "\x33\x34\x49\xf8\xe8\x3c\xfb\x4a"
			  "\x77\x89\x50\x8d\x16\x91\x8f\x03\xf5"
			  "\x3c\x52\xda\xc5\x4e\xd8\x25"
			  "\x97\x40\x05\x1e\x9c\x5f\xec\xf6\x43"
			  "\x44\xf7\xa8\x22\x60\xed\xcc"
			  "\x30\x4c\x65\x28\xf6\x59\xc7\x78"
			  "\x66\xa5\x10\xd9\xc1\xd6\xae\x5e",
		.ilen	= 64,
		.result = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}
};

#if 0 /* Match tcrypt.c comment: "Test Crashing - Skip the test */
static struct aead_testvec aes_gcm_enc_tv_template[] = {
	{ /* From McGrew & Viega - http://citeseer.ist.psu.edu/656989.html */
		.key    = zeroed_string,
		.klen	= 16,
		.result	= "\x58\xe2\xfc\xce\xfa\x7e\x30\x61"
			  "\x36\x7f\x1d\x57\xa4\xe7\x45\x5a",
		.rlen	= 16,
	}, {
		.key    = zeroed_string,
		.klen	= 16,
		.input  = zeroed_string,
		.ilen	= 16,
		.result = "\x03\x88\xda\xce\x60\xb6\xa3\x92"
			  "\xf3\x28\xc2\xb9\x71\xb2\xfe\x78"
			  "\xab\x6e\x47\xd4\x2c\xec\x13\xbd"
			  "\xf5\x3a\x67\xb2\x12\x57\xbd\xdf",
		.rlen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 16,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.ilen	= 64,
		.result = "\x42\x83\x1e\xc2\x21\x77\x74\x24"
			  "\x4b\x72\x21\xb7\x84\xd0\xd4\x9c"
			  "\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0"
			  "\x35\xc1\x7e\x23\x29\xac\xa1\x2e"
			  "\x21\xd5\x14\xb2\x54\x66\x93\x1c"
			  "\x7d\x8f\x6a\x5a\xac\x84\xaa\x05"
			  "\x1b\xa3\x0b\x39\x6a\x0a\xac\x97"
			  "\x3d\x58\xe0\x91\x47\x3f\x59\x85"
			  "\x4d\x5c\x2a\xf3\x27\xcd\x64\xa6"
			  "\x2c\xf3\x5a\xbd\x2b\xa6\xfa\xb4",
		.rlen	= 80,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 16,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.ilen	= 60,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\x42\x83\x1e\xc2\x21\x77\x74\x24"
			  "\x4b\x72\x21\xb7\x84\xd0\xd4\x9c"
			  "\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0"
			  "\x35\xc1\x7e\x23\x29\xac\xa1\x2e"
			  "\x21\xd5\x14\xb2\x54\x66\x93\x1c"
			  "\x7d\x8f\x6a\x5a\xac\x84\xaa\x05"
			  "\x1b\xa3\x0b\x39\x6a\x0a\xac\x97"
			  "\x3d\x58\xe0\x91"
			  "\x5b\xc9\x4f\xbc\x32\x21\xa5\xdb"
			  "\x94\xfa\xe9\x5a\xe7\x12\x1a\x47",
		.rlen	= 76,
	}, {
		.key    = zeroed_string,
		.klen	= 24,
		.result	= "\xcd\x33\xb2\x8a\xc7\x73\xf7\x4b"
			  "\xa0\x0e\xd1\xf3\x12\x57\x24\x35",
		.rlen	= 16,
	}, {
		.key    = zeroed_string,
		.klen	= 24,
		.input  = zeroed_string,
		.ilen	= 16,
		.result = "\x98\xe7\x24\x7c\x07\xf0\xfe\x41"
			  "\x1c\x26\x7e\x43\x84\xb0\xf6\x00"
			  "\x2f\xf5\x8d\x80\x03\x39\x27\xab"
			  "\x8e\xf4\xd4\x58\x75\x14\xf0\xfb",
		.rlen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c",
		.klen	= 24,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.ilen	= 64,
		.result = "\x39\x80\xca\x0b\x3c\x00\xe8\x41"
			  "\xeb\x06\xfa\xc4\x87\x2a\x27\x57"
			  "\x85\x9e\x1c\xea\xa6\xef\xd9\x84"
			  "\x62\x85\x93\xb4\x0c\xa1\xe1\x9c"
			  "\x7d\x77\x3d\x00\xc1\x44\xc5\x25"
			  "\xac\x61\x9d\x18\xc8\x4a\x3f\x47"
			  "\x18\xe2\x44\x8b\x2f\xe3\x24\xd9"
			  "\xcc\xda\x27\x10\xac\xad\xe2\x56"
			  "\x99\x24\xa7\xc8\x58\x73\x36\xbf"
			  "\xb1\x18\x02\x4d\xb8\x67\x4a\x14",
		.rlen	= 80,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c",
		.klen	= 24,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.ilen	= 60,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\x39\x80\xca\x0b\x3c\x00\xe8\x41"
			  "\xeb\x06\xfa\xc4\x87\x2a\x27\x57"
			  "\x85\x9e\x1c\xea\xa6\xef\xd9\x84"
			  "\x62\x85\x93\xb4\x0c\xa1\xe1\x9c"
			  "\x7d\x77\x3d\x00\xc1\x44\xc5\x25"
			  "\xac\x61\x9d\x18\xc8\x4a\x3f\x47"
			  "\x18\xe2\x44\x8b\x2f\xe3\x24\xd9"
			  "\xcc\xda\x27\x10"
			  "\x25\x19\x49\x8e\x80\xf1\x47\x8f"
			  "\x37\xba\x55\xbd\x6d\x27\x61\x8c",
		.rlen	= 76,
		.np	= 2,
		.tap	= { 32, 28 },
		.anp	= 2,
		.atap	= { 8, 12 }
	}, {
		.key    = zeroed_string,
		.klen	= 32,
		.result	= "\x53\x0f\x8a\xfb\xc7\x45\x36\xb9"
			  "\xa9\x63\xb4\xf1\xc4\xcb\x73\x8b",
		.rlen	= 16,
	}
};
#endif

static struct aead_testvec aes_gcm_dec_tv_template[] = {
	{ /* From McGrew & Viega - http://citeseer.ist.psu.edu/656989.html */
		.key    = zeroed_string,
		.klen	= 32,
		.input	= "\xce\xa7\x40\x3d\x4d\x60\x6b\x6e"
			  "\x07\x4e\xc5\xd3\xba\xf3\x9d\x18"
			  "\xd0\xd1\xc8\xa7\x99\x99\x6b\xf0"
			  "\x26\x5b\x98\xb5\xd4\x8a\xb9\x19",
		.ilen	= 32,
		.result  = zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 32,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x52\x2d\xc1\xf0\x99\x56\x7d\x07"
			  "\xf4\x7f\x37\xa3\x2a\x84\x42\x7d"
			  "\x64\x3a\x8c\xdc\xbf\xe5\xc0\xc9"
			  "\x75\x98\xa2\xbd\x25\x55\xd1\xaa"
			  "\x8c\xb0\x8e\x48\x59\x0d\xbb\x3d"
			  "\xa7\xb0\x8b\x10\x56\x82\x88\x38"
			  "\xc5\xf6\x1e\x63\x93\xba\x7a\x0a"
			  "\xbc\xc9\xf6\x62\x89\x80\x15\xad"
			  "\xb0\x94\xda\xc5\xd9\x34\x71\xbd"
			  "\xec\x1a\x50\x22\x70\xe3\xcc\x6c",
		.ilen	= 80,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.rlen	= 64,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 32,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x52\x2d\xc1\xf0\x99\x56\x7d\x07"
			  "\xf4\x7f\x37\xa3\x2a\x84\x42\x7d"
			  "\x64\x3a\x8c\xdc\xbf\xe5\xc0\xc9"
			  "\x75\x98\xa2\xbd\x25\x55\xd1\xaa"
			  "\x8c\xb0\x8e\x48\x59\x0d\xbb\x3d"
			  "\xa7\xb0\x8b\x10\x56\x82\x88\x38"
			  "\xc5\xf6\x1e\x63\x93\xba\x7a\x0a"
			  "\xbc\xc9\xf6\x62"
			  "\x76\xfc\x6e\xce\x0f\x4e\x17\x68"
			  "\xcd\xdf\x88\x53\xbb\x2d\x55\x1b",
		.ilen	= 76,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.rlen	= 60,
		.np     = 2,
		.tap    = { 48, 28 },
		.anp	= 3,
		.atap	= { 8, 8, 4 }
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 16,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x42\x83\x1e\xc2\x21\x77\x74\x24"
			  "\x4b\x72\x21\xb7\x84\xd0\xd4\x9c"
			  "\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0"
			  "\x35\xc1\x7e\x23\x29\xac\xa1\x2e"
			  "\x21\xd5\x14\xb2\x54\x66\x93\x1c"
			  "\x7d\x8f\x6a\x5a\xac\x84\xaa\x05"
			  "\x1b\xa3\x0b\x39\x6a\x0a\xac\x97"
			  "\x3d\x58\xe0\x91\x47\x3f\x59\x85"
			  "\x4d\x5c\x2a\xf3\x27\xcd\x64\xa6"
			  "\x2c\xf3\x5a\xbd\x2b\xa6\xfa\xb4",
		.ilen	= 80,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.rlen	= 64,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 16,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x42\x83\x1e\xc2\x21\x77\x74\x24"
			  "\x4b\x72\x21\xb7\x84\xd0\xd4\x9c"
			  "\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0"
			  "\x35\xc1\x7e\x23\x29\xac\xa1\x2e"
			  "\x21\xd5\x14\xb2\x54\x66\x93\x1c"
			  "\x7d\x8f\x6a\x5a\xac\x84\xaa\x05"
			  "\x1b\xa3\x0b\x39\x6a\x0a\xac\x97"
			  "\x3d\x58\xe0\x91"
			  "\x5b\xc9\x4f\xbc\x32\x21\xa5\xdb"
			  "\x94\xfa\xe9\x5a\xe7\x12\x1a\x47",
		.ilen	= 76,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.rlen	= 60,
	}, {
		.key    = zeroed_string,
		.klen	= 24,
		.input	= "\x98\xe7\x24\x7c\x07\xf0\xfe\x41"
			  "\x1c\x26\x7e\x43\x84\xb0\xf6\x00"
			  "\x2f\xf5\x8d\x80\x03\x39\x27\xab"
			  "\x8e\xf4\xd4\x58\x75\x14\xf0\xfb",
		.ilen	= 32,
		.result  = zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c",
		.klen	= 24,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x39\x80\xca\x0b\x3c\x00\xe8\x41"
			  "\xeb\x06\xfa\xc4\x87\x2a\x27\x57"
			  "\x85\x9e\x1c\xea\xa6\xef\xd9\x84"
			  "\x62\x85\x93\xb4\x0c\xa1\xe1\x9c"
			  "\x7d\x77\x3d\x00\xc1\x44\xc5\x25"
			  "\xac\x61\x9d\x18\xc8\x4a\x3f\x47"
			  "\x18\xe2\x44\x8b\x2f\xe3\x24\xd9"
			  "\xcc\xda\x27\x10\xac\xad\xe2\x56"
			  "\x99\x24\xa7\xc8\x58\x73\x36\xbf"
			  "\xb1\x18\x02\x4d\xb8\x67\x4a\x14",
		.ilen	= 80,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.rlen	= 64,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c",
		.klen	= 24,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x39\x80\xca\x0b\x3c\x00\xe8\x41"
			  "\xeb\x06\xfa\xc4\x87\x2a\x27\x57"
			  "\x85\x9e\x1c\xea\xa6\xef\xd9\x84"
			  "\x62\x85\x93\xb4\x0c\xa1\xe1\x9c"
			  "\x7d\x77\x3d\x00\xc1\x44\xc5\x25"
			  "\xac\x61\x9d\x18\xc8\x4a\x3f\x47"
			  "\x18\xe2\x44\x8b\x2f\xe3\x24\xd9"
			  "\xcc\xda\x27\x10"
			  "\x25\x19\x49\x8e\x80\xf1\x47\x8f"
			  "\x37\xba\x55\xbd\x6d\x27\x61\x8c",
		.ilen	= 76,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.rlen	= 60,
	}
};

static struct aead_testvec aes_gcm_rfc4106_enc_tv_template[] = {
	{ /* Generated using Crypto++ */
		.key    = zeroed_string,
		.klen	= 20,
		.iv	= zeroed_string,
		.input  = zeroed_string,
		.ilen   = 16,
		.assoc  = zeroed_string,
		.alen   = 16,
		.result	= "\x03\x88\xDA\xCE\x60\xB6\xA3\x92"
			  "\xF3\x28\xC2\xB9\x71\xB2\xFE\x78"
			  "\x97\xFE\x4C\x23\x37\x42\x01\xE0"
			  "\x81\x9F\x8D\xC5\xD7\x41\xA0\x1B",
		.rlen	= 32,
	},{
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = zeroed_string,
		.ilen   = 16,
		.assoc  = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen   = 16,
		.result	= "\xC0\x0D\x8B\x42\x0F\x8F\x34\x18"
			  "\x88\xB1\xC5\xBC\xC5\xB6\xD6\x28"
			  "\x6A\x9D\xDF\x11\x5E\xFE\x5E\x9D"
			  "\x2F\x70\x44\x92\xF7\xF2\xE3\xEF",
		.rlen	= 32,

	}, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = zeroed_string,
		.input  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.ilen   = 16,
		.assoc  = zeroed_string,
		.alen   = 16,
		.result	= "\x4B\xB1\xB5\xE3\x25\x71\x70\xDE"
			  "\x7F\xC9\x9C\xA5\x14\x19\xF2\xAC"
			  "\x0B\x8F\x88\x69\x17\xE6\xB4\x3C"
			  "\xB1\x68\xFD\x14\x52\x64\x61\xB2",
		.rlen	= 32,
	}, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = zeroed_string,
		.input  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.ilen   = 16,
		.assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen   = 16,
		.result	= "\x4B\xB1\xB5\xE3\x25\x71\x70\xDE"
			  "\x7F\xC9\x9C\xA5\x14\x19\xF2\xAC"
			  "\x90\x92\xB7\xE3\x5F\xA3\x9A\x63"
			  "\x7E\xD7\x1F\xD8\xD3\x7C\x4B\xF5",
		.rlen	= 32,
	}, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.ilen   = 16,
		.assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen   = 16,
		.result	= "\xC1\x0C\x8A\x43\x0E\x8E\x35\x19"
			  "\x89\xB0\xC4\xBD\xC4\xB7\xD7\x29"
			  "\x64\x50\xF9\x32\x13\xFB\x74\x61"
			  "\xF4\xED\x52\xD3\xC5\x10\x55\x3C",
		.rlen	= 32,
	}, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.ilen   = 64,
		.assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen   = 16,
		.result	= "\xC1\x0C\x8A\x43\x0E\x8E\x35\x19"
			  "\x89\xB0\xC4\xBD\xC4\xB7\xD7\x29"
			  "\x98\x14\xA1\x42\x37\x80\xFD\x90"
			  "\x68\x12\x01\xA8\x91\x89\xB9\x83"
			  "\x5B\x11\x77\x12\x9B\xFF\x24\x89"
			  "\x94\x5F\x18\x12\xBA\x27\x09\x39"
			  "\x99\x96\x76\x42\x15\x1C\xCD\xCB"
			  "\xDC\xD3\xDA\x65\x73\xAF\x80\xCD"
			  "\xD2\xB6\xC2\x4A\x76\xC2\x92\x85"
			  "\xBD\xCF\x62\x98\x58\x14\xE5\xBD",
		.rlen	= 80,
	}, {
		.key    = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = "\x00\x00\x45\x67\x89\xab\xcd\xef",
		.input  = "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff",
		.ilen   = 192,
		.assoc  = "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\x00\x00\x45\x67"
			  "\x89\xab\xcd\xef",
		.alen   = 20,
		.result	= "\xC1\x76\x33\x85\xE2\x9B\x5F\xDE"
			  "\xDE\x89\x3D\x42\xE7\xC9\x69\x8A"
			  "\x44\x6D\xC3\x88\x46\x2E\xC2\x01"
			  "\x5E\xF6\x0C\x39\xF0\xC4\xA5\x82"
			  "\xCD\xE8\x31\xCC\x0A\x4C\xE4\x44"
			  "\x41\xA9\x82\x6F\x22\xA1\x23\x1A"
			  "\xA8\xE3\x16\xFD\x31\x5C\x27\x31"
			  "\xF1\x7F\x01\x63\xA3\xAF\x70\xA1"
			  "\xCF\x07\x57\x41\x67\xD0\xC4\x42"
			  "\xDB\x18\xC6\x4C\x4C\xE0\x3D\x9F"
			  "\x05\x07\xFB\x13\x7D\x4A\xCA\x5B"
			  "\xF0\xBF\x64\x7E\x05\xB1\x72\xEE"
			  "\x7C\x3B\xD4\xCD\x14\x03\xB2\x2C"
			  "\xD3\xA9\xEE\xFA\x17\xFC\x9C\xDF"
			  "\xC7\x75\x40\xFF\xAE\xAD\x1E\x59"
			  "\x2F\x30\x24\xFB\xAD\x6B\x10\xFA"
			  "\x6C\x9F\x5B\xE7\x25\xD5\xD0\x25"
			  "\xAC\x4A\x4B\xDA\xFC\x7A\x85\x1B"
			  "\x7E\x13\x06\x82\x08\x17\xA4\x35"
			  "\xEC\xC5\x8D\x63\x96\x81\x0A\x8F"
			  "\xA3\x05\x38\x95\x20\x1A\x47\x04"
			  "\x6F\x6D\xDA\x8F\xEF\xC1\x76\x35"
			  "\x6B\xC7\x4D\x0F\x94\x12\xCA\x3E"
			  "\x2E\xD5\x03\x2E\x86\x7E\xAA\x3B"
			  "\x37\x08\x1C\xCF\xBA\x5D\x71\x46"
			  "\x80\x72\xB0\x4C\x82\x0D\x60\x3C",
		.rlen	= 208,
	}, { /* From draft-mcgrew-gcm-test-01 */
		.key	= "\x4C\x80\xCD\xEF\xBB\x5D\x10\xDA"
			  "\x90\x6A\xC7\x3C\x36\x13\xA6\x34"
			  "\x2E\x44\x3B\x68",
		.klen	= 20,
		.iv	= "\x49\x56\xED\x7E\x3B\x24\x4C\xFE",
		.input	= "\x45\x00\x00\x48\x69\x9A\x00\x00"
			  "\x80\x11\x4D\xB7\xC0\xA8\x01\x02"
			  "\xC0\xA8\x01\x01\x0A\x9B\xF1\x56"
			  "\x38\xD3\x01\x00\x00\x01\x00\x00"
			  "\x00\x00\x00\x00\x04\x5F\x73\x69"
			  "\x70\x04\x5F\x75\x64\x70\x03\x73"
			  "\x69\x70\x09\x63\x79\x62\x65\x72"
			  "\x63\x69\x74\x79\x02\x64\x6B\x00"
			  "\x00\x21\x00\x01\x01\x02\x02\x01",
		.ilen	= 72,
		.assoc	= "\x00\x00\x43\x21\x87\x65\x43\x21"
			  "\x00\x00\x00\x00\x49\x56\xED\x7E"
			  "\x3B\x24\x4C\xFE",
		.alen	= 20,
		.result	= "\xFE\xCF\x53\x7E\x72\x9D\x5B\x07"
			  "\xDC\x30\xDF\x52\x8D\xD2\x2B\x76"
			  "\x8D\x1B\x98\x73\x66\x96\xA6\xFD"
			  "\x34\x85\x09\xFA\x13\xCE\xAC\x34"
			  "\xCF\xA2\x43\x6F\x14\xA3\xF3\xCF"
			  "\x65\x92\x5B\xF1\xF4\xA1\x3C\x5D"
			  "\x15\xB2\x1E\x18\x84\xF5\xFF\x62"
			  "\x47\xAE\xAB\xB7\x86\xB9\x3B\xCE"
			  "\x61\xBC\x17\xD7\x68\xFD\x97\x32"
			  "\x45\x90\x18\x14\x8F\x6C\xBE\x72"
			  "\x2F\xD0\x47\x96\x56\x2D\xFD\xB4",
		.rlen	= 88,
	}, {
		.key	= "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\x6D\x6A\x8F\x94\x67\x30\x83\x08"
			  "\xCA\xFE\xBA\xBE",
		.klen	= 20,
		.iv	= "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.input	= "\x45\x00\x00\x3E\x69\x8F\x00\x00"
			  "\x80\x11\x4D\xCC\xC0\xA8\x01\x02"
			  "\xC0\xA8\x01\x01\x0A\x98\x00\x35"
			  "\x00\x2A\x23\x43\xB2\xD0\x01\x00"
			  "\x00\x01\x00\x00\x00\x00\x00\x00"
			  "\x03\x73\x69\x70\x09\x63\x79\x62"
			  "\x65\x72\x63\x69\x74\x79\x02\x64"
			  "\x6B\x00\x00\x01\x00\x01\x00\x01",
		.ilen	= 64,
		.assoc	= "\x00\x00\xA5\xF8\x00\x00\x00\x0A"
			  "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.alen	= 16,
		.result	= "\xDE\xB2\x2C\xD9\xB0\x7C\x72\xC1"
			  "\x6E\x3A\x65\xBE\xEB\x8D\xF3\x04"
			  "\xA5\xA5\x89\x7D\x33\xAE\x53\x0F"
			  "\x1B\xA7\x6D\x5D\x11\x4D\x2A\x5C"
			  "\x3D\xE8\x18\x27\xC1\x0E\x9A\x4F"
			  "\x51\x33\x0D\x0E\xEC\x41\x66\x42"
			  "\xCF\xBB\x85\xA5\xB4\x7E\x48\xA4"
			  "\xEC\x3B\x9B\xA9\x5D\x91\x8B\xD1"
			  "\x83\xB7\x0D\x3A\xA8\xBC\x6E\xE4"
			  "\xC3\x09\xE9\xD8\x5A\x41\xAD\x4A",
		.rlen	= 80,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x11\x22\x33\x44",
		.klen	= 36,
		.iv	= "\x01\x02\x03\x04\x05\x06\x07\x08",
		.input	= "\x45\x00\x00\x30\x69\xA6\x40\x00"
			  "\x80\x06\x26\x90\xC0\xA8\x01\x02"
			  "\x93\x89\x15\x5E\x0A\x9E\x00\x8B"
			  "\x2D\xC5\x7E\xE0\x00\x00\x00\x00"
			  "\x70\x02\x40\x00\x20\xBF\x00\x00"
			  "\x02\x04\x05\xB4\x01\x01\x04\x02"
			  "\x01\x02\x02\x01",
		.ilen	= 52,
		.assoc	= "\x4A\x2C\xBF\xE3\x00\x00\x00\x02"
			  "\x01\x02\x03\x04\x05\x06\x07\x08",
		.alen	= 16,
		.result	= "\xFF\x42\x5C\x9B\x72\x45\x99\xDF"
			  "\x7A\x3B\xCD\x51\x01\x94\xE0\x0D"
			  "\x6A\x78\x10\x7F\x1B\x0B\x1C\xBF"
			  "\x06\xEF\xAE\x9D\x65\xA5\xD7\x63"
			  "\x74\x8A\x63\x79\x85\x77\x1D\x34"
			  "\x7F\x05\x45\x65\x9F\x14\xE9\x9D"
			  "\xEF\x84\x2D\x8E\xB3\x35\xF4\xEE"
			  "\xCF\xDB\xF8\x31\x82\x4B\x4C\x49"
			  "\x15\x95\x6C\x96",
		.rlen	= 68,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x45\x00\x00\x3C\x99\xC5\x00\x00"
			  "\x80\x01\xCB\x7A\x40\x67\x93\x18"
			  "\x01\x01\x01\x01\x08\x00\x07\x5C"
			  "\x02\x00\x44\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x75\x76\x77\x61\x62\x63\x64\x65"
			  "\x66\x67\x68\x69\x01\x02\x02\x01",
		.ilen	= 64,
		.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.result	= "\x46\x88\xDA\xF2\xF9\x73\xA3\x92"
			  "\x73\x29\x09\xC3\x31\xD5\x6D\x60"
			  "\xF6\x94\xAB\xAA\x41\x4B\x5E\x7F"
			  "\xF5\xFD\xCD\xFF\xF5\xE9\xA2\x84"
			  "\x45\x64\x76\x49\x27\x19\xFF\xB6"
			  "\x4D\xE7\xD9\xDC\xA1\xE1\xD8\x94"
			  "\xBC\x3B\xD5\x78\x73\xED\x4D\x18"
			  "\x1D\x19\xD4\xD5\xC8\xC1\x8A\xF3"
			  "\xF8\x21\xD4\x96\xEE\xB0\x96\xE9"
			  "\x8A\xD2\xB6\x9E\x47\x99\xC7\x1D",
		.rlen	= 80,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E\x43",
		.klen	= 20,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.input	= "\x45\x00\x00\x3C\x99\xC3\x00\x00"
			  "\x80\x01\xCB\x7C\x40\x67\x93\x18"
			  "\x01\x01\x01\x01\x08\x00\x08\x5C"
			  "\x02\x00\x43\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x75\x76\x77\x61\x62\x63\x64\x65"
			  "\x66\x67\x68\x69\x01\x02\x02\x01",
		.ilen	= 64,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.result	= "\xFB\xA2\xCA\xA4\x85\x3C\xF9\xF0"
			  "\xF2\x2C\xB1\x0D\x86\xDD\x83\xB0"
			  "\xFE\xC7\x56\x91\xCF\x1A\x04\xB0"
			  "\x0D\x11\x38\xEC\x9C\x35\x79\x17"
			  "\x65\xAC\xBD\x87\x01\xAD\x79\x84"
			  "\x5B\xF9\xFE\x3F\xBA\x48\x7B\xC9"
			  "\x17\x55\xE6\x66\x2B\x4C\x8D\x0D"
			  "\x1F\x5E\x22\x73\x95\x30\x32\x0A"
			  "\xE0\xD7\x31\xCC\x97\x8E\xCA\xFA"
			  "\xEA\xE8\x8F\x00\xE8\x0D\x6E\x48",
		.rlen	= 80,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E\x43",
		.klen	= 20,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.input	= "\x45\x00\x00\x1C\x42\xA2\x00\x00"
			  "\x80\x01\x44\x1F\x40\x67\x93\xB6"
			  "\xE0\x00\x00\x02\x0A\x00\xF5\xFF"
			  "\x01\x02\x02\x01",
		.ilen	= 28,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.result	= "\xFB\xA2\xCA\x84\x5E\x5D\xF9\xF0"
			  "\xF2\x2C\x3E\x6E\x86\xDD\x83\x1E"
			  "\x1F\xC6\x57\x92\xCD\x1A\xF9\x13"
			  "\x0E\x13\x79\xED\x36\x9F\x07\x1F"
			  "\x35\xE0\x34\xBE\x95\xF1\x12\xE4"
			  "\xE7\xD0\x5D\x35",
		.rlen	= 44,
	}, {
		.key	= "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\x6D\x6A\x8F\x94\x67\x30\x83\x08"
			  "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\xCA\xFE\xBA\xBE",
		.klen	= 28,
		.iv	= "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.input	= "\x45\x00\x00\x28\xA4\xAD\x40\x00"
			  "\x40\x06\x78\x80\x0A\x01\x03\x8F"
			  "\x0A\x01\x06\x12\x80\x23\x06\xB8"
			  "\xCB\x71\x26\x02\xDD\x6B\xB0\x3E"
			  "\x50\x10\x16\xD0\x75\x68\x00\x01",
		.ilen	= 40,
		.assoc	= "\x00\x00\xA5\xF8\x00\x00\x00\x0A"
			  "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.alen	= 16,
		.result	= "\xA5\xB1\xF8\x06\x60\x29\xAE\xA4"
			  "\x0E\x59\x8B\x81\x22\xDE\x02\x42"
			  "\x09\x38\xB3\xAB\x33\xF8\x28\xE6"
			  "\x87\xB8\x85\x8B\x5B\xFB\xDB\xD0"
			  "\x31\x5B\x27\x45\x21\x44\xCC\x77"
			  "\x95\x45\x7B\x96\x52\x03\x7F\x53"
			  "\x18\x02\x7B\x5B\x4C\xD7\xA6\x36",
		.rlen	= 56,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xDE\xCA\xF8\x88",
		.klen	= 20,
		.iv	= "\xCA\xFE\xDE\xBA\xCE\xFA\xCE\x74",
		.input	= "\x45\x00\x00\x49\x33\xBA\x00\x00"
			  "\x7F\x11\x91\x06\xC3\xFB\x1D\x10"
			  "\xC2\xB1\xD3\x26\xC0\x28\x31\xCE"
			  "\x00\x35\xDD\x7B\x80\x03\x02\xD5"
			  "\x00\x00\x4E\x20\x00\x1E\x8C\x18"
			  "\xD7\x5B\x81\xDC\x91\xBA\xA0\x47"
			  "\x6B\x91\xB9\x24\xB2\x80\x38\x9D"
			  "\x92\xC9\x63\xBA\xC0\x46\xEC\x95"
			  "\x9B\x62\x66\xC0\x47\x22\xB1\x49"
			  "\x23\x01\x01\x01",
		.ilen	= 76,
		.assoc	= "\x00\x00\x01\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x01\xCA\xFE\xDE\xBA"
			  "\xCE\xFA\xCE\x74",
		.alen	= 20,
		.result	= "\x18\xA6\xFD\x42\xF7\x2C\xBF\x4A"
			  "\xB2\xA2\xEA\x90\x1F\x73\xD8\x14"
			  "\xE3\xE7\xF2\x43\xD9\x54\x12\xE1"
			  "\xC3\x49\xC1\xD2\xFB\xEC\x16\x8F"
			  "\x91\x90\xFE\xEB\xAF\x2C\xB0\x19"
			  "\x84\xE6\x58\x63\x96\x5D\x74\x72"
			  "\xB7\x9D\xA3\x45\xE0\xE7\x80\x19"
			  "\x1F\x0D\x2F\x0E\x0F\x49\x6C\x22"
			  "\x6F\x21\x27\xB2\x7D\xB3\x57\x24"
			  "\xE7\x84\x5D\x68\x65\x1F\x57\xE6"
			  "\x5F\x35\x4F\x75\xFF\x17\x01\x57"
			  "\x69\x62\x34\x36",
		.rlen	= 92,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x73\x61\x6C\x74",
		.klen	= 36,
		.iv	= "\x61\x6E\x64\x01\x69\x76\x65\x63",
		.input	= "\x45\x08\x00\x28\x73\x2C\x00\x00"
			  "\x40\x06\xE9\xF9\x0A\x01\x06\x12"
			  "\x0A\x01\x03\x8F\x06\xB8\x80\x23"
			  "\xDD\x6B\xAF\xBE\xCB\x71\x26\x02"
			  "\x50\x10\x1F\x64\x6D\x54\x00\x01",
		.ilen	= 40,
		.assoc	= "\x17\x40\x5E\x67\x15\x6F\x31\x26"
			  "\xDD\x0D\xB9\x9B\x61\x6E\x64\x01"
			  "\x69\x76\x65\x63",
		.alen	= 20,
		.result	= "\xF2\xD6\x9E\xCD\xBD\x5A\x0D\x5B"
			  "\x8D\x5E\xF3\x8B\xAD\x4D\xA5\x8D"
			  "\x1F\x27\x8F\xDE\x98\xEF\x67\x54"
			  "\x9D\x52\x4A\x30\x18\xD9\xA5\x7F"
			  "\xF4\xD3\xA3\x1C\xE6\x73\x11\x9E"
			  "\x45\x16\x26\xC2\x41\x57\x71\xE3"
			  "\xB7\xEE\xBC\xA6\x14\xC8\x9B\x35",
		.rlen	= 56,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E\x43",
		.klen	= 20,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.input	= "\x45\x00\x00\x49\x33\x3E\x00\x00"
			  "\x7F\x11\x91\x82\xC3\xFB\x1D\x10"
			  "\xC2\xB1\xD3\x26\xC0\x28\x31\xCE"
			  "\x00\x35\xCB\x45\x80\x03\x02\x5B"
			  "\x00\x00\x01\xE0\x00\x1E\x8C\x18"
			  "\xD6\x57\x59\xD5\x22\x84\xA0\x35"
			  "\x2C\x71\x47\x5C\x88\x80\x39\x1C"
			  "\x76\x4D\x6E\x5E\xE0\x49\x6B\x32"
			  "\x5A\xE2\x70\xC0\x38\x99\x49\x39"
			  "\x15\x01\x01\x01",
		.ilen	= 76,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.result	= "\xFB\xA2\xCA\xD1\x2F\xC1\xF9\xF0"
			  "\x0D\x3C\xEB\xF3\x05\x41\x0D\xB8"
			  "\x3D\x77\x84\xB6\x07\x32\x3D\x22"
			  "\x0F\x24\xB0\xA9\x7D\x54\x18\x28"
			  "\x00\xCA\xDB\x0F\x68\xD9\x9E\xF0"
			  "\xE0\xC0\xC8\x9A\xE9\xBE\xA8\x88"
			  "\x4E\x52\xD6\x5B\xC1\xAF\xD0\x74"
			  "\x0F\x74\x24\x44\x74\x7B\x5B\x39"
			  "\xAB\x53\x31\x63\xAA\xD4\x55\x0E"
			  "\xE5\x16\x09\x75\xCD\xB6\x08\xC5"
			  "\x76\x91\x89\x60\x97\x63\xB8\xE1"
			  "\x8C\xAA\x81\xE2",
		.rlen	= 92,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x73\x61\x6C\x74",
		.klen	= 36,
		.iv	= "\x61\x6E\x64\x01\x69\x76\x65\x63",
		.input	= "\x63\x69\x73\x63\x6F\x01\x72\x75"
			  "\x6C\x65\x73\x01\x74\x68\x65\x01"
			  "\x6E\x65\x74\x77\x65\x01\x64\x65"
			  "\x66\x69\x6E\x65\x01\x74\x68\x65"
			  "\x74\x65\x63\x68\x6E\x6F\x6C\x6F"
			  "\x67\x69\x65\x73\x01\x74\x68\x61"
			  "\x74\x77\x69\x6C\x6C\x01\x64\x65"
			  "\x66\x69\x6E\x65\x74\x6F\x6D\x6F"
			  "\x72\x72\x6F\x77\x01\x02\x02\x01",
		.ilen	= 72,
		.assoc	= "\x17\x40\x5E\x67\x15\x6F\x31\x26"
			  "\xDD\x0D\xB9\x9B\x61\x6E\x64\x01"
			  "\x69\x76\x65\x63",
		.alen	= 20,
		.result	= "\xD4\xB7\xED\x86\xA1\x77\x7F\x2E"
			  "\xA1\x3D\x69\x73\xD3\x24\xC6\x9E"
			  "\x7B\x43\xF8\x26\xFB\x56\x83\x12"
			  "\x26\x50\x8B\xEB\xD2\xDC\xEB\x18"
			  "\xD0\xA6\xDF\x10\xE5\x48\x7D\xF0"
			  "\x74\x11\x3E\x14\xC6\x41\x02\x4E"
			  "\x3E\x67\x73\xD9\x1A\x62\xEE\x42"
			  "\x9B\x04\x3A\x10\xE3\xEF\xE6\xB0"
			  "\x12\xA4\x93\x63\x41\x23\x64\xF8"
			  "\xC0\xCA\xC5\x87\xF2\x49\xE5\x6B"
			  "\x11\xE2\x4F\x30\xE4\x4C\xCC\x76",
		.rlen	= 88,
	}, {
		.key	= "\x7D\x77\x3D\x00\xC1\x44\xC5\x25"
			  "\xAC\x61\x9D\x18\xC8\x4A\x3F\x47"
			  "\xD9\x66\x42\x67",
		.klen	= 20,
		.iv	= "\x43\x45\x7E\x91\x82\x44\x3B\xC6",
		.input	= "\x01\x02\x02\x01",
		.ilen	= 4,
		.assoc	= "\x33\x54\x67\xAE\xFF\xFF\xFF\xFF"
			  "\x43\x45\x7E\x91\x82\x44\x3B\xC6",
		.alen	= 16,
		.result	= "\x43\x7F\x86\x6B\xCB\x3F\x69\x9F"
			  "\xE9\xB0\x82\x2B\xAC\x96\x1C\x45"
			  "\x04\xBE\xF2\x70",
		.rlen	= 20,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xDE\xCA\xF8\x88",
		.klen	= 20,
		.iv	= "\xCA\xFE\xDE\xBA\xCE\xFA\xCE\x74",
		.input	= "\x74\x6F\x01\x62\x65\x01\x6F\x72"
			  "\x01\x6E\x6F\x74\x01\x74\x6F\x01"
			  "\x62\x65\x00\x01",
		.ilen	= 20,
		.assoc	= "\x00\x00\x01\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x01\xCA\xFE\xDE\xBA"
			  "\xCE\xFA\xCE\x74",
		.alen	= 20,
		.result	= "\x29\xC9\xFC\x69\xA1\x97\xD0\x38"
			  "\xCC\xDD\x14\xE2\xDD\xFC\xAA\x05"
			  "\x43\x33\x21\x64\x41\x25\x03\x52"
			  "\x43\x03\xED\x3C\x6C\x5F\x28\x38"
			  "\x43\xAF\x8C\x3E",
		.rlen	= 36,
	}, {
		.key	= "\x6C\x65\x67\x61\x6C\x69\x7A\x65"
			  "\x6D\x61\x72\x69\x6A\x75\x61\x6E"
			  "\x61\x61\x6E\x64\x64\x6F\x69\x74"
			  "\x62\x65\x66\x6F\x72\x65\x69\x61"
			  "\x74\x75\x72\x6E",
		.klen	= 36,
		.iv	= "\x33\x30\x21\x69\x67\x65\x74\x6D",
		.input	= "\x45\x00\x00\x30\xDA\x3A\x00\x00"
			  "\x80\x01\xDF\x3B\xC0\xA8\x00\x05"
			  "\xC0\xA8\x00\x01\x08\x00\xC6\xCD"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.ilen	= 52,
		.assoc	= "\x79\x6B\x69\x63\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\x33\x30\x21\x69"
			  "\x67\x65\x74\x6D",
		.alen	= 20,
		.result	= "\xF9\x7A\xB2\xAA\x35\x6D\x8E\xDC"
			  "\xE1\x76\x44\xAC\x8C\x78\xE2\x5D"
			  "\xD2\x4D\xED\xBB\x29\xEB\xF1\xB6"
			  "\x4A\x27\x4B\x39\xB4\x9C\x3A\x86"
			  "\x4C\xD3\xD7\x8C\xA4\xAE\x68\xA3"
			  "\x2B\x42\x45\x8F\xB5\x7D\xBE\x82"
			  "\x1D\xCC\x63\xB9\xD0\x93\x7B\xA2"
			  "\x94\x5F\x66\x93\x68\x66\x1A\x32"
			  "\x9F\xB4\xC0\x53",
		.rlen	= 68,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E\x43",
		.klen	= 20,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.input	= "\x45\x00\x00\x30\xDA\x3A\x00\x00"
			  "\x80\x01\xDF\x3B\xC0\xA8\x00\x05"
			  "\xC0\xA8\x00\x01\x08\x00\xC6\xCD"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.ilen	= 52,
		.assoc	= "\x3F\x7E\xF6\x42\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.result	= "\xFB\xA2\xCA\xA8\xC6\xC5\xF9\xF0"
			  "\xF2\x2C\xA5\x4A\x06\x12\x10\xAD"
			  "\x3F\x6E\x57\x91\xCF\x1A\xCA\x21"
			  "\x0D\x11\x7C\xEC\x9C\x35\x79\x17"
			  "\x65\xAC\xBD\x87\x01\xAD\x79\x84"
			  "\x5B\xF9\xFE\x3F\xBA\x48\x7B\xC9"
			  "\x63\x21\x93\x06\x84\xEE\xCA\xDB"
			  "\x56\x91\x25\x46\xE7\xA9\x5C\x97"
			  "\x40\xD7\xCB\x05",
		.rlen	= 68,
	}, {
		.key	= "\x4C\x80\xCD\xEF\xBB\x5D\x10\xDA"
			  "\x90\x6A\xC7\x3C\x36\x13\xA6\x34"
			  "\x22\x43\x3C\x64",
		.klen	= 20,
		.iv	= "\x48\x55\xEC\x7D\x3A\x23\x4B\xFD",
		.input	= "\x08\x00\xC6\xCD\x02\x00\x07\x00"
			  "\x61\x62\x63\x64\x65\x66\x67\x68"
			  "\x69\x6A\x6B\x6C\x6D\x6E\x6F\x70"
			  "\x71\x72\x73\x74\x01\x02\x02\x01",
		.ilen	= 32,
		.assoc	= "\x00\x00\x43\x21\x87\x65\x43\x21"
			  "\x00\x00\x00\x07\x48\x55\xEC\x7D"
			  "\x3A\x23\x4B\xFD",
		.alen	= 20,
		.result	= "\x74\x75\x2E\x8A\xEB\x5D\x87\x3C"
			  "\xD7\xC0\xF4\xAC\xC3\x6C\x4B\xFF"
			  "\x84\xB7\xD7\xB9\x8F\x0C\xA8\xB6"
			  "\xAC\xDA\x68\x94\xBC\x61\x90\x69"
			  "\xEF\x9C\xBC\x28\xFE\x1B\x56\xA7"
			  "\xC4\xE0\xD5\x8C\x86\xCD\x2B\xC0",
		.rlen	= 48,
	}
};

static struct aead_testvec aes_gcm_rfc4106_dec_tv_template[] = {
	{ /* Generated using Crypto++ */
		.key    = zeroed_string,
		.klen	= 20,
		.iv     = zeroed_string,
		.input	= "\x03\x88\xDA\xCE\x60\xB6\xA3\x92"
			  "\xF3\x28\xC2\xB9\x71\xB2\xFE\x78"
			  "\x97\xFE\x4C\x23\x37\x42\x01\xE0"
			  "\x81\x9F\x8D\xC5\xD7\x41\xA0\x1B",
		.ilen	= 32,
		.assoc  = zeroed_string,
		.alen   = 16,
		.result = zeroed_string,
		.rlen   = 16,

	},{
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\xC0\x0D\x8B\x42\x0F\x8F\x34\x18"
			  "\x88\xB1\xC5\xBC\xC5\xB6\xD6\x28"
			  "\x6A\x9D\xDF\x11\x5E\xFE\x5E\x9D"
			  "\x2F\x70\x44\x92\xF7\xF2\xE3\xEF",
		.ilen	= 32,
		.assoc  = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen   = 16,
		.result = zeroed_string,
		.rlen   = 16,
	}, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = zeroed_string,
		.input	= "\x4B\xB1\xB5\xE3\x25\x71\x70\xDE"
			  "\x7F\xC9\x9C\xA5\x14\x19\xF2\xAC"
			  "\x0B\x8F\x88\x69\x17\xE6\xB4\x3C"
			  "\xB1\x68\xFD\x14\x52\x64\x61\xB2",
		.ilen	= 32,
		.assoc  = zeroed_string,
		.alen   = 16,
		.result = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.rlen   = 16,
	}, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = zeroed_string,
		.input	= "\x4B\xB1\xB5\xE3\x25\x71\x70\xDE"
			  "\x7F\xC9\x9C\xA5\x14\x19\xF2\xAC"
			  "\x90\x92\xB7\xE3\x5F\xA3\x9A\x63"
			  "\x7E\xD7\x1F\xD8\xD3\x7C\x4B\xF5",
		.ilen	= 32,
		.assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen   = 16,
		.result = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.rlen   = 16,

	}, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\xC1\x0C\x8A\x43\x0E\x8E\x35\x19"
			  "\x89\xB0\xC4\xBD\xC4\xB7\xD7\x29"
			  "\x64\x50\xF9\x32\x13\xFB\x74\x61"
			  "\xF4\xED\x52\xD3\xC5\x10\x55\x3C",
		.ilen	= 32,
		.assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen   = 16,
		.result = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.rlen   = 16,
	}, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\xC1\x0C\x8A\x43\x0E\x8E\x35\x19"
			  "\x89\xB0\xC4\xBD\xC4\xB7\xD7\x29"
			  "\x98\x14\xA1\x42\x37\x80\xFD\x90"
			  "\x68\x12\x01\xA8\x91\x89\xB9\x83"
			  "\x5B\x11\x77\x12\x9B\xFF\x24\x89"
			  "\x94\x5F\x18\x12\xBA\x27\x09\x39"
			  "\x99\x96\x76\x42\x15\x1C\xCD\xCB"
			  "\xDC\xD3\xDA\x65\x73\xAF\x80\xCD"
			  "\xD2\xB6\xC2\x4A\x76\xC2\x92\x85"
			  "\xBD\xCF\x62\x98\x58\x14\xE5\xBD",
		.ilen	= 80,
		.assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen   = 16,
		.result = "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.rlen   = 64,
	}, {
		.key    = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv     = "\x00\x00\x45\x67\x89\xab\xcd\xef",
		.input	= "\xC1\x76\x33\x85\xE2\x9B\x5F\xDE"
			  "\xDE\x89\x3D\x42\xE7\xC9\x69\x8A"
			  "\x44\x6D\xC3\x88\x46\x2E\xC2\x01"
			  "\x5E\xF6\x0C\x39\xF0\xC4\xA5\x82"
			  "\xCD\xE8\x31\xCC\x0A\x4C\xE4\x44"
			  "\x41\xA9\x82\x6F\x22\xA1\x23\x1A"
			  "\xA8\xE3\x16\xFD\x31\x5C\x27\x31"
			  "\xF1\x7F\x01\x63\xA3\xAF\x70\xA1"
			  "\xCF\x07\x57\x41\x67\xD0\xC4\x42"
			  "\xDB\x18\xC6\x4C\x4C\xE0\x3D\x9F"
			  "\x05\x07\xFB\x13\x7D\x4A\xCA\x5B"
			  "\xF0\xBF\x64\x7E\x05\xB1\x72\xEE"
			  "\x7C\x3B\xD4\xCD\x14\x03\xB2\x2C"
			  "\xD3\xA9\xEE\xFA\x17\xFC\x9C\xDF"
			  "\xC7\x75\x40\xFF\xAE\xAD\x1E\x59"
			  "\x2F\x30\x24\xFB\xAD\x6B\x10\xFA"
			  "\x6C\x9F\x5B\xE7\x25\xD5\xD0\x25"
			  "\xAC\x4A\x4B\xDA\xFC\x7A\x85\x1B"
			  "\x7E\x13\x06\x82\x08\x17\xA4\x35"
			  "\xEC\xC5\x8D\x63\x96\x81\x0A\x8F"
			  "\xA3\x05\x38\x95\x20\x1A\x47\x04"
			  "\x6F\x6D\xDA\x8F\xEF\xC1\x76\x35"
			  "\x6B\xC7\x4D\x0F\x94\x12\xCA\x3E"
			  "\x2E\xD5\x03\x2E\x86\x7E\xAA\x3B"
			  "\x37\x08\x1C\xCF\xBA\x5D\x71\x46"
			  "\x80\x72\xB0\x4C\x82\x0D\x60\x3C",
		.ilen	= 208,
		.assoc  = "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\x00\x00\x45\x67"
			  "\x89\xab\xcd\xef",
		.alen   = 20,
		.result = "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff",
		.rlen   = 192,
	}, {
		.key	= "\x4C\x80\xCD\xEF\xBB\x5D\x10\xDA"
			  "\x90\x6A\xC7\x3C\x36\x13\xA6\x34"
			  "\x2E\x44\x3B\x68",
		.klen	= 20,
		.iv	= "\x49\x56\xED\x7E\x3B\x24\x4C\xFE",
		.result	= "\x45\x00\x00\x48\x69\x9A\x00\x00"
			  "\x80\x11\x4D\xB7\xC0\xA8\x01\x02"
			  "\xC0\xA8\x01\x01\x0A\x9B\xF1\x56"
			  "\x38\xD3\x01\x00\x00\x01\x00\x00"
			  "\x00\x00\x00\x00\x04\x5F\x73\x69"
			  "\x70\x04\x5F\x75\x64\x70\x03\x73"
			  "\x69\x70\x09\x63\x79\x62\x65\x72"
			  "\x63\x69\x74\x79\x02\x64\x6B\x00"
			  "\x00\x21\x00\x01\x01\x02\x02\x01",
		.rlen	= 72,
		.assoc	= "\x00\x00\x43\x21\x87\x65\x43\x21"
			  "\x00\x00\x00\x00\x49\x56\xED\x7E"
			  "\x3B\x24\x4C\xFE",
		.alen	= 20,
		.input	= "\xFE\xCF\x53\x7E\x72\x9D\x5B\x07"
			  "\xDC\x30\xDF\x52\x8D\xD2\x2B\x76"
			  "\x8D\x1B\x98\x73\x66\x96\xA6\xFD"
			  "\x34\x85\x09\xFA\x13\xCE\xAC\x34"
			  "\xCF\xA2\x43\x6F\x14\xA3\xF3\xCF"
			  "\x65\x92\x5B\xF1\xF4\xA1\x3C\x5D"
			  "\x15\xB2\x1E\x18\x84\xF5\xFF\x62"
			  "\x47\xAE\xAB\xB7\x86\xB9\x3B\xCE"
			  "\x61\xBC\x17\xD7\x68\xFD\x97\x32"
			  "\x45\x90\x18\x14\x8F\x6C\xBE\x72"
			  "\x2F\xD0\x47\x96\x56\x2D\xFD\xB4",
		.ilen	= 88,
	}, {
		.key	= "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\x6D\x6A\x8F\x94\x67\x30\x83\x08"
			  "\xCA\xFE\xBA\xBE",
		.klen	= 20,
		.iv	= "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.result	= "\x45\x00\x00\x3E\x69\x8F\x00\x00"
			  "\x80\x11\x4D\xCC\xC0\xA8\x01\x02"
			  "\xC0\xA8\x01\x01\x0A\x98\x00\x35"
			  "\x00\x2A\x23\x43\xB2\xD0\x01\x00"
			  "\x00\x01\x00\x00\x00\x00\x00\x00"
			  "\x03\x73\x69\x70\x09\x63\x79\x62"
			  "\x65\x72\x63\x69\x74\x79\x02\x64"
			  "\x6B\x00\x00\x01\x00\x01\x00\x01",
		.rlen	= 64,
		.assoc	= "\x00\x00\xA5\xF8\x00\x00\x00\x0A"
			  "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.alen	= 16,
		.input	= "\xDE\xB2\x2C\xD9\xB0\x7C\x72\xC1"
			  "\x6E\x3A\x65\xBE\xEB\x8D\xF3\x04"
			  "\xA5\xA5\x89\x7D\x33\xAE\x53\x0F"
			  "\x1B\xA7\x6D\x5D\x11\x4D\x2A\x5C"
			  "\x3D\xE8\x18\x27\xC1\x0E\x9A\x4F"
			  "\x51\x33\x0D\x0E\xEC\x41\x66\x42"
			  "\xCF\xBB\x85\xA5\xB4\x7E\x48\xA4"
			  "\xEC\x3B\x9B\xA9\x5D\x91\x8B\xD1"
			  "\x83\xB7\x0D\x3A\xA8\xBC\x6E\xE4"
			  "\xC3\x09\xE9\xD8\x5A\x41\xAD\x4A",
		.ilen	= 80,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x11\x22\x33\x44",
		.klen	= 36,
		.iv	= "\x01\x02\x03\x04\x05\x06\x07\x08",
		.result	= "\x45\x00\x00\x30\x69\xA6\x40\x00"
			  "\x80\x06\x26\x90\xC0\xA8\x01\x02"
			  "\x93\x89\x15\x5E\x0A\x9E\x00\x8B"
			  "\x2D\xC5\x7E\xE0\x00\x00\x00\x00"
			  "\x70\x02\x40\x00\x20\xBF\x00\x00"
			  "\x02\x04\x05\xB4\x01\x01\x04\x02"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
		.assoc	= "\x4A\x2C\xBF\xE3\x00\x00\x00\x02"
			  "\x01\x02\x03\x04\x05\x06\x07\x08",
		.alen	= 16,
		.input	= "\xFF\x42\x5C\x9B\x72\x45\x99\xDF"
			  "\x7A\x3B\xCD\x51\x01\x94\xE0\x0D"
			  "\x6A\x78\x10\x7F\x1B\x0B\x1C\xBF"
			  "\x06\xEF\xAE\x9D\x65\xA5\xD7\x63"
			  "\x74\x8A\x63\x79\x85\x77\x1D\x34"
			  "\x7F\x05\x45\x65\x9F\x14\xE9\x9D"
			  "\xEF\x84\x2D\x8E\xB3\x35\xF4\xEE"
			  "\xCF\xDB\xF8\x31\x82\x4B\x4C\x49"
			  "\x15\x95\x6C\x96",
		.ilen	= 68,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00",
		.klen	= 20,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.result	= "\x45\x00\x00\x3C\x99\xC5\x00\x00"
			  "\x80\x01\xCB\x7A\x40\x67\x93\x18"
			  "\x01\x01\x01\x01\x08\x00\x07\x5C"
			  "\x02\x00\x44\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x75\x76\x77\x61\x62\x63\x64\x65"
			  "\x66\x67\x68\x69\x01\x02\x02\x01",
		.rlen	= 64,
		.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.input	= "\x46\x88\xDA\xF2\xF9\x73\xA3\x92"
			  "\x73\x29\x09\xC3\x31\xD5\x6D\x60"
			  "\xF6\x94\xAB\xAA\x41\x4B\x5E\x7F"
			  "\xF5\xFD\xCD\xFF\xF5\xE9\xA2\x84"
			  "\x45\x64\x76\x49\x27\x19\xFF\xB6"
			  "\x4D\xE7\xD9\xDC\xA1\xE1\xD8\x94"
			  "\xBC\x3B\xD5\x78\x73\xED\x4D\x18"
			  "\x1D\x19\xD4\xD5\xC8\xC1\x8A\xF3"
			  "\xF8\x21\xD4\x96\xEE\xB0\x96\xE9"
			  "\x8A\xD2\xB6\x9E\x47\x99\xC7\x1D",
		.ilen	= 80,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E\x43",
		.klen	= 20,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.result	= "\x45\x00\x00\x3C\x99\xC3\x00\x00"
			  "\x80\x01\xCB\x7C\x40\x67\x93\x18"
			  "\x01\x01\x01\x01\x08\x00\x08\x5C"
			  "\x02\x00\x43\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x75\x76\x77\x61\x62\x63\x64\x65"
			  "\x66\x67\x68\x69\x01\x02\x02\x01",
		.rlen	= 64,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.input	= "\xFB\xA2\xCA\xA4\x85\x3C\xF9\xF0"
			  "\xF2\x2C\xB1\x0D\x86\xDD\x83\xB0"
			  "\xFE\xC7\x56\x91\xCF\x1A\x04\xB0"
			  "\x0D\x11\x38\xEC\x9C\x35\x79\x17"
			  "\x65\xAC\xBD\x87\x01\xAD\x79\x84"
			  "\x5B\xF9\xFE\x3F\xBA\x48\x7B\xC9"
			  "\x17\x55\xE6\x66\x2B\x4C\x8D\x0D"
			  "\x1F\x5E\x22\x73\x95\x30\x32\x0A"
			  "\xE0\xD7\x31\xCC\x97\x8E\xCA\xFA"
			  "\xEA\xE8\x8F\x00\xE8\x0D\x6E\x48",
		.ilen	= 80,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E\x43",
		.klen	= 20,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.result	= "\x45\x00\x00\x1C\x42\xA2\x00\x00"
			  "\x80\x01\x44\x1F\x40\x67\x93\xB6"
			  "\xE0\x00\x00\x02\x0A\x00\xF5\xFF"
			  "\x01\x02\x02\x01",
		.rlen	= 28,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.input	= "\xFB\xA2\xCA\x84\x5E\x5D\xF9\xF0"
			  "\xF2\x2C\x3E\x6E\x86\xDD\x83\x1E"
			  "\x1F\xC6\x57\x92\xCD\x1A\xF9\x13"
			  "\x0E\x13\x79\xED\x36\x9F\x07\x1F"
			  "\x35\xE0\x34\xBE\x95\xF1\x12\xE4"
			  "\xE7\xD0\x5D\x35",
		.ilen	= 44,
	}, {
		.key	= "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\x6D\x6A\x8F\x94\x67\x30\x83\x08"
			  "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\xCA\xFE\xBA\xBE",
		.klen	= 28,
		.iv	= "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.result	= "\x45\x00\x00\x28\xA4\xAD\x40\x00"
			  "\x40\x06\x78\x80\x0A\x01\x03\x8F"
			  "\x0A\x01\x06\x12\x80\x23\x06\xB8"
			  "\xCB\x71\x26\x02\xDD\x6B\xB0\x3E"
			  "\x50\x10\x16\xD0\x75\x68\x00\x01",
		.rlen	= 40,
		.assoc	= "\x00\x00\xA5\xF8\x00\x00\x00\x0A"
			  "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.alen	= 16,
		.input	= "\xA5\xB1\xF8\x06\x60\x29\xAE\xA4"
			  "\x0E\x59\x8B\x81\x22\xDE\x02\x42"
			  "\x09\x38\xB3\xAB\x33\xF8\x28\xE6"
			  "\x87\xB8\x85\x8B\x5B\xFB\xDB\xD0"
			  "\x31\x5B\x27\x45\x21\x44\xCC\x77"
			  "\x95\x45\x7B\x96\x52\x03\x7F\x53"
			  "\x18\x02\x7B\x5B\x4C\xD7\xA6\x36",
		.ilen	= 56,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xDE\xCA\xF8\x88",
		.klen	= 20,
		.iv	= "\xCA\xFE\xDE\xBA\xCE\xFA\xCE\x74",
		.result	= "\x45\x00\x00\x49\x33\xBA\x00\x00"
			  "\x7F\x11\x91\x06\xC3\xFB\x1D\x10"
			  "\xC2\xB1\xD3\x26\xC0\x28\x31\xCE"
			  "\x00\x35\xDD\x7B\x80\x03\x02\xD5"
			  "\x00\x00\x4E\x20\x00\x1E\x8C\x18"
			  "\xD7\x5B\x81\xDC\x91\xBA\xA0\x47"
			  "\x6B\x91\xB9\x24\xB2\x80\x38\x9D"
			  "\x92\xC9\x63\xBA\xC0\x46\xEC\x95"
			  "\x9B\x62\x66\xC0\x47\x22\xB1\x49"
			  "\x23\x01\x01\x01",
		.rlen	= 76,
		.assoc	= "\x00\x00\x01\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x01\xCA\xFE\xDE\xBA"
			  "\xCE\xFA\xCE\x74",
		.alen	= 20,
		.input	= "\x18\xA6\xFD\x42\xF7\x2C\xBF\x4A"
			  "\xB2\xA2\xEA\x90\x1F\x73\xD8\x14"
			  "\xE3\xE7\xF2\x43\xD9\x54\x12\xE1"
			  "\xC3\x49\xC1\xD2\xFB\xEC\x16\x8F"
			  "\x91\x90\xFE\xEB\xAF\x2C\xB0\x19"
			  "\x84\xE6\x58\x63\x96\x5D\x74\x72"
			  "\xB7\x9D\xA3\x45\xE0\xE7\x80\x19"
			  "\x1F\x0D\x2F\x0E\x0F\x49\x6C\x22"
			  "\x6F\x21\x27\xB2\x7D\xB3\x57\x24"
			  "\xE7\x84\x5D\x68\x65\x1F\x57\xE6"
			  "\x5F\x35\x4F\x75\xFF\x17\x01\x57"
			  "\x69\x62\x34\x36",
		.ilen	= 92,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x73\x61\x6C\x74",
		.klen	= 36,
		.iv	= "\x61\x6E\x64\x01\x69\x76\x65\x63",
		.result	= "\x45\x08\x00\x28\x73\x2C\x00\x00"
			  "\x40\x06\xE9\xF9\x0A\x01\x06\x12"
			  "\x0A\x01\x03\x8F\x06\xB8\x80\x23"
			  "\xDD\x6B\xAF\xBE\xCB\x71\x26\x02"
			  "\x50\x10\x1F\x64\x6D\x54\x00\x01",
		.rlen	= 40,
		.assoc	= "\x17\x40\x5E\x67\x15\x6F\x31\x26"
			  "\xDD\x0D\xB9\x9B\x61\x6E\x64\x01"
			  "\x69\x76\x65\x63",
		.alen	= 20,
		.input	= "\xF2\xD6\x9E\xCD\xBD\x5A\x0D\x5B"
			  "\x8D\x5E\xF3\x8B\xAD\x4D\xA5\x8D"
			  "\x1F\x27\x8F\xDE\x98\xEF\x67\x54"
			  "\x9D\x52\x4A\x30\x18\xD9\xA5\x7F"
			  "\xF4\xD3\xA3\x1C\xE6\x73\x11\x9E"
			  "\x45\x16\x26\xC2\x41\x57\x71\xE3"
			  "\xB7\xEE\xBC\xA6\x14\xC8\x9B\x35",
		.ilen	= 56,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E\x43",
		.klen	= 20,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.result	= "\x45\x00\x00\x49\x33\x3E\x00\x00"
			  "\x7F\x11\x91\x82\xC3\xFB\x1D\x10"
			  "\xC2\xB1\xD3\x26\xC0\x28\x31\xCE"
			  "\x00\x35\xCB\x45\x80\x03\x02\x5B"
			  "\x00\x00\x01\xE0\x00\x1E\x8C\x18"
			  "\xD6\x57\x59\xD5\x22\x84\xA0\x35"
			  "\x2C\x71\x47\x5C\x88\x80\x39\x1C"
			  "\x76\x4D\x6E\x5E\xE0\x49\x6B\x32"
			  "\x5A\xE2\x70\xC0\x38\x99\x49\x39"
			  "\x15\x01\x01\x01",
		.rlen	= 76,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.input	= "\xFB\xA2\xCA\xD1\x2F\xC1\xF9\xF0"
			  "\x0D\x3C\xEB\xF3\x05\x41\x0D\xB8"
			  "\x3D\x77\x84\xB6\x07\x32\x3D\x22"
			  "\x0F\x24\xB0\xA9\x7D\x54\x18\x28"
			  "\x00\xCA\xDB\x0F\x68\xD9\x9E\xF0"
			  "\xE0\xC0\xC8\x9A\xE9\xBE\xA8\x88"
			  "\x4E\x52\xD6\x5B\xC1\xAF\xD0\x74"
			  "\x0F\x74\x24\x44\x74\x7B\x5B\x39"
			  "\xAB\x53\x31\x63\xAA\xD4\x55\x0E"
			  "\xE5\x16\x09\x75\xCD\xB6\x08\xC5"
			  "\x76\x91\x89\x60\x97\x63\xB8\xE1"
			  "\x8C\xAA\x81\xE2",
		.ilen	= 92,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x73\x61\x6C\x74",
		.klen	= 36,
		.iv	= "\x61\x6E\x64\x01\x69\x76\x65\x63",
		.result	= "\x63\x69\x73\x63\x6F\x01\x72\x75"
			  "\x6C\x65\x73\x01\x74\x68\x65\x01"
			  "\x6E\x65\x74\x77\x65\x01\x64\x65"
			  "\x66\x69\x6E\x65\x01\x74\x68\x65"
			  "\x74\x65\x63\x68\x6E\x6F\x6C\x6F"
			  "\x67\x69\x65\x73\x01\x74\x68\x61"
			  "\x74\x77\x69\x6C\x6C\x01\x64\x65"
			  "\x66\x69\x6E\x65\x74\x6F\x6D\x6F"
			  "\x72\x72\x6F\x77\x01\x02\x02\x01",
		.rlen	= 72,
		.assoc	= "\x17\x40\x5E\x67\x15\x6F\x31\x26"
			  "\xDD\x0D\xB9\x9B\x61\x6E\x64\x01"
			  "\x69\x76\x65\x63",
		.alen	= 20,
		.input	= "\xD4\xB7\xED\x86\xA1\x77\x7F\x2E"
			  "\xA1\x3D\x69\x73\xD3\x24\xC6\x9E"
			  "\x7B\x43\xF8\x26\xFB\x56\x83\x12"
			  "\x26\x50\x8B\xEB\xD2\xDC\xEB\x18"
			  "\xD0\xA6\xDF\x10\xE5\x48\x7D\xF0"
			  "\x74\x11\x3E\x14\xC6\x41\x02\x4E"
			  "\x3E\x67\x73\xD9\x1A\x62\xEE\x42"
			  "\x9B\x04\x3A\x10\xE3\xEF\xE6\xB0"
			  "\x12\xA4\x93\x63\x41\x23\x64\xF8"
			  "\xC0\xCA\xC5\x87\xF2\x49\xE5\x6B"
			  "\x11\xE2\x4F\x30\xE4\x4C\xCC\x76",
		.ilen	= 88,
	}, {
		.key	= "\x7D\x77\x3D\x00\xC1\x44\xC5\x25"
			  "\xAC\x61\x9D\x18\xC8\x4A\x3F\x47"
			  "\xD9\x66\x42\x67",
		.klen	= 20,
		.iv	= "\x43\x45\x7E\x91\x82\x44\x3B\xC6",
		.result	= "\x01\x02\x02\x01",
		.rlen	= 4,
		.assoc	= "\x33\x54\x67\xAE\xFF\xFF\xFF\xFF"
			  "\x43\x45\x7E\x91\x82\x44\x3B\xC6",
		.alen	= 16,
		.input	= "\x43\x7F\x86\x6B\xCB\x3F\x69\x9F"
			  "\xE9\xB0\x82\x2B\xAC\x96\x1C\x45"
			  "\x04\xBE\xF2\x70",
		.ilen	= 20,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xDE\xCA\xF8\x88",
		.klen	= 20,
		.iv	= "\xCA\xFE\xDE\xBA\xCE\xFA\xCE\x74",
		.result	= "\x74\x6F\x01\x62\x65\x01\x6F\x72"
			  "\x01\x6E\x6F\x74\x01\x74\x6F\x01"
			  "\x62\x65\x00\x01",
		.rlen	= 20,
		.assoc	= "\x00\x00\x01\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x01\xCA\xFE\xDE\xBA"
			  "\xCE\xFA\xCE\x74",
		.alen	= 20,
		.input	= "\x29\xC9\xFC\x69\xA1\x97\xD0\x38"
			  "\xCC\xDD\x14\xE2\xDD\xFC\xAA\x05"
			  "\x43\x33\x21\x64\x41\x25\x03\x52"
			  "\x43\x03\xED\x3C\x6C\x5F\x28\x38"
			  "\x43\xAF\x8C\x3E",
		.ilen	= 36,
	}, {
		.key	= "\x6C\x65\x67\x61\x6C\x69\x7A\x65"
			  "\x6D\x61\x72\x69\x6A\x75\x61\x6E"
			  "\x61\x61\x6E\x64\x64\x6F\x69\x74"
			  "\x62\x65\x66\x6F\x72\x65\x69\x61"
			  "\x74\x75\x72\x6E",
		.klen	= 36,
		.iv	= "\x33\x30\x21\x69\x67\x65\x74\x6D",
		.result	= "\x45\x00\x00\x30\xDA\x3A\x00\x00"
			  "\x80\x01\xDF\x3B\xC0\xA8\x00\x05"
			  "\xC0\xA8\x00\x01\x08\x00\xC6\xCD"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
		.assoc	= "\x79\x6B\x69\x63\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\x33\x30\x21\x69"
			  "\x67\x65\x74\x6D",
		.alen	= 20,
		.input	= "\xF9\x7A\xB2\xAA\x35\x6D\x8E\xDC"
			  "\xE1\x76\x44\xAC\x8C\x78\xE2\x5D"
			  "\xD2\x4D\xED\xBB\x29\xEB\xF1\xB6"
			  "\x4A\x27\x4B\x39\xB4\x9C\x3A\x86"
			  "\x4C\xD3\xD7\x8C\xA4\xAE\x68\xA3"
			  "\x2B\x42\x45\x8F\xB5\x7D\xBE\x82"
			  "\x1D\xCC\x63\xB9\xD0\x93\x7B\xA2"
			  "\x94\x5F\x66\x93\x68\x66\x1A\x32"
			  "\x9F\xB4\xC0\x53",
		.ilen	= 68,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E\x43",
		.klen	= 20,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.result	= "\x45\x00\x00\x30\xDA\x3A\x00\x00"
			  "\x80\x01\xDF\x3B\xC0\xA8\x00\x05"
			  "\xC0\xA8\x00\x01\x08\x00\xC6\xCD"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
		.assoc	= "\x3F\x7E\xF6\x42\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.input	= "\xFB\xA2\xCA\xA8\xC6\xC5\xF9\xF0"
			  "\xF2\x2C\xA5\x4A\x06\x12\x10\xAD"
			  "\x3F\x6E\x57\x91\xCF\x1A\xCA\x21"
			  "\x0D\x11\x7C\xEC\x9C\x35\x79\x17"
			  "\x65\xAC\xBD\x87\x01\xAD\x79\x84"
			  "\x5B\xF9\xFE\x3F\xBA\x48\x7B\xC9"
			  "\x63\x21\x93\x06\x84\xEE\xCA\xDB"
			  "\x56\x91\x25\x46\xE7\xA9\x5C\x97"
			  "\x40\xD7\xCB\x05",
		.ilen	= 68,
	}, {
		.key	= "\x4C\x80\xCD\xEF\xBB\x5D\x10\xDA"
			  "\x90\x6A\xC7\x3C\x36\x13\xA6\x34"
			  "\x22\x43\x3C\x64",
		.klen	= 20,
		.iv	= "\x48\x55\xEC\x7D\x3A\x23\x4B\xFD",
		.result	= "\x08\x00\xC6\xCD\x02\x00\x07\x00"
			  "\x61\x62\x63\x64\x65\x66\x67\x68"
			  "\x69\x6A\x6B\x6C\x6D\x6E\x6F\x70"
			  "\x71\x72\x73\x74\x01\x02\x02\x01",
		.rlen	= 32,
		.assoc	= "\x00\x00\x43\x21\x87\x65\x43\x21"
			  "\x00\x00\x00\x07\x48\x55\xEC\x7D"
			  "\x3A\x23\x4B\xFD",
		.alen	= 20,
		.input	= "\x74\x75\x2E\x8A\xEB\x5D\x87\x3C"
			  "\xD7\xC0\xF4\xAC\xC3\x6C\x4B\xFF"
			  "\x84\xB7\xD7\xB9\x8F\x0C\xA8\xB6"
			  "\xAC\xDA\x68\x94\xBC\x61\x90\x69"
			  "\xEF\x9C\xBC\x28\xFE\x1B\x56\xA7"
			  "\xC4\xE0\xD5\x8C\x86\xCD\x2B\xC0",
		.ilen	= 48,
	}
};

static struct aead_testvec aes_gcm_rfc4543_enc_tv_template[] = {
	{ /* From draft-mcgrew-gcm-test-01 */
		.key	= "\x4c\x80\xcd\xef\xbb\x5d\x10\xda"
			  "\x90\x6a\xc7\x3c\x36\x13\xa6\x34"
			  "\x22\x43\x3c\x64",
		.klen	= 20,
		.iv	= zeroed_string,
		.assoc	= "\x00\x00\x43\x21\x00\x00\x00\x07"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.input	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.ilen	= 52,
		.result	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01\xf2\xa9\xa8\x36"
			  "\xe1\x55\x10\x6a\xa8\xdc\xd6\x18"
			  "\xe4\x09\x9a\xaa",
		.rlen	= 68,
	}
};

static struct aead_testvec aes_gcm_rfc4543_dec_tv_template[] = {
	{ /* From draft-mcgrew-gcm-test-01 */
		.key	= "\x4c\x80\xcd\xef\xbb\x5d\x10\xda"
			  "\x90\x6a\xc7\x3c\x36\x13\xa6\x34"
			  "\x22\x43\x3c\x64",
		.klen	= 20,
		.iv	= zeroed_string,
		.assoc	= "\x00\x00\x43\x21\x00\x00\x00\x07"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.input	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01\xf2\xa9\xa8\x36"
			  "\xe1\x55\x10\x6a\xa8\xdc\xd6\x18"
			  "\xe4\x09\x9a\xaa",
		.ilen	= 68,
		.result	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
	}, { /* nearly same as previous, but should fail */
		.key	= "\x4c\x80\xcd\xef\xbb\x5d\x10\xda"
			  "\x90\x6a\xc7\x3c\x36\x13\xa6\x34"
			  "\x22\x43\x3c\x64",
		.klen	= 20,
		.iv	= zeroed_string,
		.assoc	= "\x00\x00\x43\x21\x00\x00\x00\x07"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.input	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01\xf2\xa9\xa8\x36"
			  "\xe1\x55\x10\x6a\xa8\xdc\xd6\x18"
			  "\x00\x00\x00\x00",
		.ilen	= 68,
		.novrfy = 1,
		.result	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
	},
};

static struct aead_testvec aes_ccm_enc_tv_template[] = {
	{ /* From RFC 3610 */
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x03\x02\x01\x00"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07",
		.alen	= 8,
		.input	= "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e",
		.ilen	= 23,
		.result	= "\x58\x8c\x97\x9a\x61\xc6\x63\xd2"
			  "\xf0\x66\xd0\xc2\xc0\xf9\x89\x80"
			  "\x6d\x5f\x6b\x61\xda\xc3\x84\x17"
			  "\xe8\xd1\x2c\xfd\xf9\x26\xe0",
		.rlen	= 31,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x07\x06\x05\x04"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b",
		.alen	= 12,
		.input	= "\x0c\x0d\x0e\x0f\x10\x11\x12\x13"
			  "\x14\x15\x16\x17\x18\x19\x1a\x1b"
			  "\x1c\x1d\x1e\x1f",
		.ilen	= 20,
		.result	= "\xdc\xf1\xfb\x7b\x5d\x9e\x23\xfb"
			  "\x9d\x4e\x13\x12\x53\x65\x8a\xd8"
			  "\x6e\xbd\xca\x3e\x51\xe8\x3f\x07"
			  "\x7d\x9c\x2d\x93",
		.rlen	= 28,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x0b\x0a\x09\x08"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07",
		.alen	= 8,
		.input	= "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20",
		.ilen	= 25,
		.result	= "\x82\x53\x1a\x60\xcc\x24\x94\x5a"
			  "\x4b\x82\x79\x18\x1a\xb5\xc8\x4d"
			  "\xf2\x1c\xe7\xf9\xb7\x3f\x42\xe1"
			  "\x97\xea\x9c\x07\xe5\x6b\x5e\xb1"
			  "\x7e\x5f\x4e",
		.rlen	= 35,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x0c\x0b\x0a\x09"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b",
		.alen	= 12,
		.input	= "\x0c\x0d\x0e\x0f\x10\x11\x12\x13"
			  "\x14\x15\x16\x17\x18\x19\x1a\x1b"
			  "\x1c\x1d\x1e",
		.ilen	= 19,
		.result	= "\x07\x34\x25\x94\x15\x77\x85\x15"
			  "\x2b\x07\x40\x98\x33\x0a\xbb\x14"
			  "\x1b\x94\x7b\x56\x6a\xa9\x40\x6b"
			  "\x4d\x99\x99\x88\xdd",
		.rlen	= 29,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\x33\x56\x8e\xf7\xb2\x63"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\x63\x01\x8f\x76\xdc\x8a\x1b\xcb",
		.alen	= 8,
		.input	= "\x90\x20\xea\x6f\x91\xbd\xd8\x5a"
			  "\xfa\x00\x39\xba\x4b\xaf\xf9\xbf"
			  "\xb7\x9c\x70\x28\x94\x9c\xd0\xec",
		.ilen	= 24,
		.result	= "\x4c\xcb\x1e\x7c\xa9\x81\xbe\xfa"
			  "\xa0\x72\x6c\x55\xd3\x78\x06\x12"
			  "\x98\xc8\x5c\x92\x81\x4a\xbc\x33"
			  "\xc5\x2e\xe8\x1d\x7d\x77\xc0\x8a",
		.rlen	= 32,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\xd5\x60\x91\x2d\x3f\x70"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\xcd\x90\x44\xd2\xb7\x1f\xdb\x81"
			  "\x20\xea\x60\xc0",
		.alen	= 12,
		.input	= "\x64\x35\xac\xba\xfb\x11\xa8\x2e"
			  "\x2f\x07\x1d\x7c\xa4\xa5\xeb\xd9"
			  "\x3a\x80\x3b\xa8\x7f",
		.ilen	= 21,
		.result	= "\x00\x97\x69\xec\xab\xdf\x48\x62"
			  "\x55\x94\xc5\x92\x51\xe6\x03\x57"
			  "\x22\x67\x5e\x04\xc8\x47\x09\x9e"
			  "\x5a\xe0\x70\x45\x51",
		.rlen	= 29,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\x42\xff\xf8\xf1\x95\x1c"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\xd8\x5b\xc7\xe6\x9f\x94\x4f\xb8",
		.alen	= 8,
		.input	= "\x8a\x19\xb9\x50\xbc\xf7\x1a\x01"
			  "\x8e\x5e\x67\x01\xc9\x17\x87\x65"
			  "\x98\x09\xd6\x7d\xbe\xdd\x18",
		.ilen	= 23,
		.result	= "\xbc\x21\x8d\xaa\x94\x74\x27\xb6"
			  "\xdb\x38\x6a\x99\xac\x1a\xef\x23"
			  "\xad\xe0\xb5\x29\x39\xcb\x6a\x63"
			  "\x7c\xf9\xbe\xc2\x40\x88\x97\xc6"
			  "\xba",
		.rlen	= 33,
	}, {
		/* This is taken from FIPS CAVS. */
		.key	= "\x83\xac\x54\x66\xc2\xeb\xe5\x05"
			  "\x2e\x01\xd1\xfc\x5d\x82\x66\x2e",
		.klen	= 16,
		.iv	= "\x03\x96\xac\x59\x30\x07\xa1\xe2\xa2\xc7\x55\x24\0\0\0\0",
		.alen	= 0,
		.input	= "\x19\xc8\x81\xf6\xe9\x86\xff\x93"
			  "\x0b\x78\x67\xe5\xbb\xb7\xfc\x6e"
			  "\x83\x77\xb3\xa6\x0c\x8c\x9f\x9c"
			  "\x35\x2e\xad\xe0\x62\xf9\x91\xa1",
		.ilen	= 32,
		.result	= "\xab\x6f\xe1\x69\x1d\x19\x99\xa8"
			  "\x92\xa0\xc4\x6f\x7e\xe2\x8b\xb1"
			  "\x70\xbb\x8c\xa6\x4c\x6e\x97\x8a"
			  "\x57\x2b\xbe\x5d\x98\xa6\xb1\x32"
			  "\xda\x24\xea\xd9\xa1\x39\x98\xfd"
			  "\xa4\xbe\xd9\xf2\x1a\x6d\x22\xa8",
		.rlen	= 48,
	}, {
		.key	= "\x1e\x2c\x7e\x01\x41\x9a\xef\xc0"
			  "\x0d\x58\x96\x6e\x5c\xa2\x4b\xd3",
		.klen	= 16,
		.iv	= "\x03\x4f\xa3\x19\xd3\x01\x5a\xd8"
			  "\x30\x60\x15\x56\x00\x00\x00\x00",
		.assoc	= "\xda\xe6\x28\x9c\x45\x2d\xfd\x63"
			  "\x5e\xda\x4c\xb6\xe6\xfc\xf9\xb7"
			  "\x0c\x56\xcb\xe4\xe0\x05\x7a\xe1"
			  "\x0a\x63\x09\x78\xbc\x2c\x55\xde",
		.alen	= 32,
		.input	= "\x87\xa3\x36\xfd\x96\xb3\x93\x78"
			  "\xa9\x28\x63\xba\x12\xa3\x14\x85"
			  "\x57\x1e\x06\xc9\x7b\x21\xef\x76"
			  "\x7f\x38\x7e\x8e\x29\xa4\x3e\x7e",
		.ilen	= 32,
		.result	= "\x8a\x1e\x11\xf0\x02\x6b\xe2\x19"
			  "\xfc\x70\xc4\x6d\x8e\xb7\x99\xab"
			  "\xc5\x4b\xa2\xac\xd3\xf3\x48\xff"
			  "\x3b\xb5\xce\x53\xef\xde\xbb\x02"
			  "\xa9\x86\x15\x6c\x13\xfe\xda\x0a"
			  "\x22\xb8\x29\x3d\xd8\x39\x9a\x23",
		.rlen	= 48,
	}, {
		.key	= "\xf4\x6b\xc2\x75\x62\xfe\xb4\xe1"
			  "\xa3\xf0\xff\xdd\x4e\x4b\x12\x75"
			  "\x53\x14\x73\x66\x8d\x88\xf6\x80",
		.klen	= 24,
		.iv	= "\x03\xa0\x20\x35\x26\xf2\x21\x8d"
			  "\x50\x20\xda\xe2\x00\x00\x00\x00",
		.assoc	= "\x5b\x9e\x13\x67\x02\x5e\xef\xc1"
			  "\x6c\xf9\xd7\x1e\x52\x8f\x7a\x47"
			  "\xe9\xd4\xcf\x20\x14\x6e\xf0\x2d"
			  "\xd8\x9e\x2b\x56\x10\x23\x56\xe7",
		.alen	= 32,
		.result	= "\x36\xea\x7a\x70\x08\xdc\x6a\xbc"
			  "\xad\x0c\x7a\x63\xf6\x61\xfd\x9b",
		.rlen	= 16,
	}, {
		.key	= "\x56\xdf\x5c\x8f\x26\x3f\x0e\x42"
			  "\xef\x7a\xd3\xce\xfc\x84\x60\x62"
			  "\xca\xb4\x40\xaf\x5f\xc9\xc9\x01",
		.klen	= 24,
		.iv	= "\x03\xd6\x3c\x8c\x86\x84\xb6\xcd"
			  "\xef\x09\x2e\x94\x00\x00\x00\x00",
		.assoc	= "\x02\x65\x78\x3c\xe9\x21\x30\x91"
			  "\xb1\xb9\xda\x76\x9a\x78\x6d\x95"
			  "\xf2\x88\x32\xa3\xf2\x50\xcb\x4c"
			  "\xe3\x00\x73\x69\x84\x69\x87\x79",
		.alen	= 32,
		.input	= "\x9f\xd2\x02\x4b\x52\x49\x31\x3c"
			  "\x43\x69\x3a\x2d\x8e\x70\xad\x7e"
			  "\xe0\xe5\x46\x09\x80\x89\x13\xb2"
			  "\x8c\x8b\xd9\x3f\x86\xfb\xb5\x6b",
		.ilen	= 32,
		.result	= "\x39\xdf\x7c\x3c\x5a\x29\xb9\x62"
			  "\x5d\x51\xc2\x16\xd8\xbd\x06\x9f"
			  "\x9b\x6a\x09\x70\xc1\x51\x83\xc2"
			  "\x66\x88\x1d\x4f\x9a\xda\xe0\x1e"
			  "\xc7\x79\x11\x58\xe5\x6b\x20\x40"
			  "\x7a\xea\x46\x42\x8b\xe4\x6f\xe1",
		.rlen	= 48,
	}, {
		.key	= "\xe0\x8d\x99\x71\x60\xd7\x97\x1a"
			  "\xbd\x01\x99\xd5\x8a\xdf\x71\x3a"
			  "\xd3\xdf\x24\x4b\x5e\x3d\x4b\x4e"
			  "\x30\x7a\xb9\xd8\x53\x0a\x5e\x2b",
		.klen	= 32,
		.iv	= "\x03\x1e\x29\x91\xad\x8e\xc1\x53"
			  "\x0a\xcf\x2d\xbe\x00\x00\x00\x00",
		.assoc	= "\x19\xb6\x1f\x57\xc4\xf3\xf0\x8b"
			  "\x78\x2b\x94\x02\x29\x0f\x42\x27"
			  "\x6b\x75\xcb\x98\x34\x08\x7e\x79"
			  "\xe4\x3e\x49\x0d\x84\x8b\x22\x87",
		.alen	= 32,
		.input	= "\xe1\xd9\xd8\x13\xeb\x3a\x75\x3f"
			  "\x9d\xbd\x5f\x66\xbe\xdc\xbb\x66"
			  "\xbf\x17\x99\x62\x4a\x39\x27\x1f"
			  "\x1d\xdc\x24\xae\x19\x2f\x98\x4c",
		.ilen	= 32,
		.result	= "\x19\xb8\x61\x33\x45\x2b\x43\x96"
			  "\x6f\x51\xd0\x20\x30\x7d\x9b\xc6"
			  "\x26\x3d\xf8\xc9\x65\x16\xa8\x9f"
			  "\xf0\x62\x17\x34\xf2\x1e\x8d\x75"
			  "\x4e\x13\xcc\xc0\xc3\x2a\x54\x2d",
		.rlen	= 40,
	}, {
		.key	= "\x7c\xc8\x18\x3b\x8d\x99\xe0\x7c"
			  "\x45\x41\xb8\xbd\x5c\xa7\xc2\x32"
			  "\x8a\xb8\x02\x59\xa4\xfe\xa9\x2c"
			  "\x09\x75\x9a\x9b\x3c\x9b\x27\x39",
		.klen	= 32,
		.iv	= "\x03\xf9\xd9\x4e\x63\xb5\x3d\x9d"
			  "\x43\xf6\x1e\x50\0\0\0\0",
		.assoc	= "\x57\xf5\x6b\x8b\x57\x5c\x3d\x3b"
			  "\x13\x02\x01\x0c\x83\x4c\x96\x35"
			  "\x8e\xd6\x39\xcf\x7d\x14\x9b\x94"
			  "\xb0\x39\x36\xe6\x8f\x57\xe0\x13",
		.alen	= 32,
		.input	= "\x3b\x6c\x29\x36\xb6\xef\x07\xa6"
			  "\x83\x72\x07\x4f\xcf\xfa\x66\x89"
			  "\x5f\xca\xb1\xba\xd5\x8f\x2c\x27"
			  "\x30\xdb\x75\x09\x93\xd4\x65\xe4",
		.ilen	= 32,
		.result	= "\xb0\x88\x5a\x33\xaa\xe5\xc7\x1d"
			  "\x85\x23\xc7\xc6\x2f\xf4\x1e\x3d"
			  "\xcc\x63\x44\x25\x07\x78\x4f\x9e"
			  "\x96\xb8\x88\xeb\xbc\x48\x1f\x06"
			  "\x39\xaf\x39\xac\xd8\x4a\x80\x39"
			  "\x7b\x72\x8a\xf7",
		.rlen	= 44,
	}, {
		.key	= "\xab\xd0\xe9\x33\x07\x26\xe5\x83"
			  "\x8c\x76\x95\xd4\xb6\xdc\xf3\x46"
			  "\xf9\x8f\xad\xe3\x02\x13\x83\x77"
			  "\x3f\xb0\xf1\xa1\xa1\x22\x0f\x2b",
		.klen	= 32,
		.iv	= "\x03\x24\xa7\x8b\x07\xcb\xcc\x0e"
			  "\xe6\x33\xbf\xf5\x00\x00\x00\x00",
		.assoc	= "\xd4\xdb\x30\x1d\x03\xfe\xfd\x5f"
			  "\x87\xd4\x8c\xb6\xb6\xf1\x7a\x5d"
			  "\xab\x90\x65\x8d\x8e\xca\x4d\x4f"
			  "\x16\x0c\x40\x90\x4b\xc7\x36\x73",
		.alen	= 32,
		.input	= "\xf5\xc6\x7d\x48\xc1\xb7\xe6\x92"
			  "\x97\x5a\xca\xc4\xa9\x6d\xf9\x3d"
			  "\x6c\xde\xbc\xf1\x90\xea\x6a\xb2"
			  "\x35\x86\x36\xaf\x5c\xfe\x4b\x3a",
		.ilen	= 32,
		.result	= "\x83\x6f\x40\x87\x72\xcf\xc1\x13"
			  "\xef\xbb\x80\x21\x04\x6c\x58\x09"
			  "\x07\x1b\xfc\xdf\xc0\x3f\x5b\xc7"
			  "\xe0\x79\xa8\x6e\x71\x7c\x3f\xcf"
			  "\x5c\xda\xb2\x33\xe5\x13\xe2\x0d"
			  "\x74\xd1\xef\xb5\x0f\x3a\xb5\xf8",
		.rlen	= 48,
	}
};

static struct aead_testvec aes_ccm_dec_tv_template[] = {
	{ /* From RFC 3610 */
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x03\x02\x01\x00"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07",
		.alen	= 8,
		.input	= "\x58\x8c\x97\x9a\x61\xc6\x63\xd2"
			  "\xf0\x66\xd0\xc2\xc0\xf9\x89\x80"
			  "\x6d\x5f\x6b\x61\xda\xc3\x84\x17"
			  "\xe8\xd1\x2c\xfd\xf9\x26\xe0",
		.ilen	= 31,
		.result	= "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e",
		.rlen	= 23,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x07\x06\x05\x04"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b",
		.alen	= 12,
		.input	= "\xdc\xf1\xfb\x7b\x5d\x9e\x23\xfb"
			  "\x9d\x4e\x13\x12\x53\x65\x8a\xd8"
			  "\x6e\xbd\xca\x3e\x51\xe8\x3f\x07"
			  "\x7d\x9c\x2d\x93",
		.ilen	= 28,
		.result	= "\x0c\x0d\x0e\x0f\x10\x11\x12\x13"
			  "\x14\x15\x16\x17\x18\x19\x1a\x1b"
			  "\x1c\x1d\x1e\x1f",
		.rlen	= 20,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x0b\x0a\x09\x08"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07",
		.alen	= 8,
		.input	= "\x82\x53\x1a\x60\xcc\x24\x94\x5a"
			  "\x4b\x82\x79\x18\x1a\xb5\xc8\x4d"
			  "\xf2\x1c\xe7\xf9\xb7\x3f\x42\xe1"
			  "\x97\xea\x9c\x07\xe5\x6b\x5e\xb1"
			  "\x7e\x5f\x4e",
		.ilen	= 35,
		.result	= "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20",
		.rlen	= 25,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x0c\x0b\x0a\x09"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b",
		.alen	= 12,
		.input	= "\x07\x34\x25\x94\x15\x77\x85\x15"
			  "\x2b\x07\x40\x98\x33\x0a\xbb\x14"
			  "\x1b\x94\x7b\x56\x6a\xa9\x40\x6b"
			  "\x4d\x99\x99\x88\xdd",
		.ilen	= 29,
		.result	= "\x0c\x0d\x0e\x0f\x10\x11\x12\x13"
			  "\x14\x15\x16\x17\x18\x19\x1a\x1b"
			  "\x1c\x1d\x1e",
		.rlen	= 19,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\x33\x56\x8e\xf7\xb2\x63"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\x63\x01\x8f\x76\xdc\x8a\x1b\xcb",
		.alen	= 8,
		.input	= "\x4c\xcb\x1e\x7c\xa9\x81\xbe\xfa"
			  "\xa0\x72\x6c\x55\xd3\x78\x06\x12"
			  "\x98\xc8\x5c\x92\x81\x4a\xbc\x33"
			  "\xc5\x2e\xe8\x1d\x7d\x77\xc0\x8a",
		.ilen	= 32,
		.result	= "\x90\x20\xea\x6f\x91\xbd\xd8\x5a"
			  "\xfa\x00\x39\xba\x4b\xaf\xf9\xbf"
			  "\xb7\x9c\x70\x28\x94\x9c\xd0\xec",
		.rlen	= 24,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\xd5\x60\x91\x2d\x3f\x70"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\xcd\x90\x44\xd2\xb7\x1f\xdb\x81"
			  "\x20\xea\x60\xc0",
		.alen	= 12,
		.input	= "\x00\x97\x69\xec\xab\xdf\x48\x62"
			  "\x55\x94\xc5\x92\x51\xe6\x03\x57"
			  "\x22\x67\x5e\x04\xc8\x47\x09\x9e"
			  "\x5a\xe0\x70\x45\x51",
		.ilen	= 29,
		.result	= "\x64\x35\xac\xba\xfb\x11\xa8\x2e"
			  "\x2f\x07\x1d\x7c\xa4\xa5\xeb\xd9"
			  "\x3a\x80\x3b\xa8\x7f",
		.rlen	= 21,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\x42\xff\xf8\xf1\x95\x1c"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\xd8\x5b\xc7\xe6\x9f\x94\x4f\xb8",
		.alen	= 8,
		.input	= "\xbc\x21\x8d\xaa\x94\x74\x27\xb6"
			  "\xdb\x38\x6a\x99\xac\x1a\xef\x23"
			  "\xad\xe0\xb5\x29\x39\xcb\x6a\x63"
			  "\x7c\xf9\xbe\xc2\x40\x88\x97\xc6"
			  "\xba",
		.ilen	= 33,
		.result	= "\x8a\x19\xb9\x50\xbc\xf7\x1a\x01"
			  "\x8e\x5e\x67\x01\xc9\x17\x87\x65"
			  "\x98\x09\xd6\x7d\xbe\xdd\x18",
		.rlen	= 23,
	}, {
		/* This is taken from FIPS CAVS. */
		.key	= "\xab\x2f\x8a\x74\xb7\x1c\xd2\xb1"
			  "\xff\x80\x2e\x48\x7d\x82\xf8\xb9",
		.klen	= 16,
		.iv	= "\x03\xc6\xfb\x7d\x80\x0d\x13\xab"
			  "\xd8\xa6\xb2\xd8\x00\x00\x00\x00",
		.alen	= 0,
		.input	= "\xd5\xe8\x93\x9f\xc7\x89\x2e\x2b",
		.ilen	= 8,
		.result	= "\x00",
		.rlen	= 0,
		.novrfy	= 1,
	}, {
		.key	= "\xab\x2f\x8a\x74\xb7\x1c\xd2\xb1"
			  "\xff\x80\x2e\x48\x7d\x82\xf8\xb9",
		.klen	= 16,
		.iv	= "\x03\xaf\x94\x87\x78\x35\x82\x81"
			  "\x7f\x88\x94\x68\x00\x00\x00\x00",
		.alen	= 0,
		.input	= "\x41\x3c\xb8\x87\x73\xcb\xf3\xf3",
		.ilen	= 8,
		.result	= "\x00",
		.rlen	= 0,
	}, {
		.key	= "\x61\x0e\x8c\xae\xe3\x23\xb6\x38"
			  "\x76\x1c\xf6\x3a\x67\xa3\x9c\xd8",
		.klen	= 16,
		.iv	= "\x03\xc6\xfb\x7d\x80\x0d\x13\xab"
			  "\xd8\xa6\xb2\xd8\x00\x00\x00\x00",
		.assoc	= "\xf3\x94\x87\x78\x35\x82\x81\x7f"
			  "\x88\x94\x68\xb1\x78\x6b\x2b\xd6"
			  "\x04\x1f\x4e\xed\x78\xd5\x33\x66"
			  "\xd8\x94\x99\x91\x81\x54\x62\x57",
		.alen	= 32,
		.input	= "\xf0\x7c\x29\x02\xae\x1c\x2f\x55"
			  "\xd0\xd1\x3d\x1a\xa3\x6d\xe4\x0a"
			  "\x86\xb0\x87\x6b\x62\x33\x8c\x34"
			  "\xce\xab\x57\xcc\x79\x0b\xe0\x6f"
			  "\x5c\x3e\x48\x1f\x6c\x46\xf7\x51"
			  "\x8b\x84\x83\x2a\xc1\x05\xb8\xc5",
		.ilen	= 48,
		.result	= "\x50\x82\x3e\x07\xe2\x1e\xb6\xfb"
			  "\x33\xe4\x73\xce\xd2\xfb\x95\x79"
			  "\xe8\xb4\xb5\x77\x11\x10\x62\x6f"
			  "\x6a\x82\xd1\x13\xec\xf5\xd0\x48",
		.rlen	= 32,
		.novrfy	= 1,
	}, {
		.key	= "\x61\x0e\x8c\xae\xe3\x23\xb6\x38"
			  "\x76\x1c\xf6\x3a\x67\xa3\x9c\xd8",
		.klen	= 16,
		.iv	= "\x03\x05\xe0\xc9\x0f\xed\x34\xea"
			  "\x97\xd4\x3b\xdf\x00\x00\x00\x00",
		.assoc	= "\x49\x5c\x50\x1f\x1d\x94\xcc\x81"
			  "\xba\xb7\xb6\x03\xaf\xa5\xc1\xa1"
			  "\xd8\x5c\x42\x68\xe0\x6c\xda\x89"
			  "\x05\xac\x56\xac\x1b\x2a\xd3\x86",
		.alen	= 32,
		.input	= "\x39\xbe\x7d\x15\x62\x77\xf3\x3c"
			  "\xad\x83\x52\x6d\x71\x03\x25\x1c"
			  "\xed\x81\x3a\x9a\x16\x7d\x19\x80"
			  "\x72\x04\x72\xd0\xf6\xff\x05\x0f"
			  "\xb7\x14\x30\x00\x32\x9e\xa0\xa6"
			  "\x9e\x5a\x18\xa1\xb8\xfe\xdb\xd3",
		.ilen	= 48,
		.result	= "\x75\x05\xbe\xc2\xd9\x1e\xde\x60"
			  "\x47\x3d\x8c\x7d\xbd\xb5\xd9\xb7"
			  "\xf2\xae\x61\x05\x8f\x82\x24\x3f"
			  "\x9c\x67\x91\xe1\x38\x4f\xe4\x0c",
		.rlen	= 32,
	}, {
		.key	= "\x39\xbb\xa7\xbe\x59\x97\x9e\x73"
			  "\xa2\xbc\x6b\x98\xd7\x75\x7f\xe3"
			  "\xa4\x48\x93\x39\x26\x71\x4a\xc6",
		.klen	= 24,
		.iv	= "\x03\xee\x49\x83\xe9\xa9\xff\xe9"
			  "\x57\xba\xfd\x9e\x00\x00\x00\x00",
		.assoc	= "\x44\xa6\x2c\x05\xe9\xe1\x43\xb1"
			  "\x58\x7c\xf2\x5c\x6d\x39\x0a\x64"
			  "\xa4\xf0\x13\x05\xd1\x77\x99\x67"
			  "\x11\xc4\xc6\xdb\x00\x56\x36\x61",
		.alen	= 32,
		.input	= "\x71\x99\xfa\xf4\x44\x12\x68\x9b",
		.ilen	= 8,
		.result	= "\x00",
		.rlen	= 0,
	}, {
		.key	= "\x58\x5d\xa0\x96\x65\x1a\x04\xd7"
			  "\x96\xe5\xc5\x68\xaa\x95\x35\xe0"
			  "\x29\xa0\xba\x9e\x48\x78\xd1\xba",
		.klen	= 24,
		.iv	= "\x03\xee\x49\x83\xe9\xa9\xff\xe9"
			  "\x57\xba\xfd\x9e\x00\x00\x00\x00",
		.assoc	= "\x44\xa6\x2c\x05\xe9\xe1\x43\xb1"
			  "\x58\x7c\xf2\x5c\x6d\x39\x0a\x64"
			  "\xa4\xf0\x13\x05\xd1\x77\x99\x67"
			  "\x11\xc4\xc6\xdb\x00\x56\x36\x61",
		.alen	= 32,
		.input	= "\xfb\xe5\x5d\x34\xbe\xe5\xe8\xe7"
			  "\x5a\xef\x2f\xbf\x1f\x7f\xd4\xb2"
			  "\x66\xca\x61\x1e\x96\x7a\x61\xb3"
			  "\x1c\x16\x45\x52\xba\x04\x9c\x9f"
			  "\xb1\xd2\x40\xbc\x52\x7c\x6f\xb1",
		.ilen	= 40,
		.result	= "\x85\x34\x66\x42\xc8\x92\x0f\x36"
			  "\x58\xe0\x6b\x91\x3c\x98\x5c\xbb"
			  "\x0a\x85\xcc\x02\xad\x7a\x96\xe9"
			  "\x65\x43\xa4\xc3\x0f\xdc\x55\x81",
		.rlen	= 32,
	}, {
		.key	= "\x58\x5d\xa0\x96\x65\x1a\x04\xd7"
			  "\x96\xe5\xc5\x68\xaa\x95\x35\xe0"
			  "\x29\xa0\xba\x9e\x48\x78\xd1\xba",
		.klen	= 24,
		.iv	= "\x03\xd1\xfc\x57\x9c\xfe\xb8\x9c"
			  "\xad\x71\xaa\x1f\x00\x00\x00\x00",
		.assoc	= "\x86\x67\xa5\xa9\x14\x5f\x0d\xc6"
			  "\xff\x14\xc7\x44\xbf\x6c\x3a\xc3"
			  "\xff\xb6\x81\xbd\xe2\xd5\x06\xc7"
			  "\x3c\xa1\x52\x13\x03\x8a\x23\x3a",
		.alen	= 32,
		.input	= "\x3f\x66\xb0\x9d\xe5\x4b\x38\x00"
			  "\xc6\x0e\x6e\xe5\xd6\x98\xa6\x37"
			  "\x8c\x26\x33\xc6\xb2\xa2\x17\xfa"
			  "\x64\x19\xc0\x30\xd7\xfc\x14\x6b"
			  "\xe3\x33\xc2\x04\xb0\x37\xbe\x3f"
			  "\xa9\xb4\x2d\x68\x03\xa3\x44\xef",
		.ilen	= 48,
		.result	= "\x02\x87\x4d\x28\x80\x6e\xb2\xed"
			  "\x99\x2a\xa8\xca\x04\x25\x45\x90"
			  "\x1d\xdd\x5a\xd9\xe4\xdb\x9c\x9c"
			  "\x49\xe9\x01\xfe\xa7\x80\x6d\x6b",
		.rlen	= 32,
		.novrfy	= 1,
	}, {
		.key	= "\xa4\x4b\x54\x29\x0a\xb8\x6d\x01"
			  "\x5b\x80\x2a\xcf\x25\xc4\xb7\x5c"
			  "\x20\x2c\xad\x30\xc2\x2b\x41\xfb"
			  "\x0e\x85\xbc\x33\xad\x0f\x2b\xff",
		.klen	= 32,
		.iv	= "\x03\xee\x49\x83\xe9\xa9\xff\xe9"
			  "\x57\xba\xfd\x9e\x00\x00\x00\x00",
		.alen	= 0,
		.input	= "\x1f\xb8\x8f\xa3\xdd\x54\x00\xf2",
		.ilen	= 8,
		.result	= "\x00",
		.rlen	= 0,
	}, {
		.key	= "\x39\xbb\xa7\xbe\x59\x97\x9e\x73"
			  "\xa2\xbc\x6b\x98\xd7\x75\x7f\xe3"
			  "\xa4\x48\x93\x39\x26\x71\x4a\xc6"
			  "\xae\x8f\x11\x4c\xc2\x9c\x4a\xbb",
		.klen	= 32,
		.iv	= "\x03\x85\x34\x66\x42\xc8\x92\x0f"
			  "\x36\x58\xe0\x6b\x00\x00\x00\x00",
		.alen	= 0,
		.input	= "\x48\x01\x5e\x02\x24\x04\x66\x47"
			  "\xa1\xea\x6f\xaf\xe8\xfc\xfb\xdd"
			  "\xa5\xa9\x87\x8d\x84\xee\x2e\x77"
			  "\xbb\x86\xb9\xf5\x5c\x6c\xff\xf6"
			  "\x72\xc3\x8e\xf7\x70\xb1\xb2\x07"
			  "\xbc\xa8\xa3\xbd\x83\x7c\x1d\x2a",
		.ilen	= 48,
		.result	= "\xdc\x56\xf2\x71\xb0\xb1\xa0\x6c"
			  "\xf0\x97\x3a\xfb\x6d\xe7\x32\x99"
			  "\x3e\xaf\x70\x5e\xb2\x4d\xea\x39"
			  "\x89\xd4\x75\x7a\x63\xb1\xda\x93",
		.rlen	= 32,
		.novrfy	= 1,
	}, {
		.key	= "\x58\x5d\xa0\x96\x65\x1a\x04\xd7"
			  "\x96\xe5\xc5\x68\xaa\x95\x35\xe0"
			  "\x29\xa0\xba\x9e\x48\x78\xd1\xba"
			  "\x0d\x1a\x53\x3b\xb5\xe3\xf8\x8b",
		.klen	= 32,
		.iv	= "\x03\xcf\x76\x3f\xd9\x95\x75\x8f"
			  "\x44\x89\x40\x7b\x00\x00\x00\x00",
		.assoc	= "\x8f\x86\x6c\x4d\x1d\xc5\x39\x88"
			  "\xc8\xf3\x5c\x52\x10\x63\x6f\x2b"
			  "\x8a\x2a\xc5\x6f\x30\x23\x58\x7b"
			  "\xfb\x36\x03\x11\xb4\xd9\xf2\xfe",
		.alen	= 32,
		.input	= "\x48\x58\xd6\xf3\xad\x63\x58\xbf"
			  "\xae\xc7\x5e\xae\x83\x8f\x7b\xe4"
			  "\x78\x5c\x4c\x67\x71\x89\x94\xbf"
			  "\x47\xf1\x63\x7e\x1c\x59\xbd\xc5"
			  "\x7f\x44\x0a\x0c\x01\x18\x07\x92"
			  "\xe1\xd3\x51\xce\x32\x6d\x0c\x5b",
		.ilen	= 48,
		.result	= "\xc2\x54\xc8\xde\x78\x87\x77\x40"
			  "\x49\x71\xe4\xb7\xe7\xcb\x76\x61"
			  "\x0a\x41\xb9\xe9\xc0\x76\x54\xab"
			  "\x04\x49\x3b\x19\x93\x57\x25\x5d",
		.rlen	= 32,
	},
};

/*
 * rfc4309 refers to section 8 of rfc3610 for test vectors, but they all
 * use a 13-byte nonce, we only support an 11-byte nonce.  Worse,
 * they use AD lengths which are not valid ESP header lengths.
 *
 * These vectors are copied/generated from the ones for rfc4106 with
 * the key truncated by one byte..
 */
static struct aead_testvec aes_ccm_rfc4309_enc_tv_template[] = {
	{ /* Generated using Crypto++ */
		.key	= zeroed_string,
		.klen	= 19,
		.iv	= zeroed_string,
		.input	= zeroed_string,
		.ilen	= 16,
		.assoc	= zeroed_string,
		.alen	= 16,
		.result	= "\x2E\x9A\xCA\x6B\xDA\x54\xFC\x6F"
			  "\x12\x50\xE8\xDE\x81\x3C\x63\x08"
			  "\x1A\x22\xBA\x75\xEE\xD4\xD5\xB5"
			  "\x27\x50\x01\xAC\x03\x33\x39\xFB",
		.rlen	= 32,
	},{
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= zeroed_string,
		.ilen	= 16,
		.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen	= 16,
		.result	= "\xCF\xB9\x99\x17\xC8\x86\x0E\x7F"
			  "\x7E\x76\xF8\xE6\xF8\xCC\x1F\x17"
			  "\x6A\xE0\x53\x9F\x4B\x73\x7E\xDA"
			  "\x08\x09\x4E\xC4\x1E\xAD\xC6\xB0",
		.rlen	= 32,

	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= zeroed_string,
		.input	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.ilen	= 16,
		.assoc	= zeroed_string,
		.alen	= 16,
		.result	= "\x33\xDE\x73\xBC\xA6\xCE\x4E\xA6"
			  "\x61\xF4\xF5\x41\x03\x4A\xE3\x86"
			  "\xA1\xE2\xC2\x42\x2B\x81\x70\x40"
			  "\xFD\x7F\x76\xD1\x03\x07\xBB\x0C",
		.rlen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= zeroed_string,
		.input	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.ilen	= 16,
		.assoc	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.result	= "\x33\xDE\x73\xBC\xA6\xCE\x4E\xA6"
			  "\x61\xF4\xF5\x41\x03\x4A\xE3\x86"
			  "\x5B\xC0\x73\xE0\x2B\x73\x68\xC9"
			  "\x2D\x8C\x58\xC2\x90\x3D\xB0\x3E",
		.rlen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.ilen	= 16,
		.assoc	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen	= 16,
		.result	= "\xCE\xB8\x98\x16\xC9\x87\x0F\x7E"
			  "\x7F\x77\xF9\xE7\xF9\xCD\x1E\x16"
			  "\x43\x8E\x76\x57\x3B\xB4\x05\xE8"
			  "\xA9\x9B\xBF\x25\xE0\x4F\xC0\xED",
		.rlen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.ilen	= 64,
		.assoc	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen	= 16,
		.result	= "\xCE\xB8\x98\x16\xC9\x87\x0F\x7E"
			  "\x7F\x77\xF9\xE7\xF9\xCD\x1E\x16"
			  "\x9C\xA4\x97\x83\x3F\x01\xA5\xF4"
			  "\x43\x09\xE7\xB8\xE9\xD1\xD7\x02"
			  "\x9B\xAB\x39\x18\xEB\x94\x34\x36"
			  "\xE6\xC5\xC8\x9B\x00\x81\x9E\x49"
			  "\x1D\x78\xE1\x48\xE3\xE9\xEA\x8E"
			  "\x3A\x2B\x67\x5D\x35\x6A\x0F\xDB"
			  "\x02\x73\xDD\xE7\x30\x4A\x30\x54"
			  "\x1A\x9D\x09\xCA\xC8\x1C\x32\x5F",
		.rlen	= 80,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x45\x67\x89\xab\xcd\xef",
		.input	= "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff",
		.ilen	= 192,
		.assoc	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\x00\x00\x45\x67"
			  "\x89\xab\xcd\xef",
		.alen	= 20,
		.result	= "\x64\x17\xDC\x24\x9D\x92\xBA\x5E"
			  "\x7C\x64\x6D\x33\x46\x77\xAC\xB1"
			  "\x5C\x9E\xE2\xC7\x27\x11\x3E\x95"
			  "\x7D\xBE\x28\xC8\xC1\xCA\x5E\x8C"
			  "\xB4\xE2\xDE\x9F\x53\x59\x26\xDB"
			  "\x0C\xD4\xE4\x07\x9A\xE6\x3E\x01"
			  "\x58\x0D\x3E\x3D\xD5\x21\xEB\x04"
			  "\x06\x9D\x5F\xB9\x02\x49\x1A\x2B"
			  "\xBA\xF0\x4E\x3B\x85\x50\x5B\x09"
			  "\xFE\xEC\xFC\x54\xEC\x0C\xE2\x79"
			  "\x8A\x2F\x5F\xD7\x05\x5D\xF1\x6D"
			  "\x22\xEB\xD1\x09\x80\x3F\x5A\x70"
			  "\xB2\xB9\xD3\x63\x99\xC2\x4D\x1B"
			  "\x36\x12\x00\x89\xAA\x5D\x55\xDA"
			  "\x1D\x5B\xD8\x3C\x5F\x09\xD2\xE6"
			  "\x39\x41\x5C\xF0\xBE\x26\x4E\x5F"
			  "\x2B\x50\x44\x52\xC2\x10\x7D\x38"
			  "\x82\x64\x83\x0C\xAE\x49\xD0\xE5"
			  "\x4F\xE5\x66\x4C\x58\x7A\xEE\x43"
			  "\x3B\x51\xFE\xBA\x24\x8A\xFE\xDC"
			  "\x19\x6D\x60\x66\x61\xF9\x9A\x3F"
			  "\x75\xFC\x38\x53\x5B\xB5\xCD\x52"
			  "\x4F\xE5\xE4\xC9\xFE\x10\xCB\x98"
			  "\xF0\x06\x5B\x07\xAB\xBB\xF4\x0E"
			  "\x2D\xC2\xDD\x5D\xDD\x22\x9A\xCC"
			  "\x39\xAB\x63\xA5\x3D\x9C\x51\x8A",
		.rlen	= 208,
	}, { /* From draft-mcgrew-gcm-test-01 */
		.key	= "\x4C\x80\xCD\xEF\xBB\x5D\x10\xDA"
			  "\x90\x6A\xC7\x3C\x36\x13\xA6\x34"
			  "\x2E\x44\x3B",
		.klen	= 19,
		.iv	= "\x49\x56\xED\x7E\x3B\x24\x4C\xFE",
		.input	= "\x45\x00\x00\x48\x69\x9A\x00\x00"
			  "\x80\x11\x4D\xB7\xC0\xA8\x01\x02"
			  "\xC0\xA8\x01\x01\x0A\x9B\xF1\x56"
			  "\x38\xD3\x01\x00\x00\x01\x00\x00"
			  "\x00\x00\x00\x00\x04\x5F\x73\x69"
			  "\x70\x04\x5F\x75\x64\x70\x03\x73"
			  "\x69\x70\x09\x63\x79\x62\x65\x72"
			  "\x63\x69\x74\x79\x02\x64\x6B\x00"
			  "\x00\x21\x00\x01\x01\x02\x02\x01",
		.ilen	= 72,
		.assoc	= "\x00\x00\x43\x21\x87\x65\x43\x21"
			  "\x00\x00\x00\x00\x49\x56\xED\x7E"
			  "\x3B\x24\x4C\xFE",
		.alen	= 20,
		.result	= "\x89\xBA\x3E\xEF\xE6\xD6\xCF\xDB"
			  "\x83\x60\xF5\xBA\x3A\x56\x79\xE6"
			  "\x7E\x0C\x53\xCF\x9E\x87\xE0\x4E"
			  "\x1A\x26\x01\x24\xC7\x2E\x3D\xBF"
			  "\x29\x2C\x91\xC1\xB8\xA8\xCF\xE0"
			  "\x39\xF8\x53\x6D\x31\x22\x2B\xBF"
			  "\x98\x81\xFC\x34\xEE\x85\x36\xCD"
			  "\x26\xDB\x6C\x7A\x0C\x77\x8A\x35"
			  "\x18\x85\x54\xB2\xBC\xDD\x3F\x43"
			  "\x61\x06\x8A\xDF\x86\x3F\xB4\xAC"
			  "\x97\xDC\xBD\xFD\x92\x10\xC5\xFF",
		.rlen	= 88,
	}, {
		.key	= "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\x6D\x6A\x8F\x94\x67\x30\x83\x08"
			  "\xCA\xFE\xBA",
		.klen	= 19,
		.iv	= "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.input	= "\x45\x00\x00\x3E\x69\x8F\x00\x00"
			  "\x80\x11\x4D\xCC\xC0\xA8\x01\x02"
			  "\xC0\xA8\x01\x01\x0A\x98\x00\x35"
			  "\x00\x2A\x23\x43\xB2\xD0\x01\x00"
			  "\x00\x01\x00\x00\x00\x00\x00\x00"
			  "\x03\x73\x69\x70\x09\x63\x79\x62"
			  "\x65\x72\x63\x69\x74\x79\x02\x64"
			  "\x6B\x00\x00\x01\x00\x01\x00\x01",
		.ilen	= 64,
		.assoc	= "\x00\x00\xA5\xF8\x00\x00\x00\x0A"
			  "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.alen	= 16,
		.result	= "\x4B\xC2\x70\x60\x64\xD2\xF3\xC8"
			  "\xE5\x26\x8A\xDE\xB8\x7E\x7D\x16"
			  "\x56\xC7\xD2\x88\xBA\x8D\x58\xAF"
			  "\xF5\x71\xB6\x37\x84\xA7\xB1\x99"
			  "\x51\x5C\x0D\xA0\x27\xDE\xE7\x2D"
			  "\xEF\x25\x88\x1F\x1D\x77\x11\xFF"
			  "\xDB\xED\xEE\x56\x16\xC5\x5C\x9B"
			  "\x00\x62\x1F\x68\x4E\x7C\xA0\x97"
			  "\x10\x72\x7E\x53\x13\x3B\x68\xE4"
			  "\x30\x99\x91\x79\x09\xEA\xFF\x6A",
		.rlen	= 80,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x11\x22\x33",
		.klen	= 35,
		.iv	= "\x01\x02\x03\x04\x05\x06\x07\x08",
		.input	= "\x45\x00\x00\x30\x69\xA6\x40\x00"
			  "\x80\x06\x26\x90\xC0\xA8\x01\x02"
			  "\x93\x89\x15\x5E\x0A\x9E\x00\x8B"
			  "\x2D\xC5\x7E\xE0\x00\x00\x00\x00"
			  "\x70\x02\x40\x00\x20\xBF\x00\x00"
			  "\x02\x04\x05\xB4\x01\x01\x04\x02"
			  "\x01\x02\x02\x01",
		.ilen	= 52,
		.assoc	= "\x4A\x2C\xBF\xE3\x00\x00\x00\x02"
			  "\x01\x02\x03\x04\x05\x06\x07\x08",
		.alen	= 16,
		.result	= "\xD6\x31\x0D\x2B\x3D\x6F\xBD\x2F"
			  "\x58\x41\x7E\xFF\x9A\x9E\x09\xB4"
			  "\x1A\xF7\xF6\x42\x31\xCD\xBF\xAD"
			  "\x27\x0E\x2C\xF2\xDB\x10\xDF\x55"
			  "\x8F\x0D\xD7\xAC\x23\xBD\x42\x10"
			  "\xD0\xB2\xAF\xD8\x37\xAC\x6B\x0B"
			  "\x11\xD4\x0B\x12\xEC\xB4\xB1\x92"
			  "\x23\xA6\x10\xB0\x26\xD6\xD9\x26"
			  "\x5A\x48\x6A\x3E",
		.rlen	= 68,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x45\x00\x00\x3C\x99\xC5\x00\x00"
			  "\x80\x01\xCB\x7A\x40\x67\x93\x18"
			  "\x01\x01\x01\x01\x08\x00\x07\x5C"
			  "\x02\x00\x44\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x75\x76\x77\x61\x62\x63\x64\x65"
			  "\x66\x67\x68\x69\x01\x02\x02\x01",
		.ilen	= 64,
		.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.result	= "\x6B\x9A\xCA\x57\x43\x91\xFC\x6F"
			  "\x92\x51\x23\xA4\xC1\x5B\xF0\x10"
			  "\xF3\x13\xF4\xF8\xA1\x9A\xB4\xDC"
			  "\x89\xC8\xF8\x42\x62\x95\xB7\xCB"
			  "\xB8\xF5\x0F\x1B\x2E\x94\xA2\xA7"
			  "\xBF\xFB\x8A\x92\x13\x63\xD1\x3C"
			  "\x08\xF5\xE8\xA6\xAA\xF6\x34\xF9"
			  "\x42\x05\xAF\xB3\xE7\x9A\xFC\xEE"
			  "\x36\x25\xC1\x10\x12\x1C\xCA\x82"
			  "\xEA\xE6\x63\x5A\x57\x28\xA9\x9A",
		.rlen	= 80,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E",
		.klen	= 19,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.input	= "\x45\x00\x00\x3C\x99\xC3\x00\x00"
			  "\x80\x01\xCB\x7C\x40\x67\x93\x18"
			  "\x01\x01\x01\x01\x08\x00\x08\x5C"
			  "\x02\x00\x43\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x75\x76\x77\x61\x62\x63\x64\x65"
			  "\x66\x67\x68\x69\x01\x02\x02\x01",
		.ilen	= 64,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.result	= "\x6A\x6B\x45\x2B\x7C\x67\x52\xF6"
			  "\x10\x60\x40\x62\x6B\x4F\x97\x8E"
			  "\x0B\xB2\x22\x97\xCB\x21\xE0\x90"
			  "\xA2\xE7\xD1\x41\x30\xE4\x4B\x1B"
			  "\x79\x01\x58\x50\x01\x06\xE1\xE0"
			  "\x2C\x83\x79\xD3\xDE\x46\x97\x1A"
			  "\x30\xB8\xE5\xDF\xD7\x12\x56\x75"
			  "\xD0\x95\xB7\xB8\x91\x42\xF7\xFD"
			  "\x97\x57\xCA\xC1\x20\xD0\x86\xB9"
			  "\x66\x9D\xB4\x2B\x96\x22\xAC\x67",
		.rlen	= 80,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E",
		.klen	= 19,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.input	= "\x45\x00\x00\x1C\x42\xA2\x00\x00"
			  "\x80\x01\x44\x1F\x40\x67\x93\xB6"
			  "\xE0\x00\x00\x02\x0A\x00\xF5\xFF"
			  "\x01\x02\x02\x01",
		.ilen	= 28,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.result	= "\x6A\x6B\x45\x0B\xA7\x06\x52\xF6"
			  "\x10\x60\xCF\x01\x6B\x4F\x97\x20"
			  "\xEA\xB3\x23\x94\xC9\x21\x1D\x33"
			  "\xA1\xE5\x90\x40\x05\x37\x45\x70"
			  "\xB5\xD6\x09\x0A\x23\x73\x33\xF9"
			  "\x08\xB4\x22\xE4",
		.rlen	= 44,
	}, {
		.key	= "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\x6D\x6A\x8F\x94\x67\x30\x83\x08"
			  "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\xCA\xFE\xBA",
		.klen	= 27,
		.iv	= "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.input	= "\x45\x00\x00\x28\xA4\xAD\x40\x00"
			  "\x40\x06\x78\x80\x0A\x01\x03\x8F"
			  "\x0A\x01\x06\x12\x80\x23\x06\xB8"
			  "\xCB\x71\x26\x02\xDD\x6B\xB0\x3E"
			  "\x50\x10\x16\xD0\x75\x68\x00\x01",
		.ilen	= 40,
		.assoc	= "\x00\x00\xA5\xF8\x00\x00\x00\x0A"
			  "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.alen	= 16,
		.result	= "\x05\x22\x15\xD1\x52\x56\x85\x04"
			  "\xA8\x5C\x5D\x6D\x7E\x6E\xF5\xFA"
			  "\xEA\x16\x37\x50\xF3\xDF\x84\x3B"
			  "\x2F\x32\x18\x57\x34\x2A\x8C\x23"
			  "\x67\xDF\x6D\x35\x7B\x54\x0D\xFB"
			  "\x34\xA5\x9F\x6C\x48\x30\x1E\x22"
			  "\xFE\xB1\x22\x17\x17\x8A\xB9\x5B",
		.rlen	= 56,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xDE\xCA\xF8",
		.klen	= 19,
		.iv	= "\xCA\xFE\xDE\xBA\xCE\xFA\xCE\x74",
		.input	= "\x45\x00\x00\x49\x33\xBA\x00\x00"
			  "\x7F\x11\x91\x06\xC3\xFB\x1D\x10"
			  "\xC2\xB1\xD3\x26\xC0\x28\x31\xCE"
			  "\x00\x35\xDD\x7B\x80\x03\x02\xD5"
			  "\x00\x00\x4E\x20\x00\x1E\x8C\x18"
			  "\xD7\x5B\x81\xDC\x91\xBA\xA0\x47"
			  "\x6B\x91\xB9\x24\xB2\x80\x38\x9D"
			  "\x92\xC9\x63\xBA\xC0\x46\xEC\x95"
			  "\x9B\x62\x66\xC0\x47\x22\xB1\x49"
			  "\x23\x01\x01\x01",
		.ilen	= 76,
		.assoc	= "\x00\x00\x01\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x01\xCA\xFE\xDE\xBA"
			  "\xCE\xFA\xCE\x74",
		.alen	= 20,
		.result	= "\x92\xD0\x53\x79\x33\x38\xD5\xF3"
			  "\x7D\xE4\x7A\x8E\x86\x03\xC9\x90"
			  "\x96\x35\xAB\x9C\xFB\xE8\xA3\x76"
			  "\xE9\xE9\xE2\xD1\x2E\x11\x0E\x00"
			  "\xFA\xCE\xB5\x9E\x02\xA7\x7B\xEA"
			  "\x71\x9A\x58\xFB\xA5\x8A\xE1\xB7"
			  "\x9C\x39\x9D\xE3\xB5\x6E\x69\xE6"
			  "\x63\xC9\xDB\x05\x69\x51\x12\xAD"
			  "\x3E\x00\x32\x73\x86\xF2\xEE\xF5"
			  "\x0F\xE8\x81\x7E\x84\xD3\xC0\x0D"
			  "\x76\xD6\x55\xC6\xB4\xC2\x34\xC7"
			  "\x12\x25\x0B\xF9",
		.rlen	= 92,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x73\x61\x6C",
		.klen	= 35,
		.iv	= "\x61\x6E\x64\x01\x69\x76\x65\x63",
		.input	= "\x45\x08\x00\x28\x73\x2C\x00\x00"
			  "\x40\x06\xE9\xF9\x0A\x01\x06\x12"
			  "\x0A\x01\x03\x8F\x06\xB8\x80\x23"
			  "\xDD\x6B\xAF\xBE\xCB\x71\x26\x02"
			  "\x50\x10\x1F\x64\x6D\x54\x00\x01",
		.ilen	= 40,
		.assoc	= "\x17\x40\x5E\x67\x15\x6F\x31\x26"
			  "\xDD\x0D\xB9\x9B\x61\x6E\x64\x01"
			  "\x69\x76\x65\x63",
		.alen	= 20,
		.result	= "\xCC\x74\xB7\xD3\xB0\x38\x50\x42"
			  "\x2C\x64\x87\x46\x1E\x34\x10\x05"
			  "\x29\x6B\xBB\x36\xE9\x69\xAD\x92"
			  "\x82\xA1\x10\x6A\xEB\x0F\xDC\x7D"
			  "\x08\xBA\xF3\x91\xCA\xAA\x61\xDA"
			  "\x62\xF4\x14\x61\x5C\x9D\xB5\xA7"
			  "\xEE\xD7\xB9\x7E\x87\x99\x9B\x7D",
		.rlen	= 56,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E",
		.klen	= 19,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.input	= "\x45\x00\x00\x49\x33\x3E\x00\x00"
			  "\x7F\x11\x91\x82\xC3\xFB\x1D\x10"
			  "\xC2\xB1\xD3\x26\xC0\x28\x31\xCE"
			  "\x00\x35\xCB\x45\x80\x03\x02\x5B"
			  "\x00\x00\x01\xE0\x00\x1E\x8C\x18"
			  "\xD6\x57\x59\xD5\x22\x84\xA0\x35"
			  "\x2C\x71\x47\x5C\x88\x80\x39\x1C"
			  "\x76\x4D\x6E\x5E\xE0\x49\x6B\x32"
			  "\x5A\xE2\x70\xC0\x38\x99\x49\x39"
			  "\x15\x01\x01\x01",
		.ilen	= 76,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.result	= "\x6A\x6B\x45\x5E\xD6\x9A\x52\xF6"
			  "\xEF\x70\x1A\x9C\xE8\xD3\x19\x86"
			  "\xC8\x02\xF0\xB0\x03\x09\xD9\x02"
			  "\xA0\xD2\x59\x04\xD1\x85\x2A\x24"
			  "\x1C\x67\x3E\xD8\x68\x72\x06\x94"
			  "\x97\xBA\x4F\x76\x8D\xB0\x44\x5B"
			  "\x69\xBF\xD5\xE2\x3D\xF1\x0B\x0C"
			  "\xC0\xBF\xB1\x8F\x70\x09\x9E\xCE"
			  "\xA5\xF2\x55\x58\x84\xFA\xF9\xB5"
			  "\x23\xF4\x84\x40\x74\x14\x8A\x6B"
			  "\xDB\xD7\x67\xED\xA4\x93\xF3\x47"
			  "\xCC\xF7\x46\x6F",
		.rlen	= 92,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x73\x61\x6C",
		.klen	= 35,
		.iv	= "\x61\x6E\x64\x01\x69\x76\x65\x63",
		.input	= "\x63\x69\x73\x63\x6F\x01\x72\x75"
			  "\x6C\x65\x73\x01\x74\x68\x65\x01"
			  "\x6E\x65\x74\x77\x65\x01\x64\x65"
			  "\x66\x69\x6E\x65\x01\x74\x68\x65"
			  "\x74\x65\x63\x68\x6E\x6F\x6C\x6F"
			  "\x67\x69\x65\x73\x01\x74\x68\x61"
			  "\x74\x77\x69\x6C\x6C\x01\x64\x65"
			  "\x66\x69\x6E\x65\x74\x6F\x6D\x6F"
			  "\x72\x72\x6F\x77\x01\x02\x02\x01",
		.ilen	= 72,
		.assoc	= "\x17\x40\x5E\x67\x15\x6F\x31\x26"
			  "\xDD\x0D\xB9\x9B\x61\x6E\x64\x01"
			  "\x69\x76\x65\x63",
		.alen	= 20,
		.result	= "\xEA\x15\xC4\x98\xAC\x15\x22\x37"
			  "\x00\x07\x1D\xBE\x60\x5D\x73\x16"
			  "\x4D\x0F\xCC\xCE\x8A\xD0\x49\xD4"
			  "\x39\xA3\xD1\xB1\x21\x0A\x92\x1A"
			  "\x2C\xCF\x8F\x9D\xC9\x91\x0D\xB4"
			  "\x15\xFC\xBC\xA5\xC5\xBF\x54\xE5"
			  "\x1C\xC7\x32\x41\x07\x7B\x2C\xB6"
			  "\x5C\x23\x7C\x93\xEA\xEF\x23\x1C"
			  "\x73\xF4\xE7\x12\x84\x4C\x37\x0A"
			  "\x4A\x8F\x06\x37\x48\xF9\xF9\x05"
			  "\x55\x13\x40\xC3\xD5\x55\x3A\x3D",
		.rlen	= 88,
	}, {
		.key	= "\x7D\x77\x3D\x00\xC1\x44\xC5\x25"
			  "\xAC\x61\x9D\x18\xC8\x4A\x3F\x47"
			  "\xD9\x66\x42",
		.klen	= 19,
		.iv	= "\x43\x45\x7E\x91\x82\x44\x3B\xC6",
		.input	= "\x01\x02\x02\x01",
		.ilen	= 4,
		.assoc	= "\x33\x54\x67\xAE\xFF\xFF\xFF\xFF"
			  "\x43\x45\x7E\x91\x82\x44\x3B\xC6",
		.alen	= 16,
		.result	= "\x4C\x72\x63\x30\x2F\xE6\x56\xDD"
			  "\xD0\xD8\x60\x9D\x8B\xEF\x85\x90"
			  "\xF7\x61\x24\x62",
		.rlen	= 20,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xDE\xCA\xF8",
		.klen	= 19,
		.iv	= "\xCA\xFE\xDE\xBA\xCE\xFA\xCE\x74",
		.input	= "\x74\x6F\x01\x62\x65\x01\x6F\x72"
			  "\x01\x6E\x6F\x74\x01\x74\x6F\x01"
			  "\x62\x65\x00\x01",
		.ilen	= 20,
		.assoc	= "\x00\x00\x01\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x01\xCA\xFE\xDE\xBA"
			  "\xCE\xFA\xCE\x74",
		.alen	= 20,
		.result	= "\xA3\xBF\x52\x52\x65\x83\xBA\x81"
			  "\x03\x9B\x84\xFC\x44\x8C\xBB\x81"
			  "\x36\xE1\x78\xBB\xA5\x49\x3A\xD0"
			  "\xF0\x6B\x21\xAF\x98\xC0\x34\xDC"
			  "\x17\x17\x65\xAD",
		.rlen	= 36,
	}, {
		.key	= "\x6C\x65\x67\x61\x6C\x69\x7A\x65"
			  "\x6D\x61\x72\x69\x6A\x75\x61\x6E"
			  "\x61\x61\x6E\x64\x64\x6F\x69\x74"
			  "\x62\x65\x66\x6F\x72\x65\x69\x61"
			  "\x74\x75\x72",
		.klen	= 35,
		.iv	= "\x33\x30\x21\x69\x67\x65\x74\x6D",
		.input	= "\x45\x00\x00\x30\xDA\x3A\x00\x00"
			  "\x80\x01\xDF\x3B\xC0\xA8\x00\x05"
			  "\xC0\xA8\x00\x01\x08\x00\xC6\xCD"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.ilen	= 52,
		.assoc	= "\x79\x6B\x69\x63\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\x33\x30\x21\x69"
			  "\x67\x65\x74\x6D",
		.alen	= 20,
		.result	= "\x96\xFD\x86\xF8\xD1\x98\xFF\x10"
			  "\xAB\x8C\xDA\x8A\x5A\x08\x38\x1A"
			  "\x48\x59\x80\x18\x1A\x18\x1A\x04"
			  "\xC9\x0D\xE3\xE7\x0E\xA4\x0B\x75"
			  "\x92\x9C\x52\x5C\x0B\xFB\xF8\xAF"
			  "\x16\xC3\x35\xA8\xE7\xCE\x84\x04"
			  "\xEB\x40\x6B\x7A\x8E\x75\xBB\x42"
			  "\xE0\x63\x4B\x21\x44\xA2\x2B\x2B"
			  "\x39\xDB\xC8\xDC",
		.rlen	= 68,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E",
		.klen	= 19,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.input	= "\x45\x00\x00\x30\xDA\x3A\x00\x00"
			  "\x80\x01\xDF\x3B\xC0\xA8\x00\x05"
			  "\xC0\xA8\x00\x01\x08\x00\xC6\xCD"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.ilen	= 52,
		.assoc	= "\x3F\x7E\xF6\x42\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.result	= "\x6A\x6B\x45\x27\x3F\x9E\x52\xF6"
			  "\x10\x60\x54\x25\xEB\x80\x04\x93"
			  "\xCA\x1B\x23\x97\xCB\x21\x2E\x01"
			  "\xA2\xE7\x95\x41\x30\xE4\x4B\x1B"
			  "\x79\x01\x58\x50\x01\x06\xE1\xE0"
			  "\x2C\x83\x79\xD3\xDE\x46\x97\x1A"
			  "\x44\xCC\x90\xBF\x00\x94\x94\x92"
			  "\x20\x17\x0C\x1B\x55\xDE\x7E\x68"
			  "\xF4\x95\x5D\x4F",
		.rlen	= 68,
	}, {
		.key	= "\x4C\x80\xCD\xEF\xBB\x5D\x10\xDA"
			  "\x90\x6A\xC7\x3C\x36\x13\xA6\x34"
			  "\x22\x43\x3C",
		.klen	= 19,
		.iv	= "\x48\x55\xEC\x7D\x3A\x23\x4B\xFD",
		.input	= "\x08\x00\xC6\xCD\x02\x00\x07\x00"
			  "\x61\x62\x63\x64\x65\x66\x67\x68"
			  "\x69\x6A\x6B\x6C\x6D\x6E\x6F\x70"
			  "\x71\x72\x73\x74\x01\x02\x02\x01",
		.ilen	= 32,
		.assoc	= "\x00\x00\x43\x21\x87\x65\x43\x21"
			  "\x00\x00\x00\x07\x48\x55\xEC\x7D"
			  "\x3A\x23\x4B\xFD",
		.alen	= 20,
		.result	= "\x67\xE9\x28\xB3\x1C\xA4\x6D\x02"
			  "\xF0\xB5\x37\xB6\x6B\x2F\xF5\x4F"
			  "\xF8\xA3\x4C\x53\xB8\x12\x09\xBF"
			  "\x58\x7D\xCF\x29\xA3\x41\x68\x6B"
			  "\xCE\xE8\x79\x85\x3C\xB0\x3A\x8F"
			  "\x16\xB0\xA1\x26\xC9\xBC\xBC\xA6",
		.rlen	= 48,
	}
};

static struct aead_testvec aes_ccm_rfc4309_dec_tv_template[]	= {
	{ /* Generated using Crypto++ */
		.key	= zeroed_string,
		.klen	= 19,
		.iv	= zeroed_string,
		.result	= zeroed_string,
		.rlen	= 16,
		.assoc	= zeroed_string,
		.alen	= 16,
		.input	= "\x2E\x9A\xCA\x6B\xDA\x54\xFC\x6F"
			  "\x12\x50\xE8\xDE\x81\x3C\x63\x08"
			  "\x1A\x22\xBA\x75\xEE\xD4\xD5\xB5"
			  "\x27\x50\x01\xAC\x03\x33\x39\xFB",
		.ilen	= 32,
	},{
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x01",
		.result	= zeroed_string,
		.rlen	= 16,
		.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen	= 16,
		.input	= "\xCF\xB9\x99\x17\xC8\x86\x0E\x7F"
			  "\x7E\x76\xF8\xE6\xF8\xCC\x1F\x17"
			  "\x6A\xE0\x53\x9F\x4B\x73\x7E\xDA"
			  "\x08\x09\x4E\xC4\x1E\xAD\xC6\xB0",
		.ilen	= 32,

	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= zeroed_string,
		.result	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.rlen	= 16,
		.assoc	= zeroed_string,
		.alen	= 16,
		.input	= "\x33\xDE\x73\xBC\xA6\xCE\x4E\xA6"
			  "\x61\xF4\xF5\x41\x03\x4A\xE3\x86"
			  "\xA1\xE2\xC2\x42\x2B\x81\x70\x40"
			  "\xFD\x7F\x76\xD1\x03\x07\xBB\x0C",
		.ilen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= zeroed_string,
		.result	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.rlen	= 16,
		.assoc	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.input	= "\x33\xDE\x73\xBC\xA6\xCE\x4E\xA6"
			  "\x61\xF4\xF5\x41\x03\x4A\xE3\x86"
			  "\x5B\xC0\x73\xE0\x2B\x73\x68\xC9"
			  "\x2D\x8C\x58\xC2\x90\x3D\xB0\x3E",
		.ilen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x01",
		.result	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.rlen	= 16,
		.assoc	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen	= 16,
		.input	= "\xCE\xB8\x98\x16\xC9\x87\x0F\x7E"
			  "\x7F\x77\xF9\xE7\xF9\xCD\x1E\x16"
			  "\x43\x8E\x76\x57\x3B\xB4\x05\xE8"
			  "\xA9\x9B\xBF\x25\xE0\x4F\xC0\xED",
		.ilen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x01",
		.result	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x01\x01\x01\x01\x01\x01\x01\x01",
		.rlen	= 64,
		.assoc	= "\x01\x01\x01\x01\x01\x01\x01\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.alen	= 16,
		.input	= "\xCE\xB8\x98\x16\xC9\x87\x0F\x7E"
			  "\x7F\x77\xF9\xE7\xF9\xCD\x1E\x16"
			  "\x9C\xA4\x97\x83\x3F\x01\xA5\xF4"
			  "\x43\x09\xE7\xB8\xE9\xD1\xD7\x02"
			  "\x9B\xAB\x39\x18\xEB\x94\x34\x36"
			  "\xE6\xC5\xC8\x9B\x00\x81\x9E\x49"
			  "\x1D\x78\xE1\x48\xE3\xE9\xEA\x8E"
			  "\x3A\x2B\x67\x5D\x35\x6A\x0F\xDB"
			  "\x02\x73\xDD\xE7\x30\x4A\x30\x54"
			  "\x1A\x9D\x09\xCA\xC8\x1C\x32\x5F",
		.ilen	= 80,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x45\x67\x89\xab\xcd\xef",
		.result	= "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff"
			  "\xff\xff\xff\xff\xff\xff\xff\xff",
		.rlen	= 192,
		.assoc	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\x00\x00\x45\x67"
			  "\x89\xab\xcd\xef",
		.alen	= 20,
		.input	= "\x64\x17\xDC\x24\x9D\x92\xBA\x5E"
			  "\x7C\x64\x6D\x33\x46\x77\xAC\xB1"
			  "\x5C\x9E\xE2\xC7\x27\x11\x3E\x95"
			  "\x7D\xBE\x28\xC8\xC1\xCA\x5E\x8C"
			  "\xB4\xE2\xDE\x9F\x53\x59\x26\xDB"
			  "\x0C\xD4\xE4\x07\x9A\xE6\x3E\x01"
			  "\x58\x0D\x3E\x3D\xD5\x21\xEB\x04"
			  "\x06\x9D\x5F\xB9\x02\x49\x1A\x2B"
			  "\xBA\xF0\x4E\x3B\x85\x50\x5B\x09"
			  "\xFE\xEC\xFC\x54\xEC\x0C\xE2\x79"
			  "\x8A\x2F\x5F\xD7\x05\x5D\xF1\x6D"
			  "\x22\xEB\xD1\x09\x80\x3F\x5A\x70"
			  "\xB2\xB9\xD3\x63\x99\xC2\x4D\x1B"
			  "\x36\x12\x00\x89\xAA\x5D\x55\xDA"
			  "\x1D\x5B\xD8\x3C\x5F\x09\xD2\xE6"
			  "\x39\x41\x5C\xF0\xBE\x26\x4E\x5F"
			  "\x2B\x50\x44\x52\xC2\x10\x7D\x38"
			  "\x82\x64\x83\x0C\xAE\x49\xD0\xE5"
			  "\x4F\xE5\x66\x4C\x58\x7A\xEE\x43"
			  "\x3B\x51\xFE\xBA\x24\x8A\xFE\xDC"
			  "\x19\x6D\x60\x66\x61\xF9\x9A\x3F"
			  "\x75\xFC\x38\x53\x5B\xB5\xCD\x52"
			  "\x4F\xE5\xE4\xC9\xFE\x10\xCB\x98"
			  "\xF0\x06\x5B\x07\xAB\xBB\xF4\x0E"
			  "\x2D\xC2\xDD\x5D\xDD\x22\x9A\xCC"
			  "\x39\xAB\x63\xA5\x3D\x9C\x51\x8A",
		.ilen	= 208,
	}, { /* From draft-mcgrew-gcm-test-01 */
		.key	= "\x4C\x80\xCD\xEF\xBB\x5D\x10\xDA"
			  "\x90\x6A\xC7\x3C\x36\x13\xA6\x34"
			  "\x2E\x44\x3B",
		.klen	= 19,
		.iv	= "\x49\x56\xED\x7E\x3B\x24\x4C\xFE",
		.result	= "\x45\x00\x00\x48\x69\x9A\x00\x00"
			  "\x80\x11\x4D\xB7\xC0\xA8\x01\x02"
			  "\xC0\xA8\x01\x01\x0A\x9B\xF1\x56"
			  "\x38\xD3\x01\x00\x00\x01\x00\x00"
			  "\x00\x00\x00\x00\x04\x5F\x73\x69"
			  "\x70\x04\x5F\x75\x64\x70\x03\x73"
			  "\x69\x70\x09\x63\x79\x62\x65\x72"
			  "\x63\x69\x74\x79\x02\x64\x6B\x00"
			  "\x00\x21\x00\x01\x01\x02\x02\x01",
		.rlen	= 72,
		.assoc	= "\x00\x00\x43\x21\x87\x65\x43\x21"
			  "\x00\x00\x00\x00\x49\x56\xED\x7E"
			  "\x3B\x24\x4C\xFE",
		.alen	= 20,
		.input	= "\x89\xBA\x3E\xEF\xE6\xD6\xCF\xDB"
			  "\x83\x60\xF5\xBA\x3A\x56\x79\xE6"
			  "\x7E\x0C\x53\xCF\x9E\x87\xE0\x4E"
			  "\x1A\x26\x01\x24\xC7\x2E\x3D\xBF"
			  "\x29\x2C\x91\xC1\xB8\xA8\xCF\xE0"
			  "\x39\xF8\x53\x6D\x31\x22\x2B\xBF"
			  "\x98\x81\xFC\x34\xEE\x85\x36\xCD"
			  "\x26\xDB\x6C\x7A\x0C\x77\x8A\x35"
			  "\x18\x85\x54\xB2\xBC\xDD\x3F\x43"
			  "\x61\x06\x8A\xDF\x86\x3F\xB4\xAC"
			  "\x97\xDC\xBD\xFD\x92\x10\xC5\xFF",
		.ilen	= 88,
	}, {
		.key	= "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\x6D\x6A\x8F\x94\x67\x30\x83\x08"
			  "\xCA\xFE\xBA",
		.klen	= 19,
		.iv	= "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.result	= "\x45\x00\x00\x3E\x69\x8F\x00\x00"
			  "\x80\x11\x4D\xCC\xC0\xA8\x01\x02"
			  "\xC0\xA8\x01\x01\x0A\x98\x00\x35"
			  "\x00\x2A\x23\x43\xB2\xD0\x01\x00"
			  "\x00\x01\x00\x00\x00\x00\x00\x00"
			  "\x03\x73\x69\x70\x09\x63\x79\x62"
			  "\x65\x72\x63\x69\x74\x79\x02\x64"
			  "\x6B\x00\x00\x01\x00\x01\x00\x01",
		.rlen	= 64,
		.assoc	= "\x00\x00\xA5\xF8\x00\x00\x00\x0A"
			  "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.alen	= 16,
		.input	= "\x4B\xC2\x70\x60\x64\xD2\xF3\xC8"
			  "\xE5\x26\x8A\xDE\xB8\x7E\x7D\x16"
			  "\x56\xC7\xD2\x88\xBA\x8D\x58\xAF"
			  "\xF5\x71\xB6\x37\x84\xA7\xB1\x99"
			  "\x51\x5C\x0D\xA0\x27\xDE\xE7\x2D"
			  "\xEF\x25\x88\x1F\x1D\x77\x11\xFF"
			  "\xDB\xED\xEE\x56\x16\xC5\x5C\x9B"
			  "\x00\x62\x1F\x68\x4E\x7C\xA0\x97"
			  "\x10\x72\x7E\x53\x13\x3B\x68\xE4"
			  "\x30\x99\x91\x79\x09\xEA\xFF\x6A",
		.ilen	= 80,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x11\x22\x33",
		.klen	= 35,
		.iv	= "\x01\x02\x03\x04\x05\x06\x07\x08",
		.result	= "\x45\x00\x00\x30\x69\xA6\x40\x00"
			  "\x80\x06\x26\x90\xC0\xA8\x01\x02"
			  "\x93\x89\x15\x5E\x0A\x9E\x00\x8B"
			  "\x2D\xC5\x7E\xE0\x00\x00\x00\x00"
			  "\x70\x02\x40\x00\x20\xBF\x00\x00"
			  "\x02\x04\x05\xB4\x01\x01\x04\x02"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
		.assoc	= "\x4A\x2C\xBF\xE3\x00\x00\x00\x02"
			  "\x01\x02\x03\x04\x05\x06\x07\x08",
		.alen	= 16,
		.input	= "\xD6\x31\x0D\x2B\x3D\x6F\xBD\x2F"
			  "\x58\x41\x7E\xFF\x9A\x9E\x09\xB4"
			  "\x1A\xF7\xF6\x42\x31\xCD\xBF\xAD"
			  "\x27\x0E\x2C\xF2\xDB\x10\xDF\x55"
			  "\x8F\x0D\xD7\xAC\x23\xBD\x42\x10"
			  "\xD0\xB2\xAF\xD8\x37\xAC\x6B\x0B"
			  "\x11\xD4\x0B\x12\xEC\xB4\xB1\x92"
			  "\x23\xA6\x10\xB0\x26\xD6\xD9\x26"
			  "\x5A\x48\x6A\x3E",
		.ilen	= 68,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00",
		.klen	= 19,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.result	= "\x45\x00\x00\x3C\x99\xC5\x00\x00"
			  "\x80\x01\xCB\x7A\x40\x67\x93\x18"
			  "\x01\x01\x01\x01\x08\x00\x07\x5C"
			  "\x02\x00\x44\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x75\x76\x77\x61\x62\x63\x64\x65"
			  "\x66\x67\x68\x69\x01\x02\x02\x01",
		.rlen	= 64,
		.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x01"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.alen	= 16,
		.input	= "\x6B\x9A\xCA\x57\x43\x91\xFC\x6F"
			  "\x92\x51\x23\xA4\xC1\x5B\xF0\x10"
			  "\xF3\x13\xF4\xF8\xA1\x9A\xB4\xDC"
			  "\x89\xC8\xF8\x42\x62\x95\xB7\xCB"
			  "\xB8\xF5\x0F\x1B\x2E\x94\xA2\xA7"
			  "\xBF\xFB\x8A\x92\x13\x63\xD1\x3C"
			  "\x08\xF5\xE8\xA6\xAA\xF6\x34\xF9"
			  "\x42\x05\xAF\xB3\xE7\x9A\xFC\xEE"
			  "\x36\x25\xC1\x10\x12\x1C\xCA\x82"
			  "\xEA\xE6\x63\x5A\x57\x28\xA9\x9A",
		.ilen	= 80,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E",
		.klen	= 19,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.result	= "\x45\x00\x00\x3C\x99\xC3\x00\x00"
			  "\x80\x01\xCB\x7C\x40\x67\x93\x18"
			  "\x01\x01\x01\x01\x08\x00\x08\x5C"
			  "\x02\x00\x43\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x75\x76\x77\x61\x62\x63\x64\x65"
			  "\x66\x67\x68\x69\x01\x02\x02\x01",
		.rlen	= 64,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.input	= "\x6A\x6B\x45\x2B\x7C\x67\x52\xF6"
			  "\x10\x60\x40\x62\x6B\x4F\x97\x8E"
			  "\x0B\xB2\x22\x97\xCB\x21\xE0\x90"
			  "\xA2\xE7\xD1\x41\x30\xE4\x4B\x1B"
			  "\x79\x01\x58\x50\x01\x06\xE1\xE0"
			  "\x2C\x83\x79\xD3\xDE\x46\x97\x1A"
			  "\x30\xB8\xE5\xDF\xD7\x12\x56\x75"
			  "\xD0\x95\xB7\xB8\x91\x42\xF7\xFD"
			  "\x97\x57\xCA\xC1\x20\xD0\x86\xB9"
			  "\x66\x9D\xB4\x2B\x96\x22\xAC\x67",
		.ilen	= 80,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E",
		.klen	= 19,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.result	= "\x45\x00\x00\x1C\x42\xA2\x00\x00"
			  "\x80\x01\x44\x1F\x40\x67\x93\xB6"
			  "\xE0\x00\x00\x02\x0A\x00\xF5\xFF"
			  "\x01\x02\x02\x01",
		.rlen	= 28,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.input	= "\x6A\x6B\x45\x0B\xA7\x06\x52\xF6"
			  "\x10\x60\xCF\x01\x6B\x4F\x97\x20"
			  "\xEA\xB3\x23\x94\xC9\x21\x1D\x33"
			  "\xA1\xE5\x90\x40\x05\x37\x45\x70"
			  "\xB5\xD6\x09\x0A\x23\x73\x33\xF9"
			  "\x08\xB4\x22\xE4",
		.ilen	= 44,
	}, {
		.key	= "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\x6D\x6A\x8F\x94\x67\x30\x83\x08"
			  "\xFE\xFF\xE9\x92\x86\x65\x73\x1C"
			  "\xCA\xFE\xBA",
		.klen	= 27,
		.iv	= "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.result	= "\x45\x00\x00\x28\xA4\xAD\x40\x00"
			  "\x40\x06\x78\x80\x0A\x01\x03\x8F"
			  "\x0A\x01\x06\x12\x80\x23\x06\xB8"
			  "\xCB\x71\x26\x02\xDD\x6B\xB0\x3E"
			  "\x50\x10\x16\xD0\x75\x68\x00\x01",
		.rlen	= 40,
		.assoc	= "\x00\x00\xA5\xF8\x00\x00\x00\x0A"
			  "\xFA\xCE\xDB\xAD\xDE\xCA\xF8\x88",
		.alen	= 16,
		.input	= "\x05\x22\x15\xD1\x52\x56\x85\x04"
			  "\xA8\x5C\x5D\x6D\x7E\x6E\xF5\xFA"
			  "\xEA\x16\x37\x50\xF3\xDF\x84\x3B"
			  "\x2F\x32\x18\x57\x34\x2A\x8C\x23"
			  "\x67\xDF\x6D\x35\x7B\x54\x0D\xFB"
			  "\x34\xA5\x9F\x6C\x48\x30\x1E\x22"
			  "\xFE\xB1\x22\x17\x17\x8A\xB9\x5B",
		.ilen	= 56,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xDE\xCA\xF8",
		.klen	= 19,
		.iv	= "\xCA\xFE\xDE\xBA\xCE\xFA\xCE\x74",
		.result	= "\x45\x00\x00\x49\x33\xBA\x00\x00"
			  "\x7F\x11\x91\x06\xC3\xFB\x1D\x10"
			  "\xC2\xB1\xD3\x26\xC0\x28\x31\xCE"
			  "\x00\x35\xDD\x7B\x80\x03\x02\xD5"
			  "\x00\x00\x4E\x20\x00\x1E\x8C\x18"
			  "\xD7\x5B\x81\xDC\x91\xBA\xA0\x47"
			  "\x6B\x91\xB9\x24\xB2\x80\x38\x9D"
			  "\x92\xC9\x63\xBA\xC0\x46\xEC\x95"
			  "\x9B\x62\x66\xC0\x47\x22\xB1\x49"
			  "\x23\x01\x01\x01",
		.rlen	= 76,
		.assoc	= "\x00\x00\x01\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x01\xCA\xFE\xDE\xBA"
			  "\xCE\xFA\xCE\x74",
		.alen	= 20,
		.input	= "\x92\xD0\x53\x79\x33\x38\xD5\xF3"
			  "\x7D\xE4\x7A\x8E\x86\x03\xC9\x90"
			  "\x96\x35\xAB\x9C\xFB\xE8\xA3\x76"
			  "\xE9\xE9\xE2\xD1\x2E\x11\x0E\x00"
			  "\xFA\xCE\xB5\x9E\x02\xA7\x7B\xEA"
			  "\x71\x9A\x58\xFB\xA5\x8A\xE1\xB7"
			  "\x9C\x39\x9D\xE3\xB5\x6E\x69\xE6"
			  "\x63\xC9\xDB\x05\x69\x51\x12\xAD"
			  "\x3E\x00\x32\x73\x86\xF2\xEE\xF5"
			  "\x0F\xE8\x81\x7E\x84\xD3\xC0\x0D"
			  "\x76\xD6\x55\xC6\xB4\xC2\x34\xC7"
			  "\x12\x25\x0B\xF9",
		.ilen	= 92,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x73\x61\x6C",
		.klen	= 35,
		.iv	= "\x61\x6E\x64\x01\x69\x76\x65\x63",
		.result	= "\x45\x08\x00\x28\x73\x2C\x00\x00"
			  "\x40\x06\xE9\xF9\x0A\x01\x06\x12"
			  "\x0A\x01\x03\x8F\x06\xB8\x80\x23"
			  "\xDD\x6B\xAF\xBE\xCB\x71\x26\x02"
			  "\x50\x10\x1F\x64\x6D\x54\x00\x01",
		.rlen	= 40,
		.assoc	= "\x17\x40\x5E\x67\x15\x6F\x31\x26"
			  "\xDD\x0D\xB9\x9B\x61\x6E\x64\x01"
			  "\x69\x76\x65\x63",
		.alen	= 20,
		.input	= "\xCC\x74\xB7\xD3\xB0\x38\x50\x42"
			  "\x2C\x64\x87\x46\x1E\x34\x10\x05"
			  "\x29\x6B\xBB\x36\xE9\x69\xAD\x92"
			  "\x82\xA1\x10\x6A\xEB\x0F\xDC\x7D"
			  "\x08\xBA\xF3\x91\xCA\xAA\x61\xDA"
			  "\x62\xF4\x14\x61\x5C\x9D\xB5\xA7"
			  "\xEE\xD7\xB9\x7E\x87\x99\x9B\x7D",
		.ilen	= 56,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E",
		.klen	= 19,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.result	= "\x45\x00\x00\x49\x33\x3E\x00\x00"
			  "\x7F\x11\x91\x82\xC3\xFB\x1D\x10"
			  "\xC2\xB1\xD3\x26\xC0\x28\x31\xCE"
			  "\x00\x35\xCB\x45\x80\x03\x02\x5B"
			  "\x00\x00\x01\xE0\x00\x1E\x8C\x18"
			  "\xD6\x57\x59\xD5\x22\x84\xA0\x35"
			  "\x2C\x71\x47\x5C\x88\x80\x39\x1C"
			  "\x76\x4D\x6E\x5E\xE0\x49\x6B\x32"
			  "\x5A\xE2\x70\xC0\x38\x99\x49\x39"
			  "\x15\x01\x01\x01",
		.rlen	= 76,
		.assoc	= "\x42\xF6\x7E\x3F\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.input	= "\x6A\x6B\x45\x5E\xD6\x9A\x52\xF6"
			  "\xEF\x70\x1A\x9C\xE8\xD3\x19\x86"
			  "\xC8\x02\xF0\xB0\x03\x09\xD9\x02"
			  "\xA0\xD2\x59\x04\xD1\x85\x2A\x24"
			  "\x1C\x67\x3E\xD8\x68\x72\x06\x94"
			  "\x97\xBA\x4F\x76\x8D\xB0\x44\x5B"
			  "\x69\xBF\xD5\xE2\x3D\xF1\x0B\x0C"
			  "\xC0\xBF\xB1\x8F\x70\x09\x9E\xCE"
			  "\xA5\xF2\x55\x58\x84\xFA\xF9\xB5"
			  "\x23\xF4\x84\x40\x74\x14\x8A\x6B"
			  "\xDB\xD7\x67\xED\xA4\x93\xF3\x47"
			  "\xCC\xF7\x46\x6F",
		.ilen	= 92,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\x73\x61\x6C",
		.klen	= 35,
		.iv	= "\x61\x6E\x64\x01\x69\x76\x65\x63",
		.result	= "\x63\x69\x73\x63\x6F\x01\x72\x75"
			  "\x6C\x65\x73\x01\x74\x68\x65\x01"
			  "\x6E\x65\x74\x77\x65\x01\x64\x65"
			  "\x66\x69\x6E\x65\x01\x74\x68\x65"
			  "\x74\x65\x63\x68\x6E\x6F\x6C\x6F"
			  "\x67\x69\x65\x73\x01\x74\x68\x61"
			  "\x74\x77\x69\x6C\x6C\x01\x64\x65"
			  "\x66\x69\x6E\x65\x74\x6F\x6D\x6F"
			  "\x72\x72\x6F\x77\x01\x02\x02\x01",
		.rlen	= 72,
		.assoc	= "\x17\x40\x5E\x67\x15\x6F\x31\x26"
			  "\xDD\x0D\xB9\x9B\x61\x6E\x64\x01"
			  "\x69\x76\x65\x63",
		.alen	= 20,
		.input	= "\xEA\x15\xC4\x98\xAC\x15\x22\x37"
			  "\x00\x07\x1D\xBE\x60\x5D\x73\x16"
			  "\x4D\x0F\xCC\xCE\x8A\xD0\x49\xD4"
			  "\x39\xA3\xD1\xB1\x21\x0A\x92\x1A"
			  "\x2C\xCF\x8F\x9D\xC9\x91\x0D\xB4"
			  "\x15\xFC\xBC\xA5\xC5\xBF\x54\xE5"
			  "\x1C\xC7\x32\x41\x07\x7B\x2C\xB6"
			  "\x5C\x23\x7C\x93\xEA\xEF\x23\x1C"
			  "\x73\xF4\xE7\x12\x84\x4C\x37\x0A"
			  "\x4A\x8F\x06\x37\x48\xF9\xF9\x05"
			  "\x55\x13\x40\xC3\xD5\x55\x3A\x3D",
		.ilen	= 88,
	}, {
		.key	= "\x7D\x77\x3D\x00\xC1\x44\xC5\x25"
			  "\xAC\x61\x9D\x18\xC8\x4A\x3F\x47"
			  "\xD9\x66\x42",
		.klen	= 19,
		.iv	= "\x43\x45\x7E\x91\x82\x44\x3B\xC6",
		.result	= "\x01\x02\x02\x01",
		.rlen	= 4,
		.assoc	= "\x33\x54\x67\xAE\xFF\xFF\xFF\xFF"
			  "\x43\x45\x7E\x91\x82\x44\x3B\xC6",
		.alen	= 16,
		.input	= "\x4C\x72\x63\x30\x2F\xE6\x56\xDD"
			  "\xD0\xD8\x60\x9D\x8B\xEF\x85\x90"
			  "\xF7\x61\x24\x62",
		.ilen	= 20,
	}, {
		.key	= "\xAB\xBC\xCD\xDE\xF0\x01\x12\x23"
			  "\x34\x45\x56\x67\x78\x89\x9A\xAB"
			  "\xDE\xCA\xF8",
		.klen	= 19,
		.iv	= "\xCA\xFE\xDE\xBA\xCE\xFA\xCE\x74",
		.result	= "\x74\x6F\x01\x62\x65\x01\x6F\x72"
			  "\x01\x6E\x6F\x74\x01\x74\x6F\x01"
			  "\x62\x65\x00\x01",
		.rlen	= 20,
		.assoc	= "\x00\x00\x01\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x01\xCA\xFE\xDE\xBA"
			  "\xCE\xFA\xCE\x74",
		.alen	= 20,
		.input	= "\xA3\xBF\x52\x52\x65\x83\xBA\x81"
			  "\x03\x9B\x84\xFC\x44\x8C\xBB\x81"
			  "\x36\xE1\x78\xBB\xA5\x49\x3A\xD0"
			  "\xF0\x6B\x21\xAF\x98\xC0\x34\xDC"
			  "\x17\x17\x65\xAD",
		.ilen	= 36,
	}, {
		.key	= "\x6C\x65\x67\x61\x6C\x69\x7A\x65"
			  "\x6D\x61\x72\x69\x6A\x75\x61\x6E"
			  "\x61\x61\x6E\x64\x64\x6F\x69\x74"
			  "\x62\x65\x66\x6F\x72\x65\x69\x61"
			  "\x74\x75\x72",
		.klen	= 35,
		.iv	= "\x33\x30\x21\x69\x67\x65\x74\x6D",
		.result	= "\x45\x00\x00\x30\xDA\x3A\x00\x00"
			  "\x80\x01\xDF\x3B\xC0\xA8\x00\x05"
			  "\xC0\xA8\x00\x01\x08\x00\xC6\xCD"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
		.assoc	= "\x79\x6B\x69\x63\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\x33\x30\x21\x69"
			  "\x67\x65\x74\x6D",
		.alen	= 20,
		.input	= "\x96\xFD\x86\xF8\xD1\x98\xFF\x10"
			  "\xAB\x8C\xDA\x8A\x5A\x08\x38\x1A"
			  "\x48\x59\x80\x18\x1A\x18\x1A\x04"
			  "\xC9\x0D\xE3\xE7\x0E\xA4\x0B\x75"
			  "\x92\x9C\x52\x5C\x0B\xFB\xF8\xAF"
			  "\x16\xC3\x35\xA8\xE7\xCE\x84\x04"
			  "\xEB\x40\x6B\x7A\x8E\x75\xBB\x42"
			  "\xE0\x63\x4B\x21\x44\xA2\x2B\x2B"
			  "\x39\xDB\xC8\xDC",
		.ilen	= 68,
	}, {
		.key	= "\x3D\xE0\x98\x74\xB3\x88\xE6\x49"
			  "\x19\x88\xD0\xC3\x60\x7E\xAE\x1F"
			  "\x57\x69\x0E",
		.klen	= 19,
		.iv	= "\x4E\x28\x00\x00\xA2\xFC\xA1\xA3",
		.result	= "\x45\x00\x00\x30\xDA\x3A\x00\x00"
			  "\x80\x01\xDF\x3B\xC0\xA8\x00\x05"
			  "\xC0\xA8\x00\x01\x08\x00\xC6\xCD"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6A\x6B\x6C"
			  "\x6D\x6E\x6F\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
		.assoc	= "\x3F\x7E\xF6\x42\x10\x10\x10\x10"
			  "\x10\x10\x10\x10\x4E\x28\x00\x00"
			  "\xA2\xFC\xA1\xA3",
		.alen	= 20,
		.input	= "\x6A\x6B\x45\x27\x3F\x9E\x52\xF6"
			  "\x10\x60\x54\x25\xEB\x80\x04\x93"
			  "\xCA\x1B\x23\x97\xCB\x21\x2E\x01"
			  "\xA2\xE7\x95\x41\x30\xE4\x4B\x1B"
			  "\x79\x01\x58\x50\x01\x06\xE1\xE0"
			  "\x2C\x83\x79\xD3\xDE\x46\x97\x1A"
			  "\x44\xCC\x90\xBF\x00\x94\x94\x92"
			  "\x20\x17\x0C\x1B\x55\xDE\x7E\x68"
			  "\xF4\x95\x5D\x4F",
		.ilen	= 68,
	}, {
		.key	= "\x4C\x80\xCD\xEF\xBB\x5D\x10\xDA"
			  "\x90\x6A\xC7\x3C\x36\x13\xA6\x34"
			  "\x22\x43\x3C",
		.klen	= 19,
		.iv	= "\x48\x55\xEC\x7D\x3A\x23\x4B\xFD",
		.result	= "\x08\x00\xC6\xCD\x02\x00\x07\x00"
			  "\x61\x62\x63\x64\x65\x66\x67\x68"
			  "\x69\x6A\x6B\x6C\x6D\x6E\x6F\x70"
			  "\x71\x72\x73\x74\x01\x02\x02\x01",
		.rlen	= 32,
		.assoc	= "\x00\x00\x43\x21\x87\x65\x43\x21"
			  "\x00\x00\x00\x07\x48\x55\xEC\x7D"
			  "\x3A\x23\x4B\xFD",
		.alen	= 20,
		.input	= "\x67\xE9\x28\xB3\x1C\xA4\x6D\x02"
			  "\xF0\xB5\x37\xB6\x6B\x2F\xF5\x4F"
			  "\xF8\xA3\x4C\x53\xB8\x12\x09\xBF"
			  "\x58\x7D\xCF\x29\xA3\x41\x68\x6B"
			  "\xCE\xE8\x79\x85\x3C\xB0\x3A\x8F"
			  "\x16\xB0\xA1\x26\xC9\xBC\xBC\xA6",
		.ilen	= 48,
	}
};

/*
 * ARC4 test vectors from OpenSSL
 */
#define ARC4_ENC_TEST_VECTORS	7
#define ARC4_DEC_TEST_VECTORS	7

static struct cipher_testvec arc4_enc_tv_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.ilen	= 8,
		.result	= "\x75\xb7\x87\x80\x99\xe0\xc5\x96",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\x74\x94\xc2\xe7\x10\x4b\x08\x79",
		.rlen	= 8,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 8,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\xde\x18\x89\x41\xa3\x37\x5d\x3a",
		.rlen	= 8,
	}, {
		.key	= "\xef\x01\x23\x45",
		.klen	= 4,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00",
		.ilen	= 20,
		.result	= "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf"
			  "\xbd\x61\x5a\x11\x62\xe1\xc7\xba"
			  "\x36\xb6\x78\x58",
		.rlen	= 20,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78",
		.ilen	= 28,
		.result	= "\x66\xa0\x94\x9f\x8a\xf7\xd6\x89"
			  "\x1f\x7f\x83\x2b\xa8\x33\xc0\x0c"
			  "\x89\x2e\xbe\x30\x14\x3c\xe2\x87"
			  "\x40\x01\x1e\xcf",
		.rlen	= 28,
	}, {
		.key	= "\xef\x01\x23\x45",
		.klen	= 4,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00",
		.ilen	= 10,
		.result	= "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf"
			  "\xbd\x61",
		.rlen	= 10,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xAB\xCD\xEF"
			"\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 16,
		.input	= "\x01\x23\x45\x67\x89\xAB\xCD\xEF",
		.ilen	= 8,
		.result	= "\x69\x72\x36\x59\x1B\x52\x42\xB1",
		.rlen	= 8,
	},
};

static struct cipher_testvec arc4_dec_tv_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x75\xb7\x87\x80\x99\xe0\xc5\x96",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x74\x94\xc2\xe7\x10\x4b\x08\x79",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 8,
		.input	= "\xde\x18\x89\x41\xa3\x37\x5d\x3a",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, {
		.key	= "\xef\x01\x23\x45",
		.klen	= 4,
		.input	= "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf"
			  "\xbd\x61\x5a\x11\x62\xe1\xc7\xba"
			  "\x36\xb6\x78\x58",
		.ilen	= 20,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00",
		.rlen	= 20,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x66\xa0\x94\x9f\x8a\xf7\xd6\x89"
			  "\x1f\x7f\x83\x2b\xa8\x33\xc0\x0c"
			  "\x89\x2e\xbe\x30\x14\x3c\xe2\x87"
			  "\x40\x01\x1e\xcf",
		.ilen	= 28,
		.result	= "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78",
		.rlen	= 28,
	}, {
		.key	= "\xef\x01\x23\x45",
		.klen	= 4,
		.input	= "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf"
			  "\xbd\x61",
		.ilen	= 10,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00",
		.rlen	= 10,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xAB\xCD\xEF"
			"\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 16,
		.input	= "\x69\x72\x36\x59\x1B\x52\x42\xB1",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xAB\xCD\xEF",
		.rlen	= 8,
	},
};

#endif	/* _CRYPTO_TCRYPT_H */
