/*
 *  Oops dump MMC
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/kmsg_dump.h>
#include <linux/slab.h>
#include <linux/mmc/mmc.h>
#include <linux/mmc/host.h>
#include <linux/mmc/card.h>
#include <linux/scatterlist.h>
#include <linux/platform_device.h>
#include <linux/mmc/core.h>
#include <linux/slab.h>
#include <linux/moduleparam.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/hdreg.h>
#include <linux/kdev_t.h>
#include <linux/blkdev.h>
#include <linux/mutex.h>
#include <linux/scatterlist.h>
#include <linux/string_helpers.h>
#include <linux/delay.h>
#include <linux/capability.h>
#include <linux/compat.h>
#include <linux/pm_runtime.h>
#include <linux/mmc/ioctl.h>
#include <linux/mmc/sd.h>
#include <linux/uaccess.h>
#include <linux/file.h>
#include "core.h"


/* TODO Unify the oops header, mmtoops, ramoops, mmcoops */
#define OOPS_MMC_DUMP_SIGNATURE "BRCM-OOPS-DUMP-MMC"
#define OOPS_MMC_DUMP_HDR_LENGTH \
	((sizeof(OOPS_MMC_DUMP_SIGNATURE))-1+(sizeof(unsigned long)))
static char *dump_mark =
	"================================";
static char *dump_start_str =
	"PREVIOUS_KERNEL_OOPS_DUMP_START";
static char *dump_end_str =
	"PREVIOUS_KERNEL_OOPS_DUMP_END";

/* At a time how many blocks we read or write */
#define OOPS_MMC_NUM_RW_BLOCKS   128
#define OOPS_MMC_INVALID 0xFFFFFFFF

static struct platform_device *dummy;
static struct oops_mmc_platform_data *dummy_data;

static char mmcdev[80];
module_param_string(mmcdev, mmcdev, 80, 0400);
MODULE_PARM_DESC(mmcdev,
		"Name or path of the MMC block device to use");

static unsigned long partition_start_block = OOPS_MMC_INVALID;
module_param(partition_start_block, ulong, 0400);
MODULE_PARM_DESC(partition_start_block,
		"partition start block to dump the panic");

static unsigned long partition_size = OOPS_MMC_INVALID;
module_param(partition_size, ulong, 0400);
MODULE_PARM_DESC(partition_size,
		"size of the partition to dump the panic");

static unsigned long mmc_blksz = OOPS_MMC_INVALID;
module_param(mmc_blksz, ulong, 0400);
MODULE_PARM_DESC(mmc_blksz,
		"each block size of MMC, usually 512");

static char *dump_file_path;
module_param(dump_file_path, charp, 0400);
MODULE_PARM_DESC(dump_file_path,
		"Dump the panics to file instead of console");

static char *dump_to_console;
module_param(dump_to_console, charp, 0400);
MODULE_PARM_DESC(dump_to_console,
		"Specify whether to dump to console or not");

static unsigned long rw_blocks = OOPS_MMC_NUM_RW_BLOCKS;
module_param(rw_blocks, ulong, 0400);
MODULE_PARM_DESC(rw_blocks,
		"record size for MMC OOPS in blocks (default 128)");

static struct oops_mmc_context {
	struct kmsg_dumper	dump;
	struct mmc_card		*card;
	/* start block address to dump*/
	unsigned long		start_block_addr;
	/* max blocks available to dump */
	unsigned long		max_block_count;
	/* each MMC block size */
	unsigned long		mmc_blksz;
	/* Total bytes that we write or read each time */
	unsigned long		num_rw_bytes;
	char				*buff;
} mmc_oops_context;


struct oops_mmc_platform_data {
	struct platform_device	*pdev;
	char			*mmcdev;
	unsigned long		partition_start_block;
	unsigned long		partition_size;
	unsigned long		mmc_blksz;
	char			*dump_file_path;
	char			*dump_to_console;
};

static inline bool is_mmc_ongoing_tfr(struct mmc_host *host)
{
	struct mmc_request *ongoing_mrq = READ_ONCE(host->ongoing_mrq);

	/*
	 * If there is an ongoing transfer, wait for the command line to become
	 * available.
	 */
	if (ongoing_mrq && !completion_done(&ongoing_mrq->cmd_completion))
		return true;

	return false;
}

static void oops_mmc_panic_read_write(struct oops_mmc_context *cxt,
	char *buf, unsigned long start, int num_blocks, char rw, bool tfr)
{
	struct mmc_card *card = cxt->card;
	struct mmc_host *host = card->host;
	struct mmc_request mrq;
	struct mmc_command cmd;
	struct mmc_command stop;
	struct mmc_data data;
	struct scatterlist sg;

	memset(&mrq, 0, sizeof(struct mmc_request));
	memset(&cmd, 0, sizeof(struct mmc_command));
	memset(&stop, 0, sizeof(struct mmc_command));
	memset(&data, 0, sizeof(struct mmc_data));

	mrq.cmd = &cmd;
	mrq.data = &data;
	mrq.stop = &stop;
	mrq.cap_cmd_during_tfr = tfr;

	data.blksz = cxt->mmc_blksz;
	data.blocks = num_blocks;
	data.sg = &sg;
	data.sg_len = 1;

	sg_init_one(&sg, buf, (cxt->mmc_blksz*num_blocks));

	if (rw == 'r') {
		if (data.blocks == 1)
			cmd.opcode = MMC_READ_SINGLE_BLOCK;
		else
			cmd.opcode = MMC_READ_MULTIPLE_BLOCK;
		data.flags = MMC_DATA_READ;
	} else if (rw == 'w') {
		if (data.blocks == 1)
			cmd.opcode = MMC_WRITE_BLOCK;
		else
			cmd.opcode = MMC_WRITE_MULTIPLE_BLOCK;
		data.flags = MMC_DATA_WRITE;
	} else {
		pr_err("oops_mmc_panic_%s : wrong rw code =%d\n",
				(rw == 'r')?"read":"write", rw);
		return;
	}

	cmd.arg = start;
	cmd.flags = MMC_RSP_R1 | MMC_CMD_ADTC;

	if (data.blocks == 1)
		mrq.stop = NULL;
	else {
		stop.opcode = MMC_STOP_TRANSMISSION;
		stop.arg = 0;
		stop.flags = MMC_RSP_R1B | MMC_CMD_AC;
	}
	mmc_claim_host(host);
	if (tfr) {
		if(!is_mmc_ongoing_tfr(host))
			mmc_wait_for_req(host, &mrq);
		else
			pr_info("Opps dump not possible, since previous"
					" mmc transaction not complete\n");
	}
	else
		mmc_wait_for_req(host, &mrq);
	mmc_release_host(host);

	if (cmd.error)
		pr_err("oops_mmc_panic_%s[%d] cmd error %d\n",
			(rw == 'r')?"read":"write", __LINE__, cmd.error);
	if (data.error)
		pr_err("oops_mmc_panic_%s[%d] data error %d\n",
			(rw == 'r')?"read":"write", __LINE__, data.error);
}

static void oops_mmc_do_dump(struct kmsg_dumper *dumper,
			enum kmsg_dump_reason reason)
{
	struct oops_mmc_context *cxt =
		container_of(dumper, struct oops_mmc_context, dump);
	char *buff = cxt->buff;
	bool rc = true;
	size_t header_size, text_len = 0;
	int text_length = 0, read_cnt = 0;

	if (!in_nmi()) {
		pr_info("oops_mmc_do_dump: Not in NMI Routine...\n");
		return;
	}
	/* Add Dump signature and
	* block size before kernel log
	*/
	memset(buff, '\0', cxt->num_rw_bytes);
	memcpy(buff, OOPS_MMC_DUMP_SIGNATURE,
			strlen(OOPS_MMC_DUMP_SIGNATURE));
	header_size = strlen(OOPS_MMC_DUMP_SIGNATURE)
						+ sizeof(int);
	read_cnt = cxt->num_rw_bytes-header_size-text_length;
	buff += header_size;
	while (1) {
		rc = kmsg_dump_get_buffer(dumper, false,
				buff, read_cnt, &text_len);
		text_length += text_len;
		if (!rc) {
			pr_err("oops_mmc_do_dump: end of read from the kmsg dump\n");
			break;
		}
		buff = (char *)cxt->buff+header_size+text_length;
		read_cnt = cxt->num_rw_bytes-header_size-text_length;
		if (read_cnt<=0) {
			pr_err("oops_mmc_do_dump: read limit reached from the kmsg dump\n");
			break;
		}
	}
	pr_info("oops_mmc_do_dump: writing to MMC = %d\n",
				text_length);
	buff = (char *)cxt->buff+strlen(OOPS_MMC_DUMP_SIGNATURE);
	memcpy(buff, &text_length, sizeof(int));
	/* Write the emmc flash at one iteration, the emmc write
	   is interrupt driven, it will hang on futher emmc write
	   since we are executing under NMI routine */
	oops_mmc_panic_read_write(cxt, cxt->buff,
		(cxt->start_block_addr), rw_blocks, 'w', false);
}

static int oops_mmc_probe(struct platform_device *pdev)
{
	struct oops_mmc_platform_data *pdata = pdev->dev.platform_data;
	struct oops_mmc_context *context = &mmc_oops_context;
	int err = -EINVAL;
	int i;
	char *buf;
	int text_len = 0;
	loff_t pos = 0;
	struct file *file = NULL;
	mm_segment_t old_fs;
	char marker_string[200]="";

	if (!pdata) {
		pr_err("oops_mmc_probe: No platform data. Error!\n");
		return -EINVAL;
	}

	if (pdata->mmcdev == NULL) {
		pr_err("oops_mmc_driver_probe: mmcdev is NULL\n");
		return err;
	}

	context->mmc_blksz = pdata->mmc_blksz;
	context->start_block_addr = pdata->partition_start_block;
	context->max_block_count = (pdata->partition_size)/(pdata->mmc_blksz);
	context->num_rw_bytes = (rw_blocks * context->mmc_blksz);
	context->card = mmc_blk_dev_to_card(pdata->mmcdev);

	if (context->card == NULL) {
		pr_err("oops_mmc_driver_probe: card is NULL\n");
		return err;
	}

	if (mmc_card_mmc(context->card))
		pr_info("oops_mmc_driver_probe: its MMC card..\n");
	else if (mmc_card_sd(context->card))
		pr_info("oops_mmc_driver_probe: its SD card..\n");
	else {
		pr_err("oops_mmc_driver_probe: Not and SD or MMC! return error!\n");
		return err;
	}

	pr_info("oops_mmc_probe: partition_start_block=0x%lx, mmc_blksz=0x%lx, max_blocks=%lu\n",
			context->start_block_addr,
			context->mmc_blksz, context->max_block_count);

	/* Allocate min io size buffer to be used in do_dump */
	context->buff = kmalloc(context->num_rw_bytes, GFP_KERNEL);
	if (context->buff == NULL) {
		err = -EINVAL;
		goto kmalloc_failed;
	}

	context->dump.dump = oops_mmc_do_dump;
	err = kmsg_dump_register(&context->dump);
	if (err) {
		pr_err("oops_mmc_probe: registering kmsg dumper failed\n");
		goto kmsg_dump_register_failed;
	}

	if (pdata->dump_file_path) {
		pr_info("oops_mmc_probe: dump_file_path = %s\n",
				pdata->dump_file_path);
		file = filp_open(pdata->dump_file_path, O_WRONLY|O_CREAT, 0644);
		if (IS_ERR(file)) {
			pr_err("oops_mmc_probe: filp_open failed, dump to console only\n");
			file = NULL;
		} else {
			old_fs = get_fs();
			set_fs(KERNEL_DS);
		}
	} else if ((pdata->dump_to_console) &&
			((!strcmp(pdata->dump_to_console, "n")) ||
			(!strcmp(pdata->dump_to_console, "no"))))
		pr_info("oops_mmc_probe: dump_to_console=no,OEM has own script");
	else
		pr_info("oops_mmc_probe: If any panic , it will be dumped to console\n");

	buf = (char *)context->buff;

	oops_mmc_panic_read_write(context, buf,
		context->start_block_addr, 1, 'r', false);
	if (!strncmp(OOPS_MMC_DUMP_SIGNATURE,
			buf, strlen(OOPS_MMC_DUMP_SIGNATURE))) {
		sprintf(marker_string, "\n%s%s%s\n", dump_mark, dump_start_str, dump_mark);
		pr_err("%s", marker_string);
		if (file) {
			kernel_write(file, marker_string,
				strlen(marker_string), &pos);
			pos = pos+strlen(marker_string);

		}
		if (pdata->dump_file_path)
			pr_info("oops_mmc_probe: panics dumped to the file [%s]\n",
				pdata->dump_file_path);
		else if ((pdata->dump_to_console) &&
			((!strcmp(pdata->dump_to_console, "n")) ||
			(!strcmp(pdata->dump_to_console, "no")))) {
				pr_info("oops_mmc_probe:OEM has own script to read!\n");
				pr_err("\n%s%s%s\n", dump_mark,
					dump_end_str, dump_mark);
				return 0;
		}

		for (i = 0; i < context->max_block_count; i+=rw_blocks) {
			oops_mmc_panic_read_write(context, buf,
				(context->start_block_addr+i),
				rw_blocks, 'r', false);
			if (strncmp(OOPS_MMC_DUMP_SIGNATURE, buf,
				    strlen(OOPS_MMC_DUMP_SIGNATURE))) {
				break;
			}
			memcpy(&text_len, &buf[strlen(OOPS_MMC_DUMP_SIGNATURE)],
						sizeof(int));
			buf = buf+strlen(OOPS_MMC_DUMP_SIGNATURE)+sizeof(int);

			if ((text_len == 0) || (text_len > context->num_rw_bytes)) {
				pr_info("oops_mmc_probe:Invalid text length[%d]\n",
						text_len);
				break;
			}
			pr_info("oops_mmc_probe: printing text length = %d\n",
								text_len);

			if (file) {
				kernel_write(file, buf, text_len, &pos);
				pos = pos+text_len;
			} else {
				char *ptr = buf;
				char *line;
				while ((line = strsep(&ptr, "\n")) != NULL) {
					pr_info("%s\n", line);
				}
			}
			buf = (char *)context->buff;
		}
		sprintf(marker_string,"\n%s%s%s\n", dump_mark, dump_end_str, dump_mark);
		pr_err("%s", marker_string);
		if (file) {
			kernel_write(file, marker_string,
				strlen(marker_string), &pos);
			pos = pos+strlen(marker_string);
		}
		/* Clear buffer */
		buf = (char *)context->buff;
		memset(buf, '\0', context->num_rw_bytes);
		for (i = 0; i < context->max_block_count; i+=rw_blocks) {
			oops_mmc_panic_read_write(context, buf,
				(context->start_block_addr+i),
				rw_blocks, 'w', false);
		}
	} else
		pr_info("oops_mmc_probe: There was no panic in earlier run\n");

	if (file) {
		filp_close(file, NULL);
		set_fs(old_fs); /*Reset to save FS*/
	}
	return 0;

kmsg_dump_register_failed:
	kfree(context->buff);
	context->buff = NULL;
kmalloc_failed:
	return err;

}

static int oops_mmc_remove(struct platform_device *pdev)
{
	struct oops_mmc_context *cxt = &mmc_oops_context;

	pr_info("oops_mmc_remove\n");
	if (kmsg_dump_unregister(&cxt->dump) < 0)
		pr_err("oops_mmc_remove: colud not unregister kmsg dumper\n");
	kfree(cxt->buff);
	cxt->buff = NULL;
	return 0;
}

static struct platform_driver oops_mmc_driver = {
	.remove			= __exit_p(oops_mmc_remove),
	.probe                  = oops_mmc_probe,
	.driver			= {
		.name		= "oops_mmc",
		.owner		= THIS_MODULE,
	},
};

static int __init oops_mmc_init(void)
{
	int ret;

	pr_info("oops_mmc_init\n");
	ret = platform_driver_probe(&oops_mmc_driver, oops_mmc_probe);
	if (ret == -ENODEV) {
		/*
		* If we didn't find a platform device, we use module parameters
		* building platform data on the fly.
		*/
		pr_info("platform device not found, using module parameters\n");

		if (strlen(mmcdev) == 0) {
			pr_err("oops_mmc_init: mmc device (mmcdev=name) must be supplied\n");
			return -EINVAL;
		}

		if (partition_start_block == OOPS_MMC_INVALID ||
				partition_size == OOPS_MMC_INVALID ||
				mmc_blksz == OOPS_MMC_INVALID){
			pr_err("oops_mmc_init : Invalid module params\n");
			return -EINVAL;
		}

		dummy_data = kzalloc(sizeof(struct oops_mmc_platform_data),
					GFP_KERNEL);
		if (!dummy_data)
			return -ENOMEM;

		dummy_data->mmcdev = mmcdev;
		dummy_data->partition_start_block = partition_start_block;
		dummy_data->partition_size = partition_size;
		dummy_data->mmc_blksz = mmc_blksz;
		dummy_data->dump_file_path = dump_file_path;
		dummy_data->dump_to_console = dump_to_console;
		dummy = platform_create_bundle(&oops_mmc_driver, oops_mmc_probe,
			NULL, 0, dummy_data,
			sizeof(struct oops_mmc_platform_data));
		if (IS_ERR(dummy)) {
			ret = PTR_ERR(dummy);
			kfree(dummy_data);
		} else
			ret = 0;
	}
	return ret;

}
static void __exit oops_mmc_exit(void)
{
	kfree(dummy_data);
	dummy_data = NULL;
	pr_info("oops_mmc_exit\n");
	platform_device_unregister(dummy);
	platform_driver_unregister(&oops_mmc_driver);
	dummy = NULL;
}

module_init(oops_mmc_init);
module_exit(oops_mmc_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("BROADCOM");
MODULE_DESCRIPTION("MMC Oops/Panic Looger");
